<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\AccountEvents;
use App\Models\AccountEventMapping;
use Illuminate\Support\Facades\Bus;
use App\Models\Account;
use App\Models\AccountControl;
use App\Models\AccountMain;
use App\Models\AccountSubControl;
use App\Models\AccountHeads;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;
use Carbon\Carbon;
use App\Services\FiltersService;

use Illuminate\Support\Facades\DB;

use function PHPSTORM_META\map;

class AccountsController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function chart_of_accounts(Request $request)
    {   
        $business_id = $request->input('business_id');
        if($business_id == "" || $business_id == 0) {
                $userRole = session('user_role');
                $allowedRoles = ['Super User', 'Super Admin', 'Accountant', 'Admin'];
                if (in_array($userRole, $allowedRoles)) {
                    if (session('ho') == 'Yes') {
                        $business_id = session('business_id');
                    } elseif (session('ho_accounts') == 'Yes') {
                        return response()->view('error.403', ['message' => 'Your account is handled by the Head Office'], 403);
                    } else {
                       $business_id = session('business_id');
                    }
                } else {
                     return "You do not have the permission to access this resource.";
                }
        }
        
        $account_main = AccountMain::where('business_id', $business_id)->get();

        $account_control = AccountControl::where('business_id', $business_id)->get();

        $account_sub_control = AccountSubControl::where('business_id', $business_id)->get();

        $account_heads = AccountHeads::where('business_id', $business_id)->get();
        
        $accounts = AccountMain::select(
            'business.id_business',
            'business.business_name',
            'id_account_main',
            'account_main',
            'account_main_number',
            'id_account_control',
            'account_control',
            'account_control_number',
            'id_account_sub_control',
            'account_sub_control',
            'account_sub_control_number',
            'id_account_heads',
            'account_head',
            'account_head_number',
            'till_account',
            DB::raw("CONCAT(account_main_number, '-', account_control.account_control_number, '-', account_sub_control_number, '-', account_head_number) as account_number")
        )     
        ->leftJoin('account_control', function($join) {
            $join->on('account_control.account_main_id', '=', 'account_main.id_account_main')
                ->where('account_control.account_control_status', '=', 'Active');
        })
        ->leftJoin('account_sub_control', function($join) {
            $join->on('account_control.id_account_control', '=', 'account_sub_control.account_control_id')
                ->where('account_sub_control.account_sub_control_status', '=', 'Active');
        })
        ->leftJoin('account_heads', function($join) use ($business_id) {
            $join->on('account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
                ->where('account_heads.account_head_status', '=', 'Active')
                ->where('account_heads.business_id', '=', $business_id);
        })
        ->leftJoin('business', function($join) use ($business_id) {
            $join->on('business.id_business', '=', 'account_heads.business_id')            
             ->where('business.id_business', '=', $business_id);       
        })
        ->where('account_main.account_main_status', '=', 'Active');
        $accounts = $accounts->orderBy('id_account_main')
        ->orderBy('id_account_control')
        ->orderBy('id_account_sub_control')
        ->orderBy('id_account_heads')
        ->get();

        if(session("ho")=="Yes"){
            $branches = Business::where('ho_accounts', 'Yes')->orwhere('ho','Yes')->get();
        } else {
            $branches = Business::where('id_business',session('business_id'))->get();
        }   
        $this_business = Business::where('id_business', $business_id)->first();
        return view('accounts.chart_of_accounts')
        ->with(compact('account_main', 'account_control', 'account_sub_control', 'account_heads', 'accounts', 'branches', 'business_id', 'this_business'));

    }

    public function add_main(Request $request){
        try {
            if(session('user_role')=="Super User" || session('user_role')=='Super Admin'){
                if(session('ho') == "Yes"){
                    $businesses = Business::where('ho', "Yes")->get();
                } else if(session('ho_accounts') == "No"){
                    $businesses = Business::where("id_business", session('business_id'))->get();
                } else if(session('ho_accounts') == "Yes" && session('ho') == "No"){
                    return response()->json([
                        "message" => "Your Accounts are handled by the Head Office and you dont have permission to Add new Main Account.",
                        "error" => 'error'
                    ]);
                }
            } else {
                return response()->json([
                    "message" => "You dont have permission to Add new Main Account.",
                    "error" => 'error'
                ]);
            }
            foreach($businesses as $business){
                //Add a new Main Account
                $mainAccount = new AccountMain();            
                $mainAccount->account_main = $request->input('account_main');
                $mainAccount->account_main_number = $request->input('account_main_number');
                $mainAccount->account_main_status = $request->input('account_main_status');
                $mainAccount->business_id = $business->id_business;
                $mainAccount->save();
            }
            return response()->json([
                "account_main" => $mainAccount,
                "message" => "success",
                "success" => true
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    
    }
    public function edit_main(Request $request)
    {
        try{
            if(session('user_role') != 'Super User') {
                return response()->json([
                    "message" => "You do not have permission to change the Main Accounts.",
                    "error" => "error"
                ], 403);
            }   

            $id = $request->id_account_main;
            $account_main = AccountMain::where('id_account_main', $id)->first();
            return response()->json([
                "account_main" => $account_main,
                "message" => "success",
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }
    public function update_main(Request $request){
        try{
            $id_account_main = $request->input('id_account_main');
            if(session('user_role') != 'Super User') {
                return response()->json([
                    "message" => "You do not have permission to change the Main Accounts.",
                    "error" => "error"
                ], 403);
            }
            $account_main = AccountMain::where('id_account_main', $id_account_main)->first();
            if (!$account_main) {
                return response()->json([
                    "message" => "Main Account not found.",
                    "error" => "error"
                ], 404);
            }
            $account_main->account_main = $request->input('account_main');
            $account_main->account_main_number = $request->input('account_main_number');
            $account_main->account_main_status = $request->input('account_main_status');
            $account_main->save();

            return response()->json([
                "message" => "success",
                "success" => true
            ]);
        }catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function add_control(Request $request){
        try {

            if(session('user_role')=="Super User" || session('user_role')=='Super Admin'){
                if(session('ho') == "Yes"){
                    $businesses = Business::where('ho', "Yes")->get();
                } else if(session('ho_accounts') == "No"){
                    $businesses = Business::where("id_business", session('business_id'))->get();
                } else if(session('ho_accounts') == "Yes" && session('ho') == "No"){                   
                    return response()->json([
                        "message" => "Your Accounts are handled by the Head Office and you dont have permission to Add new Main Account.",
                        "error" => 'error'
                    ]);
                }
            } else {
                return response()->json([
                    "message" => "You dont have permission to Add new Control Account.",
                    "error" => 'error'
                ]);
            }

            foreach($businesses as $business){
                //Add a new Control Account
                $account_control = new AccountControl();            
                $account_control->account_control = $request->input('account_control');
                $account_control->account_control_number = $request->input('account_control_number');
                $account_control->account_control_status = $request->input('account_control_status');
                $account_control->account_main_id = $request->input('account_main_id');
                $account_control->business_id = $business->id_business;
                $account_control->save();
            }

            return response()->json([
                "message" => "success",
                "success" => true
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function edit_control(Request $request)
    {
        try{
            $id = $request->id_account_control;
        
            $account_control = AccountControl::select('*')->where('id_account_control', $id)->first();

            return response()->json([
                "account_control" => $account_control,
                "message" => "success",
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function update_control(Request $request){
        try{
            $id_account_control = $request->input('id_account_control');

            if(session('user_role') != 'Super User') {
                return response()->json([
                    "message" => "You do not have permission to change the Control Accounts.",
                    "error" => "error"
                ], 403);
            }

            $account_control = AccountControl::where('id_account_control', $id_account_control)->first();
           
            if (!$account_control) {
                return response()->json([
                    "message" => "Control Account not found",
                    "error" => "error"
                ], 404);
            }
            $account_control->account_control = $request->input('account_control');
            $account_control->account_control_status = $request->input('account_control_status');
            $account_control->account_main_id = $request->input('account_main_id');
            $account_control->account_control_number = $request->input('account_control_number');
            $account_control->save();

            return response()->json([
                "message" => "success",
                "success" => true
            ]);
        }catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function edit_sub_control(Request $request)
    {
        try{
            $id = $request->id_account_sub_control;
        
            $account_sub_control = AccountSubControl::where('id_account_sub_control', $id)->first();

            return response()->json([
                "account_sub_control" => $account_sub_control,
                "message" => "success",
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function add_sub_control(Request $request){
        try {
            if(session('user_role')=="Super User" || session('user_role')=='Super Admin'){
                if(session('ho') == "Yes"){
                    $businesses = Business::where('ho', "Yes")->get();
                } else if(session('ho_accounts') == "No"){
                    $businesses = Business::where("id_business", session('business_id'))->get();
                } else if(session('ho_accounts') == "Yes" && session('ho') == "No"){                   
                    return response()->json([
                        "message" => "Your Accounts are handled by the Head Office and you dont have permission to Add new Main Account.",
                        "error" => 'error'
                    ]);
                }
            } else {
                return response()->json([
                    "message" => "You dont have permission to Add new Sub Control Account.",
                    "error" => 'error'
                ]);
            }

            foreach($businesses as $business){
                //Add a new Sub Control Account
                $account_sub_control = new AccountSubControl();            
                $account_sub_control->account_sub_control = $request->input('account_sub_control_name');
                $account_sub_control->account_sub_control_number = $request->input('account_sub_control_number');
                $account_sub_control->account_sub_control_status = $request->input('account_sub_control_status');
                $account_sub_control->account_control_id = $request->input('account_control_id');
                $account_sub_control->business_id = $business->id_business;
                $account_sub_control->save();
            }

            return response()->json([
                "message" => "success",
                "success" => true
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function update_sub_control(Request $request){
        try{
            $id_account_sub_control = $request->input('id_account_sub_control');

            if(session('user_role') != 'Super User') {
                return response()->json([
                    "message" => "You do not have permission to change the Sub Control Accounts.",
                    "error" => "error"
                ], 403);
            }

            $account_sub_control = AccountSubControl::where('id_account_sub_control', $id_account_sub_control)->first();
           
            if (!$account_sub_control) {
                return response()->json([
                    "message" => "Sub Control Account not found.",
                    "error" => "error"
                ], 404);
            }
            $account_sub_control->account_sub_control = $request->input('account_sub_control_name');
            $account_sub_control->account_sub_control_status = $request->input('account_sub_control_status');
            $account_sub_control->account_control_id = $request->input('account_control_id');
            $account_sub_control->account_sub_control_number = $request->input('account_sub_control_number');
            $account_sub_control->save();

            return response()->json([
                "message" => "success",
                "success" => true
            ]);
        }catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function edit_head(Request $request)
    {
        try{
            $id = $request->id_account_heads;

            $account_heads = AccountHeads::where('id_account_heads', $id)->first();
            
            return response()->json([
                "account_heads" => $account_heads,
                "message" => "success",
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function add_head(Request $request){
        try {

            if(session('user_role')=="Super User" || session('user_role')=='Super Admin' || session('user_role')=='Accountant' || session('user_role')=='Admin'){
                if(session('ho') == "Yes"){
                    $businesses = Business::where('ho', "Yes")->get();
                } else if(session('ho_accounts') == "No"){
                    $businesses = Business::where("id_business", session('business_id'))->get();
                } else if(session('ho_accounts') == "Yes" && session('ho') == "No"){                   
                    return response()->json([
                        "message" => "Your Accounts are handled by the Head Office and you dont have permission to Add new Account Head.",
                        "error" => 'error'
                    ]);
                }
            } else {
                return response()->json([
                    "message" => "You dont have permission to Add new Account Head.",
                    "error" => 'error'
                ]);
            }

            foreach($businesses as $business){
                //Add a new Account Head
                $account_head = new AccountHeads();            
                $account_head->account_head = $request->input('account_head');
                $account_head->account_head_number = $request->input('account_head_number');
                $account_head->account_head_status = $request->input('account_head_status');
                $account_head->account_sub_control_id = $request->input('account_sub_control_id');
                $account_head->business_id = $business->id_business;
                $account_head->save();
            }

            return response()->json([
                "message" => "success",
                "success" => true
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }    
    }
    
    public function update_head(Request $request)
    {
        try {
            $id_account_heads = $request->input('id_account_heads');
            $account_head = $request->input('account_head');

            $userRole = session('user_role');
            $businessId = session('business_id');

            if(session('ho_accounts') == "Yes" && session('ho') == "No"){                   
                    return response()->json([
                        "message" => "Your Accounts are handled by the Head Office and you dont have permission to Add new Account Head.",
                        "error" => 'error'
                    ]);
            }

            if (($userRole == 'Super Admin' || $userRole == 'Super User' || $userRole =='Accountant' || $userRole=='Admin')) {
                 $account_heads = AccountHeads::where('id_account_heads', $id_account_heads);
                    if (session('ho') === "Yes" || session('ho_accounts') === "Yes") {
                        $account_heads->whereIn('business_id', function ($q) {
                            $q->select('id_business')
                            ->from('business')
                            ->where('ho', 'Yes');
                        });
                    } else {
                        $account_heads->where('business_id', session('business_id'));
                    }

                    $account_heads = $account_heads->get();

                if ($account_heads->isEmpty()) {
                    return response()->json([
                        "message" => "No account heads found to update",
                        "error" => "error"
                    ], 404);
                }

                foreach ($account_heads as $head) {
                    $head->account_head = $request->input('account_head');
                    $head->account_head_status = $request->input('account_head_status');
                    $head->account_sub_control_id = $request->input('account_sub_control_id');
                    $head->account_head_number = $request->input('account_head_number');
                    $head->save();
                }
                
            } else {
                return response()->json([
                    "message" => "You do not have permission to change the Account Heads.",
                    "error" => "error"
                ], 403);
            }

            return response()->json([
                "message" => "success",
                "success" => true
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function open_account_vouchers(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
            
            } else {
                $business_id = session('business_id');
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
              return "You do not have the permission to access this resource.";
        }

        $business_id = $request->input('business_id');

        if(session('ho') == 'Yes') {
           // $business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
            $business_id = Business::where('ho', 'Yes')->value('id_business'); //get any HO business id
            //get all businesses
            $businesses = Business::where('ho_accounts', "Yes")->orWhere('ho','Yes')->get();
        } else if(session('ho_accounts') == 'Yes'){
            $business_id = Business::where('ho', 'Yes')->value('id_business'); //get any HO business id
            //get all businesses
            $businesses = Business::where('id_business',session('business_id'))->get();
        } else {
            $business_id = session('business_id');
            //get all businesses
            $businesses = Business::where('id_business', $business_id)->get(); 
        }

        return view('accounts.account_vouchers')
            ->with(compact('businesses'));

    }

    public function account_vouchers(Request $request)
    {
        $business_id = $request->get('business_id');
        $start_date  = $request->get('start_date');
        $end_date    = $request->get('end_date');
        $voucher_type = $request->get('voucher_type');
        $auto_voucher = $request->get('auto_voucher');
        $reftype = $request->get('reftype') ?? '';
        $refid = $request->get('refid') ?? '';

        if (empty($start_date) || empty($end_date) || !strtotime($start_date) || !strtotime($end_date)) {
            $start_date = date('Y-m-01');
            $end_date   = date('Y-m-t');
        }

        //check if start_date > end_date
        if (strtotime($start_date) > strtotime($end_date)) {
            return response()->json([
                "message" => "error",
                "error" => "Start date cannot be greater than end date."
            ]);
        }

        if($business_id == "" || $business_id == 0) {
            if(session('ho') == 'Yes') {          
                //$business_id = Business::where('ho', 'Yes')->value('id_business'); //get any HO business id
                $business_id = 0; // HO mode
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get();
            } else {
                $business_id = session('business_id');         
                $businesses = Business::where('id_business', $business_id)->get();
            }
        } else {
            $businesses = Business::where('id_business', $business_id)->get();
        }

        // DataTables parameters
        $start  = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->input('search.value');
        $orderColumnIndex = $request->input('order.0.column', 0);
        $orderDir = $request->input('order.0.dir', 'asc');

        // Define columns array (match the JS DataTable columns)
        $columns = [
            'v.id_account_vouchers',
            'v.voucher_type',
            'v.voucher_date',
            'v.created_on',
            'v.description',
            'b.business_name',
            'v.business_partner_name',
            'v.created_by',
            'h.account_head',
            'd.debit',
            'd.credit',
            'd.auto_voucher',
        ];

        $orderColumn = $columns[$orderColumnIndex] ?? 'v.voucher_date';

        // Base query
        $query = DB::table('account_vouchers as v')
            ->join('account_voucher_detail as d', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->leftJoin('account_heads as h', 'd.account_head_id', '=', 'h.id_account_heads')
            ->leftJoin('business as b', 'v.business_id', '=', 'b.id_business')      
            ->leftJoin('account_voucher_type as vt', 'v.voucher_type', '=', 'vt.id_account_voucher_type')      
            ->select(
                'v.id_account_vouchers',
                'vt.account_voucher_type as voucher_type',
                DB::RAW('date_format(v.voucher_date, "%d-%m-%Y") as voucher_date'),
                DB::RAW('date_format(v.created_on, "%d-%m-%Y %H:%i") as created_on'),
                'v.description',
                'v.voucher_status',
                'b.business_name',
                'v.business_partner_name',
                'v.created_by',
                'h.account_head',
                'd.debit as total_debit',
                'd.credit as total_credit',
                DB::RAW('"' . session('user_role') . '" as user_role'),
                'v.auto_voucher',
            );
            if(empty($refid)){
                // $query->whereBetween('v.voucher_date', [$start_date, $end_date]);
                $query->whereRaw("DATE(v.voucher_date) BETWEEN ? AND ?", [$start_date, $end_date]);
            }

            if(!empty($refid)){
                $query->where($reftype, $refid);
            }             

        // Apply voucher type filter
        if ($voucher_type != 0 && $voucher_type != '' && empty($refid)) {
            $query->where('v.voucher_type', $voucher_type);
        }

        // Apply auto voucher filter
        if ($auto_voucher != '' && empty($refid)) {
            if($auto_voucher == 'Yes' || $auto_voucher == 'No'){
                $query->where('v.auto_voucher', $auto_voucher);
            }           
        }

        // Apply business filter
        if ($business_id != 0) {
            $query->where('v.business_id', $business_id);
        } else {
            $query->whereIn('v.business_id', function ($q) {
                $q->select('id_business')->from('business')->where('ho_accounts', 'Yes')->orWhere('ho','Yes');
            });
        } 
        // Total record count (before filtering)
        $recordsTotal = $query->count();
        // Apply search filter
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('v.voucher_type', 'like', "%{$search}%")
                ->orWhere('v.description', 'like', "%{$search}%")
                ->orWhere('h.account_head', 'like', "%{$search}%")
                ->orWhere('b.business_name', 'like', "%{$search}%")
                ->orWhere('v.business_partner_name', 'like', "%{$search}%");
            });
        }

        // Count after filtering
        $recordsFiltered = $query->count();

        // Totals (Active only)
        $totalDebit = (clone $query)->where('v.voucher_status', 'Active')->sum('d.debit');
        $totalCredit = (clone $query)->where('v.voucher_status', 'Active')->sum('d.credit');



        // Pagination + ordering
        $data = $query
            ->orderBy($orderColumn, $orderDir)
            ->offset($start)
            ->limit($length)
            ->get();

       // print_r($data); die;

        // Return JSON in DataTables format
        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
            'totals' => [
                'debit' => round($totalDebit, 2),
                'credit' => round($totalCredit, 2),
            ],
            'businesses' => $businesses,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'business_id' => $business_id,
        ]);
    }

    public function open_create_voucher(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->select('id_business','business_name')->get();
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                //redirect or return with error
                // return redirect()->back()->withErrors("Your account is handled by the Head Office.")->withInput();
                // return "Your account is handled by the Head Office.";
                $business_id = Business::where('ho', 'Yes')->value('id_business');
                $businesses = Business::where('id_business', session("business_id"))->select('id_business','business_name')->get();
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', session("business_id"))->select('id_business','business_name')->get();
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        //get all account voucher types
        $account_voucher_type = DB::table('account_voucher_type')
            ->get();

        //get all business partners
        $business_partners = DB::table('account_business_partner')
            ->get();

        //get all cost centers
        $cost_centers = DB::table('account_cost_center')
            ->get();

        //get all account heads
        $account_heads = AccountHeads::where('business_id', $business_id)
            ->where('account_head_status', 'Active')
            ->get();

        $permission = 'create-Account_Vouchers';

        return view('accounts.create_voucher')->with(compact('account_voucher_type', 'account_heads', 'business_partners', 'cost_centers', 'businesses', 'permission'));
    }

    public function open_edit_voucher(Request $request, $id)
    {
        $business_id = session('business_id');
        $permission = 'edit-Account_Vouchers';

        $voucher = AccountVouchers::where('id_account_vouchers', $id)
            ->first();

        if (!$voucher) {
            return redirect()->back()->withErrors(['Voucher not found or access denied.']);
        }

        if ($voucher->auto_voucher == 'Yes' && session('user_role') != 'Super Admin' && session('user_role') != 'Super User') {
            // return redirect()->back()->withErrors(['Auto vouchers cannot be edited manually.']);
            return "Auto vouchers cannot be edited manually.";
        }

        $business_id = $voucher->business_id;
        $voucherBusinessId = $voucher->business_id; 
        // Fiscal Year Closing Validation
        $voucher_date = date('Y-m-d', strtotime($voucher->voucher_date));
        if (session('user_role') != "Super User") {
            $lastClosing = DB::table('fiscal_year_closings')
                ->selectRaw("DATE(closing_date) as closing_date")
                ->where('business_id', $voucherBusinessId)
                ->where('status', 'Active')
                ->orderBy('closing_date', 'desc')
                ->first();

            if ($lastClosing && $voucher_date <= $lastClosing->closing_date) {
                return "Voucher cannot be edited as its date falls within the last closed fiscal period ({$lastClosing->closing_date}).";
            }
        }

        if (session('user_role') != "Super User" && session('user_role') != "Super Admin"){
            if ($voucherBusinessId != session('business_id')) {
                return "You are attempting to edit a voucher for a different branch. You do not have permission to edit this voucher.";
            }
        }
        
        // Load voucher details and other necessary data for editing
        $details = AccountVoucherDetail::where('account_voucher_id', $id)
        ->leftJoin('account_heads', 'account_voucher_detail.account_head_id', '=', 'account_heads.id_account_heads')
        ->leftJoin('account_cost_center', 'account_cost_center.id_account_cost_center', '=', 'account_voucher_detail.cost_center_type')
        ->select('account_voucher_detail.*', 'account_heads.*', 'account_cost_center.account_cost_center as account_cost_center_type_name')
        ->orderBy('account_voucher_detail.id_account_voucher_detail', 'asc')  
        ->get();

        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->select('id_business','business_name')->get();
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                //redirect or return with error
                // return redirect()->back()->withErrors("Your account is handled by the Head Office.")->withInput();
                // return "Your account is handled by the Head Office."; 
                $business_id = Business::where('ho', 'Yes')->value('id_business');
                $businesses = Business::where('id_business', session("business_id"))->select('id_business','business_name')->get();
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', session("business_id"))->select('id_business','business_name')->get();
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        //get all account voucher types
        $account_voucher_type = DB::table('account_voucher_type')
            ->get();

        //get all business partners
        $business_partners = DB::table('account_business_partner')
            ->get();

        //get all cost centers
        $cost_centers = DB::table('account_cost_center')
            ->get();

        //get all account heads
        $account_heads = AccountHeads::where('business_id', $business_id)
            ->where('account_head_status', 'Active')
            ->get();

        return view('accounts.create_voucher', compact('voucher', 'details', 'account_voucher_type', 'account_heads', 'business_partners', 'cost_centers','voucherBusinessId','businesses', 'permission'));
    }

    public function save_voucher(Request $request)
    {   
        $business_id_for_voucher =$request->business_id;
        //code to save voucher
        $voucher_date = date('Y-m-d', strtotime($request->input('voucher_date')));
        if (session('user_role') != "Super User") {
            $lastClosing = DB::table('fiscal_year_closings')
                ->selectRaw("DATE(closing_date) as closing_date")
                ->where('business_id', $business_id_for_voucher)
                ->where('status', 'Active')
                ->orderBy('closing_date', 'desc')
                ->first();

            if ($lastClosing && $voucher_date <= $lastClosing->closing_date) {
                return response()->json(['success' => false, 'message_type' => 'error', 'message' => "Voucher cannot be added as its date falls within the last closed fiscal period ({$lastClosing->closing_date})."]);
            }
        }

        try{
           
            DB::beginTransaction();
            $id_account_voucher = $request->input('id_account_voucher');
            $account_voucher_type = $request->input('voucher_type');
            $account_voucher_date = $request->input('voucher_date');
            $description = $request->input('description');
            $business_partner_id = $request->input('business_partner_id');
            $business_partner_name = $request->input('business_partner_name');
            $business_partner = $request->input('business_partner');
            $created_by = session('username');
            $description = $description;
            $cost_center = $request->input('cost_center') ?? 1;
            $cost_center_name = $request->input('cost_center_name') ?? 'Main Outlet';
            $payment_mode = $request->input('payment_mode');
            $instrument_number = $request->input('instrument_number');
            $voucher_details = $request->input('voucher_details', []);
            $voucher_status =  $request->input('voucher_status', 'Active'); 


            //check if voucher date is valid
            if(!strtotime($account_voucher_date)){
                return response()->json([
                    "message_type" => "error",
                    "message" => "Invalid voucher date.",
                    "message_btn" => "btn btn-danger",
                    "success" => false,
                ]);
            }
            if($id_account_voucher == null || $id_account_voucher == 0){ 


                //new voucher
                $voucher = new AccountVouchers();
                $voucher->voucher_type = $account_voucher_type;
                $voucher->voucher_status = $voucher_status;
                $voucher->voucher_date = $account_voucher_date;
                $voucher->description = $description;
                $voucher->business_id = $business_id_for_voucher;
                $voucher->created_by = session('user_name');
                $voucher->business_partner_id = $business_partner_id;
                $voucher->business_partner_name = $business_partner_name;
                $voucher->business_partner = $business_partner;
                $voucher->cost_center = $voucher_details[0]['cost_center_type'] ?? 1;
                $voucher->cost_center_name = $voucher_details[0]['cost_center'] ?? 'Main Outlet';
                $voucher->payment_mode = $payment_mode;
                $voucher->voucher_status = 'Active';
                $voucher->auto_voucher = $request->input('auto_voucher', 'No');
                $voucher->save();
                
                //check if debit and credit are balanced
                $total_debit = 0;
                $total_credit = 0;
                foreach($voucher_details as $detail){
                    $total_debit += floatval($detail['debit']);
                    $total_credit += floatval($detail['credit']);
                }
                if(round($total_debit, 2) != round($total_credit, 2)){
                    DB::rollBack();
                    return response()->json([
                        "message_type" => "error",
                        "message" => "Total Debit and Credit amounts are not balanced.",
                        "message_btn" => "btn btn-danger",
                        "success" => false,
                    ]);
                }

                foreach($voucher_details as $detail){
                    $voucher_detail = new AccountVoucherDetail();
                    $voucher_detail->account_voucher_id = $voucher->id_account_vouchers;
                    $voucher_detail->account_head_id = $detail['account_head_id'];
                    $voucher_detail->debit = $detail['debit'];
                    $voucher_detail->credit = $detail['credit'];
                    $voucher_detail->cost_center = $detail['cost_center']  ?? 1;
                    $voucher_detail->cost_center_type = $detail['cost_center_type'] ?? 'Main Outlet';
                    $voucher_detail->detail_remarks = $detail['detail_remarks'];
                    $voucher_detail->instrument_number = $instrument_number = $detail['instrument_number'] ?? '';
                    $voucher_detail->payment_mode = $payment_mode ?? '';

                    $voucher_detail->save();
                }
                DB::commit();
                return response()->json([
                    "message" => "Voucher created successfully . ID: " . $voucher->id_account_vouchers,
                    "message_type" => "success",
                    "message_btn" => "btn btn-success",
                    "success" => true,
                ]);
            } else {
                //editing existing voucher
                $voucher = AccountVouchers::find($id_account_voucher);
                if (!$voucher) {
                    DB::rollBack();
                    return response()->json([
                        "message_type" => "error",
                        "message" => "Voucher not found.",
                        "message_btn" => "btn btn-danger",
                        "success" => false,
                    ]);
                }
                //Update the existing voucher using $id_account_voucher
                $voucher->voucher_type = $account_voucher_type;
                $voucher->voucher_status = $voucher_status;
                $voucher->voucher_date = $account_voucher_date;
                $voucher->description = $description;
                $voucher->business_partner_id = $business_partner_id;
                $voucher->business_partner_name = $business_partner_name;
                $voucher->business_partner = $business_partner;
                $voucher->cost_center = $voucher_details[0]['cost_center_type'] ?? 1;
                $voucher->cost_center_name = $voucher_details[0]['cost_center'] ?? 'Main Outlet';
                $voucher->payment_mode = $payment_mode;
                $voucher->business_id = $business_id_for_voucher;
                $voucher->save();

                //update or insert voucher details depending on whether id_account_voucher_detail is present
                $total_debit = 0;
                $total_credit = 0;
                foreach($voucher_details as $detail){
                    $total_debit += floatval($detail['debit']);
                    $total_credit += floatval($detail['credit']);
                }
                if(round($total_debit, 2) != round($total_credit, 2)){
                    DB::rollBack();
                    return response()->json([
                        "message_type" => "error",
                        "message" => "Total Debit and Credit amounts are not balanced.",
                        "message_btn" => "btn btn-danger",
                        "success" => false,
                    ]);
                }
                foreach($voucher_details as $detail){
                    if(isset($detail['id_account_voucher_detail']) && $detail['id_account_voucher_detail'] > 0){
                        //update existing detail
                        $voucher_detail = AccountVoucherDetail::find($detail['id_account_voucher_detail']);
                        if($voucher_detail){
                            $voucher_detail->account_head_id = $detail['account_head_id'];
                            $voucher_detail->debit = $detail['debit'];
                            $voucher_detail->credit = $detail['credit'];
                            $voucher_detail->cost_center = $detail['cost_center']  ?? 1;
                            $voucher_detail->cost_center_type = $detail['cost_center_type'] ?? 'Main Outlet';
                            $voucher_detail->detail_remarks = $detail['detail_remarks'];
                            $voucher_detail->instrument_number = $instrument_number = $detail['instrument_number'] ?? '';
                            $voucher_detail->payment_mode = $payment_mode ?? '';
                            $voucher_detail->save();
                        }
                    } else {
                        //new detail
                        $voucher_detail = new AccountVoucherDetail();
                        $voucher_detail->account_voucher_id = $voucher->id_account_vouchers;
                        $voucher_detail->account_head_id = $detail['account_head_id'];
                        $voucher_detail->debit = $detail['debit'];
                        $voucher_detail->credit = $detail['credit'];
                        $voucher_detail->cost_center = $detail['cost_center']  ?? 1;
                        $voucher_detail->cost_center_type = $detail['cost_center_type'] ?? 'Main Outlet';
                        $voucher_detail->detail_remarks = $detail['detail_remarks'];
                        $voucher_detail->instrument_number = $instrument_number = $detail['instrument_number'] ?? '';
                        $voucher_detail->payment_mode = $payment_mode ?? '';
                        $voucher_detail->save();
                    }
                }

                DB::commit();
                return response()->json([
                    "message_type" => "success",
                    "message" => "Voucher updated successfully. ID: " . $voucher->id_account_vouchers,
                    "message_btn" => "btn btn-success",
                    "success" => true,
                ]);
            }
            

        }catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                "message_type" => "error",
                "message" => $e->getMessage(),
                "message_btn" => "btn btn-danger",
                "success" => false,
            ], 500);
        }
    }

    public function open_profit_and_loss(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                 $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                 $business_id = Business::where('ho', 'Yes')->value('id_business'); 
                 $businesses = Business::where('id_business', session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', session("business_id"))->get(); 
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        return view('accounts.profit_and_loss')
            ->with(compact( 'businesses'));
    }

    public function profit_and_loss(Request $request)
    {
        $business_id = $request->input('business_id') ?? "";

        if ($business_id == "") {
            if (session('ho') === 'Yes') {
                $business_id = 0; // HO mode
            } else {
                $business_id = session('business_id');
            }
        } 

        
       $business_id_for_account_head_id =session('business_id');
        if(session('ho_accounts') == 'Yes'){
            $business_id_for_account_head_id = Business::where('ho', 'Yes')->value('id_business'); 
        }   
        
        $start_date = $request->input('start_date');
        $end_date   = $request->input('end_date');

        if (empty($start_date) || empty($end_date) || !strtotime($start_date) || !strtotime($end_date)) {
            $start_date = date('Y-m-01');
            $end_date   = date('Y-m-t');
        }

        $period_type = $request->input('period_type'); // month/year/day or null
        // $period_type ='month';

        $profitLoss = DB::table('account_vouchers as v')
            ->join('account_voucher_detail as d', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->join('account_heads as h', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_sub_control as sc', 'h.account_sub_control_id', '=', 'sc.id_account_sub_control')
            ->join('account_control as c', 'sc.account_control_id', '=', 'c.id_account_control')
            ->join('account_main as m', 'c.account_main_id', '=', 'm.id_account_main')
            ->where('v.voucher_status', 'Active')
            ->where('h.account_head_status', 'Active')
            ->where('h.business_id', $business_id_for_account_head_id)
            // ->whereBetween('v.voucher_date', [$start_date, $end_date])
            ->whereRaw("DATE(v.voucher_date) BETWEEN ? AND ?", [$start_date, $end_date]) 
            ->whereIn('m.account_main', ['Revenue','Expenses']);
        $business_name = "All Branches";
        if ($business_id != 0) {
            $profitLoss->where('v.business_id', $business_id);
            $business_name = Business::where('id_business', $business_id)->value('business_name');
        }

        // Period select
        $selects = [
            'm.account_main',
            'm.id_account_main',
            'c.id_account_control',
            'c.account_control',
            'sc.id_account_sub_control',
            'sc.account_sub_control',
            'h.account_head_number',
            'h.account_head',
            'h.id_account_heads',
            DB::raw('ROUND(SUM(d.debit), 2) as total_debit'),
            DB::raw('ROUND(SUM(d.credit), 2) as total_credit'),
            DB::raw('ROUND(
                CASE 
                    WHEN m.account_main = "Revenue" THEN SUM(d.credit) - SUM(d.debit)
                    WHEN m.account_main = "Expenses" THEN SUM(d.debit) - SUM(d.credit)
                    ELSE 0
                END, 2) as net_amount')
        ];

        $periodSelect = null;
        if ($period_type) {
            switch($period_type){
                case 'year':
                    $periodSelect = DB::raw('YEAR(v.voucher_date) as period');
                    break;
                case 'day':
                    $periodSelect = DB::raw('DATE(v.voucher_date) as period');
                    break;
                case 'month': 
                $periodSelect = DB::raw('DATE_FORMAT(v.voucher_date, "%Y-%m") as period'); // month 
                break;   
                default:
                    $periodSelect = DB::raw('DATE_FORMAT(v.voucher_date, "%Y-%m") as period'); // month default
                    break;
            }
            $selects[] = $periodSelect;
        }

        $profitLoss->select($selects);

        // Group by
        $groupBy = [
            'm.id_account_main',
            'c.id_account_control',
            'sc.id_account_sub_control',
            'h.id_account_heads'
        ];

        if ($periodSelect) $groupBy[] = 'period';

        $profitLoss->groupBy($groupBy)
                ->orderBy('m.id_account_main')
                ->orderBy('c.id_account_control')
                ->orderBy('sc.id_account_sub_control')
                ->orderBy('h.id_account_heads');

        if ($periodSelect) $profitLoss->orderBy('period');

        $profit_loss = $profitLoss->get();
        // Build hierarchy
        $structuredData = [];
        $all_periods = [];

        foreach ($profit_loss as $item) {

            $main = $item->account_control === 'Cost of Goods' || $item->account_sub_control === 'Cost of Goods'
                ? 'Cost of Goods'
                : $item->account_main;

            $controlKey = $item->id_account_control.'|'.$item->account_control;
            $subKey = $item->id_account_sub_control.'|'.$item->account_sub_control;

            if (!isset($structuredData[$main])) {
                $structuredData[$main] = [
                    'name' => $main,
                    'total' => 0,
                    'controls' => []
                ];
            }

            if (!isset($structuredData[$main]['controls'][$controlKey])) {
                $structuredData[$main]['controls'][$controlKey] = [
                    'id' => $item->id_account_control,
                    'name' => $item->account_control,
                    'total' => 0,
                    'sub_controls' => []
                ];
            }

            if (!isset($structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey])) {
                $structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey] = [
                    'id' => $item->id_account_sub_control,
                    'name' => $item->account_sub_control,
                    'total' => 0,
                    'heads' => []
                ];
            }

            // If period_type not set, just totals
            if (!$period_type) {
                $structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey]['heads'][] = [
                    'id' => $item->id_account_heads,
                    'account_head_number' => $item->account_head_number,
                    'account_head' => $item->account_head,
                    'net_amount' => $item->net_amount
                ];
            } else {
                // Period-wise totals
                $period = $item->period;
                $all_periods[$period] = $period;

                if (!isset($structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey]['heads'][$item->id_account_heads])) {
                    $structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey]['heads'][$item->id_account_heads] = [
                        'id' => $item->id_account_heads,
                        'account_head_number' => $item->account_head_number,
                        'account_head' => $item->account_head,
                        'period_totals' => []
                    ];
                }

                $structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey]['heads'][$item->id_account_heads]['period_totals'][$period] = $item->net_amount;
            }

            // Totals
            $structuredData[$main]['controls'][$controlKey]['sub_controls'][$subKey]['total'] += $item->net_amount;
            $structuredData[$main]['controls'][$controlKey]['total'] += $item->net_amount;
            $structuredData[$main]['total'] += $item->net_amount;
        }

        ksort($all_periods);

        // Fixed ordering: Revenue > COGS > Expenses
        $orderedData = [];
        if(isset($structuredData['Revenue'])) $orderedData['Revenue'] = $structuredData['Revenue'];
        if(isset($structuredData['Cost of Goods'])) $orderedData['Cost of Goods'] = $structuredData['Cost of Goods'];
        foreach($structuredData as $name => $data){
            if($name !== 'Revenue' && $name !== 'Cost of Goods'){
                $orderedData[$name] = $data;
            }
        }
        return response()->json([
            'structured_data' => $orderedData,
            'all_periods' => $period_type ? array_keys($all_periods) : [], // only periods if type set
            'start_date' => $start_date,
            'end_date' => $end_date,
            'business_name' => $business_name,
            'period_type' => $period_type,
        ]);
    }

    public function open_trial_balance(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                //redirect or return with error
                // return redirect()->back()->withErrors("Your account is handled by the Head Office.")->withInput();
                // return "Your account is handled by the Head Office.";
            } else {
                $business_id = session('business_id');
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        $business_id = $request->input('business_id');

        if(session('ho') == 'Yes') {
           // $business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
            $business_id = Business::where('ho', 'Yes')->value('id_business'); //get any HO business id
            //get all businesses
            $businesses = Business::where('ho_accounts', "Yes")->orWhere('ho','Yes')->get();
        } else if(session('ho_accounts') == 'Yes' ){
            $business_id = Business::where('ho', 'Yes')->value('id_business');
            $businesses = Business::where('id_business',session('business_id'))->get();
        } else {
            $business_id = session('business_id');
            //get all businesses
            $businesses = Business::where('id_business', $business_id)->get();
        }

        //get all account heads for the business         
        $account_heads = AccountHeads::select('business_name', 'id_account_heads', 'account_main', 'account_head', 'account_head_number')
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
        ->join('business', 'account_heads.business_id', '=', 'business.id_business')
        ->where('account_heads.business_id', $business_id)
        ->where('account_head_status', 'Active')
        ->orderBy('id_account_main')
        ->orderBy('id_account_control')
        ->orderBy('id_account_sub_control')
        ->orderBy('account_head_number')
        ->get();  
        $account_sub_control = AccountSubControl::where('account_sub_control_status', 'Active')->get();
        $business_partners = DB::table('account_business_partner')->get();
        $cost_centers = DB::table('account_cost_center')->get(); 
        return view('accounts.trial_balance')->with(compact( 'businesses', 'account_heads','account_sub_control','business_partners','cost_centers'));
    }

    public function trial_balance(Request $request)
    {
        $business_id = $request->input('business_id');
        
        if($business_id == "" || $business_id == 0) {
            if(session('ho') == 'Yes'){
                $business_id = 0; 
            }else if(session('ho_accounts') == 'Yes'){
                $business_id = 0; 
            }else{
                $business_id = session('business_id');
            }
        } 

        $business_id_for_account_head =0;
        if(session('ho_accounts') == 'Yes'){
          $business_id_for_account_head = Business::where('ho','Yes')->value('id_business');
        }else{
          $business_id_for_account_head = session('business_id'); 
        }   

        $business_name = Business::where('id_business', $business_id)->value('business_name') ?? 'All Branches';

        $start_date = $request->input('start_date');
        $end_date   = $request->input('end_date');

        $account_head_id = $request->input('account_head_id') ?? 0;
        $sub_control_account_id = $request->input('sub_control_account_id') ?? 0;

	    $business_partner_id = $request->input('business_partner_id') ?? 0;
        $business_partner_type_id = $request->input('business_partner_type_id') ?? 0;
        $cost_center_type = $request->input('cost_center_type') ?? 0;
        $cost_center_name = $request->input('cost_center_name') ?? 0;

        if (empty($start_date) || empty($end_date) || !strtotime($start_date) || !strtotime($end_date)) {
            $start_date = date('Y-m-01');
            $end_date   = date('Y-m-t');
        }

        $opening_balance = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->whereDate('v.voucher_date', '<', $start_date)
            ->where('h.business_id',$business_id_for_account_head)
            ->where('h.account_head_status', 'Active')
            ->select(
                'm.id_account_main',
                'm.account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.id_account_heads as account_head_number',
                'h.account_head',                
                DB::raw('ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2) as balance')
            );
            // Apply business filter
            if ($business_id != 0) {
                $opening_balance->where('v.business_id', $business_id);
            } else {
                $opening_balance->whereIn('v.business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('ho_accounts', 'Yes')->orwhere('ho','Yes');
                });
            } 

            if (!empty($account_head_id) &&  $account_head_id > 0) {
                $opening_balance->where('h.id_account_heads', $account_head_id);
            } else if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
                // Apply account control filter
                $opening_balance->where('s.id_account_sub_control', $sub_control_account_id);
            } 


            // Apply business partner filter
            if (!empty($business_partner_id) && $business_partner_id > 0) {
                $opening_balance->where('v.business_partner_id', $business_partner_id);
                $opening_balance->where('v.business_partner', $business_partner_type_id); 
            } 

            if(!empty($cost_center_type) && $cost_center_type > 0  ){
                if(!empty($cost_center_name)){
                    $opening_balance->where('d.cost_center', $cost_center_name);            
                    $opening_balance->where('d.cost_center_type', $cost_center_type);           
                }else{
                    $opening_balance->where('d.cost_center_type', $cost_center_type);           
                }
            }

            $opening_balance = $opening_balance->groupBy(
                'm.id_account_main', 'm.account_main',
                'c.id_account_control', 'c.account_control',
                's.id_account_sub_control', 's.account_sub_control',
                'h.id_account_heads'
            )
            ->orderBy('m.account_main')
            ->orderBy('c.account_control')
            ->orderBy('s.account_sub_control')
            ->orderBy('h.id_account_heads')
            ->get();

        $current_balance = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            // ->whereBetween('v.voucher_date', [$start_date, $end_date])        
            ->whereRaw("DATE(v.voucher_date) BETWEEN ? AND ?", [$start_date, $end_date]) 
            ->where('h.business_id',$business_id_for_account_head)   
            ->select(
                'm.id_account_main',
                'm.account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.id_account_heads as account_head_number',
                'h.account_head',
                DB::raw('ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2)  as balance')
            );
            // Apply business filter
            if ($business_id != 0) {
                $current_balance->where('v.business_id', $business_id);
            } else {                
                $current_balance->whereIn('v.business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('ho_accounts', 'Yes');
                });
            } 

            if (!empty($account_head_id) &&  $account_head_id > 0) {
                $current_balance->where('h.id_account_heads', $account_head_id);
            } else if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
                // Apply account control filter
                $current_balance->where('s.id_account_sub_control', $sub_control_account_id);
            } 


            // Apply business partner filter
            if (!empty($business_partner_id) && $business_partner_id > 0) {
                $current_balance->where('v.business_partner_id', $business_partner_id);
                $current_balance->where('v.business_partner', $business_partner_type_id); 
            } 

            if(!empty($cost_center_type) && $cost_center_type > 0  ){
                if(!empty($cost_center_name)){
                    $current_balance->where('d.cost_center', $cost_center_name);            
                    $current_balance->where('d.cost_center_type', $cost_center_type);           
                }else{
                    $current_balance->where('d.cost_center_type', $cost_center_type);           
                }
            }

            $current_balance = $current_balance->groupBy(
                'm.id_account_main', 'm.account_main',
                'c.id_account_control', 'c.account_control',
                's.id_account_sub_control', 's.account_sub_control',
                'h.id_account_heads'
            )
            ->orderBy('m.account_main')
            ->orderBy('c.account_control')
            ->orderBy('s.account_sub_control')
            ->orderBy('h.id_account_heads')
            ->get();

        return response()->json([
            'opening_balance' => $opening_balance,
            'current_balance' => $current_balance,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'business_name' => $business_name
        ]);
    }

    public function open_trial_movement(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                //redirect or return with error
                // return redirect()->back()->withErrors("Your account is handled by the Head Office.")->withInput();
                // return "Your account is handled by the Head Office.";
            } else {
                $business_id = session('business_id');
            }           
        } else {
            return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
        }

        $business_id = $request->input('business_id');
        
        if(session('ho') == 'Yes') {
            $business_id = Business::where('ho', 'Yes')->value('id_business'); 
            $businesses = Business::where('ho_accounts', "Yes")->orWhere('ho','Yes')->get();
        } else if(session('ho_accounts') == 'Yes' ){
            $business_id = Business::where('ho', 'Yes')->value('id_business');
            $businesses = Business::where('id_business',session('business_id'))->get();
        } else {
            $business_id = session('business_id');
            //get all businesses
            $businesses = Business::where('id_business', $business_id)->get();
        }

        //get all account heads for the business         
        $account_heads = AccountHeads::select('business_name', 'id_account_heads', 'account_main', 'account_head', 'account_head_number')
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
        ->join('business', 'account_heads.business_id', '=', 'business.id_business')
        ->where('account_heads.business_id', $business_id)
        ->where('account_head_status', 'Active')
        ->orderBy('id_account_main')
        ->orderBy('id_account_control')
        ->orderBy('id_account_sub_control')
        ->orderBy('account_head_number')
        ->get();
        $account_sub_control = AccountSubControl::where('account_sub_control_status', 'Active')->get();
        $business_partners = DB::table('account_business_partner')->get();
        $cost_centers = DB::table('account_cost_center')->get(); 
        return view('accounts.trial_movement')->with(compact( 'businesses', 'account_heads','account_sub_control','business_partners','cost_centers'));
    }

    public function trial_movement(Request $request)
    {
        $business_id = $request->input('business_id');

        if($business_id == "" || $business_id == 0) {
            if (session('ho_accounts') == 'Yes' && session('ho') == 'Yes') {
                $business_id = 0; // HO mode
            } else{
                $business_id = session('business_id');
            }
        }

        $business_id_for_account_head =0;
        if(session('ho_accounts') == 'Yes'){
          $business_id_for_account_head = Business::where('ho','Yes')->value('id_business');
        }else{
          $business_id_for_account_head = session('business_id'); 
        }   

        $business_name = Business::where('id_business', $business_id)->value('business_name') ?? 'All Branches';
        
        $start_date = $request->input('start_date');
        $end_date   = $request->input('end_date');

        if (empty($start_date) || empty($end_date) || !strtotime($start_date) || !strtotime($end_date)) {
            $start_date = date('Y-m-01');
            $end_date   = date('Y-m-t');
        } 


        $account_head_id = $request->input('account_head_id') ?? 0;
        $sub_control_account_id = $request->input('sub_control_account_id') ?? 0;

	    $business_partner_id = $request->input('business_partner_id') ?? 0;
        $business_partner_type_id = $request->input('business_partner_type_id') ?? 0;
        $cost_center_type = $request->input('cost_center_type') ?? 0;
        $cost_center_name = $request->input('cost_center_name') ?? 0;

        $opening_balance = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->where('h.account_head_status', 'Active')
            ->whereDate('v.voucher_date', '<', $start_date)
            ->where('h.business_id',$business_id_for_account_head)
            ->select(
                'm.id_account_main',
                'm.account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.id_account_heads as account_head_number',
                'h.account_head',                
                DB::raw('ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2) as balance'),
                DB::raw('ROUND(IFNULL(SUM(d.debit), 0), 2) as debit'),
                DB::raw('ROUND(IFNULL(SUM(d.credit), 0), 2) as credit')
            );
            // Apply business filter
            if ($business_id != 0) {
                $opening_balance->where('v.business_id', $business_id);
            } else {
                $opening_balance->whereIn('v.business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('ho_accounts', 'Yes')->orWhere('ho',"Yes");
                });
            } 

            if (!empty($account_head_id) &&  $account_head_id > 0) {
                $opening_balance->where('h.id_account_heads', $account_head_id);
            } else if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
                // Apply account control filter
                $opening_balance->where('s.id_account_sub_control', $sub_control_account_id);
            } 


            // Apply business partner filter
            if (!empty($business_partner_id) && $business_partner_id > 0) {
                $opening_balance->where('v.business_partner_id', $business_partner_id);
                $opening_balance->where('v.business_partner', $business_partner_type_id); 
            } 

            if(!empty($cost_center_type) && $cost_center_type > 0  ){
                if(!empty($cost_center_name)){
                    $opening_balance->where('d.cost_center', $cost_center_name);            
                    $opening_balance->where('d.cost_center_type', $cost_center_type);           
                }else{
                    $opening_balance->where('d.cost_center_type', $cost_center_type);           
                }
            }

            $opening_balance = $opening_balance->groupBy(
                'm.id_account_main', 'm.account_main',
                'c.id_account_control', 'c.account_control',
                's.id_account_sub_control', 's.account_sub_control',
                'h.id_account_heads'
            )
            ->orderBy('m.account_main')
            ->orderBy('c.account_control')
            ->orderBy('s.account_sub_control')
            ->orderBy('h.id_account_heads')
            ->get();

        $current_balance = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->where('h.account_head_status', 'Active')
            // ->whereBetween('v.voucher_date', [$start_date, $end_date])  
            ->whereRaw("DATE(v.voucher_date) BETWEEN ? AND ?", [$start_date, $end_date])
            ->where('h.business_id',$business_id_for_account_head)          
            ->select(
                'm.id_account_main',
                'm.account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.id_account_heads as account_head_number',
                'h.account_head',
                DB::raw('ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2)  as balance'),
                 DB::raw('ROUND(IFNULL(SUM(d.debit), 0), 2) as debit'),
                DB::raw('ROUND(IFNULL(SUM(d.credit), 0), 2) as credit')
            );
            // Apply business filter
            if ($business_id != 0) {
                $current_balance->where('v.business_id', $business_id);
            } else {                
                $current_balance->whereIn('v.business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('ho_accounts', 'Yes')->orWhere('ho',"Yes");
                });
            } 

            if (!empty($account_head_id) &&  $account_head_id > 0) {
                $current_balance->where('h.id_account_heads', $account_head_id);
            } else if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
                // Apply account control filter
                $current_balance->where('s.id_account_sub_control', $sub_control_account_id);
            } 


            // Apply business partner filter
            if (!empty($business_partner_id) && $business_partner_id > 0) {
                $current_balance->where('v.business_partner_id', $business_partner_id);
                $current_balance->where('v.business_partner', $business_partner_type_id); 
            } 

            if(!empty($cost_center_type) && $cost_center_type > 0  ){
                if(!empty($cost_center_name)){
                    $current_balance->where('d.cost_center', $cost_center_name);            
                    $current_balance->where('d.cost_center_type', $cost_center_type);           
                }else{
                    $current_balance->where('d.cost_center_type', $cost_center_type);           
                }
            }

            $current_balance = $current_balance->groupBy(
                'm.id_account_main', 'm.account_main',
                'c.id_account_control', 'c.account_control',
                's.id_account_sub_control', 's.account_sub_control',
                'h.id_account_heads'
            )
            ->orderBy('m.account_main')
            ->orderBy('c.account_control')
            ->orderBy('s.account_sub_control')
            ->orderBy('h.id_account_heads')
            ->get();

        return response()->json([
            'opening_balance' => $opening_balance,
            'current_balance' => $current_balance,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'business_name' => $business_name
        ]);
    }

    public function open_balance_sheet(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get();
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                $business_id = Business::where('ho', 'Yes')->value('id_business'); 
                $businesses = Business::where('id_business', session("business_id"))->get();
            } else {
                $business_id = session('business_id');
                 $businesses = Business::where('id_business', $business_id)->get();
            }           
        } else {
            return "You do not have the permission to access this resource.";
        }

     
        $as_on_date = $request->input('as_on_date');

        $account_heads_assest = AccountHeads::select(
        'account_main.account_main',
        'account_main.id_account_main',
        'account_control.id_account_control',
        'account_control.account_control',
        'account_sub_control.id_account_sub_control',
        'account_sub_control.account_sub_control',
        'account_heads.id_account_heads',
        'account_heads.account_head_number',
        'account_heads.account_head',
        DB::raw('0 as total_debit'),
        DB::raw('0 as total_credit'),
        DB::raw('0 as closing_balance'))
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
        ->join('business', 'account_heads.business_id', '=', 'business.id_business')
        ->whereIn('account_main.account_main', ['Assets'])
        ->where('account_heads.business_id', $business_id)
        ->where('account_head_status', 'Active')
        ->groupBy(
                'account_main.id_account_main',
                'account_main.account_main',
                'account_control.id_account_control',
                'account_control.account_control',
                'account_sub_control.id_account_sub_control',
                'account_sub_control.account_sub_control',
                'account_heads.id_account_heads'
        )
        ->orderBy('id_account_main')
        ->orderBy('id_account_control')
        ->orderBy('id_account_sub_control')
        ->orderBy('id_account_heads')
        ->get();  
        $account_heads_liabilities= AccountHeads::select(
        'account_main.account_main',
        'account_main.id_account_main',
        'account_control.id_account_control',
        'account_control.account_control',
        'account_sub_control.id_account_sub_control',
        'account_sub_control.account_sub_control',
        'account_heads.id_account_heads',
        'account_heads.account_head_number',
        'account_heads.account_head',
        DB::raw('0.00 as total_debit'),
        DB::raw('0.00 as total_credit'),
        DB::raw('0.00 as closing_balance'))
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
        ->join('business', 'account_heads.business_id', '=', 'business.id_business')
        ->whereIn('account_main.account_main', ['Liabilities','Equity'])
        ->where('account_heads.business_id', $business_id)
        ->where('account_head_status', 'Active')
        ->groupBy(
                'account_main.id_account_main',
                'account_main.account_main',
                'account_control.id_account_control',
                'account_control.account_control',
                'account_sub_control.id_account_sub_control',
                'account_sub_control.account_sub_control',
                'account_heads.id_account_heads'
        )
        ->orderBy('id_account_main')
        ->orderBy('id_account_control')
        ->orderBy('id_account_sub_control')
        ->orderBy('id_account_heads')
        ->get();  

         $account_heads_assest = $this->buildHierarchicalStructure($account_heads_assest);
        $account_heads_liabilities = $this->buildHierarchicalStructure($account_heads_liabilities);
        $business_name = Business::where('id_business', session('business_id'))->value('business_name');
        return view('accounts.balance_sheet')->with(compact('businesses', 'account_heads_assest','account_heads_liabilities' ,'as_on_date', 'business_name'));
    }

    public function balance_sheet(Request $request)
    {
        $business_id = $request->input('business_id');

        if($business_id == "" || $business_id == 0) {
            if (session('ho') == 'Yes') { 
                $business_id=0;
            } else if(session('ho_accounts') == 'Yes'){
                 $business_id=0;
            }  else{
                $business_id = session('business_id');
            }
        }      
        
        if($business_id != 0) {
            $business_name = Business::where('id_business', $business_id)->value('business_name');
        } else {
            $business_name = "All Branches";
        }

        $business_id_for_account_head_id =session('business_id');
        if(session('ho_accounts') == 'Yes'){
            $business_id_for_account_head_id = Business::where('ho', 'Yes')->value('id_business'); 
        }   
    
        $as_on_date = $request->input('as_on_date');

        if (empty($as_on_date) || !strtotime($as_on_date)) {
            $as_on_date = date('Y-m-t');
        }

        // Liabilities and Equity Query
        $balanceSheetLiabities = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->whereIn('m.account_main', ['Liabilities', 'Equity'])
            ->whereDate('v.voucher_date', '<=', $as_on_date)
            ->where('h.business_id', '=', $business_id_for_account_head_id)
            ->where('h.account_head_status', 'Active')
            ->select(
                'm.account_main',
                'm.id_account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.account_head_number',
                'h.account_head',
                DB::raw('IFNULL(SUM(d.debit), 0) as total_debit'),
                DB::raw('IFNULL(SUM(d.credit), 0) as total_credit'),
                DB::raw('IFNULL((SUM(d.credit) - SUM(d.debit)), 0) as closing_balance')
            );

        // Apply business filter
        if ($business_id != 0) {
            $balanceSheetLiabities->where('v.business_id', $business_id);
        } else {
            $balanceSheetLiabities->whereIn('v.business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes');
            });
        }
        
        // Grouping
        $balanceSheetLiabities->groupBy(
        'm.id_account_main', 'm.account_main',
        'c.id_account_control', 'c.account_control', 
        's.id_account_sub_control', 's.account_sub_control',
        'h.id_account_heads' 
        );
        
       
        $balanceSheetLiabities
            ->orderBy('m.id_account_main')
            ->orderBy('c.id_account_control')
            ->orderBy('s.id_account_sub_control')
            ->orderBy('h.id_account_heads');
        
        $balance_sheet_liabilities_raw = $balanceSheetLiabities->get();

      
        $balanceSheetAssets = DB::table('account_main as m')
            ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->whereIn('m.account_main', ['Assets'])
            ->whereDate('v.voucher_date', '<=', $as_on_date)
            ->where('h.business_id', '=', $business_id_for_account_head_id)
            ->where('h.account_head_status', 'Active')
            ->select(
                'm.account_main',
                'm.id_account_main',
                'c.id_account_control',
                'c.account_control',
                's.id_account_sub_control',
                's.account_sub_control',
                'h.id_account_heads',
                'h.account_head_number',
                'h.account_head',
                DB::raw('IFNULL(SUM(d.debit), 0) as total_debit'),
                DB::raw('IFNULL(SUM(d.credit), 0) as total_credit'),
                DB::raw('IFNULL((SUM(d.debit) - SUM(d.credit)), 0) as closing_balance')
            );
        
        if ($business_id != 0) {
            $balanceSheetAssets->where('v.business_id', $business_id);
        } else {
            $balanceSheetAssets->whereIn('v.business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes');
            });
        }   
        
      
        $balanceSheetAssets->groupBy(
                'm.id_account_main', 'm.account_main',
                'c.id_account_control', 'c.account_control', 
                's.id_account_sub_control', 's.account_sub_control',
                'h.id_account_heads' 
        );
        
       
        $balanceSheetAssets
            ->orderBy('m.id_account_main')
            ->orderBy('c.id_account_control')
            ->orderBy('s.id_account_sub_control')
            ->orderBy('h.id_account_heads');
        
        $balance_sheet_assets_raw = $balanceSheetAssets->get();  
        $profitAndLossAmount = $this->getProfitAndLossAmount($as_on_date, $business_id);
 
        $balance_sheet_liabilities = $this->buildHierarchicalStructure($balance_sheet_liabilities_raw);
        $balance_sheet_assets = $this->buildHierarchicalStructure($balance_sheet_assets_raw);

        return response()->json([
            'balance_sheet_liabilities' => $balance_sheet_liabilities,
            'balance_sheet_assets' => $balance_sheet_assets,
            'as_on_date' => date('d-m-Y', strtotime($as_on_date)),
            'business_name' => $business_name,
            'profitAndLossAmount' => $profitAndLossAmount,
        ]);
    }

    public function open_general_ledger(Request $request)
    {
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                // return "Your account is handled by the Head Office.";
            } else {
                $business_id = session('business_id');
            }           
        } else { 
            return "You do not have the permission to access this resource.";
        }

        $business_id = $request->input('business_id');
        if($business_id == "" || $business_id == 0) {
            if(session('ho') == 'Yes') {          
                $business_id = 0; 
                $businesses = Business::where('ho_accounts', "Yes")->orWhere('ho','Yes')->get();
            }else if(session('ho_accounts') == 'Yes'){
                $business_id = 0; 
                $businesses = Business::where('id_business',session('business_id'))->get();
            }  else {
                $business_id = session('business_id');         
                $businesses = Business::where('id_business', $business_id)->get();
            }
        }

        $account_heads = AccountHeads::select('*')->where('account_head_status', 'Active');
        if($business_id != 0) {
            $account_heads = $account_heads->where('business_id', $business_id);
        } else {
            $account_heads = $account_heads->whereIn('business_id', function ($query) {
                $query->select('id_business')->from('business')->Where('ho', 'Yes');
            });
        }
        $account_heads = $account_heads->get();
        $account_sub_control = AccountSubControl::where('account_sub_control_status', 'Active')->get();
        $business_partners = DB::table('account_business_partner')->get();
        $cost_centers = DB::table('account_cost_center')->get();    
        
        return view('accounts.general_ledger', compact('businesses', 'business_id', 'account_heads', 'account_sub_control','business_partners','cost_centers'));
    }
    
    public function general_ledger(Request $request)
    {
        $business_id = $request->input('business_id');

        if($business_id == "" || $business_id == 0) {
            if (session('ho') == 'Yes') {
                $business_id = 0; // HO mode
            } else{
                $business_id = session('business_id');
            }
        }

        $account_head_id = $request->input('account_head_id') ?? 0;
        $sub_control_account_id = $request->input('sub_control_account_id') ?? 0;


         $business_id_for_account_head_id =session('business_id');
        if(session('ho_accounts') == 'Yes'){
            $business_id_for_account_head_id = Business::where('ho', 'Yes')->value('id_business'); 
        }   
    


        if(($account_head_id == "" || $account_head_id == 0) &&  ($sub_control_account_id == "" || $sub_control_account_id == 0)) {
            return response()->json([
                'opening_balance' => '',
                'ledger_entries' => '',
            ]);
        }

        $start_date = $request->input('start_date');
        $end_date   = $request->input('end_date');

        if (empty($start_date) || empty($end_date) || !strtotime($start_date) || !strtotime($end_date)) {
            $start_date = date('Y-m-01');
            $end_date   = date('Y-m-t');
        }

        $business_partner_id = $request->input('business_partner_id') ?? 0;
        $business_partner_type_id = $request->input('business_partner_type_id') ?? 0;
        $cost_center_type = $request->input('cost_center_type') ?? 0;
        $cost_center_name = $request->input('cost_center_name') ?? 0;
        $groupBy = $request->input('groupByFilter');

        // Opening balance
        $opening_balance_query = DB::table('account_heads as h')
            ->join('account_sub_control as s', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_control as c', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_main as m', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->where('v.voucher_status', 'Active')
            ->where('h.account_head_status', 'Active')
            ->where('h.business_id', $business_id_for_account_head_id)
            ->whereDate('v.voucher_date', '<', $start_date);

        // Apply business filter
        if ($business_id != 0) {
            $opening_balance_query->where('v.business_id', $business_id);
        } else {
            $opening_balance_query->whereIn('v.business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes');
            });
        }

        // Apply account head filter
        if (!empty($account_head_id) &&  $account_head_id > 0) {
            $opening_balance_query->where('h.id_account_heads', $account_head_id);
        } else if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
            // Apply account control filter
            $opening_balance_query->where('s.id_account_sub_control', $sub_control_account_id);
        }

        // Apply business partner filter
        if (!empty($business_partner_id) && $business_partner_id > 0) {
            $opening_balance_query->where('v.business_partner_id', $business_partner_id);
            $opening_balance_query->where('v.business_partner', $business_partner_type_id); 
        } 

        if(!empty($cost_center_type) && $cost_center_type > 0  ){
            if(!empty($cost_center_name)){
                $opening_balance_query->where('d.cost_center', $cost_center_name);            
                $opening_balance_query->where('d.cost_center_type', $cost_center_type);           
            }else{
                $opening_balance_query->where('d.cost_center_type', $cost_center_type);           
            }
        }

       $opening_balance_query->select(
                'h.id_account_heads',
                'h.account_head_number',
                'h.account_head',
                DB::raw('CASE WHEN m.account_main = "Assets" OR m.account_main = "Expenses" THEN ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2) '. 
                ' WHEN m.account_main = "Liabilities" OR m.account_main = "Equity" OR m.account_main = "Revenue" THEN ROUND(IFNULL(SUM(d.credit) - SUM(d.debit), 0), 2) '.
                'END as opening_balance')                
            );
        if (!empty($sub_control_account_id) && $sub_control_account_id > 0) {
            $opening_balance_query->groupBy(
                'm.id_account_main',
                'c.id_account_control',
                's.id_account_sub_control'
            );

        } else {
            $opening_balance_query->groupBy(
                'm.id_account_main',
                'c.id_account_control',
                's.id_account_sub_control',
                'h.id_account_heads'
            );
        }

        $opening_balance = $opening_balance_query
            ->orderBy('h.id_account_heads')
            ->first();

        if (!$opening_balance) {
            $opening_balance = (object)[
                'id_account_heads' => null,
                'account_head_number' => null,
                'account_head' => null,
                'debit' => 0,
                'credit' => 0,
                'opening_balance' => 0,
            ];
            $opening_balance->debit = 0;
            $opening_balance->credit = 0;
        }
        
        // Ledger entries
        $ledger_entries_query = DB::table('account_heads as h')
            ->join('account_sub_control as s', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
            ->join('account_control as c', 's.account_control_id', '=', 'c.id_account_control')
            ->join('account_main as m', 'c.account_main_id', '=', 'm.id_account_main')
            ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->join('account_voucher_type as vt', 'v.voucher_type', '=', 'vt.id_account_voucher_type')
            ->leftJoin('business as b', 'v.business_id', '=', 'b.id_business')
            ->where('v.voucher_status', 'Active')
            ->where('h.business_id', $business_id_for_account_head_id)
            ->where('h.account_head_status', 'Active')
            ->whereRaw("DATE(v.voucher_date) BETWEEN ? AND ?", [$start_date, $end_date]);
        // Apply business filter
        if ($business_id != 0) {
            $ledger_entries_query->where('v.business_id', $business_id);
        } else {
            $ledger_entries_query->whereIn('v.business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes');
            });
        }

        // Apply account head filter
        if (!empty($account_head_id) && $account_head_id > 0) {
            $ledger_entries_query->where('h.id_account_heads', $account_head_id);
        } else if (!empty($sub_control_account_id)  && $sub_control_account_id > 0) {
            // Apply account control filter
            $ledger_entries_query->where('s.id_account_sub_control', $sub_control_account_id);
        }

        if (!empty($business_partner_id) && $business_partner_id > 0) {
            $ledger_entries_query->where('v.business_partner_id', $business_partner_id);
            $ledger_entries_query->where('v.business_partner', $business_partner_type_id);
        } 

        if(!empty($cost_center_type) && $cost_center_type > 0  ){
            if(!empty($cost_center_name)){
                $ledger_entries_query->where('d.cost_center', $cost_center_name);            
                $ledger_entries_query->where('d.cost_center_type', $cost_center_type);           
            }else{
                $ledger_entries_query->where('d.cost_center_type', $cost_center_type);           
            }
        } 

        $id_account_vouchers = 'v.id_account_vouchers as id_account_vouchers';
        $voucherDateSelect = 'DATE_FORMAT(v.voucher_date, "%d-%m-%Y") as voucher_date';
        $voucherCreatedOnSelect = 'DATE_FORMAT(v.created_on, "%d-%m-%Y %H:%i") as created_on';
        $voucherDescription = 'GROUP_CONCAT(DISTINCT CONCAT(v.description, " || ", b.business_name) SEPARATOR ", ") as description';
        $businessPartner = 'GROUP_CONCAT(DISTINCT v.business_partner_name SEPARATOR ", ") as business_partner_name';
        $payment_mode = 'GROUP_CONCAT(DISTINCT v.payment_mode SEPARATOR ", ") as payment_mode';
        $instrument_number = 'GROUP_CONCAT(DISTINCT d.instrument_number SEPARATOR ", ") as instrument_number';
        $created_by = 'GROUP_CONCAT(DISTINCT v.created_by SEPARATOR ", ") as created_by'; 
        $account_voucher_type = 'GROUP_CONCAT(DISTINCT vt.account_voucher_type SEPARATOR ", ") as account_voucher_type'; 

        $cost_center = 'GROUP_CONCAT(DISTINCT d.cost_center SEPARATOR ", ") as cost_center '; 
        $business_name = 'GROUP_CONCAT(DISTINCT b.business_name  SEPARATOR ", ") as business_name '; 

        if (!empty($groupBy)) {
            switch ($groupBy) {
                case 'id_account_vouchers':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('v.id_account_vouchers')->orderBy('v.business_id')->orderBy('v.id_account_vouchers');
                    break;
                case 'instrument_number':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('v.id_account_vouchers')->groupBy('d.instrument_number')->orderBy('v.business_id')->orderBy('d.instrument_number');
                    break;
                case 'account_head_id':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('h.account_head_number')->orderBy('v.business_id')->orderBy('h.account_head_number');
                    $id_account_vouchers =' "" as id_account_vouchers';
                    $voucherDescription = 'GROUP_CONCAT(DISTINCT CONCAT(h.account_head, " || ",h.account_head_number, " || ", b.business_name) SEPARATOR ", ") as description';
                    $businessPartner = ' "" as business_partner_name';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by'; 
                    $account_voucher_type = ' "" as account_voucher_type';  
                    $voucherDateSelect = ' "" as voucher_date';
                    $voucherCreatedOnSelect =' "" as created_on';
                    $cost_center = ' "" as cost_center ';
                    break;
                case 'voucher_date':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('v.voucher_date')->orderBy('v.business_id')->orderBy('v.voucher_date');

                    
                    $voucherDateSelect = 'DATE_FORMAT(v.voucher_date, "%d-%m-%Y") as voucher_date';
                    $voucherCreatedOnSelect = 'DATE_FORMAT(v.created_on, "%d-%m-%Y") as created_on';
                    $id_account_vouchers =' "" as id_account_vouchers';
                    $voucherDescription = '"" as description';
                    $businessPartner = ' "" as business_partner_name';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by';
                    $account_voucher_type = ' "" as account_voucher_type'; 
                    $cost_center = ' "" as cost_center '; 
                    break;
                case 'cost_center':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('d.cost_center')->orderBy('v.business_id')->orderBy('d.cost_center');
                    $id_account_vouchers =' "" as id_account_vouchers';
                    // $voucherDescription = 'GROUP_CONCAT(DISTINCT b.business_name SEPARATOR ", ") as description';
                    $voucherDescription = 'GROUP_CONCAT(DISTINCT CONCAT(d.cost_center, " || ", b.business_name) SEPARATOR ", ") as description';
                    $businessPartner = ' "" as business_partner_name';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by';
                    $account_voucher_type = ' "" as account_voucher_type'; 
                    $voucherDateSelect = ' "" as voucher_date';
                    $voucherCreatedOnSelect =' "" as created_on';
                    break;
                case 'business_partner_id':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('v.business_partner_id')->orderBy('v.business_id')->orderBy('v.business_partner_id'); 

                    $id_account_vouchers =' "" as id_account_vouchers';
                    $voucherDescription = 'GROUP_CONCAT(DISTINCT b.business_name SEPARATOR ", ") as description';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by';
                    $account_voucher_type = ' "" as account_voucher_type';
                    
                    $voucherDateSelect = ' "" as voucher_date';
                    $voucherCreatedOnSelect =' "" as created_on';
                    $cost_center = ' "" as cost_center '; 

                    break;
                case 'month':
                    $ledger_entries_query->groupBy('v.business_id')->groupBy(DB::raw('DATE_FORMAT(v.voucher_date, "%Y-%m")'))->orderBy('v.business_id')->orderBy('v.voucher_date');
                    $id_account_vouchers =' "" as id_account_vouchers';
                    $voucherDateSelect = 'DATE_FORMAT(v.voucher_date, "%Y-%m") as voucher_date';
                    $voucherCreatedOnSelect = 'DATE_FORMAT(v.created_on, "%Y-%m") as created_on';
                    

                    $voucherDescription = 'GROUP_CONCAT(DISTINCT b.business_name SEPARATOR ", ") as description';
                    $businessPartner = ' "" as business_partner_name';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by';
                    $account_voucher_type = ' "" as account_voucher_type'; 
                    $cost_center = ' "" as cost_center '; 
                    
                    break;
                case 'year':
                    $id_account_vouchers =' "" as id_account_vouchers';
                    $ledger_entries_query->groupBy('v.business_id')->groupBy(DB::raw('YEAR(v.voucher_date)'))->orderBy('v.business_id')->orderBy('v.voucher_date');
                    $voucherDateSelect = 'YEAR(v.voucher_date) as voucher_date';
                    $voucherCreatedOnSelect = 'YEAR(v.voucher_date) as created_on';

                    $voucherDescription = 'GROUP_CONCAT(DISTINCT b.business_name SEPARATOR ", ") as description';
                    $businessPartner = ' "" as business_partner_name';
                    $payment_mode = ' "" as payment_mode';
                    $instrument_number = ' ""  as instrument_number';
                    $created_by = ' ""  as created_by'; 
                    $account_voucher_type = ' "" as account_voucher_type';
                    $cost_center = ' "" as cost_center '; 

                    break;
                default:
                    $ledger_entries_query->groupBy('v.business_id')->groupBy('v.id_account_vouchers')->orderBy('v.voucher_date')->orderBy('v.business_id')->orderBy('v.voucher_date');
                    break;
            }
        } else {
            $ledger_entries_query->groupBy('v.id_account_vouchers')->orderBy('v.voucher_date')->orderBy('v.business_id')->orderBy('v.voucher_date');
        }
        $ledger_entries = $ledger_entries_query
        ->select(
            DB::raw($id_account_vouchers),
            DB::raw($voucherDateSelect),
            DB::raw($voucherCreatedOnSelect),
            DB::raw($account_voucher_type),
            DB::raw($voucherDescription),
            DB::raw($businessPartner),
            DB::RAW('ROUND(SUM(IFNULL(d.debit,0)), 2) as debit'),
            DB::RAW('ROUND(SUM(IFNULL(d.credit,0)), 2) as credit'),
            DB::raw('CASE WHEN m.account_main = "Assets" OR m.account_main = "Expenses" THEN ROUND(IFNULL(SUM(d.debit) - SUM(d.credit), 0), 2) '.
                     'WHEN m.account_main = "Liabilities" OR m.account_main = "Equity" OR m.account_main = "Revenue" THEN ROUND(IFNULL(SUM(d.credit) - SUM(d.debit), 0), 2) END as net_amount'),
            DB::raw('"' . session('user_role') . '" as user_role'),
            'v.auto_voucher',
            'v.voucher_status',
            DB::RAW($payment_mode),
            DB::RAW($instrument_number),
            DB::RAW($created_by),
            DB::RAW($business_name),
            DB::RAW($cost_center)
        )
        ->get();
    
        return response()->json([
            'opening_balance' => $opening_balance,            
            'data' => $ledger_entries
        ]);
    }

    public function account_voucher_print(Request $request)
    {
        $business_id = $request->input('business_id'); 

        $id = $request->input('id');

        if(session('ho_accounts') == 'Yes' && session('ho') == 'Yes') {          
            //$business_id = Business::where('ho', 'Yes')->value('id_business'); //get any HO business id
            $business_id = 0; // HO mode
           
        } else {
            $business_id = session('business_id');         
           
        }

        $account_voucher = AccountVouchers::select('*')
        ->join('business as b', 'account_vouchers.business_id', '=', 'b.id_business')
        ->join('account_voucher_type as vt', 'account_vouchers.voucher_type', '=', 'vt.id_account_voucher_type')
            ->where('id_account_vouchers', $id)
            ->where(function ($query) use ($business_id) {
                if ($business_id != 0) {
                    $query->where('account_vouchers.business_id', $business_id);
                } else {
                    $query->whereIn('account_vouchers.business_id', function ($q) {
                        $q->select('id_business')
                            ->from('business')
                            ->where('ho_accounts', 'Yes')->orWhere('ho',"Yes");
                    });
                }
            })
            ->first();

        $details = AccountVoucherDetail::select('account_voucher_detail.*', 'account_heads.account_head', 
        'account_heads.account_head_number',
        DB::raw('CONCAT(account_main.account_main_number, " - ", account_control.account_control_number, " - ", account_sub_control.account_sub_control_number, " - ", account_heads.account_head_number) as account_hierarchy')
        )
            ->join('account_heads', 'account_voucher_detail.account_head_id', '=', 'account_heads.id_account_heads')
            ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
            ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
            ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
    
            ->where('account_voucher_id', $id)
            
            ->get();

        if (!$account_voucher) {
            return redirect()->back()->withErrors(['Account Voucher not found or access denied.']);
        }

        return view('accounts.account_voucher_print', compact('account_voucher', 'details', 'business_id'));

    }

    public function cancel_voucher(Request $request)
    {
        $id = $request->input('id_account_vouchers');
        $business_id = $request->input('business_id');
        $reason = $request->input('reason'); 

        $voucher = AccountVouchers::where('id_account_vouchers', $id)->first();
        $voucherBusinessId = $voucher->business_id; 
        $voucher_date = date('Y-m-d', strtotime($voucher->voucher_date));
        if (session('user_role') != "Super User") {
            $lastClosing = DB::table('fiscal_year_closings')
                ->selectRaw("DATE(closing_date) as closing_date")
                ->where('business_id', $voucherBusinessId)
                ->where('status', 'Active')
                ->orderBy('closing_date', 'desc')
                ->first();

            if ($lastClosing && $voucher_date <= $lastClosing->closing_date) {
                return response()->json(['success' => false, 'message_type' => 'error', 'message' => "Voucher cannot be cancelled as its date falls within the last closed fiscal period ({$lastClosing->closing_date})."]);
            }
        }

        if (session('user_role') != "Super User" && session('user_role') != "Super Admin"){
            if ($voucherBusinessId != session('business_id')) {
                return response()->json(['success' => false, 'message_type' => 'error', 'message' => 'You are attempting to edit a voucher for a different branch. You do not have permission to edit this voucher.']);
            }
        }

        if (empty($reason)) {
            return response()->json(['success' => false, 'message_type' => 'error', 'message' => 'Cancellation reason is required.']);
        }

        //  $business_id = session('business_id');
        $voucher = AccountVouchers::where('id_account_vouchers', $id)
            // ->where(function ($query) use ($business_id) {
            //     if ($business_id != 0) {
            //         $query->where('business_id', $business_id);
            //     } else {
            //         $query->whereIn('business_id', function ($q) {
            //             $q->select('id_business')
            //                 ->from('business')
            //                 ->where('ho_accounts', 'Yes')->orWhere('ho',"Yes");
            //         });
            //     }
            // })
            ->first();

        if (!$voucher) {
            return response()->json(['success' => false, 'message_type' => 'error', 'message' => 'Voucher not found or access denied.']);
        }

        if ($voucher->voucher_status == 'Cancelled') {
            return response()->json(['success' => false, 'message_type' => 'error', 'message' => 'Voucher is already cancelled.']);
        }

        if ($voucher->auto_voucher == 'Yes' && session('user_role') != 'Super Admin' && session('user_role') != 'Super User') {
            return response()->json(['success' => false, 'message_type' => 'error', 'message' => 'Auto vouchers cannot be cancelled Manually.']);
        }

        // Cancel the voucher
        $voucher->voucher_status = 'Cancelled';
        $voucher->cancellation_reason = $reason;
        $voucher->cancelled_by = session('user_id');
        $voucher->cancelled_on = date('Y-m-d H:i:s');
        $voucher->save();

        return response()->json([
            'success' => true,
            'message_type' => 'success', 
            'message_btn' => 'btn btn-success', 
            'message' => 'Voucher cancelled successfully.'
        ]);
    } 

    public function get_account_heads(Request $request)
    {
        $business_id = $request->input('business_id');
        if($business_id == "" || $business_id == 0) {
            if (session('ho_accounts') == 'Yes') {
                //get headoffice id
                $business_id = Business::where('ho', 'Yes')->value('id_business');
            } else{
                $business_id = session('business_id');
            }
        }

        //get all account heads for the business         
        $account_heads = AccountHeads::select('id_account_heads', 'account_head', 'account_head_number', 
        'account_sub_control.account_sub_control', 'account_control.account_control', 'account_main.account_main')
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->join('account_main', 'account_control.account_main_id', '=', 'account_main.id_account_main')
        ->where('account_heads.business_id', $business_id)
        ->where('account_head_status', 'Active');

        if($request->has('account_sub_control')) {
            $account_sub_control = $request->input('account_sub_control');
            if(!empty($account_sub_control)) {
                $account_heads = $account_heads->where('account_sub_control', 'like', "%{$account_sub_control}%");
            }
        }

        if($request->has('account_control')) {
            $account_control = $request->input('account_control');
            if(!empty($account_control)) {
                $account_heads = $account_heads->where('account_control', 'like', "%{$account_control}%");
            }
        }

        if($request->has('account_main')) {
            $account_main = $request->input('account_main');
            if(!empty($account_main)) {
                $account_heads = $account_heads->where('account_main', 'like', "%{$account_main}%");
            }
        }

        $account_heads = $account_heads->orderBy('id_account_main')
            ->orderBy('account_control')
            ->orderBy('account_sub_control')
            ->orderBy('account_head_number')
        ->get();

        return response()->json([
            'account_heads' => $account_heads
        ]);
    }
    
    function getProfitAndLossAmount($asOnDate, $business_id = 0)
    {   

        $business_id_for_account_head_id =session('business_id');
        if(session('ho_accounts') == 'Yes'){
            $business_id_for_account_head_id = Business::where('ho', 'Yes')->value('id_business'); 
        }   

        $profitLoss = DB::table('account_vouchers as v')
            ->join('account_voucher_detail as d', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
            ->join('account_heads as h', 'd.account_head_id', '=', 'h.id_account_heads')
            ->join('account_sub_control as sc', 'h.account_sub_control_id', '=', 'sc.id_account_sub_control')
            ->join('account_control as c', 'sc.account_control_id', '=', 'c.id_account_control')
            ->join('account_main as m', 'c.account_main_id', '=', 'm.id_account_main')
            ->where('h.business_id', $business_id_for_account_head_id)
            ->where('v.voucher_status', 'Active')
            ->where('h.account_head_status', 'Active')
            ->whereIn('m.account_main', ['Revenue','Expenses'])
            ->whereDate('v.voucher_date', '<=', $asOnDate);
            
        if ($business_id != 0) {
            $profitLoss->where('v.business_id', $business_id);
        } else {

            $profitLoss->whereIn('v.business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where(function ($q) {
                        $q->where('ho_accounts', 'Yes')
                        ->orWhere('ho', 'Yes');
                    });
            });

        }

        $profitLoss->select(
            'm.account_main',
            DB::raw('SUM(d.debit) as total_debit'),
            DB::raw('SUM(d.credit) as total_credit')
        )
        ->groupBy('m.id_account_main');

        $data = $profitLoss->get();
        $totalRevenue = 0;
        $totalExpenses = 0;

        foreach ($data as $row) {
            if ($row->account_main == "Revenue") {
                $totalRevenue += $row->total_credit - $row->total_debit;
            }
            if ($row->account_main == "Expenses") {
                $totalExpenses += $row->total_debit - $row->total_credit;
            }
        }
        $netProfitLoss = $totalRevenue - $totalExpenses;
        return round($netProfitLoss, 2);  
    }

    // New function to build hierarchical structure
    private function buildHierarchicalStructure($data)
    {
        $hierarchicalData = [];
        
        foreach ($data as $item) {
            $mainId = $item->id_account_main;
            $controlId = $item->id_account_control;
            $subControlId = $item->id_account_sub_control;
            
            // Initialize main category if not exists
            if (!isset($hierarchicalData[$mainId])) {
                $hierarchicalData[$mainId] = [
                    'id' => $mainId,
                    'name' => $item->account_main,
                    'type' => 'main',
                    'closing_balance' => 0.00,
                    'children' => []
                ];
            }
            
            // Initialize control category if not exists
            if (!isset($hierarchicalData[$mainId]['children'][$controlId])) {
                $hierarchicalData[$mainId]['children'][$controlId] = [
                    'id' => $controlId,
                    'name' => $item->account_control,
                    'type' => 'control',
                    'closing_balance' => 0.00,
                    'children' => []
                ];
            }
            
            // Initialize sub-control category if not exists
            if (!isset($hierarchicalData[$mainId]['children'][$controlId]['children'][$subControlId])) {
                $hierarchicalData[$mainId]['children'][$controlId]['children'][$subControlId] = [
                    'id' => $subControlId,
                    'name' => $item->account_sub_control,
                    'type' => 'sub_control',
                    'closing_balance' => 0.00,
                    'children' => []
                ];
            }
            
            // Add account head
            $hierarchicalData[$mainId]['children'][$controlId]['children'][$subControlId]['children'][] = [
                'id' => $item->id_account_heads,
                'name' => $item->account_head,
                'number' => $item->account_head_number,
                'type' => 'head',
                'closing_balance' => $item->closing_balance,
                'total_debit' => $item->total_debit,
                'total_credit' => $item->total_credit
            ];
            // Update balances upward
            $hierarchicalData[$mainId]['children'][$controlId]['children'][$subControlId]['closing_balance'] += $item->closing_balance;
            $hierarchicalData[$mainId]['children'][$controlId]['closing_balance'] += $item->closing_balance;
            $hierarchicalData[$mainId]['closing_balance'] += $item->closing_balance;
        }
        
        // Convert to simple array and reset keys
        $result = [];
        foreach ($hierarchicalData as $main) {
            $main['children'] = array_values($main['children']);
            foreach ($main['children'] as &$control) {
                $control['children'] = array_values($control['children']);
                foreach ($control['children'] as &$subControl) {
                    $subControl['children'] = array_values($subControl['children']);
                }
            }
            $result[] = $main;
        }
        
        return $result;
    } 

    public function accounts_closing(Request $request)
    {  
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                 $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                $business_id = Business::where('ho', 'Yes')->value('id_business');
                 $businesses = Business::where('id_business',  session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', $business_id)->get();
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        $business_id_for_voucher = $request->input('selected_business_id') ?? session("business_id");

          $lastClosing = DB::table('fiscal_year_closings')
            ->selectRaw("DATE_FORMAT(closing_date, '%Y-%m') as closing_ym")
            ->selectRaw("fiscal_year_closings.closing_date as closing_date ")
            ->where('business_id', $business_id_for_voucher)
            ->where('status', 'Active')
            ->orderBy('closing_date', 'desc')
            ->first();

        $fiscalYearStartFrom = [
            'last_closed' => null,
            'current_running' => Carbon::now()->format('d-M-Y') // Current date
        ];
        if ($lastClosing) {
            $fiscalYearStartFrom['last_closed'] = Carbon::parse($lastClosing->closing_date)->format('d-M-Y');   
        }


        $mybusinessname =session('business_name');
        $closing_months = DB::select("SELECT * from months"); 
        
        $closing_month = $request->closing_month;
        $closing_year  =  $request->closing_year;

        $last_year = $closing_year - 1;
        $last_month = sprintf("%02d", $closing_month - 1);

        $closing_month = (int)$closing_month; 
        $closing_month = sprintf("%02d", $closing_month); 

        $expense_accounts=[];
        $expense_accounts_summary =[];
        $retained_account =[];
        $income_accounts =[];
        $income_accounts_summary =[]; 
        if($closing_month != '' && $closing_year != ''){ 
            

            $selectedYearAndMonth = $closing_year . '-' . str_pad($closing_month, 2, '0', STR_PAD_LEFT);
            if ($lastClosing) {
                $lastYM = $lastClosing->closing_ym; // e.g., "2025-12"
                if ($selectedYearAndMonth <= $lastYM) {

                    return redirect()->route('accounts.accounts_closing')->with('warning', "You have already closed accounts up to $lastYM. You cannot close a previous month.");
                }
            }

        
            $sql = "SELECT account_heads.*, sum(account_voucher_detail.debit) as debit, sum(account_voucher_detail.credit) as credit,
            round((sum(account_voucher_detail.debit) - sum(account_voucher_detail.credit)),2) as balance
            FROM account_vouchers           
            JOIN account_voucher_detail ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            JOIN account_heads ON account_voucher_detail.account_head_id = account_heads.id_account_heads
            Join account_sub_control ON account_heads.account_sub_control_id = account_sub_control.id_account_sub_control
            join account_control ON account_sub_control.account_control_id = account_control.id_account_control
            join account_main ON account_control.account_main_id = account_main.id_account_main
            WHERE voucher_status = 'Active'
            AND account_heads.business_id = '".$business_id."'
            AND account_vouchers.business_id = '".$business_id_for_voucher."'
            AND account_main.account_main Like 'Expense%' 
            #AND account_vouchers.voucher_date > '".$closing_year-1 ."-".$closing_month."-31 23:59:59' "." 
            AND account_vouchers.voucher_date <= '".$closing_year."-".$closing_month."-31 23:59:59' ". 
            " GROUP BY account_heads.id_account_heads;";
            $expense_accounts = collect(DB::select($sql))->toArray(); 

            $sql = "SELECT account_heads.*, sum(account_voucher_detail.debit) as debit, sum(account_voucher_detail.credit) as credit,
            round((sum(account_voucher_detail.debit) - sum(account_voucher_detail.credit)),2) as balance
            FROM account_vouchers
            JOIN account_voucher_detail ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            JOIN account_heads ON account_voucher_detail.account_head_id = account_heads.id_account_heads
            Join account_sub_control ON account_heads.account_sub_control_id = account_sub_control.id_account_sub_control
            join account_control ON account_sub_control.account_control_id = account_control.id_account_control
            join account_main ON account_control.account_main_id = account_main.id_account_main
            WHERE voucher_status = 'Active'
            AND account_heads.business_id = '".$business_id."'
            AND account_vouchers.business_id = '".$business_id_for_voucher."'
            AND account_main.account_main Like 'Expense%' 
            #AND account_vouchers.voucher_date > '".$closing_year-1 ."-".$closing_month."-31 23:59:59' "." 
            AND account_vouchers.voucher_date <= '".$closing_year."-".$closing_month."-31 23:59:59' ". 
            " GROUP BY account_main.id_account_main;";
            $expense_accounts_summary = collect(DB::select($sql))->toArray(); 


            $sql = "select account_heads.* from account_heads 
            Join account_sub_control ON account_heads.account_sub_control_id = account_sub_control.id_account_sub_control
            join account_control ON account_sub_control.account_control_id = account_control.id_account_control
            join account_main ON account_control.account_main_id = account_main.id_account_main
            where account_heads.account_head like 'Retained%' and account_main.account_main like 'Equity%' 
            AND account_heads.business_id = '".$business_id."'";
            
            $retained_account = collect(DB::select($sql))->first(); 

            ///get the income accounts closing data
           $sql = "SELECT account_heads.*, sum(account_voucher_detail.debit) as debit, sum(account_voucher_detail.credit) as credit,
            round((sum(account_voucher_detail.credit) - sum(account_voucher_detail.debit)),2) as balance
            FROM account_vouchers           
            JOIN account_voucher_detail ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            JOIN account_heads ON account_voucher_detail.account_head_id = account_heads.id_account_heads
            Join account_sub_control ON account_heads.account_sub_control_id = account_sub_control.id_account_sub_control
            join account_control ON account_sub_control.account_control_id = account_control.id_account_control
            join account_main ON account_control.account_main_id = account_main.id_account_main
            WHERE voucher_status = 'Active'
            AND account_heads.business_id = '".$business_id."'
            AND account_vouchers.business_id = '".$business_id_for_voucher."'
            AND account_main.account_main Like 'Revenue%' 
            #AND account_vouchers.voucher_date > '".$closing_year-1 ."-".$closing_month."-31 23:59:59' "." 
            AND account_vouchers.voucher_date <= '".$closing_year."-".$closing_month."-31 23:59:59' ". 
            " GROUP BY account_heads.id_account_heads;";
            $income_accounts = collect(DB::select($sql))->toArray(); 


            //get the income accounts summary
            $sql = "SELECT account_heads.*, sum(account_voucher_detail.debit) as debit, sum(account_voucher_detail.credit) as credit,
            round((sum(account_voucher_detail.credit) - sum(account_voucher_detail.debit)),2) as balance
            FROM account_vouchers           
            JOIN account_voucher_detail ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            JOIN account_heads ON account_voucher_detail.account_head_id = account_heads.id_account_heads
            Join account_sub_control ON account_heads.account_sub_control_id = account_sub_control.id_account_sub_control
            join account_control ON account_sub_control.account_control_id = account_control.id_account_control
            join account_main ON account_control.account_main_id = account_main.id_account_main
            WHERE voucher_status = 'Active'
            AND account_heads.business_id = '".$business_id."'
            AND account_vouchers.business_id = '".$business_id_for_voucher."'
            AND account_main.account_main Like 'Revenue%' 
            #AND account_vouchers.voucher_date > '".$closing_year-1 ."-".$closing_month."-31 23:59:59' "." 
            AND account_vouchers.voucher_date <= '".$closing_year."-".$closing_month."-31 23:59:59' ". 
            " GROUP BY account_main.id_account_main;";

            $income_accounts_summary = collect(DB::select($sql))->toArray(); 

           if($request->close_accounts == "Close Accounts"){
         
            //check if the accounts are already closed for the month
                $sql = "SELECT * 
                        FROM account_vouchers 
                        WHERE voucher_status = 'Active' 
                        AND id_account_head = -333 
                        AND YEAR(voucher_date) = ? 
                        AND MONTH(voucher_date) = ? 
                        AND business_id = ?";
                $result = DB::select($sql, [$closing_year,$closing_month, $business_id_for_voucher]);
                if (!empty($result)) {
                     $voucher = $result[0];
                        $voucherPrintUrl = route('accounts.account_voucher_print', ['id' => $voucher->id_account_vouchers]);
                        $actionButtons = '&nbsp &nbsp &nbsp &nbsp &nbsp<a href="' . $voucherPrintUrl . '" target="voucherprint" class="btn btn-primary btn-sm">
                         <iconify-icon icon="line-md:watch-loop" class="fs-20"></iconify-icon> Voucher ID is ' . $voucher->id_account_vouchers . '</a>';
                     return redirect()->route('accounts.accounts_closing')->with('warning', 'Accounts are already closed using voucher ID ' . $actionButtons. 
                        ' year ' . date("M", mktime(0, 0, 0, $closing_month, 1)) . '-' . $closing_year);

                } else {
                    DB::beginTransaction(); 
                    try { 
                        // No record found, continue process 

                            $lastday = cal_days_in_month(CAL_GREGORIAN, $closing_month, $closing_year);
                            $month = (int)$closing_month; // Ensure it's an integer
                            $month = sprintf("%02d", $month);
                            $voucher_date = $closing_year."-".$month."-".$lastday." 23:59:59";
                            $voucher_amount = $income_accounts_summary[0]->balance + $expense_accounts_summary[0]->balance;

                           //------ CLOSING YEAR NOTE ENTRY 
                            $fiscal_year_closing_id = DB::table('fiscal_year_closings')->insertGetId([
                                    'business_id'   => $business_id_for_voucher,
                                    'closing_month' => $closing_month,
                                    'closing_year'  => $closing_year,
                                    'closing_date'  => $voucher_date,
                                    'status'        => 'Active',
                                    'created_by'    => session('user_name'),
                                    'created_at'    => now(),
                            ]);

                            //------ VOUCHER CREATION START
                            $closing_voucher = array(
                                'voucher_date' => $voucher_date,
                                'description' => 'Accounts Closing on '.$voucher_date. ' by '.session('user_name'),
                                'voucher_amount' => $voucher_amount,
                                'voucher_type' => 3, // 3 is the type for journal voucher,
                                'voucher_status' => 'Active',
                                'business_id' => $business_id_for_voucher,
                                'created_by' => session('user_name'),
                                'cost_center' => 1,
                                'cost_center_name'=> 'Back Office',
                                'business_partner' => 1, // 1 is the business partner for the business itself
                                'business_partner_id' => $business_id_for_voucher,
                                'business_partner_name' => $mybusinessname,
                                'payment_mode' => 'Cash',
                                'id_account_head' => -333,
                                'auto_voucher' => 'Yes', 
                                'fiscal_year_closing_id' => $fiscal_year_closing_id 
                            );

                            $closing_voucher_id = DB::table('account_vouchers')->insertGetId($closing_voucher);
                            
                            $retained_earnings_data = array(
                                'account_voucher_id' => $closing_voucher_id,
                                'account_head_id' => $retained_account->id_account_heads,
                                'debit' => $expense_accounts_summary[0]->balance,
                                'credit' => 0,
                                'detail_remarks' => 'Debit Retained Earnings Account Against Closing Expense Accounts'
                            );
                            DB::table('account_voucher_detail')->insert($retained_earnings_data);

                            foreach($expense_accounts as $expense_account){ 
                                $closing_data = array(
                                    'account_voucher_id' => $closing_voucher_id,
                                    'account_head_id' => $expense_account->id_account_heads,
                                    'debit' => 0,
                                    'credit' => $expense_account->balance,
                                    'detail_remarks' => 'Closing Expense Account: '.$expense_account->account_head
                                );
                                DB::table('account_voucher_detail')->insert($closing_data);
                            }
                            
                            //insert the retained earnings account closing data against the income accounts
                            $retained_earnings_data = array(
                                'account_voucher_id' => $closing_voucher_id,
                                'account_head_id' => $retained_account->id_account_heads,
                                'debit' => 0,
                                'credit' => $income_accounts_summary[0]->balance,
                                'detail_remarks' => 'Credit Retained Earnings Account Against Closing Income Accounts'
                            );
                            DB::table('account_voucher_detail')->insert($retained_earnings_data);
                            //insert the income accounts closing data
                            foreach($income_accounts as $income_account){
                                $closing_data = array(
                                    'account_voucher_id' => $closing_voucher_id,
                                    'account_head_id' => $income_account->id_account_heads,
                                    'debit' => $income_account->balance,
                                    'credit' => 0,
                                    'detail_remarks' => 'Closing Income Account: '.$income_account->account_head
                                );
                                DB::table('account_voucher_detail')->insert($closing_data);
                            }

                            // $this->session->set_flashdata('Success', 'Accounts Closed Successfully for the month of '.$data['closing_month'].'-'.$data['closing_year'].' and Voucher ID is '.$closing_voucher_id);

                            $income_accounts = array();
                            $expense_accounts = array();
                            $income_accounts_summary = array();
                            $expense_accounts_summary = array();
                            $closing_month="";
                         DB::commit();  

                        $voucherPrintUrl = route('accounts.account_voucher_print', ['id' => $closing_voucher_id]);

                        $actionButtons = '&nbsp &nbsp &nbsp &nbsp &nbsp <a href="' . $voucherPrintUrl . '" target="voucherprint" class="btn btn-primary btn-sm">
                         <iconify-icon icon="line-md:watch-loop" class="fs-20"></iconify-icon> Voucher ID is ' . $closing_voucher_id . '</a>';

                        return redirect()->route('accounts.accounts_closing')
                            ->with('success', 'Accounts Closed Successfully for the month of ' . $closing_month . '-' . $closing_year . ' ' . $actionButtons);

                    } catch (\Exception $e) {
                            DB::rollBack();
                            return redirect()->route('accounts.accounts_closing')
                            ->with('error', "Error: " . $e->getMessage());
                    } 
                }

           }

        }   
        return view('accounts.accounts_closing')->with(compact( 'businesses','mybusinessname','closing_months','closing_month','closing_year','last_year','last_month','expense_accounts','expense_accounts_summary','retained_account','income_accounts','income_accounts_summary','business_id_for_voucher','fiscalYearStartFrom'));
    }

    public function accounts_closing_history(Request $request){ 

        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                 $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                $business_id = Business::where('ho', 'Yes')->value('id_business');
                 $businesses = Business::where('id_business',  session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', $business_id)->get();
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }

        $selected_business_id = $request->input('business_id') ?? session("business_id"); 

        $lastClosing = DB::table('fiscal_year_closings')
            ->selectRaw("DATE_FORMAT(closing_date, '%Y-%m') as closing_ym")
            ->selectRaw("fiscal_year_closings.closing_date as closing_date ")
            ->where('business_id', $selected_business_id)
            ->where('status', 'Active')
            ->orderBy('closing_date', 'desc')
            ->first();

        $fiscalYearStartFrom = [
            'last_closed' => null,
            'current_running' => Carbon::now()->format('d-M-Y') // Current date
        ];
        if ($lastClosing) {
            $fiscalYearStartFrom['last_closed'] = Carbon::parse($lastClosing->closing_date)->format('d-M-Y');   
        }


        $ClosingData = [];
        if ($request->ajax()) {
            $businessIdForHistory = $request->business_id ?? 0;

            $ClosingDataQuery = DB::table('fiscal_year_closings')
                ->join('account_vouchers', 'account_vouchers.fiscal_year_closing_id', '=', 'fiscal_year_closings.id')
                ->join('business','business.id_business','account_vouchers.business_id')
                ->selectRaw("fiscal_year_closings.*, DATE_FORMAT(closing_date,'%d-%b-%Y') as closing_date, account_vouchers.id_account_vouchers ,business.business_name");

            if ($businessIdForHistory > 0) {
                $ClosingDataQuery = $ClosingDataQuery->where('fiscal_year_closings.business_id', $businessIdForHistory);
            }
            $ClosingData = $ClosingDataQuery->orderBy('closing_date', 'desc')->get();
            return response()->json([
                'ClosingData' => $ClosingData,
                'fiscalYearStartFrom' => $fiscalYearStartFrom,
            ]);
        }
        return view('accounts.accounts_closing_history')->with(compact( 'businesses','selected_business_id','ClosingData'));
    }

    public function cancel_fiscal_year_closing(Request $request)
    {
        $id = $request->id;
        $reason = $request->reason;
        try {
            DB::beginTransaction();


            // Update fiscal year closing
            DB::table('fiscal_year_closings')
                ->where('id', $id)
                ->update([
                    'status' => 'Cancelled',
                    'cancelled_on' => now(),
                    'cancelled_by' => session('user_name'),
                    'cancellation_reason' => $reason
                ]);

            // Update account vouchers
            DB::table('account_vouchers')
                ->where('fiscal_year_closing_id', $id)
                ->update([
                    'voucher_status'       => 'Cancelled',
                    'cancellation_reason'  => $reason,
                    'cancelled_by'         => session('user_id'),
                    'cancelled_on'         => now(),
                ]);

            DB::commit();

            $fiscal_year_closings = DB::table('fiscal_year_closings')->where('id', $id)->first();

            return response()->json([
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success',
                'message' => 'Cancelled successfully.',
                'fiscal_year_closings' => $fiscal_year_closings
            ]);

        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
                'message' => 'Error occurred while cancelling.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function businessPartnerLedger(Request $request, $type){
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                 $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                 $business_id = Business::where('ho', 'Yes')->value('id_business'); 
                 $businesses = Business::where('id_business', session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', $business_id)->get(); 
            }           
        } else {
            // return redirect()->back()->withErrors("You do not have the permission to access this resource.")->withInput();
            return "You do not have the permission to access this resource.";
        }
        $start_date = $request->input('start_date') ?? date('Y-m-01');
        $end_date   = $request->input('end_date') ?? date('Y-m-t'); 
        $account_business_partner = DB::table('account_business_partner')
         ->select('id_account_business_partner','account_business_partner')->whereRaw('LOWER(account_business_partner) = LOWER(?)', [$type])
         ->first();

        $business_partner = $account_business_partner->id_account_business_partner;
        $account_business_partner_name = $account_business_partner->account_business_partner;
        $selected_business_id =  $request->selected_business_id ?? $business_id = session('business_id');
        return view('accounts.business_partner_ledger')->with(compact( 'businesses','account_business_partner','businesses','selected_business_id'));
    }

    public function get_business_partner_ledger(Request $request){
        $selected_business_id = $request->business_id ?? 0;
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        $account_type = $request->account_type;
        $business_partner_type = $request->business_partner_type;
        $business_partner_id = $request->business_partner; 


        if($business_partner_id ==0 || $business_partner_id ==""){
            return response()->json([
                    'opening_balance' => 0,
                    'business_partner_ledger' =>[]
            ]);
        }


        $account_business_partner = DB::table('account_business_partner')
        ->select('id_account_business_partner','account_business_partner')
        ->where('id_account_business_partner', $business_partner_type)
        ->first();
    
        $account_business_partner_name = $account_business_partner->account_business_partner;
        $accounts_payable = $this->get_event_account(20, 'payables');

        $account_head_id = 0;
        $account_head_name ="";
        if($account_business_partner_name == "Suppliers" || $account_business_partner_name == "Customers" || $account_business_partner_name == "Franchise"){
            if($account_type == "Payable"){  
                $accounts_payable = $this->get_event_account(20, 'payables');
                $account_head_id = $accounts_payable->account_head_id;
                $account_head_name =$accounts_payable->account_head;
            } else {
                $accounts_receivable = $this->get_event_account(1, 'balance');
                $account_head_id = $accounts_receivable->account_head_id;
                 $account_head_name =$accounts_receivable->account_head;
            }
        }else if($account_business_partner_name == "Staff") {
            if($account_type == "Payable"){  
                $accounts_payable = $this->get_event_account(26 , 'salary_payable');
                $account_head_id = $accounts_payable->account_head_id;
                $account_head_name =$accounts_payable->account_head;
            } else {
                $accounts_receivable = $this->get_event_account(26, 'loan_deduction');
                $account_head_id = $accounts_receivable->account_head_id;
                $account_head_name =$accounts_receivable->account_head;
            }
        }else{
             if($account_type == "Payable"){  
                $accounts_payable = $this->get_event_account(20, 'payables');
                $account_head_id = $accounts_payable->account_head_id;
                $account_head_name =$accounts_payable->account_head;
            } else {
                $accounts_receivable = $this->get_event_account(1, 'balance');
                $account_head_id = $accounts_receivable->account_head_id;
                $account_head_name =$accounts_receivable->account_head;

            }
        }

        $opening_balance = $this->businessPartnerOpeningBalance($start_date, $end_date, $business_partner_type, $business_partner_id, $selected_business_id, $account_head_id, $account_type); 


        $balanceExpression = strpos($account_type, "Payable") === false ? "IFNULL(SUM(debit) - SUM(credit), 0)" : "IFNULL(SUM(credit) - SUM(debit), 0)";
            $sql = "
                SELECT 
                    business.business_name AS Branch,
                    id_account_vouchers,
                    business_partner_name,
                    DATE_FORMAT(account_vouchers.voucher_date, '%d-%m-%Y') AS voucher_date,
                    DATE_FORMAT(account_vouchers.created_on, '%d-%m-%Y') AS create_date,
                    account_vouchers.description,
                    SUM(debit) as debit,
                    SUM(credit) as credit,
                    {$balanceExpression} AS closing_balance

                FROM account_voucher_detail
                JOIN account_vouchers 
                    ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                JOIN business 
                    ON business.id_business = account_vouchers.business_id
                WHERE account_vouchers.voucher_status = 'Active'
                    AND business_partner = {$business_partner_type}
                    AND business_partner_id = {$business_partner_id} ";
            if( $selected_business_id !=0){
                $sql .=" AND id_business = {$selected_business_id} ";
            }else{
                $business_ids = DB::table('business')->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')
                ->pluck('id_business')
                ->implode(',');
               $sql .= " AND id_business IN ($business_ids) ";
            }

            $sql .="   AND account_voucher_detail.account_head_id = {$account_head_id}
                    AND DATE(account_vouchers.voucher_date) >= '{$start_date}'
                    AND DATE(account_vouchers.voucher_date) <= '{$end_date}'
                GROUP BY   account_vouchers.id_account_vouchers 
                ORDER BY DATE(voucher_date)
            ";

        $business_partner_ledger = DB::select($sql);

        return response()->json([
            'account_head_name' => $account_head_name,
            'opening_balance' => $opening_balance,
            'business_partner_ledger' => $business_partner_ledger
        ]);
    }

    public static  function get_event_account($eventid, $entity_name, $business_id = 0)
    {
        // Determine business_id
        if (empty($business_id) || $business_id == 0) {
            if (session('ho_accounts') === 'Yes') {
                $business_id = Business::where('ho', 'Yes')->value('id_business');
            } else {
                $business_id = session('business_id');
            }
        }

        return DB::table('account_event_mapping')
            ->join('account_heads', 'account_heads.id_account_heads', '=', 'account_event_mapping.account_head_id')
            ->select('account_head_id', 'account_head')
            ->where('account_event_id', $eventid)
            ->where('entity_name', $entity_name)
            ->where('account_event_mapping.business_id', $business_id)
            ->first();
    }

    public function businessPartnerOpeningBalance($startdate, $enddate, $business_partner, $business_partner_id, $id_business, $account_id = 1, $account_type = 'Receivable') {
        $balanceExpression = strpos($account_type, "Payable") === false ? "IFNULL(SUM(debit) - SUM(credit), 0)" : "IFNULL(SUM(credit) - SUM(debit), 0)";
        if($id_business == 0 || $id_business == "" ){    
            $business_ids = DB::table('business')->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->pluck('id_business')->implode(',');
        }              

        $sql = "
            SELECT {$balanceExpression} AS balance
            FROM account_voucher_detail
            JOIN account_vouchers 
                ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            JOIN business 
                ON business.id_business = account_vouchers.business_id
            JOIN account_heads 
                ON account_heads.id_account_heads = account_voucher_detail.account_head_id
            WHERE account_vouchers.voucher_status = 'Active'
            AND business_partner = $business_partner
            AND business_partner_id = $business_partner_id ";
            // Business filter
            if ($id_business != 0) {
                $sql .= " AND id_business = $id_business ";
            } else {
                $sql .= " AND id_business IN ($business_ids) ";
            }
            $sql .="AND account_voucher_detail.account_head_id = $account_id
            AND DATE(account_vouchers.voucher_date) < '$startdate'
            GROUP BY account_voucher_detail.account_head_id
        ";
        
        $result = DB::select($sql);

        return $result ? $result[0]->balance : 0;
    } 

  // aging_payables
    public function aging_payables(Request $request){ 
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                $business_id = Business::where('ho', 'Yes')->value('id_business'); 
                $businesses = Business::where('id_business', session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', $business_id)->get(); 
            }           
        } else {
            return "You do not have the permission to access this resource.";
        }
        
        $selected_business_id = $request->input('business_id') ?? session("business_id"); 
        $as_on_date = $request->input('as_on_date') ?? date('Y-m-01'); 

        $accounts_payable = $this->get_event_account(20, 'payables');
        if($accounts_payable){
            $account_head_id = $accounts_payable->account_head_id;
            $account_head_name = $accounts_payable->account_head;
        } else {
            return 'Mapping not available';
        }

        if ($request->ajax()) {
            $searchValue = $request->input('search.value');
            $aging = DB::select($this->aging_payables_query($selected_business_id, $account_head_id, $as_on_date,$searchValue));
            // Server-side processing response
            $response = [
                'draw' => $request->input('draw', 0),
                'recordsTotal' => count($aging),
                'recordsFiltered' => count($aging),
                'data' => $aging,
                'account_head_name' => $account_head_name,
                'as_on_date' => $as_on_date
            ];
            return response()->json($response);
        }
        
        return view('accounts.reports.aging_payables')->with(compact('businesses','selected_business_id','as_on_date'));    
    }

    function aging_payables_query($business_id, $account_head_id, $as_on_date, $searchValue = '') {
       if($business_id == 0 || $business_id == "" ){    
            $business_ids = DB::table('business')->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->pluck('id_business')->implode(',');
        }   
        $sql = "
        SELECT 
            account_head_id,
            business_partner,
            branch,
            business_partner_name,
            business_partner_id,
            SUM(over90) AS over90,
            SUM(60to90) AS sixtyninty,
            SUM(30to59) AS thirtyfiftynine,
            SUM(lessthen30) AS lessthen30
        FROM (
            SELECT 
                business_partner,
                account_head_id,
                Branch,
                business_partner_name,
                business_partner_id,
                CASE WHEN lapseddays >= 90 
                    THEN SUM(credit) - SUM(debit) 
                    ELSE 0 
                END AS over90,
                CASE WHEN lapseddays < 90 AND lapseddays >= 60  
                    THEN SUM(credit) - SUM(debit) 
                    ELSE 0  
                END AS 60to90,
                CASE WHEN lapseddays < 60 AND lapseddays >= 30  
                    THEN SUM(credit) - SUM(debit) 
                    ELSE 0  
                END AS 30to59,
                CASE WHEN lapseddays < 30  
                    THEN SUM(credit) - SUM(debit) 
                    ELSE 0 
                END AS lessthen30
            FROM (
                SELECT 
                    business_partner,
                    account_voucher_detail.account_head_id,
                    purchase_order_id,
                    DATE(voucher_date),
                    business.business_name AS Branch,
                    business_partner_name,
                    account_vouchers.business_partner_id,
                    account_vouchers.business_partner AS partner_type,
                    SUM(credit) - SUM(debit) AS Balance,
                    SUM(debit) AS debit,
                    SUM(credit) AS credit,
                    DATEDIFF(NOW(), account_vouchers.voucher_date) AS lapseddays
                FROM account_voucher_detail
                JOIN account_vouchers 
                    ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                JOIN business 
                    ON business.id_business = account_vouchers.business_id
                WHERE 
                    account_vouchers.voucher_status = 'Active' ";
                if($business_id !=0){
                    $sql.=" AND id_business = ".$business_id." " ;
                }else{
                    $sql.=" AND id_business in(".$business_ids." ) " ;
                }    

                $sql.=" AND account_voucher_detail.account_head_id = ".$account_head_id."
                    AND DATE(account_vouchers.voucher_date) <= '".$as_on_date."'";
        
        if (!empty($searchValue)) {
            $sql .= " AND (business_partner_name LIKE '%".$searchValue."%' OR business.business_name LIKE '%".$searchValue."%')";
        }
        
        $sql .= "
                GROUP BY 
                    Branch,
                    account_vouchers.purchase_order_id, 
                    business_partner_id
            ) AS a
            GROUP BY 
                Branch,
                business_partner_id, 
                lapseddays
        ) AS b
        GROUP BY branch, business_partner_id ORDER BY business_partner_id";
        
        return $sql;
    }

    // Aging Receivables
    public function aging_receivables(Request $request){
        if(session('user_role')=="Super User" || session('user_role') == "Super Admin" || session('user_role')=="Accountant" || session('user_role')=="Admin"){
            if(session('ho')=="Yes"){
                $business_id = session("business_id");
                $businesses = Business::where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->get(); 
            } else if(session("ho_accounts")=="Yes" && session("ho")=="No"){
                $business_id = Business::where('ho', 'Yes')->value('id_business'); 
                $businesses = Business::where('id_business', session("business_id"))->get(); 
            } else {
                $business_id = session('business_id');
                $businesses = Business::where('id_business', $business_id)->get(); 
            }           
        } else {
            return "You do not have the permission to access this resource.";
        }

        $selected_business_id = $request->input('business_id') ?? session("business_id"); 
        $as_on_date = $request->input('as_on_date') ?? date('Y-m-01'); 

        $accounts_payable = $this->get_event_account(1, 'balance');
        if($accounts_payable){
            $account_head_id = $accounts_payable->account_head_id;
            $account_head_name = $accounts_payable->account_head;
        } else {
            return 'Mapping not available';
        }

        if ($request->ajax()) {
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            $searchValue = $request->input('search.value');
            $draw = $request->input('draw', 0);


            $totalQuery = $this->aging_receivables_query($selected_business_id, $account_head_id, $as_on_date, $searchValue, true);
            $totalRecords = DB::select($totalQuery)[0]->total;


            $agingQuery = $this->aging_receivables_query($selected_business_id, $account_head_id, $as_on_date, $searchValue, false, $start, $length);
            $aging = DB::select($agingQuery);
            // $totalRecords = count($aging);
            $response = [
                'draw' => $draw,
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $aging,
                'account_head_name' => $account_head_name,
                'as_on_date' => $as_on_date
            ];
            return response()->json($response);
        }
        return view('accounts.reports.aging_receivables')->with(compact('businesses','selected_business_id','as_on_date'));    
    }

    function aging_receivables_query($business_id, $account_head_id, $as_on_date, $searchValue = '', $countOnly = false, $start = 0, $length = 10) {
        
        if($business_id == 0 || $business_id == "" ){    
            $business_ids = DB::table('business')->where('ho_accounts', 'Yes')->orWhere('ho', 'Yes')->pluck('id_business')->implode(',');
        }   

        if ($countOnly) {
            $sql = "
                SELECT COUNT(*) as total
                FROM (
                    SELECT business_partner_id
                    FROM (
                        SELECT 
                            account_head_id,
                            business_partner,
                            branch,
                            business_partner_name,
                            business_partner_id,
                            SUM(over90) AS 'over90',
                            SUM(60to90) AS 'sixtyninty',
                            SUM(30to59) AS 'thirtyfiftynine',
                            SUM(lessthen30) AS 'lessthen30'
                        FROM (
                            SELECT 
                                account_head_id,
                                business_partner,
                                Branch,
                                business_partner_name,
                                business_partner_id,

                                CASE 
                                    WHEN lapseddays >= 90 
                                        THEN SUM(debit) - SUM(credit)
                                    ELSE 0 
                                END AS 'over90',

                                CASE 
                                    WHEN lapseddays < 90 AND lapseddays >= 60 
                                        THEN SUM(debit) - SUM(credit)
                                    ELSE 0  
                                END AS '60to90',

                                CASE 
                                    WHEN lapseddays < 60 AND lapseddays >= 30  
                                        THEN SUM(debit) - SUM(credit)
                                    ELSE 0  
                                END AS '30to59',

                                CASE 
                                    WHEN lapseddays < 30  
                                        THEN SUM(debit) - SUM(credit)
                                    ELSE 0 
                                END AS 'lessthen30'

                            FROM (
                                SELECT  
                                    account_voucher_detail.account_head_id,
                                    business_partner,
                                    purchase_order_id,
                                    DATE(voucher_date) AS voucher_date,
                                    business.business_name AS 'Branch',
                                    business_partner_name,
                                    account_vouchers.business_partner_id,
                                    account_vouchers.business_partner AS partner_type,

                                    SUM(debit) - SUM(credit) AS Balance,
                                    SUM(debit) AS debit,
                                    SUM(credit) AS credit,

                                    DATEDIFF('".$as_on_date."', account_vouchers.voucher_date) AS lapseddays

                                FROM account_voucher_detail
                                JOIN account_vouchers 
                                    ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                                JOIN business 
                                    ON business.id_business = account_vouchers.business_id

                                WHERE 
                                    account_vouchers.voucher_status = 'Active' ";
                            if($business_id !=0){
                                $sql .=" AND id_business = ".$business_id." ";
                            }else{
                                $sql .=" AND id_business in (".$business_ids.") ";
                                
                            }        
                            $sql .=" AND account_voucher_detail.account_head_id = ".$account_head_id."
                                    AND DATE(account_vouchers.voucher_date) <= '".$as_on_date."' 
            ";

            if (!empty($searchValue)) {
                $sql .= "
                                AND (
                                    business_partner_name LIKE '%".$searchValue."%' 
                                    OR business.business_name LIKE '%".$searchValue."%'
                                )
                ";
            }

            $sql .= "
                                GROUP BY 
                                    account_vouchers.invoice_id, 
                                    business_partner_id
                            ) AS a

                            GROUP BY 
                                business_partner_id, 
                                lapseddays
                        ) AS b

                        GROUP BY business_partner_id
                    ) AS final_data
                ) AS count_table
            ";
        } else {
            $sql = "
                SELECT 
                    account_head_id,
                    business_partner,
                    branch,
                    business_partner_name,
                    business_partner_id,
                    SUM(over90) AS 'over90',
                    SUM(60to90) AS 'sixtyninty',
                    SUM(30to59) AS 'thirtyfiftynine',
                    SUM(lessthen30) AS 'lessthen30'
                FROM (
                    SELECT 
                        account_head_id,
                        business_partner,
                        Branch,
                        business_partner_name,
                        business_partner_id,

                        CASE 
                            WHEN lapseddays >= 90 
                                THEN SUM(debit) - SUM(credit)
                            ELSE 0 
                        END AS 'over90',

                        CASE 
                            WHEN lapseddays < 90 AND lapseddays >= 60 
                                THEN SUM(debit) - SUM(credit)
                            ELSE 0  
                        END AS '60to90',

                        CASE 
                            WHEN lapseddays < 60 AND lapseddays >= 30  
                                THEN SUM(debit) - SUM(credit)
                            ELSE 0  
                        END AS '30to59',

                        CASE 
                            WHEN lapseddays < 30  
                                THEN SUM(debit) - SUM(credit)
                            ELSE 0 
                        END AS 'lessthen30'

                    FROM (
                        SELECT  
                            account_voucher_detail.account_head_id,
                            business_partner,
                            purchase_order_id,
                            DATE(voucher_date) AS voucher_date,
                            business.business_name AS 'Branch',
                            business_partner_name,
                            account_vouchers.business_partner_id,
                            account_vouchers.business_partner AS partner_type,

                            SUM(debit) - SUM(credit) AS Balance,
                            SUM(debit) AS debit,
                            SUM(credit) AS credit,

                            DATEDIFF('".$as_on_date."', account_vouchers.voucher_date) AS lapseddays

                        FROM account_voucher_detail
                        JOIN account_vouchers 
                            ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                        JOIN business 
                            ON business.id_business = account_vouchers.business_id

                        WHERE 
                            account_vouchers.voucher_status = 'Active' ";
                            if($business_id !=0){
                                $sql .=" AND id_business = ".$business_id." ";
                            }else{
                                $sql .=" AND id_business in (".$business_ids.") ";
                                
                            }        
                            $sql .=" AND account_voucher_detail.account_head_id = ".$account_head_id."
                            AND DATE(account_vouchers.voucher_date) <= '".$as_on_date."' 
            ";

            if (!empty($searchValue)) {
                $sql .= "
                            AND (
                                business_partner_name LIKE '%".$searchValue."%' 
                                OR business.business_name LIKE '%".$searchValue."%'
                            )
                ";
            }

            $sql .= "
                        GROUP BY
                            Branch, 
                            account_vouchers.invoice_id, 
                            business_partner_id
                    ) AS a

                    GROUP BY 
                        Branch,
                        business_partner_id, 
                        lapseddays
                ) AS b

                GROUP BY branch , business_partner_id ORDER BY business_partner_id
            ";
            
            if ($length > 0) {
                $sql .= " LIMIT " . $start . ", " . $length;
            }
        }

        return $sql;
    } 


    public function get_business_partner_default_filters(Request  $request , FiltersService $filters,)
    {   
        $business_id = $request->business_id; 
        $account_business_partner = DB::table('account_business_partner')
        ->select('id_account_business_partner','account_business_partner')
        ->where('id_account_business_partner', $request->business_partner_type)
        ->first();
    
        $account_business_partner_name = $account_business_partner->account_business_partner;
        if($account_business_partner_name == "Suppliers"){
        return response()->json([
            'staff'      => [],
            'franchises' => [],
            'suppliers'  => $filters->getSuppliers($business_id),
            'businesses' => [],
        ]);
        
        }else if($account_business_partner_name == "Franchise"){
            return response()->json([
                'staff'      => [],
                'franchises' => $filters->getFranchises($business_id),
                'suppliers'  => [],
                'businesses' => [],
            ]);
            
        }else if($account_business_partner_name == "Staff") {
            return response()->json([
                'staff'      => $filters->getStaff($business_id),
                'franchises' =>[],
                'suppliers'  =>[],
                'businesses' =>[],
            ]);
            
        }else if($account_business_partner_name == "Branch"){
            return response()->json([
                'staff'      =>[],
                'franchises' =>[],
                'suppliers'  =>[],
                'businesses' => $filters->getBusinesses(),
            ]);

        }else{
            return response()->json([
                'staff'      => [],
                'franchises' => [],
                'suppliers'  => [],
                'businesses' => [],
            ]);
          
        }
    }
} 