<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Attendance;
use App\Models\Staff;
use App\Models\Business;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Yajra\DataTables\Facades\DataTables;

class AttendanceController extends Controller
{
    private function isHoUser()
    {
        return session('ho') === 'Yes';
    }

    private function getAllBusinesses()
    {
        return Business::orderBy('business_name', 'ASC')
            ->when(session('ho_accounts') === 'Yes', function($query) {
                return $query->where('ho_accounts', 'Yes');
            })
            ->get();
    }

    private function getBusinessIdForQuery(Request $request = null)
    {
        // If HO user and business_id is provided in request, use it
        if ($this->isHoUser() && $request && $request->filled('business_id')) {
            return $request->business_id;
        }
        
        // If HO user has selected business in session, use it
        if ($this->isHoUser() && session('selected_business_id')) {
            return session('selected_business_id');
        }
        
        // For non-HO users, use their own business ID
        return session('business_id');
    }

    public function attendanceRequests()
    {
        $businesses = $this->isHoUser() ? $this->getAllBusinesses() : collect([]);
        
        // Get current business for non-HO users
        $business = Business::where('id_business', session('business_id', session('businessid', 1)))->first();
        
        return view('hrm.attendance_request', compact('business', 'businesses'));
    }

    public function getAttendanceRequestsData(Request $request)
{ 
    try {
        $query = DB::table('attendance_request as ar')
            ->select(
                'ar.id_attendance_request',
                'ar.staff_id',
                'hs.staff_fullname as staff_name',
                'b.business_name as business',
                DB::raw("DATE_FORMAT(ar.request_date, '%d-%m-%Y') as request_date"),
                'ar.request_time_in',
                'ar.request_time_out',
                DB::raw("COALESCE(ar.approved_by, '') as approved_by"),
                'ar.address',
                'ar.request_status',
                'ar.business_id'
            )
            ->join('staff as hs', 'hs.id_staff', '=', 'ar.staff_id')
            ->join('business as b', 'b.id_business', '=', 'ar.business_id');
        
        // Apply business filter based on user role
        $isHoUser = $this->isHoUser();
        $businessId = $this->getBusinessIdForQuery($request);
        
        // For HO users, handle business selection
        if ($isHoUser) {
            // If specific business is selected in dropdown
            if ($request->filled('business_id') && $request->business_id !== "") {
                $query->where('ar.business_id', $request->business_id);
            } else {
                // If "All Businesses" is selected, show only businesses with ho_accounts = 'Yes'
                $query->where('b.ho_accounts', 'Yes');
            }
        } else {
            // Non-HO users: only see their own business
            $query->where('ar.business_id', $businessId);
        }
        
        // Apply individual filters
        if ($request->filled('id_attendance_request')) {
            $query->where('ar.id_attendance_request', $request->id_attendance_request);
        }
        
        if ($request->filled('staff_id')) {
            $query->where('ar.staff_id', $request->staff_id);
        }
        
        if ($request->filled('staff_name')) {
            $query->where('hs.staff_fullname', 'like', '%' . $request->staff_name . '%');
        }
        
        // Business name filter - for HO users, only search within ho_accounts businesses
        if ($request->filled('business')) {
            if ($isHoUser && (!$request->filled('business_id') || $request->business_id === "")) {
                // If HO user is viewing "All", search within ho_accounts businesses
                $query->where('b.business_name', 'like', '%' . $request->business . '%')
                      ->where('b.ho_accounts', 'Yes');
            } else {
                // For specific business or non-HO users
                $query->where('b.business_name', 'like', '%' . $request->business . '%');
            }
        }
        
        if ($request->filled('request_date')) {
            $query->where('ar.request_date', 'like', '%' . $request->request_date . '%');
        }
        
        if ($request->filled('request_time_in')) {
            $query->where('ar.request_time_in', $request->request_time_in);
        }
        
        if ($request->filled('request_time_out')) {
            $query->where('ar.request_time_out', $request->request_time_out);
        }
        
        if ($request->filled('address')) {
            $query->where('ar.address', 'like', '%' . $request->address . '%');
        }
        
        if ($request->filled('approved_by')) {
            $query->where('ar.approved_by', 'like', '%' . $request->approved_by . '%');
        }

        // Handle status filter
        if ($request->filled('request_status')) {
            // If empty string (All) is selected, show all statuses
            if ($request->request_status === 'All') {
                // No filter - show all statuses
            } 
            // If "Pending" is selected, show only pending
            else if ($request->request_status === 'Pending') {
                $query->where('ar.request_status', 'pending');
            }
            // If "Approved" is selected, show only approved
            else if ($request->request_status === 'Approved') {
                $query->where('ar.request_status', 'approved');
            }
            // If "Rejected" is selected, show only rejected
            else if ($request->request_status === 'Rejected') {
                $query->where('ar.request_status', 'rejected');
            }
            // For any other value, use it directly
            else {
                $query->where('ar.request_status', $request->request_status);
            }
        } else {
            // If no status filter is provided (first page load), default to showing only "Pending"
            $query->where('ar.request_status', 'pending');
        }

        // Get total count before pagination
        $recordsTotal = $query->count();

        // Apply ordering
        if ($request->has('order') && count($request->order) > 0) {
            $orderColumnIndex = $request->order[0]['column'];
            $orderDirection = $request->order[0]['dir'];
            
            $columns = [
                0 => 'ar.id_attendance_request',
                1 => 'ar.staff_id',
                2 => 'hs.staff_fullname',
                3 => 'b.business_name',
                4 => 'ar.request_date',
                5 => 'ar.request_time_in',
                6 => 'ar.request_time_out',
                7 => 'ar.address',
                8 => 'ar.request_status',
                9 => 'ar.approved_by',
            ];
            
            if (isset($columns[$orderColumnIndex])) {
                $query->orderBy($columns[$orderColumnIndex], $orderDirection);
            } else {
                $query->orderBy('ar.id_attendance_request', 'desc');
            }
        } else {
            $query->orderBy('ar.id_attendance_request', 'desc');
        }

        // Apply pagination
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);
        
        if ($length != -1) {
            $query->offset($start)->limit($length);
        }

        $data = $query->get();

        return response()->json([
            'draw' => $request->input('draw', 1),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsTotal,
            'data' => $data->map(function($item) {
                return [
                    'id_attendance_request' => $item->id_attendance_request,
                    'staff_id' => $item->staff_id,
                    'staff_name' => $item->staff_name,
                    'business' => $item->business, 
                    'request_date' => $item->request_date,
                    'request_time_in' => $item->request_time_in,
                    'request_time_out' => $item->request_time_out,
                    'address' => $item->address,
                    'request_status' => $item->request_status,
                    'approved_by' => $item->approved_by,
                ];
            })
        ]);

    } catch (\Exception $e) {
        \Log::error('Error fetching attendance requests: ' . $e->getMessage());
        return response()->json([
            'draw' => $request->input('draw', 1),
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => $e->getMessage()
        ], 500);
    }
}

    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'request_status' => 'required|in:approved,rejected,Approved,Rejected'
        ]);

        try {
            $attendance = DB::table('attendance_request')
                ->where('id_attendance_request', $id)
                ->first();

            if (!$attendance) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $status = strtolower($request->request_status);

            if ($status === 'approved') {
                DB::table('attendance_request')
                    ->where('id_attendance_request', $id)
                    ->update([
                        'request_status' => $status,
                        'approved_by' => Auth::user()->user_name ?? Auth::user()->user_email,
                    ]);
                
                $attendanceRecord = DB::table('staff_attendance')
                    ->where('staff_id', $attendance->staff_id)
                    ->whereDate(
                        DB::raw('COALESCE(time_in, time_out)'),
                        $attendance->request_date
                    )
                    ->first();

                if ($attendanceRecord) {
                    $update = ['address' => $attendance->address];
                    if (!empty($attendance->request_time_in) && empty($attendanceRecord->time_in)) {
                        $update['time_in'] = $attendance->request_time_in;
                    }
                    if (!empty($attendance->request_time_out)) {
                        $update['time_out'] = $attendance->request_time_out;
                    }

                    DB::table('staff_attendance')
                        ->where('id_staff_attendance', $attendanceRecord->id_staff_attendance)
                        ->update($update);
                } else {
                    if (!empty($attendance->request_time_in)) {
                        DB::table('staff_attendance')->insert([
                            'staff_id'    => $attendance->staff_id,
                            'business_id' => $attendance->business_id,
                            'time_in'     => $attendance->request_time_in,
                            'time_out'    => $attendance->request_time_out,
                            'address'     => $attendance->address,
                        ]);
                    }
                }
            } else if ($status === 'rejected') {
                DB::table('attendance_request')
                    ->where('id_attendance_request', $id)
                    ->update([
                        'request_status' => $status
                    ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully'
            ]);

        } catch (\Exception $e) {
            \Log::error('Error updating attendance status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status'
            ], 500);
        }
    }

    public function getAttendanceRequest($id)
    {
        try {
            $attendance = DB::table('attendance_request as ar')
                ->select('ar.*', 'hs.staff_fullname')
                ->join('staff as hs', 'hs.id_staff', '=', 'ar.staff_id')
                ->where('ar.id_attendance_request', $id)
                ->first();

            if (!$attendance) {
                return response()->json(['error' => 'Record not found'], 404);
            }

            return response()->json([
                'id_attendance_request' => $attendance->id_attendance_request,
                'staff_id' => $attendance->staff_id,
                'staff_fullname' => $attendance->staff_fullname,
                'staff_name' => $attendance->staff_fullname,
                'business_id' => $attendance->business_id,
                'request_date' => $attendance->request_date,
                'request_time_in' => $attendance->request_time_in,
                'request_time_out' => $attendance->request_time_out,
                'address' => $attendance->address,
                'request_status' => $attendance->request_status,
                'approved_by' => $attendance->approved_by,
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching attendance request: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch data'], 500);
        }
    }


      // ==================== QR CODE METHODS (Using key_code table) ====================
    
    /**Main attendance marking page*/
    public function markAttendance()
    {
        $businessId = session('business_id');
        $keyCode = $this->generateRandomString(8);
        $this->updateKeyCodeTable($businessId, $keyCode);
        
        $qrData = $keyCode;
        
        // Generate QR code as HTML string
        $qrCode = QrCode::size(180)->margin(1)->errorCorrection('H')->generate($qrData);
        
        // Convert to string if it's an object
        if (is_object($qrCode) && method_exists($qrCode, 'toHtml')) {
            $qrCode = $qrCode->toHtml();
        } elseif (is_object($qrCode) && method_exists($qrCode, '__toString')) {
            $qrCode = $qrCode->__toString();
        }
        
        return view('hrm.mark_attendance', compact('qrCode', 'keyCode'));
    }
    
    /** Update QR key in key_code table*/
    private function updateKeyCodeTable($businessId, $key)
    {
        try {
            $systemUrl = env('APP_URL');
            // Check if record exists for this business
            $existingRecord = DB::table('key_code')
                ->where('business_id', $businessId)
                ->first();
            
            if ($existingRecord) {
                // Update existing record
                DB::table('key_code')
                    ->where('id_key_code', $existingRecord->id_key_code)
                    ->update([
                        'current_key' => $key,
                        'created_date' => Carbon::now(),
                        'serverurl' => rtrim($systemUrl, '/')
                    ]);
            } else {
                // Insert new record
                DB::table('key_code')->insert([
                    'business_id' => $businessId,
                    'current_key' => $key,
                    'created_date' => Carbon::now(),
                    'serverurl' => rtrim($systemUrl, '/')
                ]);
            }
            
            return true;
            
        } catch (\Exception $e) {
            \Log::error('Error updating key_code table: ' . $e->getMessage());
            return false;
        }
    }
    
    /** Generate random string for key*/
    private function generateRandomString($length = 8)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $randomString = '';
        
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        return $randomString;
    }
    
    /**Fingerprint matching API*/
    public function matchFingerprint(Request $request)
    {
        try {
            $scannedTemplate = $request->template_base64;
            $qualityThreshold = $request->quality ?? 50;

            if (empty($scannedTemplate)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No fingerprint template provided'
                ]);
            }

            // Get staff fingerprints for current business
            $staffList = DB::table('staff')
                ->select(
                    'id_staff',
                    'staff_firstname',
                    'staff_cell',
                    'fcm_id',
                    'staff_fullname',
                    'staff_fingerprint_template_base64'
                )
                ->where('business_id', session('business_id'))
                ->whereNotNull('staff_fingerprint_template_base64')
                ->where('staff_active', 'Y')
                ->get();

            // For simulation - match with first staff
            // In real implementation, compare with all templates
            if ($staffList->count() > 0) {
                $firstStaff = $staffList->first();
                
                // Return matched staff
                return response()->json([
                    'success' => true,
                    'staff_id' => $firstStaff->id_staff,
                    'staff_name' => $firstStaff->staff_firstname,
                    'staff_cell' => $firstStaff->staff_cell,
                    'fcm_id' => $firstStaff->fcm_id,
                    'match_score' => 95,
                    'message' => 'Fingerprint matched successfully'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'No registered fingerprints found'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error matching fingerprint: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /** Mark attendance via fingerprint*/
    public function markAttendanceFingerprint(Request $request)
    {
        $request->validate([
            'staff_id' => 'required|integer',
            'attendance_date' => 'required|date',
            'attendance_time' => 'required',
            'latitude' => 'nullable',
            'longitude' => 'nullable',
            'address' => 'nullable'
        ]);
        
        try {
            $staffId = $request->staff_id;
            $businessId = session('business_id');
            
            // Get staff details
            $staff = DB::table('staff')
                ->where('id_staff', $staffId)
                ->where('business_id', $businessId)
                ->first();

            if (!$staff) {
                return response()->json([
                    ['status' => false, 'message' => 'Staff not found']
                ]);
            }
            
            $locationData = [
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'address' => $request->address
            ];
            
            $result = $this->markAttendanceProcess($staffId, 'fingerprint', $locationData);
            
            if ($result['success']) {
                return response()->json([
                    ['status' => true, 'message' => $result['message']]
                ]);
            } else {
                return response()->json([
                    ['status' => false, 'message' => $result['message']]
                ]);
            }
            
        } catch (\Exception $e) {
            return response()->json([
                ['status' => false, 'message' => 'Error: ' . $e->getMessage()]
            ], 500);
        }
    }
    
    /**Mark attendance process*/
    private function markAttendanceProcess($staffId, $method, $locationData = [])
    {
        $businessId = session('business_id');
        $now = Carbon::now();
        $today = $now->toDateString();
        
        // Check existing attendance for today
        $existingAttendance = DB::table('staff_attendance')
            ->where('staff_id', $staffId)
            ->where('business_id', $businessId)
            ->whereDate('time_in', $today)
            ->first();
        
        $staff = DB::table('staff')->where('id_staff', $staffId)->first();
        
        if (!$staff) {
            return [
                'success' => false,
                'message' => 'Staff not found'
            ];
        }
        
        // Prepare location data
        $latitude = isset($locationData['latitude']) ? (float)$locationData['latitude'] : null;
        $longitude = isset($locationData['longitude']) ? (float)$locationData['longitude'] : null;
        $address = $locationData['address'] ?? null;
        
        if ($existingAttendance) {
            // Check if already completed attendance
            if ($existingAttendance->time_out) {
                return [
                    'success' => false,
                    'message' => 'Attendance already completed for today',
                    'type' => 'already_completed'
                ];
            }
            
            // Update time_out with location
            DB::table('staff_attendance')
                ->where('id_staff_attendance', $existingAttendance->id_staff_attendance)
                ->update([
                    'time_out' => $now,
                    'staff_available' => 'Y',
                    'address' => $address ?? $existingAttendance->address,
                    'latitude' => $latitude ?? $existingAttendance->latitude,
                    'longitude' => $longitude ?? $existingAttendance->longitude
                ]);
            
            return [
                'success' => true,
                'message' => 'Time out recorded for ' . $staff->staff_firstname . ' at ' . $now->format('h:i A'),
                'type' => 'time_out',
                'staff_name' => $staff->staff_firstname,
                'time' => $now->format('h:i A')
            ];
        } else {
            // Insert new attendance with location
            DB::table('staff_attendance')->insert([
                'business_id' => $businessId,
                'staff_id' => $staffId,
                'time_in' => $now,
                'staff_available' => 'Y',
                'address' => $address,
                'latitude' => $latitude,
                'longitude' => $longitude
            ]);
            
            return [
                'success' => true,
                'message' => 'Time in recorded for ' . $staff->staff_firstname . ' at ' . $now->format('h:i A'),
                'type' => 'time_in',
                'staff_name' => $staff->staff_firstname,
                'time' => $now->format('h:i A')
            ];
        }
    }
    
    /** Get current QR key for refresh functionality*/
    public function refreshQR()
    {
        try {
            $businessId = session('business_id');
            
            // Generate new random key
            $keyCode = $this->generateRandomString(8);
            
            // Update key_code table
            $this->updateKeyCodeTable($businessId, $keyCode);
            
            // Generate QR data: business_id + key_code
            $qrData = $keyCode;
            
            // Generate QR code HTML
            $qrCodeHtml = QrCode::size(180)->margin(1)->errorCorrection('H')->generate($qrData);
            
            // Convert to string if it's an object
            if (is_object($qrCodeHtml) && method_exists($qrCodeHtml, 'toHtml')) {
                $qrCodeHtml = $qrCodeHtml->toHtml();
            } elseif (is_object($qrCodeHtml) && method_exists($qrCodeHtml, '__toString')) {
                $qrCodeHtml = $qrCodeHtml->__toString();
            }
            
            return response()->json([
                'success' => true,
                'qr_code' => $qrCodeHtml,
                'key_code' => $keyCode
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to refresh QR code'
            ], 500);
        }
    }
}