<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Booking;
use App\Models\BookingVisit;
use App\Models\VisitAdvance;
use App\Models\CustomerVisits;
use App\Models\VisitServices;
use App\Models\PackageType;
use App\Models\Staff;
use App\Models\CategoryMapping;
use App\Models\CategoryMappingPackage;
use App\Models\MappingFloor;
use App\Models\BusinessTimeslot;
use App\Models\PricingBand;
use App\Models\PackageCategory;
use App\Models\VisitServiceStaff;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Bus;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class BookingController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function bookings(Request $request)
    {
        $business_id = session('business_id');
        if($request->business_id){
            $business_id = $request->business_id;
        }

        $business = Business::find($business_id);

         if(session('ho') == "Yes"){
            $businesses = Business::select('id_business', 'business_name')
            ->where('business.common_services', "Yes")->get();
        } else {
            $businesses = Business::select('id_business', 'business_name')
            ->where('business.id_business', $business_id)->get();            
        }

        $start_date = $request->start_date ?? date('Y-m-d'); //today's date
        $end_date   = $request->end_date   ?? date('Y-m-d', strtotime('+1 day')); //tommorrow date

        $agents = DB::table('users')
            ->select('id_users', 'user_name')
            ->where('business_id', $business_id)
            ->where('user_status', 'Active')
            ->get();

        

        return view('visits.bookings.bookings', compact('business', 'businesses', 'agents', 'start_date', 'end_date'));
    }


    public function bookingsdata(Request $request)
    {
        $mDate = $request->start_date ?? '2023-01-01';
        $today = Carbon::today()->toDateString();
        $businessId = $request->business_id ?? session('business_id');

        // ---- Subquery: package service rate ----
        $packageServiceRate = DB::table('visit_services')
            ->join('package_services', function ($join) {
                $join->on('visit_services.service_id', '=', 'package_services.service_id')
                    ->on('visit_services.id_service_category', '=', 'package_services.package_category_id');
            })
            ->where('service_name', 'not like', '%(nc)')
            ->groupBy('customer_visit_id')
            ->select(
                'customer_visit_id',
                DB::raw('SUM(visit_services.s_rate) as rate')
            );

        // ---- Subquery: visit advance ----
        $visitAdvance = DB::table('customer_visits')
            ->leftJoin('visit_advance', 'customer_visits.id_customer_visits', '=', 'visit_advance.customer_visit_id')
            ->groupBy('id_customer_visits')
            ->where('advance_status', 'Active')
            ->where('customer_visits.visit_status', 'open')
            ->select(
                'id_customer_visits as customer_visit_id',
                DB::raw('SUM(visit_advance.advance_amount) as advance_amount')
            );

        // ---- Main Query ----
        $query = DB::table('customer_visits')
        ->join('booking_visits', 'booking_visits.visit_id', '=', 'customer_visits.id_customer_visits')
        ->join('bookings', 'bookings.id_bookings', '=', 'booking_visits.booking_id')
        ->join('visit_services', function ($join) {
            $join->on('customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
                ->where('visit_service_status', 'Active');
        })
        ->join('business_services', 'business_services.id_business_services', '=', 'visit_services.service_id')
        ->join('package_category as pc', 'pc.id_package_category', '=', 'visit_services.id_service_category')
        ->join('package_type as pt', 'pt.id_package_type', '=', 'pc.package_type_id')
        ->join('customers', 'customers.id_customers', '=', 'customer_visits.customer_id')
        ->leftJoinSub($packageServiceRate, 'ps', function ($join) {
            $join->on('ps.customer_visit_id', '=', 'visit_services.customer_visit_id');
        })
        ->leftJoinSub($visitAdvance, 'va', function ($join) {
            $join->on('customer_visits.id_customer_visits', '=', 'va.customer_visit_id');
        })
        ->where('customer_visits.visit_status', 'open')
        ->where('visit_services.service_flag', 'packagetype')
        ->where('customer_visits.business_id', $businessId)
        ->when($request->filled('user_name') && $request->user_name !== 'all', function ($q) use ($request) {
            $q->where('customer_visits.created_by', $request->user_name);
        })
        ->when($mDate, function ($q) use ($mDate) {
            $q->whereDate('visit_services.visit_service_start',  '>=',  $mDate);
        }, function ($q) use ($today) {
            $q->whereDate('visit_services.visit_service_start', '>=', $today);
        })
        ->groupBy(
            'bookings.id_bookings',
            'customer_visits.id_customer_visits',
            'bookings.booking_date',
            'customer_visits.created_by',
            'customer_name',
            'customer_cell',
            'service_type',
            'service_category',
            
            'customer_visits.visit_status',
            'customer_visits.discount',
            
            'va.advance_amount'
        )
        ->select([
            // IDs
            'bookings.id_bookings',
            'customer_visits.id_customer_visits',

            // Dates
            DB::raw("DATE_FORMAT(visit_services.visit_service_start, '%d-%b-%Y %h:%i %p') as visit_date"),
            'bookings.booking_date',

            // User
            'customer_visits.created_by',

            // Customer
            'customer_name',
            'customer_cell',

            // Service
            'service_type',
            'service_category',
            'business_services.service_name',

            // Status
            'customer_visits.visit_status',

            // Money
            DB::raw('IFNULL(ps.rate,0) as service_rate'),
            'customer_visits.discount',
            DB::raw('IFNULL(va.advance_amount,0) as advance_amount'),
        ]);
        
        return DataTables::of($query)
        ->filterColumn('customer_name', fn ($q, $v) => $q->where('customer_name', 'like', "%$v%"))
        ->filterColumn('customer_cell', fn ($q, $v) => $q->where('customer_cell', 'like', "%$v%"))
        ->orderColumn('visit_date', fn ($q, $order) =>
            $q->orderBy('visit_services.visit_service_start', $order)
        )
        ->make(true);

    }

    public function print_booking_receipt($id_bookings)
    {
        
        $booking = Booking::where('id_bookings', $id_bookings)
            ->first();

        if (!$booking) {
            return redirect()->back()->with('error', 'Booking not found.');
        }

        $bookingVisits = BookingVisit::where('booking_id', $id_bookings)            
            ->join('customer_visits', 'customer_visits.id_customer_visits', '=', 'booking_visits.visit_id')
            ->join('visit_services', function($join) {
                $join->on('visit_services.customer_visit_id', '=', 'booking_visits.visit_id')
                ->on('visit_services.id_service_category', "=", 'booking_visits.category_id')
                    ->where('visit_services.visit_service_status', 'Active');
            })
            ->join('customers', 'customers.id_customers', '=', 'customer_visits.customer_id')
            ->join('bookings', 'bookings.id_bookings', '=', 'booking_visits.booking_id')
            ->select(
                'booking_visits.*', 'bookings.*',
                'visit_services.s_type as service_type',
                'visit_services.s_category as service_category',
                DB::raw('GROUP_CONCAT(visit_services.service_name) as service_name'),
                DB::raw('ROUND(SUM(visit_services.s_rate), 2) as service_rate'),
                DB::raw('date_format(visit_services.visit_service_start, "%d-%m-%Y %H:%i") as visit_date'),
                DB::raw('date_format(bookings.booking_date, "%D %b, %Y") as booking_date'),
                DB::raw('date_format(bookings.event_date, "%D %b, %Y") as event_date'),
                DB::raw('date_format(MIN(visit_services.visit_service_start), "%D %b, %Y") as arrival_date'),
                DB::raw("DATE_FORMAT(
                    DATE_SUB(MIN(visit_services.visit_service_start), INTERVAL 30 MINUTE),
                    '%h:%i %p'
                ) AS arrival_time"),
                DB::raw('date_format(Max(visit_services.visit_service_end), "%h:%i %p") as departure_time'),
                'customers.customer_name',
                'customers.customer_cell',                
                'customer_visits.discount',
                'customer_visits.created_by'
            )
            ->groupBy('id_bookings','booking_visits.id_booking_visits', 'customer_visits.id_customer_visits', 'visit_services.s_category', 'visit_services.s_type')
            
            ->orderBy('visit_services.visit_service_start', 'asc')
            ->get();
            //return query and checking data
            //dd($bookingVisits);
           // return $bookingVisits;
        $visitAdvance = VisitAdvance::select(
                'advance_mode as payment_mode',
                DB::raw('SUM(ifnull(advance_amount,0)) as total_advance'),                
                )->whereIn('customer_visit_id', function($query) use ($id_bookings) {
                $query->select('visit_id')
                      ->from('booking_visits')
                      ->where('booking_id', $id_bookings);
            })
            ->where('advance_status', 'Active')
            ->groupBy('customer_visit_id')
            ->get();
          

        $business = Business::select('business_name', 'business_address', 'business_phone1', 'business_email', 'business_logo', 'fine_print')
        ->where('id_business', $booking->business_id)->first();
        

        

        return view('visits.bookings.booking_receipt', compact('booking', 'bookingVisits', 'visitAdvance', 'business'));
    }

    public function new_booking(Request $request)
    {
        $business_id = session('business_id');

        if($request->business_id){
            $business_id = $request->business_id;
        }

        $business = Business::select('id_business', 'business_name', 'business_logo', 'flexi_pricing_enabled', 'business_opening_time', 'business_closing_time')
            ->where('id_business', $business_id)->first();

        if(session('ho')=="Yes"){
            $branches = Business::select('id_business', 'business_name', 'flexi_pricing_enabled')
            ->where('business.common_services', "Yes")->get();
            
        }else {
            $branches = collect([$business]);
        }

        $staff = Staff::select('id_staff', 'staff_fullname', 'staff_image')
            ->where('business_id', $business_id)
            ->where('staff_scheduler', 'On')
            ->where('staff_active', 'Y')
            ->get();


        if(session('common_services') == "Yes"){
            $package_business_id = Business::where('ho', "Yes")
                ->value('id_business');
        }else {
            $package_business_id = $business_id;
        }

        $packages = PackageType::select('package_type.id_package_type', 'service_type', 
        #DB::RAW('ROUND(SUM(service_rate), 2) AS package_price')
        DB::raw('ROUND(SUM(COALESCE(package_services_business_mapping.price, package_services.service_rate)), 2) as package_price')
        )        
        ->join('package_category', 'package_category.package_type_id', '=', 'package_type.id_package_type')
        ->join('package_services', 'package_services.package_category_id', '=', 'package_category.id_package_category')
        ->leftJoin('package_services_business_mapping', function ($join) {
                $join->on(
                        'package_services_business_mapping.package_services_id',
                        '=',
                        'package_services.id_package_services'
                    )
                    ->where(
                        'package_services_business_mapping.business_id',
                        session('business_id')
                    );
        })
        ->where(function ($query) {
                                $query->whereNull('package_services_business_mapping.is_visible')
                                    ->orWhere('package_services_business_mapping.is_visible', 'Yes');
        })
        ->where('package_type.business_id', $package_business_id)
        ->where('service_type_active', 'Yes')
        ->where('service_category_active', 'Yes')
        ->where('package_services_active', 'Yes')
        ->whereRaw('IFNULL(package_services.option_for, 0) = 0') // only main services
        ->where('expiry_date', '>=', date('Y-m-d'))
        ->groupBy('package_type.id_package_type')
        ->get();

        return view('visits.bookings.new_booking', compact('business', 'business_id',  'branches', 'packages', 'staff'));
    }

    /**
     * Get package categories for a selected package
     * 
     * Returns all categories in the package with their services and rates.
     * If flexi-pricing is enabled, also includes mapping information to show
     * which categories have flexi-pricing offers available.
     * 
     * @param Request $request
     * @param int $package_id Package type ID
     * @return \Illuminate\Http\JsonResponse
     */
    public function get_package_categories(Request $request, $package_id)
    {
        try {
            $package_id = $package_id ?: $request->input('package_id');
            $business_id = session('business_id');

            if (!$package_id) {
                return response()->json(['error' => 'Package ID is required'], 400);
            }

            $business = Business::find($business_id);
            $flexiPricingEnabled = $business && $business->flexi_pricing_enabled === 'Yes';

            // Fetch main services only (exclude option_for)
            $packageDetails = PackageType::select(
                    'package_type.id_package_type',
                    'service_type',
                    'package_category.id_package_category',
                    'package_category.service_category',
                    'service_category_image',
                    DB::raw('GROUP_CONCAT(business_services.service_name SEPARATOR "<br>") as service_names'),
                    DB::raw('GROUP_CONCAT(package_services.service_id) as service_ids'),
                    DB::raw('GROUP_CONCAT(package_services.id_package_services) as package_service_ids'),
                    DB::raw('GROUP_CONCAT(COALESCE(package_services_business_mapping.price, package_services.service_rate) SEPARATOR "<br>") as service_rates'),
                    DB::raw('ROUND(SUM(COALESCE(package_services_business_mapping.price, package_services.service_rate)), 2) as category_price')
                )
                ->join('package_category', 'package_category.package_type_id', '=', 'package_type.id_package_type')
                ->join('package_services', 'package_services.package_category_id', '=', 'package_category.id_package_category')
                ->join('business_services', 'business_services.id_business_services', '=', 'package_services.service_id')
                ->leftJoin('package_services_business_mapping', function ($join) use ($business_id) {
                    $join->on('package_services_business_mapping.package_services_id', '=', 'package_services.id_package_services')
                        ->where('package_services_business_mapping.business_id', $business_id);
                })
                ->where('package_type.id_package_type', $package_id)
                ->where('service_type_active', 'Yes')
                ->where('service_category_active', 'Yes')
                ->where('package_services_active', 'Yes')
                ->whereRaw('IFNULL(package_services.option_for, 0) = 0') // only main services
                ->groupBy('package_category.id_package_category')
                ->where(function ($query) {
                    $query->whereNull('package_services_business_mapping.is_visible')
                        ->orWhere('package_services_business_mapping.is_visible', 'Yes');
                })
                ->get();

            // Convert models to arrays and attach options for each main service
            $packageDetails = $packageDetails->map(function($category) use ($business_id) {
                $categoryArray = $category->toArray();

                $mainServiceIds = array_map('trim', explode(',', $categoryArray['package_service_ids']));
                $categoryArray['options'] = [];

                foreach ($mainServiceIds as $mainServiceId) {
                    $options = DB::table('package_services')
                        ->select(
                            'package_services.id_package_services',
                            'package_services.service_id',
                            'business_services.service_name',
                            DB::raw('COALESCE(package_services_business_mapping.price, package_services.service_rate) as service_rate')
                        )
                        ->join('business_services', 'business_services.id_business_services', '=', 'package_services.service_id')
                        ->leftJoin('package_services_business_mapping', function ($join) use ($business_id) {
                            $join->on('package_services_business_mapping.package_services_id', '=', 'package_services.id_package_services')
                                ->where('package_services_business_mapping.business_id', $business_id);
                        })
                        ->where('package_services.option_for', $mainServiceId)
                        ->where('package_services.package_services_active', 'Yes')
                        ->where(function ($query) {
                            $query->whereNull('package_services_business_mapping.is_visible')
                                ->orWhere('package_services_business_mapping.is_visible', 'Yes');
                        })
                        ->orderBy('package_services.id_package_services')
                        ->get();

                    $categoryArray['options'][$mainServiceId] = $options;
                }

                return $categoryArray;
            }); 

            // Add flexi-pricing mapping info
            if ($flexiPricingEnabled) {
                $packageDetails = $packageDetails->map(function($category) use ($business_id) {
                    try {
                        $mapping = CategoryMappingPackage::where('package_category_id', $category['id_package_category'])
                            ->whereHas('categoryMapping', function($query) use ($business_id) {
                                $query->where('business_id', $business_id)
                                    ->where('active', 1);
                            })
                            ->with('categoryMapping')
                            ->first();

                        $category['has_mapping'] = $mapping ? true : false;
                        $category['mapping_id'] = $mapping && $mapping->categoryMapping ? $mapping->categoryMapping->id_mapping : null;
                    } catch (\Exception $e) {
                        $category['has_mapping'] = false;
                        $category['mapping_id'] = null;
                        Log::warning('Error checking mapping for category ' . $category['id_package_category'] . ': ' . $e->getMessage());
                    }

                    return $category;
                });
            } else {
                $packageDetails = $packageDetails->map(function($category) {
                    $category['has_mapping'] = false;
                    $category['mapping_id'] = null;
                    return $category;
                });
            }

            return response()->json($packageDetails);

        } catch (\Exception $e) {
            Log::error('Error in get_package_categories: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'error' => 'Failed to fetch package categories',
                'message' => $e->getMessage()
            ], 500);
        }
    } 
    public function get_package_categoriesOLD(Request $request, $package_id)
    {
        try {
            // Use route parameter, fallback to request if needed
            $package_id = $package_id ?: $request->input('package_id');
            $business_id = session('business_id');

            if (!$package_id) {
                return response()->json(['error' => 'Package ID is required'], 400);
            }

            // Check if flexi-pricing is enabled for this business
            $business = Business::find($business_id);
            $flexiPricingEnabled = $business && $business->flexi_pricing_enabled == 'Yes';

            $packageDetails = PackageType::select(
                    'package_type.id_package_type',
                    'service_type',
                    'package_category.id_package_category',
                    'package_category.service_category',
                    'service_category_image',
                    DB::RAW('GROUP_CONCAT(business_services.service_name SEPARATOR "<br>") as service_names'),
                    DB::RAW('GROUP_CONCAT(package_services.service_id) as service_ids'),
                    #DB::RAW('GROUP_CONCAT(package_services.service_rate SEPARATOR "<br>") as service_rates'),
                    #DB::RAW('ROUND(SUM(package_services.service_rate), 2) AS category_price')
                    DB::raw('GROUP_CONCAT(COALESCE(package_services_business_mapping.price, package_services.service_rate) SEPARATOR "<br>") as service_rates'),
                    DB::raw('ROUND(SUM(COALESCE(package_services_business_mapping.price, package_services.service_rate)), 2) as category_price')

                )
                ->join('package_category', 'package_category.package_type_id', '=', 'package_type.id_package_type')
                ->join('package_services', 'package_services.package_category_id', '=', 'package_category.id_package_category')
                ->join('business_services', 'business_services.id_business_services', '=', 'package_services.service_id')
                ->leftJoin('package_services_business_mapping', function ($join) {
                                $join->on(
                                        'package_services_business_mapping.package_services_id',
                                        '=',
                                        'package_services.id_package_services'
                                    )
                                    ->where(
                                        'package_services_business_mapping.business_id',
                                        session('business_id')
                                    );
                })
                ->where('package_type.id_package_type', $package_id)
                ->where('service_type_active', 'Yes')
                ->where('service_category_active', 'Yes')
                ->where('package_services_active', 'Yes')
                ->groupBy('package_category.id_package_category')
                ->where(function ($query) {
                                $query->whereNull('package_services_business_mapping.is_visible')
                                    ->orWhere('package_services_business_mapping.is_visible', 'Yes');
                })
                ->get();

            // Add flexi-pricing mapping info if enabled
            // This allows the frontend to show "Offers" button for categories with mappings
            if ($flexiPricingEnabled) {
                foreach ($packageDetails as $category) {
                    try {
                        // Check if this category has an active mapping
                        $mapping = CategoryMappingPackage::where('package_category_id', $category->id_package_category)
                            ->whereHas('categoryMapping', function($query) use ($business_id) {
                                $query->where('business_id', $business_id)
                                      ->where('active', 1);
                            })
                            ->with('categoryMapping')
                            ->first();
                        
                        $category->has_mapping = $mapping ? true : false;
                        $category->mapping_id = $mapping && $mapping->categoryMapping ? $mapping->categoryMapping->id_mapping : null;
                    } catch (\Exception $e) {
                        // If there's an error checking mapping, default to no mapping
                        $category->has_mapping = false;
                        $category->mapping_id = null;
                        Log::warning('Error checking mapping for category ' . $category->id_package_category . ': ' . $e->getMessage());
                    }
                }
            } else {
                // Ensure has_mapping is set even when flexi-pricing is disabled
                foreach ($packageDetails as $category) {
                    $category->has_mapping = false;
                    $category->mapping_id = null;
                }
            }

            return response()->json($packageDetails);
        } catch (\Exception $e) {
            Log::error('Error in get_package_categories: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'error' => 'Failed to fetch package categories',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calculate flexi-pricing discounts for a package category
     * 
     * Returns discount offers for a 7-day window (3 days before, selected day, 3 days after).
     * Discounts are calculated based on:
     * - Current booking capacity percentage
     * - Day of the week
     * - Time slot (if configured)
     * - Floor assignments
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function get_flexi_price(Request $request)
    {
        try {
            $event_date = $request->input('event_date');
            $package_category_id = $request->input('package_category_id');
            $business_id = session('business_id');

            // Validate inputs
            if (empty($event_date) || empty($package_category_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Event date and package category ID are required'
                ], 400);
            }

            // Check if flexi-pricing is enabled for this business
            $business = Business::find($business_id);
            if (!$business || $business->flexi_pricing_enabled != 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Flexi-pricing is not enabled for this business'
                ], 400);
            }

            $event_date_only = date('Y-m-d', strtotime($event_date));

            // Find the active mapping for this package category
            $mappingPackage = CategoryMappingPackage::where('package_category_id', $package_category_id)
                ->whereHas('categoryMapping', function($query) use ($business_id) {
                    $query->where('business_id', $business_id)
                          ->where('active', 1);
                })
                ->with('categoryMapping')
                ->first();

            if (!$mappingPackage) {
                return response()->json([
                    'success' => false,
                    'message' => 'No active mapping found for this package category'
                ], 404);
            }

            $mapping = $mappingPackage->categoryMapping;
            
            if (!$mapping || $mapping->active != 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category mapping not found or inactive'
                ], 404);
            }

            // Get floors assigned to this mapping (used to calculate capacity)
            $mappingFloors = MappingFloor::where('mapping_id', $mapping->id_mapping)
                ->where('active', 1)
                ->with('floor')
                ->get();

            if ($mappingFloors->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No floors assigned to this mapping'
                ], 404);
            }

            // Extract floor details and calculate total capacity
            $floors = $mappingFloors->map(function($mappingFloor) {
                return [
                    'id' => $mappingFloor->floor->id_business_floors,
                    'name' => $mappingFloor->floor->floor_name,
                    'capacity' => $mappingFloor->floor->floor_workstations ?? 0
                ];
            });
            $total_capacity = $floors->sum('capacity');

            // Check if mapping uses timeslot-specific bands (vs all-day bands)
            $hasTimeslotBands = PricingBand::where('mapping_id', $mapping->id_mapping)
                ->whereNotNull('timeslot_id')
                ->where('active', 1)
                ->exists();
            
            // Get business timeslots only if mapping has timeslot-specific bands
            $timeslots = collect([]);
            if ($hasTimeslotBands) {
                $timeslots = BusinessTimeslot::where('business_id', $business_id)
                    ->where('active', 1)
                    ->orderBy('order')
                    ->get();
            }

            // Get all package category IDs in this mapping (to count bookings across all categories)
            $packageCategoryIds = CategoryMappingPackage::where('mapping_id', $mapping->id_mapping)
                ->pluck('package_category_id')
                ->toArray();
            $floorIds = $floors->pluck('id')->toArray();

            // Generate 7-day window: 3 days before, selected day, 3 days after
            $dateRange = [];
            for ($i = -3; $i <= 3; $i++) {
                $dateRange[] = date('Y-m-d', strtotime($event_date_only . " {$i} days"));
            }

            $discounts_by_date = [];

            // Calculate discounts for each day in the 7-day window
            foreach ($dateRange as $date) {
                // Convert day of week: PHP date('w') returns 0=Sunday, 6=Saturday
                // We need 1=Sunday, 2=Monday, ..., 7=Saturday (MariaDB format)
                $dayOfWeek = date('w', strtotime($date));
                $dayNumber = $dayOfWeek == 0 ? 1 : $dayOfWeek + 1;

                // Count existing bookings for this date, mapping, and floors
                $bookingsQuery = VisitServices::select(
                        DB::raw('DATE(visit_service_start) as visit_date'),
                        DB::raw('TIME(visit_service_start) as visit_time'),
                        'visit_services.customer_visit_id',
                        'visit_services.id_service_category',
                        'visit_service_staffs.staff_id',
                        'staff.floor_id'
                    )
                    ->join('customer_visits', 'customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
                    ->leftJoin('visit_service_staffs', function($join) {
                        $join->on('visit_service_staffs.visit_service_id', '=', 'visit_services.id_visit_services')
                            ->where('visit_service_staffs.visit_service_staff_status', '=', 'Active');
                    })
                    ->leftJoin('staff', 'staff.id_staff', '=', 'visit_service_staffs.staff_id')
                    ->whereIn('visit_services.id_service_category', $packageCategoryIds)
                    ->where('visit_services.service_flag', 'packagetype')
                    ->where('visit_services.business_id', $business_id)
                    ->where('visit_services.visit_service_status', 'Active')
                    ->where('customer_visits.visit_status', 'open')
                    ->whereDate('visit_services.visit_service_start', $date);

                // Filter by floor - either staff.floor_id matches or no staff assigned (use mapping floor)
                $bookingsQuery->where(function($query) use ($floorIds) {
                    $query->whereIn('staff.floor_id', $floorIds)
                        ->orWhereNull('staff.floor_id');
                });

                $bookings = $bookingsQuery->get();

                // Process discounts by timeslot if mapping uses timeslot-specific bands
                if ($timeslots->isNotEmpty()) {
                    foreach ($timeslots as $timeslot) {
                        // Filter bookings that fall within this timeslot
                        $timeslotBookings = $bookings->filter(function($booking) use ($timeslot) {
                            $visitTime = strtotime($booking->visit_time);
                            $startTime = strtotime($timeslot->start_time);
                            $endTime = strtotime($timeslot->end_time);
                            return $visitTime >= $startTime && $visitTime < $endTime;
                        });

                        // Calculate booking capacity percentage for this timeslot
                        $bookingCount = $timeslotBookings->unique('customer_visit_id')->count();
                        $percentage = $total_capacity > 0 ? ($bookingCount / $total_capacity) * 100 : 0;

                        // Find matching pricing band (inclusive range: min <= percentage <= max)
                        $band = PricingBand::where('mapping_id', $mapping->id_mapping)
                            ->where('day', $dayNumber)
                            ->where('timeslot_id', $timeslot->id_business_timeslots)
                            ->where('active', 1)
                            ->where('min_percentage', '<=', $percentage)
                            ->where('max_percentage', '>=', $percentage)
                            ->first();

                        if (!isset($discounts_by_date[$date])) {
                            $discounts_by_date[$date] = [
                                'date' => $date,
                                'day_name' => date('l', strtotime($date)),
                                'time_slots' => []
                            ];
                        }

                        // Add timeslot entry with band or default to 0 discount
                            $discounts_by_date[$date]['time_slots'][] = [
                                'timeslot_name' => $timeslot->timeslot_name,
                                'timeslot_id' => $timeslot->id_business_timeslots,
                            'bookings' => $bookingCount,
                            'capacity' => $total_capacity,
                            'percentage' => round($percentage, 2),
                            'band' => $band ? [
                                'min_percentage' => $band->min_percentage,
                                'max_percentage' => $band->max_percentage,
                                'discount_type' => $band->discount_type,
                                'discount' => $band->discount,
                            ] : [
                                'min_percentage' => 0,
                                'max_percentage' => 100,
                                'discount_type' => 'percentage',
                                'discount' => 0,
                            ]
                        ];
                    }
                } else {
                    // No timeslots configured - use all-day bands
                    $bookingCount = $bookings->unique('customer_visit_id')->count();
                    $percentage = $total_capacity > 0 ? ($bookingCount / $total_capacity) * 100 : 0;

                    // Find matching all-day pricing band (timeslot_id is null)
                    $band = PricingBand::where('mapping_id', $mapping->id_mapping)
                        ->where('day', $dayNumber)
                        ->whereNull('timeslot_id')
                        ->where('active', 1)
                        ->where('min_percentage', '<=', $percentage)
                        ->where('max_percentage', '>=', $percentage)
                        ->first();

                    // Always return discount entry (defaults to 0% if no band found)
                    $discounts_by_date[$date] = [
                        'date' => $date,
                        'day_name' => date('l', strtotime($date)),
                        'bookings' => $bookingCount,
                        'capacity' => $total_capacity,
                        'percentage' => round($percentage, 2),
                        'band' => $band ? [
                            'min_percentage' => $band->min_percentage,
                            'max_percentage' => $band->max_percentage,
                            'discount_type' => $band->discount_type,
                            'discount' => $band->discount,
                        ] : [
                            'min_percentage' => 0,
                            'max_percentage' => 100,
                            'discount_type' => 'percentage',
                            'discount' => 0,
                        ]
                    ];
                }
            }

            // Get package category info and calculate its total price
            // This is needed to calculate the discounted price in the frontend
            $packageCategory = PackageCategory::find($package_category_id);
            $packageCategoryName = $packageCategory ? $packageCategory->service_category : 'Unknown';
            
            // Calculate total price for this category (sum of all service rates)
            $packageCategoryPrice = 0;
            if ($packageCategory) {
                $packageCategoryPrice = DB::table('package_services')
                    ->where('package_category_id', $package_category_id)
                    ->where('package_services_active', 'Yes')
                    ->sum('service_rate');
            }

            return response()->json([
                'success' => true,
                'mapping_name' => $mapping->mapping_name,
                'package_category_name' => $packageCategoryName,
                'package_category_price' => $packageCategoryPrice,
                'event_date' => $event_date_only,
                'floors' => $floors,
                'timeslots' => $timeslots->map(function($ts) {
                    return [
                        'id' => $ts->id_business_timeslots,
                        'name' => $ts->timeslot_name,
                        'start_time' => date('H:i', strtotime($ts->start_time)),
                        'end_time' => date('H:i', strtotime($ts->end_time))
                    ];
                }),
                'discounts' => array_values($discounts_by_date)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error calculating flexi-price: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check if flexi-pricing is enabled for a business
     * 
     * Used by the frontend to dynamically check flexi-pricing status
     * when the user changes branches. This allows the UI to show/hide
     * flexi-pricing features accordingly.
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function check_flexi_pricing_status(Request $request)
    {
        $business_id = $request->input('business_id') ?: session('business_id');
        
        if (!$business_id) {
            return response()->json([
                'success' => false,
                'flexi_pricing_enabled' => false,
                'message' => 'Business ID is required'
            ], 400);
        }

        $business = Business::select('flexi_pricing_enabled')
            ->where('id_business', $business_id)
            ->first();

        if (!$business) {
            return response()->json([
                'success' => false,
                'flexi_pricing_enabled' => false,
                'message' => 'Business not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'flexi_pricing_enabled' => $business->flexi_pricing_enabled == 'Yes'
        ]);
    }
}


