<?php

namespace App\Http\Controllers;

use App\Models\BranchRequisition;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\BusinessType;
use App\Models\Business;
use App\Models\Stores;
use App\Models\ProductBatch;
use App\Models\TransferNotes;
use App\Models\Gtn;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Bus;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpKernel\HttpCache\Store;
use Yajra\DataTables\Facades\DataTables;
use App\Services\productBatchService;
use App\Models\AccountEventMapping;
use App\Models\AccountVoucherDetail;
use App\Models\AccountVouchers;
use Carbon\Carbon;
use Dotenv\Parser\Value;

class BranchRequisitionController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function open_requisition_list(Request $request)
    {
        
        $business_type_id = $request->input('business_type_id');
        $business_id = $request->input('business_id');

        //Get all business types
         $business_types = BusinessType::whereIn('id_business_type', function($query){
            $query->select('business_type_id')
                    ->from('business')
                    ->where('common_products', session('common_products'));
        })->get();

        //Get all branches with common_products = "Yes"
        if(session('ho')=='Yes'){
            $branches = DB::table('business')
                ->where('common_products', 'Yes');
            if ($business_type_id > 0) {
                $branches->where('business_type_id', $business_type_id);
            }
            $branches = $branches->get();     
        } else {
            
            $business_id = session('business_id');
            $branches = DB::table('business')->where('id_business', session('business_id'))
                ->get();
        }

        $ho = session('ho');

        return view('products.branch_requisitions.branch_requisitions', compact('business_types', 'branches', 'business_type_id', 'business_id', 'ho'));
    }

    public function get_requisitions_data(Request $request)
    {
        $business_type_id = $request->business_type_id;
        $business_id      = $request->business_id;
        
        $query = BranchRequisition::from('branch_requisition')
        ->select(
            'branch_requisition.requisition_number',
            DB::raw("DATE_FORMAT(branch_requisition.requisition_date, '%d-%m-%Y') as requisition_date"),
            'branch_requisition.business_id',
            'branch_requisition.created_by as requested_by',
            'business.business_name as to_branch',
            DB::raw("GROUP_CONCAT(branch_requisition.product_id SEPARATOR ',') as product_ids"),
            DB::raw("GROUP_CONCAT(business_products.product SEPARATOR ',') as product_names"),
            DB::raw("GROUP_CONCAT(branch_requisition.required_qty SEPARATOR ',') as required_qty"),
            DB::raw("GROUP_CONCAT(branch_requisition.transfered_qty SEPARATOR ',') as transfered_qty"),
            DB::raw("GROUP_CONCAT(branch_requisition.requisition_status SEPARATOR ',') as requisition_status")
        )
        ->join('business', 'business.id_business', '=', 'branch_requisition.business_id')
        ->join(
            'business_products',
            'business_products.id_business_products',
            '=',
            'branch_requisition.product_id'
        );

        if (session('ho') === 'Yes') {
            $query->whereIn('branch_requisition.requisition_status', ['Active', 'Transferred']);
        }

        if ($business_id > 0) {
            $query->where('branch_requisition.business_id', $business_id);
        }
        $query->groupBy(
            'branch_requisition.requisition_number',
            'branch_requisition.business_id'
        );
        $query->havingRaw('SUM(branch_requisition.required_qty) > SUM(branch_requisition.transfered_qty)');
        return DataTables::of($query)
            ->filter(function ($query) use ($request) {
                if (!empty($request->get('search')['value'])) {
                    $search = $request->get('search')['value'];
                    $query->where(function ($q) use ($search) {
                        $q->where('branch_requisition.requisition_number', 'like', "%{$search}%")
                        ->orWhere('business.business_name', 'like', "%{$search}%");
                    });
                }
            })
            ->make(true);
    }


    public function add_new_requisition()
    {      
       if (session('ho') == 'Yes') {
            return back();
        }
        //Get next requisition number of the logged in business
        $requisition_number = BranchRequisition::where('business_id', session('business_id'))
            ->max('requisition_number');

        if(session('ho')=="Yes"){
            $stores = Stores::select('business_stores.*')
            ->whereIn('business_id', function($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('common_products', 'Yes');
            })
            ->get();
        } else {
            $stores = Stores::select('business_stores.*')
            ->where('business_id', session('business_id'))
            ->get();
        }

        $requisition_number = $requisition_number + 1;

        $branch_requisition_status = DB::table('business')->where('id_business', session('business_id'))->value('branch_requisition_status') ?? 'Active';

        return view('products.branch_requisitions.new_requisition', compact('requisition_number', 'stores','branch_requisition_status'));
    }
   
    public function save_requisition(Request $request)
    {
        $data = $request->all();
        $items = $data['items'];
        //return $items;
        DB::beginTransaction();

        try {
            foreach ($items as $item) {
                if($item['id_branch_requisition'] == "" || $item['id_branch_requisition'] == null){ //add new record
                    $branchRequisition = new BranchRequisition();
                    $branchRequisition->requisition_number = $data['requisition_number'];
                    $branchRequisition->requisition_date = now();
                    $branchRequisition->business_id = session('business_id');
                    $branchRequisition->store_id = $item['store_id'];
                    $branchRequisition->product_id = $item['product_id'];
                    $branchRequisition->required_qty = $item['required_qty'];
                    $branchRequisition->transfered_qty = 0;
                    $branchRequisition->requisition_status = $item['requisition_status'];
                    $branchRequisition->created_by = session('user_name');
                    $branchRequisition->save();
                } else { //update existing record
                    $branchRequisition = BranchRequisition::find($item['id_branch_requisition']);                    
                    $branchRequisition->required_qty = $item['required_qty'];
                    $branchRequisition->requisition_status = $item['requisition_status'];
                    $branchRequisition->save();
                }
            }

            DB::commit();

            return response()->json([
                'message' => 'Requisition saved successfully',
                'message_type' => 'success'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => $e->getMessage(),
                'message_type' => 'error'
            ], 500);
        }
    }

    public function get_requisition_details($requisition_number, $business_id)
    {
        $requisition_number = intval($requisition_number);
        $requisition_items = BranchRequisition::where('requisition_number', $requisition_number)
            ->Join('business_stores', 'business_stores.id_business_stores', '=', 'branch_requisition.store_id')
            ->join('business_products as products', 'products.id_business_products', '=', 'branch_requisition.product_id')
            ->join('business_brands as brand', 'brand.id_business_brands', '=', 'products.brand_id')
            ->select('branch_requisition.*',
            'business_stores.business_store as store_name', 'brand.business_brand_name as brand_name',
            'products.product', 'products.measure_unit', 'products.qty_per_unit', 'products.barcode_products as barcode')
            ->where('branch_requisition.business_id',$business_id)
            ->get();
        
        if(session('ho')=="Yes"){
            $stores = Stores::select('business_stores.*')
            ->whereIn('business_id', function($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('common_products', 'Yes');
            })
            ->get();
        } else {
            $stores = Stores::select('business_stores.*')
            ->where('business_id', session('business_id'))
            ->get();
        }

        $branch_requisition_status = DB::table('business')->where('id_business', session('business_id'))->value('branch_requisition_status') ?? 'Active';

        return view('products.branch_requisitions.new_requisition', compact('requisition_number', 'stores', 'requisition_items','branch_requisition_status'));
    }

    public function ho_requisition_transfer($requisition_number , $business_id)
    {
        $requisition_number = intval($requisition_number);
        $requisition_items = BranchRequisition::where('requisition_number', $requisition_number)        
            ->Join('business_stores', 'business_stores.id_business_stores', '=', 'branch_requisition.store_id')
            ->join('business', 'business.id_business', '=', 'branch_requisition.business_id')
            ->join('business_products as products', 'products.id_business_products', '=', 'branch_requisition.product_id')
            ->join('business_brands as brand', 'brand.id_business_brands', '=', 'products.brand_id')
            ->select('branch_requisition.*', 'business.business_name as to_branch',
            'business_stores.business_store as store_name', 'brand.business_brand_name as brand_name',
            'products.product', 'products.measure_unit', 'products.qty_per_unit', 'products.barcode_products as barcode')
            ->where('branch_requisition.business_id', $business_id);
            if (session('ho') === 'Yes') {
                $requisition_items->whereIn('branch_requisition.requisition_status', ['Active', 'Transferred']);
            }
            $requisition_items = $requisition_items->get();


            $stores = Stores::select('business_stores.*')
            ->where('business_id', session('business_id'))
            ->get();
        return view('products.branch_requisitions.ho_requisition_transfer', compact('requisition_number', 'stores', 'requisition_items'));
    }

    public function process_ho_transfer(Request $request, $requisition_number)
    {
        if(!session('ho')=="Yes"){
            return response()->json([
                'message' => 'Unauthorized access',
                'message_type' => 'error'
            ], 403);
        }

        $data = $request->all();
        $items = $data['transfers'];
        $from_store_id = $data['from_store_id'];
        DB::beginTransaction();
        
        try {
            foreach ($items as $item) {
                //Get headoffice store id
                $ho_store = Stores::where('business_id', session('business_id'))->first();
                $ho_store_id = $ho_store->id_business_stores;
                $store_to = $item['store_id'];

                $branchIdTo = DB::table('business_stores')->where('business_stores.id_business_stores',$store_to)->value('business_id');

                //Get the batches of the HO stores for the product usint the productBatchService
                $productBatchService = new ProductBatchService();
                $batches = $productBatchService->findSufficientBatches($item['id_business_products'], session('business_id'), $from_store_id); //all stores of headoffice
                //Sum up the available qty using string $batches
                $batches_array = json_decode($batches, true);        
                $store_available_qty = array_sum(array_column($batches_array, 'instock'));

                $sufficient_batches = json_decode($batches);
                if(empty($sufficient_batches) || $sufficient_batches == null || $store_available_qty < $item['transfer_qty']){
                    DB::rollBack();
                    return response()->json([
                        'message' => 'Insufficient stock for product ID: '.$item['id_business_products']. ' at the Head Office, store: '.$ho_store->business_store.'',
                        'message_type' => 'error',
                        'message_btn' => 'btn btn-danger'
                    ], 400);
                }
                $remaining_qty = $item['transfer_qty'];
                //create GTN records and deduct stock from batches
                $gtn = new Gtn();
                $gtn->gtn_date = now();
                $gtn->business_id = session('business_id');
                $gtn->created_by = session('user_name');
                $gtn->to_store = $item['store_id'];
                $gtn->save();
                $total_inventory_amount =0;
                //Loop through batches and create transfernotes and deduct stock
                foreach($sufficient_batches as $batch){
                   
                    if($remaining_qty <= 0){
                        break;
                    }
                    $available_in_batch = $batch->instock;
                    $deduct_qty = min($available_in_batch, $remaining_qty);
                    //1- Create Transfer In record
                    // return $item['id_business_products'];    
                    //first create a new batch at the destination store 
                    $batches = ProductBatch::where('store_id', $item['store_id'])
                        ->where('product_id', $item['id_business_products'])
                        ->select(DB::raw('MAX(CAST(batch_number AS UNSIGNED)) as batch_number'))
                        ->first();

                    $new_batch_number = 1;
                    if ($batches && !empty($batches->batch_number)) {
                        $new_batch_number = (int) $batches->batch_number + 1;
                    }

                    // 4 digit format: 0001, 0002, 0010
                    $newbatchnumber = str_pad($new_batch_number, 4, '0', STR_PAD_LEFT);

                    $new_batch = new ProductBatch();
                    $new_batch->product_id = $item['id_business_products'];
                    $new_batch->store_id = $item['store_id'];
                    $new_batch->batch_number = $newbatchnumber;
                    $new_batch->batch_qty = $deduct_qty;
                    $new_batch->batch_amount = $batch->batch_amount;
                    $new_batch->batch_date = now();
                    $new_batch->expiry_date = $batch->expiry_date;
                    $new_batch->batch_hidden = 'No';
                    $new_batch->save();

                    $transfer_note_in = new TransferNotes();
                    $transfer_note_in->requisition_id = $item['id_branch_requisition'];
                    $transfer_note_in->product_id = $item['id_business_products'];
                    $transfer_note_in->transfer_date = now();
                    $transfer_note_in->business_id = $branchIdTo;
                    $transfer_note_in->created_by = session('user_name');
                    $transfer_note_in->status = 'Active';
                    $transfer_note_in->gtn_id = $gtn->id_gtn;
                    $transfer_note_in->tranfer_out_qty = 0;
                    $transfer_note_in->tranfer_in_qty = $deduct_qty;
                    $transfer_note_in->batch = $new_batch->batch_number;
                    $transfer_note_in->batch_id = $new_batch->id_batch;
                    $transfer_note_in->save();


                    //2 - Create Transfer Out Record
                    $transfer_note = new TransferNotes();
                    $transfer_note->requisition_id = $item['id_branch_requisition'];
                    $transfer_note->product_id = $item['id_business_products'];
                    $transfer_note->transfer_date = now();
                    $transfer_note->business_id = session('business_id');
                    $transfer_note->created_by = session('user_name');
                    $transfer_note->status = 'Active';
                    $transfer_note->gtn_id = $gtn->id_gtn;
                    $transfer_note->tranfer_out_qty = $deduct_qty;
                    $transfer_note->tranfer_in_qty = 0;
                    $transfer_note->batch = $batch->batch_no;
                    $transfer_note->batch_id = $batch->batch_id;
                    //$transfer_note->unit_type = 
                    $transfer_note->save();


                    //Update Requisition transfer qty
                    $requisition = BranchRequisition::where('id_branch_requisition', $item['id_branch_requisition'])->first();
                    $requisition->transfered_qty = $requisition->transfered_qty + $deduct_qty;
                    $requisition->save();

                    $remaining_qty -= $deduct_qty;
                    $total_inventory_amount += ($deduct_qty * $batch->batch_amount);
                }

                    /**
                     * -----------------------------
                     * TRANSFER VOUCHER START 
                     * -----------------------------
                     */
                    $branchIdFrom = session('business_id'); // from branch
                    $amount       = $total_inventory_amount;
                

                
                    $fromBusiness = Business::where('id_business', $branchIdFrom)->first();
                    $toBusiness   = Business::where('id_business', $branchIdTo)->first();

                    /**
                     * -----------------------------
                     * TRANSFER OUT (FROM BRANCH)
                     * -----------------------------
                     */
                    $mappingBusinessIdOut = session('ho_accounts') === 'Yes' ? Business::where('ho', 'Yes')->value('id_business') : $branchIdFrom;

                    $eventMappingsOut = AccountEventMapping::where('account_event_id', 34)
                        ->where('business_id', $mappingBusinessIdOut)
                        ->get();

                    $debitAccountsOut  = [];
                    $creditAccountsOut = [];

                    foreach ($eventMappingsOut as $mapping) {
                        $entry = "{$mapping->account_head_id}|{$mapping->transaction_type}|{$mapping->entity_name}";
                        if ($mapping->transaction_type === 'debit') {
                            $debitAccountsOut[] = $entry;
                        } else {
                            $creditAccountsOut[] = $entry;
                        }
                    }

                    $voucherDescOut = "Inventory transferred to Branch $toBusiness->business_name (ID: {$toBusiness->id_business}).";

                    $voucherIdOut = DB::table('account_vouchers')->insertGetId([
                        'business_id'           => $branchIdFrom,
                        'description'           => $voucherDescOut,
                        'voucher_date'          => Carbon::now(),
                        'voucher_amount'        => $amount,
                        'voucher_status'        => 'Active',
                        'created_by'            => session('user_name'),
                        'cost_center'           => '4',
                        'cost_center_name'      => $toBusiness->business_name,
                        'business_partner'      => 5,
                        'business_partner_id'   => $toBusiness->id_business,
                        'business_partner_name' => $toBusiness->business_name ,
                        'voucher_type'          => 4,
                        'payment_mode'          => 'Cash',
                        'bank_name'             => '',
                        'auto_voucher'          => 'Yes',
                    ]);

                    // Insert debit entries (Out)
                    foreach ($debitAccountsOut as $debitAccount) {
                        [$headId, $type, $entity] = explode('|', $debitAccount);
                        $debitAmount = ($entity === 'receivable') ? $amount : 0;

                        if ($debitAmount > 0) {
                            DB::table('account_voucher_detail')->insert([
                                'account_voucher_id' => $voucherIdOut,
                                'account_head_id'    => $headId,
                                'debit'              => round($debitAmount, 2),
                                'credit'             => 0,
                            ]);
                        }
                    }

                    // Insert credit entries (Out)
                    foreach ($creditAccountsOut as $creditAccount) {
                        [$headId, $type, $entity] = explode('|', $creditAccount);
                        $creditAmount = ($entity === 'inventory_out') ? $amount : 0;

                        if ($creditAmount > 0) {
                            DB::table('account_voucher_detail')->insert([
                                'account_voucher_id' => $voucherIdOut,
                                'account_head_id'    => $headId,
                                'debit'              => 0,
                                'credit'             => round($creditAmount, 2),
                            ]);
                        }
                    } 


                    // ===== CHECK OUT VOUCHER BALANCE =====
                    $totalsOut = DB::table('account_voucher_detail')
                        ->where('account_voucher_id', $voucherIdOut)
                        ->selectRaw('
                            ROUND(SUM(debit),2) as total_debit,
                            ROUND(SUM(credit),2) as total_credit
                        ')
                        ->first();

                    if ((float)$totalsOut->total_debit !== (float)$totalsOut->total_credit) {
                        DB::rollBack();
                        return response()->json([
                            'message' => "Transfer OUT Voucher imbalance. Debit ({$totalsOut->total_debit}) and Credit ({$totalsOut->total_credit}) not equal",
                            'message_type' => 'error'
                        ], 400);
                    }


                    /**
                     * -----------------------------
                     * TRANSFER IN (TO BRANCH)
                     * -----------------------------
                     */
                    
                    $mappingBusinessIdIn = $toBusiness->ho_accounts === 'Yes' ? Business::where('ho', 'Yes')->value('id_business') : $branchIdTo;
                    $eventMappingsIn = AccountEventMapping::where('account_event_id', 34)->where('business_id', $mappingBusinessIdIn)->get();

                    $debitAccountsIn  = [];
                    $creditAccountsIn = [];

                    foreach ($eventMappingsIn as $mapping) {
                        $entry = "{$mapping->account_head_id}|{$mapping->transaction_type}|{$mapping->entity_name}";
                        if ($mapping->transaction_type === 'debit') {
                            $debitAccountsIn[] = $entry;
                        } else {
                            $creditAccountsIn[] = $entry;
                        }
                    }

                    $voucherDescIn = "Inventory transferred from Branch '{$fromBusiness->business_name}' (ID: {$fromBusiness->id_business}).";
                    $voucherIdIn = DB::table('account_vouchers')->insertGetId([
                        'business_id'           => $branchIdTo,
                        'description'           => $voucherDescIn,
                        'voucher_date'          => Carbon::now(),
                        'voucher_amount'        => $amount,
                        'voucher_status'        => 'Active',
                        'created_by'            => session('user_name'),
                        'cost_center'           => '4',
                        'cost_center_name'      => $fromBusiness->business_name,
                        'business_partner'      => 5,
                        'business_partner_id'   => $fromBusiness->id_business,
                        'business_partner_name' => $fromBusiness->business_name ,
                        'voucher_type'          => 4,
                        'payment_mode'          => 'Cash',
                        'bank_name'             => '',
                        'auto_voucher'          => 'Yes',
                    ]);

                    // Insert debit entries (In)
                    foreach ($debitAccountsIn as $debitAccount) {
                        [$headId, $type, $entity] = explode('|', $debitAccount);
                        $debitAmount = ($entity === 'inventory_in') ? $amount : 0;

                        if ($debitAmount > 0) {
                            DB::table('account_voucher_detail')->insert([
                                'account_voucher_id' => $voucherIdIn,
                                'account_head_id'    => $headId,
                                'debit'              => round($debitAmount, 2),
                                'credit'             => 0,
                            ]);
                        }
                    }

                    // Insert credit entries (In)
                    foreach ($creditAccountsIn as $creditAccount) {
                        [$headId, $type, $entity] = explode('|', $creditAccount);
                        $creditAmount = ($entity === 'payable') ? $amount : 0;

                        if ($creditAmount > 0) {
                            DB::table('account_voucher_detail')->insert([
                                'account_voucher_id' => $voucherIdIn,
                                'account_head_id'    => $headId,
                                'debit'              => 0,
                                'credit'             => round($creditAmount, 2),
                            ]);
                        }
                    } 

                    $totalsIn = DB::table('account_voucher_detail')
                        ->where('account_voucher_id', $voucherIdIn)
                        ->selectRaw('
                            ROUND(SUM(debit),2) as total_debit,
                            ROUND(SUM(credit),2) as total_credit
                        ')
                        ->first();

                    if ((float)$totalsIn->total_debit !== (float)$totalsIn->total_credit) {
                        DB::rollBack();
                        return response()->json([
                            'message' => "Transfer IN Voucher imbalance. Debit ({$totalsIn->total_debit}) and Credit ({$totalsIn->total_credit}) not equal",
                            'message_type' => 'error'
                        ], 400);
                    }

                    /**
                     * -----------------------------
                     * TRANSFER VOUCHER END 
                     * -----------------------------
                     */   

            } 
            DB::commit();
            return response()->json([
                'message' => 'Items transferred successfully',
                'message_type' => 'success'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => $e->getMessage(),
                'message_type' => 'error'
            ], 500);
        }
    }

}