<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\BusinessTargets;
use App\Models\Business;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class BranchTargetsController extends Controller
{
    private function canAccessPage()
    {
        // Check ho_accounts from session. If no, open the page normally
        if (session('ho_accounts') !== 'Yes') {
            return true;
        }
        
        // If ho_accounts is yes, check ho in session. If ho is yes, open the page, if no, do not open the page
        return session('ho') === 'Yes';
    }

    private function getAllBusinesses()
    {
        return Business::orderBy('business_name', 'ASC')->get();
    }

    /**
     * Display the branch targets page
     */
    public function index(Request $request)
    {
        // Check access permissions
        if (!$this->canAccessPage()) {
            abort(403, 'Access denied. Only head office users can access this page.');
        }
        
        $businesses = $this->getAllBusinesses();
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        
        return view('branch_targets.branch_targets', compact('business', 'businesses', 'businessId'));
    }

    /**
     * Get branch targets data for DataTables
     * Returns grouped data by Business/Target Type/Year with all months
     */
    public function getData(Request $request)
    {
        try {
            if (!$this->canAccessPage()) {
                return response()->json([
                    'error' => 'Access denied'
                ], 403);
            }

            $baseQuery = BusinessTargets::join('business', 'business_targets.business_id', '=', 'business.id_business')
                ->select([
                    'business_targets.*',
                    'business.business_name'
                ]);

            // Apply filters
            if ($request->filled('business_name')) {
                $baseQuery->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }
            
            if ($request->filled('target_type')) {
                $baseQuery->where('business_targets.target_type', $request->target_type);
            }
            
            if ($request->filled('year')) {
                $baseQuery->where('business_targets.year', $request->year);
            }
            
            if ($request->filled('month')) {
                $baseQuery->where('business_targets.month', $request->month);
            }

            // Get all targets
            $targets = $baseQuery->orderBy('business.business_name')
                ->orderBy('business_targets.target_type')
                ->orderBy('business_targets.year')
                ->orderBy('business_targets.month')
                ->get();

            // Group by business_id, target_type, year
            $grouped = [];
            foreach ($targets as $target) {
                $key = $target->business_id . '_' . $target->target_type . '_' . $target->year;
                if (!isset($grouped[$key])) {
                    $grouped[$key] = [
                        'business_id' => $target->business_id,
                        'business_name' => $target->business_name,
                        'target_type' => $target->target_type,
                        'target_type_display' => ucfirst($target->target_type),
                        'year' => $target->year,
                        'months' => []
                    ];
                }
                $grouped[$key]['months'][$target->month] = [
                    'id_business_targets' => $target->id_business_targets,
                    'month' => $target->month,
                    'month_display' => $this->getMonthName($target->month),
                    'amount' => $target->amount,
                    'amount_display' => number_format($target->amount, 2)
                ];
            }

            // Build rows for DataTables
            $rows = [];
            foreach ($grouped as $group) {
                $monthCount = count($group['months']);
                $isFirstRow = true;
                
                // Create rows for each month (1-12)
                for ($month = 1; $month <= 12; $month++) {
                    $monthData = $group['months'][$month] ?? null;
                    
                    $row = [
                        'business_id' => $group['business_id'],
                        'business_name' => $group['business_name'],
                        'target_type' => $group['target_type'],
                        'target_type_display' => $group['target_type_display'],
                        'year' => $group['year'],
                        'month' => $month,
                        'month_display' => $this->getMonthName($month),
                        'amount' => $monthData ? $monthData['amount'] : null,
                        'amount_display' => $monthData ? $monthData['amount_display'] : '',
                        'id_business_targets' => $monthData ? $monthData['id_business_targets'] : null,
                        'is_first_row' => $isFirstRow,
                        'rowspan' => 12,
                        'action' => $isFirstRow ? $this->getActionButtons($group['business_id'], $group['target_type'], $group['year']) : ''
                    ];
                    
                    $rows[] = $row;
                    $isFirstRow = false;
                }
            }

            return DataTables::of(collect($rows))
                ->rawColumns(['action'])
                ->make(true);
                
        } catch (\Exception $e) {
            \Log::error('Error getting branch targets data: ' . $e->getMessage());
            return response()->json([
                'error' => 'Error loading branch targets data'
            ], 500);
        }
    }

    /**
     * Get month name by number
     */
    private function getMonthName($month)
    {
        $months = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];
        return $months[$month] ?? $month;
    }

    /**
     * Get action buttons for a year group
     */
    private function getActionButtons($businessId, $targetType, $year)
    {
        $editBtn = '';
        $deleteBtn = '';
        
        if(auth()->user()->can('edit_branch_targets')) {
            $editBtn = '<button class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-info open-edit-btn" 
                onclick="openEditModal('.$businessId.', \''.$targetType.'\', '.$year.')"
                data-bs-toggle="tooltip" title="Edit Branch Targets">
                <iconify-icon icon="line-md:edit" class="fs-20"></iconify-icon></button>';
        }
        
        if(auth()->user()->can('delete_branch_targets')) {
            $deleteBtn = '<button class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-danger" 
                onclick="deleteYearTargets('.$businessId.', \''.$targetType.'\', '.$year.')"
                data-bs-toggle="tooltip" title="Delete All Targets for This Year">
                <iconify-icon icon="solar:trash-bin-trash-bold-duotone" class="fs-20"></iconify-icon></button>';
        }
        
        return '<div class="d-flex gap-1">' . $editBtn . $deleteBtn . '</div>';
    }

    /**
     * Get all branch targets for a year (for editing)
     */
    public function getTargetsForYear(Request $request)
    {
        try {
            if (!$this->canAccessPage()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied'
                ], 403);
            }

            $request->validate([
                'business_id' => 'required|integer|exists:business,id_business',
                'target_type' => 'required|in:Service,Retail',
                'year' => 'required|integer|min:2000|max:2100',
            ]);

            $targets = BusinessTargets::where('business_id', $request->business_id)
                ->where('target_type', $request->target_type)
                ->where('year', $request->year)
                ->orderBy('month')
                ->get();

            // Build array with all 12 months
            $monthsData = [];
            for ($month = 1; $month <= 12; $month++) {
                $target = $targets->firstWhere('month', $month);
                $monthsData[$month] = [
                    'id_business_targets' => $target ? $target->id_business_targets : null,
                    'amount' => $target ? $target->amount : null,
                ];
            }

            return response()->json([
                'success' => true,
                'targets' => $monthsData,
                'business_id' => $request->business_id,
                'target_type' => $request->target_type,
                'year' => $request->year
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Error getting targets for year: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading targets'
            ], 500);
        }
    }

    /**
     * Get single branch target for editing (kept for backward compatibility)
     */
    public function getTarget($id)
    {
        try {
            $target = BusinessTargets::findOrFail($id);
            return response()->json([
                'success' => true,
                'target' => $target
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Branch target not found'
            ], 404);
        }
    }

    /**
     * Save or update branch targets (bulk update for all 12 months)
     */
    public function saveOrUpdate(Request $request)
    {
        try {
            if (!$this->canAccessPage()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied'
                ], 403);
            }

            $request->validate([
                'business_id' => 'required|integer|exists:business,id_business',
                'target_type' => 'required|in:Service,Retail',
                'year' => 'required|integer|min:2000|max:2100',
                'months' => 'required|array|size:12',
                'months.*' => 'nullable|numeric|min:0',
            ]);

            $businessId = $request->input('business_id');
            $targetType = $request->input('target_type');
            $year = $request->input('year');
            $months = $request->input('months');

            DB::beginTransaction();

            $savedCount = 0;
            $updatedCount = 0;

            foreach ($months as $month => $amount) {
                $month = (int)$month;
                
                // Skip if amount is null or empty
                if ($amount === null || $amount === '') {
                    continue;
                }

                // Check if target exists
                $target = BusinessTargets::where('business_id', $businessId)
                    ->where('target_type', $targetType)
                    ->where('year', $year)
                    ->where('month', $month)
                    ->first();

                if ($target) {
                    // Update existing
                    $target->update(['amount' => $amount]);
                    $updatedCount++;
                } else {
                    // Create new
                    BusinessTargets::create([
                        'business_id' => $businessId,
                        'target_type' => $targetType,
                        'year' => $year,
                        'month' => $month,
                        'amount' => $amount,
                    ]);
                    $savedCount++;
                }
            }

            DB::commit();

            $message = 'Branch targets saved successfully';
            if ($savedCount > 0 && $updatedCount > 0) {
                $message = "Created {$savedCount} target(s) and updated {$updatedCount} target(s)";
            } elseif ($savedCount > 0) {
                $message = "Created {$savedCount} target(s)";
            } elseif ($updatedCount > 0) {
                $message = "Updated {$updatedCount} target(s)";
            }

            return response()->json([
                'success' => true,
                'message' => $message
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error saving branch targets: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving branch targets: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete all branch targets for a year
     */
    public function deleteYearTargets(Request $request)
    {
        try {
            if (!$this->canAccessPage()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied'
                ], 403);
            }

            $request->validate([
                'business_id' => 'required|integer|exists:business,id_business',
                'target_type' => 'required|in:Service,Retail',
                'year' => 'required|integer|min:2000|max:2100',
            ]);

            $deleted = BusinessTargets::where('business_id', $request->business_id)
                ->where('target_type', $request->target_type)
                ->where('year', $request->year)
                ->delete();

            return response()->json([
                'success' => true,
                'message' => "Deleted {$deleted} branch target(s) successfully"
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Error deleting branch targets: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting branch targets'
            ], 500);
        }
    }

    /**
     * Delete single branch target (kept for backward compatibility)
     */
    public function delete($id)
    {
        try {
            if (!$this->canAccessPage()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied'
                ], 403);
            }

            $target = BusinessTargets::findOrFail($id);
            $target->delete();

            return response()->json([
                'success' => true,
                'message' => 'Branch target deleted successfully'
            ]);
        } catch (\Exception $e) {
            \Log::error('Error deleting branch target: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting branch target'
            ], 500);
        }
    }
}
