<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\ColorRecords;

class ColorRecordsController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function index(Request $request)
    {
        return view('customers.color_records.list');
    }

    public function recordsData(Request $request)
    {
        $business_id = session('business_id');
        
        $query = ColorRecords::where('business_id', $business_id);

        // Apply filters
        if ($request->filled('filter_id')) {
            $query->where('id', 'LIKE', '%' . $request->filter_id . '%');
        }
        if ($request->filled('filter_customer')) {
            $query->where('customer', 'LIKE', '%' . $request->filter_customer . '%');
        }
        if ($request->filled('filter_colortype')) {
            $query->where('colortype', 'LIKE', '%' . $request->filter_colortype . '%');
        }
        if ($request->filled('filter_colorno')) {
            $query->where('colorno', 'LIKE', '%' . $request->filter_colorno . '%');
        }
        if ($request->filled('filter_watercontent')) {
            $query->where('watercontent', 'LIKE', '%' . $request->filter_watercontent . '%');
        }
        
        if ($request->filled('filter_date')) {
            $query->where('date', 'LIKE', '%' . $request->filter_date . '%');
        }
        if ($request->filled('filter_time')) {
            $query->where('time', 'LIKE', '%' . $request->filter_time . '%');
        }
        if ($request->filled('filter_charge')) {
            $query->where('charge', 'LIKE', '%' . $request->filter_charge . '%');
        }
        if ($request->filled('filter_remarks')) {
            $query->where('remarks', 'LIKE', '%' . $request->filter_remarks . '%');
        }
        if ($request->filled('filter_recommendation')) {
            $query->where('recommendation', 'LIKE', '%' . $request->filter_recommendation . '%');
        }

        $records = $query->orderBy('id', 'DESC')->get();

        // Format time for display (convert minutes to H:MM:SS)
        $records->transform(function ($record) {
            if ($record->time) {
                $minutes = (int)$record->time;
                $hours = floor($minutes / 60);
                $mins = $minutes % 60;
                $seconds = 0;
                $record->formatted_time = sprintf('%d:%02d:%02d', $hours, $mins, $seconds);
            } else {
                $record->formatted_time = '0:00:00';
            }
            return $record;
        });

        return response()->json([
            'data' => $records
        ]);
    }

    public function save_or_update_record(Request $request)
    {
        try {
            $request->validate([
                'id' => 'nullable|integer',
                'customer_id' => 'nullable|integer',
                'customer' => 'required|string|max:255',
                'color_type' => 'nullable|string|max:255',
                'color_no' => 'nullable|string|max:255',
                'water_content' => 'nullable|string|max:255',
                'date' => 'nullable|date',
                'time_minutes' => 'required|integer|min:0',
                'charge' => 'nullable|numeric|min:0',
                'remarks' => 'nullable|string',
                'recommendation' => 'nullable|string',
            ]);

            $id = $request->input('id');
            $business_id = session('business_id');
            $timeMinutes = $request->input('time_minutes');

            // Store as minutes (as integer string) - per spec "Stored in database as minutes"
            $data = [
                'business_id' => $business_id,
                'customer_id' => $request->input('customer_id'),
                'customer' => $request->input('customer'),
                'color_type' => $request->input('color_type'),
                'color_no' => $request->input('color_no'),
                'water_content' => $request->input('water_content'),
                'date' => $request->input('date'),
                'time' => (string)$timeMinutes, // Store as minutes
                'charge' => $request->input('charge'),
                'remarks' => $request->input('remarks'),
            ];

            if (empty($id)) {
                // Create new record - let database handle auto-increment
                $data['created_date'] = now();
                $record = ColorRecords::create($data);
            } else {
                // Update existing record
                $record = ColorRecords::updateOrCreate(
                    ['id' => $id],
                    $data
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Color record saved successfully',
                'record' => $record,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
