<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Customers;
use App\Models\Invoice;
use App\Models\CityAreas;
use App\Models\CustomerVisits;
use App\Services\customerDetailsService;

use Illuminate\Support\Facades\Bus;
use Pest\ArchPresets\Custom;
use Illuminate\Support\Facades\DB;
use App\Models\AccountEventMapping;
use App\Models\AccountVoucherDetail;
use App\Models\AccountVouchers;
use Carbon\Carbon;


class CustomerController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;


    public function customer_search(Request $request)
    {
        $term = $request->input('term');
       
        $customers = Customers::where(function ($query) use ($term) {
                $query->where('customer_name', 'LIKE', '%' . $term . '%')
                      ->orWhere('customer_cell', 'LIKE', '%' . $term . '%')
                      ->orWhere('customer_email', 'LIKE', '%' . $term . '%')
                      ->orWhere('customer_card', 'LIKE', '%' . $term . '%');
            })
            ->select('id_customers as id', DB::raw('concat(customer_name, " ", customer_cell) as text'), 'customer_name', DB::raw("concat(customer_cell, ' ', coalesce(customer_area,'')) as customer_cell"), 'customer_email', 'customer_card', DB::raw('COALESCE(customer_type, "primary") as customer_type'))
            ->get();
           // print_r($customers); exit();
        return response()->json($customers);
    }

    public function get_customer_details(Request $request)
    {
        $customer_id = $request->input('id_customer');
        $customer = Customers::select('id_customers', 'customer_name', 'customer_cell', 'customer_email', 'customer_card', 
        'customer_address', 'customer_area', 'customer_allergies', 'customer_alert',  'allowed_balance', 'customer_gender', 'customer_birthmonth', 'customer_birthday',
        'customer_type as customer_flag',
        'customer_area', 'customer_allergies', 'customer_alert',
        DB::raw('COALESCE(customer_type, "primary") as customer_type'))
        
        ->where('id_customers', '=', $customer_id)        
        ->first();

         $recoveries = Invoice::select('id_invoice', 'balance')
        ->where('is_recovery', '=', 'Yes')
        ->where('customer_id', '=', $customer_id)
        ->where('invoice_status', '=', 'valid')
        ->get();

             
        return response()->json([           
            "status" => 'success',
            "customer_details" => $customer,
            "recoveries" => $recoveries, 
            
        ]);
       
    }

    public function customers(Request $request)
    {
        $business = Business::select('business_name', 'id_business', 'business_logo', 'hide_cell')
            ->where('id_business', '=', session('business_id'))
            ->first();

        return view('customers.customer_list')->with(compact('business'));
    }

    public function get_city_areas(Request $request)
    {

        $business_ids = [];
        if(session('common_products') == "No"){
            $business_ids = [session('business_id')];
        } else {
            $query = Business::select('id_business')
            ->where(function($query){
                $query->where('common_products', '=', "Yes");
            })
            ->get();
            $business_ids = $query->pluck('id_business')->toArray();
        }

       $areas = CityAreas::select('city_area', 'id_city_areas', 'cities.city as city_name')
            ->join('cities', 'cities.id_cities', '=', 'city_areas.city_id')
            ->whereIn('cities.business_id', $business_ids)
            ->get();

        return response()->json($areas);
    }

    public function customersdata(Request $request)
    {
        $columns = [
            'id_customers',
            'customer_name',
            'customer_cell',
            'customer_email',
            'customer_card',
            'customer_area',
            'customer_created_at',
            'last_updated_on',
            'customer_gender',
            'customer_birthdate',
            'credit_limit',
            'customer_type',
            'branch_name',
            'actions'
        ];

        $search = $request->input('search.value');
        $orderColumnIndex = $request->input('order.0.column', 0);
        $orderColumn = $columns[$orderColumnIndex] ?? 'id_customers';
        $orderDir = $request->input('order.0.dir', 'asc');
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        // Base query
        $query = Customers::join('business', 'business.id_business', '=', 'customers.business_id')
            ->select(
                'id_customers',
                'customer_name',
                'customer_cell',
                'customer_email',
                'customer_card',
                'customer_area',
                DB::raw('COALESCE(DATE_FORMAT(created_on, "%d-%m-%Y %H:%i"), "N/A") AS customer_created_at'),
                DB::raw('COALESCE(DATE_FORMAT(customers.updated_at, "%d-%m-%Y %H:%i"), "N/A") AS last_updated_on'),
                DB::raw('CASE 
                            WHEN customer_gender = "F" THEN "Female" 
                            WHEN customer_gender = "M" THEN "Male" 
                            ELSE COALESCE(customer_gender, "N/A") 
                        END AS customer_gender'),
                DB::raw('COALESCE(CONCAT(customer_birthday, "-", customer_birthmonth), "N/A") AS customer_birthdate'),
                'allowed_balance AS credit_limit',
                DB::raw('COALESCE(customer_type, "N/A") AS customer_type'),
                'business.business_name AS branch_name',
            );
            
        // Total count before filtering
        $recordsTotal = $query->count();

        // Search filter
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('customer_name', 'like', "%{$search}%")
                    ->orWhere('customer_cell', 'like', "%{$search}%")
                    ->orWhere('customer_email', 'like', "%{$search}%")
                    ->orWhere('business.business_name', 'like', "%{$search}%");
            });
        }

        // Count after filtering
        $recordsFiltered = $query->count();

        // Apply order, limit, offset
        $data = $query
            ->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        // Return JSON in DataTables expected structure
        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    public function get_customer_edit_data(Request $request)
    {
        $customer_id = $request->input('id_customer');
        $customer = Customers::where('id_customers', '=', $customer_id)->first();

        return response()->json($customer);
    }

    public function update_customer(Request $request)
    {
        
        try {
            DB::beginTransaction();
            $customer_id = $request->input('id_customers');
            $customer = Customers::find($customer_id);
                $customer->customer_name = $request->input('customer_name');
                $customer->customer_cell = $request->input('customer_cell');
                $customer->customer_gender = $request->input('customer_gender');
                $customer->customer_birthday = $request->input('customer_birthday');
                $customer->customer_birthmonth = $request->input('customer_birthmonth');
                $customer->customer_type = $request->input('customer_type');
                $customer->customer_phone1 = $request->input('customer_phone1');
                $customer->customer_phone2 = $request->input('customer_phone2');
                $customer->customer_email = $request->input('customer_email');
                $customer->customer_card = $request->input('customer_card');
                $customer->customer_address = $request->input('customer_address');
                $customer->customer_area = $request->input('customer_area');
                $customer->customer_allergies = $request->input('customer_allergies');
                $customer->customer_alert = $request->input('customer_alert');
                $customer->allowed_balance = $request->input('allowed_balance');
                $customer->customer_careof = $request->input('customer_careof_name');
                                
                $customer->save();
                DB::commit();
                return response()->json(
                    [
                        'success' => true,
                        'message' => 'Customer updated successfully.',
                        'btn' => 'btn btn-success',
                        'id_customers' => $customer->id_customers
                    ]
                );
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => 'Error updating customer: ' . $e->getMessage(),
                'btn' => 'btn btn-danger'
            ]);
        }
    }

    public function check_mobile_exists(Request $request)
    {
        $mobile = $request->input('customer_cell');
        
        if (empty($mobile)) {
            return response()->json([
                'exists' => false,
                'message' => ''
            ]);
        }

        $exists = Customers::where('customer_cell', $mobile)->exists();

        return response()->json([
            'exists' => $exists,
            'message' => $exists ? 'Customer with this phone number already exists, please use "Care Of" option if you wish to save this customer under the same phone number' : ''
        ]);
    }

    public function add_customer(Request $request)
    {
        try {
            // Check if mobile number already exists (only for new customers)
            $customer_id = $request->input('id_customers');
            if (empty($customer_id)) {
                // Only check for duplicates when creating a new customer
                $mobile = $request->input('customer_cell');
                if (!empty($mobile)) {
                    $mobileExists = Customers::where('customer_cell', $mobile)->exists();
                    if ($mobileExists) {
                        return response()->json([
                            'error' => true,
                            'message' => 'Customer with this phone number already exists, please use "Care Of" option if you wish to save this customer under the same phone number',
                            'btn' => 'btn btn-danger'
                        ]);
                    }
                }
            }

            DB::beginTransaction();

            //Add Customer
            Customers::updateOrCreate(
                ['id_customers' => $request->input('id_customers')],
                [
                    'business_id' => session('business_id'),
                    'customer_name' => $request->input('customer_name'),
                    'customer_cell' => $request->input('customer_cell'),
                    'customer_gender' => $request->input('customer_gender') ?? 'F',
                    'customer_birthday' => $request->input('customer_birthday'),
                    'customer_birthmonth' => $request->input('customer_birthmonth'),
                    'customer_type' => $request->input('customer_type') ?? 'blue',
                    'customer_phone1' => $request->input('customer_phone1'),
                    'customer_phone2' => $request->input('customer_phone2'),
                    'customer_email' => $request->input('customer_email'),
                    'customer_card' => $request->input('customer_card'),
                    'customer_address' => $request->input('customer_address'),
                    'customer_area' => $request->input('customer_area'),
                    'customer_allergies' => $request->input('customer_allergies') ?? '',
                    'customer_alert' => $request->input('customer_alert') ?? '',
                    'allowed_balance' => $request->input('allowed_balance') ?? 50000,
                    'customer_careof' => $request->input('customer_careof_name'),
                    'created_on' => now(),
                    'profession' => $request->input('profession') ?? '',
                ]
            );

            DB::commit();
            return response()->json(
                [
                    'success' => true,
                    'message' => 'Customer Added/Updated successfully.',
                    'btn' => 'btn btn-success',
                    'id_customers' => $request->input('id_customers'),
                    'customer_name' => $request->input('customer_name'),
                    'customer_cell' => $request->input('customer_cell'),
                ]
            );

        } catch (\Exception $e) {
            return response()->json(['error' => true, 
            'message' => 'Error adding customer: ' . $e->getMessage(),
                'btn' => 'btn btn-danger'
            ]);
        }
    }

    public function open_customer_account($customer_id)
    {

        if(session('common_products') == "No"){
            $businesses = Business::select('business_name', 'id_business')
            ->where('id_business', '=', session('business_id'))
            ->get();
        } else {
            $businesses = Business::select('business_name', 'id_business')
            ->where(function($query){
                $query->where('common_products', '=', "Yes")
                ->where('id_business', '!=', session('business_id'));
            })
            ->get();
        }

       // $customer_id = $request->input('id_customer');
        $customer = Customers::select('id_customers', 'customer_name', 'customer_cell', 'customer_email', 'customer_card', 
        'customer_address', 'customer_area', 'customer_allergies', 'customer_alert',  'customer_careof',
        DB::RAW('CASE WHEN customer_gender = "F" THEN "Female" WHEN customer_gender = "M" THEN "Male" WHEN customer_gender = "O" THEN "Other" ELSE COALESCE(customer_gender, "N/A") END AS customer_gender'),
        DB::RAW('COALESCE(CONCAT(customer_birthday, "-", customer_birthmonth), "N/A") AS customer_birthdate'),
        DB::raw('CASE WHEN customer_type = "orange" THEN "Star Customer"  WHEN customer_type = "green" THEN "Important Customer"  WHEN customer_type = "red" THEN "Flagged Customer"  WHEN customer_type = "blue" THEN "Regular Customer" ELSE COALESCE(customer_type, "Regular Customer") END AS customer_type'),
        DB::raw('COALESCE(created_on, "N/A") AS customer_created_on'),
        DB::RAW('COALESCE(allowed_balance, 0) AS allowed_balance')
        )
        ->where('id_customers', $customer_id)
        ->first();

        $customerDetailsService = new customerDetailsService();

        $loyalty_points = $customerDetailsService->getCustomerLoyaltyPoints($customer_id);
        $customer_loyalty_points = $loyalty_points->first()->loyalty_points;
        $customer_retained_amount = $customerDetailsService->getCustomerRetainedAmount($customer_id, session('business_id'));
        $customer_retained_amount = floatval($customer_retained_amount);
        

        $customer_balance_invoices = $customerDetailsService->getCustomerBalanceInvoices($customer_id);

        return view('customers.customer_account')->with(compact('customer', 'businesses', 'customer_loyalty_points', 'customer_retained_amount', 'customer_balance_invoices'));
    }

    public function get_customer_service_category_data($id_customers, Request $request)
    {
        $customer_id = $request->input('customer_id');

        $query = Invoice::select('invoice_details.service_category as label', 
        DB::RAW('count(invoice_details.service_category) as item'))
            ->join('invoice_details', 'invoice.id_invoice', '=', 'invoice_details.invoice_id')
            ->where('invoice.customer_id', $id_customers)
            ->where('invoice.invoice_status', 'valid')
            ->groupBy('invoice_details.service_category')
            ->limit(12)
            // ->orderBy('item', 'DESC')
            ->get();
        $service_category_data = [];

        foreach ($query as $data) {
            $service_category_data['labels'][] = $data->label;
            $service_category_data['series'][] = intval($data->item);
        }

        return response()->json($service_category_data);
    }

    public function get_customer_staff_preference_data($id_customers, Request $request)
    {
        $customer_id = $request->input('customer_id');

        $query = Invoice::select(DB::RAW("invoice_staff.staff_name as label"), 
        DB::RAW('count(invoice_staff.staff_id) as item'))
            ->join('invoice_staff', 'invoice.id_invoice', '=', 'invoice_staff.invoice_id')
            ->where('invoice.customer_id', $id_customers)
            ->where('invoice.invoice_status', 'valid')
            ->groupBy('invoice_staff.staff_id')
            // ->orderBy('item', 'DESC')
            ->limit(12)
            ->get();
        $staff_preference_data = [];

        foreach ($query as $data) {
            $staff_preference_data['labels'][] = $data->label;
            $staff_preference_data['series'][] = intval($data->item);
        }

        return response()->json($staff_preference_data);
    }

    public function get_customer_facial_records($id_customers , Request $request)
    {
        $columns = [
            'id',
            'customer',
            'record_date',
            'facial',
            'exfoliant',
            'mask',
            'cleanser',
            'charge',
            'remarks',
            'days_ago',
        ];

        $query = DB::table('facial_records')
            ->select(
                'id',
                'customer_id',
                'customer',
                DB::raw("DATE_FORMAT(date, '%d-%m-%Y') as record_date"),
                'facial as facial_name',
                'exfoliant',
                'mask',
                'cleanser',
                'charge',
                'remarks',
                'time as record_time',
                'visit_id',
                'created_date',
                'facial_records.time',
                DB::raw("DATEDIFF(NOW(), date) as days_ago")
            )
            ->where('customer_id', $id_customers);

        // Search filter
        if ($request->has('search') && $request->input('search.value') !== '') {
            $searchValue = $request->input('search.value');
            $query->where(function ($q) use ($searchValue) {
                $q->where('customer', 'like', "%{$searchValue}%")
                ->orWhere('facial', 'like', "%{$searchValue}%")
                ->orWhere('exfoliant', 'like', "%{$searchValue}%")
                ->orWhere('mask', 'like', "%{$searchValue}%")
                ->orWhere('cleanser', 'like', "%{$searchValue}%")
                ->orWhere('remarks', 'like', "%{$searchValue}%");
            });
        }

        // Order
        if ($request->has('order')) {
            $order = $request->input('order')[0];
            $columnIndex = $order['column'];
            $columnName = $columns[$columnIndex] ?? 'id';
            $columnSortOrder = $order['dir'] ?? 'desc';
            $query->orderBy($columnName, $columnSortOrder);
        } else {
            $query->orderBy('date', 'desc');
        }

        // Pagination
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        $recordsFiltered = $query->count();
        $data = $query->skip($start)->take($length)->get();

        $recordsTotal = DB::table('facial_records')
            ->where('customer_id', $id_customers)
            ->count();

        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    public function get_customer_color_records($id_customers, Request $request)
    {
        $columns = [
            'id',
            'color_type',
            'color_number',
            'water_content',
            'color_record_date',
            'time',
            'charge',
            'color_remarks',
            'days_ago'
        ];

        $baseQuery = DB::table('color_records')
            ->where('color_records.customer_id', $id_customers)
            ->groupBy('color_records.id');

        // Apply select after defining base query so it's consistent for both count and data
        $query = clone $baseQuery;
        $query->select(
            'color_records.id',
            'color_type',
            'color_number',
            'water_content',
            'color_records.recommendation',
            DB::raw("DATE_FORMAT(date, '%d-%m-%Y') as color_record_date"),
            'time as time_min',
            'charge',
            'remarks as color_remarks',
            DB::raw("DATEDIFF(NOW(), date) as days_ago")
        );

        // Search filter
        if ($request->has('search') && $request->input('search.value') !== '') {
            $searchValue = $request->input('search.value');
            $query->where(function ($q) use ($searchValue) {
                $q->where('color_type', 'like', "%{$searchValue}%")
                ->orWhere('color_number', 'like', "%{$searchValue}%")
                ->orWhere('water_content', 'like', "%{$searchValue}%")
                ->orWhere('remarks', 'like', "%{$searchValue}%")
                ->orWhere('recommendation', 'like', "%{$searchValue}%");
            });
        }

        // Order
        if ($request->has('order')) {
            $order = $request->input('order')[0];
            $columnIndex = $order['column'];
            $columnName = $columns[$columnIndex] ?? 'color_records.id';
            $columnSortOrder = $order['dir'] ?? 'desc';
            $query->orderBy($columnName, $columnSortOrder);
        } else {
            $query->orderBy('date', 'desc');
        }

        // Pagination
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        $recordsFiltered = (clone $query)->count(DB::raw('distinct color_records.id'));

        $data = $query->skip($start)->take($length)->get();

        $recordsTotal = DB::table('color_records')
            ->where('customer_id', $id_customers)
            ->count();

        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    public function get_customer_eyelash_records($id_customers, Request $request)
    {
        $columns = [
            'id_eyelashes',
            'eyelash_type',
            'thickness',
            'length',
            'curl',
            'full_set_refill',
            'date',
            'remarks',
            'visit_id',
            'created_date',
            'price'
        ];

        $baseQuery = DB::table('eyelashes_record')
            ->where('eyelashes_record.customer_id', $id_customers)
            ->groupBy('eyelashes_record.id_eyelashes');

        // Apply select after defining base query so it's consistent for both count and data
        $query = clone $baseQuery;
        $query->select(
            'id_eyelashes',
            'eyelash_type',
            'thickness',
            'length',
            'curl',
            'full_set_refill',           
            'remarks as lash_remarks',
            'visit_id',
            'created_date',
            'price as charge',
            DB::raw("DATE_FORMAT(date, '%d-%m-%Y') as lash_record_date"),
            DB::raw("DATEDIFF(NOW(), date) as days_ago")
        );

        // Search filter
        if ($request->has('search') && $request->input('search.value') !== '') {
            $searchValue = $request->input('search.value');
            $query->where(function ($q) use ($searchValue) {
                $q->where('eyelash_type', 'like', "%{$searchValue}%")
                ->orWhere('length', 'like', "%{$searchValue}%")
                ->orWhere('remarks', 'like', "%{$searchValue}%");
            });
        }

        // Order
        if ($request->has('order')) {
            $order = $request->input('order')[0];
            $columnIndex = $order['column'];
            $columnName = $columns[$columnIndex] ?? 'eyelashes_record.id_eyelashes';
            $columnSortOrder = $order['dir'] ?? 'desc';
            $query->orderBy($columnName, $columnSortOrder);
        } else {
            $query->orderBy('date', 'desc');
        }

        // Pagination
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        $recordsFiltered = (clone $query)->count(DB::raw('distinct eyelashes_record.id_eyelashes'));

        $data = $query->skip($start)->take($length)->get();

        $recordsTotal = DB::table('eyelashes_record')
            ->where('customer_id', $id_customers)
            ->count();
        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

  
    public function transfer_retained_to_branch(Request $request)
    {   

        try {
            DB::beginTransaction();

            $branchIdFrom = session('business_id'); // from branch
            $branchIdTo   = $request->branch_id;    // to branch
            $amount       = $request->amount;
            $customerId   = $request->customer_id;
            $customerName = DB::table('customers')->where('id_customers',$customerId)->value('customer_name');

           
            $fromBusiness = Business::where('id_business', $branchIdFrom)->first();
            $toBusiness   = Business::where('id_business', $branchIdTo)->first();

            /**
             * -----------------------------
             * TRANSFER OUT (FROM BRANCH)
             * -----------------------------
             */
            $mappingBusinessIdOut = session('ho_accounts') === 'Yes' ? Business::where('ho', 'Yes')->value('id_business') : $branchIdFrom;

            $eventMappingsOut = AccountEventMapping::where('account_event_id', 35)
                ->where('business_id', $mappingBusinessIdOut)
                ->get();

            $debitAccountsOut  = [];
            $creditAccountsOut = [];

            foreach ($eventMappingsOut as $mapping) {
                $entry = "{$mapping->account_head_id}|{$mapping->transaction_type}|{$mapping->entity_name}";
                if ($mapping->transaction_type === 'debit') {
                    $debitAccountsOut[] = $entry;
                } else {
                    $creditAccountsOut[] = $entry;
                }
            }

            $voucherDescOut = "Retained Amount for customer $customerName (ID: $customerId) Transferred to Branch: $toBusiness->business_name";

            $voucherIdOut = DB::table('account_vouchers')->insertGetId([
                'business_id'           => $branchIdFrom,
                'description'           => $voucherDescOut,
                'voucher_date'          => Carbon::now(),
                'voucher_amount'        => $amount,
                'voucher_status'        => 'Active',
                'created_by'            => session('user_name'),
                'cost_center'           => '4',
                'cost_center_name'      => $toBusiness->business_name,
                'business_partner'      => 1,
                'business_partner_id'   => $customerId,
                'business_partner_name' => $customerName,
                'voucher_type'          => 4,
                'payment_mode'          => 'Cash',
                'bank_name'             => '',
                'auto_voucher'          => 'Yes',
            ]);

            // Insert debit entries (Out)
            foreach ($debitAccountsOut as $debitAccount) {
                [$headId, $type, $entity] = explode('|', $debitAccount);
                $debitAmount = ($entity === 'retained_amount') ? $amount : 0;

                if ($debitAmount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherIdOut,
                        'account_head_id'    => $headId,
                        'debit'              => round($debitAmount, 2),
                        'credit'             => 0,
                    ]);
                }
            }

            // Insert credit entries (Out)
            foreach ($creditAccountsOut as $creditAccount) {
                [$headId, $type, $entity] = explode('|', $creditAccount);
                $creditAmount = ($entity === 'payable') ? $amount : 0;

                if ($creditAmount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherIdOut,
                        'account_head_id'    => $headId,
                        'debit'              => 0,
                        'credit'             => round($creditAmount, 2),
                    ]);
                }
            }


            $totalsOut = DB::table('account_voucher_detail')
                        ->where('account_voucher_id', $voucherIdOut)
                        ->selectRaw('
                            ROUND(SUM(debit),2) as total_debit,
                            ROUND(SUM(credit),2) as total_credit
                        ')
                        ->first();

            if ((float)$totalsOut->total_debit !== (float)$totalsOut->total_credit) {
                    DB::rollBack();
                    return response()->json([
                        'succes'=>false,
                        'message' => "Transfer OUT Voucher imbalance. Debit ({$totalsOut->total_debit}) and Credit ({$totalsOut->total_credit}) not equal",
                        'message_type' => 'error'
                    ]);
            }

            /**
             * -----------------------------
             * TRANSFER IN (TO BRANCH)
             * -----------------------------
             */
            
            $mappingBusinessIdIn = $toBusiness->ho_accounts === 'Yes' ? Business::where('ho', 'Yes')->value('id_business') : $branchIdTo;

            $eventMappingsIn = AccountEventMapping::where('account_event_id', 36)->where('business_id', $mappingBusinessIdIn)->get();

            $debitAccountsIn  = [];
            $creditAccountsIn = [];

            foreach ($eventMappingsIn as $mapping) {
                $entry = "{$mapping->account_head_id}|{$mapping->transaction_type}|{$mapping->entity_name}";
                if ($mapping->transaction_type === 'debit') {
                    $debitAccountsIn[] = $entry;
                } else {
                    $creditAccountsIn[] = $entry;
                }
            }

            $voucherDescIn = "Retained Amount for customer $customerName (ID: $customerId) Transferred from Branch: $fromBusiness->business_name";
            $voucherIdIn = DB::table('account_vouchers')->insertGetId([
                'business_id'           => $branchIdTo,
                'description'           => $voucherDescIn,
                'voucher_date'          => Carbon::now(),
                'voucher_amount'        => $amount,
                'voucher_status'        => 'Active',
                'created_by'            => session('user_name'),
                'cost_center'           => '4',
                'cost_center_name'      => $fromBusiness->business_name,
                'business_partner'      => 1,
                'business_partner_id'   => $customerId,
                'business_partner_name' => $customerName,
                'voucher_type'          => 4,
                'payment_mode'          => 'Cash',
                'bank_name'             => '',
                'auto_voucher'          => 'Yes',
            ]);

            // Insert debit entries (In)
            foreach ($debitAccountsIn as $debitAccount) {
                [$headId, $type, $entity] = explode('|', $debitAccount);
                $debitAmount = ($entity === 'receivable') ? $amount : 0;

                if ($debitAmount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherIdIn,
                        'account_head_id'    => $headId,
                        'debit'              => round($debitAmount, 2),
                        'credit'             => 0,
                    ]);
                }
            }

            // Insert credit entries (In)
            foreach ($creditAccountsIn as $creditAccount) {
                [$headId, $type, $entity] = explode('|', $creditAccount);
                $creditAmount = ($entity === 'retained_amount') ? $amount : 0;

                if ($creditAmount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherIdIn,
                        'account_head_id'    => $headId,
                        'debit'              => 0,
                        'credit'             => round($creditAmount, 2),
                    ]);
                }
            } 


            $totalsIN = DB::table('account_voucher_detail')
                        ->where('account_voucher_id', $voucherIdIn)
                        ->selectRaw('
                            ROUND(SUM(debit),2) as total_debit,
                            ROUND(SUM(credit),2) as total_credit
                        ')
                        ->first();

            if ((float)$totalsIN->total_debit !== (float)$totalsIN->total_credit) {
                    DB::rollBack();
                    return response()->json([
                        'succes'=>false,
                        'message' => "Transfer OUT Voucher imbalance. Debit ({$totalsIN->total_debit}) and Credit ({$totalsIN->total_credit}) not equal",
                        'message_type' => 'error'
                    ]);
            }

            DB::commit(); 
            
            return response()->json([
                'succes'=>true,
                'message'        => 'Retained amount transferred successfully',
                'voucher_id_out' => $voucherIdOut,
                'voucher_id_in'  => $voucherIdIn
            ]);

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback if any error occurs
            return response()->json([
                'succes'=>false,
                'message' => 'Error transferring retained amount'.$e->getMessage(),
                'error'   => $e->getMessage()
            ], 500);
        }
    }
    
}
