<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Franchise;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;
use App\Models\Invoice;
use App\Models\InvoiceDetail;
use App\Models\InvoiceProducts;
use App\Models\InvoiceStaff;
use App\Models\CashRegister;
use App\Models\InvoiceDetails;
use App\Models\CustomerVisits;
use App\Models\VisitServices;
use App\Models\Customers;
use App\Models\VisitAdvance;
use App\Models\Staff;
use App\Models\OrderVouchers;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Bus;
use App\Services\ProductsStockService; 
use Carbon\Carbon;
use App\Http\Controllers\AccountsController;
use App\Models\AccountEventMapping;
use Carbon\CarbonPeriod;
use PhpOffice\PhpSpreadsheet\Calculation\MathTrig\Round;
use Illuminate\Support\Facades\Log;

class DashboardController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function daily_sheet(Request $request)
    {
        $today = $request->input('date', date('Y-m-d'));
        if (!isset($today)) {
            $today = date('Y-m-d');
        }
        $yesterday = date('Y-m-d', strtotime($today . ' -1 day'));

        $businessId = $request->input('business_id');

        if (!$businessId) {
           $businessId = session('business_id');
        }
        $business = Business::find($businessId)->business_name;

       // 1. Yesterday Till
        $yesterdayTill = CashRegister::where('business_id', $businessId)
            ->whereDate('cash_register_date', '=', $yesterday)
            ->orderByDesc('cash_register_date')
            ->first();

        // 2. Today Cash Register
        $cashRegister = CashRegister::where('business_id', $businessId)
            ->whereDate('cash_register_date', $today)
            ->first();

        // 3. Today Invoices Total
        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $businessId;
        }

        $vouchers = DB::table('account_vouchers as av')
            ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->join('account_voucher_detail as vd', 'vd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('account_event_mapping as aem', function ($join) use ($common_business_id) {
                $join->on('aem.account_head_id', '=', 'vd.account_head_id')
                    ->where('aem.business_id', '=', $common_business_id)
                    ->where('aem.account_event_id', '=', 1);
            })
            ->select([
                'i.id_invoice',
                'i.invoice_type',
                'i.invoice_day_seq',
                'i.invoice_seq',
                'i.invoice_number',
                DB::raw("DATE_FORMAT(i.invoice_date, '%d-%m-%Y %H:%i') as invoice_date"),
                'i.invoice_status',
                'i.invoice_extra',
                'i.customer_id',
                'i.customer_name',
                'i.customer_cell',
                'i.visit_id',
                'i.created_by',
                'av.created_on',
                DB::raw("
                    SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN vd.debit ELSE 0 END) AS cash,
                    SUM(CASE WHEN aem.entity_name = 'paid_card' THEN vd.debit ELSE 0 END) AS card,
                    SUM(CASE WHEN aem.entity_name = 'paid_check' THEN vd.debit ELSE 0 END) AS bank,
                    SUM(CASE WHEN aem.entity_name = 'paid_voucher' THEN vd.debit ELSE 0 END) AS voucher,
                    SUM(CASE WHEN aem.entity_name = 'paid_online' THEN vd.debit ELSE 0 END) AS online,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_used' THEN vd.debit ELSE 0 END) AS loyalty_used,
                    SUM(CASE WHEN aem.entity_name = 'retained_amount_used' THEN vd.debit ELSE 0 END) AS retained_used,
                    SUM(CASE WHEN aem.entity_name = 'advance_adjusted' THEN vd.debit ELSE 0 END) AS advance_adjusted,
                    SUM(CASE WHEN aem.entity_name = 'balance' THEN vd.debit ELSE 0 END) AS balance,
                    SUM(CASE WHEN aem.entity_name = 'sale_discount_service' THEN vd.debit ELSE 0 END) AS service_discount,
                    SUM(CASE WHEN aem.entity_name = 'sale_disount_retail' THEN vd.debit ELSE 0 END) AS retail_discount,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_expense' THEN vd.debit ELSE 0 END) AS loyalty_received,
                    SUM(CASE WHEN aem.entity_name = 'receivable' THEN vd.credit ELSE 0 END) AS recovery,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_earned' THEN vd.credit ELSE 0 END) AS loyalty_earned,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_training' THEN vd.credit ELSE 0 END) AS training_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_service' THEN vd.credit ELSE 0 END) AS service_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_retail' THEN vd.credit ELSE 0 END) AS retail_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_recurring' THEN vd.credit ELSE 0 END) AS recurring_income,
                    SUM(CASE WHEN aem.entity_name = 'return_cash' THEN vd.credit ELSE 0 END) AS return_cash,
                    SUM(CASE WHEN aem.entity_name = 'cc_fee' THEN vd.credit ELSE 0 END) AS cc_fee,
                    SUM(CASE WHEN aem.entity_name = 'retained_amount' THEN vd.credit ELSE 0 END) AS retained,
                    SUM(CASE WHEN aem.entity_name = 'sales_tax' THEN vd.credit ELSE 0 END) AS sales_tax,
                    SUM(CASE WHEN aem.entity_name = 'cc_tip' THEN vd.credit ELSE 0 END) AS cc_tip
                ")
            ])
            ->whereDate('av.voucher_date', '=', $today)
            ->where('av.business_id', $businessId)
            ->groupBy([
                'i.id_invoice',
                'i.invoice_type',
                'i.invoice_day_seq',
                'i.invoice_seq',
                'i.invoice_number',
                'i.invoice_date',
                'i.invoice_status',
                'i.invoice_extra',
                'i.customer_id',
                'i.customer_name',
                'i.customer_cell',
                'i.visit_id'
            ])
            ->get();

            foreach ($vouchers as $voucher) {
                //get invoice products
                $invoiceProducts = InvoiceProducts::where('invoice_id', $voucher->id_invoice)->get();
                //attach to voucher
                $voucher->products = $invoiceProducts;

                //get invoice services
                $invoiceServices = InvoiceDetails::select('*', 
                DB::raw("TIMESTAMPDIFF(MINUTE, visit_service_start, visit_service_end) AS duration"),
                DB::raw("date_format(visit_service_start, '%H:%i') AS service_time")
                )
                ->join('visit_services as vs', 'vs.id_visit_services', '=', 'invoice_details.visit_service_id')
                ->where('invoice_id', $voucher->id_invoice)->get();
                //attach to voucher
                $voucher->services = $invoiceServices;
            }

        // 4. Today Gift Voucher Total
        $gift_vouchers = DB::table('account_vouchers as av')
            ->select(
                'av.id_account_vouchers',
                'customers.customer_name',
                'av.created_on',
                DB::raw("DATE_FORMAT(av.created_on, '%H:%i') AS voucher_date"),
                'av.voucher_status',
                'av.order_voucher_id',
                'gv.id_order_vouchers',
                'av.created_by',
                DB::raw("
                    SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN vd.debit ELSE 0 END) AS cash,
                    SUM(CASE WHEN aem.entity_name = 'paid_card' THEN vd.debit ELSE 0 END) AS card,
                    SUM(CASE WHEN aem.entity_name = 'paid_check' THEN vd.debit ELSE 0 END) AS bank,
                    SUM(CASE WHEN aem.entity_name = 'paid_online' THEN vd.debit ELSE 0 END) AS online,
                    SUM(CASE WHEN aem.entity_name = 'cc_fee_expense' THEN vd.debit ELSE 0 END) AS cc_fee_expense,
                    SUM(CASE WHEN aem.entity_name = 'advance_amout' THEN vd.credit ELSE 0 END) AS advance,
                    SUM(CASE WHEN aem.entity_name = 'cc_fee' THEN vd.credit ELSE 0 END) AS cc_fee,
                    SUM(CASE WHEN aem.entity_name = 'sales_tax' THEN vd.credit ELSE 0 END) AS sales_tax,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_service' THEN vd.credit ELSE 0 END) AS grosstotal_service                    
                ")
            )
            ->join('account_voucher_detail as vd', 'vd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('order_vouchers as gv', 'gv.id_order_vouchers', '=', 'av.order_voucher_id')
            ->join('customers', 'av.business_partner_id', '=', 'customers.id_customers')
            
            ->join('account_event_mapping as aem', function ($join) use ($common_business_id) {
                $join->on('aem.account_head_id', '=', 'vd.account_head_id')
                    ->where('aem.business_id', '=', $common_business_id)
                    ->where('aem.account_event_id', '=', 1);
            })
            
            ->where('av.business_id', $businessId)
            ->whereDate('av.voucher_date', '=', $today)
            ->groupBy('av.id_account_vouchers', 
                'customers.customer_name',
                'av.voucher_date',
                'av.voucher_status',
                'av.order_voucher_id',
                'gv.id_order_vouchers',
                'av.created_by'
            )
            ->get();

        
        // 5. Today Training Invoices Total
        $training_vouchers = DB::table('account_vouchers as av')
            ->select(
                'av.id_account_vouchers',
                'customers.customer_name',
                'av.created_on',
                DB::raw("DATE_FORMAT(av.created_on, '%H:%i') AS voucher_date"),
                'av.voucher_status',
                'av.program_invoice_payment_id',
                'pi.id_program_invoices',
                'av.program_enrollment_id',
                'pt.program_type',
                'av.created_by',
                DB::raw("
                    SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN vd.debit ELSE 0 END) AS cash,
                    SUM(CASE WHEN aem.entity_name = 'cash_return' THEN vd.debit ELSE 0 END) AS cash_return,
                    SUM(CASE WHEN aem.entity_name = 'paid_card' THEN vd.debit ELSE 0 END) AS card,
                    SUM(CASE WHEN aem.entity_name = 'paid_check' THEN vd.debit ELSE 0 END) AS bank,
                    SUM(CASE WHEN aem.entity_name = 'paid_voucher' THEN vd.debit ELSE 0 END) AS voucher,
                    SUM(CASE WHEN aem.entity_name = 'paid_online' THEN vd.debit ELSE 0 END) AS online,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_used' THEN vd.debit ELSE 0 END) AS loyalty_used,
                    SUM(CASE WHEN aem.entity_name = 'retained_amount_used' THEN vd.debit ELSE 0 END) AS retained_used,
                    SUM(CASE WHEN aem.entity_name = 'advance_adjusted' THEN vd.debit ELSE 0 END) AS advance_adjusted,
                    SUM(CASE WHEN aem.entity_name = 'balance' THEN vd.debit ELSE 0 END) AS balance,
                    SUM(CASE WHEN aem.entity_name = 'sale_discount_service' THEN vd.debit ELSE 0 END) AS service_discount,
                    SUM(CASE WHEN aem.entity_name = 'sale_disount_retail' THEN vd.debit ELSE 0 END) AS retail_discount,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_expense' THEN vd.debit ELSE 0 END) AS loyalty_received,
                    SUM(CASE WHEN aem.entity_name = 'receivable' THEN vd.credit ELSE 0 END) AS recovery,
                    SUM(CASE WHEN aem.entity_name = 'loyalty_earned' THEN vd.credit ELSE 0 END) AS loyalty_earned,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_training' THEN vd.credit ELSE 0 END) AS training_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_service' THEN vd.credit ELSE 0 END) AS service_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_retail' THEN vd.credit ELSE 0 END) AS retail_income,
                    SUM(CASE WHEN aem.entity_name = 'grosstotal_recurring' THEN vd.credit ELSE 0 END) AS recurring_income,
                    SUM(CASE WHEN aem.entity_name = 'return_cash' THEN vd.credit ELSE 0 END) AS return_cash,
                    SUM(CASE WHEN aem.entity_name = 'cc_fee' THEN vd.credit ELSE 0 END) AS cc_fee,
                    SUM(CASE WHEN aem.entity_name = 'retained_amount' THEN vd.credit ELSE 0 END) AS retained,
                    SUM(CASE WHEN aem.entity_name = 'sales_tax' THEN vd.credit ELSE 0 END) AS sales_tax,
                    SUM(CASE WHEN aem.entity_name = 'cc_tip' THEN vd.credit ELSE 0 END) AS cc_tip
                ")
            )
            ->join('account_voucher_detail as vd', 'vd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('program_invoices as pi', 'pi.id_program_invoices', '=', 'av.program_invoice_id')
            ->join('program_enrollment as pe', 'av.program_enrollment_id', '=', 'pe.id_program_enrollment')
            ->join('program_types as pt', 'pt.id_program_types', '=', 'pe.program_type_id')
            ->join('customers', 'av.business_partner_id', '=', 'customers.id_customers')
            
            ->join('account_event_mapping as aem', function ($join) use ($common_business_id) {
                $join->on('aem.account_head_id', '=', 'vd.account_head_id')
                    ->where('aem.business_id', '=', $common_business_id)
                    ->where('aem.account_event_id', '=', 1);
            })
            
            ->where('av.business_id', $businessId)
            ->whereDate('av.voucher_date', '=', $today)
            ->groupBy('av.id_account_vouchers', 
                'customers.customer_name',
                'av.voucher_date',
                'av.voucher_status',
                'av.program_invoice_payment_id',
                'pi.id_program_invoices',
                'av.program_enrollment_id',
                'pt.program_type',
                'av.created_by'
            )
            ->get();
 
            //5. Get All Advance Vouchers
            $advance_vouchers = DB::table('account_vouchers as v')
                ->select(
                    'v.id_account_vouchers',
                    'va.customer_visit_id',
                    'customers.customer_name', 
                    DB::raw("GROUP_CONCAT(DISTINCT vs.s_type SEPARATOR ', ') as service_names"),
                    'v.voucher_status',
                    'v.created_by',
                    'v.created_on',
                    DB::raw("DATE_FORMAT(v.created_on, '%H:%i') AS voucher_date"),
                    'va.id_visit_advance',
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN vd.debit ELSE 0 END) AS cash"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_card' THEN vd.debit ELSE 0 END) AS card"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_check' THEN vd.debit ELSE 0 END) AS bank"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_online' THEN vd.debit ELSE 0 END) AS online"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'cc_fee_expense' THEN vd.debit ELSE 0 END) AS cc_fee_expense"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'advance_amout' THEN vd.credit ELSE 0 END) AS advance"),
                    DB::raw("SUM(CASE WHEN aem.entity_name = 'cc_fee' THEN vd.credit ELSE 0 END) AS cc_fee")
                )
                ->join('account_voucher_detail as vd', 'v.id_account_vouchers', '=', 'vd.account_voucher_id')
                ->join('visit_advance as va', 'v.visit_advance_id', '=', 'va.id_visit_advance')
                ->join('customer_visits as cv', 'va.customer_visit_id', '=', 'cv.id_customer_visits')
                ->join('visit_services as vs', 'cv.id_customer_visits', '=', 'vs.customer_visit_id')
                ->join('customers', 'v.business_partner_id', '=', 'customers.id_customers')
                ->join(DB::raw("(
                    SELECT account_head_id, MIN(entity_name) AS entity_name
                    FROM account_event_mapping
                    WHERE account_event_id = 1 AND business_id = {$common_business_id}
                    GROUP BY account_head_id
                ) as aem"), 'aem.account_head_id', '=', 'vd.account_head_id')
                ->whereDate('v.voucher_date', $today)
                ->where('v.business_id', $businessId)
                ->groupBy('v.id_account_vouchers')
                ->get();

                // 5. Get All Expense Vouchers
                
                $expenses = DB::table('account_vouchers as v')
                    ->select(
                        'v.id_account_vouchers',
                        'v.business_partner_name',
                        'v.description',
                        'v.voucher_status',
                        'v.created_by',
                        DB::raw("DATE_FORMAT(v.created_on, '%H:%i') AS voucher_date"),
                        DB::raw("SUM(vd.debit) AS expense_amount"),
                        DB::raw("GROUP_CONCAT(ah.account_head SEPARATOR ' <iconify-icon icon=\'emojione:left-right-arrow\' class=\'icon fs-12\'></iconify-icon> ') AS account_heads")
                    )
                    ->join('account_voucher_detail as vd', 'v.id_account_vouchers', '=', 'vd.account_voucher_id')
                    ->join('account_heads as ah', 'vd.account_head_id', '=', 'ah.id_account_heads')
                    ->whereDate('v.voucher_date', $today)
                    ->where('v.business_id', $businessId)
                    ->where('v.voucher_type', 1) // Payment Voucher Type
                    ->groupBy('v.id_account_vouchers')
                    ->get();

                // 6. Get Cash Register Details for Yesterday    
                $yesterdayCashRegister = DB::table('cash_register')
                    ->whereDate('cash_register_date', $yesterday)
                    ->where('business_id', $businessId)
                    ->first();

                // 7. Get Cash Register Details for Today
                $todayCashRegister = DB::table('cash_register')
                    ->whereDate('cash_register_date', $today)
                    ->where('business_id', $businessId)
                    ->first();
                $todayCashRegisterjson=json_encode($todayCashRegister);

        //return response()->json(compact('yesterdayTill', 'cashRegister', 'vouchers'));
         return view('dashboards.daily_sheet')
        ->with(compact('yesterdayTill', 'cashRegister', 'vouchers', 'today', 'yesterday', 'businessId', 'gift_vouchers', 'training_vouchers', 'advance_vouchers', 'business', 'expenses', 'yesterdayCashRegister', 'todayCashRegister', 'todayCashRegisterjson'));

    }

    public function main_dashboard(Request $request){
      $branches = DB::table('business')->select('id_business','business_name')->where('ho_accounts','Yes')->orwhere('ho','Yes')->get();
      return view('dashboards.dashboard',compact('branches'));
    }    
    /**
     * Get dashboard statistics (visits, revenue, customers, satisfaction)
     */
    public function getDashboardStats(Request $request)
    {   
        $from = Carbon::parse($request->from);
        $to = Carbon::parse($request->to);
        $business_id = $request->business_id;

        $days = \Carbon\Carbon::parse($from)->diffInDays($to) + 1;
    
        $months = $from->diffInMonths($to) + 1;
        $prevFrom = $from->copy()
            ->subMonths($months)
            ->startOfMonth();

        $prevTo = $from->copy()
            ->subMonth()
            ->endOfMonth();


        $business_id_for_account_head = session('business_id');
        if (session('ho_accounts') === 'Yes') {
            $business_id_for_account_head = DB::table('business')->where('ho', 'Yes')->value('id_business');
        }

        $businessId = $request->input('business_id', $business_id);
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        $currentDate = date('Y-m-d');
        $currentMonthStart = date('Y-m-01');
        $lastMonthStart = date('Y-m-01', strtotime('-1 month'));
        $lastMonthEnd = date('Y-m-t', strtotime('-1 month'));

        // Get Total Visits (open + invoiced) and Invoiced Visits from customer_visits table for current month
        $visitStats = DB::table('customer_visits')
            ->select(
                DB::raw('COUNT(*) as total_visits'),
                DB::raw('SUM(CASE WHEN visit_status = "invoiced" THEN 1 ELSE 0 END) as invoiced_visits')
            )
            ->where('business_id', $businessId)
            ->whereIn('visit_status', ['open', 'invoiced'])
            ->whereBetween(DB::raw('DATE(customer_visit_date)'), [$from, $to])
            ->first();

        $totalVisitsCurrent = (int)($visitStats->total_visits ?? 0);
        $invoicedVisitsCurrent = (int)($visitStats->invoiced_visits ?? 0);
        $invoicedPercentage = $totalVisitsCurrent > 0 
            ? round(($invoicedVisitsCurrent / $totalVisitsCurrent) * 100, 1) 
            : 0;

        // Get service account head ID (same approach as getMonthlyPerformance)
        $serviceAccount = AccountsController::get_event_account('1', 'grosstotal_service');
        $discountAccount = AccountsController::get_event_account('1', 'sale_discount_service');
        if (!$serviceAccount || !$discountAccount) {
            return response()->json(['error' => 'Service or discount account not found'], 400);
        }

        // Get Total Revenue from account vouchers for current month (Service only - discount added - retail removed)
        // Updated to match getMonthlyPerformance approach: directly target service account head ID
        $currentMonthStats = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->select(
                DB::raw("
                    COALESCE(SUM(
                        CASE 
                            WHEN avd.account_head_id IN ({$serviceAccount->account_head_id}, {$discountAccount->account_head_id})
                            THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0)
                            ELSE 0
                        END
                    ), 0) as total_revenue
                ")
            )
            ->where('av.voucher_status', 'Active')
            ->where('i.invoice_status', 'valid')
            ->where('i.business_id', $businessId)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
            ->first();

        // Get Total Revenue from account vouchers for last month (Service only - retail removed)
        // Updated to match getMonthlyPerformance approach: directly target service account head ID
        $lastMonthStats = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->select(
                DB::raw("
                    COALESCE(SUM(
                        CASE 
                            WHEN avd.account_head_id IN ({$serviceAccount->account_head_id}, {$discountAccount->account_head_id})
                            THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0)
                            ELSE 0
                        END
                    ), 0) as total_revenue
                ")
            )
            ->where('av.voucher_status', 'Active')
            ->where('i.invoice_status', 'valid')
            ->where('i.business_id', $businessId)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$prevFrom, $prevTo])
            ->first();

        $revenueCurrent = (float)($currentMonthStats->total_revenue ?? 0);
        $revenueLast = (float)($lastMonthStats->total_revenue ?? 0);

        // Repeat Customers (customers with 2+ visits in last 3 months)
        $threeMonthsAgo = date('Y-m-d', strtotime('-3 months'));
        
        // Get customer visit counts for last 3 months
        $customerVisitCounts = DB::table('customer_visits')
            ->select(
                'customer_id',
                DB::raw('COUNT(*) as visit_count')
            )
            ->where('business_id', $businessId)
            ->whereNotNull('customer_id')
            ->where('visit_status', 'invoiced')
            ->whereBetween(DB::raw('DATE(customer_visit_date)'), [$prevFrom, $prevTo])
            ->groupBy('customer_id')
            ->get();

        $totalUniqueCustomers = $customerVisitCounts->count();
        $repeatCustomers = $customerVisitCounts->filter(function($customer) {
            return $customer->visit_count >= 2;
        })->count();

        $repeatPercentage = $totalUniqueCustomers > 0 
            ? round(($repeatCustomers / $totalUniqueCustomers) * 100, 1) 
            : 0;

        // Customer Satisfaction
        $totalInvoicesWithFeedback = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            // ->whereBetween('invoice_date', [$from, $to])
            ->whereBetween(DB::raw('DATE(invoice_date)'), [$from, $to])
            ->whereNotNull('feedback_status')
            ->where('feedback_status', '!=', '')
            ->count();

        $positiveFeedback = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            // ->whereBetween('invoice_date', [$from, $to])
            ->whereBetween(DB::raw('DATE(invoice_date)'), [$from, $to])
            ->whereIn('feedback_status', ['positive', 'good'])
            ->count();

        $satisfactionPercentage = $totalInvoicesWithFeedback > 0 
            ? round(($positiveFeedback / $totalInvoicesWithFeedback) * 100, 2) 
            : 0;

        return response()->json([
            'total_visits' => [
                'total' => $totalVisitsCurrent,
                'invoiced' => $invoicedVisitsCurrent,
                'percentage' => $invoicedPercentage
            ],
            'total_revenue' => [
                'current' => round($revenueCurrent, 2),
                'last' => round($revenueLast, 2),
                'change' => $revenueLast > 0 
                    ? round((($revenueCurrent - $revenueLast) / $revenueLast) * 100, 1)
                    : 0
            ],
            'repeat_customers' => [
                'repeat' => $repeatCustomers,
                'total' => $totalUniqueCustomers,
                'percentage' => $repeatPercentage
            ],
            'customer_satisfaction' => [
                'percentage' => $satisfactionPercentage,
                'total_feedback' => $totalInvoicesWithFeedback,
                'positive' => $positiveFeedback
            ],
            'from'=>Carbon::parse($from)->format('d-M-Y'),
            'to'=>Carbon::parse($to)->format('d-M-Y'),
            'prevFrom' =>Carbon::parse($prevFrom)->format('M'),
            'prevTo'=>Carbon::parse($prevTo)->format('M'),
        ]);
    }
    
    public function getTopCustomers(Request $request)
    {   
        $from = $request->from;
        $to = $request->to;
        $businessId = $request->input('business_id');

        // Allow dynamic limit (default 10, max 30)
        $limit = (int) $request->input('limit', 10);
        if ($limit <= 0) {
            $limit = 10;
        }
        if ($limit > 30) {
            $limit = 30;
        }

        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        try {

            /* ================= Get Accounts ================= */
            // Retail account - commented out as retail part removed from calculation
            // $salesAccount   = AccountsController::get_event_account('1', 'grosstotal_retail');
            $serviceAccount = AccountsController::get_event_account('1', 'grosstotal_service');
            $receivableAccount = AccountsController::get_event_account('1', 'receivable');
            $discountAccount   = AccountsController::get_event_account('1', 'sale_discount_service');

            // Updated check - removed salesAccount requirement
            if (!$serviceAccount || !$receivableAccount) {
                return response()->json(['error' => 'Required accounts not found'], 400);
            }

            /* ================= STEP-1: Top Customers by REVENUE (Service only - retail removed) ================= */
            $topCustomerIds = DB::table('account_voucher_detail as avd')
                ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
                ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
                ->where('av.voucher_status', 'Active')
                ->where('i.invoice_status', 'valid')
                ->where('i.invoice_type', 'service')
                ->where('i.business_id', $businessId)
                ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
                // Retail account removed - only service account used
                ->whereIn('avd.account_head_id', [
                    // $salesAccount->account_head_id, // Retail - commented out
                    $serviceAccount->account_head_id,
                    $discountAccount->account_head_id
                ])
                ->groupBy('i.customer_id')
                ->orderByRaw('SUM(IFNULL(avd.credit,0) - IFNULL(avd.debit,0)) DESC')
                ->limit($limit)
                ->pluck('customer_id');

            if ($topCustomerIds->isEmpty()) {
                // return response()->json([]);
            }


            /* ================= STEP-2: Customer Detail + Revenue + Receivable ================= */
            $customers = DB::table('account_voucher_detail as avd')
                ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
                ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
                ->leftJoin('customers as c', 'i.customer_id', '=', 'c.id_customers')
                ->where('av.voucher_status', 'Active')
                ->where('i.invoice_status', 'valid')
                ->where('i.invoice_type', 'service')
                ->where('i.business_id', $businessId)
                ->whereIn('i.customer_id', $topCustomerIds)
                ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
                ->selectRaw("
                    i.customer_id as id_customers,
                    COALESCE(c.customer_name, MAX(i.customer_name), 'Unknown') as customer_name,
                    COALESCE(c.customer_cell, MAX(i.customer_cell), '') as customer_cell,

                    COUNT(DISTINCT i.id_invoice) as total_visits,

                    ROUND(SUM(
                        CASE 
                            WHEN avd.account_head_id IN ({$serviceAccount->account_head_id}, {$discountAccount->account_head_id})
                            THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0)
                            ELSE 0
                        END
                    ),2) as total_revenue,

                    ROUND(SUM(
                        CASE 
                            WHEN avd.account_head_id = {$receivableAccount->account_head_id}
                            THEN IFNULL(avd.debit,0) - IFNULL(avd.credit,0)
                            ELSE 0
                        END
                    ), 2) as total_balance,

                    MAX(i.invoice_date) as last_visit_date,
                    DATEDIFF(CURDATE(), MAX(i.invoice_date)) as days_since_last_visit
                ")
                ->groupBy('i.customer_id')
                ->orderByDesc('total_revenue')
                ->get();


            /* ================= STEP-3: Response Format ================= */
            $formatted = $customers->map(function ($customer) {
                return [
                    'id_customers' => $customer->id_customers,
                    'customer_name' => $customer->customer_name,
                    'customer_cell' => $customer->customer_cell,
                    'total_visits' => (int) $customer->total_visits,
                    'total_revenue' => (float) $customer->total_revenue,
                    'total_balance' => (float) $customer->total_balance,
                    'days_since_last_visit' => $customer->days_since_last_visit !== null
                        ? (int) $customer->days_since_last_visit
                        : null,
                    'last_visit_date' => $customer->last_visit_date
                        ? Carbon::parse($customer->last_visit_date)->format('d-m-Y')
                        : null
                ];
            });

            $data['customers'] = $formatted;
            $data['from'] = Carbon::parse($from)->format('d-M-Y');
            $data['to'] = Carbon::parse($to)->format('d-M-Y');
            return response()->json($data);

        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading customers'], 500);
        }
    }

    /**
     * Get top 10 most popular services
     */
    public function getTopServices(Request $request)
    {
        $businessId = $request->input('business_id', session('business_id'));
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        $lastMonthStart = date('Y-m-01', strtotime('-1 month'));
        $currentDate = date('Y-m-d');

        try {
            $services = DB::table('visit_services as vs')
                ->select(
                    'vs.service_id',
                    DB::raw("COALESCE(vs.service_name, 'Unknown Service') as service_name"),
                    DB::raw("COALESCE(sc.service_category, 'Uncategorized') as service_category"),
                    DB::raw("COUNT(*) as booking_count"),
                    DB::raw("COUNT(DISTINCT vs.customer_visit_id) as unique_visits"),
                    DB::raw("COALESCE(ROUND(SUM(vs.s_rate), 2), 0) as total_revenue"),
                    DB::raw("COALESCE(ROUND(AVG(vs.s_rate), 2), 0) as avg_service_rate")
                )
                ->join('customer_visits as cv', 'vs.customer_visit_id', '=', 'cv.id_customer_visits')
                ->leftJoin('business_services as bs', function($join) use ($businessId) {
                    $join->on('vs.service_id', '=', 'bs.id_business_services')
                         ->where('bs.business_id', '=', $businessId);
                })
                ->leftJoin('service_category as sc', 'bs.service_category_id', '=', 'sc.id_service_category')
                ->where('cv.business_id', $businessId)
                ->where('vs.visit_service_status', 'Active')
                ->whereNotNull('vs.visit_service_start')
                ->whereBetween(DB::raw('DATE(vs.visit_service_start)'), [$lastMonthStart, $currentDate])
                ->groupBy('vs.service_id', 'vs.service_name', 'sc.service_category')
                ->orderBy('booking_count', 'DESC')
                ->orderBy('total_revenue', 'DESC')
                ->limit(10)
                ->get();

            return response()->json($services);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading services: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get monthly performance data (last 31 days) - matching previous implementation
     */

    public function getMonthlyPerformance(Request $request)
    {
        try {
            $businessId = $request->input('business_id');
            if (!$businessId) {
                return response()->json(['error' => 'Business ID required'], 400);
            }

            // ===================== Dynamic Dates =====================
            $from = Carbon::parse($request->from)->startOfDay();
            $to   = Carbon::parse($request->to)->endOfDay();

            $diffInMonths =  $from->diffInMonths($to); // calculate months difference

            $diffInMonths = round($diffInMonths);
            // ===================== Accounts =====================
            // Retail account - commented out as retail part removed from calculation
            // $salesAccount      = AccountsController::get_event_account('1', 'grosstotal_retail');
            $serviceAccount    = AccountsController::get_event_account('1', 'grosstotal_service');
            $receivableAccount = AccountsController::get_event_account('1', 'receivable');
            $advanceAccount    = AccountsController::get_event_account('2', 'advance_amount');
            $discountAccount   = AccountsController::get_event_account('1', 'sale_discount_service');

            // ===================== Determine Grouping =====================
            $groupBy =  $diffInMonths > 2 ? DB::raw('YEAR(av.voucher_date), MONTH(av.voucher_date)') : DB::raw('DATE(av.voucher_date)');
            $selectDate = $diffInMonths > 2 ? DB::raw('DATE_FORMAT(av.voucher_date, "%Y-%m") as invoice_date') : DB::raw('DATE(av.voucher_date) as invoice_date');

            // ===================== Invoice Revenue & Receivable =====================
            $invoiceData = DB::table('account_voucher_detail as avd')
                ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
                ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
                ->where('av.voucher_status', 'Active')
                ->where('i.invoice_status', 'valid')
                ->where('i.business_id', $businessId)
                ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from->toDateString(), $to->toDateString()])
                ->groupBy($groupBy)
                ->select(
                    $selectDate,
                    DB::raw("
                        SUM(
                            CASE 
                                WHEN avd.account_head_id IN ({$serviceAccount->account_head_id}, {$discountAccount->account_head_id})
                                THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0)
                                ELSE 0
                            END
                        ) as revenue
                    "),
                    DB::raw("
                        SUM(
                            CASE 
                                WHEN avd.account_head_id = {$receivableAccount->account_head_id}
                                THEN IFNULL(avd.debit,0) - IFNULL(avd.credit,0)
                                ELSE 0
                            END
                        ) as receivable
                    ")
                )
                ->get()
                ->keyBy('invoice_date');

            // ===================== Advance (Voucher Based) =====================
            $groupByAdvance = $diffInMonths > 2 ? DB::raw('YEAR(av.voucher_date), MONTH(av.voucher_date)') : DB::raw('DATE(av.voucher_date)');
            $selectAdvance = $diffInMonths > 2 ? DB::raw('DATE_FORMAT(av.voucher_date, "%Y-%m") as advance_date') : DB::raw('DATE(av.voucher_date) as advance_date');

            $advanceData = DB::table('account_voucher_detail as avd')
                ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
                ->join('visit_advance as va', 'av.visit_advance_id', '=', 'va.id_visit_advance')
                ->join('customer_visits as cv', 'va.customer_visit_id', '=', 'cv.id_customer_visits')
                ->where('av.voucher_status', 'Active')
                ->where('cv.business_id', $businessId)
                ->where('cv.visit_status', '!=', 'canceled')
                ->where('avd.account_head_id', $advanceAccount->account_head_id)
                ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from->toDateString(), $to->toDateString()])
                ->groupBy($groupByAdvance)
                ->select(
                    $selectAdvance,
                    DB::raw('COALESCE(SUM(IFNULL(avd.credit,0) - IFNULL(avd.debit,0)),0) as advance_amount')
                )
                ->get()
                ->keyBy('advance_date');

            // ===================== Services & Customers =====================
            $groupByServices = $diffInMonths > 2 ? DB::raw('YEAR(visit_services.visit_service_start), MONTH(visit_services.visit_service_start)') : DB::raw('DATE(visit_services.visit_service_start)');
            $selectServices = $diffInMonths > 2 ? DB::raw('DATE_FORMAT(visit_services.visit_service_start, "%Y-%m") as service_date') : DB::raw('DATE(visit_services.visit_service_start) as service_date');

            $servicesData = DB::table('visit_services')
                ->join('customer_visits', 'visit_services.customer_visit_id', '=', 'customer_visits.id_customer_visits')
                ->where('visit_services.business_id', $businessId)
                ->where('customer_visits.visit_status', '!=', 'canceled')
                ->where('visit_services.visit_service_status', 'Active')
                ->whereBetween(DB::raw('DATE(visit_services.visit_service_start)'), [$from->toDateString(), $to->toDateString()])
                ->groupBy($groupByServices)
                ->select(
                    $selectServices,
                    DB::raw('COUNT(visit_services.service_id) as services_count'),
                    DB::raw('COUNT(DISTINCT visit_services.customer_visit_id) as customers_count')
                )
                ->get()
                ->keyBy('service_date');

            // ===================== Build Final Response =====================
            $performanceData = [];

            // Create date or month list
            if ($diffInMonths > 2) {
                $period = CarbonPeriod::create($from->copy()->startOfMonth(), '1 month', $to->copy()->endOfMonth());
                foreach ($period as $date) {
                    $key = $date->format('Y-m');
                    $invoice  = $invoiceData->get($key);
                    $advance  = $advanceData->get($key);
                    $services = $servicesData->get($key);
                    // Retail voucher data - commented out as retail part removed from calculation
                    // $voucher  = $voucherData->get($key);

                    // Revenue calculation - retail voucher amount removed
                    $paidAmount    = $invoice ? (float)$invoice->revenue : 0;
                    // $voucherAmount = $voucher ? (float)$voucher->voucher_amount : 0;
                    // $revenue       = $paidAmount + $voucherAmount; // Old calculation with retail
                    $revenue = $paidAmount; // New calculation without retail

                    $performanceData[] = [
                        'date'            => $date->format('M Y'),
                        'date_full'       => $key,
                        'visits_count'    => (int)($services->customers_count ?? 0),
                        'services_count'  => (int)($services->services_count ?? 0),
                        'revenue'         => round($revenue, 2),
                        'balances'        => round($invoice ? (float)$invoice->receivable : 0, 2),
                        'advances'        => round($advance ? (float)$advance->advance_amount : 0, 2),
                    ];
                }
            } else {
                $period = CarbonPeriod::create($from, $to);
                foreach ($period as $date) {
                    $key = $date->format('Y-m-d');
                    $invoice  = $invoiceData->get($key);
                    $advance  = $advanceData->get($key);
                    $services = $servicesData->get($key);
                    // Retail voucher data - commented out as retail part removed from calculation
                    // $voucher  = $voucherData->get($key);

                    // Revenue calculation - retail voucher amount removed
                    $paidAmount    = $invoice ? (float)$invoice->revenue : 0;
                    // $voucherAmount = $voucher ? (float)$voucher->voucher_amount : 0;
                    // $revenue       = $paidAmount + $voucherAmount; // Old calculation with retail
                    $revenue = $paidAmount; // New calculation without retail

                    $performanceData[] = [
                        'date'            => $date->format('D jS M'),
                        'date_full'       => $key,
                        'visits_count'    => (int)($services->customers_count ?? 0),
                        'services_count'  => (int)($services->services_count ?? 0),
                        'revenue'         => round($revenue, 2),
                        'balances'        => round($invoice ? (float)$invoice->receivable : 0, 2),
                        'advances'        => round($advance ? (float)$advance->advance_amount : 0, 2),
                    ];
                }
            }

            $data['performanceData'] = $performanceData;
            $data['from'] = $from->format('d-M-Y');
            $data['to']   = $to->format('d-M-Y');

            return response()->json($data);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error loading monthly performance',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today's overview stats
     */
    public function getTodayOverview(Request $request)
    {
        $businessId = $request->input('business_id', session('business_id'));
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        $today = date('Y-m-d');

        $todayAppointments = CustomerVisits::where('business_id', $businessId)
            ->whereDate('customer_visit_date', $today)
            ->whereIn('visit_status', ['open', 'invoiced'])
            ->count();

        $todayOpenVisits = CustomerVisits::where('business_id', $businessId)
            ->whereDate('customer_visit_date', $today)
            ->where('visit_status', 'open')
            ->count();

        $todayInvoicedVisits = CustomerVisits::where('business_id', $businessId)
            ->whereDate('customer_visit_date', $today)
            ->where('visit_status', 'invoiced')
            ->count();

        $todayRevenue = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            ->whereDate('invoice_date', $today)
            ->sum('gross_wo_tax');

        $todayPaid = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            ->whereDate('invoice_date', $today)
            ->sum('paid_amount');

        $todayBalance = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            ->whereDate('invoice_date', $today)
            ->sum('balance');

        $pendingInvoicesCount = Invoice::where('business_id', $businessId)
            ->where('invoice_status', 'valid')
            ->whereDate('invoice_date', $today)
            ->where('balance', '>', 0)
            ->count();

        return response()->json([
            'today_appointments' => $todayAppointments,
            'today_open_visits' => $todayOpenVisits,
            'today_invoiced_visits' => $todayInvoicedVisits,
            'today_revenue' => round($todayRevenue, 2),
            'today_paid' => round($todayPaid, 2),
            'today_balance' => round($todayBalance, 2),
            'pending_invoices_count' => $pendingInvoicesCount
        ]);
    }

    /**
     * Get top 10 service/package categories for current month
     */
    public function getTopServiceCategories(Request $request)
    {   
        $from =  $request->from;
        $to   =  $request->to;

        $businessId = $request->input('business_id');
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        // Get start of current month (first day)
        $currentMonthStart = date('Y-m-01');
        // Get current date
        $currentDate = date('Y-m-d');
        try {
            // Get service categories
            $serviceCategories = DB::table('visit_services as vs')
                ->select(
                    DB::raw("COALESCE(sc.service_category, 'Uncategorized') as category_name"),
                    DB::raw("'Service' as category_type"),
                    #DB::raw("COALESCE(SUM(vs.s_rate), 0) as total_revenue"),
                    DB::raw("
                        COALESCE(
                            SUM(
                                (COALESCE(invoice_details.price, 0) 
                                - COALESCE(invoice_details.discount, 0)) 
                                + COALESCE(invoice_details.service_addition, 0)
                            ), 
                        0) as total_revenue
                    ")

                )
                ->join('customer_visits as cv', 'vs.customer_visit_id', '=', 'cv.id_customer_visits')
                ->leftJoin('business_services as bs', function($join) use ($businessId) {
                    $join->on('vs.service_id', '=', 'bs.id_business_services')
                         ->where('bs.business_id', '=', $businessId);
                })
                ->leftJoin('service_category as sc', 'bs.service_category_id', '=', 'sc.id_service_category')
                ->join('invoice_details','invoice_details.visit_service_id' ,'vs.id_visit_services')
                ->join('invoice','invoice.id_invoice','invoice_details.invoice_id')
                ->where('cv.business_id', $businessId)
                ->where('cv.visit_status', 'invoiced')
                ->where(DB::raw('LOWER(invoice.invoice_status)'), '!=','cancelled')
                ->where('vs.visit_service_status', 'Active')
                ->whereNotNull('vs.visit_service_start')
                ->where(DB::raw('LOWER(invoice.invoice_status)'),'!=','cancelled')
                ->whereBetween(DB::raw('DATE(invoice.invoice_date)'), [$from, $to])
                ->groupBy('sc.service_category')
                ->get();

            // Get package categories (if they exist in visit_services)
            // Note: Package categories might need different join logic based on your schema
            // For now, we'll focus on service categories

            // Combine and get top 10
            $combined = $serviceCategories->map(function($item) {
                return [
                    'category_name' => $item->category_name ?: 'Uncategorized',
                    'category_type' => $item->category_type,
                    'total_revenue' => (float)$item->total_revenue
                ];
            })->sortByDesc('total_revenue')->take(10)->values();

            $data['combined'] = $combined;
            $data['from'] = Carbon::parse($from)->format('d-M-Y');
            $data['to'] = Carbon::parse($to)->format('d-M-Y');

            return response()->json($data);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading categories: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get top 20 performing staff members (current month)
     */
    public function getTopStaff(Request $request)
    {   
        $from =  $request->from;
        $to   =  $request->to;
        $businessId = $request->input('business_id');

        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        $currentMonthStart = date('Y-m-01');
        $currentDate = date('Y-m-d');

        try {
            // Service invoices only - retail removed
            $staff = DB::table('invoice_staff as isf')
                ->select(
                    's.id_staff',
                    DB::raw("COALESCE(s.staff_fullname, 'Unknown Staff') as staff_fullname"),
                    DB::raw("COALESCE(s.staff_image, '') as staff_image"),
                    DB::raw("COALESCE(SUM(isf.final_price), 0) as total_earnings"),
                    DB::raw("COUNT(isf.id_invoice_staff) as services_count")
                )
                ->join('staff as s', 'isf.staff_id', '=', 's.id_staff')
                ->join('invoice as i', 'isf.invoice_id', '=', 'i.id_invoice')
                ->where('s.business_id', $businessId)
                ->where('i.business_id', $businessId)
                ->where('i.invoice_status', 'valid')
                ->where('i.invoice_type', 'service') // Filter to service invoices only - retail removed
                ->whereBetween(DB::raw('DATE(i.invoice_date)'), [$from, $to])
                ->whereNotNull('isf.final_price')
                ->groupBy('s.id_staff', 's.staff_fullname', 's.staff_image')
                ->orderBy('total_earnings', 'DESC')
                ->limit(12)
                ->get();

            $data['staff'] = $staff;
            $data['from'] = Carbon::parse($from)->format('d-M-Y');
            $data['to'] = Carbon::parse($to)->format('d-M-Y');

            return response()->json($data);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading staff: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get monthly sales from business_service_sales table
     */
    public function getMonthlySales(Request $request)
    {
        $businessId = $request->input('business_id');
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        try {
            // Yearly Service Comparison - ignores selected date range
            // Uses all available data from business_account_monthly_summary table
            // Service-only query - using service_sale fields only (retail removed)
            // This matches the service-only approach used in other dashboard functions
            $sales = DB::table('business_account_monthly_summary')
            ->select(
                DB::raw('month_name AS month'), 
                'year',
                'month_num',
                DB::raw('
                    IFNULL(service_sale,0) 
                  + IFNULL(gift_vocuher_sale,0) 
                  + IFNULL(service_sale_discount,0) 
                  AS total_sale
                ')
            )
            ->where('business_id', $businessId)
            ->where('account_main', 'Revenue')
            ->orderBy('year')
            ->orderBy('month_num')
            ->get();


            // If no data, return empty array
            if ($sales->isEmpty()) {
                return response()->json([]);
            }

            // Create a map of existing sales data
            // Map month names to abbreviated format for consistent key matching
            $monthNameMap = [
                'January' => 'Jan', 'February' => 'Feb', 'March' => 'Mar', 'April' => 'Apr',
                'May' => 'May', 'June' => 'Jun', 'July' => 'Jul', 'August' => 'Aug',
                'September' => 'Sep', 'October' => 'Oct', 'November' => 'Nov', 'December' => 'Dec'
            ];
            
            $salesMap = [];
            $yearsSet = [];
            
            foreach ($sales as $item) {
                // Convert month name to abbreviated format if needed
                $monthAbbr = $monthNameMap[$item->month] ?? $item->month;
                $key = $monthAbbr . '_' . $item->year;
                $salesMap[$key] = (int)$item->total_sale;
                $yearsSet[] = (int)$item->year;
            }

            // Find min and max year from unique years
            $uniqueYears = array_unique($yearsSet);
            if (empty($uniqueYears)) {
                return response()->json([]);
            }
            
            $minYear = min($uniqueYears);
            $maxYear = max($uniqueYears);
            
            // Validate years
            if (!$minYear || !$maxYear || $minYear <= 0 || $maxYear <= 0) {
                return response()->json([]);
            }

            // Generate all months from min year to max year
            $months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
            $formatted = [];
            
            for ($year = $minYear; $year <= $maxYear; $year++) {
                foreach ($months as $month) {
                    $key = $month . '_' . $year;
                    $formatted[] = [
                        'month_year' => $month . ' ' . $year,
                        'month' => $month,
                        'year' => $year,
                        'total_sale' => isset($salesMap[$key]) ? (int)$salesMap[$key] : 0
                    ];
                }
            }

            return response()->json($formatted);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading monthly sales: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get footfall heatmap data (13 months, 31 days per month)
     * Counts invoiced visits per day from 13 months ago to now
     * Always uses past 13 months from current date - ignores selected date range
     */
    public function getFootfallHeatmap(Request $request)
    {   
        $businessId = $request->input('business_id');
        
        if (!$businessId) {
            return response()->json(['error' => 'Business ID required'], 400);
        }

        // Always use past 13 months from current date - ignore selected date range
        // Example: If current date is Feb 2026 → data from Feb 2025 to Feb 2026 (13 months inclusive)
        $currentDate = Carbon::now();
        $thirteenMonthsAgo = Carbon::now()->subMonths(12)->startOfMonth(); // 12 months back = 13th month (inclusive)
        
        // Calculate display range: from 13 months ago to end of current month
        $displayFrom = $thirteenMonthsAgo;
        $displayTo = $currentDate->copy()->endOfMonth(); // Include full current month
        
        $totalMonths = 13; // Always 13 months

        try {
            // Get all invoiced visits in the date range
            $visits = DB::table('customer_visits')
                ->select(
                    DB::raw('DATE(customer_visit_date) as visit_day'),
                    DB::raw('COUNT(*) as visit_count')
                )
                ->where('business_id', $businessId)
                ->where('visit_status', 'invoiced')
                ->whereBetween(DB::raw('DATE(customer_visit_date)'), [$displayFrom, $displayTo])
                ->groupBy(DB::raw('DATE(customer_visit_date)'))
                ->get()
                ->keyBy('visit_day');

            // Generate data structure for heatmap
            // Y-axis: 13 months (most recent first)
            // X-axis: 31 days

            $series = [];
            $monthNames = [];

            // Generate month names for all 13 months
            $temp = $displayFrom->copy();
            for ($i = 0; $i < $totalMonths; $i++) {
                $monthNames[] = $temp->format('M'); // Jan, Feb, etc
                $temp->addMonth();
            }
            
            // Use displayFrom as base for month calculations
            $baseDate = $displayFrom->copy();
            
            // Start from 13 months ago and work forward to current month
            for ($monthOffset = 0; $monthOffset < $totalMonths; $monthOffset++) {
                $monthDate = $baseDate->copy()->addMonths($monthOffset);
                $year = $monthDate->year;
                $monthNum = $monthDate->month;
                $monthName = $monthNames[$monthOffset];
                $daysInMonth = $monthDate->daysInMonth;

                $monthData = [];

                for ($day = 1; $day <= 31; $day++) {
                    if ($day > $daysInMonth) {
                        continue;
                    }

                    $dayDate = $monthDate->copy()->day($day)->format('Y-m-d');
                    $visitCount = isset($visits[$dayDate]) ? (int)$visits[$dayDate]->visit_count : 0;

                    $monthData[] = [
                        'x' => (string)$day,
                        'y' => $visitCount
                    ];
                }

                $series[] = [
                    'name' => $monthName . ' ' . $year,
                    'data' => $monthData
                ];
            }

            $data['series'] = $series;
            $data['from'] = $displayFrom->format('d-M-Y');
            $data['to'] = $displayTo->format('d-M-Y');
            return response()->json($data);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error loading footfall data: ' . $e->getMessage()], 500);
        }
    }

    public function getMonthlyServicePaymentSummary(Request $request)
    {
        try {
            $businessId = $request->business_id;
            $from = $request->from;
            $to = $request->to;

            if (!$businessId) {
                return response()->json(['error' => 'Business ID required'], 400);
            }

            if(session('ho_accounts') == "Yes") {
                $common_business_id = DB::table('business')->where('ho', 'Yes')->first()->id_business;
            } else {
                $common_business_id = session('business_id');
            }

            $service_bank_cash_online_card = DB::table('account_main as m')
                ->join('account_control as c', 'c.account_main_id', '=', 'm.id_account_main')
                ->join('account_sub_control as s', 's.account_control_id', '=', 'c.id_account_control')
                ->join('account_heads as h', 'h.account_sub_control_id', '=', 's.id_account_sub_control')
                ->join('account_voucher_detail as d', 'd.account_head_id', '=', 'h.id_account_heads')
                ->join('account_vouchers as v', 'v.id_account_vouchers', '=', 'd.account_voucher_id')
                ->join('invoice', 'invoice.id_invoice', '=', 'v.invoice_id')
                ->where('v.voucher_status', 'Active')
                ->whereBetween(DB::raw('DATE(v.voucher_date)'), [$from, $to])
                ->where('v.voucher_type', 2)
                ->whereIn('s.account_sub_control', ['Cash', 'Banks', 'Online', 'Card'])
                ->where('v.business_id', $businessId)
                ->where('h.business_id', $common_business_id)
                ->where('invoice.invoice_status', 'valid')
                ->where('invoice.invoice_type', 'service')
                ->selectRaw("
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit ELSE 0 END), 2) AS cash_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.credit ELSE 0 END), 2) AS cash_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit - d.credit ELSE 0 END), 2) AS cash_balance,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit ELSE 0 END), 2) AS bank_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.credit ELSE 0 END), 2) AS bank_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit - d.credit ELSE 0 END), 2) AS bank_balance,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit ELSE 0 END), 2) AS online_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.credit ELSE 0 END), 2) AS online_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit - d.credit ELSE 0 END), 2) AS online_balance,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit ELSE 0 END), 2) AS card_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.credit ELSE 0 END), 2) AS card_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit - d.credit ELSE 0 END), 2) AS card_balance
                ")
                ->get();

            $totals = [
                'cash_debit'    => $service_bank_cash_online_card->sum('cash_debit'),
                'cash_credit'   => $service_bank_cash_online_card->sum('cash_credit'),
                'cash_balance'  => $service_bank_cash_online_card->sum('cash_balance'),

                'bank_debit'    => $service_bank_cash_online_card->sum('bank_debit'),
                'bank_credit'   => $service_bank_cash_online_card->sum('bank_credit'),
                'bank_balance'  => $service_bank_cash_online_card->sum('bank_balance'),

                'online_debit'  => $service_bank_cash_online_card->sum('online_debit'),
                'online_credit' => $service_bank_cash_online_card->sum('online_credit'),
                'online_balance'=> $service_bank_cash_online_card->sum('online_balance'),

                'card_debit'    => $service_bank_cash_online_card->sum('card_debit'),
                'card_credit'   => $service_bank_cash_online_card->sum('card_credit'),
                'card_balance'  => $service_bank_cash_online_card->sum('card_balance'),
            ]; 

            $fromFormatted = Carbon::parse($from)->format('d-M-Y');
            $toFormatted   = Carbon::parse($to)->format('d-M-Y');
            
            return response()->json([
                'success' => true,
                'from' => $fromFormatted,
                'to' => $toFormatted,
                'data' => $totals
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
    
    // ------------------------> Retail Dashboard Start <----------------------------
    function retaildashboard(Request $request){
        $branches = DB::table('business')->select('id_business','business_name')->where('ho_accounts','Yes')->orwhere('ho','Yes')->get();
        return view('dashboards.retaildashboard',compact('branches'));
    }

    function product_categories_stock_status(Request $request){
       $business_id  = $request->business_id; 
       $where =''; 
    //    if(session('ho') =='No'){
        $where." where business_id = $business_id ";
    //    }

        $sql =" SELECT
                    store_id,
                    store,
                    business_name,
                    category,
                    product_count,
                    addition_qty,
                    purchased_qty,
                    transfer_in_qty,
                    transfer_out_qty,
                    sold_qty,
                    used_qty,
                    returned_qty,
                    franchise_sale_qty,
                    instock,
                    total_stock_value,

                    ROUND(
                        (instock / NULLIF(SUM(instock) OVER (), 0)) * 100,
                        2
                    ) AS stock_qty_percentage,

                    ROUND(
                        (total_stock_value / NULLIF(SUM(total_stock_value) OVER (), 0)) * 100,
                        2
                    ) AS stock_value_percentage

                FROM (
                    SELECT
                        store_id,
                        store,
                        business_name,
                        CASE
                            WHEN category IS NULL OR TRIM(category) = ''
                            THEN 'Uncategorized'
                            ELSE category
                        END AS category,

                        COUNT(DISTINCT product_id) AS product_count,

                        SUM(addition_qty) AS addition_qty,
                        SUM(purchased_qty) AS purchased_qty,
                        SUM(transfer_in_qty) AS transfer_in_qty,
                        SUM(transfer_out_qty) AS transfer_out_qty,
                        SUM(sold_qty) AS sold_qty,
                        SUM(used_qty) AS used_qty,
                        SUM(returned_qty) AS returned_qty,
                        SUM(franchise_sale_qty) AS franchise_sale_qty,

                        SUM(instock) AS instock,
                        SUM(total_value) AS total_stock_value

                    FROM mis_store_product_stock

                    $where 

                    GROUP BY
                        store_id,
                        CASE
                            WHEN category IS NULL OR TRIM(category) = ''
                            THEN 'Uncategorized'
                            ELSE category
                        END
                ) t
                where instock > 0
                ORDER BY store_id,  total_stock_value DESC;";
        $categories_data = DB::select($sql);

        $lastUpdatedAt = DB::table('mis_store_product_stock')
        ->selectRaw("DATE_FORMAT(MAX(updated_at), '%d-%b-%Y, %h:%i %p') as last_updated_at")
        ->value('last_updated_at') ?? '';

        if($categories_data){
             return response()->json([
                'success' => true,
                'data' => $categories_data,
                'lastUpdatedAt' => $lastUpdatedAt,
                'message_type' => 'error',
                'message' => '',
                'message_btn' => 'btn btn-danger',
            ]);
        }else{
             return response()->json([
                'success' => false,
                'data' => [],
                'lastUpdatedAt' => '',
                'message_type' => 'error',
                'message' => 'data not found',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    } 

    public function product_categories_sales_perc(Request $request)
    {
        $ProductsStockService = new ProductsStockService();
        $business_id =$request->business_id; 
     
        $from = $request->input('from');
        $to = $request->input('to');
        
 
        if (empty($from) || empty($to)) {
            $from = Carbon::now()->startOfMonth()->toDateString();
            $to   = Carbon::now()->endOfMonth()->toDateString();
        } else {
         
            try {
                $from = Carbon::parse($from)->toDateString();
                $to = Carbon::parse($to)->toDateString();
            } catch (\Exception $e) {
                $from = Carbon::now()->startOfMonth()->toDateString();
                $to   = Carbon::now()->endOfMonth()->toDateString();
            }
        }

       
        $stores = DB::table('business_stores')
                    ->select('id_business_stores', 'business_id', 'business_store');
        if (session('ho') == 'No') {
            $stores->where('business_id', $business_id);
        }
        $stores = $stores->get();

        $finalData = collect();

        foreach ($stores as $store) {
         
            $productsData = $ProductsStockService
                ->getInvoiceSaleAndFranchiseSaleQty(
                    $store->id_business_stores,
                    $store->business_id,
                    $from,  
                    $to     
                );


            $productsData = $productsData->map(function ($item) {
                $item['category'] = (empty($item['category']) || trim($item['category']) === '')
                    ? 'Uncategorized'
                    : trim($item['category']);
                $item['total_sale_qty'] = $item['sold_qty'] + $item['franchise_sale_qty'];
                return $item;
            });

            $finalData = $finalData->merge($productsData);
        }


        $combinedData = $finalData
            ->groupBy('category')
            ->map(function ($items, $category) {
                return [
                    'category'           => $category,
                    'sold_qty'           => $items->sum('sold_qty'),
                    'franchise_sale_qty' => $items->sum('franchise_sale_qty'),
                    'total_sale_qty'     => $items->sum('total_sale_qty'),
                ];
            })
            ->filter(function ($item) {
                return $item['total_sale_qty'] > 0;
            })
            ->values();


        $fromFormatted = Carbon::parse($from)->format('d-M-Y');
        $toFormatted   = Carbon::parse($to)->format('d-M-Y');

        return response()->json([
            'success' => true,
            'data'    => $combinedData,
            'from'    => $fromFormatted,
            'to'      => $toFormatted,
            'raw_from' => $from,
            'raw_to'   => $to,  
        ]);
    }
    
    public function getting_top_retail_customers(Request $request)
    {
        $businessId = $request->business_id;
        
        $from = $request->input('from');
        $to   = $request->input('to');

        if (empty($from) || empty($to)) {
            $from = Carbon::now()->subMonths(2)->startOfMonth()->toDateString();
            $to   = Carbon::now()->endOfMonth()->toDateString();
        } else {
            try {
                $from = Carbon::parse($from)->toDateString();
                $to   = Carbon::parse($to)->toDateString();
                if ($from > $to) {
                    [$from, $to] = [$to, $from];
                }
            } catch (\Exception $e) {
                $from = Carbon::now()->subMonths(2)->startOfMonth()->toDateString();
                $to   = Carbon::now()->endOfMonth()->toDateString();
            }
        }

        // ===== Sales & Discount Account =====
        $salesAccount = AccountsController::get_event_account('1', 'grosstotal_retail');
        $discountAccount = AccountsController::get_event_account('1', 'sale_discount_retail');
        if (!$salesAccount || !$discountAccount) {
            return response()->json(['success' => false, 'message' => 'Sales or discount account not found']);
        }

        // ===== Receivable Account =====
        $receivableAccount = AccountsController::get_event_account('1', 'receivable');
        if (!$receivableAccount) {
            return response()->json(['success' => false, 'message' => 'Receivable account not found']);
        }

        $customers = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            // ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->join('customers as c', 'av.business_partner_id', '=', 'c.id_customers')
           
            // ->where('av.voucher_status', 'Active')
            // ->where('i.invoice_status', 'valid')
            // ->where('i.invoice_type', 'sale')
            // ->where('i.business_id', $businessId)
            ->where('av.voucher_status', 'Active')
            ->where('av.business_id', $businessId)
            ->where('av.business_partner', 1) 
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
            ->selectRaw("
                c.id_customers as customer_id,
                c.customer_name,
                c.customer_cell,
                c.customer_email,

                SUM(CASE 
                    WHEN avd.account_head_id IN ({$salesAccount->account_head_id}, {$discountAccount->account_head_id})
                    THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0)
                    ELSE 0 END
                ) AS total_sales,

                SUM(CASE 
                    WHEN avd.account_head_id = {$receivableAccount->account_head_id}
                    THEN IFNULL(avd.debit,0) - IFNULL(avd.credit,0)
                    ELSE 0 END
                ) AS receivable_balance
            ")
            ->groupBy('c.id_customers')
            ->orderByDesc('total_sales')
            ->limit(10)
            ->get();
            
        $fromFormatted = Carbon::parse($from)->format('d-M-Y');
        $toFormatted = Carbon::parse($to)->format('d-M-Y');    

        return response()->json([
            'success' => true,
            'from' => $fromFormatted,
            'to' => $toFormatted,
            'data' => $customers
        ]);
    } 

    public function getRetailCustomerAndFranchiseBalances(Request $request)
    {
        $businessId = $request->business_id;
        $start = $request->input('from');
        $end = $request->input('to');
        

        if (empty($start) || empty($end)) {
            $start = now()->startOfMonth()->toDateString();
            $end   = now()->endOfMonth()->toDateString();
        } else {
           
            try {
                $start = Carbon::parse($start)->toDateString();
                $end = Carbon::parse($end)->toDateString();
                
   
                if ($start > $end) {
                
                    $temp = $start;
                    $start = $end;
                    $end = $temp;
                }
            } catch (\Exception $e) {

                $start = now()->startOfMonth()->toDateString();
                $end   = now()->endOfMonth()->toDateString();
            }
        }

        $account = AccountsController::get_event_account('1', 'balance');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }

        $accountHeadId = $account->account_head_id;

        /*
        |--------------------------------------------------------------------------
        | Retail Customers (Invoices)
        |--------------------------------------------------------------------------
        */
        $retail = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            // ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->join('customers as c', 'av.business_partner_id', '=', 'c.id_customers')
            ->where('avd.account_head_id', $accountHeadId)
            // ->where('av.voucher_status', 'Active')
            // ->where('i.invoice_status', 'valid')
            // ->where('i.invoice_type', 'sale')
            // ->where('i.business_id', $businessId)
            ->where('av.voucher_status', 'Active')
            ->where('av.business_id', $businessId)
            ->where('av.business_partner', 1)
            // ->whereBetween('i.invoice_date', [$start, $end])
            ->whereDate('av.voucher_date', '<=', $end)  
            ->selectRaw('
                SUM(IFNULL(avd.debit,0))  AS total_receivable,
                SUM(IFNULL(avd.credit,0)) AS total_received,
                SUM(IFNULL(avd.debit,0)) - SUM(IFNULL(avd.credit,0)) AS balance
            ')
            ->first();

        /*
        |--------------------------------------------------------------------------
        | Franchise Orders
        |--------------------------------------------------------------------------
        */

        $account = AccountsController::get_event_account('37', 'total_receivable');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }
        $accountHeadId = $account->account_head_id;

        $franchise = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            // ->join('franchise_orders as fo', 'fo.id_franchise_orders', '=', 'av.franchise_order_id')
            ->join('franchises as f', 'f.id_franchises', '=', 'av.business_partner_id')
            // ->where('av.voucher_status', 'Active')
            // ->where('fo.business_id', $businessId)
            ->where('avd.account_head_id', $accountHeadId)
            ->where('av.voucher_status', 'Active')
            ->where('av.business_id', $businessId)
            ->where('av.business_partner', 4)
            // ->whereRaw('LOWER(fo.order_status) != ?', ['cancelled'])
            // ->whereBetween('fo.created_at', [$start, $end]) 
            ->whereDate('av.voucher_date', '<=', $end) 
            ->selectRaw('
                SUM(IFNULL(avd.debit,0))  AS total_receivable,
                SUM(IFNULL(avd.credit,0)) AS total_received,
                SUM(IFNULL(avd.debit,0)) - SUM(IFNULL(avd.credit,0)) AS balance
            ')
            ->first();


        $fromFormatted = Carbon::parse($start)->format('d-M-Y');
        $toFormatted = Carbon::parse($end)->format('d-M-Y');

        return response()->json([
            'success' => true,
            'data' => [
                'retail' => [
                    'total_receivable' => $retail->total_receivable ?? 0,
                    'total_received'   => $retail->total_received ?? 0,
                    'balance'          => $retail->balance ?? 0,
                ],
                'franchise' => [
                    'total_receivable' => $franchise->total_receivable ?? 0,
                    'total_received'   => $franchise->total_received ?? 0,
                    'balance'          => $franchise->balance ?? 0,
                ],
                'from' => $fromFormatted,
                'to'   => $toFormatted,
            ]
        ]);
    }

    public function getRetailCustomerPayableAndFranchisePayable(Request $request)
    {
        $businessId = $request->business_id;
        $start = $request->input('from');
        $end = $request->input('to');

        if (empty($start) || empty($end)) {
            $start = now()->startOfMonth()->toDateString();
            $end   = now()->endOfMonth()->toDateString();
        } else {
         
            try {
                $start = Carbon::parse($start)->toDateString();
                $end = Carbon::parse($end)->toDateString();
                
                
                if ($start > $end) {
                    
                    $temp = $start;
                    $start = $end;
                    $end = $temp;
                }
            } catch (\Exception $e) {
                
                $start = now()->startOfMonth()->toDateString();
                $end   = now()->endOfMonth()->toDateString();
            }
        }

        $account = AccountsController::get_event_account('1', 'advance_adjusted');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }
        
        $accountHeadId = $account->account_head_id;

        /*
        |--------------------------------------------------------------------------
        | Retail Customers (Invoices)
        |--------------------------------------------------------------------------
        */
        $retail = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->where('avd.account_head_id', $accountHeadId)
            ->where('av.voucher_status', 'Active')
            ->where('i.invoice_status', 'valid')
            ->where('i.invoice_type', 'sale')
            ->where('i.business_id', $businessId)
            // ->whereBetween('i.invoice_date', [$start, $end]) 
            ->whereDate('av.voucher_date', '<=', $end) 
            ->selectRaw('
                SUM(IFNULL(avd.credit,0)) AS total_payable,
                SUM(IFNULL(avd.debit,0))  AS total_paid,
                SUM(IFNULL(avd.credit,0)) - SUM(IFNULL(avd.debit,0)) AS balance
            ')
            ->first();

        /*
        |--------------------------------------------------------------------------
        | Franchise Orders
        |--------------------------------------------------------------------------
        */

        $account = AccountsController::get_event_account('37', 'increase_advance_liability');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }
        
        $accountHeadId = $account->account_head_id;
        
        $franchise = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('franchise_orders as fo', 'fo.id_franchise_orders', '=', 'av.franchise_order_id')
            ->where('av.voucher_status', 'Active')
            ->where('avd.account_head_id', $accountHeadId)
            ->where('fo.business_id', $businessId)
            ->whereRaw('LOWER(fo.order_status) != ?', ['cancelled'])
            // ->whereBetween('fo.created_at', [$start, $end])
            ->whereDate('av.voucher_date', '<=', $end) 
            ->selectRaw('
                SUM(IFNULL(avd.credit,0)) AS total_payable,
                SUM(IFNULL(avd.debit,0))  AS total_paid,
                SUM(IFNULL(avd.credit,0)) - SUM(IFNULL(avd.debit,0)) AS balance
            ')
            ->first();


        $fromFormatted = Carbon::parse($start)->format('d-M-Y');
        $toFormatted = Carbon::parse($end)->format('d-M-Y');

        return response()->json([
            'success' => true,
            'data' => [
                'retail' => [
                    'total_payable' => $retail->total_payable ?? 0,
                    'total_paid'    => $retail->total_paid ?? 0,
                    'balance'       => $retail->balance ?? 0,
                ],
                'franchise' => [
                    'total_payable' => $franchise->total_payable ?? 0,
                    'total_paid'    => $franchise->total_paid ?? 0,
                    'balance'       => $franchise->balance ?? 0,
                ],
                'from' => $fromFormatted,
                'to'   => $toFormatted,
            ]
        ]);
    } 

    public function getting_top_franchise(Request $request) {
        $businessId = $request->business_id;

        // Date range: either from request or current month
        $from = $request->from ?? Carbon::now()->startOfMonth()->toDateString();
        $to   = $request->to ?? Carbon::now()->endOfMonth()->toDateString();

        // Get account head for total invoice value
        $account = AccountsController::get_event_account('37', 'total_invoice_value');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }
        $accountHeadId = $account->account_head_id;

        // Fetch top 5 franchises by sales
        $franchises = DB::table('account_vouchers as av')
            ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('franchises as f', 'f.id_franchises', '=', 'av.business_partner_id')
            ->select(
                'f.id_franchises as franchise_id',
                'f.franchise_name as franchise_name',
                DB::raw('IFNULL(SUM(avd.credit), 0) as total_sale')  // debit = total sale
            )
            ->where('av.voucher_status', 'Active')
            ->where('av.business_partner', 4)
            ->where('avd.account_head_id', $accountHeadId)
            ->where('av.business_id', $businessId)
            ->whereBetween('av.voucher_date', [$from, $to])
            ->groupBy('f.id_franchises')
            ->orderByDesc('total_sale')
            ->limit(5)
            ->get();

        // Calculate total of top 5 sales
        $total = $franchises->sum('total_sale');

        // Format data exactly like mData for JS chart
        $mData = $franchises->map(function($item) use ($total) {
            return [
                'name' => $item->franchise_name,
                'x' => (float) $item->total_sale,
                'y' => $total > 0 ? round(($item->total_sale / $total) * 100, 1) : 0
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $mData,
            'total' => (float) $total,
            'from' => Carbon::parse($from)->format('d-M-Y'),
            'to' => Carbon::parse($to)->format('d-M-Y'),
        ]);
    } 


    public function getRetailRevenueByDateRange($start ,$end , $business_id){

        $businessId = $business_id;
        $retailAccount = AccountsController::get_event_account('1', 'grosstotal_retail');
        $discountAccount = AccountsController::get_event_account('1', 'sale_discount_retail');

        if (!$retailAccount || !$discountAccount) {
            return response()->json([
                'success' => false,
                'message' => 'Retail or discount account mapping not found.'
            ]);
        }
        $retailAccountHeadId = $retailAccount->account_head_id;
        $discountAccountHeadId = $discountAccount->account_head_id;

        /*
        |--------------------------------------------------------------------------
        | Retail Customers (Invoices) Revenue
        |--------------------------------------------------------------------------
        */

        $retail = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            // ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->join('customers as c', 'av.business_partner_id', '=', 'c.id_customers')
            // ->where('av.voucher_status', 'Active')
            // ->where('i.invoice_status', 'valid')
            // ->where('i.invoice_type', 'sale')
            // ->where('i.business_id', $businessId)
            ->whereIn('avd.account_head_id', [$retailAccount->account_head_id, $discountAccount->account_head_id])
            ->where('av.voucher_status', 'Active')
            ->where('av.business_id', $businessId)
            ->where('av.business_partner', 1)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$start, $end])
            ->selectRaw('
                SUM(IFNULL(avd.credit,0)) AS credit,
                SUM(IFNULL(avd.debit,0))  AS debit,
                SUM(IFNULL(avd.credit,0)) - SUM(IFNULL(avd.debit,0)) AS balance
            ')
            ->first();

        /*
        |--------------------------------------------------------------------------
        | Franchise Orders Revenue
        |--------------------------------------------------------------------------
        */

        $franchiseAccount = AccountsController::get_event_account('37', 'total_invoice_value');
        if (!$franchiseAccount) {
            return response()->json([
                'success' => false,
                'message' => 'Franchise account mapping not found.'
            ]);
        }
        $franchiseAccountHeadId = $franchiseAccount->account_head_id;
        $franchise = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            // ->join('franchise_orders as fo', 'fo.id_franchise_orders', '=', 'av.franchise_order_id')
            ->join('franchises as f', 'f.id_franchises', '=', 'av.business_partner_id')
            // ->where('av.voucher_status', 'Active')
            // ->where('fo.business_id', $businessId)
            // ->whereRaw('LOWER(fo.order_status) != ?', ['cancelled'])
            ->where('avd.account_head_id', $franchiseAccountHeadId)
            ->where('av.voucher_status', 'Active')
            ->where('av.business_id', $businessId)
            ->where('av.business_partner', 4)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$start, $end])
            ->selectRaw('
                SUM(IFNULL(avd.credit,0)) AS credit,
                SUM(IFNULL(avd.debit,0))  AS debit,
                SUM(IFNULL(avd.credit,0)) - SUM(IFNULL(avd.debit,0)) AS balance
            ')
            ->first();

        return response()->json([
            'success' => true,
            'data' => [
                'retail' => [
                    'credit' => $retail->credit ?? 0,
                    'debit'   => $retail->debit ?? 0,
                    'balance'          => $retail->balance ?? 0,
                ],
                'franchise' => [
                    'credit' => $franchise->credit ?? 0,
                    'debit'   => $franchise->debit ?? 0,
                    'balance'          => $franchise->balance ?? 0,
                ],
                'from' => Carbon::parse($start)->format('d-M-Y'),
                'to'   => Carbon::parse($end)->format('d-M-Y'),
            ]
        ]);

    } 

    public function retail_revenue(Request $request)
    {
        $businessId = $request->business_id;
        

        $selectedFrom = Carbon::parse($request->input('from'));
        $selectedTo = Carbon::parse($request->input('to'));
    
        $days = \Carbon\Carbon::parse($selectedFrom)->diffInDays($selectedTo) + 1;

        // $prevFrom = \Carbon\Carbon::parse($selectedFrom)->subDays($days);
        // $prevTo   = \Carbon\Carbon::parse($selectedFrom)->subDay();
        $months = $selectedFrom->diffInMonths($selectedTo) + 1;
        $prevFrom = $selectedFrom->copy()
            ->subMonths($months)
            ->startOfMonth();

        $prevTo = $selectedFrom->copy()
            ->subMonth()
            ->endOfMonth();

        $current = $this->getRetailRevenueByDateRange($selectedFrom, $selectedTo, $businessId)->getData()->data ?? [];
        

        $last    = $this->getRetailRevenueByDateRange($prevFrom, $prevTo, $businessId)->getData()->data ?? [];

        $currentTotal = ($current->retail->balance ?? 0) + ($current->franchise->balance ?? 0);
        $lastTotal    = ($last->retail->balance ?? 0) + ($last->franchise->balance ?? 0);

        $totalPerc = $lastTotal != 0
            ? (($currentTotal - $lastTotal) / $lastTotal) * 100
            : ($currentTotal > 0 ? 100 : 0);

        return response()->json([
            'success' => true,
            'data' => [
                'current_month' => [
                    'retail'    => $current->retail ?? 0,
                    'franchise' => $current->franchise ?? 0,
                    'total'     => $currentTotal ?? 0,
                ],
                'last_month' => [
                    'retail'    => $last->retail ?? 0,
                    'franchise' => $last->franchise ?? 0,
                    'total'     => $lastTotal ?? 0,
                ],
                'percentage_change' => round($totalPerc, 2)
            
            ],
            'from' => Carbon::parse($selectedFrom)->format('d-M-Y'),
            'to'   => Carbon::parse($selectedTo)->format('d-M-Y'),
            'previous_from' => Carbon::parse($prevFrom)->format('M'),
            'previous_to'   => Carbon::parse($prevTo)->format('M'),
        ]);
    }

    public function accountsPayable(Request $request){
        $businessId = $request->business_id;
        $from = $request->from ?? Carbon::now()->startOfMonth()->toDateString();
        $to   = $request->to ?? Carbon::now()->endOfMonth()->toDateString();
        
        $account = AccountsController::get_event_account('20', 'payables');
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account mapping not found.'
            ]);
        }

        $accountHeadId = $account->account_head_id;

        $accounts_payable = DB::table('account_voucher_detail as avd')
        ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
        ->where('av.voucher_status', 'Active')
        ->where('avd.account_head_id', $accountHeadId)
        ->where('av.business_id', $businessId)
        ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
        ->selectRaw('
            SUM(IFNULL(avd.credit,0)) AS total_payable,
            SUM(IFNULL(avd.debit,0))  AS total_paid,
            SUM(IFNULL(avd.credit,0)) - SUM(IFNULL(avd.debit,0)) AS balance
        ')
        ->first();

        return response()->json([
            'success' => true,
            'data' => [
                'payable' => $accounts_payable->total_payable ?? 0,
                'paid'   => $accounts_payable->total_paid ?? 0,
                'balance' => $accounts_payable->balance ?? 0,
                'from' => Carbon::parse($from)->format('d-M-Y'),
                'to'   => Carbon::parse($to)->format('d-M-Y'),
            ]
        ]);
    } 


    public function top_product_sales_perc(Request $request)
    {   
        $business_id = $request->business_id; 
        $ProductsStockService = new ProductsStockService();

        $from = $request->input('from');
        $to = $request->input('to');
        

        if (empty($from) || empty($to)) {
            $from = Carbon::now()->startOfMonth()->toDateString();
            $to   = Carbon::now()->endOfMonth()->toDateString();
        } else {

            try {
                $from = Carbon::parse($from)->toDateString();
                $to = Carbon::parse($to)->toDateString();
                

                if ($from > $to) {

                    $temp = $from;
                    $from = $to;
                    $to = $temp;
                }
            } catch (\Exception $e) {

                $from = Carbon::now()->startOfMonth()->toDateString();
                $to   = Carbon::now()->endOfMonth()->toDateString();
            }
        }


        $stores = DB::table('business_stores')->select('id_business_stores', 'business_id', 'business_store');
        $stores->where('business_id', $business_id);
        

        $stores = $stores->get();

        $finalData = collect();

        foreach ($stores as $store) {

            $productsData = $ProductsStockService
                ->getInvoiceSaleAndFranchiseSaleQty(
                    $store->id_business_stores,
                    $store->business_id,
                    $from, 
                    $to    
                );

            $productsData = $productsData->map(function ($item) {
                $item['total_sale_qty'] = $item['sold_qty'] + $item['franchise_sale_qty'];
                return $item;
            });

            $groupedProducts = $productsData
                ->groupBy('product')
                ->map(function ($items, $product) use ($store) {
                    return [
                        'product'            => $product,
                        'store_id'           => $store->id_business_stores,
                        'store'              => $store->business_store,
                        'sold_qty'           => $items->sum('sold_qty'),
                        'franchise_sale_qty' => $items->sum('franchise_sale_qty'),
                        'total_sale_qty'     => $items->sum('total_sale_qty'),
                    ];
                })
                ->values();

            $finalData = $finalData->merge($groupedProducts);
        }

    
        $combinedData = $finalData
            ->groupBy('product')
            ->map(function ($items, $product) {
                return [
                    'product'            => $product,
                    'sold_qty'           => $items->sum('sold_qty'),
                    'franchise_sale_qty' => $items->sum('franchise_sale_qty'),
                    'total_sale_qty'     => $items->sum('total_sale_qty'),
                ];
            })
            ->values();

        $topProducts = $combinedData
            ->sortByDesc('total_sale_qty')
            ->take(5)
            ->values();

        return response()->json([
            'success' => true,
            'data'    => $topProducts,
            'from'    => Carbon::parse($from)->format('d-M-Y'),
            'to'      => Carbon::parse($to)->format('d-M-Y'),
        ]);
    }

    public function getMonthlyRetailPaymentSummary(Request $request)
    {
        try {

        $businessId = $request->business_id;
        // Current month date range
        $from = $request->input('from');
        $to = $request->input('to');

    
        if (empty($from) || empty($to)) {
            $from = Carbon::now()->startOfMonth()->toDateString();
            $to   = Carbon::now()->endOfMonth()->toDateString();
        } else {

            try {
                $from = Carbon::parse($from)->toDateString();
                $to = Carbon::parse($to)->toDateString();
                

                if ($from > $to) {

                    $temp = $from;
                    $from = $to;
                    $to = $temp;
                }
            } catch (\Exception $e) {

                $from = Carbon::now()->startOfMonth()->toDateString();
                $to   = Carbon::now()->endOfMonth()->toDateString();
            }
        }

            if(session('ho_accounts') == "Yes") {
                $common_business_id = DB::table('business')->where('ho', 'Yes')->first()->id_business;
            } else {
                $common_business_id = session('business_id');
            }

            $sql ="SELECT 
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit ELSE 0 END), 2) AS cash_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.credit ELSE 0 END), 2) AS cash_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit - d.credit ELSE 0 END), 2) AS cash_balance,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit ELSE 0 END), 2) AS bank_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.credit ELSE 0 END), 2) AS bank_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit - d.credit ELSE 0 END), 2) AS bank_balance ,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit ELSE 0 END), 2) AS online_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.credit ELSE 0 END), 2) AS online_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit - d.credit ELSE 0 END), 2) AS online_balance,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit ELSE 0 END), 2) AS card_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.credit ELSE 0 END), 2) AS card_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit - d.credit ELSE 0 END), 2) AS card_balance

                FROM account_main AS m
                JOIN account_control AS c ON c.account_main_id = m.id_account_main
                JOIN account_sub_control AS s ON s.account_control_id = c.id_account_control
                JOIN account_heads AS h ON h.account_sub_control_id = s.id_account_sub_control
                JOIN account_voucher_detail AS d ON d.account_head_id = h.id_account_heads
                JOIN account_vouchers AS v ON v.id_account_vouchers = d.account_voucher_id
                JOIN franchise_orders ON franchise_orders.id_franchise_orders = v.franchise_order_id
                WHERE v.voucher_status = 'Active'
                AND DATE(v.voucher_date) BETWEEN '$from' AND '$to'
                AND v.voucher_type = 2
                AND s.account_sub_control IN ('Cash', 'Banks' ,'Online' ,'Card')  
                AND v.business_id = $businessId
                AND h.business_id = $common_business_id;
            ";
            $franchise_bank_cash_online_card= DB::select($sql); 

            $sql ="SELECT 
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit ELSE 0 END), 2) AS cash_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.credit ELSE 0 END), 2) AS cash_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Cash' THEN d.debit - d.credit ELSE 0 END), 2) AS cash_balance,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit ELSE 0 END), 2) AS bank_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.credit ELSE 0 END), 2) AS bank_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Banks' THEN d.debit - d.credit ELSE 0 END), 2) AS bank_balance ,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit ELSE 0 END), 2) AS online_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.credit ELSE 0 END), 2) AS online_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Online' THEN d.debit - d.credit ELSE 0 END), 2) AS online_balance,

                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit ELSE 0 END), 2) AS card_debit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.credit ELSE 0 END), 2) AS card_credit,
                    ROUND(SUM(CASE WHEN s.account_sub_control = 'Card' THEN d.debit - d.credit ELSE 0 END), 2) AS card_balance

                FROM account_main AS m
                JOIN account_control AS c ON c.account_main_id = m.id_account_main
                JOIN account_sub_control AS s ON s.account_control_id = c.id_account_control
                JOIN account_heads AS h ON h.account_sub_control_id = s.id_account_sub_control
                JOIN account_voucher_detail AS d ON d.account_head_id = h.id_account_heads
                JOIN account_vouchers AS v ON v.id_account_vouchers = d.account_voucher_id
                JOIN invoice ON invoice.id_invoice = v.invoice_id
                WHERE v.voucher_status = 'Active'
                AND DATE(v.voucher_date) BETWEEN '$from' AND '$to'
                AND v.voucher_type = 2
                AND s.account_sub_control IN ('Cash', 'Banks' ,'Online' ,'Card')  
                AND v.business_id = $businessId
                AND h.business_id = $common_business_id 
                AND invoice.invoice_status = 'valid'
                AND invoice.invoice_type = 'sale';
            ";
             $invoice_bank_cash_online_card = DB::select($sql);

            $merged = collect($franchise_bank_cash_online_card)
             ->merge($invoice_bank_cash_online_card);

            $totals = [
                'cash_debit'    => $merged->sum('cash_debit'),
                'cash_credit'   => $merged->sum('cash_credit'),
                'cash_balance'  => $merged->sum('cash_balance'),

                'bank_debit'    => $merged->sum('bank_debit'),
                'bank_credit'   => $merged->sum('bank_credit'),
                'bank_balance'  => $merged->sum('bank_balance'),

                'online_debit'  => $merged->sum('online_debit'),
                'online_credit' => $merged->sum('online_credit'),
                'online_balance'=> $merged->sum('online_balance'),

                'card_debit'    => $merged->sum('card_debit'),
                'card_credit'   => $merged->sum('card_credit'),
                'card_balance'  => $merged->sum('card_balance'),
            ]; 

            $from = Carbon::parse($from)->format('d-M-Y');
            $to   = Carbon::parse($to)->format('d-M-Y');
            
            return response()->json([
                'success' => true,
                'from'=>$from,
                'to'=>$to,
                'data'    => $totals
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function purchase_order_and_grn_detail_with_separte_date_for_po_grn_rn(Request $request)
    {
        $businessId = $request->business_id;
        $start = $request->from ?? Carbon::now()->startOfMonth()->toDateString();
        $end   = $request->to ?? Carbon::now()->endOfMonth()->toDateString();

        // Purchase Order
        $purchase_order = DB::select("
            SELECT
                SUM(IFNULL(pod.product_qty, 0)) AS purchase_qty,
                SUM(pod.product_qty * pod.product_purchase_price) AS total_purchase_amount
            FROM purchase_order po
            JOIN purchase_order_details pod 
                ON pod.purchase_order_id = po.idpurchase_order
            WHERE po.business_id = $businessId
            AND po.status != 'Cancelled'
            AND DATE(po.order_date) BETWEEN '$start' AND '$end'
        ");

        // GRN
        $grn_details = DB::select("
            SELECT 
                SUM(gd.grn_qty_received) AS total_grn_qty,
                ROUND(SUM(gd.grn_qty_received * gd.grn_unit_price), 2) AS total_grn_amount
            FROM goods_received_note grn
            JOIN grn_details gd 
                ON gd.grn_id = grn.grn_id
            WHERE grn.business_id = $businessId
            AND DATE(grn.grn_created_date) BETWEEN '$start' AND '$end'
        ");

        // Return Notes
        $return_notes = DB::select("
            SELECT 
                SUM(rn.return_qty) AS total_return_qty,
                ROUND(SUM(rn.return_qty * gd.grn_unit_price), 2) AS total_return_amount
            FROM return_notes rn
            JOIN grn_details gd 
                ON gd.grn_product_id = rn.product_id
            JOIN goods_received_note grn 
                ON grn.grn_id = rn.grn_id
            WHERE rn.return_status = 'active'
            AND grn.business_id = $businessId
            AND DATE(rn.return_date) BETWEEN '$start' AND '$end'
        ");
        
        $from = Carbon::parse($start)->format('d-M-Y');
        $to   = Carbon::parse($end)->format('d-M-Y');
        return response()->json([
            'success' => true,
            'data' => [
                'purchase' => $purchase_order[0] ?? [],
                'grn'      => $grn_details[0] ?? [],
                'return'   => $return_notes[0] ?? []
            ],
            'from'=>$from,
            'to'=>$to,
        ]);
    }
    public function purchase_order_and_grn_detail(Request $request)
    {
        $businessId = $request->business_id;
        $start = $request->from ?? Carbon::now()->startOfMonth()->toDateString();
        $end   = $request->to ?? Carbon::now()->endOfMonth()->toDateString();

        $data = DB::select("
                SELECT
                    SUM(t.purchase_qty) AS purchase_qty,
                    SUM(t.total_purchase_amount) AS total_purchase_amount,

                    SUM(t.total_grn_qty) AS total_grn_qty,
                    SUM(t.total_grn_amount) AS total_grn_amount,

                    SUM(t.total_return_qty) AS total_return_qty,
                    SUM(t.total_return_amount) AS total_return_amount
                FROM (

                SELECT 
                     
                            po.idpurchase_order,
                            SUM(pod.product_qty) AS purchase_qty,
                            SUM(pod.product_qty * pod.product_purchase_price) AS total_purchase_amount,

                         
                            IFNULL((grn_data.total_grn_qty), 0) AS total_grn_qty,
                            IFNULL((grn_data.total_grn_amount), 0) AS total_grn_amount,

                        
                            IFNULL((grn_data.total_return_qty), 0) AS total_return_qty,
                            IFNULL((grn_data.total_return_amount), 0) AS total_return_amount

                        FROM purchase_order po
                        INNER JOIN purchase_order_details pod
                            ON pod.purchase_order_id = po.idpurchase_order

               
                        LEFT JOIN (
                            SELECT 
                                grn.purchase_order_id,
                                SUM(gd.grn_qty_received) AS total_grn_qty,
                                SUM(gd.grn_qty_received * gd.grn_unit_price) AS total_grn_amount,
                                (IFNULL(return_data.total_return_qty, 0)) AS total_return_qty,
                                (IFNULL(return_data.total_return_amount, 0)) AS total_return_amount

                            FROM goods_received_note grn
                            INNER JOIN grn_details gd 
                                ON gd.grn_id = grn.grn_id

                            LEFT JOIN (
                                SELECT 
                                rn.grn_id,
                                SUM(rn.return_qty) AS total_return_qty,
                                SUM(rn.return_qty * gd.grn_unit_price) AS total_return_amount
                                FROM return_notes rn
                                INNER JOIN grn_details gd
                                ON gd.grn_id = rn.grn_id AND gd.grn_product_id = rn.product_id
                                WHERE LOWER(rn.return_status) = 'active'
                                GROUP BY rn.grn_id
                            ) return_data ON return_data.grn_id = grn.grn_id

                            GROUP BY grn.purchase_order_id
                        ) grn_data ON grn_data.purchase_order_id = po.idpurchase_order

                        WHERE po.business_id = ?
                        AND po.status != 'Cancelled'
                        AND DATE(po.order_date) BETWEEN ? AND ?
                        GROUP BY po.idpurchase_order
                ) t
        ", [$businessId, $start, $end]);

        return response()->json([
            'success' => true,
            'data' => $data[0] ?? [
                'purchase_qty' => 0,
                'total_purchase_amount' => 0,
                'total_grn_qty' => 0,
                'total_grn_amount' => 0,
                'total_return_qty' => 0,
                'total_return_amount' => 0,
            ],
            'from' => Carbon::parse($start)->format('d-M-Y'),
            'to'   => Carbon::parse($end)->format('d-M-Y'),
        ]);
    }

    public function getting_revenue_income_cogs(Request $request)
    {
        $businessId = $request->business_id;

        $from = $request->input('from');
        $to   = $request->input('to');

        if (empty($from) || empty($to)) {
            $from = Carbon::now()->subMonths(2)->startOfMonth()->toDateString();
            $to   = Carbon::now()->endOfMonth()->toDateString();
        } else {
            try {
                $from = Carbon::parse($from)->toDateString();
                $to   = Carbon::parse($to)->toDateString();
                if ($from > $to) {
                    [$from, $to] = [$to, $from];
                }
            } catch (\Exception $e) {
                $from = Carbon::now()->subMonths(2)->startOfMonth()->toDateString();
                $to   = Carbon::now()->endOfMonth()->toDateString();
            }
        }

        $periodDays = Carbon::parse($from)->diffInDays(Carbon::parse($to)) + 1;

        // Determine grouping
        if ($periodDays <= 31) {
            $groupBy = "DATE(av.voucher_date)"; // daily
            $dateFormat = "%a %D %b";
            $periodType = 'daily';
        } elseif ($periodDays <= 62) {
            $groupBy = "DATE(av.voucher_date)"; // daily
            $dateFormat = "%a %D %b";
            $periodType = 'daily';
        } else {
            $groupBy = "DATE_FORMAT(av.voucher_date, '%Y-%m')"; // monthly
            $dateFormat = "%b-%Y";
            $periodType = 'monthly';
        }
        
        // ===== Accounts =====
        $salesAccount = AccountsController::get_event_account('1', 'grosstotal_retail');
        $cogsAccount = AccountsController::get_event_account('23', 'cogs');
        $franchiseSalesAccount = AccountsController::get_event_account('37', 'total_invoice_value');

        if (!$salesAccount || !$cogsAccount || !$franchiseSalesAccount) {
            return response()->json(['success' => false, 'message' => 'One or more accounts not found']);
        }

        // ===== Retail Customer Data =====
        $retailCustomerData = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('invoice as i', 'i.id_invoice', '=', 'av.invoice_id')
            ->where('av.voucher_status', 'Active')
            ->where('i.invoice_status', 'valid')
            ->where('i.invoice_type', 'sale')
            ->where('i.business_id', $businessId)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
            ->selectRaw("
                DATE_FORMAT(av.voucher_date, '{$dateFormat}') AS voucher_date,
                SUM(CASE WHEN avd.account_head_id = {$salesAccount->account_head_id} THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0) ELSE 0 END) AS total_sales,
                SUM(CASE WHEN avd.account_head_id = {$cogsAccount->account_head_id} THEN IFNULL(avd.debit,0) - IFNULL(avd.credit,0) ELSE 0 END) AS cogs_balance
            ")
            ->groupBy(DB::raw($groupBy))
            ->get();

        // ===== Franchise Data =====
        $retailFranchiseData = DB::table('account_voucher_detail as avd')
            ->join('account_vouchers as av', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('franchise_orders as fo', 'fo.id_franchise_orders', '=', 'av.franchise_order_id')
            ->join('franchises as f', 'fo.franchise_id', '=', 'f.id_franchises')
            ->where('av.voucher_status', 'Active')
            ->where('fo.order_status', '!=', 'Cancelled')
            ->where('f.business_id', $businessId)
            ->whereBetween(DB::raw('DATE(av.voucher_date)'), [$from, $to])
            ->selectRaw("
                DATE_FORMAT(av.voucher_date, '{$dateFormat}') AS voucher_date,
                SUM(CASE WHEN avd.account_head_id = {$franchiseSalesAccount->account_head_id} THEN IFNULL(avd.credit,0) - IFNULL(avd.debit,0) ELSE 0 END) AS total_sales,
                SUM(CASE WHEN avd.account_head_id = {$cogsAccount->account_head_id} THEN IFNULL(avd.debit,0) - IFNULL(avd.credit,0) ELSE 0 END) AS cogs_balance
            ")
            ->groupBy(DB::raw($groupBy))
            ->get();

        // ===== Merge Data =====
        $mergedData = [];

        foreach ($retailCustomerData as $data) {
            $mergedData[$data->voucher_date] = [
                'voucher_date' => $data->voucher_date,
                'total_sales' => $data->total_sales,
                'cogs_balance' => $data->cogs_balance,
            ];
        }

        foreach ($retailFranchiseData as $data) {
            if (isset($mergedData[$data->voucher_date])) {
                $mergedData[$data->voucher_date]['total_sales'] += $data->total_sales;
                $mergedData[$data->voucher_date]['cogs_balance'] += $data->cogs_balance;
            } else {
                $mergedData[$data->voucher_date] = [
                    'voucher_date' => $data->voucher_date,
                    'total_sales' => $data->total_sales,
                    'cogs_balance' => $data->cogs_balance,
                ];
            }
        }

        $mergedData = array_map(function ($item) {
            $item['net_profit'] = $item['total_sales'] - $item['cogs_balance'];
            return $item;
        }, $mergedData);

           // ===== Fill missing dates with zero =====
            $start = Carbon::parse($from);
            $end   = Carbon::parse($to);

            $dates = [];
            if ($periodType === 'daily') {
                for ($date = $start; $date <= $end; $date->addDay()) {
                    $formatted = $date->format('D jS M'); // same as your daily format
                    $dates[$formatted] = [
                        'voucher_date' => $formatted,
                        'total_sales' => 0,
                        'cogs_balance' => 0,
                    ];
                }
            } else {
                // monthly
                for ($date = $start->copy()->startOfMonth(); $date <= $end; $date->addMonth()) {
                    $formatted = $date->format('M-Y');
                    $dates[$formatted] = [
                        'voucher_date' => $formatted,
                        'total_sales' => 0,
                        'cogs_balance' => 0,
                    ];
                }
            }

            // Merge actual data into all dates
            foreach ($mergedData as $key => $value) {
                $dates[$value['voucher_date']] = $value;
            }

            // Calculate net profit
            $finalData = array_map(function ($item) {
                $item['net_profit'] = $item['total_sales'] - $item['cogs_balance'];
                return $item;
            }, $dates);

            // Sort
            usort($finalData, function ($a, $b) {
                return strtotime($a['voucher_date']) <=> strtotime($b['voucher_date']);
            });

            return response()->json([
                'success' => true,
                'from' => Carbon::parse($from)->format('d-M-Y'),
                'to' => Carbon::parse($to)->format('d-M-Y'),
                'data' => $finalData
            ]);
    }

    // ------------------------> Retail Dashboard End <----------------------------
    
    // ------------------------>Head Office Dashboard Start<--------------------------
    public function head_office_dashboard(Request $request){
      $branches = DB::table('business')->select('id_business','business_name')->where('ho_accounts','Yes')->orwhere('ho','Yes')->get();
      $business_type = DB::table('business_type')
      ->select('id_business_type','business_type')
      ->join('business' ,'business.business_type_id' , 'business_type.id_business_type')
      ->where('ho_accounts','Yes')->orwhere('ho','Yes')
      ->groupBy('business.business_type_id')
      ->get();     

      $account_main = DB::table('account_main')->select('id_account_main','account_main')->get();  

      if(session('ho') !='Yes'){
        return response()->view('error.403', ['message' => 'You are not authorized to view this page.'], 403);
      }

      $sectors =  DB::table('sectors')
      ->select('sector','id_sector')
      ->join('business' ,'business.sector_id' , 'sectors.id_sector')
      ->where('ho_accounts','Yes')->orwhere('ho','Yes')
      ->groupBy('business.sector_id')
      ->get(); 
      
      return view('dashboards.head_office_dashboard',compact('branches','business_type','account_main','sectors'));
    }

    public function getBusinessRevenueTargetsPerformance(Request $request){ 
        $currentYear = $request->target_year ??  date('Y');
        $by_sector_or_location = $request->by_sector_or_location_revenue_targets ??  'by_location';

        $targetYear  = $request->input('target_year',$currentYear);
        $prevYear    = $targetYear - 1;

        $joinForSectorOrBusinessType ="";
        $selectForSectorOrBusinessType ="";
        $groupColum ="";
        if ($by_sector_or_location == 'by_location') {
            $selectForSectorOrBusinessType =" bt.id_business_type , bt.business_type ,";
            $joinForSectorOrBusinessType =" JOIN business_type bt ON bt.id_business_type = b.business_type_id ";
            $groupColum =" bt.id_business_type , ";

        }elseif ($by_sector_or_location == 'by_sector') {
            $selectForSectorOrBusinessType =" bt.id_sector as id_business_type , bt.sector as business_type ,";
            $joinForSectorOrBusinessType =" JOIN sectors bt ON bt.id_sector = b.sector_id ";
             $groupColum =" bt.id_sector , ";
        }else if($by_sector_or_location =='all_branches'){

            $selectForSectorOrBusinessType =" b.id_business as id_business_type , b.business_name as business_type ,";
            $joinForSectorOrBusinessType ="  ";
            $groupColum ="  ";
        }

        $sql=" SELECT
                $selectForSectorOrBusinessType
                b.business_name AS branch,

                ROUND(
                    MAX(CASE WHEN t.year = $targetYear  THEN t.amount ELSE 0 END),
                2) AS target_year,

                ROUND(
                    SUM(CASE WHEN r.year = $prevYear THEN r.revenue ELSE 0 END),
                2) AS actual_prev_year,

                ROUND(
                    SUM(CASE WHEN r.year = $targetYear THEN r.revenue ELSE 0 END),
                2) AS actual_target_year,

                ROUND(
                    CASE
                        WHEN SUM(CASE WHEN r.year = $prevYear THEN r.revenue ELSE 0 END) > 0
                        THEN (
                            (
                                SUM(CASE WHEN r.year = $targetYear THEN r.revenue ELSE 0 END)
                            - SUM(CASE WHEN r.year = $prevYear THEN r.revenue ELSE 0 END)
                            )
                            / SUM(CASE WHEN r.year = $prevYear THEN r.revenue ELSE 0 END)
                        ) * 100
                        ELSE 0 
                    END, 
                2) AS yoy_percentage,
                
                ROUND(
                    CASE
                        WHEN SUM(CASE WHEN t.year = $targetYear THEN t.amount ELSE 0 END) > 0
                        THEN (
                            SUM(CASE WHEN r.year = $targetYear THEN r.revenue ELSE 0 END)
                            / SUM(CASE WHEN t.year = $targetYear THEN t.amount ELSE 0 END)
                        ) * 100
                        ELSE 0
                    END,
                2) AS achieved_percentage

            FROM business b 

            $joinForSectorOrBusinessType

            LEFT JOIN ( 
                select
                business_targets.year,
                business_id,
                SUM(amount) as amount
                from business_targets 
                where  business_targets.year = $targetYear  AND deleted_at IS NULL 
                group by business_id , business_targets.year
                
            ) AS t on t.business_id = b.id_business  AND   t.year = $targetYear 

            LEFT JOIN (
                SELECT
                    b.business_id,
                    b.year,
                    SUM(IFNULL(b.balance, 0)) AS revenue
                FROM business_account_monthly_summary b
                WHERE b.account_main = 'Revenue'
                AND b.year IN ($prevYear, $targetYear)
                GROUP BY b.business_id, b.year
            ) r ON r.business_id = b.id_business
            where (b.ho_accounts ='Yes' OR b.ho ='Yes' )
            GROUP BY 
                $groupColum
                b.id_business ";

        $data = DB::select($sql);
    
        if($data){
            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        }else{
            return response()->json([
                'success' =>false,
                'data' => []
            ]);
        } 
    }

    public function getBusinessRevenueTargetsPerformanceMonthly(Request $request)
    {   
        $targetYear = $request->target_year ?? date('Y');
        $businessTypeId = $request->business_type_id;
        $sector_id = $request->sector_id;
        $businessId = $request->business_id;
        $revenue_type = $request->revenue_type ?? 'All';
        $by_sector_or_location_or_all_branches = $request->by_sector_or_location_or_all_branches ?? 'all_branches';

        $account_head_where="";
        $revenue_amount_select =" SUM(IFNULL(b.balance, 0)) AS revenue_amount ";
        //  retail 
        if($revenue_type =="retail"){
            $revenue_amount_select = "
                SUM(
                    IFNULL(b.retail_sale, 0)
                + IFNULL(b.franchise_sale, 0)
                + IFNULL(b.retail_sale_discount, 0)
                + IFNULL(b.franchise_sale_discount, 0)
                ) AS revenue_amount
                ";
        } 
        //  services 
        if($revenue_type =="services"){
            $revenue_amount_select = "
                SUM(
                    IFNULL(b.service_sale, 0)
                + IFNULL(b.gift_vocuher_sale, 0)
                + IFNULL(b.service_sale_discount, 0)
                ) AS revenue_amount
                ";
        }

        // --- Business where ---
        $business_where ='';

        // ---- Target WHERE ----
        $target_type_where ='';
        if($revenue_type !='All'){
         $target_type_where = " AND t.target_type = '$revenue_type' ";
        }

        $targetWhere = "
            t.year = $targetYear
            AND t.deleted_at IS NULL
            $target_type_where
        ";
        $fromDate = $targetYear . '-01-01';
        $toDate   = $targetYear . '-12-31';
        // ---- Revenue WHERE ----
        $revenueWhere = "
            b.account_main = 'Revenue'
            AND b.year = $targetYear 
            $account_head_where
        ";

        // ---- Business filters ----
            if($businessTypeId ==0 && $sector_id ==0){
                $businessIds = DB::table('business')
                    ->where(function($query) {
                        $query->where('ho_accounts', 'Yes')
                            ->orWhere('ho', 'Yes');
                    })
                    ->pluck('id_business')
                    ->toArray();
                if (!empty($businessIds)) {
                    $ids = implode(',', $businessIds);
                    $targetWhere  .= " AND t.business_id IN ($ids) ";
                    $revenueWhere .= " AND b.business_id IN ($ids)";
                } 
            }else if($businessTypeId > 0){
                $business_where =" where b.business_type_id =$businessTypeId ";
                $businessIds = DB::table('business')
                    ->where('business_type_id', $businessTypeId)
                    ->where(function($query) {
                        $query->where('ho_accounts', 'Yes')
                            ->orWhere('ho', 'Yes');
                    })
                    ->pluck('id_business')
                    ->toArray();
                if (!empty($businessIds)) {
                    $ids = implode(',', $businessIds);
                    $targetWhere  .= " AND t.business_id IN ($ids) ";
                    $revenueWhere .= " AND b.business_id IN ($ids)";
                }

            }else if($sector_id > 0){
                $business_where =" where b.sector_id =$sector_id ";
                $businessIds = DB::table('business')
                    ->where('sector_id', $sector_id)
                    ->where(function($query) {
                        $query->where('ho_accounts', 'Yes')
                            ->orWhere('ho', 'Yes');
                    })
                    ->pluck('id_business')
                    ->toArray();
                if (!empty($businessIds)) {
                    $ids = implode(',', $businessIds);
                    $targetWhere  .= " AND t.business_id IN ($ids) ";
                    $revenueWhere .= " AND b.business_id IN ($ids)";
                }

            }

        $query = "
                    SELECT 
                        b.id_business AS business_id,
                        b.business_name,
                        months.month,
                        months.month_name,

                        COALESCE(SUM(targets.target_amount), 0) AS target,
                        COALESCE(SUM(revenue.revenue_amount), 0) AS revenue,

                        CASE 
                            WHEN COALESCE(SUM(targets.target_amount), 0) > 0 
                            THEN ROUND(
                                (COALESCE(SUM(revenue.revenue_amount), 0) /
                                COALESCE(SUM(targets.target_amount), 0)) * 100, 2
                            )
                            ELSE 0
                        END AS achieved_percentage,

                        COALESCE(SUM(revenue.revenue_amount), 0)
                        - COALESCE(SUM(targets.target_amount), 0) AS variance

                    FROM business b
                    JOIN months ON 1 = 1

                    LEFT JOIN (
                        SELECT 
                            t.business_id,
                            t.month,
                            SUM(t.amount) AS target_amount
                        FROM business_targets t
                        WHERE $targetWhere
                        GROUP BY t.business_id, t.month
                    ) targets ON targets.business_id = b.id_business AND targets.month = months.month

                    LEFT JOIN (
                        SELECT
                            b.business_id,
                            b.year,
                            b.month_num as month,
                            $revenue_amount_select

                        FROM business_account_monthly_summary b
                        WHERE $revenueWhere
                        GROUP BY b.business_id, b.month_num 
                    ) revenue ON revenue.business_id = b.id_business AND revenue.month = months.month

                    $business_where

                    GROUP BY 
                        b.id_business,
                        months.month

                    ORDER BY 
                        b.id_business,
                        months.month
                ";
        // return $query;
        $monthlyData = DB::select($query);
        $totalTarget = 0;
        $totalRevenue = 0;
        foreach ($monthlyData as $data) {
            $totalTarget += $data->target;
            $totalRevenue += $data->revenue;
        }
        $overallAchievedPercentage = $totalTarget > 0 ? ($totalRevenue / $totalTarget) * 100 : 0;

        $businessTypes = DB::table('business_type')
            ->select('id_business_type', 'business_type')
            ->orderBy('business_type')
            ->get();

        $businessesQuery = DB::table('business')
            ->select('id_business', 'business_name')
            ->orderBy('business_name');
            
        if ($businessTypeId) {
            $businessesQuery->where('business_type_id', $businessTypeId);
        }

        $businesses = $businessesQuery->get();
        return response()->json([
            'success' => true,
            'data' => [
                'monthly_data' => $monthlyData,
                'summary' => [
                    'total_target' => round($totalTarget, 2),
                    'total_revenue' => round($totalRevenue, 2),
                    'overall_achieved_percentage' => round($overallAchievedPercentage, 2),
                    'total_variance' => round($totalRevenue - $totalTarget, 2)
                ],
                'filters' => [
                    'business_types' => $businessTypes,
                    'businesses' => $businesses,
                    'selected_year' => $targetYear,
                    'selected_business_type' => $businessTypeId,
                    'selected_business' => $businessId
                ]
            ]
        ]);
    }

    public function branches_comparison(Request $request){
       
        $comparison_year = $request->comparison_year ?? date('Y');
        $businessTypeId = $request->business_type_id_for_comparison ?? 0;
        $sector_id = $request->sector_id ?? 0;
        $main_account = $request->account_main ?? "Revenue";
        
        $validAccounts = ['Revenue', 'Liabilities', 'Equity', 'Non Financial', 'Expenses', 'Assets'];
        if (!in_array($main_account, $validAccounts)) {
            $main_account = 'Revenue';
        }
        //-------------> MIS data query start<--------------------------    
            $sql =" SELECT
                        s.account_main,
                        s.business_id,
                        b.business_name,
                        s.month_name AS month,
                        s.month_num,
                        SUM( IFNULL(s.balance,0)) AS balance
                    FROM
                        business_account_monthly_summary AS s
                    INNER JOIN business AS b
                    ON
                        b.id_business = s.business_id
                    WHERE
                        s.account_main = '$main_account' AND s.year = $comparison_year AND(
                            b.ho = 'Yes' OR b.ho_accounts = 'Yes'
                        )";
                    if ($businessTypeId && $businessTypeId > 0) {
                       $sql .=" AND b.business_type_id = $businessTypeId ";
                    }
                    if ($sector_id && $sector_id > 0) {
                       $sql .=" AND b.sector_id = $sector_id ";
                    }

                    $sql .=" GROUP BY
                            s.business_id,
                            s.month_num,
                            s.month_name,
                            s.account_main
                            ORDER BY
                                s.business_id ASC ,
                            s.month_num ASC; ";

               $branches_comparison_data = DB::select($sql);
            //-------------> MIS data query start<--------------------------

        if($branches_comparison_data){
            return response()->json([
                'success' => true,
                'data' => $branches_comparison_data,
                'filters' => [
                    'year' => $comparison_year,
                    'account_main' => $main_account,
                    'business_type' => $businessTypeId
                ]
            ]);
        } else {
            return response()->json([
                'success' => false,
                'data' => [],
                'message' => 'No data found for the selected filters'
            ]);
        }    
    }
    
    // ------------------------>Head Office Dashboard End<----------------------------
} 