<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\DiscountReason;
use App\Models\DiscountPassword;

class DiscountConfigurationController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function index(Request $request)
    {
        $business_id = session('business_id');
        
        // Fetch discount reasons
        $discountReasons = DiscountReason::where('business_id', $business_id)
            ->orderBy('id_discount_reasons', 'ASC')
            ->get();

        // Fetch discount users only if user has permission to view them
        $discountUsers = collect([]);
        if (auth()->user()->can('view_discount_users')) {
            $discountUsers = DiscountPassword::where('business_id', $business_id)
                ->orderBy('id', 'ASC')
                ->get();
        }

        return view('discount_configuration.list', compact('discountReasons', 'discountUsers'));
    }

    public function save_or_update_reason(Request $request)
    {
        try {
            $id = $request->input('id_discount_reasons');
            $isAddMode = empty($id);

            // Check permissions
            if ($isAddMode && !auth()->user()->can('add_discount_reasons')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to add discount reasons.',
                ], 403);
            }

            if (!$isAddMode && !auth()->user()->can('edit_discount_reasons')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to edit discount reasons.',
                ], 403);
            }

            $request->validate([
                'id_discount_reasons' => 'nullable|integer',
                'discount_reason' => 'required|string|max:255',
                'type' => 'required|in:Percentage,Fixed Amount',
                'value' => 'required|numeric|min:0',
                'active' => 'required|in:Yes,No',
            ]);

            $type = $request->input('type');
            $value = $request->input('value');
            $business_id = session('business_id');

            // Set discount_perc and fixed_amount based on type
            $discount_perc = 0;
            $fixed_amount = 0;
            
            if ($type === 'Percentage') {
                $discount_perc = $value;
            } else if ($type === 'Fixed Amount') {
                $fixed_amount = $value;
            }

            $discountReason = DiscountReason::updateOrCreate(
                ['id_discount_reasons' => $id],
                [
                    'discount_reason' => $request->input('discount_reason'),
                    'business_id' => $business_id,
                    'active' => $request->input('active'),
                    'discount_perc' => $discount_perc,
                    'fixed_amount' => $fixed_amount,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Discount reason saved successfully',
                'reason' => $discountReason,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function save_or_update_user(Request $request)
    {
        try {
            $id = $request->input('id');
            $isAddMode = empty($id);

            // Check permissions
            if ($isAddMode && !auth()->user()->can('add_discount_users')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to add discount users.',
                ], 403);
            }

            if (!$isAddMode && !auth()->user()->can('edit_discount_users')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to edit discount users.',
                ], 403);
            }

            $validationRules = [
                'name' => 'required|string|max:255',
                'username' => 'required|string|max:255',
                'email' => 'required|email|max:255',
            ];

            if ($isAddMode) {
                // For add mode, password is required
                $validationRules['new_password'] = 'required|string|min:1';
                $validationRules['confirm_password'] = 'required|string|same:new_password';
            } else {
                // For edit mode, id is required and password is optional
                $validationRules['id'] = 'required|integer';
                $validationRules['old_password'] = 'nullable|string';
                $validationRules['new_password'] = 'nullable|string|min:1';
                $validationRules['confirm_password'] = 'nullable|string|same:new_password';
            }

            $request->validate($validationRules);

            $business_id = session('business_id');

            if ($isAddMode) {
                // Create new user
                // Get the next ID (since id is not auto-increment, we need to find max)
                $maxId = DiscountPassword::max('id') ?? 0;
                $newId = $maxId + 1;

                $user = DiscountPassword::create([
                    'id' => $newId,
                    'business_id' => $business_id,
                    'name' => $request->input('name'),
                    'username' => $request->input('username'),
                    'email' => $request->input('email'),
                    'password' => md5($request->input('new_password')),
                    'created_on' => now(),
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Discount user created successfully',
                    'user' => $user,
                ]);
            } else {
                // Update existing user
                $user = DiscountPassword::where('id', $id)->first();

                if (!$user) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Discount user not found',
                    ], 404);
                }

                // Verify old password if new password is provided
                if ($request->filled('new_password')) {
                    if (!$request->filled('old_password')) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Old password is required to change password',
                        ], 400);
                    }

                    // Verify old password (using MD5 as per login system)
                    if (md5($request->input('old_password')) !== $user->password) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Old password is incorrect',
                        ], 400);
                    }

                    // Hash new password using MD5 (same as login system)
                    $user->password = md5($request->input('new_password'));
                }

                $user->name = $request->input('name');
                $user->username = $request->input('username');
                $user->email = $request->input('email');
                $user->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Discount user updated successfully',
                    'user' => $user,
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
