<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\BusinessType;
use App\Models\AccountEvents;
use App\Models\AccountEventMapping;
use App\Models\Account;
use App\Models\AccountControl;
use App\Models\AccountMain;
use App\Models\AccountSubControl;
use App\Models\AccountHeads;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;
use App\Models\Stores;
use App\Models\Dispatch;
use App\Models\Staff;
use App\Models\Products;
use App\Models\ProductBatch;
use App\Models\DispatchRequest;
use App\Models\AdjustmentNotes;

use App\Services\productBatchService;
use App\Services\dispatchRequestService;


use yajra\Datatables\Datatables;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DispatchController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    function dispatchNotes(Request $request)
    {
        $business_id = session('business_id');
        $store_id = $request->input('store_id', '');

        if($request->has('business_id')){
            $business_id = $request->business_id;
        }

        if(session('ho')=='Yes'){
            //Get all business stores of businesses with common_products = "Yes"            
            $stores = Stores::select('business_stores.*',
                'business.business_name'
            )
            ->join('business', 'business_stores.business_id', '=', 'business.id_business')
            ->whereIn('business_id', function($query){
                $query->select('id_business')
                        ->from('business')
                        ->where('common_products', session('common_products'));
            })->get();
        } else {
            //get stores of current business only
            $stores = Stores::where('business_id', session('business_id'))->get();
        }

        $business=Business::where('id_business', $business_id)->first();

        $staff = Staff::select(
            'staff.id_staff', 
            DB::RAW('CONCAT(staff.staff_fullname, " - ", business.business_name) as staff_fullname')
        )
        ->join('business', 'staff.business_id', '=', 'business.id_business');
        if(session('ho')=='Yes'){
            //Get staff of all businesses with common_products = "Yes"
            $staff = $staff->whereIn('staff.business_id', function($query){
                $query->select('id_business')
                        ->from('business')
                        ->where('common_products', session('common_products'));
            });
        } else {
            $staff = $staff->where('staff.business_id', $business_id);
        }

        $staff = $staff->where('staff_active', 'Y')
            ->get();


        return view('products.dispatch.dispatch_notes', compact('stores', 'store_id', 'business_id', 'staff'));
    }

    function get_dispatch_notes_data(Request $request)
    {
        $business_id = session('business_id');
        if($request->has('business_id')){
            $business_id = $request->business_id;
        }

        $dispatch_notes = Dispatch::select('dispatch_notes.*' , 
        DB::RAW('date_format(dispatch_date, "%d-%m-%Y %H:%i") as formatted_dispatch_date'),
        'business.business_name', DB::RAW('staff.staff_fullname as dispatched_to')
        )
        ->join('business', 'dispatch_notes.business_id', '=', 'business.id_business')
        ->join('staff', 'dispatch_notes.dispatch_to_staff', '=', 'staff.id_staff')
            ->where('dispatch_notes.business_id', $business_id)
            ->get();

        return response()->json([
            'data' => $dispatch_notes,
            'message' => 'Dispatch notes retrieved successfully.',
            'message_type' => 'success',
            'message_button' => 'btn btn-success',
        ]);
    }

    function cancel_dispatch($id){
        //code to cancel dispatch
        $dispatch = Dispatch::where('id_dispatch_note', $id)->first();
        if(session('user_role') != 'Super User' && session('user_role') != 'Super Admin'){
            return response()->json([
                'message' => 'You do not have permission to cancel dispatch notes.',
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
            ]);
        } else {

            try{
                DB::beginTransaction();
                if($dispatch){
                    $dispatch->status = 'Cancelled';
                    $dispatch->cancelled_by = session('user_name');
                    $dispatch->cancelled_on = now();
                    $dispatch->save();

                    //cancel all account vouchers linked to this dispatch note
                    AccountVouchers::where('dispatch_id', $dispatch->id_dispatch_note)
                    ->update([
                        'voucher_status' => 'Cancelled',
                        'cancelled_by' => session('user_name'),
                        'cancelled_on' => now(),
                        'cancellation_reason' => 'Cancelled due to Dispatch Note Cancellation ID: ' . $dispatch->id_dispatch_note
                    ]);

                    DB::commit();
                    return response()->json([
                        'message' => 'Dispatch note cancelled successfully.',
                        'message_type' => 'success',
                        'message_button' => 'btn btn-success',
                    ]);
                } else {
                    DB::rollBack();
                    return response()->json([
                        'message' => 'Dispatch note not found.',
                        'message_type' => 'error',
                        'message_button' => 'btn btn-danger',
                    ]);
                }
            } catch(\Exception $e){
                DB::rollBack();
                return response()->json([
                    'message' => 'Error cancelling dispatch note: ' . $e->getMessage(),
                    'message_type' => 'error',
                    'message_button' => 'btn btn-danger',
                ]);
            }
        }
    }

    function add_dispatch_note(Request $request, $isKitAdjustmentVoucher = 'No'){
        try{
            DB::beginTransaction();
            
            $business_id = session('business_id');

            if($request->has('business_id')){
                $business_id = $request->business_id;
            }

            //business id for Ho if 'ho_accounts' = 'Yes'            
            if(session('ho_accounts')=='Yes'){
                $new_business_id = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $new_business_id = $business_id;
            }

            if($request->dispatch_batch_id){
                $batch = ProductBatch::select('product_batch.*', 'business.business_name', 'business.id_business')
                ->join('business_stores', 'product_batch.store_id', '=', 'business_stores.id_business_stores')
                ->join('business', 'business_stores.business_id', '=', 'business.id_business')
                ->where('id_batch', $request->dispatch_batch_id)->first();
                $batch_amount = (float) ($batch ? ($batch->batch_amount ?? 0) : 0);

                //Insert only for the selected batch
                $batch_dispatch = new Dispatch();
                $batch_dispatch->batch_id = $request->dispatch_batch_id;
                $batch_dispatch->batch = $request->dispatch_batch;
                $batch_dispatch->business_id = $business_id;
                $batch_dispatch->product_id = $request->dispatch_product_id;
                $batch_dispatch->dispatch_to_staff = $request->dispatched_to;
                $batch_dispatch->dispatch_qty = $request->calculated_dispatch_qty;
                $batch_dispatch->dispatch_measure = $request->calculated_dispatch_measure;
                $batch_dispatch->unit_type = $request->dispatch_unit_type;
                $batch_dispatch->measure_unit = $request->dispatch_measure_unit;                
                $batch_dispatch->dispatch_date = $request->dispatch_date ? date('Y-m-d H:i:s', strtotime($request->dispatch_date)) : now();
                $batch_dispatch->created_by = session('user_name');
                $batch_dispatch->status = 'Active';
                $batch_dispatch->dispatch_comment = $request->dispatch_comment;
                $batch_dispatch->visit_id = $request->visit_id ?? 0;
                $batch_dispatch->visit_service_id = $request->visit_service_id ?? 0;
                $batch_dispatch->request_id = $request->request_id ?? 0;
                $batch_dispatch->adjustment_note_id = $request->adjustment_note_id ?? 0;            
                $batch_dispatch->save();

                $voucherDescriptin = 'Cost Of Goods for Dispatch Note ID: ' . $batch_dispatch->id_dispatch_note . ' Product: ' . $request->dispatch_product .' with unit price' . $batch_amount. ' Qty: ' . $request->calculated_dispatch_qty;
                if($isKitAdjustmentVoucher =='Yes'){
                $voucherDescriptin = 'Cost Of Goods for Dispatch Note ID: ' . $batch_dispatch->id_dispatch_note . ' Product: ' . $request->dispatch_product .' with unit price' . $batch_amount. ' Qty: ' . $request->calculated_dispatch_qty.' During Compiled Kit Inventory Adjustment';
                }    

                //Add Cost Of goods Account Voucher
                $new_voucher = new AccountVouchers();
                $new_voucher->business_id = $business_id;
                $new_voucher->description = $voucherDescriptin;
                $new_voucher->voucher_type = 3; //Journal Voucher
                $new_voucher->voucher_date = $request->dispatch_date ? date('Y-m-d H:i:s', strtotime($request->dispatch_date)) : now();
                $new_voucher->dispatch_id = $batch_dispatch->id_dispatch_note;
                $new_voucher->created_by = session('user_name');
                $new_voucher->voucher_status = 'Active';
                $new_voucher->voucher_amount = $batch_amount * $request->calculated_dispatch_qty;
                $new_voucher->cost_center = 1;
                $new_voucher->cost_center_name = 'Back Office';
                $new_voucher->business_partner=5; //Branch
                $new_voucher->business_partner_id = $business_id;
                $new_voucher->business_partner_name = $batch->business_name;
                $new_voucher->payment_mode = 'Cash';
                $new_voucher->auto_voucher = 'Yes';
                $new_voucher->visit_id = $request->visit_id ?? 0;                                
                $new_voucher->save();
                $id_voucher = $new_voucher->id_account_vouchers;

                //Get the event mappings for Invoice Creation id_events = 23
                $event_mappings = AccountEventMapping::where('account_event_id', 23)->where('business_id', $new_business_id)->get();
                $debit_amounts = [];
                $credit_amounts = [];
                $debit_accounts = array();
                $credit_accounts = array();
                foreach ($event_mappings as $mapping) {
                    if($mapping['transaction_type']=='debit'){
                        array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    } else {
                        array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    }
                    
                    if($mapping['transaction_type']=='debit'){
                        //make debit entry data
                        if($isKitAdjustmentVoucher =='Yes'){
                            if($mapping['entity_name']=='debit_inventory_on_compile_kit'){array_push($debit_amounts, [ 'entity_name' => 'debit_inventory_on_compile_kit', 'amount' => round((float) $batch_amount * (float) $request->calculated_dispatch_qty,2)]);}    
                        }else{
                            if($mapping['entity_name']=='cogu'){array_push($debit_amounts, [ 'entity_name' => 'cogu', 'amount' => round((float) $batch_amount * (float) $request->calculated_dispatch_qty,2)]);}    
                        }
                    } else if($mapping['transaction_type']=='credit'){
                        //make credit entry data
                        if($mapping['entity_name']=='inventory'){array_push($credit_amounts, [ 'entity_name' => 'inventory', 'amount' => round((float) $batch_amount * (float) $request->calculated_dispatch_qty,2) ]);}                                        
                    }
                }
                //get the sum of debit amounts and credit amounts
                $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
                $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

                if(bccomp($debit_sum, $credit_sum, 2) !== 0){ 
                    DB::rollBack(); //Rollback the transaction
                    return response()->json([
                        "success" => false,
                        "message" => " debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                        "message_type" => "error",
                        "message_btn" => "btn btn-danger"
                    ]);
                
                } else {
                    //Insert voucher Details
                    foreach($debit_accounts as $debit){
                        $debit_parts = explode("|", $debit);
                        $new_receiving_voucher_details = new AccountVoucherDetail();
                        $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
                        $new_receiving_voucher_details->detail_remarks = 'Dispatch Note for Product  '. $request->dispatch_product  .' Batch ID : '. $request->dispatch_batch_id;
                        $new_receiving_voucher_details->account_head_id = $debit_parts[0];
                        $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                        $new_receiving_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $debit_amounts));
                        $new_receiving_voucher_details->credit = 0;    
                        if($new_receiving_voucher_details->debit != 0){                                            
                            $new_receiving_voucher_details->save();
                        }
                    }
                    foreach($credit_accounts as $credit){
                        $credit_parts = explode("|", $credit);
                        $new_receiving_voucher_details = new AccountVoucherDetail();
                        $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
                        $new_receiving_voucher_details->detail_remarks = 'Dispatch Note for Product  '. $request->dispatch_product  .' Batch ID : '. $request->dispatch_batch_id;
                        $new_receiving_voucher_details->account_head_id = $credit_parts[0];
                        $new_receiving_voucher_details->debit = 0;
                        $entity = $credit_parts[2]; // this is 'paid_cash' etc.
                        $new_receiving_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $credit_amounts));
                        if($new_receiving_voucher_details->credit != 0){           
                            $new_receiving_voucher_details->save();
                        }
                    }
                }

                DB::commit();
                return response()->json([
                    "success" => true,
                    'message' => 'Dispatch note added successfully.',
                    'message_type' => 'success',
                    'message_button' => 'btn btn-success',
                ]);

            }else {
                //find batchs with sufficient qty using productBatchService
                $productBatchService = new productBatchService();
                $batches = $productBatchService->findSufficientBatches(                   
                    $request->dispatch_product_id,
                    0,                    
                    $request->dispatch_store_id
                );

                $batches = json_decode($batches, true);

                if(empty($batches) || $batches == null ){
                    // Suggest alternative batches
                    DB::rollBack();
                    return response()->json([
                        "success" => false,
                        "message" => "None of the batches have sufficient quantity for product: " . $request->dispatch_product . " in Store ID: " . $request->dispatch_store_id,
                        "message_type" => "error",
                        "message_btn" => "danger"
                    ]);
                }

                $business_name = Business::where('id_business', $business_id)->value('business_name');
                $cogs_total = 0;
                // Insert dispatch for each batch until qty is fulfilled
                $batch_quantity_needed = floatval($request->calculated_dispatch_qty); // string or decimal
                $epsilon = '0.000001'; // safety margin for tiny float dust

                foreach ($batches as $batch) {

                    // Stop if remaining quantity is effectively zero
                    if (bccomp($batch_quantity_needed, '0', 6) <= 0) {
                        break;
                    }

                    // Skip batches with zero or negative stock
                    if (bccomp($batch['instock'], '0', 6) <= 0) {
                        continue;
                    }

                    // Determine how much stock is available in this batch
                    $batch_instock = $batch['instock'];

                    // floor() without float problems (BC math way)
                    // /$batch_quantity = floor($batch_instock);
                    $batch_quantity = floatval($batch['instock']);

                    // Deduct the smaller of batch stock or needed qty
                    $quantity_to_deduct = min($batch_quantity, $batch_quantity_needed);

                    // If min() returns 0, nothing to deduct → skip
                    if ($quantity_to_deduct <= 0) {
                        continue;
                    }

                    // Calculate measure qty (rounded to 4 decimals)
                    $qty_per_unit = (float) $request->dispatch_qty_per_unit;
                    $measure_qty_to_deduct = round($quantity_to_deduct * $qty_per_unit, 4);

                    // Create dispatch
                    $dispatch = new Dispatch();
                    $dispatch->batch_id           = $batch['batch_id'];
                    $dispatch->batch              = $batch['batch_no'];
                    $dispatch->business_id        = $business_id;
                    $dispatch->product_id         = $request->dispatch_product_id;
                    $dispatch->dispatch_to_staff  = $request->dispatched_to;
                    $dispatch->dispatch_qty       = $quantity_to_deduct;
                    $dispatch->dispatch_measure   = $measure_qty_to_deduct;
                    $dispatch->unit_type          = $request->dispatch_unit_type;
                    $dispatch->measure_unit       = $request->dispatch_measure_unit;
                    $dispatch->dispatch_date      = $request->dispatch_date 
                                                    ? date('Y-m-d H:i:s', strtotime($request->dispatch_date)) 
                                                    : now();
                    $dispatch->created_by         = session('user_name');
                    $dispatch->status             = 'Active';
                    $dispatch->dispatch_comment   = $request->dispatch_comment;
                    $dispatch->visit_id           = $request->visit_id ?? 0;
                    $dispatch->visit_service_id   = $request->visit_service_id ?? 0;
                    $dispatch->request_id         = $request->request_id ?? 0;
                    $dispatch->adjustment_note_id = $request->adjustment_note_id ?? 0;
                    $dispatch->save();

                    // COGS
                    $batch_amount = $batch['batch_amount'];
                    $cogs_total += $batch_amount * $quantity_to_deduct;

                    // Update remaining quantity needed
                    // (use bc math to prevent drift)
                    $batch_quantity_needed = bcsub($batch_quantity_needed, $quantity_to_deduct, 6);

                    // ------- ACCOUNT VOUCHER CREATION -------- //

                    $new_voucher = new AccountVouchers();
                    $new_voucher->business_id          = $business_id;
                    $new_voucher->description          = 'Cost Of Goods for Dispatch Note ID: ' . 
                                                        $dispatch->id_dispatch_note . 
                                                        ' Product: ' . $request->dispatch_product . 
                                                        ' with unit price ' . $batch_amount . 
                                                        ' Qty: ' . $quantity_to_deduct;
                    $new_voucher->voucher_type         = 3;
                    $new_voucher->voucher_date         = $dispatch->dispatch_date;
                    $new_voucher->dispatch_id          = $dispatch->id_dispatch_note;
                    $new_voucher->created_by           = session('user_name');
                    $new_voucher->voucher_status       = 'Active';
                    $new_voucher->voucher_amount       = $batch_amount * $quantity_to_deduct;
                    $new_voucher->cost_center          = 1;
                    $new_voucher->cost_center_name     = 'Back Office';
                    $new_voucher->business_partner     = 5;
                    $new_voucher->business_partner_id  = $business_id;
                    $new_voucher->business_partner_name = $business_name;
                    $new_voucher->payment_mode         = 'Cash';
                    $new_voucher->auto_voucher         = 'Yes';
                    $new_voucher->visit_id             = $request->visit_id ?? 0;
                    $new_voucher->save();

                    // Fetch mappings
                    $event_mappings = AccountEventMapping::where('account_event_id', 23)
                                        ->where('business_id', $new_business_id)
                                        ->get();

                    $debit_amounts  = [];
                    $credit_amounts = [];
                    $debit_accounts = [];
                    $credit_accounts = [];

                    foreach ($event_mappings as $mapping) {

                        if ($mapping['transaction_type'] == 'debit') {
                            $debit_accounts[] = $mapping['account_head_id'] . "|" . $mapping['transaction_type'] . "|" . $mapping['entity_name'];

                            if ($mapping['entity_name'] == 'cogu') {
                                $debit_amounts[] = [
                                    'entity_name' => 'cogu',
                                    'amount' => round($batch_amount * $quantity_to_deduct, 2)
                                ];
                            }

                        } else {
                            $credit_accounts[] = $mapping['account_head_id'] . "|" . $mapping['transaction_type'] . "|" . $mapping['entity_name'];

                            if ($mapping['entity_name'] == 'inventory') {
                                $credit_amounts[] = [
                                    'entity_name' => 'inventory',
                                    'amount' => round($batch_amount * $quantity_to_deduct, 2)
                                ];
                            }
                        }
                    }

                    // Validate debit/credit
                    $debit_sum  = !empty($debit_amounts)  ? array_sum(array_column($debit_amounts, 'amount')) : null;
                    $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

                    if ($debit_sum === null || $credit_sum === null || $debit_sum != $credit_sum) {
                        DB::rollBack();
                        return response()->json([
                            "message" => "debit=$debit_sum credit=$credit_sum Error in voucher creation.",
                            "message_type" => "error",
                            "success" => false
                        ]);
                    }

                    if ($debit_sum == 0 || $credit_sum == 0) {
                        DB::rollBack();
                        return response()->json([
                            "message" => "debit=$debit_sum credit=$credit_sum Zero debit/credit.",
                            "message_type" => "error",
                            "success" => false
                        ]);
                    }

                    // Insert voucher details
                    foreach ($debit_accounts as $db) {
                        [$acc_id, , $entity] = explode("|", $db);
                        $amount = array_sum(array_map(fn($x) => $x['entity_name'] === $entity ? $x['amount'] : 0, $debit_amounts));

                        if ($amount > 0) {
                            $d = new AccountVoucherDetail();
                            $d->account_voucher_id = $new_voucher->id_account_vouchers;
                            $d->detail_remarks     = 'Dispatch Note for Product ' . $request->dispatch_product;
                            $d->account_head_id    = $acc_id;
                            $d->debit              = $amount;
                            $d->credit             = 0;
                            $d->save();
                        }
                    }

                    foreach ($credit_accounts as $cr) {
                        [$acc_id, , $entity] = explode("|", $cr);
                        $amount = array_sum(array_map(fn($x) => $x['entity_name'] === $entity ? $x['amount'] : 0, $credit_amounts));

                        if ($amount > 0) {
                            $c = new AccountVoucherDetail();
                            $c->account_voucher_id = $new_voucher->id_account_vouchers;
                            $c->detail_remarks     = 'Dispatch Note for Product ' . $request->dispatch_product;
                            $c->account_head_id    = $acc_id;
                            $c->debit              = 0;
                            $c->credit             = $amount;
                            $c->save();
                        }
                    }
                }

                DB::commit();
                return response()->json([
                    'message' => 'Dispatch notes added successfully.',
                    'message_type' => 'success',
                    'message_button' => 'btn btn-success',
                    "success" => true
                ]);
            }

           
        } catch(\Exception $e){
            DB::rollBack();
            return response()->json([
                'message' => 'Error adding dispatch note: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
                "success" => false
            ]);
        }
    }

    function visit_dispatch_request($id_visit){
        try {
            $visit_id = $id_visit;
            
            $dispatchRequestService = new dispatchRequestService();
            $response = $dispatchRequestService->createDispatchRequest($visit_id);

            log::info('Dispatch Request for Visit ID: '.$visit_id. ' '. $response);

            return $response;
        } catch (\Exception $e) {            
            return response()->json([
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
                'message' => 'Error creating dispatch request: ' . $e->getMessage(),
            ], 500);
        }
    }

    function dispatch_requests(Request $request)
    {
        $business_id = session('business_id');
        if($request->has('business_id')){
            $business_id = $request->business_id;
        }
        
        if(session('ho')=='Yes'){
            //Get all business with common_products = "Yes"            
            $businesses = Business::where('common_products', session('common_products'))->get();
        } else {
            //get current business only
            $businesses = Business::where('id_business', session('business_id'))->get();
        }
        $businesses = Business::where('common_products', "Yes")->get();

        $stores = Stores::where('business_id', $business_id)->get();
        $store_id = $request->input('store_id', 0);

        return view('products.dispatch.dispatch_requests', compact('business_id', 'businesses', 'stores', 'store_id'));
    }

    public function get_dispatch_requests_data(Request $request)
    {
        $business_id = $request->business_id ?? session('business_id');
        $start_date  = $request->input('start_date', date('Y-m-d'));

        $query = DispatchRequest::select(
                'dispatch_request.*', 
                DB::raw('Concat(dispatch_request.product_id, "-", dispatch_request.product_name) as product'), 
                'business_products.unit_type',
                'id_customer_visits', 'customers.customer_name',
                'visit_services.service_name', 'business_products.qty_per_unit', 'visit_service_staffs.staff_id',
                DB::raw('DATE_FORMAT(dispatch_request_date, "%d-%m-%Y %H:%i") as formatted_request_date'),
                DB::raw('visit_service_staffs.staff_name as staff'),
                DB::RAW('ROUND(SUM(COALESCE(dispatch_notes.dispatch_measure, 0)), 2) as dispatched_qty'),                            
                DB::RAW('ROUND(dispatch_request.product_qty - SUM(COALESCE(dispatch_notes.dispatch_measure, 0)), 2) as remaining_qty')
            )
            ->join('customer_visits', 'dispatch_request.customer_visit_id', '=', 'customer_visits.id_customer_visits')
            ->join('customers', 'customer_visits.customer_id', '=', 'customers.id_customers')
            ->join('visit_services', 'dispatch_request.visit_service_id', '=', 'visit_services.id_visit_services')
            ->join('visit_service_staffs', 'dispatch_request.visit_service_staff_id', '=', 'visit_service_staffs.id_visit_service_staffs')
            ->join('staff', 'visit_service_staffs.staff_id', '=', 'staff.id_staff')
            ->join('business_products', 'dispatch_request.product_id', '=', 'business_products.id_business_products')
            ->leftJoin('dispatch_notes', function($join) {
                $join->on('dispatch_request.id_dispatch_request', '=', 'dispatch_notes.request_id')
                     ->where('dispatch_notes.status', 'Active');
            })
            ->where('customer_visits.business_id', $business_id)
            ->whereDate('dispatch_request_date', '=', $start_date)
            ->groupBy('dispatch_request.id_dispatch_request');

        return DataTables::of($query)->make(true);
    }

    public function change_add_product(Request $request)
    {
        try{
            DB::beginTransaction();

            $id_dispatch_request = $request->input('dispatch_request_id');
             $dispatch_request = DispatchRequest::where('id_dispatch_request', $id_dispatch_request)->first();
                if(!$dispatch_request){
                    DB::rollBack();
                    return response()->json([
                        'message' => 'Dispatch Request not found.',
                        'message_type' => 'error',
                        'message_button' => 'btn btn-danger',
                    ]);
                }
           
            if($request->has('current_product_id') && $request->input('current_product_id') != ''){
                //Change Product         

                $dispatch_request->product_id = $request->input('new_product_id');
                $dispatch_request->product_name = $request->input('new_product_name');
                $dispatch_request->save();
            } else {
                //Add Product
                $new_product = Products::where('id_business_products', $request->input('new_product_id'))->first();

                $new_dispatch_request = new DispatchRequest();
                $new_dispatch_request->customer_visit_id = $dispatch_request->customer_visit_id;
                $new_dispatch_request->created_on = now();
                $new_dispatch_request->customer_id = $dispatch_request->customer_id;
                $new_dispatch_request->service_id = $dispatch_request->service_id;
                $new_dispatch_request->created_by = session('user_name');
                $new_dispatch_request->dispatch_request_status = 'Active';
                $new_dispatch_request->dispatch_request_date = now();                
                $new_dispatch_request->visit_service_id = $dispatch_request->visit_service_id;
                $new_dispatch_request->visit_service_staff_id = $dispatch_request->visit_service_staff_id;
                $new_dispatch_request->product_id = $new_product->id_business_products;
                $new_dispatch_request->product_name = $new_product->product;
                $new_dispatch_request->product_qty = $dispatch_request->product_qty;
                $new_dispatch_request->product_unit = $new_product->measure_unit;

                $new_dispatch_request->save();
            }

            DB::commit();
            return response()->json([
                'message' => 'Product changed/added successfully.',
                'message_type' => 'success',
                'message_button' => 'btn btn-success',
            ]);

        } catch(\Exception $e){
            DB::rollBack();
            return response()->json([
                'message' => 'Error changing/adding product: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
            ]);
        }
    }

    public function save_kit_batch(Request $request) {
       // return $request->all();
        try{

            //Dispatch the Components if sufficient stock is available in the store
            $components = json_decode($request->input('components'), true); //Array of component products with required_qty
            $store_id = $request->input('store_id');
            $store_name = Stores::where('id_business_stores', $store_id)->value('business_store');

            $business_id = session('business_id');
            DB::beginTransaction(); 

            $batchService = new productBatchService();
            $batchesData = [];
            $shortageList = [];  
            foreach($components as $component){
                $productId = $component['component_id'];
                $qty_per_kit = $component['qty_per_kit'];
                $measure_qty_per_kit = $component['measure_qty_per_kit'];
                $calculated_kit_qty = $component['total_consumption']; //Total measure qty needed for the kit batch
                
                //Calculate required qty from qty_per_unit of the component product
                $componentDetail = Products::select('qty_per_unit', 'purchase_price','product')->where('id_business_products', $productId)->first();
                $component_qty_per_unit = $componentDetail->qty_per_unit;
                $component_cost_per_unit = $componentDetail->purchase_price;
                $requiredQty = $calculated_kit_qty / $component_qty_per_unit;

                $batches = $batchService->findSufficientBatches($productId, 0, $store_id);
                $batches = json_decode($batches, true);

                $availableQty = 0;
                if (!empty($batches)) {
                    foreach ($batches as $b) {                        
                        $availableQty += floatval($b['instock']);
                    }
                }

                if ($availableQty < $requiredQty) { 
                    $product_batch_route = route('products.batches', $productId);
                    $shortageList[] = [
                        "product_batch_route"     => $product_batch_route,
                        "productid"     => $productId,
                        "productname"   => $componentDetail->product,
                        "required_qty"  => $requiredQty,
                        "available_qty" => $availableQty,
                        "shortage"      => $requiredQty - $availableQty
                    ];
                    continue;
                }

                $remainingQty = $requiredQty;
                foreach ($batches as $batch) {
                    if ($remainingQty <= 0) break;
                    $instock = floatval($batch['instock']);
                    if ($instock <= 0) continue;

                    $deductQty = min($instock, $remainingQty , $requiredQty); // third parameter for capacity

                    $batchesData[] = [
                        "productid" => $batch['id'],
                        "brandname" => $batch['business_brand_name'],
                        "productname" => $batch['product'],
                        "category" => $batch['category'],
                        "batch_id" => $batch['batch_id'],
                        "batch" => $batch['batch_no'],
                        "batch_amount" => $batch['batch_amount'] ?? 0,
                        "qty" => $deductQty,
                        'measure_unit' => $batch['measure_unit'],
                        'qty_per_unit' => $batch['qty_per_unit'],
                        'unit_type' => $batch['unit_type'],
                        'store' => $batch['store'],
                        'store_id' => $batch['store_id'],
                    ];
                    $remainingQty -= $deductQty;
                }
            }
            if (!empty($shortageList)) {
                DB::rollBack();
                return response()->json([
                    "message" => "Insufficient stock for products.",
                    "message_type" => "error",
                    "data" => $shortageList ,
                    "success" => false 
                ]);
            }
        
            $total_kit_cost=0;
            foreach($batchesData as $row){
                  $dispatch_request = new Request([
                    'dispatch_batch_id' => $row['batch_id'],
                    'dispatch_batch' => $row['batch'],
                    'dispatch_product_id' => $row['productid'],
                    'dispatch_product' => $row['productname'],
                    'dispatched_to' => 'Kit Assembly',
                    'calculated_dispatch_qty' => $row['qty'],
                    'calculated_dispatch_measure' => $row['qty'] * $row['qty_per_unit'] ,
                    'dispatch_unit_type' => $row['unit_type'],
                    'dispatch_measure_unit' => $row['measure_unit'],
                    'dispatch_store_id' => $row['store_id'],
                    'dispatch_date' => now(),
                    'dispatch_comment' => 'Dispatch for Kit Batch Creation',
                ]);
                $total_kit_cost += ($row['qty'] * $row['batch_amount']);
                $dispatch_result = $this->add_dispatch_note($dispatch_request ,'Yes');
                $dresponse = json_decode($dispatch_result->getContent(), true);
                
                if(!$dispatch_result || (isset($dresponse['success']) && $dresponse['success'] == false)){
                    DB::rollBack();
                    return response()->json([
                        'message' => $dresponse['message'].'Failed to dispatch component product '.$row['productname'].' from store '.$store_name.' for kit batch creation',
                        'success' => false,
                        'message_type' => 'error',
                        'message_btn' => 'btn btn-danger',
                    ]);
                }

            }

            $kit_id = $request->input('id_business_products');
            $store_id = $request->input('store_id');
            $batch_number = $request->input('batch_number');
            $expiry_date = $request->input('expiry_date');
            $initial_stock = $request->input('batch_qty');

            $batch_amount =  $initial_stock > 0 ?  $total_kit_cost/ $initial_stock : 0;

            $new_batch = new ProductBatch();            
            $new_batch->product_id = $kit_id;
            $new_batch->store_id = $store_id;
            $new_batch->batch_number = $batch_number;
            $new_batch->expiry_date = $expiry_date;
            $new_batch->batch_amount =  $initial_stock > 0 ?  $total_kit_cost/ $initial_stock : 0;
            $new_batch->batch_qty = $initial_stock;                        
            $new_batch->created_at = now();
            $new_batch->save();
            
            
            $adjustment_note = new AdjustmentNotes();
            $adjustment_note->batch_id = $new_batch->id_batch;
            $adjustment_note->product_id = $kit_id;
            $adjustment_note->adjustment_qty = $initial_stock;
            $adjustment_note->adjustment_remarks = 'Initial stock for Kit Batch Creation';
            $adjustment_note->unit_price = $initial_stock > 0 ?  $total_kit_cost/ $initial_stock : 0;
            $adjustment_note->created_by = session('user_name');
            $adjustment_note->adjustment_date = now();
            $adjustment_note->save(); 

            // if(session('ho_accounts')=='Yes'){
            //     $business_id_for_account_head= Business::where('business.ho', 'Yes')->value('id_business');
            // } else {
            //     $business_id_for_account_head = $business_id;
            // }

            // $new_voucher = new AccountVouchers();
            // $new_voucher->business_id = $business_id;
            // $new_voucher->description = 'Cost of Goods Adjusted for goods found in inventory through Adjustment Note ID: ' . $adjustment_note->id_adjustment_notes . ' Product ID: ' . $request->kit_id .' with unit price' . $batch_amount. ' Qty: ' . $initial_stock;
            // $new_voucher->voucher_type = 3; 
            // $new_voucher->voucher_date =  now();
            // $new_voucher->created_by = session('user_name');
            // $new_voucher->voucher_status = 'Active';
            // $new_voucher->voucher_amount = $total_kit_cost;
            // $new_voucher->cost_center = 1;
            // $new_voucher->cost_center_name = 'Front Desk';
            // $new_voucher->business_partner=5; 
            // $new_voucher->business_partner_id = $business_id;
            // $new_voucher->business_partner_name = session('business_name');
            // $new_voucher->payment_mode = 'Cash';
            // $new_voucher->auto_voucher = 'Yes';            
            // $new_voucher->adjustment_id = $adjustment_note->id_adjustment_notes;            
            // $new_voucher->save();
            // $id_voucher = $new_voucher->id_account_vouchers;

            // $event_mappings = AccountEventMapping::where('account_event_id', 22)->where('business_id', $business_id_for_account_head)->get();
            // $debit_amounts = [];
            // $credit_amounts = [];
            // $debit_accounts = array();
            // $credit_accounts = array();
            // foreach ($event_mappings as $mapping) {
            //     if($mapping['transaction_type']=='debit'){
            //         array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
            //     } else {
            //         array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
            //     }

            //     if($mapping['transaction_type']=='debit'){
                    
            //         if($mapping['entity_name']=='inventory'){array_push($debit_amounts, [ 'entity_name' => 'inventory', 'amount' => round((float) $total_kit_cost ,2)]);}                
            //     } else if($mapping['transaction_type']=='credit'){
                    
            //         if($mapping['entity_name']=='cogd'){array_push($credit_amounts, [ 'entity_name' => 'cogd', 'amount' => round((float) $total_kit_cost ,2) ]);}                                        
            //     }
            // }
         
            // $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
            // $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

            // if(bccomp($debit_sum, $credit_sum, 2) !== 0){
            //     DB::rollBack(); 
            //     return response()->json([
            //         "success" => false,
            //         "message" => " debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
            //         "message_type" => "error",
            //         "message_btn" => "btn btn-danger"
            //     ]);
            
            // } else {
                    
            //     foreach($debit_accounts as $debit){
            //         $debit_parts = explode("|", $debit);
            //         $new_receiving_voucher_details = new AccountVoucherDetail();
            //         $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
            //         $new_receiving_voucher_details->detail_remarks = 'Adjustment Note ID '. $adjustment_note->id_adjustment_notes  .' Batch ID : '. $new_batch->id_batch;
            //         $new_receiving_voucher_details->account_head_id = $debit_parts[0];
            //         $entity = $debit_parts[2]; 
            //         $new_receiving_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
            //             return $item['entity_name'] === $entity ? $item['amount'] : 0;
            //         }, $debit_amounts));
            //         $new_receiving_voucher_details->credit = 0;                                  
            //         $new_receiving_voucher_details->save();
            //     }
            //     foreach($credit_accounts as $credit){
            //         $credit_parts = explode("|", $credit);
            //         $new_receiving_voucher_details = new AccountVoucherDetail();
            //         $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
            //         $new_receiving_voucher_details->detail_remarks = 'Adjustment Note ID '. $adjustment_note->id_adjustment_notes  .' Batch ID : '. $new_batch->id_batch;
            //         $new_receiving_voucher_details->account_head_id = $credit_parts[0];
            //         $new_receiving_voucher_details->debit = 0;
            //         $entity = $credit_parts[2]; 
            //         $new_receiving_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
            //             return $item['entity_name'] === $entity ? $item['amount'] : 0;
            //         }, $credit_amounts));
            //         $new_receiving_voucher_details->save();
            //     }
            // }

            DB::commit();
            return response()->json([
                'message' => 'Compiled kit batch saved and adjustment note created successfully',
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to save compiled kit batch: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

}