<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\AccountEvents;
use App\Models\AccountEventMapping;
use Illuminate\Support\Facades\Bus;
use App\Models\Account;
use App\Models\AccountControl;
use App\Models\AccountMain;
use App\Models\AccountSubControl;
use App\Models\AccountHeads;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;

use Illuminate\Support\Facades\DB;

use function PHPSTORM_META\map;

class ExpensesController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function daily_expenses(Request $request)
    {

        $mDate = $request->input('date', now()->toDateString());

        if(!$mDate) {
            $mDate = now()->toDateString();
        }


        $business_id = session('business_id');
        if($request->has('business_id')){
            $business_id = $request->business_id;
        }

        if(session('ho')=='Yes' && session('user_role')=='Super User'){
            $branches = Business::where('common_products', session('common_products'))->get();
        } else {
            $branches = Business::where('id_business', $business_id)->get();
        }

        $business=Business::where('id_business', $business_id)->first();

        $expenses = AccountVouchers::select(
                'account_vouchers.id_account_vouchers',
                'account_vouchers.created_on',
                'account_vouchers.voucher_date',         
                DB::raw('GROUP_CONCAT(DISTINCT account_heads.account_head SEPARATOR "=> ") as account_head'),
                'account_vouchers.description',
                DB::raw('GROUP_CONCAT(DISTINCT account_sub_control.account_sub_control SEPARATOR "=> ") as account_sub_control'),
                DB::raw('SUM(CASE WHEN account_sub_control.account_sub_control = "Cash" THEN ifnull(account_voucher_detail.credit, 0) - ifnull(account_voucher_detail.debit, 0) ELSE 0 END) as is_cash'),
                DB::raw('SUM(CASE WHEN account_sub_control.account_sub_control != "Cash" THEN ifnull(account_voucher_detail.debit, 0) - ifnull(account_voucher_detail.credit, 0) ELSE "No" END) as not_cash')
            )
            ->leftJoin('account_voucher_detail', 'account_voucher_detail.account_voucher_id', 'account_vouchers.id_account_vouchers')
            ->leftJoin('account_heads', 'account_heads.id_account_heads', 'account_voucher_detail.account_head_id')
            ->leftJoin('account_sub_control', 'account_sub_control.id_account_sub_control', 'account_heads.account_sub_control_id')
            ->where('voucher_type', 1) // 1 represents payment vouchers
            ->where('account_vouchers.voucher_status', 'Active')
            ->whereDate('voucher_date', $mDate)
            ->where('account_vouchers.business_id', $business_id)
            ->groupBy(
                'account_vouchers.id_account_vouchers',
                'account_vouchers.created_on',
                'account_vouchers.voucher_date',                         
                'account_vouchers.description'
            );

        if(session('user_role') != 'Super User' && session('user_role') != 'Super Admin' && session('user_role') != 'Admin'){
            $expenses = $expenses->whereNotLike('account_sub_control.account_sub_control', 'Salar%');
        }

        $expenses = $expenses->get();

        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $business_id;
        }

         //get cash accounts
        $cashAccounts = AccountSubControl::whereLike('account_sub_control', '%Cash%')
        ->leftJoin('account_heads', 'account_heads.account_sub_control_id', 'account_sub_control.id_account_sub_control')
        ->where('account_heads.business_id', $common_business_id)
        ->get();

        $expenseAccounts = AccountSubControl::whereLike('account_sub_control', '%Ad% %Expense%')
        ->leftJoin('account_heads', 'account_heads.account_sub_control_id', 'account_sub_control.id_account_sub_control')
        ->where('account_heads.business_id', $common_business_id)
        ->get();

         //get all business partners
        $business_partners = DB::table('account_business_partner')
            ->get();

        //return $cashAccounts;

        return view('expenses.daily_expenses', compact('branches', 'business', 'expenses', 'mDate', 'business_id', 'cashAccounts', 'expenseAccounts', 'business_partners'));
    }

    public function save_expense(Request $request)
    {
        // Validate the request data
        $validatedData = $request->validate([
            'voucher_date' => 'required|date',
            'description' => 'required|string|max:255',
            'business_id' => 'required|integer',
            'payment_account' => 'required|integer',
            'expense_account' => 'required|integer',
            'amount' => 'required|numeric|min:0.01',
            'business_partner_type' => 'nullable|string',
            'business_partner' => 'nullable|integer',
        ]);

        DB::beginTransaction();

        try {
            // Create a new Account Voucher
            $voucher = new AccountVouchers();
            $voucher->voucher_date = $validatedData['voucher_date'];
            $voucher->description = $validatedData['description'];
            $voucher->business_id = $validatedData['business_id'];
            $voucher->voucher_type = 1; // Payment voucher
            $voucher->voucher_status = 'Active';
            $voucher->business_partner = $validatedData['business_partner_type'] ?? 0;
            $voucher->business_partner_id = $validatedData['business_partner'] ?? 0;
            $voucher->business_partner_name = $validatedData['business_partner_name'] ?? null;
            $voucher->save();

            // Create Debit Entry for Expense Account
            $debitEntry = new AccountVoucherDetail();
            $debitEntry->account_voucher_id = $voucher->id_account_vouchers;
            $debitEntry->account_head_id = $validatedData['expense_account'];
            $debitEntry->debit = $validatedData['amount'];
            $debitEntry->credit = 0;
            $debitEntry->save();

            // Create Credit Entry for Payment Account
            $creditEntry = new AccountVoucherDetail();
            $creditEntry->account_voucher_id = $voucher->id_account_vouchers;
            $creditEntry->account_head_id = $validatedData['payment_account'];
            $creditEntry->debit = 0;
            $creditEntry->credit = $validatedData['amount'];
            $creditEntry->save();

            DB::commit();

            return response()->json([
                'success' => true, 
                'message' => 'Expense saved successfully.',
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
                'message' => $e->getMessage()
            ], 500);
        }
    }

}