<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\EyelashRecord;
use App\Models\EyelashType;
use App\Models\EyelashThickness;
use App\Models\EyelashLength;
use App\Models\EyelashCurl;

class EyelashRecordsController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function index(Request $request)
    {
        $business_id = session('business_id');
        
        // Fetch configuration data for dropdowns
        $eyelashTypes = EyelashType::where('business_id', $business_id)->orderBy('eyelash_type')->get();
        $thicknesses = EyelashThickness::where('business_id', $business_id)->orderBy('thickness')->get();
        $lengths = EyelashLength::where('business_id', $business_id)->orderBy('length')->get();
        $curls = EyelashCurl::where('business_id', $business_id)->orderBy('curl')->get();

        return view('customers.eyelash_records.list', compact('eyelashTypes', 'thicknesses', 'lengths', 'curls'));
    }

    public function recordsData(Request $request)
    {
        $business_id = session('business_id');
        
        $query = EyelashRecord::where('business_id', $business_id);

        // Apply filters
        if ($request->filled('filter_id')) {
            $query->where('id_eyelashes', 'LIKE', '%' . $request->filter_id . '%');
        }
        if ($request->filled('filter_customer')) {
            $query->where('customer_name', 'LIKE', '%' . $request->filter_customer . '%');
        }
        if ($request->filled('filter_type')) {
            $query->where('eyelash_type', 'LIKE', '%' . $request->filter_type . '%');
        }
        if ($request->filled('filter_thickness')) {
            $query->where('thickness', 'LIKE', '%' . $request->filter_thickness . '%');
        }
        if ($request->filled('filter_length')) {
            $query->where('length', 'LIKE', '%' . $request->filter_length . '%');
        }
        if ($request->filled('filter_curl')) {
            $query->where('curl', 'LIKE', '%' . $request->filter_curl . '%');
        }
        if ($request->filled('filter_full_set_refill')) {
            $query->where('full_set_refill', 'LIKE', '%' . $request->filter_full_set_refill . '%');
        }
        if ($request->filled('filter_date')) {
            $query->where('date', 'LIKE', '%' . $request->filter_date . '%');
        }
        if ($request->filled('filter_price')) {
            $query->where('price', 'LIKE', '%' . $request->filter_price . '%');
        }
        if ($request->filled('filter_remarks')) {
            $query->where('remarks', 'LIKE', '%' . $request->filter_remarks . '%');
        }

        $records = $query->orderBy('id_eyelashes', 'DESC')->get();

        return response()->json([
            'data' => $records
        ]);
    }

    public function save_or_update_record(Request $request)
    {
        try {
            $request->validate([
                'id_eyelashes' => 'nullable|integer',
                'customer_id' => 'nullable|integer',
                'customer_name' => 'required|string|max:255',
                'eyelash_type' => 'required|string|max:255',
                'thickness' => 'nullable|string|max:255',
                'length' => 'nullable|string|max:255',
                'curl' => 'nullable|string|max:255',
                'full_set_refill' => 'nullable|string|max:255',
                'date' => 'nullable|string|max:255',
                'price' => 'nullable|string|max:255',
                'remarks' => 'nullable|string',
            ]);

            $id = $request->input('id_eyelashes');
            $business_id = session('business_id');

            $data = [
                'business_id' => $business_id,
                'customer_id' => $request->input('customer_id'),
                'customer_name' => $request->input('customer_name'),
                'eyelash_type' => $request->input('eyelash_type'),
                'thickness' => $request->input('thickness'),
                'length' => $request->input('length'),
                'curl' => $request->input('curl'),
                'full_set_refill' => $request->input('full_set_refill'),
                'date' => $request->input('date'),
                'price' => $request->input('price'),
                'remarks' => $request->input('remarks'),
            ];

            if (empty($id)) {
                // Create new record - let database handle auto-increment
                $data['created_date'] = now();
                $record = EyelashRecord::create($data);
            } else {
                // Update existing record
                $record = EyelashRecord::updateOrCreate(
                    ['id_eyelashes' => $id],
                    $data
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Eyelash record saved successfully',
                'record' => $record,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    // Configuration CRUD methods
    public function save_type(Request $request)
    {
        try {
            $request->validate([
                'id_eyelash_types' => 'nullable|integer',
                'eyelash_type' => 'required|string|max:255',
            ]);

            $id = $request->input('id_eyelash_types');
            $business_id = session('business_id');

            $type = EyelashType::updateOrCreate(
                ['id_eyelash_types' => $id],
                [
                    'eyelash_type' => $request->input('eyelash_type'),
                    'business_id' => $business_id,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Type saved successfully',
                'type' => $type,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function delete_type($id)
    {
        try {
            EyelashType::where('id_eyelash_types', $id)->delete();
            return response()->json(['success' => true, 'message' => 'Type deleted successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function save_thickness(Request $request)
    {
        try {
            $request->validate([
                'id_eyelash_thickness' => 'nullable|integer',
                'thickness' => 'required|string|max:255',
            ]);

            $id = $request->input('id_eyelash_thickness');
            $business_id = session('business_id');

            $thickness = EyelashThickness::updateOrCreate(
                ['id_eyelash_thickness' => $id],
                [
                    'thickness' => $request->input('thickness'),
                    'business_id' => $business_id,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Thickness saved successfully',
                'thickness' => $thickness,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function delete_thickness($id)
    {
        try {
            EyelashThickness::where('id_eyelash_thickness', $id)->delete();
            return response()->json(['success' => true, 'message' => 'Thickness deleted successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function save_length(Request $request)
    {
        try {
            $request->validate([
                'id_eyelash_length' => 'nullable|integer',
                'length' => 'required|string|max:255',
            ]);

            $id = $request->input('id_eyelash_length');
            $business_id = session('business_id');

            $length = EyelashLength::updateOrCreate(
                ['id_eyelash_length' => $id],
                [
                    'length' => $request->input('length'),
                    'business_id' => $business_id,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Length saved successfully',
                'length' => $length,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function delete_length($id)
    {
        try {
            EyelashLength::where('id_eyelash_length', $id)->delete();
            return response()->json(['success' => true, 'message' => 'Length deleted successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function save_curl(Request $request)
    {
        try {
            $request->validate([
                'id_eyelash_curl' => 'nullable|integer',
                'curl' => 'required|string|max:255',
            ]);

            $id = $request->input('id_eyelash_curl');
            $business_id = session('business_id');

            $curl = EyelashCurl::updateOrCreate(
                ['id_eyelash_curl' => $id],
                [
                    'curl' => $request->input('curl'),
                    'business_id' => $business_id,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Curl saved successfully',
                'curl' => $curl,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function delete_curl($id)
    {
        try {
            EyelashCurl::where('id_eyelash_curl', $id)->delete();
            return response()->json(['success' => true, 'message' => 'Curl deleted successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function get_configuration_data()
    {
        $business_id = session('business_id');
        
        return response()->json([
            'types' => EyelashType::where('business_id', $business_id)->orderBy('eyelash_type')->get(),
            'thicknesses' => EyelashThickness::where('business_id', $business_id)->orderBy('thickness')->get(),
            'lengths' => EyelashLength::where('business_id', $business_id)->orderBy('length')->get(),
            'curls' => EyelashCurl::where('business_id', $business_id)->orderBy('curl')->get(),
        ]);
    }
}
