<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\FacialRecord;

class FacialRecordsController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function index(Request $request)
    {
        return view('customers.facial_records.list');
    }

    public function recordsData(Request $request)
    {
        $business_id = session('business_id');
        
        $query = FacialRecord::where('business_id', $business_id);

        // Apply filters
        if ($request->filled('filter_id')) {
            $query->where('id', 'LIKE', '%' . $request->filter_id . '%');
        }
        if ($request->filled('filter_customer')) {
            $query->where('customer', 'LIKE', '%' . $request->filter_customer . '%');
        }
        if ($request->filled('filter_facial')) {
            $query->where('facial', 'LIKE', '%' . $request->filter_facial . '%');
        }
        if ($request->filled('filter_exfoliant')) {
            $query->where('exfoliant', 'LIKE', '%' . $request->filter_exfoliant . '%');
        }
        if ($request->filled('filter_mask')) {
            $query->where('mask', 'LIKE', '%' . $request->filter_mask . '%');
        }
        if ($request->filled('filter_cleanser')) {
            $query->where('cleanser', 'LIKE', '%' . $request->filter_cleanser . '%');
        }
        if ($request->filled('filter_date')) {
            $query->where('date', 'LIKE', '%' . $request->filter_date . '%');
        }
        if ($request->filled('filter_time')) {
            $query->where('time', 'LIKE', '%' . $request->filter_time . '%');
        }
        if ($request->filled('filter_charge')) {
            $query->where('charge', 'LIKE', '%' . $request->filter_charge . '%');
        }
        if ($request->filled('filter_remarks')) {
            $query->where('remarks', 'LIKE', '%' . $request->filter_remarks . '%');
        }

        $records = $query->orderBy('id', 'DESC')->get();

        // Format time for display (convert minutes to H:MM:SS)
        $records->transform(function ($record) {
            if ($record->time) {
                $minutes = (int)$record->time;
                $hours = floor($minutes / 60);
                $mins = $minutes % 60;
                $seconds = 0;
                $record->formatted_time = sprintf('%d:%02d:%02d', $hours, $mins, $seconds);
            } else {
                $record->formatted_time = '0:00:00';
            }
            return $record;
        });

        return response()->json([
            'data' => $records
        ]);
    }

    public function save_or_update_record(Request $request)
    {
        try {
            $request->validate([
                'id' => 'nullable|integer',
                'customer_id' => 'nullable|integer',
                'customer' => 'required|string|max:255',
                'facial' => 'nullable|string|max:255',
                'exfoliant' => 'nullable|string|max:255',
                'mask' => 'nullable|string|max:255',
                'cleanser' => 'nullable|string|max:255',
                'date' => 'nullable|date',
                'time_minutes' => 'required|integer|min:0',
                'charge' => 'nullable|numeric|min:0',
                'remarks' => 'nullable|string',
            ]);

            $id = $request->input('id');
            $business_id = session('business_id');
            $timeMinutes = $request->input('time_minutes');

            // Store as minutes (as integer string) - per spec "Stored in database as minutes"
            $data = [
                'business_id' => $business_id,
                'customer_id' => $request->input('customer_id'),
                'customer' => $request->input('customer'),
                'facial' => $request->input('facial'),
                'exfoliant' => $request->input('exfoliant'),
                'mask' => $request->input('mask'),
                'cleanser' => $request->input('cleanser'),
                'date' => $request->input('date'),
                'time' => (string)$timeMinutes, // Store as minutes
                'charge' => $request->input('charge'),
                'remarks' => $request->input('remarks'),
            ];

            if (empty($id)) {
                // Create new record - let database handle auto-increment
                $data['created_date'] = now();
                $record = FacialRecord::create($data);
            } else {
                // Update existing record
                $record = FacialRecord::updateOrCreate(
                    ['id' => $id],
                    $data
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Facial record saved successfully',
                'record' => $record,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
