<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\CategoryMapping;
use App\Models\CategoryMappingPackage;
use App\Models\MappingFloor;
use App\Models\BusinessTimeslot;
use App\Models\PricingBand;
use App\Models\PackageCategory;
use App\Models\Floor;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

/**
 * FlexiPricingController
 * 
 * Handles all flexi-pricing configuration and management operations.
 * Flexi-pricing allows businesses to offer dynamic discounts based on:
 * - Day of the week
 * - Booking capacity percentage
 * - Time slots (optional)
 * - Floor assignments
 */
class FlexiPricingController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    /**
     * Display the flexi-pricing configuration page
     * 
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $business_id = session('business_id');
        $business = Business::find($business_id);
        
        return view('flexi_pricing.list', compact('business'));
    }

    /**
     * Get all active category mappings for the current business
     * 
     * Returns mappings with their associated package categories and package types.
     * Only returns mappings where active = 1 (soft delete support).
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMappings(Request $request)
    {
        $business_id = session('business_id');
        
        // Only get active mappings (active = 1)
        $mappings = CategoryMapping::where('business_id', $business_id)
            ->where('active', 1)
            ->with([
                'categoryMappingPackages.packageCategory.package_type'
            ])
            ->orderBy('mapping_name')
            ->get();

        // Transform data to ensure consistent structure for frontend
        $mappingsData = $mappings->map(function($mapping) {
            return [
                'id_mapping' => $mapping->id_mapping,
                'mapping_name' => $mapping->mapping_name,
                'business_id' => $mapping->business_id,
                'active' => $mapping->active,
                'created_at' => $mapping->created_at,
                'updated_at' => $mapping->updated_at,
                'category_mapping_packages' => $mapping->categoryMappingPackages->map(function($cmp) {
                    $pkg = $cmp->packageCategory;
                    if (!$pkg) {
                        return null;
                    }
                    return [
                        'id' => $cmp->id,
                        'mapping_id' => $cmp->mapping_id,
                        'package_category_id' => $cmp->package_category_id,
                        'package_category' => [
                            'id_package_category' => $pkg->id_package_category,
                            'service_category' => $pkg->service_category,
                            'package_type_id' => $pkg->package_type_id,
                            'package_type' => $pkg->package_type ? [
                                'id_package_type' => $pkg->package_type->id_package_type,
                                'service_type' => $pkg->package_type->service_type,
                            ] : null
                        ]
                    ];
                })->filter() // Remove null entries
            ];
        });

        return response()->json(['success' => true, 'data' => $mappingsData]);
    }

    /**
     * Create or update a category mapping
     * 
     * A mapping groups multiple package categories together under a single name.
     * This allows applying the same pricing bands to multiple related categories.
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeMapping(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mapping_name' => 'required|string|max:255',
            'mapping_id' => 'nullable|exists:category_mapping,id_mapping',
            'package_category_ids' => 'required|array|min:1',
            'package_category_ids.*' => 'exists:package_category,id_package_category',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 400);
        }

        $business_id = session('business_id');

        DB::beginTransaction();
        try {
            if ($request->mapping_id) {
                // Update existing mapping
                $mapping = CategoryMapping::findOrFail($request->mapping_id);
                $mapping->mapping_name = $request->mapping_name;
                $mapping->save();
            } else {
                // Create new mapping - check for duplicate name
                $exists = CategoryMapping::where('business_id', $business_id)
                    ->where('mapping_name', $request->mapping_name)
                    ->exists();
                
                if ($exists) {
                    return response()->json(['success' => false, 'message' => 'Mapping name already exists'], 400);
                }

                $mapping = CategoryMapping::create([
                    'mapping_name' => $request->mapping_name,
                    'business_id' => $business_id,
                    'active' => 1,
                ]);
            }

            // Sync package categories - remove old associations and create new ones
            CategoryMappingPackage::where('mapping_id', $mapping->id_mapping)->delete();
            foreach ($request->package_category_ids as $package_category_id) {
                CategoryMappingPackage::create([
                    'mapping_id' => $mapping->id_mapping,
                    'package_category_id' => $package_category_id,
                ]);
            }

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Mapping saved successfully', 'mapping' => $mapping]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error saving mapping: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Soft delete a category mapping
     * 
     * Sets active = 0 instead of physically deleting the record.
     * This preserves historical data and allows for easy reactivation.
     * 
     * @param int $id Mapping ID
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteMapping($id)
    {
        try {
            $mapping = CategoryMapping::findOrFail($id);
            
            // Security check: ensure business owns this mapping
            if ($mapping->business_id != session('business_id')) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
            }

            // Soft delete: set active to 0 instead of deleting
            $mapping->active = 0;
            $mapping->save();
            
            return response()->json(['success' => true, 'message' => 'Mapping deactivated successfully']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error deleting mapping: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get detailed information about a mapping for editing
     * 
     * Returns the mapping with all its associations:
     * - Package categories
     * - Floor assignments
     * - Pricing bands
     * - Available options for selection
     * 
     * @param int $id Mapping ID (0 for new mapping)
     * @return \Illuminate\Http\JsonResponse
     */
    public function getMappingDetails($id)
    {
        try {
            $business_id = session('business_id');
            
            // Get all available package categories for this business with package type info
            try {
                $availablePackages = PackageCategory::with('package_type')
                    ->where('business_id', $business_id)
                    ->where('service_category_active', 'Yes')
                    ->get()
                    ->filter(function($pkg) {
                        // Filter out packages where package_type is null or inactive
                        return $pkg->package_type && $pkg->package_type->service_type_active == 'Yes';
                    })
                    ->map(function($pkg) {
                        return [
                            'id_package_category' => $pkg->id_package_category,
                            'service_category' => $pkg->service_category,
                            'package_type_id' => $pkg->package_type_id,
                            'service_type' => $pkg->package_type ? $pkg->package_type->service_type : 'Unknown'
                        ];
                    })
                    ->values(); // Re-index array
            } catch (\Exception $e) {
                \Log::error('Error loading package categories: ' . $e->getMessage());
                $availablePackages = collect([]);
            }

            // Get all available floors for this business
            $availableFloors = Floor::where('business_id', $business_id)
                ->where('floor_status', 'Active')
                ->get();

            // If ID is 0 or '0', return only available packages/floors (for new mapping)
            if ($id == 0 || $id == '0') {
                return response()->json([
                    'success' => true,
                    'mapping' => null,
                    'availablePackages' => $availablePackages->toArray(),
                    'availableFloors' => $availableFloors,
                ]);
            }

            // Get existing mapping with all relationships
            $mapping = CategoryMapping::with([
                'categoryMappingPackages.packageCategory.package_type',
                'mappingFloors.floor',
                'pricingBands.businessTimeslot'
            ])->findOrFail($id);
            
            // Get business timeslots separately (universal timeslots, not mapping-specific)
            $businessTimeslots = BusinessTimeslot::where('business_id', $business_id)
                ->where('active', 1)
                ->orderBy('order')
                ->get();

            // Security check: ensure business owns this mapping
            if ($mapping->business_id != $business_id) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
            }

            return response()->json([
                'success' => true,
                'mapping' => $mapping,
                'availablePackages' => $availablePackages,
                'availableFloors' => $availableFloors,
                'businessTimeslots' => $businessTimeslots,
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error fetching mapping details: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get all active business timeslots
     * 
     * Timeslots are universal (business-level) and can be used across multiple mappings.
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBusinessTimeslots(Request $request)
    {
        try {
            $business_id = session('business_id');
            $timeslots = BusinessTimeslot::where('business_id', $business_id)
                ->where('active', 1)
                ->orderBy('order')
                ->get();
            
            return response()->json([
                'success' => true,
                'timeslots' => $timeslots
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error fetching timeslots: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Save business timeslots
     * 
     * Creates or updates timeslots for the business.
     * Timeslots not included in the request are deactivated (soft delete).
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeTimeslots(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'timeslots' => 'required|array',
            'timeslots.*.timeslot_name' => 'required|string|max:255',
            'timeslots.*.start_time' => 'required|date_format:H:i',
            'timeslots.*.end_time' => 'required|date_format:H:i|after:timeslots.*.start_time',
            'timeslots.*.id' => 'nullable|exists:business_timeslots,id_business_timeslots',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 400);
        }

        $business_id = session('business_id');

        DB::beginTransaction();
        try {
            // Get existing timeslot IDs from request
            $existingIds = collect($request->timeslots)->pluck('id')->filter();
            
            // Deactivate timeslots that are not in the request (soft delete)
            BusinessTimeslot::where('business_id', $business_id)
                ->whereNotIn('id_business_timeslots', $existingIds)
                ->update(['active' => 0]);

            // Update or create timeslots
            foreach ($request->timeslots as $index => $timeslotData) {
                if (isset($timeslotData['id']) && $timeslotData['id']) {
                    // Update existing timeslot
                    $timeslot = BusinessTimeslot::findOrFail($timeslotData['id']);
                    // Security check: ensure business owns this timeslot
                    if ($timeslot->business_id != $business_id) {
                        throw new \Exception('Unauthorized: Timeslot does not belong to this business');
                    }
                    $timeslot->update([
                        'timeslot_name' => $timeslotData['timeslot_name'],
                        'start_time' => $timeslotData['start_time'],
                        'end_time' => $timeslotData['end_time'],
                        'order' => $index,
                        'active' => 1,
                    ]);
                } else {
                    // Create new timeslot
                    BusinessTimeslot::create([
                        'business_id' => $business_id,
                        'timeslot_name' => $timeslotData['timeslot_name'],
                        'start_time' => $timeslotData['start_time'],
                        'end_time' => $timeslotData['end_time'],
                        'order' => $index,
                        'active' => 1,
                    ]);
                }
            }

            DB::commit();
            
            // Return updated timeslots
            $updatedTimeslots = BusinessTimeslot::where('business_id', $business_id)
                ->where('active', 1)
                ->orderBy('order')
                ->get();
            
            return response()->json([
                'success' => true, 
                'message' => 'Time slots saved successfully',
                'timeslots' => $updatedTimeslots
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error saving time slots: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Save pricing bands for a mapping
     * 
     * Pricing bands define discount ranges based on booking capacity percentage.
     * Bands must:
     * - Start at 0% and end at 100%
     * - Have no gaps or overlaps
     * - Be unique per day and timeslot combination
     * 
     * Existing bands are deactivated and new ones are created/updated.
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storePricingBands(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mapping_id' => 'required|exists:category_mapping,id_mapping',
            'bands' => 'required|array',
            'bands.*.day' => 'required|integer|between:1,7', // 1=Sunday, 7=Saturday
            'bands.*.timeslot_id' => 'nullable|exists:business_timeslots,id_business_timeslots',
            'bands.*.min_percentage' => 'required|numeric|min:0|max:100',
            'bands.*.max_percentage' => 'required|numeric|min:0|max:100',
            'bands.*.discount_type' => 'required|in:percentage,fixed',
            'bands.*.discount' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 400);
        }

        $mapping = CategoryMapping::findOrFail($request->mapping_id);
        
        // Security check: ensure business owns this mapping
        if ($mapping->business_id != session('business_id')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        // Validate bands meet requirements (0-100%, no gaps, no overlaps)
        $validationResult = $this->validateBands($request->bands);
        if (!$validationResult['valid']) {
            return response()->json([
                'success' => false, 
                'message' => $validationResult['message'],
                'validation_errors' => $validationResult['errors'] ?? []
            ], 400);
        }

        DB::beginTransaction();
        try {
            // First, deactivate all existing bands for this mapping
            PricingBand::where('mapping_id', $request->mapping_id)
                ->update(['active' => 0]);

            // Create/update bands from request
            foreach ($request->bands as $bandData) {
                // Try to find existing band with same day, timeslot, and percentage range
                $existingBand = PricingBand::where('mapping_id', $request->mapping_id)
                    ->where('day', $bandData['day'])
                    ->where(function($query) use ($bandData) {
                        if ($bandData['timeslot_id'] ?? null) {
                            $query->where('timeslot_id', $bandData['timeslot_id']);
                        } else {
                            $query->whereNull('timeslot_id'); // All-day band
                        }
                    })
                    ->where('min_percentage', $bandData['min_percentage'])
                    ->where('max_percentage', $bandData['max_percentage'])
                    ->first();
                
                if ($existingBand) {
                    // Update existing band and activate it
                    $existingBand->discount_type = $bandData['discount_type'];
                    $existingBand->discount = $bandData['discount'];
                    $existingBand->active = 1;
                    $existingBand->save();
                } else {
                    // Create new band
                    PricingBand::create([
                        'mapping_id' => $request->mapping_id,
                        'day' => $bandData['day'],
                        'timeslot_id' => $bandData['timeslot_id'] ?? null, // null = all-day
                        'min_percentage' => $bandData['min_percentage'],
                        'max_percentage' => $bandData['max_percentage'],
                        'discount_type' => $bandData['discount_type'],
                        'discount' => $bandData['discount'],
                        'active' => 1,
                    ]);
                }
            }

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Pricing bands saved successfully']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error saving pricing bands: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Save floor assignments for a mapping
     * 
     * Associates floors with a mapping. Only bookings on assigned floors
     * are counted when calculating booking capacity.
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeFloors(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mapping_id' => 'required|exists:category_mapping,id_mapping',
            'floors' => 'required|array',
            'floors.*.business_floor_id' => 'required|exists:business_floors,id_business_floors',
            'floors.*.active' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 400);
        }

        $mapping = CategoryMapping::findOrFail($request->mapping_id);
        
        // Security check: ensure business owns this mapping
        if ($mapping->business_id != session('business_id')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        DB::beginTransaction();
        try {
            // Get all floor IDs from request
            $requestFloorIds = collect($request->floors)->pluck('business_floor_id')->toArray();
            
            // Deactivate floors not in the request (soft delete)
            MappingFloor::where('mapping_id', $request->mapping_id)
                ->whereNotIn('business_floor_id', $requestFloorIds)
                ->update(['active' => 0]);

            // Update or create floor mappings
            foreach ($request->floors as $floorData) {
                $mappingFloor = MappingFloor::where('mapping_id', $request->mapping_id)
                    ->where('business_floor_id', $floorData['business_floor_id'])
                    ->first();
                
                if ($mappingFloor) {
                    // Update existing record
                    $mappingFloor->active = $floorData['active'] ? 1 : 0;
                    $mappingFloor->save();
                } else {
                    // Create new record
                    MappingFloor::create([
                        'mapping_id' => $request->mapping_id,
                        'business_floor_id' => $floorData['business_floor_id'],
                        'active' => $floorData['active'] ? 1 : 0,
                    ]);
                }
            }

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Floors saved successfully']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error saving floors: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Validate pricing bands
     * 
     * Ensures bands meet the following requirements:
     * 1. Start at 0% (first band min_percentage = 0)
     * 2. End at 100% (last band max_percentage = 100)
     * 3. No gaps (each band's max = next band's min)
     * 4. No overlaps (bands don't overlap)
     * 
     * Validation is done per day and timeslot combination.
     * 
     * @param array $bands Array of band data
     * @return array ['valid' => bool, 'message' => string, 'errors' => array]
     */
    private function validateBands($bands)
    {
        if (empty($bands)) {
            return ['valid' => false, 'message' => 'At least one pricing band is required'];
        }
        
        // Group bands by day and timeslot_id (null timeslot_id = all-day)
        $grouped = [];
        foreach ($bands as $band) {
            $key = $band['day'] . '_' . ($band['timeslot_id'] ?? 'null');
            if (!isset($grouped[$key])) {
                $grouped[$key] = [];
            }
            $grouped[$key][] = $band;
        }

        $errors = [];
        $dayNames = [1 => 'Sunday', 2 => 'Monday', 3 => 'Tuesday', 4 => 'Wednesday', 5 => 'Thursday', 6 => 'Friday', 7 => 'Saturday'];

        // Validate each group (day + timeslot combination)
        foreach ($grouped as $key => $groupBands) {
            // Sort by min_percentage
            usort($groupBands, function($a, $b) {
                return $a['min_percentage'] <=> $b['min_percentage'];
            });

            $day = $groupBands[0]['day'];
            $dayName = $dayNames[$day] ?? 'Day ' . $day;
            $timeslotInfo = isset($groupBands[0]['timeslot_id']) ? ' (Timeslot ID: ' . $groupBands[0]['timeslot_id'] . ')' : ' (All Day)';

            // Check if starts at 0%
            if ($groupBands[0]['min_percentage'] != 0) {
                $errors[] = "Bands for {$dayName}{$timeslotInfo} must start at 0%";
            }

            // Check if ends at 100%
            $lastBand = end($groupBands);
            if ($lastBand['max_percentage'] != 100) {
                $errors[] = "Bands for {$dayName}{$timeslotInfo} must end at 100%";
            }

            // Check for gaps and overlaps
            for ($i = 0; $i < count($groupBands) - 1; $i++) {
                $current = $groupBands[$i];
                $next = $groupBands[$i + 1];

                // Check for gaps (max != next min)
                if ($current['max_percentage'] != $next['min_percentage']) {
                    $errors[] = "Bands for {$dayName}{$timeslotInfo} must be continuous with no gaps. Gap found between {$current['max_percentage']}% and {$next['min_percentage']}%";
                }

                // Check for invalid range (min >= max)
                if ($current['min_percentage'] >= $current['max_percentage']) {
                    $errors[] = "Invalid band range for {$dayName}{$timeslotInfo}: min ({$current['min_percentage']}%) must be less than max ({$current['max_percentage']}%)";
                }
            }
            
            // Check last band range
            if ($lastBand['min_percentage'] >= $lastBand['max_percentage']) {
                $errors[] = "Invalid band range for {$dayName}{$timeslotInfo}: min ({$lastBand['min_percentage']}%) must be less than max ({$lastBand['max_percentage']}%)";
            }
        }

        if (!empty($errors)) {
            return [
                'valid' => false, 
                'message' => 'Validation failed: ' . implode('. ', $errors),
                'errors' => $errors
            ];
        }

        return ['valid' => true];
    }
}
