<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;

use App\Models\Floor;
use App\Models\Staff;

class FloorController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function service_floors(Request $request)
    {
        $business_id = session('business_id');

        $floors = Floor::where('business_id', $business_id)
            ->orderBy('id_business_floors', 'ASC')
            ->get();

        $staffMembers = Staff::select('id_staff', 'staff_fullname')
            ->where('business_id', $business_id)
            ->orderBy('staff_fullname', 'ASC')
            ->get();

        return view('floors/service_floors_listing', compact('floors', 'staffMembers', 'business_id'));
    }

    public function save_or_update(Request $request)
    {
        try {
            $request->validate([
                'id_business_floors' => 'nullable|integer',
                'floor_name' => 'required|string|max:255',
                'floor_workstations' => 'required|integer|min:0',
                'floormanager_id' => 'nullable|integer',
                'floor_status' => 'required|in:Active,InActive',
            ]);

            $business_id = session('business_id');

            $managerId = $request->input('floormanager_id');
            $managerName = null;
            if (!empty($managerId)) {
                $staff = Staff::where('business_id', $business_id)->where('id_staff', $managerId)->first();
                $managerName = $staff->staff_fullname ?? null;
            }

            $id = $request->input('id_business_floors');

            $floor = Floor::updateOrCreate(
                ['id_business_floors' => $id],
                [
                    'business_id' => $business_id,
                    'floor_name' => $request->input('floor_name'),
                    'floor_workstations' => $request->input('floor_workstations'),
                    'floormanager_id' => $managerId,
                    'floormanager_name' => $managerName,
                    'floor_status' => $request->input('floor_status'),
                    'created_by' => session('user_name'),
                    'created_on' => now(),
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Floor saved successfully',
                'floor' => $floor,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
