<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Franchise;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Bus;
use App\Http\Controllers\AccountsController;
use App\Services\FiltersService;
use Yajra\DataTables\Facades\DataTables;
use Exception;
use App\Models\FranchiseOrder;
use App\Models\FranchiseOrderProduct;
use App\Models\Stores;
use App\Services\productBatchService;
use PhpParser\Node\Stmt\Foreach_;
use App\Models\AccountEventMapping;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;

class FranchiseController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function franchise_search(Request $request)
    {
        $business_id = $request->input('business.id');
        $search_term = $request->input('term');

        $franchises = Franchise::select('id_franchises as id', 
        'business.business_name', 
        'franchises.business_id', 
        'franchises.franchise_name as text',         
        'franchises.franchise_name as franchise_name',         
        'franchises.franchise_contact as contact_person ', 
        'franchises.franchise_email as email', 
        'franchises.franchise_office_phone as franchise_office_phone',
        'franchises.franchise_address as address',
        'franchises.franchise_owner as owner',
        'franchises.franchise_status'
        )
        ->join('business', 'business.id_business', '=', 'franchises.business_id')
        ->where(function ($query) use ($search_term) {
            $query->where('franchises.franchise_name', 'LIKE', '%' . $search_term . '%')
                  ->orWhere('franchises.franchise_contact', 'LIKE', '%' . $search_term . '%')
                  ->orWhere('franchises.franchise_email', 'LIKE', '%' . $search_term . '%')
                  ->orWhere('franchises.franchise_office_phone', 'LIKE', '%' . $search_term . '%')
                  ->orWhere('franchises.franchise_address', 'LIKE', '%' . $search_term . '%')
                  ->orWhere('franchises.franchise_owner', 'LIKE', '%' . $search_term . '%')
                  ;
            });

         // Apply business filter
            if ($business_id != 0) {
                $franchises->where('business_id', $business_id);
            } else {
                $franchises->whereIn('business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('common_products', 'Yes');
                });
            }

        $franchises = $franchises->get();

        return response()->json($franchises);
    }

   public function franchise(Request $request)
   {
        $defultFranchiseTerms = DB::table('business')->where('id_business',session('business_id'))->value('franchise_payment_terms') ?? '';  
        return view('franchise.franchise',compact('defultFranchiseTerms'));
   }

   public function franchisedata(Request $request)
    {
        $business_id = session('business_id');

        $franchises = Franchise::select(
            'id_franchises',
            'business.business_name',
            'franchises.business_id',
            'franchises.franchise_name as franchise_name',
            'franchises.franchise_ntn as franchise_ntn',
            'franchises.franchise_contact as franchise_contact',
            'franchises.franchise_email as franchise_email',
            'franchises.franchise_office_phone as franchise_office_phone',
            'franchises.franchise_address as franchise_address',
            'franchises.franchise_owner as franchise_owner',
            'franchises.franchise_status',
            'franchises.franchise_limit',
            'franchises.franchise_discount',
            'franchises.franchise_payment_terms'
        )
        ->join('business', 'business.id_business', '=', 'franchises.business_id')
        ->where('franchises.business_id', $business_id);

        return DataTables::of($franchises)
            ->addIndexColumn()

            // Action Column
            ->addColumn('action', function($row){
                if(!auth()->user()->can('edit-Franchise'))
                    return '';

                return '
                    <button class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-info open-btn" onclick="openEditModal(this)"
                        data-id="'.$row->id_franchises.'"
                        data-name="'.$row->franchise_name.'"
                        data-ntn="'.$row->franchise_ntn.'"
                        data-contact="'.$row->franchise_contact.'"
                        data-owner="'.$row->franchise_owner.'"
                        data-office="'.$row->franchise_office_phone.'"
                        data-email="'.$row->franchise_email.'"
                        data-address="'.$row->franchise_address.'"
                        data-discount="'.$row->franchise_discount.'"
                        data-limit="'.$row->franchise_limit.'"
                        data-franchise_payment_terms="'.htmlspecialchars($row->franchise_payment_terms, ENT_QUOTES).'"
                        data-bs-toggle="tooltip" title="Edit Franchise">
                        <iconify-icon icon="line-md:edit" class="fs-20" ></iconify-icon></button>
                    <button class="d-none" onclick="cancelFranchise('.$row->id_franchises.')"  class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-danger open-btn"><iconify-icon icon="solar:trash-bin-trash-bold-duotone" class="fs-20"></iconify-icon></button>
                    ';
            })

            ->rawColumns(['action'])  
            ->make(true);
    }


    public function save_franchise(Request $request)
    {
        try {
            DB::beginTransaction();

            $id = $request->id_franchises;

            $request->validate([
                'franchise_name' => 'required|string|max:255',
                'franchise_owner' => 'required|string|max:255',
                'franchise_email' => 'nullable|email',
            ]);

            $data = [
                'franchise_name' => $request->franchise_name,
                'franchise_address' => $request->franchise_address,
                'franchise_contact' => $request->franchise_contact,
                'franchise_ntn' => $request->franchise_ntn,
                'franchise_owner' => $request->franchise_owner,
                'franchise_office_phone' => $request->franchise_office_phone,
                'franchise_email' => $request->franchise_email,
                'franchise_discount' => $request->franchise_discount ?? 0,
                'franchise_limit' => $request->franchise_limit ?? 0,
                'franchise_payment_terms' => $request->franchise_payment_terms ?? '',
                'business_id' => session('business_id'),
            ];

            if ($id) {
                Franchise::where('id_franchises', $id)->update($data);
                DB::commit();
                return response()->json([
                    'success' => true,
                    'message' => "Franchise updated successfully!"
                ]);

            } else {
                $data['created_by'] = session('user_name');
                $data['created_on'] = now();
                Franchise::create($data);

                DB::commit();
                return response()->json([
                    'success' => true,
                    'message' => "Franchise added successfully!"
                ]);
            }

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
    // franchise_orders 
    public function franchise_orders(Request $request)
    {
        return view('franchise.franchise_order');
    }

    public function franchise_order_data(Request $request)
    {
        $franchiseid = $request->input('franchiseid', 'all');
        $dtype       = $request->input('dtype', 'order');
        $status      = $request->input('status', 'all');
        $startdate   = $request->input('startdate', date('Y-m-d'));
        $enddate     = $request->input('enddate', date('Y-m-d'));
        $businessId  = session('business_id'); 

        $accounts =  AccountsController::get_event_account('37', 'total_receivable');
        $account_head_id = $accounts->account_head_id;
        $sql = "
            SELECT 
                fo.id_franchise_orders,
                fo.franchise_id,
                fo.franchise_name,
                fo.franchise_ntn,
                fo.order_status,
                fo.created_by,
                DATE_FORMAT(fo.created_on, '%d-%m-%Y %h:%i:%s') AS created_on,
                DATE_FORMAT(fo.order_date, '%d-%m-%Y') AS order_date,
                DATE_FORMAT(fo.invoice_date, '%d-%m-%Y') AS invoice_date,
                fo.franchise_order_value,
                fo.invoice_number,
                fo.invoice_discount,
                fo.invoice_other_charges,
                fo.invoice_sales_tax,
                fo.product_tax,
                fo.invoice_gross_total,
                fo.invoice_total_receivable,
                f.franchise_name AS f_name,
                COALESCE(av_tot.debit, 0) AS debit,
                COALESCE(av_tot.credit, 0) AS credit,
                COALESCE(av_tot.balance, 0) AS balance
            FROM franchise_orders fo
            JOIN franchises f ON f.id_franchises = fo.franchise_id
            LEFT JOIN (
                SELECT 
                    av.franchise_order_id,
                    COALESCE(SUM(avd.debit),0) AS debit ,
                    COALESCE(SUM(avd.credit),0) AS credit ,
                    COALESCE(SUM(avd.debit - avd.credit),0) AS balance
                FROM account_voucher_detail avd
                JOIN account_vouchers av ON av.id_account_vouchers = avd.account_voucher_id
                WHERE avd.account_head_id = ? AND LOWER(voucher_status)= 'active' and av.franchise_order_id > 0
                GROUP BY av.franchise_order_id
            ) AS av_tot ON av_tot.franchise_order_id = fo.id_franchise_orders
            WHERE fo.business_id = ?
        ";

        $bindings = [$account_head_id, $businessId];

        if($startdate !== date('Y-m-d') || $enddate !== date('Y-m-d')) {
            if($dtype == 'invoice'){
                $sql .= " AND DATE(fo.invoice_date) BETWEEN ? AND ? ";
            } elseif($dtype == 'order'){
                $sql .= " AND DATE(fo.order_date) BETWEEN ? AND ? ";
            } else {
                $sql .= " AND DATE(fo.created_on) BETWEEN ? AND ? ";
            }
            $bindings[] = $startdate;
            $bindings[] = $enddate;
        }


        if($franchiseid !== 'All'){
            $sql .= " AND fo.franchise_id = ? ";
            $bindings[] = $franchiseid;
        }


        if($status !== 'All'){
            if($status == 'Invoiced'){
                $sql .= " AND fo.order_status IN ('Invoiced', 'Paid') ";
            } else {
                $sql .= " AND fo.order_status = ? ";
                $bindings[] = $status;
            }
        }
        $sql .= " GROUP BY fo.id_franchise_orders";

        $data = DB::select($sql, $bindings);

        return DataTables::of($data)
        ->addColumn('actions', function ($row) {
            $edit_url = route('franchise.create_franchise_order', [
                'selected_franchise' => $row->franchise_id,
                'order_id' => $row->id_franchise_orders
            ]);
            $invoice_url = route('franchise.franchise_order_invoice', [
                'order_id' => $row->id_franchise_orders
            ]); 
            $view_invoice_url = route('franchise.view_franchise_order_invoice', [
                'order_id' => $row->id_franchise_orders
            ]); 
            $payment_receiving_note_url = route('franchise.payment_receiving_note', [
                'order_id' => $row->id_franchise_orders
            ]); 
            $return_note_url = route('franchise.franchise_return_note', [
                'order_id' => $row->id_franchise_orders
            ]); 

            $buttons_html = '<div class="d-flex gap-1">';

            if ($row->order_status == "Open") {
                
                if(auth()->user()->can('edit-Franchise_Orders')){

                    $buttons_html .= '<a href="'.$edit_url.'" target="_blank" class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-info" data-bs-toggle="tooltip" title="Edit Order">
                            <iconify-icon icon="line-md:edit" class="fs-20"></iconify-icon>
                        </a>';
                }

                if(auth()->user()->can('delete-Franchise_Orders')){
                    
                    $buttons_html .= '<button onclick="cancelFranchiseOrder('.$row->id_franchise_orders.')" 
                    class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-danger" data-bs-toggle="tooltip" title="Cancel Order">
                    <iconify-icon icon="solar:trash-bin-trash-bold-duotone" class="fs-20"></iconify-icon>
                    </button>';
                }
                
                if(auth()->user()->can('create_invoice-Franchise_Orders')){
                    $buttons_html .= '<a href="'.$invoice_url.'" target="_blank" 
                    class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-info" data-bs-toggle="tooltip" title="Create Invoice">
                        <iconify-icon icon="material-symbols:receipt-long" class="fs-20"></iconify-icon>
                    </a>';
                }


            } elseif ($row->order_status == "Invoiced" ||  $row->order_status == "Paid") {
                $buttons_html .= '<a href="'.$view_invoice_url.'" target="_blank" class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-primary" data-bs-toggle="tooltip" title="View Invoice"><iconify-icon icon="line-md:watch-loop" class="fs-20"></iconify-icon></a>';
                $buttons_html .='<a href="'.$payment_receiving_note_url.'" target="_blank" class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-success" data-bs-toggle="tooltip" title="Receive Payment"><iconify-icon icon="ph:money-wavy-thin" class="fs-20"></iconify-icon></a>';
                // if($row->order_status == "Invoiced"){
                    $buttons_html .=' <a href="'.$return_note_url.'" target="_blank" class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-success" data-bs-toggle="tooltip" title="Return Note"><iconify-icon icon="ph:arrow-u-up-left" class="fs-20"></iconify-icon></a>';
                // }
                if($row->credit > 0 && $row->order_status == "Paid"){
                  $buttons_html .= '<button type="button" onclick ="view_franchise_order_invoice_payment('.$row->id_franchise_orders.' , this)" target="_blank" class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-secondary" data-bs-toggle="tooltip" title="Payment History"><iconify-icon icon="bi:clock-history" class="fs-20"></iconify-icon></button>';
                }
            }

        $buttons_html .= '</div>';
        return $buttons_html;
    })
     ->rawColumns(['actions'])
     ->make(true);
    }

    public function create_franchise_order(Request $request , FiltersService $filters ,$selected_franchise = 'all', $order_id = 0){

        $business_id  = session('business_id'); 
        if($selected_franchise > 0){
            $franchiseid = $request->selected_franchise ?? 'all';
        }else{
            $franchiseid = $selected_franchise;
        }
        $pageTitle ="Create Franchise Order";
        $order_date = $request->order_date ?? date('Y-m-d');
        
        if($order_id > 0){
            $orderid=$order_id;
            $pageTitle = "Update Franchise Order";
        }else{
            $orderid=$request->order_id ?? 0;
        }

           
        $franchise = DB::table('franchises')
        ->where('id_franchises', $franchiseid)
        ->where('franchise_status', 'Active')
        ->first();
        $selected_franchise = $franchiseid;
         $order=[];
     
        if($orderid>0){
             $order = DB::table('franchise_orders')
                ->select(
                    'franchise_orders.*','franchise_order_products.*',
                    DB::raw('DATE(franchise_orders.order_date) as order_date'),
                    DB::raw('DATE_FORMAT(invoice_date, "%d-%m-%Y") as formatted_invoice_date'),
                    DB::raw('DATE_FORMAT(DATE_ADD(invoice_date, INTERVAL 7 DAY), "%d-%m-%Y") as due_date')
                )
                ->join('franchise_order_products', 'franchise_order_products.franchise_order_id', '=', 'franchise_orders.id_franchise_orders')
                ->leftjoin('product_batch', 'product_batch.id_batch', '=', 'franchise_order_products.batch_id')
                ->join('business_products', 'business_products.id_business_products', '=', 'franchise_order_products.product_id')
                ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
                ->where('franchise_orders.id_franchise_orders', $orderid)
                ->get()
                    ->map(function($item) {
                    return (array) $item;
                })
                ->toArray();

                if (!$order) {
                  return redirect()->back()->with('error', "Order detail not found.");
                }

                if($order[0]['order_status'] =='Invoiced'){
                    return redirect()->back()->with('error', "An invoice has already been generated for this order ID ".$orderid);
                }else if($order[0]['order_status'] =='Cancelled'){
                 return redirect()->back()->with('error', "This order cannot be edited as it has already been canceled.");
                }   
        }

        $franchises =    $filters->getFranchises($business_id);
        $getStaff   =    $filters->getStaff($business_id);
        $months = DB::select("SELECT * from months"); 
        $currency = config('constants.CURRENCY');
        

        return view('franchise.create_franchise_order',compact('franchises','selected_franchise','franchise','selected_franchise','order','getStaff','currency','months','pageTitle','order_date'));
    }

    public function save_franchise_order(Request $request) 
    {
        $tableData = json_decode($request->orderdata, true);
        $orderid = $request->orderid ?? 0;
        DB::beginTransaction();

        try {
            if ($orderid > 0) {
                $order = FranchiseOrder::find($orderid);
                if (!$order) {
                    throw new Exception("Order not found!");
                }

                $order->update([
                    'order_date' => $request->order_date,
                    'franchise_id' => $request->franchiseid,
                    'franchise_name' => $request->franchisename,
                    'franchise_ntn' => $request->franchisentn,
                    'order_status' => $request->orderstatus,
                    'franchise_order_value' => $request->franchise_order_value,
                    'franchise_order_discount' => $request->franchise_order_discount,
                    'franchise_order_extra' => $request->order_extra,
                    'updated_by' => session('user_name'),
                ]);
                FranchiseOrderProduct::where('franchise_order_id', $orderid)->delete();
            } else {
            
                $order = FranchiseOrder::create([
                    'order_date' => $request->order_date,
                    'business_id' => session('business_id'),
                    'franchise_id' => $request->franchiseid,
                    'franchise_name' => $request->franchisename,
                    'franchise_ntn' => $request->franchisentn,
                    'order_status' => $request->orderstatus,
                    'franchise_order_value' => $request->franchise_order_value,
                    'franchise_order_discount' => $request->franchise_order_discount,
                    'created_by' => session('user_name'),
                    'franchise_order_extra' => $request->order_extra,
                ]);
            }

            foreach ($tableData as $row) {
                FranchiseOrderProduct::create([
                    'business_id' => session('business_id'),
                    'franchise_order_id' => $order->id_franchise_orders,
                    'product_id' => $row['productid'],
                    'product_name' => $row['productname'],
                    'product_detail' => $row['productdetail'],
                    'category' => $row['category'],
                    'staff_id' => $row['staffid'],
                    'staff_name' => $row['staff'],
                    'qty' => $row['qty'],
                    'stockfrom' => $row['stockfrom'],
                    'batch' => null,
                    'batch_id' => null,
                    'unit_price' => $row['unit_price'],
                    'unit_discount' => $row['unit_discount'],
                    'discounted_price' => $row['discounted_price'],
                    'order_month' => $row['month'] ?? null,
                    'order_year' => $row['year'] ?? null,
                ]);
            }

            DB::commit();

            $invoice_url = route('franchise.franchise_order_invoice', [
                'order_id' => $order->id_franchise_orders
            ]);

            return response()->json([
                'success' => true,
                'message_type' => 'success',
                'message' => $orderid > 0
                    ? 'Franchise order has been successfully updated. Order ID: ' . $order->id_franchise_orders
                    : 'Franchise order has been successfully created. Order ID: ' . $order->id_franchise_orders,
                'message_btn' => 'btn btn-success',
                'order_id' => $order->id_franchise_orders,
                'invoice_url' => $invoice_url,
            ]);

        } catch (Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message_type' => 'error',
                'message' => 'Order could not be saved. Transaction rolled back. ' . $e->getMessage(),
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

    function cancel_order(Request $request){
          try {
                    DB::beginTransaction();

                    DB::table('franchise_orders')
                    ->where('id_franchise_orders', $request->id_franchise_orders)
                    ->update([
                        'order_status' => 'Cancelled',
                    ]);
                    DB::table('account_vouchers')
                        ->where('franchise_order_id', $request->id_franchise_orders)
                        ->update([
                            'voucher_status' => 'Cancelled',
                            'cancelled_by'   => session('user_name'),
                            'cancelled_on'   => now(),
                    ]);

                   DB::commit();
                return response()->json([
                    'success' => true,
                    'message_type' => 'success',
                    'message' => 'Order cancelled successfully',
                    'message_btn' => 'btn btn-success',
                    'order_id' => $request->id_franchise_orders,
                ]);

                } catch (\Exception $e) {
                    DB::rollBack();

                    return response()->json([
                        'success' => false,
                        'message_type' => 'error',
                        'message' => 'Failed to cancel order: ' . $e->getMessage(),
                        'message_btn' => 'btn btn-danger',
                    ]);
                }
    }
    
    public function franchise_order_invoice(Request $request, $order_id){

        $business = Business::where("id_business", session('business_id'))->first();

        $order_detail = FranchiseOrder::where('id_franchise_orders', $order_id)
                ->where('order_status','!=', 'Cancelled')
                ->select( 'franchise_orders.*',DB::raw('DATE(franchise_orders.order_date) as order_date'))
                ->first();
        if(!$order_detail){ 
         return redirect()->back()->with('error', "Order detail not found.");
        }

        if($order_detail->order_status =='Invoiced'){
          return redirect()->back()->with('error', "An invoice has already been generated for this order.");
        }

        $franchise = Franchise::where('id_franchises', $order_detail->franchise_id)->first();
        if(!$franchise){
            return "Franchise detail not found.";   
        }

        $franchiseid =  $franchise->id_franchises;
        $franchiseinvoicenumber = DB::table('franchise_orders')
            ->where('franchise_id', $franchiseid)
            ->select(DB::raw('COALESCE(MAX(invoice_number), 0) + 1 AS invoice_number'))
            ->first();

        $arrears = $this->get_receivable_amount($franchiseid, $order_id);

        $order = DB::table('franchise_orders')
                ->select(
                    'franchise_orders.*',
                    'franchise_order_products.id_franchise_order_products',
                    'franchise_order_products.franchise_order_id',
                    'franchise_order_products.order_month',
                    'franchise_order_products.order_year',
                    'franchise_order_products.product_name',
                    'franchise_order_products.category',
                    'franchise_order_products.product_detail',
                    'franchise_order_products.batch_id',
                    'franchise_order_products.batch',
                    'product_batch.batch_amount',
                    'franchise_order_products.staff_id',
                    'franchise_order_products.staff_name',
                    'business_products.product_sales_tax',
                    'business_brands.business_brand_name as business_brand_name',
                    'business_products.product_type',
                    DB::raw('franchise_order_products.product_id as product_id'),
                    DB::raw('SUM(franchise_order_products.qty) as qty'),
                    DB::raw('SUM(franchise_order_products.qty * franchise_order_products.unit_price) as total_amount'),
                    DB::raw('
                        CASE 
                            WHEN SUM(franchise_order_products.qty) > 0 
                            THEN ROUND( SUM(franchise_order_products.qty * franchise_order_products.unit_price) 
                                / SUM(franchise_order_products.qty) ,2)
                            ELSE 0
                        END AS unit_price
                    '),
                    DB::raw('SUM(franchise_order_products.discounted_price) as discounted_price'),
                    DB::raw('SUM(franchise_order_products.product_tax) as product_tax'),
                    DB::raw('(franchise_order_products.unit_discount) as unit_discount'),
                    DB::raw('DATE(franchise_orders.order_date) as order_date'),
                    DB::raw('DATE_FORMAT(invoice_date, "%d-%m-%Y") as formatted_invoice_date'),
                    DB::raw('DATE_FORMAT(DATE_ADD(invoice_date, INTERVAL 7 DAY), "%d-%m-%Y") as due_date')
                )
                ->join('franchise_order_products', 'franchise_order_products.franchise_order_id', '=', 'franchise_orders.id_franchise_orders')
                ->leftjoin('product_batch', 'product_batch.id_batch', '=', 'franchise_order_products.batch_id')
                ->join('business_products', 'business_products.id_business_products', '=', 'franchise_order_products.product_id')
                ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
                ->where('franchise_orders.id_franchise_orders', $order_id)
                ->groupBy(
                        'franchise_orders.id_franchise_orders',
                        'franchise_order_products.product_id'
                    )
                ->get()
                    ->map(function($item) {
                    return (array) $item;
                })
                ->toArray(); 

        $stores = Stores::select('business_stores.*',
            'business.business_name'
        )
        ->join('business', 'business_stores.business_id', '=', 'business.id_business')
        ->where('business_id', session('business_id'))->get();

        return view('franchise.franchise_order_invoice_creation',compact('business','franchise','franchiseinvoicenumber','arrears','order','order_detail','stores','order_id'));        
    }

    function get_receivable_amount($franchiseid, $order_id=null){
        $sql =" SELECT 
                    account_vouchers.business_partner_id, 
                    IFNULL(SUM(debit) - SUM(credit), 0) AS balance,
                    IFNULL(SUM(debit), 0) AS debit,
                    IFNULL(SUM(credit), 0) AS credit
                FROM 
                    account_vouchers
                JOIN 
                    account_voucher_detail 
                    ON account_voucher_detail.account_voucher_id = account_vouchers.id_account_vouchers
                JOIN 
                    account_event_mapping 
                    ON account_event_mapping.account_head_id = account_voucher_detail.account_head_id
                WHERE  
                    entity_name = 'received_amount' AND
                    account_event_id = 37 AND
                    account_vouchers.voucher_status = 'Active' AND
                    account_vouchers.business_partner_id = " . $franchiseid . " AND
                    account_vouchers.business_partner = 4 AND
                    franchise_order_id > 0 AND
                    franchise_order_id != " . $order_id . " AND
                    account_event_mapping.entity_name IN ('total_receivable', 'received_amount')
                GROUP BY 
                    business_partner_id;";
        $result = DB::select($sql);

        return $result[0] ?? null; 
    }


    public function save_franchise_invoice(Request $request)
    {
        DB::beginTransaction();

        try {
            $tableData = json_decode($request->tabledata, true);
            if (!is_array($tableData)) {
                return response()->json([
                    "message" => "Invalid table data",
                    "message_type" => "error",
                    "success" => false
                ]);
            }


            $order_detail = FranchiseOrder::where('id_franchise_orders', $request->orderid)
                    ->where('order_status','!=', 'Cancelled')
                    ->select( 'franchise_orders.*',DB::raw('DATE(franchise_orders.order_date) as order_date'))
                    ->first();
            if($order_detail->order_status =='Invoiced'){
                  return response()->json([
                    "message" => "An invoice has already been generated for this order.",
                    "message_type" => "error",
                    "success" => false
                ]);
            }


            $business_id = session('business_id');
            $orgsubtotal = $request->orgsubtotal;
            $discount= $request->discount;
            $discount_perc= $request->discount_perc;
            $product_discount = $request->product_discount;
            $other_charges = $request->other_charges;
            $grosstotal = $request->grosstotal;
            $tax_in_perc = $request->tax_in_perc;
            $salestax = $request->salestax;
            $othertax = $request->othertax;
            $totalpayable = $request->totalpayable;
            $store_id = $request->store_id; 
            $orderid = $request->orderid;  
            $remarks = $request->remarks;  
            $invoicedate = $request->invoicedate;   




            $franchiseinvoicenumber = DB::table('franchise_orders')
            ->where('franchise_id', $request->franchise_id)
            ->select(DB::raw('COALESCE(MAX(invoice_number), 0) + 1 AS invoice_number'))
            ->first();

            // $invoicenumber = $request->invoicenumber;  
            $invoicenumber = $franchiseinvoicenumber->invoice_number ?? 0;  

            $batchService = new productBatchService();
            $batchesData = [];
            $shortageList = [];  

            foreach ($tableData as $row) {
                $productId   = $row['productid'];
                $requiredQty = floatval($row['qty']);

                $batches = $batchService->findSufficientBatches($productId, 0, $store_id);
                $batches = json_decode($batches, true);

                $availableQty = 0;
                if (!empty($batches)) {
                    foreach ($batches as $b) {                        
                        $availableQty += floatval($b['instock']);
                    }
                }

                if ($availableQty < $requiredQty) { 
                    $product_batch_route = route('products.batches', $row['productid']);
                    $shortageList[] = [
                        "product_batch_route"     => $product_batch_route,
                        "productid"     => $row['productid'],
                        "productname"   => $row['productname'],
                        "required_qty"  => $requiredQty,
                        "available_qty" => $availableQty,
                        "shortage"      => $requiredQty - $availableQty
                    ];
                    continue;
                }

                $remainingQty = $requiredQty;
                foreach ($batches as $batch) {
                    if ($remainingQty <= 0) break;
                    $instock = floatval($batch['instock']);
                    if ($instock <= 0) continue;
                    
                    $deductQty = min($instock, $remainingQty , $requiredQty); // third parameter for capacity
                    $amount_with_qty = $row['unit_price'] * $deductQty;
                    $unit_discount = ($row['unit_price'] * $row['discount_perc']) / 100;
                    $total_discount_amount = ($amount_with_qty * $row['discount_perc']) / 100;
                    $discounted_price = $amount_with_qty - $total_discount_amount;
                    $tax_amount = ($discounted_price * $row['tax_perc']) / 100;
                    $final_amount = $discounted_price + $tax_amount;

                    $batchesData[] = [
                        "id_franchise_order_products" => $row['id_franchise_order_products'],
                        "seq" => $row['seq'],
                        "productid" => $row['productid'],
                        "brandname" => $row['brandname'],
                        "productname" => $row['productname'],
                        "product_detail" => $row['product_detail'],
                        "category" => $row['category'],
                        "batch_id" => $batch['batch_id'],
                        "batch" => $batch['batch_no'],
                        "batch_amount" => $batch['batch_amount'],
                        "staff_name" => $row['staff_name'],
                        "staff_id" => $row['staff_id'],
                        "unit_price" => $row['unit_price'],
                        "qty" => $deductQty,
                        "product_price" => $amount_with_qty,
                        "product_discount" => $total_discount_amount,
                        "unit_discount" => $unit_discount,
                        "discount_perc" => $row['discount_perc'],
                        "discounted_price" => $discounted_price,
                        "tax_perc" => $row['tax_perc'],
                        "product_tax" => $tax_amount,
                        "final_price" => $final_amount,
                        "year" => $row['year'],
                        "month" => $row['month'],
                    ];

                    $remainingQty -= $deductQty;
                }
            }

            if (!empty($shortageList)) {
                DB::rollBack();
                return response()->json([
                    "message" => "Insufficient stock for products.",
                    "message_type" => "error",
                    "data" => $shortageList ,
                    "success" => false 
                ]);
            }

            FranchiseOrderProduct::where('franchise_order_id', $orderid)->delete();
            $batch_amount = 0;
            foreach ($batchesData as $data) {
                FranchiseOrderProduct::create([
                    'business_id' => session('business_id'),
                    'franchise_order_id' => $orderid,
                    'product_id' => $data['productid'],
                    'product_name' => $data['productname'],
                    'product_detail' => $data['product_detail'],
                    'category' => $data['category'],
                    'staff_id' => $data['staff_id'],
                    'staff_name' => $data['staff_name'],
                    'qty' => $data['qty'],
                    'stockfrom' => 'in_stock',
                    'batch' => $data['batch'],
                    'batch_id' => $data['batch_id'],
                    'unit_price' => $data['unit_price'],
                    'unit_discount' => $data['unit_discount'],
                    'discounted_price' => $data['discounted_price'],
                    'order_month' => $data['month'],
                    'order_year' => $data['year'],
                    'product_tax' => $data['product_tax'],
                ]);

                if ($data['qty'] > 0) {
                    $batch_amount += $data['qty'] * ($data['batch_amount'] > 0 ? $data['batch_amount'] : 0);
                }
            }


            $FranchiseOrder = FranchiseOrder::find($orderid);
            $FranchiseOrder->update(attributes: [
                'invoice_discount' => $discount,
                'product_discount' => $product_discount,
                'invoice_discount_perc' => $discount_perc,
                'invoice_other_charges' => $other_charges,
                'invoice_gross_total' => $grosstotal,
                'invoice_sales_tax' => $othertax,
                'invoice_sales_tax_perc' => $tax_in_perc,
                'product_tax' => $salestax,
                'invoice_total_receivable' => $totalpayable,
                'order_status' => 'Invoiced',
                'invoice_remarks' => $remarks,
                'invoice_number' => $invoicenumber,
                'invoice_date' => $invoicedate
            ]);


            // Franchise Order Invoice  VOUCHER  START
            if(session('ho_accounts')=='Yes'){
                $business_id_for_accounts_head = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $business_id_for_accounts_head = $business_id;
            }

            $event_mappings = AccountEventMapping::where('account_event_id', 37)->where('business_id', $business_id_for_accounts_head)->get();
            $debit_accounts=[];
            $credit_accounts=[];

            foreach($event_mappings as $ah){
                if($ah['transaction_type']=='debit'){
                   array_push($debit_accounts,$ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                } else {
                    array_push($credit_accounts, $ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                }
            }

            $data = [
                'business_id' => $business_id,
                'description' => 'Franchise Sale Order Voucher on invoicing for Order #' . $orderid . 
                                ', Invoice # ' . $invoicenumber . 
                                ' on ' . date('d-m-Y') . 
                                ', for ' . $request->franchise_name . 
                                ', customer ID : ' . $request->franchise_id . 
                                ' with comments : ' . $remarks,
                'voucher_date' => $invoicedate,
                'voucher_amount' => $request->totalpayable,
                'voucher_status' => 'Active',
                'created_by' => session('user_name'), 
                'cost_center' => '1',
                'cost_center_name' => 'Back Office',
                'business_partner' => 4,
                'business_partner_id' => $request->franchise_id,
                'business_partner_name' => $request->franchise_name,
                'voucher_type' => 3,
                'payment_mode' => 'Cash',
                'bank_name' => '',
                'franchise_order_id' => $orderid,
                'auto_voucher' => 'Yes',
                'sale_type' => 'franchise'
            ];

            $accountVoucher = AccountVouchers::create($data);
            $account_voucherid = $accountVoucher->id_account_vouchers; 

            // enter debit entries
            foreach ($debit_accounts as $debit_account) {

                $debitamount = 0;
                $debit = explode("|", $debit_account);

                // check entity
                if ($debit[2] == "total_receivable" && $request->totalpayable > 0) {
                    $debitamount = $request->totalpayable;
                }

                if ($debit[2] == "invoice_discount" && ($request->discount + $request->product_discount) > 0) {
                    $debitamount = $request->discount + $request->product_discount;
                }

                if ($debitamount > 0) {
                    AccountVoucherDetail::create([
                        'account_voucher_id' => $account_voucherid,
                        'account_head_id'    => $debit[0],
                        'debit'              => round($debitamount, 2),
                        'credit'             => 0,
                    ]);
                }
            }


            // enter credit entries
            foreach ($credit_accounts as $credit_account) {

                $creditamount = 0;
                $credit = explode("|", $credit_account);

                // check entity
                if ($credit[2] == "total_invoice_value") {
                    $creditamount = $request->orgsubtotal;
                }

                if ($credit[2] == "other_charges") {
                    $creditamount = $request->other_charges;
                }

                if ($credit[2] == "sales_tax") {
                    $creditamount = $request->salestax + $request->othertax;
                }

                if ($creditamount > 0) {
                    AccountVoucherDetail::create([
                        'account_voucher_id' => $account_voucherid,
                        'account_head_id'    => $credit[0],
                        'debit'              => 0,
                        'credit'             => round($creditamount, 2),
                    ]);
                }
            }
            // Franchise Order Invoice  VOUCHER  END

            // Creating Cost of goods Voucher/
            
            $event_mappings = AccountEventMapping::where('account_event_id', operator: 23)->where('business_id', $business_id_for_accounts_head)->get();
            $debit_accounts=[];
            $credit_accounts=[];

            foreach($event_mappings as $ah){
                if($ah['transaction_type']=='debit'){
                   array_push($debit_accounts,$ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                } else {
                    array_push($credit_accounts, $ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                }
            } 

            $data = array( 
                'business_id' => $business_id,
                'description' => 'Cost of Goods Solds Voucher against reference voucher ID '. $account_voucherid . ' Franchise invoice ' . $orderid . ' on '. date('d-m-Y').', for '.$request->franchise_name.' Franchise ID : '. $request->franchise_id ,
                'voucher_date' => $invoicedate,
                'voucher_amount' => $batch_amount,
                'voucher_status' => 'Active',
                'created_by' =>  session('user_name'),
                'cost_center'=> '1',
                'cost_center_name'=> 'Front Desk',
                'business_partner'=> 1,
                'business_partner_id' => $request->franchise_id,
                'business_partner_name' => $request->franchise_name,
                'voucher_type'=>3,
                'payment_mode'=>'Cash',
                'bank_name'=>'',
                'franchise_order_id' => $orderid,
                'auto_voucher'=>'Yes'
                //'sale_type'=>'sale'
            ); 

            $accountVoucher = AccountVouchers::create($data);
            $cogs_voucherid = $accountVoucher->id_account_vouchers; 

            // enter debit entries
            foreach ($debit_accounts as $debit_account) {

                $debitamount = 0;
                $debit = explode("|", $debit_account);

                // check entity
                if ($debit[2] == "cogs") {
                    $debitamount = $batch_amount;
                }

                if ($debitamount > 0) {
                    AccountVoucherDetail::create([
                        'account_voucher_id' => $cogs_voucherid,
                        'account_head_id'    => $debit[0],
                        'debit'              => round($debitamount, 2),
                        'credit'             => 0,
                    ]);
                }
            }

            // enter credit entries
            foreach ($credit_accounts as $credit_account) {

                $creditamount = 0;
                $credit = explode("|", $credit_account);

                // check entity
                if ($credit[2] == "inventory") {
                    $creditamount = $batch_amount;
                }

                if ($creditamount > 0) {
                    AccountVoucherDetail::create(attributes: [
                        'account_voucher_id' => $cogs_voucherid,
                        'account_head_id'    => $credit[0],
                        'debit'              => 0,
                        'credit'             => round($creditamount, 2),
                    ]);
                }
            }

            $print_view_route = route('franchise.view_franchise_order_invoice', $orderid);

            DB::commit();
            return response()->json([
                "message" => "Invoice processed successfully.",
                "message_type" => "success",
                "batches" => $batchesData,
                "success" => true,
                "franchise_print_url" => $print_view_route,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                "message" => "Something went wrong: " . $e->getMessage(),
                "message_type" => "error",
                "success" => false
            ]);
        }
    }

    public function view_franchise_order_invoice(Request $request , $order_id){

       $business = Business::where("id_business", session('business_id'))->first();

        $order_detail = FranchiseOrder::where('id_franchise_orders', $order_id)
                ->where('order_status','!=', 'Cancelled')
                ->select( 'franchise_orders.*',DB::raw('DATE(franchise_orders.order_date) as order_date'))
                ->first();
        if(!$order_detail){
          return redirect()->back()->with('error', "Order detail not found.");
        }  

        $franchise = Franchise::where('id_franchises', $order_detail->franchise_id)->first();

        $franchiseid =  $franchise->id_franchises;
        $franchiseinvoicenumber = DB::table('franchise_orders')
            ->where('franchise_id', $franchiseid)
            ->select(DB::raw('COALESCE(MAX(invoice_number), 0) + 1 AS invoice_number'))
            ->first();

        $arrears = $this->get_receivable_amount($franchiseid, $order_id);
        
        $order = DB::table('franchise_orders')
                ->select(
                    'franchise_orders.*','franchise_order_products.*','business_products.*','business_brands.*','product_batch.*',
                    DB::raw('DATE(franchise_orders.order_date) as order_date'),
                    DB::raw('DATE_FORMAT(franchise_orders.invoice_date, "%d-%m-%Y") as formatted_invoice_date'),
                    DB::raw('DATE_FORMAT(DATE_ADD(franchise_orders.invoice_date, INTERVAL 7 DAY), "%d-%m-%Y") as due_date'),
                    DB::raw('franchise_order_products.product_id as product_id')
                )
                ->join('franchise_order_products', 'franchise_order_products.franchise_order_id', '=', 'franchise_orders.id_franchise_orders')
                ->leftjoin('product_batch', 'product_batch.id_batch', '=', 'franchise_order_products.batch_id')
                ->join('business_products', 'business_products.id_business_products', '=', 'franchise_order_products.product_id')
                ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
                ->where('franchise_orders.id_franchise_orders', $order_id)
                ->get()
                    ->map(function($item) {
                    return (array) $item;
                })
                ->toArray(); 

        return view('franchise.franchise_order_invoice_print_view',compact('business','franchise','franchiseinvoicenumber','arrears','order','order_detail','order_id'));  
    }

    public function cancel_order_invoice(Request $request)
    {
        $franchiseorderid = $request->id_franchise_orders;
        
        // Fetch order status
        $order_detail = FranchiseOrder::where('id_franchise_orders', $franchiseorderid)
            ->select('order_status')
            ->first();

        if (!$order_detail) {
            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Failed to get invoice details.',
                'message_btn'   => 'btn btn-danger',
            ]);
        }

        if ($order_detail->order_status !== 'Invoiced') {
            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'This order is not invoiced yet.',
                'message_btn'   => 'btn btn-danger',
            ]);
        } 


        $accounts =  AccountsController::get_event_account('37', 'total_receivable');
        if(!$accounts){ 
            return redirect()->back()->with('error', "Account mapping not found.");
        }
        $account_head_id = $accounts->account_head_id;

        $sql = "
            SELECT 
                SUM(IFNULL(debit,0)) AS total_receivable,
                SUM(IFNULL(credit,0)) AS total_received,
                SUM(IFNULL(debit,0)) - SUM(IFNULL(credit,0)) AS balance
            FROM account_voucher_detail
            JOIN account_vouchers 
                ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            WHERE 
                account_vouchers.voucher_status = 'Active'
                AND account_vouchers.franchise_order_id = {$franchiseorderid}
                AND account_voucher_detail.account_head_id = {$account_head_id}
        ";

        $result = DB::select($sql);
        $total_receivable = $result[0]->total_receivable ?? 0;
        $total_received   = $result[0]->total_received ?? 0;
        $balance          = $result[0]->balance ?? 0;

        if ($total_received > 0) {
            return response()->json([
                'success'      => false,
                'message_type' => 'warning',
                'message'      => 'Invoice cannot be cancelled because some amount has already been received against this invoice, either through return notes or payment receipts.',
                'message_btn'  => 'btn btn-warning',
            ]);
        }


        try {
            DB::beginTransaction();

            // Reset the franchise order invoice fields
            FranchiseOrder::where('id_franchise_orders', $franchiseorderid)->update([
                'order_status'              => 'Open',
                'invoice_discount'          => 0,
                'invoice_discount_perc'     => 0,
                'invoice_other_charges'     => 0,
                'invoice_gross_total'       => 0,
                'invoice_sales_tax'         => 0,
                'invoice_sales_tax_perc'    => 0,
                'invoice_total_receivable'  => 0,
                'invoice_remarks'           => null,
                'invoice_date'              => null,
                'invoice_number'            => null,
            ]);

            // Cancel the related account voucher
            AccountVouchers::where('franchise_order_id', $franchiseorderid)->update([
                'voucher_status' => 'cancelled',
                'cancelled_by'   => session('user_name'),
                'cancelled_on'   => now(),
            ]);

            // Reset franchise order product batches
            DB::table('franchise_order_products')
                ->where('franchise_order_id', $franchiseorderid)
                ->update([
                    'batch'        => null,
                    'batch_id'     => null,
                    'product_tax'  => null,
                ]);

            DB::commit();

        $franchise_orders_route = route('franchise.franchise_orders');

            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Invoice cancelled successfully.',
                'message_btn'   => 'btn btn-success',
                'order_id'      => $franchiseorderid,
                'franchise_orders_route'      => $franchise_orders_route,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Failed to cancel invoice: ' . $e->getMessage(),
                'message_btn'   => 'btn btn-danger',
            ]);
        }
    }

    public function payment_receiving_note(Request $request , $order_id){
        $business = Business::where("id_business", session('business_id'))->first();
        $business_id =session('business_id');
        $order_detail = FranchiseOrder::where('id_franchise_orders', $order_id)
                ->where('order_status','!=', 'Cancelled')
                ->select( 'franchise_orders.*',DB::raw('DATE(franchise_orders.order_date) as order_date'))
                ->first();
        if(!$order_detail){ 
         return redirect()->back()->with('error', "Order detail not found.");
        }

        $total_tax_on_invoice = (($order_detail->invoice_sales_tax ?? 0 ) + ($order_detail->product_tax ?? 0 ));

        $franchise = Franchise::where('id_franchises', $order_detail->franchise_id)->first(); 
        $accounts =  AccountsController::get_event_account('37', 'total_receivable');
        if(!$accounts){ 
            return redirect()->back()->with('error', "Account mapping not found.");
        }
        $account_head_id = $accounts->account_head_id;

        $sql = "
            SELECT 
                SUM(IFNULL(debit,0)) AS total_receivable,
                SUM(IFNULL(credit,0)) AS total_received,
                SUM(IFNULL(debit,0)) - SUM(IFNULL(credit,0)) AS balance
            FROM account_voucher_detail
            JOIN account_vouchers 
                ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
            WHERE 
                account_vouchers.voucher_status = 'Active'
                AND account_vouchers.franchise_order_id = {$order_id}
                AND account_voucher_detail.account_head_id = {$account_head_id}
        ";

        $result = DB::select($sql);
        $total_receivable = $result[0]->total_receivable ?? 0;
        $total_received   = $result[0]->total_received ?? 0;
        $balance          = $result[0]->balance ?? 0;

        $cost_centers = DB::table('account_cost_center')->get();
        $franchise_pending_payments = $this->pending_payment_of_franchise($business_id, $order_detail->franchise_id);



        $advance_accounts =  AccountsController::get_event_account('37', 'increase_advance_liability');
        if(!$advance_accounts){ 
            return redirect()->back()->with('error', "Advance Account mapping not found.");
        }
        $advance_account_head_id = $advance_accounts->account_head_id;
        $franchise_over_all_advance_balance = DB::select("SELECT
                                                    account_vouchers.business_partner_id,
                                                    SUM(credit) as total_advance,
                                                    SUM(debit) as total_adjustment,
                                                    SUM(credit) - SUM(debit) AS balance
                                                FROM
                                                    account_vouchers   
                                                JOIN account_voucher_detail ON account_voucher_detail.account_voucher_id = account_vouchers.id_account_vouchers
                                                WHERE
                                                    account_voucher_detail.account_head_id = $advance_account_head_id 
                                                    AND account_vouchers.voucher_status = 'Active' 
                                                    AND account_vouchers.business_id = $business_id
                                                    AND account_vouchers.business_partner = 4
                                                    AND account_vouchers.business_partner_id =$order_detail->franchise_id
                                                GROUP BY
                                                    account_vouchers.business_partner_id;");


                                                        

        return view('franchise.payment_receiving_note',compact('business','order_detail','franchise','total_receivable','total_received','balance','cost_centers','franchise_pending_payments','franchise_over_all_advance_balance','total_tax_on_invoice'));
    }

    public function save_invoice_payment(Request $request){

        $business_id =session('business_id');

        $franchise_id =    $request->franchise_id;
        $franchise_order_id =    $request->franchise_order_id;
        $invoice_number =    $request->invoice_number;
        $franchise_name =    $request->franchise_name;
        $payment_date =    $request->payment_date;
        $payment_mode =    $request->payment_mode;
        $account_head_id =    $request->account_head_id;
        $instrument_number =    $request->instrument_number ?? '';
        $cost_center_type =    $request->cost_center_type;
        $cost_center =    $request->cost_center;
        $receiving_now =    $request->receiving_now;
        $withholding_tax_percentage =    $request->withholding_tax_percentage;
        $withholding_tax =    $request->withholding_tax;
        $notes =    $request->notes;
        $receiving_after_withholding =    $request->receiving_after_withholding;  
        $excess =    $request->excess ?? 0;  
        $useAvaibaleAdvance =    $request->useAvaibaleAdvance ?? 0;  
        $receiving_after_advance_adjustment =    $request->receiving_after_advance_adjustment ?? 0;   
        $withholding_sales_tax =    $request->withholding_sales_tax ?? 0;   


        $advance_accounts =  AccountsController::get_event_account('37', 'increase_advance_liability');
                if(!$advance_accounts){ 
                    return redirect()->back()->with('error', "Advance Account mapping not found.");
                }
        $advance_account_head_id = $advance_accounts->account_head_id;
        $franchise_over_all_advance_balance = DB::select("SELECT
                                            account_vouchers.business_partner_id,
                                            SUM(credit) as total_advance,
                                            SUM(debit) as total_adjustment,
                                            SUM(credit) - SUM(debit) AS balance
                                        FROM
                                            account_vouchers   
                                        JOIN account_voucher_detail ON account_voucher_detail.account_voucher_id = account_vouchers.id_account_vouchers
                                        WHERE
                                            account_voucher_detail.account_head_id = $advance_account_head_id 
                                            AND account_vouchers.voucher_status = 'Active' 
                                            AND account_vouchers.business_id = $business_id
                                            AND account_vouchers.business_partner = 4
                                            AND account_vouchers.business_partner_id =$franchise_id
                                        GROUP BY
                                            account_vouchers.business_partner_id;");

        $total_avaiable_balance = $franchise_over_all_advance_balance[0]->balance ?? 0;  

        if($useAvaibaleAdvance > $total_avaiable_balance){
           return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Your available advance balance is '.$total_avaiable_balance.' but you are trying to use '.$useAvaibaleAdvance.', which exceeds the available limit. I have updated your advance fields with the available balance. Please review and proceed again.',
                'message_btn'   => 'btn btn-danger',
                'total_avaiable_balance' => $total_avaiable_balance
            ]);
        }

        DB::beginTransaction();
        try {
            // Create account voucher
            $voucherData = [
                'business_id'           => session('business_id'),
                'description'           => $notes,
                'voucher_date'          => $payment_date,
                'voucher_amount'        => $receiving_now,
                'voucher_status'        => 'Active',
                'created_by'            => session('user_name'),
                'cost_center'           => $cost_center_type,
                'cost_center_name'      => $cost_center,
                'business_partner'      => 4,
                'business_partner_id'   => $franchise_id,
                'business_partner_name' => $franchise_name,
                'voucher_type'          => 2,
                'payment_mode'          => $payment_mode,
                'franchise_order_id'    => $franchise_order_id,
                'auto_voucher'          => 'Yes',
                'sale_type'             => 'franchise'
            ];

            $account_voucherid = DB::table('account_vouchers')->insertGetId($voucherData);

            // Get mapped accounts
            if(session('ho_accounts')=='Yes'){
                $business_id_for_accounts_head = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $business_id_for_accounts_head = session('business_id'); 
            } 


            $event_mappings = AccountEventMapping::where('account_event_id', 37)->where('business_id', $business_id)->get();
            $debit_accounts = [];
            $credit_accounts = [];

            foreach($event_mappings as $ah){
                if($ah['transaction_type']=='debit'){
                    array_push($debit_accounts,$ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                } else {
                    array_push($credit_accounts, $ah['account_head_id']."|".$ah['transaction_type']."|".$ah['entity_name']);
                }
            }  

            // Debit fixed
            DB::table('account_voucher_detail')->insert([
                'account_voucher_id' => $account_voucherid,
                'account_head_id'    => $account_head_id,
                'debit'              => $receiving_after_withholding,
                'credit'             => 0,
                'cost_center_type'   => $cost_center_type,
                'cost_center'        => $cost_center,
                'instrument_number'  => $instrument_number,
                'payment_mode'       => $request->voucher_payment_mode
            ]);

            // Debit entries
            foreach ($debit_accounts as $acc) {

                $exp = explode("|", $acc);
                $id = $exp[0];
                $entity = $exp[2];

                $debitamount = 0;
                $detail_remarks ='';
                if ($entity == "withholding" && $withholding_tax > 0) {
                    $debitamount = $withholding_tax;
                }
                if ($entity == "sale_tax_withholding" && $withholding_sales_tax > 0) {
                    $debitamount = $withholding_sales_tax;
                }
                if ($entity == "decrease_advance_liability" && $useAvaibaleAdvance > 0) {
                    $debitamount = $useAvaibaleAdvance;
                }

                if ($debitamount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $account_voucherid,
                        'account_head_id'    => $id,
                        'debit'              => round($debitamount, 2),
                        'credit'             => 0,
                        'instrument_number'  => $instrument_number
                    ]);
                }
            }

            // Credit entries
            foreach ($credit_accounts as $acc) {

                $exp = explode("|", $acc);
                $id = $exp[0];
                $entity = $exp[2];
                $creditamount = 0;

                if ($entity == "received_amount" && $receiving_after_withholding > 0) {
                    $creditamount = $receiving_now - $excess;
                }
                if ($entity == "increase_advance_liability" && $excess > 0) {
                    $creditamount = $excess;
                }

                if ($creditamount > 0) {
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $account_voucherid,
                        'account_head_id'    => $id,
                        'debit'              => 0,
                        'credit'             => round($creditamount, 2),
                        'instrument_number'  => $instrument_number
                    ]);
                }
            }

            DB::table('franchise_orders')
            ->where('id_franchise_orders', $franchise_order_id)
            ->update(['order_status' => 'Paid']);

            DB::commit();
            $franchise_orders_route = route('franchise.franchise_orders');
            $account_voucher_print_url = route('accounts.account_voucher_print', ['id' => $account_voucherid]); 
            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Invoice Payment Received successfully Voucher ID is '.$account_voucherid,
                'message_btn'   => 'btn btn-success',
                'account_voucherid'   => $account_voucherid,
                'order_id'      => $franchise_id,
                'franchise_orders_route'      => $franchise_orders_route,
                'account_voucher_print_url' =>$account_voucher_print_url
            ]);

        } catch (\Exception $e) {

            DB::rollBack();
            return response()->json([
            'success'       => false,
            'message_type'  => 'error',
            'message'       => 'Failed Received Invoice Payment : ' . $e->getMessage(),
            'message_btn'   => 'btn btn-danger',
         ]);
       }
    }
    
    public function invoice_payment_history(Request $request)
    {
        $order_id = $request->id_franchise_orders;
        // Get mapped account
        $accounts = AccountsController::get_event_account('37', 'total_receivable');

        if (!$accounts) {
            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Account mapping not found.',
                'message_btn'   => 'btn btn-danger',
            ]);
        }

        $account_head_id = $accounts->account_head_id;

        // Payment history using Query Builder
        $result = DB::table('account_voucher_detail')
            ->join('account_vouchers', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
            ->select(
                'account_vouchers.id_account_vouchers',
                'account_vouchers.description',
                'account_vouchers.created_by',
                DB::raw('DATE_FORMAT(account_vouchers.voucher_date, "%d-%m-%Y") as voucher_date'),
                DB::raw('SUM(IFNULL(debit,0)) AS total_receivable'),
                DB::raw('SUM(IFNULL(credit,0)) AS total_received'),
                DB::raw('(SUM(IFNULL(debit,0)) - SUM(IFNULL(credit,0))) AS balance')
            )
            ->where('account_vouchers.voucher_type', '2')
            ->where('account_vouchers.voucher_status', 'Active')
            ->where('account_vouchers.franchise_order_id', $order_id)
            ->where('account_voucher_detail.account_head_id', $account_head_id)
            ->groupBy('account_vouchers.id_account_vouchers')
            ->get();

        if ($result->isNotEmpty()) {
            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Payment History',
                'message_btn'   => 'btn btn-success',
                'payment_history' => $result
            ]);
        }

        return response()->json([
            'success'       => true,
            'message_type'  => 'success',
            'message'       => 'Payment history not found.',
            'message_btn'   => 'btn btn-danger',
        ]);
    }


    public function cancel_invoice_payment(Request $request){
        $franchiseorderid = $request->id_franchise_orders;
        $id_account_vouchers = $request->id_account_vouchers;

     
        try {
            DB::beginTransaction();

            AccountVouchers::where('id_account_vouchers', $id_account_vouchers)
            ->update([
                'voucher_status' => 'cancelled',
                'cancelled_by'   => session('user_name'),
                'cancelled_on'   => now(),
            ]);

            // Reset franchise order product batches 

               $accounts =  AccountsController::get_event_account('37', 'total_receivable');
                if(!$accounts){ 
                    return response()->json([
                        'success'       => false,
                        'message_type'  => 'error',
                        'message'       => 'Mapping not found',
                        'message_btn'   => 'btn btn-danger',
                    ]);
                    return false;
                }
                $account_head_id = $accounts->account_head_id;

                $sql = "
                    SELECT 
                        SUM(IFNULL(debit,0)) AS total_receivable,
                        SUM(IFNULL(credit,0)) AS total_received,
                        SUM(IFNULL(debit,0)) - SUM(IFNULL(credit,0)) AS balance
                    FROM account_voucher_detail
                    JOIN account_vouchers 
                        ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                    WHERE 
                        account_vouchers.voucher_status = 'Active'
                        AND account_vouchers.franchise_order_id = {$franchiseorderid}
                        AND account_voucher_detail.account_head_id = {$account_head_id}
                        AND account_vouchers.id_account_vouchers != {$id_account_vouchers}
                ";

                $result = DB::select($sql);

                $total_receivable = $result[0]->total_receivable ?? 0;
                $total_received   = $result[0]->total_received ?? 0;
                $balance          = $result[0]->balance ?? 0;

            if($total_received ==0){
                DB::table('franchise_orders')
                ->where('id_franchise_orders', $franchiseorderid)
                ->update([
                    'order_status' => 'Invoiced',
                ]);
            }
         
         
            DB::commit();
            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Payment cancelled successfully.',
                'message_btn'   => 'btn btn-success'
            ]);

        } catch (\Exception $e) {
             DB::rollBack();
            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Failed to cancel Payment: ' . $e->getMessage(),
                'message_btn'   => 'btn btn-danger',
            ]);
        }
    }

    public function franchise_return_note(Request $request , $order_id){

        $business = Business::where("id_business", session('business_id'))->first();

        $order_detail = FranchiseOrder::where('id_franchise_orders', $order_id)
                ->where('order_status','!=', 'Cancelled')
                ->where('order_status','!=', 'Open')
                ->select( 'franchise_orders.*',DB::raw('DATE(franchise_orders.order_date) as order_date'))
                ->first();
        if(!$order_detail){
          return redirect()->back()->with('error', "Order is not Invoiced.");
        }  

        $franchise = Franchise::where('id_franchises', $order_detail->franchise_id)->first();

        $franchiseid =  $franchise->id_franchises;
        $franchiseinvoicenumber = DB::table('franchise_orders')
            ->where('franchise_id', $franchiseid)
            ->select(DB::raw('COALESCE(MAX(invoice_number), 0) + 1 AS invoice_number'))
            ->first();

        $arrears = $this->get_receivable_amount($franchiseid, $order_id);

        $order = DB::table('franchise_orders')
                ->select(
                    'franchise_orders.*','franchise_order_products.*','business_products.*','business_brands.*','product_batch.*',
                    DB::raw('DATE(franchise_orders.order_date) as order_date'),
                    DB::raw('DATE_FORMAT(franchise_orders.invoice_date, "%d-%m-%Y") as formatted_invoice_date'),
                    DB::raw('DATE_FORMAT(DATE_ADD(franchise_orders.invoice_date, INTERVAL 7 DAY), "%d-%m-%Y") as due_date'),
                    DB::raw('franchise_order_products.product_id as product_id'),
                    DB::raw('IFNULL(adj.addition,0) as already_retured_qty'),
                )
                ->join('franchise_order_products', 'franchise_order_products.franchise_order_id', '=', 'franchise_orders.id_franchise_orders')
                ->leftjoin('product_batch', 'product_batch.id_batch', '=', 'franchise_order_products.batch_id')
                ->join('business_products', 'business_products.id_business_products', '=', 'franchise_order_products.product_id')
                ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id') 
                    ->leftJoin(DB::raw("(
                        SELECT batch_id, product_id,franchise_order_product_id, SUM(IFNULL(adjustment_qty,0)) as addition
                        FROM adjustment_notes where  franchise_return_note_id > 0
                        GROUP BY franchise_order_product_id
                    ) as adj"), function($join) {
                        $join->on('adj.franchise_order_product_id', '=', 'franchise_order_products.id_franchise_order_products');
                    })
                ->where('franchise_orders.id_franchise_orders', $order_id)
                ->get()
                    ->map(function($item) {
                    return (array) $item;
                })
                ->toArray(); 
        return view('franchise.franchise_order_return_note',compact('business','franchise','franchiseinvoicenumber','arrears','order','order_detail','order_id'));  
    }

    public function save_franchise_return(Request $request){
        DB::beginTransaction();
        try {

            if (!$request->has('order_id') || !$request->has('table_data')) {
                return response()->json([
                    'success'       => false,
                    'message_type'  => 'error',
                    'message'       => 'Required data is missing.',
                    'message_btn'   => 'btn btn-danger',
                ]);
            }

            $orderId = $request->order_id;
            $tableData = $request->table_data;
            $summaryData = $request->summary_data ?? [];

            $business_id = session('business_id');

            $order = DB::table('franchise_orders as fo')
                        ->where('fo.id_franchise_orders', $orderId)
                        ->where('fo.order_status','!=','Cancelled')
                        ->first();

            $order_status = $order->order_status;

            if (!$order) {
                return response()->json([
                    'success'       => false,
                    'message_type'  => 'error',
                    'message'       => 'Order not found.',
                    'message_btn'   => 'btn btn-danger',
                ]);
            }

            $franchise_return_notes_id = DB::table('franchise_return_notes')->insertGetId([
                'business_id'=>session('business_id'),
                'franchise_order_id'=>$orderId,
                'return_date'=>$summaryData['return_date'] ?? date('Y-m-d'),
                'status' => 'Open',
                'created_by' => session('user_name'),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $adjustmentNotes = [];
            $totalReturnInventoryValue = 0;
            foreach ($tableData as $product) {
                $returnQty = floatval($product['return_qty'] ?? 0);
                if ($returnQty > 0) {
                    $productDetails = DB::table('business_products')
                                    ->where('id_business_products', $product['product_id'])
                                    ->first();

                    $unitPrice = floatval($product['batch_amount'] ?? 0);
                    $adjustmentNoteId = DB::table('adjustment_notes')->insertGetId([
                        'batch_id' => $product['batch_id'] ?? null,
                        'product_id' => $product['product_id'],
                        'adjustment_qty' => $returnQty,
                        'adjustment_date' => $summaryData['return_date'] ?? date('Y-m-d'),
                        'adjustment_remarks' =>
                            "Returned {$returnQty} units from Franchise Order #{$orderId} ({$productDetails->product})",
                        'unit_price' => $unitPrice,
                        'created_by' => session('user_name'),
                        'created_on' => now(),
                        'franchise_return_note_id' => $franchise_return_notes_id,
                        'franchise_order_product_id' => $product['id_franchise_order_products'] ?? null
                    ]);
                    $totalReturnInventoryValue +=($returnQty*$unitPrice);   
                    // $cogs_amount =($returnQty*$unitPrice);   


                $adjustmentNotes[] = [
                        'id' => $adjustmentNoteId,
                        'product_id' => $product['product_id'],
                        'qty' => $returnQty
                    ];
                }
            }   


              // -------------------> START  Reverse COGS Voucher   <--------------------------------------
                $accounts = AccountsController::get_event_account('23', 'cogs');
                if (!$accounts) {
                    return response()->json([
                        'success'       => false,
                        'message_type'  => 'error',
                        'message'       => 'Account mapping not found.',
                        'message_btn'   => 'btn btn-danger',
                    ]);
                }

                $account_head_cogs = $accounts->account_head_id; 
                $accounts = AccountsController::get_event_account('23', 'inventory');
                if (!$accounts) {
                    return response()->json([
                        'success'       => false,
                        'message_type'  => 'error',
                        'message'       => 'Account mapping not found.',
                        'message_btn'   => 'btn btn-danger',
                    ]);
                }
                $account_head_inventory = $accounts->account_head_id; 
                // Voucher main record
                $voucher = AccountVouchers::create([
                    'business_id'        => $business_id,
                    'description'        => 'Reverse Voucher - COGS reversed for Order #'.
                                            $order->id_franchise_orders.' returned by '.$order->franchise_name,
                    'voucher_date'       => now(),
                    'voucher_amount'     => $totalReturnInventoryValue,
                    'voucher_status'     => 'Active',
                    'created_by'         => session('user_name'),
                    'cost_center'        => 1,
                    'cost_center_name'   => 'Main Outlet',
                    'business_partner'   => 4,
                    'business_partner_id'=> $order->franchise_id,
                    'business_partner_name' => $order->franchise_name,
                    'voucher_type'       => 3,
                    'payment_mode'       => 'Cash',
                    'auto_voucher'       => 'Yes',
                    'sale_type'          => 'franchise',
                    'franchise_order_id' => $order->id_franchise_orders,
                    'franchise_return_note_id' => $franchise_return_notes_id,
                ]);
                $voucher_id = $voucher->id_account_vouchers;
                // Debit entries (Inventory)
                    $debit_amount = $totalReturnInventoryValue;
                    if ($debit_amount > 0) {
                        AccountVoucherDetail::create([
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $account_head_inventory,
                            'debit'              => round($debit_amount, 2),
                            'credit'             => 0,
                        ]);
                    }
                // Credit entries (COGS)
                    $credit_amount = $totalReturnInventoryValue;
                    if ($credit_amount > 0) {
                        AccountVoucherDetail::create([
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $account_head_cogs,
                            'debit'              => 0,
                            'credit'             => round($credit_amount, 2),
                        ]);
                    }
            // -------------------> END  Reverse COGS Voucher <--------------------------------------



          // -------------------> START  Knock off the receivable amount  <-------------------------------------- 


                $accounts =  AccountsController::get_event_account('37', 'total_receivable');
                if(!$accounts){ 
                    return redirect()->back()->with('error', "Account mapping not found.");
                }
                $account_head_id = $accounts->account_head_id;

                $sql = "
                    SELECT 
                        SUM(IFNULL(debit,0)) AS total_receivable,
                        SUM(IFNULL(credit,0)) AS total_received,
                        SUM(IFNULL(debit,0)) - SUM(IFNULL(credit,0)) AS balance
                    FROM account_voucher_detail
                    JOIN account_vouchers 
                        ON account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                    WHERE 
                        account_vouchers.voucher_status = 'Active'
                        AND account_vouchers.franchise_order_id = {$order->id_franchise_orders}
                        AND account_voucher_detail.account_head_id = {$account_head_id}
                ";
                $result = DB::select($sql);
                $total_receivable = $result[0]->total_receivable ?? 0;
                $total_received   = $result[0]->total_received ?? 0;
                $balance          = $result[0]->balance ?? 0;    

                $return_amount =  $summaryData['amount_inclusive_of_tax'];

                $save_as_advance = 0;
                $save_as_receivable = 0;
                if($balance == 0){
                    $save_as_advance =$return_amount;
                }elseif($return_amount > $balance  &&  $balance > 0){
                    $save_as_receivable = $balance;
                    $save_as_advance = $return_amount - $balance;
                }else if($balance > $return_amount){
                  $save_as_receivable  = $return_amount;
                }

                $accounts = AccountsController::get_event_account('37', 'total_receivable');
                $receivable_id = $accounts->account_head_id;


                $accounts = AccountsController::get_event_account('37', 'increase_advance_liability');
                $advance_payable_id = $accounts->account_head_id; 
        
                $accounts = AccountsController::get_event_account('37', 'total_invoice_value');
                $sale_id = $accounts->account_head_id;

                $accounts = AccountsController::get_event_account('37', 'sales_tax');
                $sales_tax_account_head_id = $accounts->account_head_id; 

                $accounts = AccountsController::get_event_account('37', 'invoice_discount');
                $invoice_discount_account_head_id = $accounts->account_head_id; 



                // Voucher main record
                $voucher = AccountVouchers::create([
                    'business_id'        => $business_id,
                    'description'        => 'Reverse Voucher. Receivable Knocked off for Order #'.
                                            $order->id_franchise_orders.' returned by '.$order->franchise_name,
                    'voucher_date'       => now(),
                    'voucher_amount'     => $summaryData['subtotal'] ?? 0,
                    'voucher_status'     => 'Active',
                    'created_by'         => session('user_name'),
                    'cost_center'        => 1,
                    'cost_center_name'   => 'Main Outlet',
                    'business_partner'   => 4,
                    'business_partner_id'=> $order->franchise_id,
                    'business_partner_name' => $order->franchise_name,
                    'voucher_type'       => 3,
                    'payment_mode'       => 'Cash',
                    'auto_voucher'       => 'Yes',
                    'sale_type'          => 'franchise',
                    'franchise_order_id' => $order->id_franchise_orders,
                    'franchise_return_note_id' => $franchise_return_notes_id,
                ]);
                $voucher_id = $voucher->id_account_vouchers;
                // Debit entries 
                    $debit_amount = $summaryData['subtotal'] ?? 0;
                    if ($debit_amount > 0) {
                        AccountVoucherDetail::create([
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $sale_id,
                            'debit'              => round($debit_amount, 2),
                            'credit'             => 0,
                        ]);
                    }
                    $total_sale_tax_amount =   ($summaryData['amount_of_product_sales_tax'] ?? 0) + ($summaryData['invoice_sales_tax_amount'] ?? 0);
                    $debit_amount = $total_sale_tax_amount ?? 0;
                    if ($debit_amount > 0) {
                        AccountVoucherDetail::create([
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $sales_tax_account_head_id,
                            'debit'              => round($debit_amount, 2),
                            'credit'             => 0,
                        ]);
                    }


                // Credit entries 
                    // RECEIVABLE KNOCK OFF
                    $credit_amount = $save_as_receivable ?? 0;
                    if ($credit_amount > 0) {
                        AccountVoucherDetail::create(attributes: [
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $receivable_id,
                            'debit'              => 0,
                            'credit'             => round($credit_amount, 2),
                        ]);
                    }

                    // RECEIVABLE KNOCK OFF
                    $credit_amount = $save_as_advance ?? 0;
                    if ($credit_amount > 0) {
                        AccountVoucherDetail::create(attributes: [
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $advance_payable_id,
                            'debit'              => 0,
                            'credit'             => round($credit_amount, 2),
                        ]);
                    }

                    $total_discount =   ($summaryData['invoice_discount'] ?? 0) + ($summaryData['total_product_discount'] ?? 0);
                    $credit_amount = $total_discount?? 0;
                    if ($credit_amount > 0) {
                        AccountVoucherDetail::create([
                            'account_voucher_id' => $voucher_id,
                            'account_head_id'    => $invoice_discount_account_head_id,
                            'debit'              => 0,
                            'credit'             => round($credit_amount, 2),
                        ]);
                    }
            // -------------------> END  Knock off the receivable amount  <--------------------------------------

            DB::commit();
            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Return note saved successfully.',
                'message_btn'   => 'btn btn-success',

                'data' => [
                    'return_note_id' => $franchise_return_notes_id,
                    'adjustment_notes' => $adjustmentNotes,
                    'total_items' => count($adjustmentNotes),
                    'totalReturnInventoryValue' => $totalReturnInventoryValue
                ],
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Failed to save return note: '.$e->getMessage(),
                'message_btn'   => 'btn btn-danger'
            ], 500);
        }
    } 

    public function franchise_order_return_history(Request $request)
    {
        $order_id = $request->id_franchise_orders;

        // Safe SQL query using binding + return as collection
        $result = collect(DB::select("SELECT 
                                        frn.id_franchise_return_notes as id,
                                        DATE_FORMAT(frn.return_date, '%d-%m-%Y') AS return_date,
                                        frn.status,
                                        frn.created_by,
                                        SUM(IFNULL(an.adjustment_qty, 0)) AS return_qty,
                                        IFNULL(voucher_data.voucher_ids, '') AS voucher_ids,
                                        IFNULL(voucher_data.description, '') AS voucher_description
                                    FROM franchise_return_notes AS frn
                                    INNER JOIN franchise_orders AS fo 
                                            ON fo.id_franchise_orders = frn.franchise_order_id
                                    INNER JOIN adjustment_notes AS an 
                                            ON an.franchise_return_note_id = frn.id_franchise_return_notes
                                    LEFT JOIN (
                                            SELECT 
                                                GROUP_CONCAT(DISTINCT account_vouchers.id_account_vouchers) AS voucher_ids,
                                                GROUP_CONCAT(DISTINCT account_vouchers.description) AS description,
                                                account_vouchers.franchise_return_note_id
                                            FROM account_vouchers
                                            JOIN account_voucher_detail 
                                                    ON account_voucher_detail.account_voucher_id = account_vouchers.id_account_vouchers
                                            WHERE account_vouchers.voucher_status = 'Active'
                                            AND account_vouchers.franchise_order_id = $order_id
                                            GROUP BY account_vouchers.franchise_return_note_id
                                    ) AS voucher_data 
                                            ON voucher_data.franchise_return_note_id = frn.id_franchise_return_notes
                                    WHERE frn.status != 'Cancelled'
                                    AND frn.deleted_at IS NULL
                                    AND frn.franchise_order_id = $order_id
                                    AND an.adjustment_qty > 0
                                    GROUP BY frn.id_franchise_return_notes;"));
        if ($result->isNotEmpty()) {
            return response()->json([
                'success'          => true,
                'message_type'     => 'success',
                'message'          => 'Return History',
                'message_btn'      => 'btn btn-success',
                'return_history'  => $result
            ]);
        }

        return response()->json([
            'success'       => true,
            'message_type'  => 'success',
            'message'       => 'Return history not found.',
            'message_btn'   => 'btn btn-danger',
        ]);
    }


    public function cancel_franchise_return_note(Request $request){
        $franchiseorderid = $request->id_franchise_orders;
        $return_note_id = $request->return_note_id;

        try {
            DB::beginTransaction();
            AccountVouchers::where('franchise_return_note_id', $return_note_id)
            ->update([
                'voucher_status' => 'cancelled',
                'cancelled_by'   => session('user_name'),
                'cancelled_on'   => now(),
            ]);


            DB::table('franchise_return_notes')
            ->where('id_franchise_return_notes', $return_note_id)
            ->update(['status' =>'Cancelled']);

            $adjustment_notes = DB::table('adjustment_notes')
            ->where('franchise_return_note_id', $return_note_id)
            ->get();

            if ($adjustment_notes->count() > 0) {
                foreach ($adjustment_notes as $product) {

                    $returnQty = floatval($product->adjustment_qty ?? 0);

                    if ($returnQty > 0) {

                        $productDetails = DB::table('business_products')
                            ->where('id_business_products', $product->product_id)
                            ->first();

                        $unitPrice = floatval($product->unit_price ?? 0);

                        DB::table('adjustment_notes')->insert([
                            'batch_id' => $product->batch_id ?? null,
                            'product_id' => $product->product_id,
                            'adjustment_qty' => -$returnQty,   // minus qty
                            'adjustment_date' => date('Y-m-d'),
                            'adjustment_remarks' =>
                                "Reverse Returned {$returnQty} units from Franchise Order #{$franchiseorderid} ({$productDetails->product})",
                            'unit_price' => $unitPrice,
                            'created_by' => session('user_name'),
                            'created_on' => now(),
                            'franchise_return_note_id' => $return_note_id,
                            'franchise_order_product_id' => $product->franchise_order_product_id ?? null
                        ]);
                    }
                }
            }

            DB::commit();
            return response()->json([
                'success'       => true,
                'message_type'  => 'success',
                'message'       => 'Return Note cancelled successfully.',
                'message_btn'   => 'btn btn-success'
            ]);

        } catch (\Exception $e) {
             DB::rollBack();
            return response()->json([
                'success'       => false,
                'message_type'  => 'error',
                'message'       => 'Failed to cancel Return Note: ' . $e->getMessage(),
                'message_btn'   => 'btn btn-danger',
            ]);
        }
    } 

    function pending_payment_of_franchise($business_id, $franchise_id)
    {
        $accounts = AccountsController::get_event_account('37', 'total_receivable');
        $account_head_id = $accounts->account_head_id;

        $sql = "
            SELECT 
                fo.id_franchise_orders,
                fo.franchise_id,
                fo.franchise_name,
                fo.franchise_ntn,
                fo.order_status,
                fo.created_by,
                DATE_FORMAT(fo.created_on, '%d-%m-%Y %h:%i:%s') AS created_on,
                DATE_FORMAT(fo.order_date, '%d-%m-%Y') AS order_date,
                DATE_FORMAT(fo.invoice_date, '%d-%m-%Y') AS invoice_date,
                fo.franchise_order_value,
                fo.invoice_number,
                f.franchise_name AS f_name,

                IFNULL(SUM(avd.debit), 0)  AS debit,
                IFNULL(SUM(avd.credit), 0) AS credit,
                IFNULL(SUM(avd.debit - avd.credit), 0) AS balance

            FROM franchise_orders AS fo

            INNER JOIN franchises AS f
                ON f.id_franchises = fo.franchise_id

            INNER JOIN account_vouchers AS av
                ON av.franchise_order_id = fo.id_franchise_orders
            AND av.voucher_status = 'active'

            INNER JOIN account_voucher_detail AS avd
                ON avd.account_voucher_id = av.id_account_vouchers
            AND avd.account_head_id = ?

            WHERE 
                fo.business_id = ?
                AND fo.franchise_id = ?
                AND fo.order_status != 'Cancelled'

            GROUP BY fo.id_franchise_orders 

            HAVING balance != 0 ORDER BY fo.id_franchise_orders
        ";
        return DB::select($sql, [$account_head_id, $business_id, $franchise_id]);
    } 

    public function franchise_aging(Request $request){
    
        $as_on_date = $request->input('as_on_date') ?? date('Y-m-d'); 
        $business_id =session('business_id');
        $accounts = AccountsController::get_event_account('37', 'total_receivable');
        if($accounts){
            $account_head_id = $accounts->account_head_id;
            $account_head_name = $accounts->account_head;
        } else {
            return 'Mapping not available';
        } 

        //--------------- Ajax Request Start
        if ($request->ajax()) {
            $as_on_date = $request->input('as_on_date') ?? date('Y-m-d'); 
            $sql = "
                        SELECT
                            franchise_name,
                            franchise_id,
                            id_franchise_orders,
                            invoice_number,
                            DATE_FORMAT(invoice_date, '%d-%m-%Y') AS invoice_date,
                            bal.balance AS invoice_total_receivable,
                            DATEDIFF('$as_on_date', invoice_date) AS lapsed_days,
                            franchise_order_extra,
                            a.product_name,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) < 10 THEN bal.balance ELSE 0 END AS bucket_0_9,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) BETWEEN 10 AND 19 THEN bal.balance ELSE 0 END AS bucket_10_19,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) BETWEEN 20 AND 39 THEN bal.balance ELSE 0 END AS bucket_20_39,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) BETWEEN 40 AND 59 THEN bal.balance ELSE 0 END AS bucket_40_58,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) BETWEEN 60 AND 89 THEN bal.balance ELSE 0 END AS bucket_60_89,
                            CASE WHEN DATEDIFF('$as_on_date', invoice_date) >= 90 THEN bal.balance ELSE 0 END AS bucket_90_plus
                        FROM
                            franchise_orders
                        LEFT JOIN (
                                SELECT 
                                    franchise_order_id, 
                                    GROUP_CONCAT(
                                        DISTINCT franchise_order_products.product_name
                                        SEPARATOR ','
                                    ) AS product_name
                                FROM 
                                    franchise_order_products
                                GROUP BY 
                                    franchise_order_id
                        ) AS a ON a.franchise_order_id = franchise_orders.id_franchise_orders

                        JOIN (
                            SELECT 
                                franchise_order_id,
                                SUM(debit) - SUM(credit) AS balance
                            FROM account_vouchers
                            JOIN account_voucher_detail
                                ON account_voucher_detail.account_voucher_id = account_vouchers.id_account_vouchers
                            WHERE account_voucher_detail.account_head_id = $account_head_id
                            AND account_vouchers.voucher_status = 'Active'
                            AND account_vouchers.business_id = $business_id
                            GROUP BY franchise_order_id
                        ) AS bal
                        ON bal.franchise_order_id = franchise_orders.id_franchise_orders
                        WHERE invoice_number IS NOT NULL
                        AND order_status != 'Cancelled'
                        AND bal.balance != 0
                        AND franchise_orders.business_id = $business_id
                        ORDER BY franchise_id;
                    ";

                $results = DB::select($sql);
                // Send results to Yajra DataTables 
                return DataTables::of($results)->make(true);
        }            
        //--------------- Ajax Request END 
        return view('franchise.reports.aging')->with(compact('as_on_date','account_head_name'));    
    }

    public function franchise_product_sale_details(Request $request){
        $start_date  = $request->get('start_date') ?? date('Y-m-01');
        $end_date    = $request->get('end_date') ?? date('Y-m-t') ;
     
        $business_id =session('business_id');

        //--------------- Ajax Request Start
        if ($request->ajax()) { 
              $start_date  = $request->get('start_date') ?? date('Y-m-01');
              $end_date    = $request->get('end_date') ?? date('Y-m-t') ;
              $sql = "
                      SELECT
                            fo.franchise_id,
                            fo.id_franchise_orders ,
                            fo.invoice_number,
                            fo.franchise_name,
                            IFNULL(franchises.franchise_office_phone , '') as franchise_office_phone,
                            DATE_FORMAT(fo.order_date , '%d-%m-%Y') AS order_date,
                            DATE_FORMAT(fo.invoice_date, '%d-%m-%Y') AS invoice_date,
                            bb.business_brand_name,
                            fop.product_name,
                            fop.product_id,
                            bp.category,
                            business_stores.business_store,
                            fop.staff_name,
                            SUM(fop.qty) AS qty,
                            SUM(fop.qty * fop.unit_price) AS total_amount,

                            CASE 
                                WHEN SUM(fop.qty) > 0 
                                THEN ROUND(
                                    SUM(fop.qty * fop.unit_price) / SUM(fop.qty),
                                    2
                                )
                                ELSE 0
                            END AS unit_price,

                            /* Product level */
                            SUM(fop.discounted_price) AS discounted_price,
                            SUM(fop.product_tax) AS product_tax,
                            
                            (SUM(fop.discounted_price) +     SUM(fop.product_tax)) as amount_after_product_disc_and_tax,

                            /* Invoice discount on discounted_price ONLY */
                            ROUND(
                                SUM(fop.discounted_price)
                                * IFNULL(fo.invoice_discount_perc, 0) / 100,
                                2
                            ) AS invoice_discount_amount,

                            /* After invoice discount */
                            ROUND(
                                SUM(fop.discounted_price)
                                - (
                                    SUM(fop.discounted_price)
                                    * IFNULL(fo.invoice_discount_perc, 0) / 100
                                ),
                                2
                            ) AS after_invoice_discount,

                            /* Invoice sales tax on (discounted_price - invoice discount) */
                            ROUND(
                                (
                                    SUM(fop.discounted_price)
                                    - (
                                        SUM(fop.discounted_price)
                                        * IFNULL(fo.invoice_discount_perc, 0) / 100
                                    )
                                ) * IFNULL(fo.invoice_sales_tax_perc, 0) / 100,
                                2
                            ) AS invoice_sales_tax_amount,

                            /* Final net amount */
                            ROUND(
                                (
                                    SUM(fop.discounted_price)
                                    - (
                                        SUM(fop.discounted_price)
                                        * IFNULL(fo.invoice_discount_perc, 0) / 100
                                    )
                                )
                                + (
                                    (
                                        SUM(fop.discounted_price)
                                        - (
                                            SUM(fop.discounted_price)
                                            * IFNULL(fo.invoice_discount_perc, 0) / 100
                                        )
                                    ) * IFNULL(fo.invoice_sales_tax_perc, 0) / 100
                                )
                                + SUM(fop.product_tax),
                                2
                            ) AS net_amount ,
                            
                            IFNULL(SUM(fop.qty*product_batch.batch_amount),0)  cog


                        FROM franchise_orders fo
                        INNER JOIN franchises on franchises.id_franchises = fo.franchise_id
                        INNER JOIN franchise_order_products fop
                            ON fop.franchise_order_id = fo.id_franchise_orders

                        INNER JOIN business_products bp
                            ON bp.id_business_products = fop.product_id
                        INNER JOIN product_batch on product_batch.id_batch =fop.batch_id
                        JOIN business_stores on business_stores.id_business_stores =product_batch.store_id

                        INNER JOIN business_brands bb
                            ON bb.id_business_brands = bp.brand_id

                        WHERE  fo.order_status IN ('Invoiced', 'Paid')
                        AND   DATE(fo.invoice_date) BETWEEN '".$start_date."' AND '".$end_date."' 
                        AND fo.business_id = $business_id
                        GROUP BY
                            fo.id_franchise_orders,
                            fop.product_id;
                     ";

                $results = DB::select($sql);
                // Send results to Yajra DataTables 
                return DataTables::of($results)->make(true);
         
        }            
        //--------------- Ajax Request END 
        return view('franchise.reports.franchise_product_sale_details')->with(compact('start_date','end_date'));    
    }
}