<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Staff;
use App\Models\Business;
use App\Models\LoanAdvSalary;
use App\Models\Allowance;
use App\Models\StaffAllowance;
use App\Models\Holiday;
use App\Models\OvertimeDay;
use App\Models\AccountEventMapping;
use App\Models\LeaveType;
use App\Models\LeavePolicy;
use App\Models\LeavePolicyDefinition;
use App\Models\DeductionPolicy;
use App\Models\LeaveApplication;
use App\Models\StaffLeave;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Log;


class HrmController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            return $next($request);
        });
    }

    /* Get staff by business for HO users*/
    public function getStaffByBusiness(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            $businessId = $request->input('business_id') ?? $request->query('business_id');
            
            if (!$businessId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Business ID is required.'
                ], 400);
            }
            
            $business = Business::where('id_business', $businessId)->first();
                
            if (!$business) {
                return response()->json([
                    'success' => false,
                    'message' => 'Business not found or not accessible.'
                ], 404);
            }
            
            $staffList = Staff::where('staff_active', 'Y')
                ->where('business_id', $businessId)
                ->orderBy('staff_fullname', 'asc')
                ->get(['id_staff', 'staff_fullname', 'business_id']);
                
            return response()->json([
                'success' => true,
                'staff' => $staffList,
                'business_id' => $businessId,
                'business_name' => $business->business_name
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error getting staff by business: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading staff list: ' . $e->getMessage()
            ], 500);
        }
    }


    /* Display loan applications page*/
    public function advanceSalaryApplications()
    {
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        $isHoUser = session('ho') === 'Yes';
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        $staffList = collect([]);
        if (!$isHoUser) {
            $staffList = Staff::where('staff_active', 'Y')
                ->where('business_id', $businessId)
                ->orderBy('staff_fullname', 'asc')
                ->get();
        } else {
            if (session('selected_business_id')) {
                $staffList = Staff::where('staff_active', 'Y')
                    ->where('business_id', session('selected_business_id'))
                    ->orderBy('staff_fullname', 'asc')
                    ->get();
            }
        }
        
        return view('hrm.advance_salary_applications', compact('business', 'staffList', 'businesses'));
    }

    /* Get loan applications data for DataTables*/
    public function getLoanApplicationsData(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            if ($isHoUser && $request && $request->filled('business_id')) {
                $businessId = $request->business_id;
            }else{
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }

             $query = LoanAdvSalary::join('staff', 'loan_applications.staff_id', '=', 'staff.id_staff')
            ->join('business', 'staff.business_id', '=', 'business.id_business');

            if (!$isHoUser) {
                $query->where('staff.business_id', $businessId);
            } else if ($request->filled('business_id') && $request->business_id !== "") {
                $query->where('staff.business_id', $request->business_id);
            }else{
                 $query->where('business.ho_accounts', 'Yes');
            }
            
            $query = $query->select([
                'loan_applications.*',
                'staff.staff_fullname',
                'staff.business_id',
                DB::raw("DATE_FORMAT(loan_applications.created_on, '%d-%m-%Y') as created_on_formatted"),
                DB::raw("DATE_FORMAT(loan_applications.approved_on, '%d-%m-%Y %H:%i:%s') as approved_on_formatted"),
                DB::raw("COALESCE(loan_applications.approved_by, '') as approved_by"),
                DB::raw("COALESCE(loan_applications.staff_name, '') as staff_name")
            ]);

            // Apply filters from request
            if ($request->filled('id_loan_applications')) {
                $query->where('loan_applications.id_loan_applications', $request->id_loan_applications);
            }
            
            if ($request->filled('staff_id')) {
                $query->where('loan_applications.staff_id', $request->staff_id);
            }
            
            if ($request->filled('staff_name')) {
                $query->where(function($q) use ($request) {
                    $q->where('loan_applications.staff_name', 'like', '%' . $request->staff_name . '%')
                    ->orWhere('staff.staff_fullname', 'like', '%' . $request->staff_name . '%');
                });
            }
            
            if ($request->filled('created_on')) {
                $query->whereDate('loan_applications.created_on', $request->created_on);
            }
            
            if ($request->filled('required_loan_amount')) {
                $query->where('loan_applications.required_loan_amount', 'like', '%' . $request->required_loan_amount . '%');
            }
            
            if ($request->filled('approved_amount')) {
                $query->where('loan_applications.approved_amount', 'like', '%' . $request->approved_amount . '%');
            }
            
            if ($request->filled('approved_by')) {
                $query->where('loan_applications.approved_by', 'like', '%' . $request->approved_by . '%');
            }
            
            if ($request->filled('approved_on')) {
                $query->whereDate('loan_applications.approved_on', $request->approved_on);
            }
            
            if ($request->filled('installment')) {
                $query->where('loan_applications.installment', $request->installment);
            }
            
            if ($request->filled('num_of_months')) {
                $query->where('loan_applications.num_of_months', $request->num_of_months);
            }
            
            if ($request->filled('application_status')) {
                $query->where('loan_applications.application_status', $request->application_status);
            }

            // Get total count
            $recordsTotal = $query->count();
            
            // Apply ordering
            if ($request->has('order') && count($request->order) > 0) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                $columns = [
                    0 => 'loan_applications.id_loan_applications',
                    1 => 'loan_applications.staff_id',
                    2 => 'loan_applications.staff_name',
                    3 => 'loan_applications.created_on',
                    4 => 'loan_applications.required_loan_amount',
                    5 => 'loan_applications.approved_amount',
                    6 => 'loan_applications.approved_by',
                    7 => 'loan_applications.approved_on',
                    8 => 'loan_applications.installment',
                    9 => 'loan_applications.num_of_months',
                    10 => 'loan_applications.application_status',
                ];
                
                if (isset($columns[$orderColumnIndex])) {
                    $query->orderBy($columns[$orderColumnIndex], $orderDirection);
                } else {
                    $query->orderBy('loan_applications.id_loan_applications', 'desc');
                }
            } else {
                $query->orderBy('loan_applications.id_loan_applications', 'desc');
            }

            // Apply pagination
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $data = $query->get();

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $data->map(function($loan) {
                    return [
                        'id_loan_applications' => $loan->id_loan_applications,
                        'staff_id' => $loan->staff_id,
                        'staff_name' => $loan->staff_name ?: $loan->staff_fullname,
                        'created_on' => $loan->created_on_formatted,
                        'required_loan_amount' => number_format($loan->required_loan_amount, 2),
                        'approved_amount' => $loan->approved_amount ? number_format($loan->approved_amount, 2) : '0.00',
                        'approved_by' => $loan->approved_by,
                        'approved_on' => $loan->approved_on_formatted,
                        'installment' => $loan->installment ? number_format($loan->installment, 2) : '0.00',
                        'num_of_months' => $loan->num_of_months,
                        'application_status' => $loan->application_status,
                        'business_id' => $loan->business_id
                    ];
                })
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching loan applications: ' . $e->getMessage());
            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /*Get single loan application*/
    public function getLoanApplication($id)
    {
        try {
            $isHoUser = session('ho') === 'Yes';;
            
            $loan = LoanAdvSalary::join('staff', 'loan_applications.staff_id', '=', 'staff.id_staff')
                ->select('loan_applications.*', 'staff.business_id');
            
            if (!$isHoUser) {
                $loan = $loan->where('staff.business_id', session('business_id'));
            }
            
            $loan = $loan->where('loan_applications.id_loan_applications', $id)
                ->first();
            
            if (!$loan) {
                return response()->json([
                    'success' => false,
                    'message' => 'Loan application not found.'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id_loan_applications' => $loan->id_loan_applications,
                    'staff_id' => $loan->staff_id,
                    'staff_name' => $loan->staff_name,
                    'required_loan_amount' => $loan->required_loan_amount,
                    'approved_amount' => $loan->approved_amount,
                    'application_status' => $loan->application_status,
                    'created_by' => $loan->created_by,
                    'approved_by' => $loan->approved_by,
                    'approved_on' => $loan->approved_on,
                    'comments' => $loan->comments,
                    'num_of_months' => $loan->num_of_months,
                    'installment' => $loan->installment,
                    'voucher_id' => $loan->voucher_id,
                    'business_id' => $loan->business_id,
                    'created_on' => $loan->created_on
                ]
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching loan application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching loan application.'
            ], 500);
        }
    }
    /* Create new loan application*/
    public function insertLoanApplication(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'staff_id' => 'required|exists:staff,id_staff',
                'required_loan_amount' => 'required|numeric|min:1',
                'num_of_months' => 'required|integer|min:1|max:12',
                'comments' => 'nullable|string',
                'business_id' => 'nullable|exists:business,id_business'
            ]);

            if (session('ho') === 'Yes' && $request->filled('business_id')) {
                $businessId = $request->business_id;
            } else {
                $businessId = session('business_id');
            }
            
            $userName = session('user_name');
            $staff = Staff::where('id_staff', $request->staff_id);
            
            
            if (session('ho') === 'Yes' && $request->filled('business_id')) {
                $staff = $staff->where('business_id', $request->business_id);
            } else {
                $staff = $staff->where('business_id', $businessId);
            }
            
            $staff = $staff->first();
                
            if (!$staff) {
                \Log::error('Staff not found or wrong business', [
                    'staff_id' => $request->staff_id,
                    'business_id' => $businessId,
                    'request_business_id' => $request->business_id
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found or does not belong to the selected business.'
                ], 404);
            }
            
           
            $installment = $request->required_loan_amount / $request->num_of_months;
            
           
            $loan = LoanAdvSalary::create([
                'staff_id' => $request->staff_id,
                'staff_name' => $staff->staff_fullname,
                'required_loan_amount' => $request->required_loan_amount,
                'application_status' => 'Pending',
                'created_by' => $userName,
                'comments' => $request->comments,
                'num_of_months' => $request->num_of_months,
                'installment' => round($installment, 2),
                'created_on' => now(),
                'business_id' => $businessId
            ]);
            
            DB::commit();
            
            \Log::info('Loan application created successfully', [
                'loan_id' => $loan->id_loan_applications,
                'business_id' => $businessId
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Loan application created successfully.',
                'data' => $loan
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error creating loan application: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Error creating loan application: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Update loan application*/
    public function updateLoanApplication(Request $request)
    {
        try {
            DB::beginTransaction();
            $validationRules = [
                'id_loan_applications' => 'required|exists:loan_applications,id_loan_applications',
                'num_of_months' => 'required|integer|min:1|max:12',
                'application_status' => 'required|in:Pending,Approved,Rejected,Disbursed',
                'comment' => 'nullable|string'
            ];
            
            if (in_array($request->application_status, ['Approved', 'Disbursed'])) {
                $validationRules['approved_loan_amount'] = 'required|numeric|min:1';
            } else {
                $validationRules['approved_loan_amount'] = 'required|numeric|min:0';
            }
            
            $request->validate($validationRules);
            $userName = session('user_name');
            $loan = DB::table('loan_applications')
                ->where('id_loan_applications', $request->id_loan_applications)
                ->first();
            
            if (!$loan) {
                return response()->json([
                    'success' => false,
                    'message' => 'Loan application not found.'
                ], 404);
            }
            
            $isNewlyApproved = ($request->application_status == 'Approved' && $loan->application_status != 'Approved');
            $installment = $request->approved_loan_amount > 0 ? 
                $request->approved_loan_amount / $request->num_of_months : 0;

            $updateData = [
                'approved_amount' => $request->approved_loan_amount,
                'num_of_months' => $request->num_of_months,
                'installment' => round($installment, 2),
                'comments' => $request->comment,
                'application_status' => $request->application_status
            ];
            
            if ($request->application_status == 'Approved') {
                $updateData['approved_by'] = $userName;
                $updateData['approved_on'] = now()->format('Y-m-d H:i:s');
                
                // Create vouchers
                if ($isNewlyApproved && !$loan->voucher_id && $request->approved_loan_amount > 0) {
                    $voucherInfo = $this->createLoanAccountingEntry((object) array_merge((array) $loan, $updateData));
                    $updateData['voucher_id'] = $voucherInfo['regular_voucher_id'];
                } 
            }
            
            DB::table('loan_applications')
                ->where('id_loan_applications', $request->id_loan_applications)
                ->update($updateData);

            $updatedLoan = DB::table('loan_applications')
                ->where('id_loan_applications', $request->id_loan_applications)
                ->first();
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Loan application updated successfully.',
                'data' => $updatedLoan
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error updating loan application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating loan application: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Create loan accounting vouchers) */
    private function createLoanAccountingEntry($loan)
    {
        try {
            $businessId = $loan->business_id;
            $userName = session('user_name');

            if(session('ho_accounts') == 'Yes') {
                $accounts_businessId = DB::table('business')->where('ho', 'Yes')->value('id_business');
            } else {
                $accounts_businessId = session('business_id');
            }
            
            // Get account mappings for loan approval (Event ID: 25)
            $accounts = DB::table('account_event_mapping')
                ->where('account_event_id', 25)
                ->where('business_id', $accounts_businessId)
                ->get()
                ->map(function($item) {
                    return [
                        'account_head_id' => $item->account_head_id,
                        'transaction_type' => $item->transaction_type,
                        'entity_name' => $item->entity_name,
                        'description' => $item->description
                    ];
                })
                ->toArray();

            $loanAccounts = [
                'debit' => null,
                'credit' => null
            ];
            
            foreach ($accounts as $account) {
                if ($account['entity_name'] == 'loan_amount') {
                    $loanAccounts['debit'] = $account;
                } elseif ($account['entity_name'] == 'salary_payable') {
                    $loanAccounts['credit'] = $account; 
                }
            }
            
            if (empty($loanAccounts['debit']) || empty($loanAccounts['credit'])) {
                throw new \Exception('Account event is not mapped for loan approval (Event ID: 25)');
            }

            $description = 'Loan Payment booked for Staff ' . $loan->staff_name . ' ID: ' . $loan->staff_id . ' on ' . date('d-m-Y') . ' will be Return in ' . $loan->num_of_months . ' Months';
            $amount = round($loan->approved_amount, 2);
            $currentDateTime = date('Y-m-d H:i:s');
            
            // ============================
            // 1. CREATE REGULAR VOUCHER
            // ============================
            $regularVoucherType = 3;
            $regularVoucherId = DB::table('account_vouchers')->insertGetId([
                'business_id' => $businessId,
                'description' => $description,
                'voucher_date' => $currentDateTime,
                'voucher_status' => 'Active',
                'created_by' => $userName,
                'business_partner' => 3,
                'business_partner_id' => $loan->staff_id,
                'business_partner_name' => $loan->staff_name,
                'voucher_type' => $regularVoucherType,
                'staffpayment_id' => 0,
                'loan_application_id' => $loan->id_loan_applications,
                'created_at' => $currentDateTime
            ]);

            // Regular Voucher - Debit Entry
            DB::table('account_voucher_detail')->insert([
                'account_voucher_id' => $regularVoucherId,
                'created_date' => $currentDateTime,
                'account_head_id' => $loanAccounts['debit']['account_head_id'],
                'debit' => $amount,
                'credit' => 0,
                'detail_remarks' => $description,
                'cost_center_type' => 'Branches',
                'cost_center' => 'Main Outlet',
                'payment_mode' => 'Cash',
                'instrument_number' => '',
                'created_at' => $currentDateTime,
                'created_by' => $userName
            ]);

            // Regular Voucher - Credit Entry
            DB::table('account_voucher_detail')->insert([
                'account_voucher_id' => $regularVoucherId,
                'created_date' => $currentDateTime,
                'account_head_id' => $loanAccounts['credit']['account_head_id'],
                'debit' => 0,
                'credit' => $amount,
                'detail_remarks' => $description,
                'cost_center_type' => 'Branches',
                'cost_center' => 'Main Outlet',
                'payment_mode' => 'Cash',
                'instrument_number' => '',
                'created_at' => $currentDateTime,
                'created_by' => $userName
            ]);

            return [
                'regular_voucher_id' => $regularVoucherId,
            ];
            
        } catch (\Exception $e) {
            \Log::error('Error creating loan accounting entry: ' . $e->getMessage());
            throw $e;
        }
    }

    /* Display holidays page with business selector for HO users*/
    public function markHolidays()
    {
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        $isHO = session('ho') === 'Yes';
        $hoAccounts = session('ho_accounts', 'No');
        $showAllTabs = ($isHO === false && $hoAccounts === 'No');
        
        if ($isHO) {
            $businesses = Business::orderBy('business_name', 'ASC')->get();
        } else {
            $businesses = Business::where('id_business', $businessId)->get();
        }
        return view('hrm.mark_holidays', compact('business', 'businesses', 'isHO', 'businessId', 'showAllTabs', 'hoAccounts'));
    }

    
    /*** Save Holiday*/
    public function saveHoliday(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'holiday_dates' => 'required|array',
                'holiday_dates.*' => 'required|date',
                'holiday_reason' => 'required|string|max:255'
            ]);
            
            $businessId = session('business_id');
            $userName = session('user_name');
            $holidayReason = $request->holiday_reason;
            $holidayDates = $request->holiday_dates;

            $results = [];
            $errors = [];
            
            foreach ($holidayDates as $holidayDate) {
                // Check if holiday already exists for this business and date
                $exists = Holiday::where('business_id', $businessId)
                    ->where('holiday_date', $holidayDate)
                    ->first();
                
                if ($exists) {
                    // If updating, update it
                    if ($request->filled('event_id')) {
                        $exists->update([
                            'holiday_reason' => $holidayReason
                        ]);
                        
                        $results[] = [
                            'id' => $exists->id_holidays,
                            'title' => $exists->holiday_reason,
                            'start' => $exists->holiday_date,
                            'status' => 'updated'
                        ];
                    } else {
                        $errors[] = "Holiday already exists for date: " . $holidayDate;
                    }
                } else {
                    // Create new holiday
                    $day = date('D', strtotime($holidayDate));
                    $holiday = Holiday::create([
                        'holiday_date' => $holidayDate,
                        'holiday_reason' => $holidayReason,
                        'holiday_day' => $day,
                        'business_id' => $businessId,
                        'created_by' => $userName
                    ]);
                    
                    $results[] = [
                        'id' => $holiday->id_holidays,
                        'title' => $holiday->holiday_reason,
                        'start' => $holiday->holiday_date,
                        'status' => 'created'
                    ];
                    
                    \Log::info('Holiday created:', [
                        'id' => $holiday->id_holidays,
                        'business_id' => $businessId,
                        'date' => $holidayDate,
                        'user' => $userName
                    ]);
                }
            }
            
            DB::commit();
            
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some holidays already exist.',
                    'errors' => $errors,
                    'results' => $results
                ], 400);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Holiday(s) saved successfully.',
                'count' => count($results),
                'results' => $results
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error saving holiday: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving holiday: ' . $e->getMessage()
            ], 500);
        }
    }

    
    /*** Delete Holiday */
    public function deleteHoliday(Request $request, $id = null)
    {
        try {
            if ($request->has('event_id')) {
                $id = $request->event_id;
            }
            
            $businessId = session('business_id');
            
            $holiday = Holiday::where('id_holidays', $id)
                ->where('business_id', $businessId)
                ->first();
            
            if (!$holiday) {
                return response()->json([
                    'success' => false,
                    'message' => 'Holiday not found or you do not have permission to delete it.'
                ], 404);
            }
            
            $holiday->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Holiday deleted successfully.'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error deleting holiday: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting holiday: ' . $e->getMessage()
            ], 500);
        }
    }
    /* Get Holiday Events for Calendar */
    public function getHolidayEvents(Request $request)
    {
        try {
            $year = $request->get('year', date('Y'));
            $businessId = session('business_id');
            
            $holidays = Holiday::where('business_id', $businessId)
                ->whereYear('holiday_date', $year)
                ->orderBy('holiday_date', 'asc')
                ->get()
                ->map(function($holiday) {
                    $date = $holiday->holiday_date;
                    
                    return [
                        'id' => $holiday->id_holidays,
                        'title' => $holiday->holiday_reason,
                        'start' => $date,
                        'allDay' => true,
                        'className' => 'holiday-event',
                        'business_id' => $holiday->business_id
                    ];
                });
            
            return response()->json($holidays);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching holidays: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /* Save Overtime Day*/
    public function saveOvertimeDay(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'overtime_dates' => 'required|array',
                'overtime_dates.*' => 'required|date',
                'overtime_reason' => 'required|string|max:255'
            ]);
            
            $businessId = session('business_id');
            $userName = session('user_name');
            $overtimeReason = $request->overtime_reason;
            $overtimeDates = $request->overtime_dates;

            $results = [];
            $errors = [];
            
            foreach ($overtimeDates as $overtimeDate) {
                $exists = OvertimeDay::where('business_id', $businessId)
                    ->where('overtime_date', $overtimeDate)
                    ->first();
                
                if ($exists) {
                    // If updating, update it
                    if ($request->filled('event_id')) {
                        $exists->update([
                            'overtime_reason' => $overtimeReason,
                        ]);
                        
                        $results[] = [
                            'id' => $exists->id_overtime,
                            'title' => $exists->overtime_reason,
                            'start' => $exists->overtime_date,
                            'status' => 'updated'
                        ];
                    } else {
                        $errors[] = "Overtime day already exists for date: " . $overtimeDate;
                    }
                } else {
                    // Create new overtime day
                    $day = date('D', strtotime($overtimeDate));
                    $overtime = OvertimeDay::create([
                        'overtime_date' => $overtimeDate,
                        'overtime_reason' => $overtimeReason,
                        'overtime_day' => $day,
                        'business_id' => $businessId,
                        'created_by' => $userName,
                        'created_on' => now()
                    ]);
                    
                    $results[] = [
                        'id' => $overtime->id_overtime,
                        'title' => $overtime->overtime_reason,
                        'start' => $overtime->overtime_date,
                        'status' => 'created'
                    ];
                    
                    \Log::info('Overtime day created:', [
                        'id' => $overtime->id_overtime,
                        'business_id' => $businessId,
                        'date' => $overtimeDate,
                        'user' => $userName
                    ]);
                }
            }
            
            DB::commit();
            
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some overtime days already exist.',
                    'errors' => $errors,
                    'results' => $results
                ], 400);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Overtime day(s) saved successfully.',
                'count' => count($results),
                'results' => $results
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error saving overtime day: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving overtime day: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Delete Overtime Day*/
    public function deleteOvertimeDay(Request $request, $id = null)
    {
        try {
            if ($request->has('event_id')) {
                $id = $request->event_id;
            }
            $businessId = session('business_id');
            
            $overtime = OvertimeDay::where('id_overtime', $id)
                ->where('business_id', $businessId)
                ->first();
            
            if (!$overtime) {
                return response()->json([
                    'success' => false,
                    'message' => 'Overtime day not found or you do not have permission to delete it.'
                ], 404);
            }
            
            $overtime->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Overtime day deleted successfully.'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error deleting overtime day: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting overtime day: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Get getOvertimeDayEvents for Calendar*/
    public function getOvertimeDayEvents(Request $request)
    {
        try {
            $year = $request->get('year', date('Y'));
            $businessId = session('business_id');
            
            $overtimeDays = OvertimeDay::where('business_id', $businessId)
                ->whereYear('overtime_date', $year)
                ->orderBy('overtime_date', 'asc')
                ->get()
                ->map(function($overtime) {
                    $date = $overtime->overtime_date;
                    
                    return [
                        'id' => $overtime->id_overtime,
                        'title' => $overtime->overtime_reason,
                        'start' => $date,
                        'allDay' => true,
                        'className' => 'overtime-event',
                        'business_id' => $overtime->business_id,
                        'type' => 'overtime'
                    ];
                });
            
            return response()->json($overtimeDays);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching overtime days: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }


    /*Get staff leave types for a specific staff*/
    public function getStaffLeaveTypes(Request $request)
    {
        try {
            $staffId = $request->input('staff_id');
            
            if (!$staffId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff ID is required.'
                ], 400);
            }

            $staff = Staff::where('id_staff', $staffId)->first();
            
            if (!$staff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found.'
                ], 404);
            }

            if (!$staff->leave_policy_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please assign leave policy to this staff. Staff has no leave policy assigned.',
                    'has_policy' => false
                ], 400);
            }
            $policy = LeavePolicy::where('id_leave_policy', $staff->leave_policy_id)
                ->whereNull('deleted_at')
                ->first();
                
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'The assigned leave policy has been deleted. Please assign a new leave policy to this staff.',
                    'has_policy' => false
                ], 400);
            }

            $leaveTypes = collect();
            $policyDefinitions = LeavePolicyDefinition::where('leave_policy_id', $staff->leave_policy_id)
                ->with(['leaveType' => function($query) {
                    $query->where('status', 'Active');
                }])
                ->get()
                ->filter(function($definition) {
                    return $definition->leaveType !== null;
                })
                ->map(function($definition) {
                    return [
                        'id_leave_type' => $definition->leave_type_id,
                        'leave_type' => $definition->leaveType->leave_type,
                        'allowed_count' => $definition->allowed_count
                    ];
                });
                
            $leaveTypes = $policyDefinitions;
            if ($leaveTypes->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No leave types configured in the assigned policy. Please configure leave types in the policy.',
                    'has_policy' => true,
                    'policy_id' => $staff->leave_policy_id
                ], 400);
            }
            
            return response()->json([
                'success' => true,
                'leaveTypes' => $leaveTypes,
                'has_policy' => true,
                'policy_name' => $policy->policy_name
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching staff leave types: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave types.'
            ], 500);
        }
    }

    /*Get staff leave balance*/
    public function getStaffLeaveBalance(Request $request)
    {
        try {
            $request->validate([
                'staff_id' => 'required|exists:staff,id_staff',
                'leave_type_id' => 'required|exists:leave_type,id_leave_type',
                'required_from' => 'required|date',
                'required_to' => 'required|date|after_or_equal:required_from',
                'year' => 'nullable|integer'
            ]);
            
            $staffId = $request->staff_id;
            $leaveTypeId = $request->leave_type_id;
            $requiredFrom = $request->required_from;
            $requiredTo = $request->required_to;
            $year = $request->year ?? date('Y', strtotime($requiredFrom));
            
            $staff = Staff::where('id_staff', $staffId)->first();
            if (!$staff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found.'
                ], 404);
            }
            
            // Check if staff has leave policy assigned
            if (!$staff->leave_policy_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please assign leave policy to this staff. Staff has no leave policy assigned.',
                    'has_policy' => false
                ], 400);
            }
            
            // Get the leave_type record
            $leaveTypeRecord = LeaveType::find($leaveTypeId);
            
            if (!$leaveTypeRecord) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found.'
                ], 404);
            }
            
            // Get staff's leave policy
            $policyId = $staff->leave_policy_id;
            $allowedCount = 0;
            $policyName = 'No Policy';
            $policyLeaves = 0;
            
            if ($policyId) {
                // Check if policy exists and is not deleted
                $policy = DB::table('leave_policies')
                    ->where('id_leave_policy', $policyId)
                    ->whereNull('deleted_at')
                    ->first();
                
                if ($policy) {
                    $policyName = $policy->policy_name;
                    
                    // Get allowed leaves from policy definition using leave_type_id
                    $policyDefinition = DB::table('leave_policies_definition')
                        ->where('leave_policy_id', $policyId)
                        ->where('leave_type_id', $leaveTypeId)
                        ->first();
                    
                    if ($policyDefinition) {
                        $allowedCount = $policyDefinition->allowed_count;
                        $policyLeaves = $policyDefinition->allowed_count;
                    }
                } else {
                    // Policy is deleted
                    return response()->json([
                        'success' => false,
                        'message' => 'The assigned leave policy has been deleted. Please assign a new leave policy to this staff.',
                        'has_policy' => false
                    ], 400);
                }
            }
            
            // Calculate consumed leaves for the requested year
            $consumedLeaves = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('leave_type_id', $leaveTypeId)
                ->where('staff_leave_status', 'Active')
                ->whereYear('staff_leave_date', $year)
                ->count();
            
            // Calculate remaining leaves
            $remainingLeaves = max(0, $allowedCount - $consumedLeaves);
            
            // Check for overlapping leaves in the requested date range
            $overlappingLeaves = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('staff_leave_status', 'Active')
                ->whereBetween('staff_leave_date', [$requiredFrom, $requiredTo])
                ->count();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'staff_id' => $staffId,
                    'staff_name' => $staff->staff_fullname,
                    'leave_type' => $leaveTypeRecord->leave_type,
                    'leave_type_id' => $leaveTypeId,
                    'allowed_count' => $allowedCount,
                    'consumed_leaves' => $consumedLeaves,
                    'remaining_leaves' => $remainingLeaves,
                    'policy_name' => $policyName,
                    'policy_id' => $policyId,
                    'policy_leaves' => $policyLeaves,
                    'requested_year' => $year,
                    'overlapping_leaves' => $overlappingLeaves,
                    'requested_from' => $requiredFrom,
                    'requested_to' => $requiredTo,
                    'has_policy' => true
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching staff leave balance: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave balance: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getStaffLeaveEvents(Request $request)
    {
        try {
            $year = $request->get('year', date('Y'));
            $businessId = $request->get('business_id', session('business_id'));
            $staffId = $request->get('staff_id');
            
            // Validate user has access to the business
            $isHoUser = session('ho') === 'Yes';
            if (!$isHoUser && $businessId != session('business_id')) {
                return response()->json([], 403);
            }
            
            $staffLeaves = DB::table('staff_leaves as sl')
                ->join('staff as s', 'sl.staff_id', '=', 's.id_staff')
                ->leftJoin('leave_type as lt', 'sl.leave_type_id', '=', 'lt.id_leave_type')
                ->where('sl.staff_leave_status', 'Active')
                ->whereYear('sl.staff_leave_date', $year);
            
            // Apply business filter
            if ($businessId) {
                $staffLeaves = $staffLeaves->where('s.business_id', $businessId);
            } else if (!$isHoUser) {
                // Non-HO users can only see their own business
                $staffLeaves = $staffLeaves->where('s.business_id', session('business_id'));
            }
            
            // Apply staff filter if provided
            if ($staffId) {
                $staffLeaves = $staffLeaves->where('sl.staff_id', $staffId);
            }
            
            $staffLeaves = $staffLeaves->orderBy('sl.staff_leave_date', 'asc')
                ->select(
                    'sl.*',
                    's.staff_fullname as staff_name',
                    'lt.leave_type',
                    's.business_id'
                )
                ->get()
                ->map(function($leave) {
                    return [
                        'id' => $leave->id_staff_leaves,
                        'title' => $leave->staff_name . ' - ' . $leave->leave_type,
                        'start' => $leave->staff_leave_date,
                        'allDay' => true,
                        'className' => 'staff-leave-event',
                        'staff_id' => $leave->staff_id,
                        'staff_name' => $leave->staff_name,
                        'leave_type' => $leave->leave_type,
                        'leave_type_id' => $leave->leave_type_id,
                        'reason' => $leave->leave_reason,
                        'business_id' => $leave->business_id
                    ];
                });
            
            return response()->json($staffLeaves);
            
        } catch (\Exception $e) {
            Log::error('Error fetching staff leave events: ' . $e->getMessage());
            return response()->json([], 500);
        }
    }

    /*Save staff leave*/
    public function saveStaffLeave(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'leave_dates' => 'required|array',
                'leave_dates.*' => 'required|date',
                'staff_id' => 'required|exists:staff,id_staff',
                'leave_type_id' => 'required|exists:leave_type,id_leave_type',
                'leave_reason' => 'required|string|max:255'
            ]);
            
            $businessId = session('business_id');
            $userName = session('user_name');
            $staffId = $request->staff_id;
            $leaveTypeId = $request->leave_type_id;
            $leaveReason = $request->leave_reason;
            $leaveDates = $request->leave_dates;

            // Get staff and business info
            $staff = Staff::where('id_staff', $staffId)->first();
            if (!$staff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found.'
                ], 404);
            }
            
            // Verify staff belongs to the business (for non-HO users)
            $isHoUser = session('ho') === 'Yes';
            if (!$isHoUser && $staff->business_id != $businessId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff does not belong to your business.'
                ], 403);
            }
            
            // Get leave type name
            $leaveType = LeaveType::find($leaveTypeId);
            if (!$leaveType) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found.'
                ], 404);
            }

            $results = [];
            $errors = [];
            
            foreach ($leaveDates as $leaveDate) {
                // Check if leave already exists for this staff and date
                $exists = DB::table('staff_leaves')
                    ->where('staff_id', $staffId)
                    ->where('staff_leave_date', $leaveDate)
                    ->where('staff_leave_status', 'Active')
                    ->first();
                
                if ($exists) {
                    // If updating, update it
                    if ($request->filled('event_id')) {
                        DB::table('staff_leaves')
                            ->where('id_staff_leaves', $exists->id_staff_leaves)
                            ->update([
                                'leave_type_id' => $leaveTypeId,
                                'leave_type' => $leaveType->leave_type,
                                'leave_reason' => $leaveReason
                            ]);
                        
                        $results[] = [
                            'id' => $exists->id_staff_leaves,
                            'title' => $staff->staff_fullname . ' - ' . $leaveType->leave_type,
                            'start' => $exists->staff_leave_date,
                            'status' => 'updated'
                        ];
                    } else {
                        $errors[] = "Leave already exists for staff on date: " . date('d-m-Y', strtotime($leaveDate));
                    }
                } else {
                    // Create new staff leave
                    $leaveId = DB::table('staff_leaves')->insertGetId([
                        'staff_id' => $staffId,
                        'staff_leave_date' => $leaveDate,
                        'leave_type' => $leaveType->leave_type,
                        'leave_type_id' => $leaveTypeId,
                        'leave_reason' => $leaveReason,
                        'staff_leave_status' => 'Active',
                        'created_by' => $userName,
                        'created_on' => now()
                    ]);
                    
                    $results[] = [
                        'id' => $leaveId,
                        'title' => $staff->staff_fullname . ' - ' . $leaveType->leave_type,
                        'start' => $leaveDate,
                        'status' => 'created'
                    ];
                    
                    Log::info('Staff leave created:', [
                        'id' => $leaveId,
                        'staff_id' => $staffId,
                        'date' => $leaveDate,
                        'user' => $userName
                    ]);
                }
            }
            
            DB::commit();
            
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some leaves already exist.',
                    'errors' => $errors,
                    'results' => $results
                ], 400);
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Staff leave(s) saved successfully.',
                'count' => count($results),
                'results' => $results
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving staff leave: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving staff leave: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Delete staff leave*/
    public function deleteStaffLeave(Request $request, $id = null)
    {
        try {
            if ($request->has('event_id')) {
                $id = $request->event_id;
            }
            
            if (!$id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave ID is required.'
                ], 400);
            }
            
            $businessId = session('business_id');
            $isHoUser = session('ho') === 'Yes';
            
            $leave = DB::table('staff_leaves as sl')
                ->join('staff as s', 'sl.staff_id', '=', 's.id_staff')
                ->where('sl.id_staff_leaves', $id)
                ->select('sl.*', 's.business_id')
                ->first();
            
            if (!$leave) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff leave not found.'
                ], 404);
            }
            
            // Check permissions
            if (!$isHoUser && $leave->business_id != $businessId) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to delete this leave.'
                ], 403);
            }
            
            // Soft delete by updating status
            DB::table('staff_leaves')
                ->where('id_staff_leaves', $id)
                ->update([
                    'staff_leave_status' => 'Inactive',
                ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Staff leave deleted successfully.'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error deleting staff leave: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting staff leave: ' . $e->getMessage()
            ], 500);
        }
    }

    

    /* Display allowances list */
    public function staff_allowances(Request $request)
    {
        if (session('ho') === 'Yes') {
            $businessId = session('business_id');
        } else {
            if (session('ho_accounts') === 'No') {
                $businessId = session('business_id');
            } else {
                return ( 'Access denied. This page is managed by Head Office.');
            }
        }
        

        $businesses = collect([]);
        $business = Business::find($businessId);
        $allowances = Allowance::where('business_id', $businessId)
            ->orderBy('id_allowances', 'DESC')
            ->get();
        
        $mappedEntities = [];
        
        if($business && $business->ho_accounts === 'Yes'){
            // If business has HO accounts access, get the main HO business
            $HObusiness = Business::where('ho', 'Yes')->first();
            if($HObusiness){
                $mappedEntities = AccountEventMapping::where('account_event_id', 26)
                    ->where('business_id', $HObusiness->id_business)
                    ->where('entity_name', 'like', '%allowance%')
                    ->whereNot('entity_name', 'like', '%overtime_allowance%')
                    ->pluck('entity_name')
                    ->unique()
                    ->sort()
                    ->toArray();
            }
        } else {
            // Regular business - use its own mapping
            $mappedEntities = AccountEventMapping::where('account_event_id', 26)
                ->where('business_id', $businessId)
                ->where('entity_name', 'like', '%allowance%')
                ->whereNot('entity_name', 'like', '%overtime_allowance%')
                ->pluck('entity_name')
                ->unique()
                ->sort()
                ->toArray();
        }
        return view('hrm.allowances_list', compact('allowances', 'mappedEntities', 'business'));
    }

    /* Save allowance*/
    public function save_staff_allowance(Request $request)
    {
        try {
            // Access control check - same as deduction policies
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $validatedData = $request->validate([
                'allowance_name' => 'required|string|max:255',
                'entity_name' => 'required|string|max:255',
            ]);

            // Always use session business_id - no business selection
            $businessId = session('business_id');
            
            // Determine which business to check account event mapping from
            $currentBusiness = Business::find($businessId);
            $mappingBusinessId = $businessId;
            
            if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
                // If business has HO accounts access, check mapping from main HO business
                $HObusiness = Business::where('ho', 'Yes')->first();
                if ($HObusiness) {
                    $mappingBusinessId = $HObusiness->id_business;
                }
            }
            
            // Check if entity exists in Account Event Mapping
            $entityExists = AccountEventMapping::where('account_event_id', 26)
                ->where('entity_name', $validatedData['entity_name'])
                ->where('business_id', $mappingBusinessId)
                ->exists();

            if (!$entityExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Entity name not found in Account Event Mapping for event ID 26. Please contact administrator to map this entity first.'
                ], 400);
            }

            // Check if allowance already exists with same entity name in THIS BUSINESS
            $existingAllowance = Allowance::where('entity_name', $validatedData['entity_name'])
                ->where('business_id', $businessId)
                ->first();

            if ($request->id_allowances && $request->id_allowances != '') {
                $allowance = Allowance::where('id_allowances', $request->id_allowances)
                    ->where('business_id', $businessId)
                    ->first();
                
                if (!$allowance) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Allowance not found.'
                    ], 404);
                }
                
                // Check if trying to change to an entity name that already exists in this business
                if ($allowance->entity_name !== $validatedData['entity_name'] && $existingAllowance) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Allowance with this entity name already exists in this business.'
                    ], 400);
                }
                
                $allowance->update([
                    'allowance_name' => $validatedData['allowance_name'],
                    'entity_name' => $validatedData['entity_name']
                ]);
                $message = 'Allowance updated successfully.';
                
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'data' => $allowance
                ]);
                
            } else {
                // Creating new allowance
                if ($existingAllowance) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Allowance with this entity name already exists in this business.'
                    ], 400);
                }
                
                $allowance = Allowance::create([
                    'allowance_name' => $validatedData['allowance_name'],
                    'entity_name' => $validatedData['entity_name'],
                    'business_id' => $businessId,
                    'created_by' => session('user_name'),
                    'created_at' => now()
                ]);
                
                $message = 'Allowance created successfully.';
                
                return response()->json([
                    'success' => true,
                    'message' => $message,
                    'data' => $allowance
                ]);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error saving allowance: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Get allowance by ID*/
    public function get_allowance($id)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $businessId = session('business_id');
            
            $allowance = Allowance::select('allowances.*', 'business.business_name')
                ->leftJoin('business', 'allowances.business_id', '=', 'business.id_business')
                ->where('allowances.id_allowances', $id)
                ->where('allowances.business_id', $businessId)
                ->first();
            
            if (!$allowance) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Allowance not found.'
                ], 404);
            }
            
            return response()->json([
                'success' => true, 
                'data' => $allowance
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false, 
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /*DataTables server-side processing for allowances*/
    public function allowances_data(Request $request)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'draw' => $request->input('draw', 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            // Always use session business_id - no business filter needed
            $businessId = session('business_id');
            
            $query = Allowance::select('allowances.*', 'business.business_name')
                ->leftJoin('business', 'allowances.business_id', '=', 'business.id_business')
                ->where('allowances.business_id', $businessId); // Only show allowances for current business

            $filters = $request->all();
            
            if (!empty($filters['id_allowances'])) {
                $query->where('allowances.id_allowances', $filters['id_allowances']);
            }
            
            if (!empty($filters['allowance_name'])) {
                $query->where('allowances.allowance_name', 'like', '%' . $filters['allowance_name'] . '%');
            }
            
            if (!empty($filters['entity_name'])) {
                $query->where('allowances.entity_name', 'like', '%' . $filters['entity_name'] . '%');
            }
            
            if (!empty($filters['created_by'])) {
                $query->where('allowances.created_by', 'like', '%' . $filters['created_by'] . '%');
            }
            
            if (!empty($filters['created_at'])) {
                $query->whereDate('allowances.created_at', $filters['created_at']);
            }
            
            $totalRecords = $query->count();
            
            $orderColumn = $request->input('order.0.column', 0);
            $orderDirection = $request->input('order.0.dir', 'desc');
            $columns = ['allowances.id_allowances', 'allowances.allowance_name', 'allowances.entity_name', 'business.business_name', 'allowances.created_by', 'allowances.created_at'];
            
            if (isset($columns[$orderColumn])) {
                $query->orderBy($columns[$orderColumn], $orderDirection);
            }
        
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            $query->skip($start)->take($length);
            
            $allowances = $query->get();
            
            return response()->json([
                'draw' => intval($request->input('draw', 1)),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $allowances
            ]);
        } catch (\Exception $e) {
            \Log::error('Error fetching allowances data: ' . $e->getMessage());
            return response()->json([
                'draw' => intval($request->input('draw', 1)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }


    /* Display leave policies page*/
    public function leavePolicies()
    {
        if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
            return abort(403, 'Access denied. This page is managed by Head Office.');
        }
        
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        
        $leaveTypes = LeaveType::where(function($query) use ($businessId) {
            $query->where('business_id', $businessId)
                ->orWhereNull('business_id');
        })
        ->where('status', 'Active')
        ->orderBy('leave_type', 'ASC')
        ->get();
        
        // Only send necessary variables
        return view('hrm.leave_policies', compact('business', 'leaveTypes'));
    }
   
    /* Get leave policies data for DataTables*/
    public function getLeavePoliciesData(Request $request)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'draw' => $request->input('draw', 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            $businessId = session('business_id');
            
            $query = LeavePolicy::select('leave_policies.*', 'business.business_name')
                ->leftJoin('business', 'leave_policies.business_id', '=', 'business.id_business')
                ->whereNull('leave_policies.deleted_at')
                ->where('leave_policies.business_id', $businessId);

            // Apply filters from request
            if ($request->filled('id_leave_policy')) {
                $query->where('leave_policies.id_leave_policy', $request->id_leave_policy);
            }
            
            if ($request->filled('policy_name')) {
                $query->where('leave_policies.policy_name', 'like', '%' . $request->policy_name . '%');
            }
            
            if ($request->filled('policy_description')) {
                $query->where('leave_policies.policy_description', 'like', '%' . $request->policy_description . '%');
            }
            
            if ($request->filled('business_name')) {
                $query->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }
            
            if ($request->filled('created_at')) {
                $query->whereDate('leave_policies.created_at', $request->created_at);
            }

            $recordsTotal = $query->count();

            if ($request->has('order') && count($request->order) > 0) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                $columns = [
                    0 => 'leave_policies.id_leave_policy',
                    1 => 'leave_policies.policy_name',
                    2 => 'leave_policies.policy_description',
                    3 => 'total_leaves',
                    4 => 'business.business_name',
                    5 => 'leave_policies.created_at',
                ];
                
                if (isset($columns[$orderColumnIndex])) {
                    if ($columns[$orderColumnIndex] === 'total_leaves') {
                        $query->orderBy(
                            DB::raw('(SELECT SUM(allowed_count) FROM leave_policies_definition WHERE leave_policy_id = leave_policies.id_leave_policy)'),
                            $orderDirection
                        );
                    } else {
                        $query->orderBy($columns[$orderColumnIndex], $orderDirection);
                    }
                } else {
                    $query->orderBy('leave_policies.id_leave_policy', 'desc');
                }
            } else {
                $query->orderBy('leave_policies.id_leave_policy', 'desc');
            }

            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $policies = $query->get();

            $policyIds = $policies->pluck('id_leave_policy')->toArray();
            
            $totalLeaves = [];
            if (!empty($policyIds)) {
                $leavesData = DB::table('leave_policies_definition')
                    ->whereIn('leave_policy_id', $policyIds)
                    ->select('leave_policy_id', DB::raw('SUM(allowed_count) as total'))
                    ->groupBy('leave_policy_id')
                    ->get()
                    ->keyBy('leave_policy_id');
                
                foreach ($leavesData as $policyId => $data) {
                    $totalLeaves[$policyId] = $data->total;
                }
            }

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $policies->map(function($policy) use ($totalLeaves) {
                    $isAssigned = DB::table('staff')
                        ->where('leave_policy_id', $policy->id_leave_policy)
                        ->where('business_id', $policy->business_id)
                        ->exists();
                    
                    return [
                        'id_leave_policy' => $policy->id_leave_policy,
                        'policy_name' => $policy->policy_name,
                        'policy_description' => $policy->policy_description,
                        'total_leaves' => $totalLeaves[$policy->id_leave_policy] ?? 0,
                        'business_name' => $policy->business_name ?? 'N/A',
                        'created_at' => $policy->created_at ? $policy->created_at->format('d-m-Y') : '',
                        'is_assigned' => $isAssigned,
                        'business_id' => $policy->business_id
                    ];
                })
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching leave policies: ' . $e->getMessage());
            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /*Get single leave policy with definition*/
    public function getLeavePolicy($id)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $businessId = session('business_id');
            
            $policy = LeavePolicy::with(['definitions' => function($query) {
                $query->select('leave_policies_definition.*', 'leave_type.leave_type as leave_type_name')
                    ->leftJoin('leave_type', 'leave_policies_definition.leave_type_id', '=', 'leave_type.id_leave_type');
            }])
            ->select('leave_policies.*', 'business.business_name')
            ->leftJoin('business', 'leave_policies.business_id', '=', 'business.id_business')
            ->where('leave_policies.id_leave_policy', $id)
            ->where('leave_policies.business_id', $businessId)
            ->whereNull('leave_policies.deleted_at')
            ->first();
            
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave policy not found.'
                ], 404);
            }
            
            $isAssigned = DB::table('staff')
                ->where('leave_policy_id', $id)
                ->where('business_id', $businessId)
                ->exists();
            
            return response()->json([
                'success' => true,
                'data' => $policy,
                'is_assigned' => $isAssigned
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching leave policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave policy.'
            ], 500);
        }
    }

    /*Save leave policy (create or update)*/
    public function saveLeavePolicy(Request $request)
    {
        try {
            DB::beginTransaction();
            
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $request->validate([
                'policy_name' => 'required|string|max:255',
                'policy_description' => 'nullable|string',
                'leave_types' => 'required|array|min:1',
                'leave_types.*.leave_type_id' => 'required|exists:leave_type,id_leave_type',
                'leave_types.*.allowed_count' => 'required|integer|min:1',
            ]);
            
            $policyId = $request->input('id_leave_policy');
            $businessId = session('business_id');
            $userName = session('user_name');
            
            // Check permissions - only allow creating/updating for current business
            if ($policyId) {
                $policyExists = LeavePolicy::where('id_leave_policy', $policyId)
                    ->where('business_id', $businessId)
                    ->exists();
                
                if (!$policyExists) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Leave policy not found or you do not have permission to edit it.'
                    ], 404);
                }
                
                $isAssigned = DB::table('staff')
                    ->where('leave_policy_id', $policyId)
                    ->where('business_id', $businessId)
                    ->exists();
                
                if ($isAssigned) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot edit policy. This policy is currently assigned to staff members.'
                    ], 400);
                }
            }

            $policyData = [
                'policy_name' => $request->policy_name,
                'policy_description' => $request->policy_description,
                'business_id' => $businessId,
                'updated_at' => now()
            ];
            
            if ($policyId) {
                $policy = LeavePolicy::where('id_leave_policy', $policyId)
                    ->where('business_id', $businessId)
                    ->first();
                
                if (!$policy) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Leave policy not found.'
                    ], 404);
                }
                
                $policy->update($policyData);
                LeavePolicyDefinition::where('leave_policy_id', $policyId)->delete();
            } else {
                // Check if policy with same name already exists in this business
                $existingPolicy = LeavePolicy::where('policy_name', $request->policy_name)
                    ->where('business_id', $businessId)
                    ->first();
                
                if ($existingPolicy) {
                    return response()->json([
                        'success' => false,
                        'message' => 'A policy with this name already exists in your business.'
                    ], 400);
                }
                
                $policyData['created_by'] = $userName;
                $policyData['created_at'] = now();
                $policy = LeavePolicy::create($policyData);
                $policyId = $policy->id_leave_policy;
            }
            
            $definitionsData = [];
            foreach ($request->leave_types as $leaveType) {
                $definitionsData[] = [
                    'leave_policy_id' => $policyId,
                    'leave_type_id' => $leaveType['leave_type_id'],
                    'leave_type' => $leaveType['leave_type_name'] ?? '',
                    'allowed_count' => $leaveType['allowed_count'],
                    'created_at' => now()
                ];
            }
            
            if (!empty($definitionsData)) {
                LeavePolicyDefinition::insert($definitionsData);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $policyId ? 'Leave policy updated successfully.' : 'Leave policy created successfully.',
                'data' => ['id_leave_policy' => $policyId]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error saving leave policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving leave policy: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Delete leave policy*/
    public function deleteLeavePolicy(Request $request, $id = null)
    {
        try {
            if (!$id && $request->has('id')) {
                $id = $request->input('id');
            }
            
            if (!$id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid request. Policy ID is required.'
                ], 400);
            }
            
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $policyId = $id;
            $businessId = session('business_id');
            
            // Get the policy from current business only
            $policy = LeavePolicy::where('id_leave_policy', $policyId)
                ->where('business_id', $businessId)
                ->first();
                
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave policy not found.'
                ], 404);
            }
            
            // Check if policy is assigned to staff
            $assignedStaffCount = Staff::where('leave_policy_id', $policyId)
                ->where('business_id', $businessId)
                ->count();
            
            if ($assignedStaffCount > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete policy. It is assigned to ' . $assignedStaffCount . ' staff member(s) in your business.'
                ], 400);
            }
            
            $policy->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Policy deleted successfully.'
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error deleting leave policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting leave policy.'
            ], 500);
        }
    }
    /*Get total leaves for a policy*/
    public function getPolicyTotalLeaves(Request $request)
    {
        try {
            $policyId = $request->input('policy_id');
            $businessId = session('business_id');
            
            if (!$policyId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Policy ID is required.'
                ]);
            }
            $policyExists = LeavePolicy::where('id_leave_policy', $policyId)
                ->where('business_id', $businessId)
                ->exists();
                
            if (!$policyExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Policy not found.'
                ], 404);
            }
            
            $totalLeaves = LeavePolicyDefinition::where('leave_policy_id', $policyId)
                ->sum('allowed_count');
            
            return response()->json([
                'success' => true,
                'total_leaves' => $totalLeaves ?? 0
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error getting policy total leaves: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error getting total leaves.'
            ], 500);
        }
    }

  
    public function calculatePolicyLeaves($id)
    {
        try {
            $currentBusiness = Business::find(session('business_id'));
            
            // Determine which business to fetch policies from
            if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
                // If current business has HO accounts access, get policies from HO business
                $hoBusiness = Business::where('ho', 'Yes')->first();
                $policyBusinessId = $hoBusiness ? $hoBusiness->id_business : session('business_id');
            } else {
                // Otherwise, use current business
                $policyBusinessId = session('business_id');
            }
            
            $policy = LeavePolicy::with('definitions')
                ->where('id_leave_policy', $id)
                ->where(function($query) use ($policyBusinessId) {
                    $query->where('business_id', $policyBusinessId)
                        ->orWhereNull('business_id'); // Also include null/global policies
                })
                ->first();
                
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave policy not found.'
                ], 404);
            }
            
            $totalLeaves = $policy->definitions->sum('allowed_count');
            
            return response()->json([
                'success' => true,
                'total_leaves' => $totalLeaves,
                'policy_name' => $policy->policy_name
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error calculating leaves: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getStaffRemainingLeaves($id)
    {
        try {
            $businessId = session('business_id');
            
            $staff = Staff::where('id_staff', $id)
                ->where('business_id', $businessId)
                ->first();
                
            if (!$staff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found.'
                ], 404);
            }
            
            $currentYear = date('Y');
            $usedLeaves = DB::table('staff_leaves')
                ->where('staff_id', $id)
                ->whereYear('staff_leave_date', $currentYear)
                ->where('staff_leave_status', 'Active')
                ->count();
            
            $totalLeaves = $staff->staff_annual_leaves;
            if($totalLeaves == null && $totalLeaves == ''){ 
                $totalLeaves = 0;
            }
            $remainingLeaves = max(0, $totalLeaves - $usedLeaves);
            
            return response()->json([
                'success' => true,
                'remaining_leaves' => $remainingLeaves,
                'used_leaves' => $usedLeaves,
                'total_leaves' => $totalLeaves
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error calculating remaining leaves: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Create new leave type*/
    public function createLeaveType(Request $request)
    {
        try {
            $request->validate([
                'leave_type' => 'required|string|max:255',
                'business_id' => 'nullable|exists:business,id_business'
            ]);
            
            $userName = session('user_name');
            $businessId = session('business_id');
            $existingLeaveType = LeaveType::where('leave_type', $request->leave_type)
                ->where(function($query) use ($businessId) {
                    $query->where('business_id', $businessId)
                        ->orWhereNull('business_id');
                })
                ->first();
                
            if ($existingLeaveType) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type already exists.'
                ], 400);
            }
            
            $leaveType = LeaveType::create([
                'leave_type' => $request->leave_type,
                'business_id' => $businessId,
                'created_by' => $userName,
                'created_on' => now(),
                'status' => 'Active'
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Leave type created successfully.',
                'data' => $leaveType
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error creating leave type: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error creating leave type: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getLeaveTypes(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            if ($isHoUser && $request && $request->filled('business_id')) {
                $businessId = $request->business_id;
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }
            $leaveTypes = LeaveType::where(function($query) use ($businessId) {
                $query->where('business_id', $businessId)
                    ->orWhereNull('business_id');
            })
            ->where('status', 'Active')
            ->orderBy('leave_type', 'ASC')
            ->get();
            
            return response()->json([
                'success' => true,
                'leaveTypes' => $leaveTypes
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error fetching leave types: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave types'
            ], 500);
        }
    }


    /*Display deduction policies page*/
    public function deductionPolicies()
    {
        if (session('ho') === 'Yes') {
            $businessId = session('business_id');
        } else {
            if (session('ho_accounts') === 'No') {
                $businessId = session('business_id');
            } else {
                return ( 'Access denied. This page is managed by Head Office.');
            }
        }
        
        $business = Business::where('id_business', $businessId)->first();
        return view('hrm.deduction_policies', compact('business'));
    }
    
    /* Get deduction policies data for DataTables*/
    public function getDeductionPoliciesData(Request $request)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'draw' => $request->input('draw', 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            // Always use session business_id - no business filter needed
            $businessId = session('business_id');
            
            $query = DeductionPolicy::select('deduction_policy.*', 'business.business_name')
                ->leftJoin('business', 'deduction_policy.business_id', '=', 'business.id_business')
                ->where('deduction_policy.business_id', $businessId); // Only show policies for current business

            // Apply filters from request
            if ($request->filled('id_deduction_policy')) {
                $query->where('deduction_policy.id_deduction_policy', $request->id_deduction_policy);
            }
            
            if ($request->filled('deduction_policy')) {
                $query->where('deduction_policy.deduction_policy', 'like', '%' . $request->deduction_policy . '%');
            }
            
            if ($request->filled('business_name')) {
                $query->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }
            
            if ($request->filled('deduction_days_absent')) {
                $query->where('deduction_policy.deduction_days_absent', $request->deduction_days_absent);
            }
            
            if ($request->filled('deduction_days_late')) {
                $query->where('deduction_policy.deduction_days_late', $request->deduction_days_late);
            }
            
            if ($request->filled('total_late_allowed')) {
                $query->where('deduction_policy.total_late_allowed', $request->total_late_allowed);
            }
            
            if ($request->filled('total_absents_allowed')) {
                $query->where('deduction_policy.total_absents_allowed', $request->total_absents_allowed);
            }
            
            if ($request->filled('created_at')) {
                $query->whereDate('deduction_policy.created_at', $request->created_at);
            }

            $recordsTotal = $query->count();

            if ($request->has('order') && count($request->order) > 0) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                $columns = [
                    0 => 'deduction_policy.id_deduction_policy',
                    1 => 'deduction_policy.deduction_policy',
                    2 => 'business.business_name',
                    3 => 'deduction_policy.deduction_days_absent',
                    4 => 'deduction_policy.deduction_days_late',
                    5 => 'deduction_policy.total_late_allowed',
                    6 => 'deduction_policy.total_absents_allowed',
                    7 => 'deduction_policy.created_at',
                ];
                
                if (isset($columns[$orderColumnIndex])) {
                    $query->orderBy($columns[$orderColumnIndex], $orderDirection);
                } else {
                    $query->orderBy('deduction_policy.id_deduction_policy', 'desc');
                }
            } else {
                $query->orderBy('deduction_policy.id_deduction_policy', 'desc');
            }

            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $policies = $query->get();
            $policyIds = $policies->pluck('id_deduction_policy')->toArray();
            $staffCounts = [];
            
            if (!empty($policyIds)) {
                $staffData = DB::table('staff')
                    ->whereIn('deduction_policy', $policyIds)
                    ->where('business_id', $businessId)
                    ->select('deduction_policy', DB::raw('COUNT(*) as count'))
                    ->groupBy('deduction_policy')
                    ->get()
                    ->keyBy('deduction_policy');
                
                foreach ($staffData as $policyId => $data) {
                    $staffCounts[$policyId] = $data->count;
                }
            }

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $policies->map(function($policy) use ($staffCounts) {
                    $staffCount = $staffCounts[$policy->id_deduction_policy] ?? 0;
                    $isAssigned = $staffCount > 0;
                    
                    return [
                        'id_deduction_policy' => $policy->id_deduction_policy,
                        'deduction_policy' => $policy->deduction_policy,
                        'business_name' => $policy->business_name ?? 'N/A',
                        'deduction_days_absent' => $policy->deduction_days_absent,
                        'deduction_days_late' => $policy->deduction_days_late,
                        'total_late_allowed' => $policy->total_late_allowed,
                        'total_absents_allowed' => $policy->total_absents_allowed,
                        'created_at' => $policy->created_at ? date('d-m-Y', strtotime($policy->created_at)) : '',
                        'staff_count' => $staffCount,
                        'is_assigned' => $isAssigned,
                        'business_id' => $policy->business_id
                    ];
                })
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching deduction policies: ' . $e->getMessage());
            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }
        
    /* Get single deduction policy*/
    public function getDeductionPolicy($id)
    {
        try {
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $businessId = session('business_id');
            
            $policy = DeductionPolicy::select('deduction_policy.*', 'business.business_name')
                ->leftJoin('business', 'deduction_policy.business_id', '=', 'business.id_business')
                ->where('deduction_policy.id_deduction_policy', $id)
                ->where('deduction_policy.business_id', $businessId)
                ->first();
            
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Deduction policy not found.'
                ], 404);
            }
            
            $staffCount = Staff::where('deduction_policy', $id)
                ->where('business_id', $businessId)
                ->count();
            
            $isAssigned = $staffCount > 0;
            
            return response()->json([
                'success' => true,
                'data' => $policy,
                'is_assigned' => $isAssigned,
                'staff_count' => $staffCount
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching deduction policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching deduction policy.'
            ], 500);
        }
    }
    
 
    /* Save deduction policy (create or update)*/
    public function saveDeductionPolicy(Request $request)
    {
        try {
            DB::beginTransaction();
            
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $request->validate([
                'deduction_policy' => 'required|string|max:45',
                'deduction_days_absent' => 'required|numeric|min:0',
                'deduction_days_late' => 'required|numeric|min:0',
                'total_late_allowed' => 'required|integer|min:0',
                'total_absents_allowed' => 'required|integer|min:0',
            ]);
            
            $policyId = $request->input('id_deduction_policy');
            $businessId = session('business_id');
            
            if ($policyId) {
                $staffCount = Staff::where('deduction_policy', $policyId)
                    ->where('business_id', $businessId)
                    ->count();
                
                if ($staffCount > 0) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot edit policy. This policy is currently assigned to ' . $staffCount . ' staff member(s).'
                    ], 400);
                }
            }
            $existingPolicy = DeductionPolicy::where('deduction_policy', $request->deduction_policy)
                ->where('business_id', $businessId);
            
            if ($policyId) {
                $existingPolicy = $existingPolicy->where('id_deduction_policy', '!=', $policyId);
            }
            
            if ($existingPolicy->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'A deduction policy with this name already exists in this business.'
                ], 400);
            }
            
            $policyData = [
                'deduction_policy' => $request->deduction_policy,
                'deduction_days_absent' => $request->deduction_days_absent,
                'deduction_days_late' => $request->deduction_days_late,
                'total_late_allowed' => $request->total_late_allowed,
                'total_absents_allowed' => $request->total_absents_allowed,
                'business_id' => $businessId,
            ];
            
            if ($policyId) {
                // Check if policy exists and belongs to the current business
                $policy = DeductionPolicy::where('id_deduction_policy', $policyId)
                    ->where('business_id', $businessId)
                    ->first();
                
                if (!$policy) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Deduction policy not found.'
                    ], 404);
                }
                
                $policy->update($policyData);
                $message = 'Deduction policy updated successfully.';
            } else {
                $policy = DeductionPolicy::create($policyData);
                $policyId = $policy->id_deduction_policy;
                $message = 'Deduction policy created successfully.';
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => ['id_deduction_policy' => $policyId]
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving deduction policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving deduction policy: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Delete deduction policy*/
    public function deleteDeductionPolicy(Request $request, $id = null)
    {
        try {
            if (!$id && $request->has('id')) {
                $id = $request->input('id');
            }
            
            if (!$id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid request. Policy ID is required.'
                ], 400);
            }
            
            // Access control check
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'success' => false,
                    'message' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $policyId = $id;
            $businessId = session('business_id');
            
            // Get the policy from current business only
            $policy = DeductionPolicy::where('id_deduction_policy', $policyId)
                ->where('business_id', $businessId)
                ->first();
                
            if (!$policy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Deduction policy not found.'
                ], 404);
            }
            
            // Check if policy is assigned to staff
            $assignedStaffCount = Staff::where('deduction_policy', $policyId)
                ->where('business_id', $businessId)
                ->count();
            
            if ($assignedStaffCount > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete policy. It is assigned to ' . $assignedStaffCount . ' staff member(s).'
                ], 400);
            }
            
            $policy->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Deduction policy deleted successfully.'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error deleting deduction policy: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting deduction policy.'
            ], 500);
        }
    }
    

    /*Display leave types page*/
    public function leaveTypes(Request $request)
    {
        // Access control check - same as deduction policies
        if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
            return abort(403, 'Access denied. This page is managed by Head Office.');
        }
        
        // Clear session messages
        session()->forget(['Success', 'error']);
        
        $businessId = session('business_id');
        $business = Business::find($businessId);
        
        // Get leave types for current business only
        $leaveTypes = LeaveType::where('business_id', $businessId)
            ->whereNull('deleted_at')
            ->orderBy('created_on', 'DESC')
            ->get();
        
        return view('hrm.leave_types', compact(
            'business', 
            'leaveTypes'
        ));
    }

    /* Add new leave type*/
    public function addLeaveType(Request $request)
    {
        // Access control check - same as deduction policies
        if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
            return redirect()->route('hrm.leave_types')
                ->with('Error', 'Access denied. This page is managed by Head Office.');
        }
        
        $validator = \Validator::make($request->all(), [
            'leave_type' => 'required|string|max:255',
        ]);

        if ($validator->fails() || !$request->leave_type) {
            return redirect()->route('hrm.leave_types')
                ->with('Error', 'Leave type is required.');
        }

        $businessId = session('business_id');
        
        $existingLeaveType = LeaveType::where('leave_type', $request->leave_type)
            ->where('business_id', $businessId)
            ->whereNull('deleted_at')
            ->first();
        
        if ($existingLeaveType) {
            return redirect()->route('hrm.leave_types')
                ->with('Error', 'Leave type already exists for this business.');
        }

        $data = [
            'leave_type' => $request->leave_type,
            'created_by' => session('user_name'),
            'business_id' => $businessId,
            'created_on' => now(),
            'status' => 'Active'
        ];

        LeaveType::create($data);

        return redirect()->route('hrm.leave_types')
            ->with('Success', 'New leave type added successfully.');
    }

    /* Toggle leave type status (from CodeIgniter conversion)*/
    public function toggleLeaveStatus(Request $request)
    {
        // Access control check - same as deduction policies
        if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
            return response()->json(['status' => false, 'message' => 'Access denied.'], 403);
        }
        
        $validator = \Validator::make($request->all(), [
            'id' => 'required|exists:leave_type,id_leave_type',
            'status' => 'required|in:Active,Inactive'
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false], 400);
        }

        $businessId = session('business_id');
        
        $leaveType = LeaveType::where('id_leave_type', $request->id)
            ->where('business_id', $businessId)
            ->whereNull('deleted_at')
            ->first();
        
        if (!$leaveType) {
            return response()->json(['status' => false], 404);
        }

        $leaveType->update(['status' => $request->status]);

        return response()->json(['status' => true]);
    }

    /* Get leave types data for DataTables*/
    public function getLeaveTypesData(Request $request)
    {
        try {
            // Access control check - same as deduction policies
            if (session('ho') === 'No' && session('ho_accounts') === 'Yes') {
                return response()->json([
                    'draw' => $request->input('draw', 1),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Access denied. This page is managed by Head Office.'
                ], 403);
            }
            
            $businessId = session('business_id');
            
            $query = LeaveType::select('leave_type.*', 'business.business_name')
                ->leftJoin('business', 'leave_type.business_id', '=', 'business.id_business')
                ->whereNull('leave_type.deleted_at')
                ->where('leave_type.business_id', $businessId);
            // Apply filters
            if ($request->filled('id_leave_type')) {
                $query->where('leave_type.id_leave_type', $request->id_leave_type);
            }
            
            if ($request->filled('leave_type')) {
                $query->where('leave_type.leave_type', 'like', '%' . $request->leave_type . '%');
            }
            
            if ($request->filled('status')) {
                $query->where('leave_type.status', $request->status);
            }
            
            if ($request->filled('created_on')) {
                $query->whereDate('leave_type.created_on', $request->created_on);
            }
            
            if ($request->filled('business_name')) {
                $query->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }

            $recordsTotal = $query->count();

            // Apply ordering
            if ($request->has('order') && count($request->order) > 0) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                $columns = [
                    0 => 'leave_type.id_leave_type',
                    1 => 'leave_type.leave_type',
                    2 => 'leave_type.status',
                    3 => 'leave_type.created_on',
                    4 => 'business.business_name',
                ];
                
                if (isset($columns[$orderColumnIndex])) {
                    $query->orderBy($columns[$orderColumnIndex], $orderDirection);
                } else {
                    $query->orderBy('leave_type.created_on', 'desc');
                }
            } else {
                $query->orderBy('leave_type.created_on', 'desc');
            }

            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $leaveTypes = $query->get();

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $leaveTypes->map(function($leaveType) {
                    $createdOnFormatted = '';
                    if ($leaveType->created_on) {
                        if (is_string($leaveType->created_on)) {
                            $createdOnFormatted = date('d-m-Y', strtotime($leaveType->created_on));
                        } elseif ($leaveType->created_on instanceof \DateTime) {
                            $createdOnFormatted = $leaveType->created_on->format('d-m-Y');
                        }
                    }
                    
                    return [
                        'id_leave_type' => $leaveType->id_leave_type,
                        'leave_type' => $leaveType->leave_type,
                        'status' => $leaveType->status,
                        'created_on' => $leaveType->created_on,
                        'business_name' => $leaveType->business_name,
                        'business_id' => $leaveType->business_id,
                        'created_by' => $leaveType->created_by,
                        'created_on_formatted' => $createdOnFormatted,
                    ];
                })
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching leave types data: ' . $e->getMessage());
            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /* Display leave applications page */
    public function leaveApplications(Request $request)
    {
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        $isHoUser = session('ho') === 'Yes';
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        
        // Get staff list based on user type
        $staffList = collect([]);
        if (!$isHoUser) {
            $staffList = Staff::where('staff_active', 'Y')
                ->where('business_id', $businessId)
                ->orderBy('staff_fullname', 'asc')
                ->get();
        } else {
            if (session('selected_business_id')) {
                $staffList = Staff::where('staff_active', 'Y')
                    ->where('business_id', session('selected_business_id'))
                    ->orderBy('staff_fullname', 'asc')
                    ->get();
            }
        }
        
        // Get status filter
        $status = $request->get('statusfilter', 'Pending');
        $staffId = $request->get('stafffilter', 0);
        
        return view('hrm.leave_applications', compact('business', 'staffList', 'businesses', 'status', 'staffId'));
    }
   
    /* Get leave applications data for DataTables*/
    public function getLeaveApplicationsData(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            
            $query = DB::table('leave_applications')
                ->select(
                    'leave_applications.*', 
                    'staff.staff_fullname', 
                    'business.business_name',
                    'business.ho_accounts'
                )
                ->join('staff', 'leave_applications.staff_id', '=', 'staff.id_staff')
                ->join('business', 'staff.business_id', '=', 'business.id_business');

            // Determine which business to filter by
            if ($isHoUser) {
                if ($request->filled('business_id') && $request->business_id !== "") {
                    $query->where('staff.business_id', $request->business_id);
                } else {
                    $query->where('business.ho_accounts', 'Yes');
                }
            } else {
                $query->where('staff.business_id', session('business_id'));
            }

            // Apply application status filter
            if ($request->filled('application_status') && $request->application_status !== 'All') {
                $query->where('leave_applications.application_status', $request->application_status);
            }

            // Apply other filters
            if ($request->filled('id_leave_applications')) {
                $query->where('leave_applications.id_leave_applications', $request->id_leave_applications);
            }
            
            if ($request->filled('staff_id')) {
                $query->where('leave_applications.staff_id', $request->staff_id);
            }
            
            if ($request->filled('staff_name')) {
                $query->where(function($q) use ($request) {
                    $q->where('leave_applications.staff_name', 'like', '%' . $request->staff_name . '%')
                    ->orWhere('staff.staff_fullname', 'like', '%' . $request->staff_name . '%');
                });
            }
            
            if ($request->filled('business_name')) {
                $query->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }
            
            if ($request->filled('leave_type')) {
                $query->where('leave_applications.leave_type', 'like', '%' . $request->leave_type . '%');
            }
            
            if ($request->filled('required_from')) {
                $query->whereDate('leave_applications.required_from', $request->required_from);
            }
            
            if ($request->filled('required_to')) {
                $query->whereDate('leave_applications.required_to', $request->required_to);
            }
            
            if ($request->filled('approved_by')) {
                $query->where('leave_applications.approved_by', 'like', '%' . $request->approved_by . '%');
            }
            
            if ($request->filled('approved_on')) {
                $query->whereDate('leave_applications.approved_on', $request->approved_on);
            }

            // Get total count
            $recordsTotal = $query->count();

            // Apply ordering
            if ($request->has('order') && count($request->order) > 0) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                $columns = [
                    0 => 'leave_applications.id_leave_applications',
                    1 => 'leave_applications.staff_id',
                    2 => 'leave_applications.staff_name',
                    3 => 'business.business_name',
                    4 => 'leave_applications.leave_type',
                    5 => 'leave_applications.required_from',
                    6 => 'leave_applications.required_to',
                    7 => 'leave_applications.application_status',
                    8 => 'leave_applications.approved_by',
                    9 => 'leave_applications.approved_on',
                ];
                
                if (isset($columns[$orderColumnIndex])) {
                    $query->orderBy($columns[$orderColumnIndex], $orderDirection);
                } else {
                    $query->orderBy('leave_applications.id_leave_applications', 'desc');
                }
            } else {
                $query->orderBy('leave_applications.id_leave_applications', 'desc');
            }

            // Apply pagination
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $applications = $query->get();

            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $applications->map(function($application) {
                    $requiredFrom = $application->required_from ? date('Y-m-d', strtotime($application->required_from)) : '';
                    $requiredTo = $application->required_to ? date('Y-m-d', strtotime($application->required_to)) : '';
                    $approvedOn = $application->approved_on ? date('Y-m-d H:i:s', strtotime($application->approved_on)) : '';
                    
                    return [
                        'id_leave_applications' => $application->id_leave_applications,
                        'staff_id' => $application->staff_id,
                        'staff_name' => $application->staff_name ?: $application->staff_fullname,
                        'business_name' => $application->business_name,
                        'leave_type' => $application->leave_type,
                        'required_from' => $requiredFrom,
                        'required_to' => $requiredTo,
                        'application_status' => $application->application_status,
                        'approved_by' => $application->approved_by ?? '',
                        'approved_on' => $approvedOn,
                        'business_id' => $application->business_id,
                        'staff_fullname' => $application->staff_fullname,
                        'comments' => $application->comments,
                        'leave_type_id' => $application->leave_type_id,
                    ];
                })
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching leave applications: ' . $e->getMessage());
            return response()->json([
                'draw' => $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }
    /* Get single leave application data for modal*/
    public function getLeaveApplicationData($id)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            
            $application = DB::table('leave_applications')
                ->select('leave_applications.*', 'staff.staff_fullname', 'staff.business_id as staff_business_id', 'staff.leave_policy_id')
                ->join('staff', 'leave_applications.staff_id', '=', 'staff.id_staff')
                ->where('leave_applications.id_leave_applications', $id);
            
            if (!$isHoUser) {
                $application = $application->where('staff.business_id', session('business_id'));
            }
            
            $application = $application->first();
            
            if (!$application) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave application not found.'
                ], 404);
            }
            
            // Get leave types based on staff's policy
            $leaveTypes = collect();
            
            if ($application->leave_policy_id) {
                // Get leave types from the staff's assigned policy definition
                $policyDefinitions = LeavePolicyDefinition::where('leave_policy_id', $application->leave_policy_id)
                    ->with('leaveType')
                    ->get();
                
                // Filter out only active leave types
                $leaveTypes = $policyDefinitions->filter(function ($definition) {
                    return $definition->leaveType && $definition->leaveType->status === 'Active';
                })->map(function ($definition) {
                    return $definition->leaveType;
                })->unique('id_leave_type')->sortBy('leave_type')->values();
            } else {
                // If staff has no policy assigned, get all leave types based on business ho_accounts
                $business = Business::find($application->staff_business_id);
                
                if ($business && $business->ho_accounts === 'Yes') {
                    // If business has HO accounts, get leave types from HO business
                    $hoBusiness = Business::where('ho', 'Yes')->first();
                    $leaveTypesBusinessId = $hoBusiness ? $hoBusiness->id_business : $application->staff_business_id;
                } else {
                    $leaveTypesBusinessId = $application->staff_business_id;
                }
                
                $leaveTypes = LeaveType::where(function($query) use ($leaveTypesBusinessId) {
                        $query->where('business_id', $leaveTypesBusinessId)
                            ->orWhereNull('business_id');
                    })
                    ->where('status', 'Active')
                    ->orderBy('leave_type', 'ASC')
                    ->get();
            }
            
            return response()->json([
                'success' => true,
                'application' => [
                    'id_leave_applications' => $application->id_leave_applications,
                    'staff_id' => $application->staff_id,
                    'staff_name' => $application->staff_name,
                    'required_from' => $application->required_from ? date('Y-m-d', strtotime($application->required_from)) : '',
                    'required_to' => $application->required_to ? date('Y-m-d', strtotime($application->required_to)) : '',
                    'leave_type' => $application->leave_type,
                    'leave_type_id' => $application->leave_type_id,
                    'application_status' => $application->application_status,
                    'comments' => $application->comments,
                    'approved_by' => $application->approved_by,
                    'approved_on' => $application->approved_on ? date('Y-m-d H:i:s', strtotime($application->approved_on)) : '',
                    'business_id' => $application->business_id,
                ],
                'staff_info' => [
                    'staff_fullname' => $application->staff_fullname,
                    'business_id' => $application->staff_business_id,
                    'leave_policy_id' => $application->leave_policy_id,
                ],
                'leave_types' => $leaveTypes
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching leave application data: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave application data.'
            ], 500);
        }
    }
   
    /*** Get leave balance for staff */
    public function getLeaveBalance(Request $request)
    {
        try {
            $request->validate([
                'staff_id' => 'required|exists:staff,id_staff',
                'leave_type_id' => 'required|exists:leave_type,id_leave_type',
                'required_from' => 'required|date', // Add requested date
                'required_to' => 'required|date|after_or_equal:required_from' // Add requested date
            ]);
            
            $staffId = $request->staff_id;
            $leaveTypeId = $request->leave_type_id;
            $requiredFrom = $request->required_from;
            $requiredTo = $request->required_to;
            
            $staff = Staff::where('id_staff', $staffId)->first();
            if (!$staff) {
                return response()->json([
                    'success' => false,
                    'message' => 'Staff not found.'
                ], 404);
            }
            
            // Get the leave_type record
            $leaveTypeRecord = LeaveType::find($leaveTypeId);
            
            if (!$leaveTypeRecord) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found.'
                ], 404);
            }
            
            // Get staff's leave policy - check if it's not deleted
            $policyId = $staff->leave_policy_id;
            $allowedCount = 0;
            $requestedYear = date('Y', strtotime($requiredFrom)); // Get year from requested date
            $policyName = 'No Policy';
            $policyLeaves = 0;
            
            if ($policyId) {
                // Check if policy exists and is not deleted
                $policy = DB::table('leave_policies')
                    ->where('id_leave_policy', $policyId)
                    ->whereNull('deleted_at') // IMPORTANT: Check if policy is not deleted
                    ->first();
                
                if ($policy) {
                    $policyName = $policy->policy_name;
                    
                    // Get allowed leaves from policy definition using leave_type_id
                    $policyDefinition = DB::table('leave_policies_definition')
                        ->where('leave_policy_id', $policyId)
                        ->where('leave_type_id', $leaveTypeId)
                        ->first();
                    
                    if ($policyDefinition) {
                        $allowedCount = $policyDefinition->allowed_count;
                        $policyLeaves = $policyDefinition->allowed_count;
                    }
                } else {
                    // Policy is deleted, reset policy ID
                    $policyId = null;
                    $policyName = 'Policy Deleted';
                }
            }
            
            // Calculate consumed leaves for the REQUESTED YEAR (not current year)
            $consumedLeaves = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('leave_type_id', $leaveTypeId)
                ->where('staff_leave_status', 'Active')
                ->whereYear('staff_leave_date', $requestedYear) // Use requested year
                ->count();
            
            // Calculate remaining leaves
            $remainingLeaves = max(0, $allowedCount - $consumedLeaves);
            
            // Also check if there are overlapping leaves in the requested date range
            $overlappingLeaves = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('leave_type_id', $leaveTypeId)
                ->where('staff_leave_status', 'Active')
                ->whereBetween('staff_leave_date', [$requiredFrom, $requiredTo])
                ->count();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'staff_id' => $staffId,
                    'staff_name' => $staff->staff_fullname,
                    'leave_type' => $leaveTypeRecord->leave_type,
                    'leave_type_id' => $leaveTypeId,
                    'allowed_count' => $allowedCount,
                    'consumed_leaves' => $consumedLeaves,
                    'remaining_leaves' => $remainingLeaves,
                    'policy_name' => $policyName,
                    'policy_id' => $policyId,
                    'policy_leaves' => $policyLeaves,
                    'requested_year' => $requestedYear,
                    'overlapping_leaves' => $overlappingLeaves,
                    'requested_from' => $requiredFrom,
                    'requested_to' => $requiredTo
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching leave balance: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave balance: ' . $e->getMessage()
            ], 500);
        }
    }

    /* Update leave application*/
   public function updateLeaveApplication(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'txtleaveid' => 'required|exists:leave_applications,id_leave_applications',
                'txtstaffid' => 'required|exists:staff,id_staff',
                'txt_leave_type' => 'required',
                'txtreqfrom' => 'required|date',
                'txtreqto' => 'required|date|after_or_equal:txtreqfrom',
                'txt_leave_status' => 'required|in:Pending,Approved,Rejected',
                'status_reason' => 'required_if:txt_leave_status,Approved,Rejected|nullable|string',
                'txt_comment' => 'nullable|string',
            ]);
            
            $applicationId = $request->txtleaveid;
            $staffId = $request->txtstaffid;
            $newStatus = $request->txt_leave_status;
            
            // Check permissions
           $isHoUser = session('ho') === 'Yes';
            $application = DB::table('leave_applications')
                ->join('staff', 'leave_applications.staff_id', '=', 'staff.id_staff')
                ->where('leave_applications.id_leave_applications', $applicationId)
                ->select('leave_applications.*', 'staff.business_id as staff_business_id', 'staff.leave_policy_id')
                ->first();
            
            if (!$application) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave application not found.'
                ], 404);
            }
            
            if (!$isHoUser && $application->staff_business_id != session('business_id')) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to update this leave application.'
                ], 403);
            }
            
            // ========== ADDED CHECK: If staff doesn't have leave policy assigned ==========
            if (empty($application->leave_policy_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please assign a leave policy to this staff member first.'
                ], 400);
            }
            // =================================================================================
            
          
            $leaveTypeId = $request->txt_leave_type;
            $oldStatus = $application->application_status;
            $userName = session('user_name');
            

            if ($newStatus == 'Approved' && $application->application_status != 'Approved') {
                $existingLeaves = DB::table('staff_leaves')
                    ->where('staff_id', $staffId)
                    ->where('staff_leave_status', 'Active')
                    ->where('leave_type_id', $leaveTypeId)
                    ->whereBetween('staff_leave_date', [$request->txtreqfrom, $request->txtreqto])
                    ->exists();
                
                if ($existingLeaves) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Active leaves already exist for these dates.'
                    ], 400);
                }
            }
            



            // Prepare update data
            $updateData = [
                'leave_type' => $request->txt_leave_type,
                'leave_type_id' => $leaveTypeId,
                'required_from' => $request->txtreqfrom,
                'required_to' => $request->txtreqto,
                'comments' => $request->txt_comment,
                'application_status' => $newStatus,
            ];
            
            // If status is changing to Approved or Rejected
            if (in_array($newStatus, ['Approved', 'Rejected'])) {
                $updateData['approved_by'] = $userName;
                $updateData['approved_on'] = now();
                
                // If approving, create staff leave records
                if ($newStatus == 'Approved' && $oldStatus != 'Approved') {
                    $this->createStaffLeaveRecords($staffId, $request, $leaveTypeId);
                }
                
                // If status changed from Approved to something else, remove staff leave records for this date range
                if ($oldStatus == 'Approved' && $newStatus != 'Approved') {
                    $this->removeStaffLeaveRecords($staffId, $request, $leaveTypeId);
                }
            }
            
            // Update the application
            DB::table('leave_applications')
                ->where('id_leave_applications', $applicationId)
                ->update($updateData);
            
            DB::commit();
            
            // Get updated application
            $updatedApplication = DB::table('leave_applications')
                ->where('id_leave_applications', $applicationId)
                ->first();
            
            return response()->json([
                'success' => true,
                'message' => 'Leave application updated successfully.',
                'data' => $updatedApplication
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating leave application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error updating leave application: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Create staff leave records for approved application*/
    private function createStaffLeaveRecords($staffId, $request, $leaveTypeId)
    {
        try {
            $startDate = new \DateTime($request->txtreqfrom);
            $endDate = new \DateTime($request->txtreqto);
            $interval = new \DateInterval('P1D');
            $dateRange = new \DatePeriod($startDate, $interval, $endDate->modify('+1 day'));
            
            $leaveType = LeaveType::find($leaveTypeId);
            $leaveTypeName = $leaveType ? $leaveType->leave_type : 'Unknown';
            
            $leaveRecords = [];
            foreach ($dateRange as $date) {
                $leaveRecords[] = [
                    'staff_id' => $staffId,
                    'staff_leave_date' => $date->format('Y-m-d'),
                    'leave_type' => $leaveTypeName,
                    'leave_type_id' => $leaveTypeId,
                    'leave_reason' => $request->txt_comment,
                    'staff_leave_status' => 'Active',
                    'created_by' => session('user_name'),
                    'created_on' => now(),
                ];
            }
            
            if (!empty($leaveRecords)) {
                DB::table('staff_leaves')->insert($leaveRecords);
            }
            
        } catch (\Exception $e) {
            Log::error('Error creating staff leave records: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /*** Display staff payment calculator*/
    public function staffPaymentCalculator(Request $request)
    {
       $isHoUser = session('ho') === 'Yes';
        if ($isHoUser) {
            $businessId = $request->business_id;
        }else{
            $businessId = session('business_id');
        }
        
        if ($isHoUser && session('selected_business_id')) {
            $businessId = session('selected_business_id');
        }else{
             $businessId = session('business_id');
        }
        
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        if ($isHoUser) {
            if ($request->filled('business_id')) {
                $selectedBusinessId = $request->business_id;
            } elseif (session('selected_business_id')) {
                $selectedBusinessId = session('selected_business_id');
            } else {
                $selectedBusinessId = $businessId;
            }

            $staff = Staff::where('business_id', $selectedBusinessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();

        } else {
            $staff = Staff::where('business_id', $businessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();
        }
        
        // Get month, year, and staff from request or use defaults
        if ($request->has('month')) {
            $month = (int) $request->input('month');
            $year = (int) $request->input('year');
            $staffId = (int) $request->input('staff', 0);
        } else {
            // Get previous month as default
            $today = new \DateTime();
            $lastMonth = clone $today;
            $lastMonth->modify('-1 month');
            
            $month = (int) $lastMonth->format('m');
            $year = (int) $lastMonth->format('Y');
            $staffId = 0;
        }

        $checkHo_accounts = Business::find($businessId);

        if ($checkHo_accounts && $checkHo_accounts->ho_accounts === 'Yes') {
            $getHoBusinessId = Business::where('ho', 'Yes')->pluck('id_business');
            $businessIdForQuery = $getHoBusinessId[0];
        } else {
           $businessIdForQuery = $businessId;
        }


        // Get allowances for the business
        $allowances = Allowance::where('business_id', $businessIdForQuery)->get();
        
        $payables = [];
        $selectedStaff = null;
        
        // Check if date is valid and staff is selected
        if ($staffId > 0) {
            $selectedStaff = Staff::find($staffId);
            $selected_staffBsuinessId = $selectedStaff->business_id;
            if ($selectedStaff) {
                // Get all data in a single comprehensive query
               $staffPayable = $this->calculateStaffMonthPayables($staffId, $year, $month, $selectedStaff,$businessIdForQuery);
                
                if (!empty($staffPayable)) {
                    // Get commissions
                    
                    // errror in this will fixed this after getting commision logic
                    $commission = $this->calculateServiceCommission($staffId, $year, $month, $selectedStaff, $businessIdForQuery);
                    $retailCommission = $this->calculateRetailCommission($staffId, $year, $month, $selectedStaff, $businessIdForQuery);
                    
                    // Get allowances for staff
                    $staffAllowances = $this->getStaffMonthAllowances($staffId);
                    $loanData = $this->calculateLoanDeductions($staffId,$selected_staffBsuinessId, $year, $month, $businessIdForQuery);
                    $record = array_merge(
                        [
                            'id_staff' => $staffPayable['id_staff'] ?? null,
                            'staff_fullname' => $staffPayable['staff_fullname'] ?? 'N/A',
                            'business_name' => $staffPayable['business_name'] ?? 'N/A',
                            'staff_salary' => $staffPayable['staff_salary'] ?? 0,
                            'staff_salary_type' => $staffPayable['salary_type'] ?? 'monthly', 
                            'Present' => $staffPayable['present_days'] ?? 0,
                            'Absent' => $staffPayable['absent_days'] ?? 0,
                            'Late' => $staffPayable['late_days'] ?? 0,
                            'WorkHours' => $staffPayable['work_hours'] ?? 0, 
                            'calculated_salary' => $staffPayable['gross_salary'] ?? 0,
                            'total_overtime_amount' => $staffPayable['total_overtime_amount'] ?? 0,
                            'late_deduction' => $staffPayable['late_deduction'] ?? 0,
                            'absent_deduction' =>$staffPayable['absent_deduction'] ?? 0,
                            'provident_fund' => $staffPayable['provident_fund'] ?? 0,
                            'regular_shift_hours' => $staffPayable['regular_shift_hours'] ?? 8,
                            'hourly_rate' => $staffPayable['hourly_rate'] ?? 0,
                            'overtime_rate' => $staffPayable['overtime_rate'] ?? 0,
                            'holiday_days' => $staffPayable['holiday_days'] ?? 0,
                            'weekend_days' => $staffPayable['weekend_days'] ?? 0,
                            'daily_logs' => $staffPayable['daily_logs'] ?? [],
                            'calculation_log' => $staffPayable['calculation_log'] ?? [],
                        ],
                        $commission,
                        $retailCommission,
                        $loanData,
                        ['allowances' => $staffAllowances]
                    );
                    $payables[] = $record;
                }
            }
        }

        // Get month name
        $dateObj = \DateTime::createFromFormat('!m', $month);
        $monthName = $dateObj->format('F');

        return view('hrm.staff_payment_calculator', compact(
            
            'allowances',
            'staff',
            'payables',
            'month',
            'year',
            'staffId',
            'monthName',
            'businesses',
            'selectedStaff',
            'businessId',
            'selectedBusinessId'
        ));
    }
    
    private function calculateStaffMonthPayables($staffId, $year, $month, $staff,$businessIdForQuery)
    {
        try {
            /* =====================================================
            BASIC DATES
            ===================================================== */
            $totalDaysInMonth = date("t", strtotime("$year-$month-01"));
            $startDate = "$year-$month-01";
            $endDate   = date("Y-m-t", strtotime($startDate));

            /* =====================================================
            STAFF + BUSINESS INFO
            ===================================================== */
            $result = DB::table('staff as s')
                ->where('s.id_staff', $staffId)
                ->leftJoin('deduction_policy as dp', 'dp.id_deduction_policy', '=', 's.deduction_policy')
                ->leftJoin('business as b', 'b.id_business', '=', 's.business_id')
                ->select(
                    's.*',
                    'dp.deduction_days_absent',
                    'dp.deduction_days_late',
                    'dp.total_late_allowed',
                    'dp.total_absents_allowed',
                    'b.timein_margin',
                    'b.business_name',
                    'b.business_closing_time'
                )
                ->first();

            if (!$result) return [];

            /* =====================================================
            ROUNDING RULE (.5 UP)
            ===================================================== */
            $roundVal = fn($v) => (int) round($v, 0, PHP_ROUND_HALF_UP);

            /* =====================================================
            REGULAR SHIFT HOURS
            ===================================================== */
            $regularShiftHours = 8;
            if ($result->day_start_time && $result->day_end_time) {
                $s = strtotime($result->day_start_time);
                $e = strtotime($result->day_end_time);
                if ($e < $s) $e += 86400;
                $regularShiftHours = ($e - $s) / 3600;
            }

            /* =====================================================
            WEEK OFF / HOLIDAYS
            ===================================================== */
            $weekOffs = array_map('trim', explode(',', $result->week_day_off ?? ''));
            
            // Get all dates in the month
            $allDates = [];
            $currentDate = new \DateTime($startDate);
            $endDateTime = new \DateTime($endDate);
            
            while ($currentDate <= $endDateTime) {
                $allDates[] = $currentDate->format('Y-m-d');
                $currentDate->modify('+1 day');
            }

            $holidays = DB::table('holidays')
                ->where('business_id', $businessIdForQuery)
                ->whereBetween('holiday_date', [$startDate, $endDate])
                ->pluck('holiday_date')
                ->map(fn($d) => date('Y-m-d', strtotime($d)))
                ->toArray();

            /* =====================================================
            ATTENDANCE
            ===================================================== */
            $attendances = DB::table('staff_attendance')
                ->where('staff_id', $staffId)
                ->whereBetween('time_in', [$startDate.' 00:00:00', $endDate.' 23:59:59'])
                ->get();

            /* =====================================================
            COUNTERS
            ===================================================== */
            $presentDays = 0;
            $lateDays = 0;
            $absentDays = 0;
            $actualHolidayDays = 0;
            $actualWeekendDays = 0;
            $totalWorkedMinutes = 0;
            
            // Track which dates have attendance
            $attendanceDates = [];
            foreach ($attendances as $att) {
                $attendanceDates[] = date('Y-m-d', strtotime($att->time_in));
            }

            /* =====================================================
            SALARY RATES
            ===================================================== */
            $salary = 0;
            $daySalary = 0;
            $hourlyRate = 0;

            if ($result->salary_type === 'monthly') {
                $salary = $result->staff_salary;
                $daySalary = $salary / $totalDaysInMonth;
                $hourlyRate = $daySalary / $regularShiftHours;
            } elseif ($result->salary_type === 'daily') {
                $daySalary = $result->staff_salary;
                $hourlyRate = $daySalary / $regularShiftHours;
            } elseif (in_array($result->salary_type, ['hourly', 'weekly'])) {
                $hourlyRate = $result->staff_salary;
            }

            /* =====================================================
            OVERTIME CONFIG
            ===================================================== */
            $overtimeAllowed = ($result->overtime_allowed === 'Yes');
            $overtimeRate = $result->overtime_hourly_rate ?? $hourlyRate;

            $workingDayOTHours = 0;
            $holidayOTHours = 0;
            $workingDayOTAmount = 0;
            $holidayOTAmount = 0;

            $dailyLogs = [];

            /* =====================================================
            MAIN LOOP - Process attendance records
            ===================================================== */
            foreach ($attendances as $att) {
                $date = date('Y-m-d', strtotime($att->time_in));
                $dayName = date('l', strtotime($date));
                $dayShort = substr($dayName, 0, 3);

                $isHoliday = in_array($date, $holidays);
                $isWeekend = in_array($dayName, $weekOffs) || in_array($dayShort, $weekOffs);

                $timeIn = strtotime($att->time_in);
                $timeOut = $att->time_out
                    ? strtotime($att->time_out)
                    : strtotime($date.' '.($result->business_closing_time ?? $result->day_end_time));

                if ($timeOut <= $timeIn) continue;

                $workedMinutes = ($timeOut - $timeIn) / 60;
                $workedHours = $workedMinutes / 60;
                $totalWorkedMinutes += $workedMinutes;

                $log = [
                    'date' => $date,
                    'day' => $dayName,
                    'time_in' => date('H:i', $timeIn),
                    'time_out' => date('H:i', $timeOut),
                    'worked_hours' => round($workedHours, 2),
                    'late' => false,
                    'overtime_hours' => 0,
                    'type' => 'Working Day'
                ];

                if ($isHoliday || $isWeekend) {
                    $log['type'] = $isHoliday ? 'Holiday' : 'Weekend';

                    if ($isHoliday) $actualHolidayDays++;
                    if ($isWeekend) $actualWeekendDays++;

                    if ($overtimeAllowed) {
                        $holidayOTHours += $workedHours;
                        $holidayOTAmount += ($workedHours * $overtimeRate);
                    }

                    // $dailyLogs[] = $log;
                    continue;
                }

                /* ===== WORKING DAY ===== */
                $presentDays++;

                $margin = ($result->timein_margin ?? 0) * 60;
                $start = strtotime($date.' '.$result->day_start_time);

                if ($timeIn > ($start + $margin)) {
                    $lateDays++;
                    $log['late'] = true;
                }

                $dayEnd = strtotime($date.' '.$result->day_end_time);
                if ($overtimeAllowed && $timeOut > $dayEnd) {
                    $ot = ($timeOut - $dayEnd) / 3600;
                    $workingDayOTHours += $ot;
                    $workingDayOTAmount += ($ot * $overtimeRate);
                }

                // $dailyLogs[] = $log;
            }

            /* =====================================================
            CALCULATE ABSENT DAYS - NEW LOGIC
            ===================================================== */
            // For monthly staff only, calculate absents based on working days
            if ($result->salary_type === 'monthly') {
                $totalWorkingDays = 0;
                $absentDays = 0;
                
                // Loop through all dates in the month
                foreach ($allDates as $date) {
                    $dayName = date('l', strtotime($date));
                    $dayShort = substr($dayName, 0, 3);
                    
                    $isHoliday = in_array($date, $holidays);
                    $isWeekend = in_array($dayName, $weekOffs) || in_array($dayShort, $weekOffs);
                    
                    // Skip weekends and holidays for absent calculation
                    if (!$isHoliday && !$isWeekend) {
                        $totalWorkingDays++;
                        
                        // Check if staff has attendance for this working day
                        if (!in_array($date, $attendanceDates)) {
                            $absentDays++;
                        }
                    }
                }
                
                // Debug log
                \Log::info('Absent Calculation:', [
                    'staff_id' => $staffId,
                    'month' => $month,
                    'year' => $year,
                    'total_working_days' => $totalWorkingDays,
                    'present_days' => $presentDays,
                    'absent_days' => $absentDays,
                    'total_absents_allowed' => $result->total_absents_allowed ?? 0,
                ]);
            }

            /* =====================================================
            WORK HOURS
            ===================================================== */
            $workHours = $totalWorkedMinutes / 60;

            /* =====================================================
            FINAL SALARY (HOURLY / WEEKLY)
            ===================================================== */
            if (in_array($result->salary_type, ['hourly', 'weekly'])) {
                $salary = $hourlyRate * $workHours;
            }

            if (in_array($result->salary_type, ['daily'])) {
                $salary =  $daySalary * $presentDays;
            }

            /* =====================================================
            LATE DEDUCTION (ONLY MONTHLY & DAILY)
            ===================================================== */
            $lateDeduction = 0;
            $excessLates = 0;

            if (
                in_array($result->salary_type, ['monthly', 'daily']) &&
                ($result->total_late_allowed ?? 0) > 0 &&
                $lateDays > $result->total_late_allowed &&
                ($result->deduction_days_late ?? 0) > 0
            ) {
                $excessLates = $lateDays - $result->total_late_allowed;

                if ($result->salary_type === 'monthly') {
                    $lateDeduction =
                        ($result->staff_salary / $totalDaysInMonth)
                        * $result->deduction_days_late
                        * $excessLates;
                } else {
                    $lateDeduction =
                        $result->staff_salary
                        * $result->deduction_days_late
                        * $excessLates;
                }
            }

            $lateDeduction = $roundVal($lateDeduction);

            /* =====================================================
            ABSENT DEDUCTION (ONLY FOR MONTHLY STAFF)
            ===================================================== */
            $absentDeduction = 0;
            
            // Only calculate absent deduction for monthly staff
            if ($result->salary_type === 'monthly') {
                // Check if deduction policy allows absent deduction
                if (($result->deduction_days_absent ?? 0) > 0) {
                    // Calculate excess absents after allowed limit
                    $allowedAbsents = $result->total_absents_allowed ?? 0;
                    $excessAbsents = max(0, $absentDays - $allowedAbsents);
                    
                    \Log::info('Absent Deduction Calculation:', [
                        'staff_id' => $staffId,
                        'absent_days' => $absentDays,
                        'allowed_absents' => $allowedAbsents,
                        'excess_absents' => $excessAbsents,
                        'deduction_days_absent' => $result->deduction_days_absent,
                        'day_salary' => $daySalary,
                    ]);
                    
                    if ($excessAbsents > 0) {
                        $absentDeduction = $daySalary * $result->deduction_days_absent * $excessAbsents;
                        $absentDeduction = $roundVal($absentDeduction);
                    }
                }
            }

            /* =====================================================
            PROVIDENT FUND CALCULATION
            ===================================================== */
            $providentFundAmount = 0;
            $providentFundEmployee = 0;
            $providentFundEmployer = 0;
            
            // Get PF rate from staff record
            $providentFundRate = $result->provident_fund_rate ?? 0;
            
            // Calculate PF based on staff salary
            if ($providentFundRate > 0) {
                // Use staff's basic salary (not gross calculated salary) for PF calculation
                $baseSalaryForPF = $result->staff_salary; // Basic salary
                
                // Calculate PF amount - same logic as CodeIgniter
                $providentFundAmount = ($baseSalaryForPF * $providentFundRate) / 100;
                
                // Split equally between employee and employer
                // $providentFundEmployee = $providentFundAmount / 2;
                // $providentFundEmployer = $providentFundAmount / 2;
            }

            /* =====================================================
            RESPONSE (FULL LOG)
            ===================================================== */
            return [
                'id_staff' => $result->id_staff,
                'staff_fullname' => $result->staff_fullname,
                'business_name' => $result->business_name,

                'staff_salary' => $result->staff_salary,
                'salary_type' => $result->salary_type,

                'present_days' => $presentDays,
                'late_days' => $lateDays,
                'absent_days' => $absentDays,
                'holiday_days' => $actualHolidayDays,
                'weekend_days' => $actualWeekendDays,

                'regular_shift_hours' => round($regularShiftHours, 2),
                'hourly_rate' => round($hourlyRate, 2),
                'overtime_rate' => round($overtimeRate, 2),

                'work_hours' => $workHours,
                'gross_salary' => $salary,

                'late_deduction' => $lateDeduction,
                'absent_deduction' => round($absentDeduction, 2),

                // PF data
                'provident_fund' => round($providentFundAmount, 2),
                // 'provident_fund_employee' => round($providentFundEmployee, 2),
                // 'provident_fund_employer' => round($providentFundEmployer, 2),

                'total_overtime_hours' => $workingDayOTHours + $holidayOTHours,
                'total_overtime_amount' => $workingDayOTAmount + $holidayOTAmount,

                'daily_logs' => $dailyLogs,

                'calculation_log' => [
                    'total_days_in_month' => $totalDaysInMonth,
                    'total_working_days' => $totalWorkingDays ?? 0,
                    'total_late_allowed' => $result->total_late_allowed ?? 0,
                    'total_absents_allowed' => $result->total_absents_allowed ?? 0,
                    'excess_lates' => $excessLates,
                    'excess_absents' => $excessAbsents ?? 0,
                    'late_deduction_multiplier' => $result->deduction_days_late ?? 0,
                    'absent_deduction_multiplier' => $result->deduction_days_absent ?? 0,
                    'attendance_dates_count' => count($attendanceDates),
                    'all_dates_count' => count($allDates),
                ]
            ];

        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return [];
        }
    }

    
    private function calculateServiceCommission($staffId, $year, $month, $staff, $businessIdForQuery)
    {
        try {
            $staffInfo = DB::table('staff as s')
                ->where('s.id_staff', $staffId)
                ->select(
                    's.id_staff',
                    's.staff_fullname',
                    's.staff_image',
                    's.staff_salary',
                    's.commission_mode',
                    's.commission_target',
                    's.staff_commission_perc',
                    's.commission_fixed_amount'
                )
                ->first();
            
            if (!$staffInfo) {
                return [
                    'service_income' => 0,
                    'service_commission' => 0,
                    'commission_mode' => 'Services',
                    'commission_rate' => 0,
                    'commission_target' => 0,
                    'commission_fixed_amount' => 0
                ];
            }
            
            $monthStartDate = "$year-$month-01";
            $monthEndDate = date("Y-m-t", strtotime($monthStartDate));
            
            // Get total service sales
            $serviceSales = DB::table('invoice_staff as is')
                ->join('invoice as i', 'i.id_invoice', '=', 'is.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('is.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    DB::raw('COALESCE(SUM(is.discounted_price), 0) as total_discounted_price'),
                    DB::raw('COALESCE(SUM(is.final_price), 0) as total_final_price'),
                    DB::raw('COALESCE(SUM(is.commission), 0) as total_commission'),
                    DB::raw('COALESCE(SUM(is.paid), 0) as total_paid')
                )
                ->first();
            
            // Get retail sales to check if retail invoices exist
            $retailSales = DB::table('invoice_products as ip')
                ->join('invoice as i', 'i.id_invoice', '=', 'ip.invoice_id')
                ->where('i.invoice_status', 'valid')
                ->where('ip.staff_id', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    DB::raw('COALESCE(SUM(ip.discounted_price), 0) as total_discounted_price')
                )
                ->first();
            
            $serviceSalesTotal = $serviceSales->total_discounted_price ?? 0;
            $retailSalesTotal = $retailSales->total_discounted_price ?? 0;
            
            $totalCommission = 0;
            $hasServiceInvoices = $serviceSalesTotal > 0;
            $hasRetailInvoices = $retailSalesTotal > 0;
            
            // Apply commission logic based on mode
            if ($staffInfo->commission_mode === 'Services') {
                $totalCommission = $serviceSales->total_commission ?? 0;
            } 
            elseif ($staffInfo->commission_mode === 'Target') {
                // Check if target is met
                if ($serviceSalesTotal >= $staffInfo->commission_target) {
                    // Target is met (service alone)
                    if (!empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0) {
                        // Fixed amount commission
                        if ($hasRetailInvoices) {
                            // Both have invoices - split 50/50
                            $totalCommission = $staffInfo->commission_fixed_amount / 2;
                        } else {
                            // Only services have invoices - full amount to services
                            $totalCommission = $staffInfo->commission_fixed_amount;
                        }
                    } else {
                        // Percentage based commission
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        $totalCommission = ($serviceSalesTotal * $commissionRate) / 100;
                    }
                } 
                // Check collective achievement
                elseif (($serviceSalesTotal + $retailSalesTotal) >= $staffInfo->commission_target) {
                    // Collectively achieved target
                    if (!empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0) {
                        if ($hasServiceInvoices && $hasRetailInvoices) {
                            // Both have invoices - split 50/50
                            $totalCommission = $staffInfo->commission_fixed_amount / 2;
                        } elseif ($hasServiceInvoices) {
                            // Only services have invoices - full amount to services
                            $totalCommission = $staffInfo->commission_fixed_amount;
                        }
                        // Note: Retail-only case is handled in retailCommission method
                    } else {
                        // Percentage based - calculate proportionally
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        $collectiveTotal = $serviceSalesTotal + $retailSalesTotal;
                        $serviceProportion = $serviceSalesTotal / max($collectiveTotal, 1);
                        $totalCommission = (($collectiveTotal * $commissionRate) / 100) * $serviceProportion;
                    }
                }
                // If target not met, commission remains 0
            }
            
            return [
                'service_income' => round($serviceSalesTotal, 2),
                'service_final_price' => round($serviceSales->total_final_price ?? 0, 2),
                'service_paid' => round($serviceSales->total_paid ?? 0, 2),
                'service_commission' => round($totalCommission, 2),
                'precalculated_commission' => round($serviceSales->total_commission ?? 0, 2),
                'commission_mode' => $staffInfo->commission_mode,
                'commission_rate' => $staffInfo->staff_commission_perc,
                'commission_target' => $staffInfo->commission_target,
                'commission_fixed_amount' => $staffInfo->commission_fixed_amount ?? 0,
                'target_achieved' => ($serviceSalesTotal >= $staffInfo->commission_target) ? 'Yes' : 'No',
                'collective_target_achieved' => (($serviceSalesTotal + $retailSalesTotal) >= $staffInfo->commission_target) ? 'Yes' : 'No',
                'has_service_invoices' => $hasServiceInvoices,
                'has_retail_invoices' => $hasRetailInvoices,
                'total_discounted_price' => round($serviceSalesTotal, 2)
            ];
            
        } catch (\Exception $e) {
            Log::error('Error calculating service commission: ' . $e->getMessage());
            return [
                'service_income' => 0,
                'service_commission' => 0,
                'commission_mode' => 'Services',
                'commission_rate' => 0,
                'commission_target' => 0,
                'commission_fixed_amount' => 0
            ];
        }
    }

    private function calculateRetailCommission($staffId, $year, $month, $staff, $businessIdForQuery)
    {
        try {
            $staffInfo = DB::table('staff as s')
                ->where('s.id_staff', $staffId)
                ->select(
                    's.id_staff',
                    's.staff_fullname',
                    's.staff_image',
                    's.staff_salary',
                    's.commission_mode',
                    's.commission_target',
                    's.staff_commission_perc',
                    's.commission_fixed_amount'
                )
                ->first();
            
            if (!$staffInfo) {
                return [
                    'retail_income' => 0,
                    'retail_commission' => 0,
                    'retail_commission_mode' => 'Services',
                    'retail_commission_rate' => 0,
                    'commission_target' => 0,
                    'commission_fixed_amount' => 0
                ];
            }
            
            $monthStartDate = "$year-$month-01";
            $monthEndDate = date("Y-m-t", strtotime($monthStartDate));
            
            // Get retail sales data
            $result = DB::table('invoice_products as ip')
                ->join('invoice as i', 'i.id_invoice', '=', 'ip.invoice_id')
                ->where('i.invoice_status', 'valid')
                ->where('ip.staff_id', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    DB::raw('COALESCE(SUM(ip.discounted_price), 0) as total_discounted_price'),
                    DB::raw('COALESCE(SUM(ip.final_price), 0) as total_final_price'),
                    DB::raw('COALESCE(SUM(ip.commission), 0) as total_commission'),
                    DB::raw('COALESCE(SUM(ip.paid), 0) as total_paid')
                )
                ->first();
            
            // Get service sales to check if service invoices exist
            $serviceSales = DB::table('invoice_staff as is')
                ->join('invoice as i', 'i.id_invoice', '=', 'is.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('is.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    DB::raw('COALESCE(SUM(is.discounted_price), 0) as total_discounted_price')
                )
                ->first();
        
            $retailSalesTotal = $result->total_discounted_price ?? 0;
            $serviceSalesTotal = $serviceSales->total_discounted_price ?? 0;
            
            $retailCommission = 0;
            $hasRetailInvoices = $retailSalesTotal > 0;
            $hasServiceInvoices = $serviceSalesTotal > 0;
            
            // Apply commission logic based on mode
            if ($staffInfo->commission_mode === 'Services') {
                $retailCommission = $result->total_commission ?? 0;
            } 
            elseif ($staffInfo->commission_mode === 'Target') {
                // Check if retail target is met
                if ($retailSalesTotal >= $staffInfo->commission_target) {
                    // Retail target is met
                    if (!empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0) {
                        if ($hasServiceInvoices) {
                            // Both have invoices - split 50/50
                            $retailCommission = $staffInfo->commission_fixed_amount / 2;
                        } else {
                            // Only retail has invoices - full amount to retail
                            $retailCommission = $staffInfo->commission_fixed_amount;
                        }
                    } else {
                        // Percentage based
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        $retailCommission = ($retailSalesTotal * $commissionRate) / 100;
                    }
                } 
                // Check collective achievement
                elseif (($retailSalesTotal + $serviceSalesTotal) >= $staffInfo->commission_target) {
                    // Collectively achieved target
                    if (!empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0) {
                        if ($hasRetailInvoices && $hasServiceInvoices) {
                            // Both have invoices - split 50/50
                            $retailCommission = $staffInfo->commission_fixed_amount / 2;
                        } elseif ($hasRetailInvoices) {
                            // Only retail has invoices - full amount to retail
                            $retailCommission = $staffInfo->commission_fixed_amount;
                        }
                        // Note: Service-only case is handled in serviceCommission method
                    } else {
                        // Percentage based - calculate proportionally
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        $collectiveTotal = $retailSalesTotal + $serviceSalesTotal;
                        $retailProportion = $retailSalesTotal / max($collectiveTotal, 1);
                        $retailCommission = (($collectiveTotal * $commissionRate) / 100) * $retailProportion;
                    }
                }
                // If target not met, commission remains 0
            }
            
            return [
                'retail_income' => round($retailSalesTotal, 2),
                'retail_final_price' => round($result->total_final_price ?? 0, 2),
                'retail_paid' => round($result->total_paid ?? 0, 2),
                'retail_commission' => round($retailCommission, 2),
                'precalculated_commission' => round($result->total_commission ?? 0, 2),
                'retail_commission_mode' => $staffInfo->commission_mode,
                'retail_commission_rate' => $staffInfo->staff_commission_perc,
                'commission_target' => $staffInfo->commission_target,
                'commission_fixed_amount' => $staffInfo->commission_fixed_amount ?? 0,
                'target_achieved' => ($retailSalesTotal >= $staffInfo->commission_target) ? 'Yes' : 'No',
                'collective_target_achieved' => (($retailSalesTotal + $serviceSalesTotal) >= $staffInfo->commission_target) ? 'Yes' : 'No',
                'has_retail_invoices' => $hasRetailInvoices,
                'has_service_invoices' => $hasServiceInvoices,
                'total_discounted_price' => round($retailSalesTotal, 2)
            ];
            
        } catch (\Exception $e) {
            Log::error('Error calculating retail commission: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return [
                'retail_income' => 0,
                'retail_commission' => 0,
                'retail_commission_mode' => 'Services',
                'retail_commission_rate' => 0,
                'commission_target' => 0,
                'commission_fixed_amount' => 0
            ];
        }
    }


    private function calculateLoanDeductions($staffId,$selected_staffBsuinessId, $year = null, $month = null,$businessIdForQuery)
    {
        try {
           
            
            // Get loan account head ID from mapping 
            $loanAccountHead = DB::table('account_event_mapping')
                ->where('account_event_id', 25)
                ->where('entity_name', 'loan_amount')
                ->where('business_id', $businessIdForQuery)
                ->first();
            
            $loanAccountHeadId = $loanAccountHead ? $loanAccountHead->account_head_id : 0;
            
            if ($loanAccountHeadId == 0) {
                return [
                    'loan_deduction' => 0,
                    'remaining_loan_amount' => 0,
                    'loan_amount' => 0,
                    'active_loans' => []
                ];
            }

            $loanResult = DB::table('account_vouchers as av')
                ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
                ->select(DB::raw('COALESCE(SUM(COALESCE(avd.debit, 0)) - SUM(COALESCE(avd.credit, 0)), 0) as remaining_loan_amount'))
                ->where('av.voucher_status', 'Active')
                ->where('av.business_partner', 3) 
                ->where('av.business_partner_id', $staffId)
                ->where('avd.account_head_id', $loanAccountHeadId)
                ->where('av.business_id', $selected_staffBsuinessId)
                ->first();


          
            $remainingLoanAmount = $loanResult ? $loanResult->remaining_loan_amount : 0;
            $loanDeduction = 0;
            $activeLoans = [];

            $loans = DB::table('loan_applications')
                ->where('staff_id', $staffId)
                ->where('application_status', 'Approved')
                ->where('num_of_months', '>', 0)
                ->get();
            
            foreach ($loans as $loan) {
                if ($year && $month) {
                    $existingDeduction = DB::table('account_vouchers as av')
                        ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
                        ->where('av.business_partner', 3)
                        ->where('av.business_partner_id', $staffId)
                        ->where('avd.account_head_id', $loanAccountHeadId)
                        ->whereYear('av.voucher_date', $year)
                        ->whereMonth('av.voucher_date', $month)
                        ->where('avd.credit', '>', 0)
                        ->where('av.voucher_type', 3)
                        ->where('av.business_id',$selected_staffBsuinessId)
                        ->first();
                    if (!$existingDeduction && $remainingLoanAmount > 0) {
                        $monthlyInstallment = $loan->installment ?? ($remainingLoanAmount / max($loan->num_of_months, 1));
                        $currentDeduction = min($monthlyInstallment, $remainingLoanAmount);
                        
                       

                        if ($currentDeduction > 0) {
                            $loanDeduction += $currentDeduction;
                            
                            $activeLoans[] = [
                                'loan_id' => $loan->id_loan_applications,
                                'loan_amount' => $loan->approved_amount,
                                'monthly_installment' => $monthlyInstallment,
                                'current_deduction' => $currentDeduction,
                                'remaining_amount' => $remainingLoanAmount - $currentDeduction
                            ];
                        }
                    }
                }
            }

            if($currentDeduction > $remainingLoanAmount){
                $currentDeduction = $remainingLoanAmount;
            }
            
            return [
                'loan_deduction' => round($loanDeduction, 2),
                'remaining_loan_amount' => round($remainingLoanAmount, 2),
                'loan_amount' => round($remainingLoanAmount, 2),
                'active_loans' => $activeLoans
            ];
            
        } catch (\Exception $e) {
            Log::error('Error calculating loan deductions: ' . $e->getMessage());
            return [
                'loan_deduction' => 0,
                'remaining_loan_amount' => 0,
                'loan_amount' => 0,
                'active_loans' => []
            ];
        }
    }

   /* Get staff monthly allowances*/
    private function getStaffMonthAllowances($staffId)
    {
        try {
            return DB::table('staff_allowance as sa')
                ->join('allowances as a', 'a.id_allowances', '=', 'sa.allowance_id')
                ->where('sa.staff_id', $staffId)
                ->where('sa.staff_allowance_status', 'Active')
                ->select('a.allowance_name', 'sa.allowance_amount', 'a.entity_name')
                ->get()
                ->map(function ($item) {
                    return (array) $item;
                })
                ->toArray();
            
        } catch (\Exception $e) {
            Log::error('Error getting staff allowances: ' . $e->getMessage());
            return [];
        }
    }

    public function checkPayable(Request $request)
    {
        DB::beginTransaction();
        
        try {
            // Get input data
            $businessId = session('business_id');
            $month = $request->input('month');
            $year = $request->input('year');
            $voucherDate = $request->input('voucher_date');
            $voucherCarbon = \Carbon\Carbon::parse($voucherDate);
            $voucherMonth  = $voucherCarbon->month;
            $voucherYear   = $voucherCarbon->year;
            $userName = session('user_name');
            $tableData = json_decode($request->input('TableData'), true);
            $salaryDate = \Carbon\Carbon::createFromDate($year, $month, 1);
            $monthName = $salaryDate->format('F');
            // Validate input
            if (!$tableData || !is_array($tableData)) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid data provided.'
                ], 400);
            }
            
            if (!$businessId || !$month || !$year || !$voucherDate) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Missing required parameters.'
                ], 400);
            }

            $isHoUser = session('ho') === 'Yes';
            $eventbusinessId = 0;
            if ($isHoUser) {
                $eventbusinessId =  $businessId;
            }else{
                $checkhoAccounts = session('ho_accounts');
                if($checkhoAccounts === 'Yes'){
                    $getHoBusinessId = Business::where('ho', 'Yes')->pluck('id_business');
                    $eventbusinessId = $getHoBusinessId[0];
                }else{
                    $eventbusinessId = $businessId;
                    
                }
            }
            $accountHeads = DB::table('account_event_mapping')
                ->where('account_event_id', 26)
                ->where('business_id', $eventbusinessId)
                ->orderBy('id_account_event_mapping', 'asc')
                ->get();
            
            if ($accountHeads->isEmpty()) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Account event is not mapped for salary payable (Event ID: 26)'
                ], 400);
            }
            
            $createdCount = 0;
            $errors = [];
            $alreadyExists = [];
            $successStaff = [];
            foreach ($tableData as $index => $row) {
                try {
                    // Validate required fields
                    if (empty($row['staff_id']) || empty($row['staff_fullname'])) {
                        $errors[] = "Row {$index}: Missing staff ID or name";
                        continue;
                    }
                    
                    $staffId = $row['staff_id'];
                    $staffName = $row['staff_fullname'];
                    if ($isHoUser) {
                        $businessId = Staff::where('id_staff', $staffId)->value('business_id'); 
                        }
                    // Check if payment already exists for this month/year
                    $existingVoucher = DB::table('account_vouchers')
                        ->where('business_partner_id', $staffId)
                        ->where('business_partner', 3)
                        ->where('voucher_status', 'Active')
                        ->whereYear('voucher_date', $voucherYear) 
                        ->whereMonth('voucher_date', $voucherMonth)
                        ->where(function($query) {
                            $query->where('description', 'like', '%salary payable%')
                                ->orWhere('description', 'like', '%payable%')
                                ->orWhere('description', 'like', '%Salary Payable%')
                                ->orWhere('description', 'like', '%Payable%');
                        })
                        ->first(['id_account_vouchers', 'voucher_date', 'description']);
                        
                    if ($existingVoucher) {
                        $voucherId = $existingVoucher->id_account_vouchers;
                        $alreadyExists[] = "Staff {$staffName}: Salary payment already recorded for {$month}/{$year} ";
                        continue;
                    }
                    
                    $adjustment = isset($row['adjustment_amount']) ? (float)$row['adjustment_amount'] : 0;
                    $finalPayable = isset($row['final_payable']) ? (float)$row['final_payable'] : 0;
                    
                    // === CREATE VOUCHER ===
                    $voucherId = DB::table('account_vouchers')->insertGetId([
                        'business_id' => $businessId,
                        'description' => "Salary Payable booked for {$staffName} (ID: {$staffId}) on {$voucherDate} for the Month {$monthName} {$year}",
                        'voucher_amount' => $finalPayable,
                        'voucher_date' => $voucherDate,
                        'voucher_status' => 'Active',
                        'created_by' => $userName,
                        'created_on' => now(),
                        'voucher_type' => 3,
                        'cost_center' => 1,
                        'cost_center_name' => 'Main Outlet',
                        'business_partner' => 3,
                        'business_partner_id' => $staffId,
                        'business_partner_name' => $staffName,
                        'payment_mode' => 'Bank Transfer',
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                    
                    // Create voucher details
                    $voucherDetails = [];
                    $debitAccounts = [];
                    $creditAccounts = [];
                    
                    // Get adjustment account head info
                    $adjustmentAccount = null;
                    foreach ($accountHeads as $account) {
                        if (trim($account->entity_name) == "rounded_adjustment") {
                            $adjustmentAccount = [
                                'account_head_id' => $account->account_head_id,
                                'transaction_type' => strtolower($account->transaction_type),
                                'entity_name' => trim($account->entity_name)
                            ];
                            break;
                        }
                    }
                    
                    if (!$adjustmentAccount) {
                        throw new \Exception("Adjustment account (rounded_adjustment) not found in mapping");
                    }
                    
                    // Separate debit and credit accounts
                    foreach ($accountHeads as $account) {
                        $entity = trim($account->entity_name);
                        $transactionType = isset($account->transaction_type) ? strtolower($account->transaction_type) : '';
                        
                        // Skip rounded_adjustment for now
                        if ($entity == "rounded_adjustment") {
                            continue;
                        }
                        
                        if ($transactionType == 'debit') {
                            $debitAccounts[] = [
                                'account_head_id' => $account->account_head_id,
                                'entity_name' => $entity,
                                'description' => $account->description ?? ''
                            ];
                        } else {
                            $creditAccounts[] = [
                                'account_head_id' => $account->account_head_id,
                                'entity_name' => $entity,
                                'description' => $account->description ?? ''
                            ];
                        }
                    }
                    
                    // === PROCESS DEBIT ENTRIES (Expenses) ===
                    foreach ($debitAccounts as $debitAccount) {
                        $amount = 0;
                        $entity = $debitAccount['entity_name'];

                        // Map fields exactly as they come from frontend
                        if ($entity == "salary_expense") {
                            $amount = isset($row['staff_salary']) ? (float)$row['staff_salary'] : 0;
                        }
                        elseif ($entity == "overtime_allowance") {
                            // OVERTIME ALLOWANCE - add debit entry for overtime
                            $amount = isset($row['total_overtime_amount']) ? (float)$row['total_overtime_amount'] : 0;
                        } 
                        elseif ($entity == "commission_service_expense") {
                            $amount = isset($row['service_commission']) ? (float)$row['service_commission'] : 0;
                        } 
                        elseif ($entity == "commission_retail_expense") {
                            $amount = isset($row['retail_commission']) ? (float)$row['retail_commission'] : 0;
                        } 
                        elseif ($entity == "provident_fund_employer") {
                            $amount = isset($row['provident_fund_employer']) ? (float)$row['provident_fund_employer'] : 0;
                        } 
                        elseif ($entity == "reimbursement_expense") {
                            $amount = isset($row['reimbursements']) ? (float)$row['reimbursements'] : 0;
                        } 
                        elseif ($entity == "bonus_expense") {
                            $amount = isset($row['bonus_amount']) ? (float)$row['bonus_amount'] : 0;
                        } 
                        elseif ($entity == "miscellaneous_payments") {
                            $amount = isset($row['miscellanous']) ? (float)$row['miscellanous'] : 0;
                        }
                        // Check for allowances - no calculations, just get from payload
                        elseif (strpos($entity, 'allowance') !== false) {
                            // Try to find matching allowance field in the row data
                            if (isset($row[$entity])) {
                                $amount = (float)$row[$entity];
                            }
                            // Also check for allowance fields that might match
                            else {
                                // Check all allowance fields in row
                                foreach ($row as $key => $value) {
                                    if (strpos($key, 'allowance') !== false) {
                                        // Check if this matches the entity pattern
                                        $rowEntity = $key;
                                        $entityName = str_replace('allowance_', '', $entity);
                                        if (strpos($rowEntity, $entityName) !== false) {
                                            $amount = (float)$value;
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                        
                        if ($amount > 0) {
                            $voucherDetails[] = [
                                'account_voucher_id' => $voucherId,
                                'account_head_id' => $debitAccount['account_head_id'],
                                'debit' => $amount,
                                'credit' => 0,
                                'created_date' => now(),
                                'cost_center_type' => 0,
                                'cost_center' => 'Main Outlet',
                                'detail_remarks' => "Salary for {$staffName} - {$month} {$year}",
                                'payment_mode' => 'Bank Transfer',
                                'created_at' => now(),
                                'updated_at' => now(),
                                'created_by' => $userName
                            ];
                        }
                    }
                    
                    // === PROCESS CREDIT ENTRIES (Deductions and Payables) ===
                    foreach ($creditAccounts as $creditAccount) {
                        $amount = 0;
                        $entity = $creditAccount['entity_name'];
                        
                        // Check entity - EXACT MATCHING - no calculations
                        if ($entity == "absent_deduction") {
                            $amount = isset($row['deduction_absent']) ? (float)$row['deduction_absent'] : 0;
                        } 
                        elseif ($entity == "late_deduction") {
                            $amount = isset($row['deduction_late']) ? (float)$row['deduction_late'] : 0;
                        } 
                        elseif ($entity == "tax_amount") {
                            $amount = isset($row['tax_deduction']) ? (float)$row['tax_deduction'] : 0;
                        } 
                        elseif ($entity == "other_deduction") {
                            $amount = isset($row['other_deduction']) ? (float)$row['other_deduction'] : 0;
                        } 
                        elseif ($entity == "loan_deduction") {
                            $amount = isset($row['loan_deduction']) ? (float)$row['loan_deduction'] : 0;
                        } 
                        elseif ($entity == "salary_payable") {
                            // Use the final_payable directly from payload
                            $amount = $finalPayable;
                        } 
                        elseif ($entity == "provident_fund_payable") {
                            $provident_fund_employee = isset($row['provident_fund_employee']) ? (float)$row['provident_fund_employee'] : 0;
                            $provident_fund_employer = isset($row['provident_fund_employer']) ? (float)$row['provident_fund_employer'] : 0;
                            $amount = $provident_fund_employee + $provident_fund_employer;
                        }
                        // Check for allowance credits if needed
                        elseif (strpos($entity, 'allowance') !== false && $creditAccount['description'] == "Custom Allowance Increase") {
                            // Handle allowances in credit side if needed
                            $amount = 0;
                        }
                        
                        if ($amount > 0) {
                            $voucherDetails[] = [
                                'account_voucher_id' => $voucherId,
                                'account_head_id' => $creditAccount['account_head_id'],
                                'debit' => 0,
                                'credit' => $amount,
                                'created_date' => now(),
                                'cost_center_type' => 0,
                                'cost_center' => 'Main Outlet',
                                'detail_remarks' => "Salary for {$staffName} - {$month} {$year}",
                                'payment_mode' => 'Bank Transfer',
                                'created_at' => now(),
                                'updated_at' => now(),
                                'created_by' => $userName
                            ];
                        }
                    }
                    
                    // === ADD ADJUSTMENT ENTRY ===
                    if ($adjustment != 0) {
                        if ($adjustment > 0) {
                            $voucherDetails[] = [
                                'account_voucher_id' => $voucherId,
                                'account_head_id' => $adjustmentAccount['account_head_id'],
                                'debit' => abs($adjustment),
                                'credit' => 0,
                                'created_date' => now(),
                                'cost_center_type' => 0,
                                'cost_center' => 'Main Outlet',
                                'detail_remarks' => "Rounding adjustment for {$staffName} - {$month} {$year}",
                                'payment_mode' => 'Bank Transfer',
                                'created_at' => now(),
                                'updated_at' => now(),
                                'created_by' => $userName
                            ];
                            
                        } else {
                            // Negative adjustment: Add to credit
                            $voucherDetails[] = [
                                'account_voucher_id' => $voucherId,
                                'account_head_id' => $adjustmentAccount['account_head_id'],
                                'debit' => 0,
                                'credit' => abs($adjustment),
                                'created_date' => now(),
                                'cost_center_type' => 0,
                                'cost_center' => 'Main Outlet',
                                'detail_remarks' => "Rounding adjustment for {$staffName} - {$month} {$year}",
                                'payment_mode' => 'Bank Transfer',
                                'created_at' => now(),
                                'updated_at' => now(),
                                'created_by' => $userName
                            ];
                        }
                    }
                    
                    if (!empty($voucherDetails)) {
                        DB::table('account_voucher_detail')->insert($voucherDetails);
                    }

                    // Success
                    $createdCount++;
                    $successStaff[] = $staffName;
                    
                    // Log success
                    Log::info("Voucher #{$voucherId} created for {$staffName}. Final Payable: {$finalPayable}, Adjustment: {$adjustment}");
                    
                } catch (\Exception $e) {
                    $staffName = isset($row['staff_fullname']) ? $row['staff_fullname'] : 'Unknown';
                    $errorMsg = "Staff {$staffName} (Row {$index}): " . $e->getMessage();
                    $errors[] = $errorMsg;
                    Log::error($errorMsg . "\n" . $e->getTraceAsString());
                }
            }
            
            if (!empty($alreadyExists) && $createdCount === 0 && empty($errors)) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Payments already exist for selected staff',
                    'already_exists' => $alreadyExists
                ], 400);
            }
            
            // If there are errors and no vouchers were created, rollback
            if (!empty($errors) && $createdCount === 0) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create any vouchers',
                    'errors' => $errors,
                    'already_exists' => $alreadyExists
                ], 400);
            }
            
            // If we reached here, commit the transaction
            DB::commit();
            
            $response = [
                'success' => true,
                'message' => "Successfully processed vouchers",
                'count' => $createdCount,
                'successful_staff' => $successStaff
            ];
            
            if (!empty($alreadyExists)) {
                $response['has_existing'] = true;
                $response['existing_payments'] = $alreadyExists;
                $response['message'] = "Created {$createdCount} voucher(s). Some staff already have payments.";
            }
            
            if (!empty($errors)) {
                $response['has_warnings'] = true;
                $response['warnings'] = $errors;
                $response['message'] = "Created {$createdCount} voucher(s) with some warnings";
            }
            
            return response()->json($response);
            
        } catch (\Exception $e) {
            // Catch any unexpected errors and rollback
            DB::rollBack();
            Log::error('Error in checkPayable: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'System error: ' . $e->getMessage()
            ], 500);
        }
    }


    // Commissions Breakdown
    public function commissionBreakdown(Request $request, $staffId, $businessId, $year, $month)
{
    try {
        $staff = Staff::find($staffId);
        if (!$staff) {
            abort(404, 'Staff not found');
        }

        $staffBusinessId = $staff->business_id;
        $businessIdForQuery = $staffBusinessId;

        $serviceCommissionData = $this->getServiceCommissionDetails($staffId, $year, $month, $businessIdForQuery);
        $retailCommissionData = $this->getRetailCommissionDetails($staffId, $year, $month, $businessIdForQuery);
        
        $staffInfo = DB::table('staff')
            ->where('id_staff', $staffId)
            ->select(
                'id_staff',
                'staff_fullname',
                'staff_salary',
                'commission_mode',
                'commission_target',
                'staff_commission_perc',
                'commission_fixed_amount'
            )
            ->first();
        
        // Calculate total commission based on commission mode
        $hasFixedAmount = $staffInfo->commission_mode == 'Target' && 
                         !empty($staffInfo->commission_fixed_amount) && 
                         $staffInfo->commission_fixed_amount > 0;
        
        $totalCommission = 0;
        $serviceCommissionAllocated = 0;
        $retailCommissionAllocated = 0;
        
        if ($staffInfo->commission_mode === 'Services') {
            // For Services mode, use pre-calculated commission sums
            $serviceCommissionAllocated = $serviceCommissionData['summary']['total_commission'] ?? 0;
            $retailCommissionAllocated = $retailCommissionData['summary']['total_commission'] ?? 0;
            $totalCommission = $serviceCommissionAllocated + $retailCommissionAllocated;
            
        } elseif ($hasFixedAmount) {
            // For Target mode with fixed amount
            $serviceTotalDiscounted = $serviceCommissionData['summary']['monthly_sales'] ?? 0;
            $retailTotalDiscounted = $retailCommissionData['summary']['monthly_sales'] ?? 0;
            $totalAchieved = $serviceTotalDiscounted + $retailTotalDiscounted;
            $commissionTarget = $staffInfo->commission_target ?? 0;
            
            if ($totalAchieved >= $commissionTarget) {
                $totalCommission = $staffInfo->commission_fixed_amount;
                $serviceCommissionAllocated = $serviceCommissionData['summary']['service_commission_share'] ?? 0;
                $retailCommissionAllocated = $retailCommissionData['summary']['retail_commission_share'] ?? 0;
            } else {
                // If target not achieved, no commission is allocated
                $serviceCommissionAllocated = 0;
                $retailCommissionAllocated = 0;
            }
        } else {
            // For Target mode with percentage
            $serviceCommissionAllocated = $serviceCommissionData['summary']['total_commission'] ?? 0;
            $retailCommissionAllocated = $retailCommissionData['summary']['total_commission'] ?? 0;
            $totalCommission = $serviceCommissionAllocated + $retailCommissionAllocated;
        }
        
        return view('hrm.commission_breakdown', compact(
            'staff',
            'staffInfo',
            'serviceCommissionData',
            'retailCommissionData',
            'year',
            'month',
            'businessId',
            'totalCommission',
            'hasFixedAmount',
            'serviceCommissionAllocated',
            'retailCommissionAllocated'
        ));
        
    } catch (\Exception $e) {
        Log::error('Error in commission breakdown: ' . $e->getMessage());
        Log::error('Stack trace: ' . $e->getTraceAsString());
        return redirect()->back()->with('Error', 'Failed to load commission breakdown');
    }
    }

    private function getServiceCommissionDetails($staffId, $year, $month, $businessIdForQuery)
    {
        try {
            $staffInfo = DB::table('staff')
                ->where('id_staff', $staffId)
                ->select(
                    'commission_mode',
                    'commission_target',
                    'staff_commission_perc',
                    'commission_fixed_amount'
                )
                ->first();
            
            if (!$staffInfo) {
                $staffInfo = (object) [
                    'commission_mode' => 'Services',
                    'commission_target' => 0,
                    'staff_commission_perc' => 0,
                    'commission_fixed_amount' => 0
                ];
            }
            
            $monthStartDate = "$year-$month-01";
            $monthEndDate = date("Y-m-t", strtotime($monthStartDate));
            
            // Get total monthly sales
            $monthlySales = DB::table('invoice_staff as is')
                ->join('invoice as i', 'i.id_invoice', '=', 'is.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('is.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->sum('is.discounted_price');
            
            // Get retail sales for collective target calculation
            $retailMonthlySales = DB::table('invoice_products as ip')
                ->join('invoice as i', 'i.id_invoice', '=', 'ip.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('ip.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->sum('ip.discounted_price');
            
            $details = DB::table('invoice_staff as is')
                ->join('invoice as i', 'i.id_invoice', '=', 'is.invoice_id')
                ->join('business as b', 'b.id_business', '=', 'i.business_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('is.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    'is.*',
                    'i.invoice_date',
                    'i.invoice_number',
                    'b.business_name',
                    DB::raw('is.commission as original_commission')
                )
                ->orderBy('i.invoice_date', 'desc')
                ->get();
            
            $totalDiscountedPrice = 0;
            $totalFinalPrice = 0;
            $totalCommission = 0;
            $totalPaid = 0;
            
            // Check conditions
            $hasServiceInvoices = $monthlySales > 0;
            $hasRetailInvoices = $retailMonthlySales > 0;
            $serviceTargetAchieved = $monthlySales >= $staffInfo->commission_target;
            $retailTargetAchieved = $retailMonthlySales >= $staffInfo->commission_target;
            $collectiveAchieved = ($monthlySales + $retailMonthlySales) >= $staffInfo->commission_target;
            $hasFixedAmount = !empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0;
            
            // Calculate service commission share for fixed amount
            $serviceCommissionShare = 0;
            if ($hasFixedAmount && $collectiveAchieved) {
                if ($hasServiceInvoices && $hasRetailInvoices) {
                    // Both have invoices - split 50/50
                    $serviceCommissionShare = $staffInfo->commission_fixed_amount / 2;
                } elseif ($hasServiceInvoices && !$hasRetailInvoices) {
                    // Only services have invoices - full amount to services
                    $serviceCommissionShare = $staffInfo->commission_fixed_amount;
                } elseif (!$hasServiceInvoices && $hasRetailInvoices) {
                    // Only retail has invoices - no commission for services
                    $serviceCommissionShare = 0;
                }
            }
            
            // Calculate commission for each service
            foreach ($details as $detail) {
                $totalDiscountedPrice += $detail->discounted_price ?? 0;
                $totalFinalPrice += $detail->final_price ?? 0;
                $totalPaid += $detail->paid ?? 0;
                
                if ($staffInfo->commission_mode === 'Services') {
                    // Use pre-calculated commission
                    $detail->calculated_commission = $detail->original_commission;
                    $totalCommission += $detail->original_commission;
                } 
                elseif ($staffInfo->commission_mode === 'Target') {
                    if ($hasFixedAmount) {
                        // For fixed amount, don't calculate individual commission
                        $detail->calculated_commission = null;
                    } else {
                        // Percentage based commission
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        
                        if ($serviceTargetAchieved) {
                            $detail->calculated_commission = (($detail->discounted_price ?? 0) * $commissionRate) / 100;
                            $totalCommission += $detail->calculated_commission;
                        } else {
                            $detail->calculated_commission = 0;
                        }
                    }
                }
            }
            
            // For fixed amount mode, use the allocated share
            $displayCommission = $hasFixedAmount ? $serviceCommissionShare : $totalCommission;
            
            return [
                'details' => $details,
                'summary' => [
                    'total_discounted_price' => $totalDiscountedPrice,
                    'total_final_price' => $totalFinalPrice,
                    'total_paid' => $totalPaid,
                    'total_commission' => $displayCommission,
                    'total_services' => count($details),
                    'average_commission_rate' => $totalDiscountedPrice > 0 ? ($displayCommission / $totalDiscountedPrice * 100) : 0,
                    'commission_mode' => $staffInfo->commission_mode,
                    'commission_target' => $staffInfo->commission_target,
                    'target_achieved' => $serviceTargetAchieved,
                    'collective_target_achieved' => $collectiveAchieved,
                    'commission_fixed_amount' => $staffInfo->commission_fixed_amount ?? 0,
                    'monthly_sales' => $monthlySales,
                    'has_service_invoices' => $hasServiceInvoices,
                    'has_retail_invoices' => $hasRetailInvoices,
                    'retail_target_achieved' => $retailTargetAchieved,
                    'service_commission_share' => $serviceCommissionShare
                ]
            ];
            
        } catch (\Exception $e) {
            Log::error('Error getting service commission details: ' . $e->getMessage());
            return [
                'details' => collect([]),
                'summary' => [
                    'total_discounted_price' => 0,
                    'total_final_price' => 0,
                    'total_paid' => 0,
                    'total_commission' => 0,
                    'total_services' => 0,
                    'average_commission_rate' => 0,
                    'commission_mode' => 'Services',
                    'commission_target' => 0,
                    'target_achieved' => false,
                    'collective_target_achieved' => false,
                    'commission_fixed_amount' => 0,
                    'monthly_sales' => 0,
                    'has_service_invoices' => false,
                    'has_retail_invoices' => false,
                    'retail_target_achieved' => false,
                    'service_commission_share' => 0
                ]
            ];
        }
    }

    private function getRetailCommissionDetails($staffId, $year, $month, $businessIdForQuery)
    {
        try {
            $staffInfo = DB::table('staff')
                ->where('id_staff', $staffId)
                ->select(
                    'commission_mode',
                    'commission_target',
                    'staff_commission_perc',
                    'commission_fixed_amount'
                )
                ->first();
            
            if (!$staffInfo) {
                $staffInfo = (object) [
                    'commission_mode' => 'Services',
                    'commission_target' => 0,
                    'staff_commission_perc' => 0,
                    'commission_fixed_amount' => 0
                ];
            }
            
            $monthStartDate = "$year-$month-01";
            $monthEndDate = date("Y-m-t", strtotime($monthStartDate));
            
            // Get total monthly sales
            $monthlySales = DB::table('invoice_products as ip')
                ->join('invoice as i', 'i.id_invoice', '=', 'ip.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('ip.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->sum('ip.discounted_price');
            
            // Get service sales for collective target calculation
            $serviceMonthlySales = DB::table('invoice_staff as is')
                ->join('invoice as i', 'i.id_invoice', '=', 'is.invoice_id')
                ->where('i.invoice_status', '=', 'valid')
                ->where('is.staff_id', '=', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->sum('is.discounted_price');
            
            $details = DB::table('invoice_products as ip')
                ->join('invoice as i', 'i.id_invoice', '=', 'ip.invoice_id')
                ->join('business as b', 'b.id_business', '=', 'i.business_id')
                ->where('i.invoice_status', 'valid')
                ->where('ip.staff_id', $staffId)
                ->whereBetween('i.invoice_date', [$monthStartDate . ' 00:00:00', $monthEndDate . ' 23:59:59'])
                ->select(
                    'ip.*',
                    'i.invoice_date',
                    'i.invoice_number',
                    'b.business_name',
                    DB::raw('ip.commission as original_commission')
                )
                ->orderBy('i.invoice_date', 'desc')
                ->get();
            
            $totalDiscountedPrice = 0;
            $totalFinalPrice = 0;
            $totalCommission = 0;
            $totalPaid = 0;
            $totalQuantity = 0;
            
            // Check conditions
            $hasRetailInvoices = $monthlySales > 0;
            $hasServiceInvoices = $serviceMonthlySales > 0;
            $retailTargetAchieved = $monthlySales >= $staffInfo->commission_target;
            $serviceTargetAchieved = $serviceMonthlySales >= $staffInfo->commission_target;
            $collectiveAchieved = ($monthlySales + $serviceMonthlySales) >= $staffInfo->commission_target;
            $hasFixedAmount = !empty($staffInfo->commission_fixed_amount) && $staffInfo->commission_fixed_amount > 0;
            
            // Calculate retail commission share for fixed amount
            $retailCommissionShare = 0;
            if ($hasFixedAmount && $collectiveAchieved) {
                if ($hasServiceInvoices && $hasRetailInvoices) {
                    // Both have invoices - split 50/50
                    $retailCommissionShare = $staffInfo->commission_fixed_amount / 2;
                } elseif (!$hasServiceInvoices && $hasRetailInvoices) {
                    // Only retail has invoices - full amount to retail
                    $retailCommissionShare = $staffInfo->commission_fixed_amount;
                } elseif ($hasServiceInvoices && !$hasRetailInvoices) {
                    // Only services have invoices - no commission for retail
                    $retailCommissionShare = 0;
                }
            }
            
            // Calculate commission for each retail item
            foreach ($details as $detail) {
                $totalDiscountedPrice += $detail->discounted_price ?? 0;
                $totalFinalPrice += $detail->final_price ?? 0;
                $totalPaid += $detail->paid ?? 0;
                $totalQuantity += $detail->invoice_qty ?? 1;
                
                if ($staffInfo->commission_mode === 'Services') {
                    // Use pre-calculated commission
                    $detail->calculated_commission = $detail->original_commission;
                    $totalCommission += $detail->original_commission;
                } 
                elseif ($staffInfo->commission_mode === 'Target') {
                    if ($hasFixedAmount) {
                        // For fixed amount, don't calculate individual commission
                        $detail->calculated_commission = null;
                    } else {
                        // Percentage based commission
                        $commissionRate = $staffInfo->staff_commission_perc ?? 0;
                        
                        if ($retailTargetAchieved) {
                            $detail->calculated_commission = (($detail->discounted_price ?? 0) * $commissionRate) / 100;
                            $totalCommission += $detail->calculated_commission;
                        } else {
                            $detail->calculated_commission = 0;
                        }
                    }
                }
            }
            
            // For fixed amount mode, use the allocated share
            $displayCommission = $hasFixedAmount ? $retailCommissionShare : $totalCommission;
            
            return [
                'details' => $details,
                'summary' => [
                    'total_discounted_price' => $totalDiscountedPrice,
                    'total_final_price' => $totalFinalPrice,
                    'total_paid' => $totalPaid,
                    'total_commission' => $displayCommission,
                    'total_products' => count($details),
                    'total_quantity' => $totalQuantity,
                    'average_unit_price' => $totalQuantity > 0 ? ($totalFinalPrice / $totalQuantity) : 0,
                    'average_discounted_unit_price' => $totalQuantity > 0 ? ($totalDiscountedPrice / $totalQuantity) : 0,
                    'average_commission_rate' => $totalDiscountedPrice > 0 ? ($displayCommission / $totalDiscountedPrice * 100) : 0,
                    'commission_mode' => $staffInfo->commission_mode,
                    'commission_target' => $staffInfo->commission_target,
                    'target_achieved' => $retailTargetAchieved,
                    'collective_target_achieved' => $collectiveAchieved,
                    'commission_fixed_amount' => $staffInfo->commission_fixed_amount ?? 0,
                    'monthly_sales' => $monthlySales,
                    'has_retail_invoices' => $hasRetailInvoices,
                    'has_service_invoices' => $hasServiceInvoices,
                    'service_target_achieved' => $serviceTargetAchieved,
                    'retail_commission_share' => $retailCommissionShare
                ]
            ];
            
        } catch (\Exception $e) {
            Log::error('Error getting retail commission details: ' . $e->getMessage());
            return [
                'details' => collect([]),
                'summary' => [
                    'total_discounted_price' => 0,
                    'total_final_price' => 0,
                    'total_paid' => 0,
                    'total_commission' => 0,
                    'total_products' => 0,
                    'total_quantity' => 0,
                    'average_unit_price' => 0,
                    'average_discounted_unit_price' => 0,
                    'average_commission_rate' => 0,
                    'commission_mode' => 'Services',
                    'commission_target' => 0,
                    'target_achieved' => false,
                    'collective_target_achieved' => false,
                    'commission_fixed_amount' => 0,
                    'monthly_sales' => 0,
                    'has_retail_invoices' => false,
                    'has_service_invoices' => false,
                    'service_target_achieved' => false,
                    'retail_commission_share' => 0
                ]
            ];
        }
    }

    
    public function staffAttendanceReport(Request $request)
    {
        $isHoUser = session('ho') === 'Yes';
        if ($isHoUser && $request && $request->filled('business_id')) {
            $businessId = $request->business_id;
        }else{
            $businessId = session('business_id');
        }
        
        if ($isHoUser && session('selected_business_id')) {
            $businessId = session('selected_business_id');
        }
        $business = Business::find($businessId);
        
        // Get businesses for HO users - FILTER BY ho_accounts = 'Yes'
        $isHoUser = session('ho') === 'Yes';
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        // Get staff list
        $staffList = collect([]);
        if ($isHoUser) {
            if ($request->filled('business_id')) {
                $selectedBusinessId = $request->business_id;
            } elseif (session('selected_business_id')) {
                $selectedBusinessId = session('selected_business_id');
            } else {
                $selectedBusinessId = $businessId;
            }
            
            $staffList = Staff::where('business_id', $selectedBusinessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();
        } else {
            $staffList = Staff::where('business_id', $businessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();
        }
        
        // Get filter values with default to current month/year
        $month = $request->input('month', date('m'));
        $year = $request->input('year', date('Y'));
        $staffId = $request->input('staff', 0);
        $viewType = $request->input('view', 'summary'); // 'summary' or 'detail'
        
        // Get month name
        $dateObj = \DateTime::createFromFormat('!m', $month);
        $monthName = $dateObj->format('F');
        
        // Get year range from attendance data
        $earliestYear = DB::table('staff_attendance')
            ->selectRaw('MIN(YEAR(time_in)) as min_year')
            ->value('min_year');
        $earliestYear = $earliestYear ? (int)$earliestYear : date('Y');
        $currentYear = (int)date('Y');
        
        // Check if we're viewing detail for a specific staff
        if ($viewType === 'detail' && $staffId > 0) {
            $staff = Staff::find($staffId);
            
            if (!$staff) {
                abort(404, 'Staff not found');
            }
            
            // Check permissions for non-HO users
            if (!$isHoUser && $staff->business_id != session('business_id')) {
                abort(403, 'Access denied');
            }
            
            // Get staff attendance data WITHOUT salary calculations
            $payableData = $this->getStaffAttendanceData($staffId, $year, $month, $staff);
            
            // Get leaves and holidays for detail view
            $startDate = "$year-$month-01";
            $endDate = date("Y-m-t", strtotime($startDate));
            
            $leaveRecords = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('staff_leave_status', 'Active')
                ->whereBetween('staff_leave_date', [$startDate, $endDate])
                ->select('staff_leave_date', 'leave_reason')
                ->get()
                ->keyBy(function($item) {
                    return date('Y-m-d', strtotime($item->staff_leave_date));
                });
            
            $holidayRecords = DB::table('holidays')
                ->where('business_id', $staff->business_id)
                ->whereBetween('holiday_date', [$startDate, $endDate])
                ->select('holiday_date', 'holiday_reason')
                ->get()
                ->keyBy(function($item) {
                    return date('Y-m-d', strtotime($item->holiday_date));
                });
            
            // Get attendance records for detail view
            $attendanceRecords = DB::table('staff_attendance')
                ->where('staff_id', $staffId)
                ->whereBetween('time_in', [$startDate . ' 00:00:00', $endDate . ' 23:59:59'])
                ->select('*', DB::raw('DATE(time_in) as attendance_date'))
                ->get()
                ->keyBy('attendance_date');
            
            // Get overtime days
            $overtimeDays = DB::table('overtime_days')
                ->where('business_id', $staff->business_id)
                ->whereBetween('overtime_date', [$startDate, $endDate])
                ->pluck('overtime_date')
                ->map(function($date) {
                    return date('Y-m-d', strtotime($date));
                })
                ->toArray();
            $payables = [];
            
            // Calculate totals
            $presents = 0;
            $absents = 0;
            $late = 0;
            $workhours_total = 0;
            $overtime_hours_total = 0;
            $leave = 0;
            $holiday = 0;
            
            // Generate day-by-day data
            $currentDate = new \DateTime($startDate);
            $endDateTime = new \DateTime($endDate);
            
            while ($currentDate <= $endDateTime) {
                $dateStr = $currentDate->format('Y-m-d');
                $dayName = $currentDate->format('D');
                
                // Check for leave
                $isLeave = isset($leaveRecords[$dateStr]);
                $leaveReason = $isLeave ? $leaveRecords[$dateStr]->leave_reason : '';
                if ($isLeave) $leave++;
                
                // Check for holiday
                $isHoliday = isset($holidayRecords[$dateStr]);
                $holidayReason = $isHoliday ? $holidayRecords[$dateStr]->holiday_reason : '';
                if ($isHoliday) $holiday++;
                
                // Check if it's a weekend (day off)
                $isWeekend = strpos($staff->week_day_off ?? '', $dayName) !== false;
                
                // Check if it's an overtime day
                $isOvertimeDay = in_array($dateStr, $overtimeDays);
                
                // Get attendance record
                $attendance = $attendanceRecords[$dateStr] ?? null;
                
                // Calculate work hours from attendance record - NO ROUNDING
                $workHours = 0;
                if ($attendance) {
                    $timeIn = strtotime($attendance->time_in);
                    $timeOut = $attendance->time_out ? strtotime($attendance->time_out) : null;
                    
                    if (!$timeOut && $business) {
                        // Use business closing time if no time out
                        $closingTime = $business->business_closing_time ?? '18:00:00';
                        $timeOut = strtotime($dateStr . ' ' . $closingTime);
                    }
                    
                    if ($timeOut && $timeIn) {
                        $workHours = ($timeOut - $timeIn) / 3600; // No rounding
                    }
                }
                
                // Check if present (has attendance record with actual time)
                $isPresent = ($attendance && $attendance->time_in != $dateStr . ' 00:00:00') ? 1 : 0;
                if ($isPresent) $presents++;
                
                // Check if late - ONLY FOR REGULAR WORKING DAYS (not weekends/holidays/overtime days/leave days)
                $isLate = 0;
                if ($attendance && $staff->day_start_time && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay) {
                    $startTime = strtotime($dateStr . ' ' . $staff->day_start_time);
                    $attendanceTime = strtotime($attendance->time_in);
                    $margin = $business->timein_margin ?? 0;
                    $marginSeconds = $margin * 60;
                    
                    if ($attendanceTime > ($startTime + $marginSeconds)) {
                        $isLate = 1;
                        $late++;
                    }
                }
                
                // Check if absent (working day without attendance, not leave/holiday/weekend/overtime day)
                $isAbsent = (!$isPresent && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay);
                if ($isAbsent) $absents++;
                
                // Calculate overtime for this day
                $overtimeHours = 0;
                
                if ($staff->overtime_allowed === 'Yes') {
                    if ($isOvertimeDay || $isWeekend || $isHoliday) {
                        // For weekends/holidays/overtime days, all hours count as overtime
                        $overtimeHours = $workHours;
                    } elseif ($isPresent && $staff->day_end_time) {
                        // For regular working days, calculate overtime after regular hours
                        $regularHours = 8; // Default 8 hours
                        if ($staff->day_start_time && $staff->day_end_time) {
                            $start = strtotime($staff->day_start_time);
                            $end = strtotime($staff->day_end_time);
                            
                            if ($end < $start) {
                                $end += 86400; // Add 24 hours for night shift
                            }
                            
                            $regularHours = ($end - $start) / 3600;
                        }
                        
                        if ($workHours > $regularHours) {
                            $overtimeHours = $workHours - $regularHours;
                        }
                    }
                }
                
                $workhours_total += $workHours;
                $overtime_hours_total += $overtimeHours;
                
                // Add to payables array WITHOUT SALARY DATA
                $payables[] = [
                    'Date' => $dateStr,
                    'Day' => $dayName,
                    'week_day_off' => $staff->week_day_off ?? '',
                    'time_in' => $attendance ? date('H:i', strtotime($attendance->time_in)) : '',
                    'time_out' => $attendance && $attendance->time_out ? date('H:i', strtotime($attendance->time_out)) : '',
                    'Work_Hours' => $workHours, // Raw value, no rounding
                    'Present' => $isPresent,
                    'Absent' => $isAbsent ? 1 : 0,
                    'Late' => $isLate,
                    'Leaves' => $isLeave ? 1 : 0,
                    'leave_reason' => $leaveReason,
                    'Holiday' => $isHoliday ? 1 : 0,
                    'holiday_reason' => $holidayReason,
                    'address' => $attendance ? ($attendance->address ?? '') : '',
                    'overtime_hours' => $overtimeHours,
                    'is_overtime_day' => $isOvertimeDay,
                    'is_weekend' => $isWeekend
                ];
                
                $currentDate->modify('+1 day');
            }
            
            // Get year range from attendance data for detail view too
            $earliestYear = DB::table('staff_attendance')
                ->selectRaw('MIN(YEAR(time_in)) as min_year')
                ->value('min_year');
            $earliestYear = $earliestYear ? (int)$earliestYear : date('Y');
            $currentYear = (int)date('Y');
            
            return view('hrm.staff_attendance_report', compact(
                'business',
                'staffList',
                'businesses',
                'month',
                'year',
                'staffId',
                'monthName',
                'viewType',
                'staff',
                'payables',
                'presents',
                'absents',
                'late',
                'leave',
                'holiday',
                'workhours_total',
                'overtime_hours_total',
                'isHoUser',
                'earliestYear',
                'currentYear'
            ));
        }
        
        // Default: Show summary view
        return view('hrm.staff_attendance_report', compact(
            'business',
            'staffList',
            'businesses',
            'month',
            'year',
            'staffId',
            'monthName',
            'viewType',
            'isHoUser',
            'earliestYear',
            'currentYear'
        ));
    }

    /*Get staff attendance data */
    private function getStaffAttendanceData($staffId, $year, $month, $staff)
    {
        try {
            $startDate = "$year-$month-01";
            $endDate = date("Y-m-t", strtotime($startDate));
            
            // Get staff info with business
            $result = DB::table('staff as s')
                ->where('s.id_staff', $staffId)
                ->leftJoin('business as b', 'b.id_business', '=', 's.business_id')
                ->select(
                    's.*',
                    'b.timein_margin',
                    'b.business_name',
                    'b.business_closing_time'
                )
                ->first();

            if (!$result) return [];

            // Get week offs
            $weekOffs = array_map('trim', explode(',', $result->week_day_off ?? ''));
            
            // Get holidays
            $holidays = DB::table('holidays')
                ->where('business_id', $result->business_id)
                ->whereBetween('holiday_date', [$startDate, $endDate])
                ->pluck('holiday_date')
                ->map(fn($d) => date('Y-m-d', strtotime($d)))
                ->toArray();

            // Get overtime days
            $overtimeDays = DB::table('overtime_days')
                ->where('business_id', $result->business_id)
                ->whereBetween('overtime_date', [$startDate, $endDate])
                ->pluck('overtime_date')
                ->map(fn($d) => date('Y-m-d', strtotime($d)))
                ->toArray();

            // Get attendance records
            $attendances = DB::table('staff_attendance')
                ->where('staff_id', $staffId)
                ->whereBetween('time_in', [$startDate.' 00:00:00', $endDate.' 23:59:59'])
                ->get();

            // Get leaves
            $leaves = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('staff_leave_status', 'Active')
                ->whereBetween('staff_leave_date', [$startDate, $endDate])
                ->pluck('staff_leave_date')
                ->map(fn($d) => date('Y-m-d', strtotime($d)))
                ->toArray();

            // Counters
            $presentDays = 0;
            $lateDays = 0;
            $absentDays = 0;
            $totalWorkHours = 0;
            $totalOvertimeHours = 0;
            $leaveDays = count($leaves);
            $holidayDays = count($holidays);
            
            // Track attendance dates
            $attendanceDates = [];
            foreach ($attendances as $att) {
                $date = date('Y-m-d', strtotime($att->time_in));
                $attendanceDates[$date] = $att;
            }

            // Calculate regular shift hours
            $regularShiftHours = 8;
            if ($result->day_start_time && $result->day_end_time) {
                $s = strtotime($result->day_start_time);
                $e = strtotime($result->day_end_time);
                if ($e < $s) $e += 86400;
                $regularShiftHours = ($e - $s) / 3600;
            }

            // Process each day of the month
            $currentDate = new \DateTime($startDate);
            $endDateTime = new \DateTime($endDate);
            
            while ($currentDate <= $endDateTime) {
                $date = $currentDate->format('Y-m-d');
                $dayName = $currentDate->format('l');
                $dayShort = substr($dayName, 0, 3);
                
                $isHoliday = in_array($date, $holidays);
                $isLeave = in_array($date, $leaves);
                $isWeekend = in_array($dayName, $weekOffs) || in_array($dayShort, $weekOffs);
                $isOvertimeDay = in_array($date, $overtimeDays);
                
                $attendance = $attendanceDates[$date] ?? null;
                
                // Calculate work hours - SAME LOGIC AS DETAIL VIEW
                $workHours = 0;
                if ($attendance) {
                    $timeIn = strtotime($attendance->time_in);
                    $timeOut = $attendance->time_out ? strtotime($attendance->time_out) : null;
                    
                    if (!$timeOut && $result->business_closing_time) {
                        $timeOut = strtotime($date . ' ' . $result->business_closing_time);
                    }
                    
                    if ($timeOut && $timeIn) {
                        $workHours = ($timeOut - $timeIn) / 3600; // No rounding
                    }
                }
                
                // Check if present
                $isPresent = ($attendance && $attendance->time_in != $date . ' 00:00:00');
                if ($isPresent) {
                    $presentDays++;
                }
                
                // Check if late - SAME LOGIC AS DETAIL VIEW
                if ($attendance && $result->day_start_time && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay) {
                    $startTime = strtotime($date . ' ' . $result->day_start_time);
                    $attendanceTime = strtotime($attendance->time_in);
                    $margin = $result->timein_margin ?? 0;
                    $marginSeconds = $margin * 60;
                    
                    if ($attendanceTime > ($startTime + $marginSeconds)) {
                        $lateDays++;
                    }
                }
                
                // Check if absent - SAME LOGIC AS DETAIL VIEW
                $isAbsent = (!$isPresent && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay);
                if ($isAbsent) {
                    $absentDays++;
                }
                
                // Calculate overtime - SAME LOGIC AS DETAIL VIEW
                $overtimeHours = 0;
                if ($result->overtime_allowed === 'Yes') {
                    if ($isOvertimeDay || $isWeekend || $isHoliday) {
                        $overtimeHours = $workHours;
                    } elseif ($isPresent && $result->day_end_time) {
                        if ($workHours > $regularShiftHours) {
                            $overtimeHours = $workHours - $regularShiftHours;
                        }
                    }
                }
                
                $totalWorkHours += $workHours;
                $totalOvertimeHours += $overtimeHours;
                
                $currentDate->modify('+1 day');
            }

            return [
                'id_staff' => $result->id_staff,
                'staff_fullname' => $result->staff_fullname,
                'business_name' => $result->business_name,
                'salary_type' => $result->salary_type,
                'present_days' => $presentDays,
                'late_days' => $lateDays,
                'absent_days' => $absentDays,
                'leave_days' => $leaveDays,
                'holiday_days' => $holidayDays,
                'regular_shift_hours' => $regularShiftHours,
                'work_hours' => $totalWorkHours,
                'total_overtime_hours' => $totalOvertimeHours,
            ];

        } catch (\Exception $e) {
            Log::error($e->getMessage());
            return [];
        }
    }

    /*Get attendance report data */
    public function getAttendanceReportData(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            if ($isHoUser && $request && $request->filled('business_id')) {
                $businessId = $request->business_id;
            }else{
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }
            
            $query = Staff::with(['business'])
                ->where('staff_active', 'Y');
            
            // Apply business filter
            if ($isHoUser) {
                if ($request->filled('business_id') && $request->business_id !== "") {
                    $query->where('business_id', $request->business_id);
                } else {
                    $query->whereHas('business', function($q) {
                        $q->where('ho_accounts', 'Yes');
                    });
                }
            } else {
                $query->where('business_id', $businessId);
            }
            
            // Apply staff filter if provided
            if ($request->filled('staff_id') && $request->staff_id > 0) {
                $query->where('id_staff', $request->staff_id);
            }
            
            $staffList = $query->get();
            
            $month = $request->input('month', date('m'));
            $year = $request->input('year', date('Y'));
            
            // Get DataTables parameters
            $draw = $request->input('draw', 1);
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            $reportData = [];
            foreach ($staffList as $staff) {
                // Use the UPDATED calculation method
                $payableData = $this->getStaffAttendanceData($staff->id_staff, $year, $month, $staff);
                
                if (empty($payableData)) {
                    // Create empty record if no data
                    $reportData[] = [
                        'id_staff' => $staff->id_staff,
                        'staff_fullname' => $staff->staff_fullname,
                        'business_name' => $staff->business->business_name ?? 'N/A',
                        'total_present' => 0,
                        'total_absent' => 0,
                        'total_late' => 0,
                        'total_leave' => 0,
                        'total_holiday' => 0,
                        'total_work_hours' => 0,
                        'total_overtime_hours' => 0,
                        'view_link' => route('hrm.staff_attendance_report', [
                            'staff' => $staff->id_staff,
                            'month' => $month,
                            'year' => $year,
                            'view' => 'detail'
                        ])
                    ];
                    continue;
                }
                
                // Generate detail view URL with parameters
                $detailUrl = route('hrm.staff_attendance_report', [
                    'staff' => $staff->id_staff,
                    'month' => $month,
                    'year' => $year,
                    'view' => 'detail'
                ]);
                
                $reportData[] = [
                    'id_staff' => $staff->id_staff,
                    'staff_fullname' => $staff->staff_fullname,
                    'business_name' => $staff->business->business_name ?? 'N/A',
                    'total_present' => $payableData['present_days'] ?? 0,
                    'total_absent' => $payableData['absent_days'] ?? 0,
                    'total_late' => $payableData['late_days'] ?? 0,
                    'total_leave' => $payableData['leave_days'] ?? 0,
                    'total_holiday' => $payableData['holiday_days'] ?? 0,
                    'total_work_hours' => $payableData['work_hours'] ?? 0,
                    'total_overtime_hours' => $payableData['total_overtime_hours'] ?? 0,
                    'view_link' => $detailUrl
                ];
            }
            
            // Get total count (before pagination)
            $recordsTotal = count($reportData);
            $recordsFiltered = $recordsTotal; // Since we're filtering server-side via filters, all records are "filtered"
            
            // Apply pagination
            $paginatedData = array_slice($reportData, $start, $length !== -1 ? $length : $recordsTotal);
            
            // Return DataTables format
            return response()->json([
                'draw' => intval($draw),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $paginatedData
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error getting attendance report: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching attendance report: ' . $e->getMessage()
            ], 500);
        }
    }


    /*** Yearly Attendance Report*/
    public function yearlyAttendanceReport(Request $request)
    {
        $isHoUser = session('ho') === 'Yes';
        
        // Get business ID
        if ($isHoUser && $request->filled('business_id')) {
            $businessId = $request->business_id;
        } else {
            $businessId = session('business_id');
        }
        
        if ($isHoUser && session('selected_business_id')) {
            $businessId = session('selected_business_id');
        }
        
        $business = Business::find($businessId);
        
        // Get businesses for HO users
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        
        // Get staff list
        $staffList = collect([]);
        if ($isHoUser) {
            if ($request->filled('business_id')) {
                $selectedBusinessId = $request->business_id;
            } elseif (session('selected_business_id')) {
                $selectedBusinessId = session('selected_business_id');
            } else {
                $selectedBusinessId = $businessId;
            }
            
            $staffList = Staff::where('business_id', $selectedBusinessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();
        } else {
            $staffList = Staff::where('business_id', $businessId)
                ->where('staff_active', 'Y')
                ->orderBy('staff_fullname')
                ->get();
        }
        
        // Get filter values
        $year = $request->input('year', date('Y'));
        $staffId = $request->input('staff', 0);
        
        return view('hrm.yearly_attendance_report', compact(
            'business',
            'staffList',
            'businesses',
            'year',
            'staffId',
            'isHoUser'
        ));
    }

    /** Get Yearly Attendance Data*/
    public function getYearlyAttendanceData(Request $request)
    {
        try {
            $draw = $request->input('draw');
            $start = $request->input('start');
            $length = $request->input('length');
            $searchValue = $request->input('search.value');
            $orderColumn = $request->input('order.0.column');
            $orderDir = $request->input('order.0.dir');
            
            $isHoUser = session('ho') === 'Yes';
            
            if ($isHoUser && $request->filled('business_id')) {
                $businessId = $request->business_id;
            } else {
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }
            
            $query = Staff::with(['business'])
                ->where('staff_active', 'Y');
            
            // Apply business filter
            if ($isHoUser) {
                if ($request->filled('business_id') && $request->business_id !== "") {
                    $query->where('business_id', $request->business_id);
                } else {
                    $query->whereHas('business', function($q) {
                        $q->where('ho_accounts', 'Yes');
                    });
                }
            } else {
                $query->where('business_id', $businessId);
            }
            
            if ($request->filled('staff_id') && $request->staff_id > 0) {
                $query->where('id_staff', $request->staff_id);
            }
            
            // Apply search
            if (!empty($searchValue)) {
                $query->where(function($q) use ($searchValue) {
                    $q->where('staff_fullname', 'LIKE', "%{$searchValue}%")
                    ->orWhere('id_staff', 'LIKE', "%{$searchValue}%")
                    ->orWhereHas('business', function($businessQuery) use ($searchValue) {
                        $businessQuery->where('business_name', 'LIKE', "%{$searchValue}%");
                    });
                });
            }
            
            // Get total count
            $totalRecords = $query->count();
            
            // Apply ordering
            $orderColumns = [
                0 => 'staff_fullname',  // Staff Name column
                1 => 'business_name',   // Business column
                2 => 'salary_type',     // Salary Type column
            ];
            
            if (isset($orderColumns[$orderColumn])) {
                if ($orderColumns[$orderColumn] === 'business_name') {
                    $query->join('businesses', 'staff.business_id', '=', 'businesses.id_business')
                        ->orderBy('businesses.business_name', $orderDir)
                        ->select('staff.*');
                } else {
                    $query->orderBy($orderColumns[$orderColumn], $orderDir);
                }
            }
            
            // Apply pagination
            $staffList = $query->skip($start)->take($length)->get();
            
            $year = $request->input('year', date('Y'));
            $yearlyData = [];
            
            foreach ($staffList as $staff) {
                $staffData = [
                    'id_staff' => $staff->id_staff,
                    'staff_fullname' => $staff->staff_fullname,
                    'business_name' => $staff->business->business_name ?? 'N/A',
                    'salary_type' => $staff->salary_type,
                    'months' => []
                ];
                
                for ($month = 1; $month <= 12; $month++) {
                    $monthFormatted = str_pad($month, 2, '0', STR_PAD_LEFT);
                    $monthData = $this->getDetailedMonthData($staff->id_staff, $year, $monthFormatted);
                    
                    $staffData['months'][$month] = [
                        'month_number' => $monthFormatted,
                        'month_name' => date('F', mktime(0, 0, 0, $month, 1)),
                        'present' => $monthData['presents'] ?? 0,
                        'absent' => $monthData['absents'] ?? 0,
                        'late' => $monthData['late'] ?? 0,
                        'leave' => $monthData['leave'] ?? 0,
                        'holiday' => $monthData['holiday'] ?? 0,
                        'work_hours' => $monthData['workhours_total'] ?? 0,
                        'overtime_hours' => $monthData['overtime_hours_total'] ?? 0,
                        'detail_url' => route('hrm.staff_attendance_report', [
                            'staff' => $staff->id_staff,
                            'month' => $monthFormatted,
                            'year' => $year,
                            'view' => 'detail',
                            'business_id' => $request->business_id ?? session('business_id')
                        ])
                    ];
                }
                
                // Calculate yearly totals
                $staffData['yearly_totals'] = [
                    'present' => array_sum(array_column($staffData['months'], 'present')),
                    'absent' => array_sum(array_column($staffData['months'], 'absent')),
                    'late' => array_sum(array_column($staffData['months'], 'late')),
                    'leave' => array_sum(array_column($staffData['months'], 'leave')),
                    'holiday' => array_sum(array_column($staffData['months'], 'holiday')),
                    'work_hours' => array_sum(array_column($staffData['months'], 'work_hours')),
                    'overtime_hours' => array_sum(array_column($staffData['months'], 'overtime_hours'))
                ];
                
                $yearlyData[] = $staffData;
            }
            
            return response()->json([
                'draw' => (int)$draw,
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $yearlyData,
                'year' => $year
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error getting yearly attendance report: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'draw' => (int)($request->input('draw') ?? 0),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error fetching yearly attendance report: ' . $e->getMessage()
            ], 500);
        }
    }

    /** Get detailed month data*/
    private function getDetailedMonthData($staffId, $year, $month)
    {
        try {
            $staff = Staff::with('business')->find($staffId);
            if (!$staff) {
                return [
                    'presents' => 0,
                    'absents' => 0,
                    'late' => 0,
                    'leave' => 0,
                    'holiday' => 0,
                    'workhours_total' => 0,
                    'overtime_hours_total' => 0
                ];
            }
            
            $startDate = "$year-$month-01";
            $endDate = date("Y-m-t", strtotime($startDate));
            
            // Get leaves
            $leaveRecords = DB::table('staff_leaves')
                ->where('staff_id', $staffId)
                ->where('staff_leave_status', 'Active')
                ->whereBetween('staff_leave_date', [$startDate, $endDate])
                ->select('staff_leave_date')
                ->get()
                ->keyBy(function($item) {
                    return date('Y-m-d', strtotime($item->staff_leave_date));
                });
            
            // Get holidays
            $holidayRecords = DB::table('holidays')
                ->where('business_id', $staff->business_id)
                ->whereBetween('holiday_date', [$startDate, $endDate])
                ->select('holiday_date')
                ->get()
                ->keyBy(function($item) {
                    return date('Y-m-d', strtotime($item->holiday_date));
                });
            
            // Get attendance records
            $attendanceRecords = DB::table('staff_attendance')
                ->where('staff_id', $staffId)
                ->whereBetween('time_in', [$startDate . ' 00:00:00', $endDate . ' 23:59:59'])
                ->select('*', DB::raw('DATE(time_in) as attendance_date'))
                ->get()
                ->keyBy('attendance_date');
            
            // Get overtime days
            $overtimeDays = DB::table('overtime_days')
                ->where('business_id', $staff->business_id)
                ->whereBetween('overtime_date', [$startDate, $endDate])
                ->pluck('overtime_date')
                ->map(function($date) {
                    return date('Y-m-d', strtotime($date));
                })
                ->toArray();
            
            // Initialize counters - ALL START FROM 0
            $presents = 0;
            $absents = 0;
            $late = 0;
            $leave = 0;
            $holiday = 0;
            $workhours_total = 0;
            $overtime_hours_total = 0;
            
            // Process each day
            $currentDate = new \DateTime($startDate);
            $endDateTime = new \DateTime($endDate);
            
            while ($currentDate <= $endDateTime) {
                $dateStr = $currentDate->format('Y-m-d');
                $dayName = $currentDate->format('D');
                
                // Check for leave
                $isLeave = isset($leaveRecords[$dateStr]);
                if ($isLeave) $leave++;
                
                // Check for holiday
                $isHoliday = isset($holidayRecords[$dateStr]);
                if ($isHoliday) $holiday++;
                
                // Check if it's a weekend
                $isWeekend = strpos($staff->week_day_off ?? '', $dayName) !== false;
                
                // Check if it's an overtime day
                $isOvertimeDay = in_array($dateStr, $overtimeDays);
                
                // Get attendance record
                $attendance = $attendanceRecords[$dateStr] ?? null;
                
                // Calculate work hours
                $workHours = 0;
                if ($attendance) {
                    $timeIn = strtotime($attendance->time_in);
                    $timeOut = $attendance->time_out ? strtotime($attendance->time_out) : null;
                    
                    if (!$timeOut && $staff->business) {
                        $closingTime = $staff->business->business_closing_time ?? '18:00:00';
                        $timeOut = strtotime($dateStr . ' ' . $closingTime);
                    }
                    
                    if ($timeOut && $timeIn) {
                        $workHours = ($timeOut - $timeIn) / 3600;
                        // Ensure workHours is not negative
                        $workHours = max(0, $workHours);
                    }
                }
                
                // Check if present - ensure we properly detect presence
                $isPresent = false;
                if ($attendance) {
                    // Check if time_in is not midnight and not null
                    $timeInStr = $attendance->time_in;
                    $isPresent = ($timeInStr && $timeInStr != $dateStr . ' 00:00:00' && $timeInStr != '0000-00-00 00:00:00');
                }
                
                if ($isPresent) $presents++;
                
                // Check if late
                if ($attendance && $staff->day_start_time && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay) {
                    $startTime = strtotime($dateStr . ' ' . $staff->day_start_time);
                    $attendanceTime = strtotime($attendance->time_in);
                    $margin = $staff->business->timein_margin ?? 0;
                    $marginSeconds = $margin * 60;
                    
                    if ($attendanceTime > ($startTime + $marginSeconds)) {
                        $late++;
                    }
                }
                
                // Check if absent - updated logic
                $isAbsent = (!$isPresent && !$isHoliday && !$isLeave && !$isWeekend && !$isOvertimeDay);
                if ($isAbsent) $absents++;
                
                // Calculate overtime
                $overtimeHours = 0;
                if ($staff->overtime_allowed === 'Yes') {
                    if ($isOvertimeDay || $isWeekend || $isHoliday) {
                        $overtimeHours = $workHours;
                    } elseif ($isPresent && $staff->day_end_time) {
                        $regularHours = 8;
                        if ($staff->day_start_time && $staff->day_end_time) {
                            $start = strtotime($staff->day_start_time);
                            $end = strtotime($staff->day_end_time);
                            
                            if ($end < $start) $end += 86400;
                            $regularHours = ($end - $start) / 3600;
                        }
                        
                        if ($workHours > $regularHours) {
                            $overtimeHours = $workHours - $regularHours;
                        }
                    }
                }
                
                $workhours_total += $workHours;
                $overtime_hours_total += $overtimeHours;
                
                $currentDate->modify('+1 day');
            }
            
            // Ensure all values are numbers (not null)
            return [
                'presents' => (int)$presents,
                'absents' => (int)$absents,
                'late' => (int)$late,
                'leave' => (int)$leave,
                'holiday' => (int)$holiday,
                'workhours_total' => (float)$workhours_total,
                'overtime_hours_total' => (float)$overtime_hours_total
            ];
            
        } catch (\Exception $e) {
            Log::error('Error getting detailed month data: ' . $e->getMessage());
            // Return empty data instead of empty array
            return $this->getEmptyMonthData();
        }
    }

    
    /* Display staff payables/receivables page*/
    public function view_staff_payables_receivables($businessid = null)
    {
        $isHoUser = session('ho');
        $businessId = $businessid ?? session('selected_business_id') ?? session('business_id');
        $business = DB::table('business')->where('id_business', $businessId)->first();
        if ($isHoUser === 'Yes' && $businessid) {
            session(['selected_business_id' => $businessid]);
            $businessId = $businessid;
        }
        
        if (!$business) {
            abort(404, 'Business not found');
        }
        $businesses = collect([]);
        if ($isHoUser === 'Yes') {
            $businesses = Business::where('ho_accounts', 'Yes')
                ->orderBy('business_name', 'ASC')
                ->get();
        }
        
        $mappingBusinessId = $businessId;
        $accountsBusinessId = $businessId;
        
        $currentBusiness = Business::where('id_business', $businessId)->first();
        
        if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
            $hoBusiness = Business::where('ho', 'Yes')->first();
            if ($hoBusiness) {
                $accountsBusinessId = $hoBusiness->id_business;
                $mappingBusinessId = $hoBusiness->id_business;
            }
        } else {
            $accountsBusinessId = $businessId;
            $mappingBusinessId = $businessId;
        }
        
        $bank_accounts = DB::table('account_heads')
            ->where('account_type', 'Bank')
            ->where('business_id', $accountsBusinessId)
            ->where('account_head_status', 'Active')
            ->orderBy('account_head')
            ->get();

        $cash_accounts = DB::table('account_heads')
            ->where('account_type', 'Cash')
            ->where('business_id', $accountsBusinessId)
            ->where('account_head_status', 'Active')
            ->orderBy('account_head')
            ->get();

        $card_accounts = DB::table('account_heads')
            ->where('account_type', 'Card')
            ->where('business_id', $accountsBusinessId)
            ->where('account_head_status', 'Active')
            ->orderBy('account_head')
            ->get();
    
        $salary_account = DB::table('account_event_mapping')
            ->where('account_event_id', 26)
            ->where('entity_name', 'salary_payable')
            ->where('business_id', $mappingBusinessId)
            ->first();
        
        $salaryAccountId = $salary_account ? $salary_account->account_head_id : null;
        $payables = [];
        
        if ($salaryAccountId) {
            $payables = DB::table('staff as hs')
                ->select(
                    'hs.id_staff',
                    'hs.staff_fullname',
                    DB::raw('MONTH(av.voucher_date) as voucher_month'),
                    DB::raw('YEAR(av.voucher_date) as voucher_year'),
                    'av.id_account_vouchers',
                    'av.voucher_date',
                    'av.business_id',
                    'av.description',
                    DB::raw('SUM(avd.credit) as credit'),
                    DB::raw('SUM(COALESCE(sp.payment_amount, 0)) as paid'),
                    DB::raw('ROUND(SUM(avd.credit) - SUM(COALESCE(sp.payment_amount, 0)), 2) as salary_payable')
                )
                ->join('account_vouchers as av', function($join) use ($businessId) {
                    $join->on('hs.id_staff', '=', 'av.business_partner_id')
                        ->where('av.business_partner', '=', 3)
                        ->where('av.business_id', '=', $businessId) 
                        ->where('av.voucher_type', '=', 3)
                        ->where('av.voucher_status', '=', 'Active'); 
                })
                ->join('account_voucher_detail as avd', function($join) use ($salaryAccountId) {
                    $join->on('av.id_account_vouchers', '=', 'avd.account_voucher_id')
                        ->where('avd.account_head_id', '=', $salaryAccountId)
                        ->where('avd.credit', '>', 0);
                })
                ->leftJoin('staff_payments as sp', function($join) {
                    $join->on('av.id_account_vouchers', '=', 'sp.account_voucher_id')
                        ->where('sp.payment_type', '=', 'Salary'); 
                })
                ->where('hs.business_id', $businessId)
                ->where('hs.staff_active', 'Y')
                ->groupBy('hs.id_staff', 'av.id_account_vouchers', 'av.voucher_date')
                ->orderBy('av.voucher_date', 'ASC')
                ->get()
                ->map(function($item) {
                    return (array) $item;
                })
                ->toArray();
        }

        return view('hrm.staff_payables_receivables', compact(
            'business',
            'businesses',
            'bank_accounts',
            'cash_accounts',
            'card_accounts',
            'payables'
        ));
    }

    /*Check and process staff payments*/
    public function check_staff_payment(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $businessId = $request->input('business_id') ?? session('selected_business_id') ?? session('business_id');
            $voucher_date = $request->input('account_voucher_date');
            $payment_mode = $request->input('payment_mode');
            $instrument_number = $request->input('instrument_number');
            $bank_account = $request->input('bank_account');
            $cash_account = $request->input('cash_account');
            $tableData = json_decode($request->input('TableData'), true);
            
            $userName = session('user_name');
            
            // Validate payment date
            if (!$voucher_date) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment date is required'
                ], 400);
            }
            
            // Get business
            $business = DB::table('business')->where('id_business', $businessId)->first();
            if (!$business) {
                return response()->json([
                    'success' => false,
                    'message' => 'Business not found'
                ], 400);
            }

            $accountsBusinessId = $businessId;
            
            if ($business->ho_accounts === 'Yes') {
                // If business has HO accounts, voucher goes to HO business
                $hoBusiness = Business::where('ho', 'Yes')->first();
                if ($hoBusiness) {
                    $accountsBusinessId = $hoBusiness->id_business;
                }
            }
            
            // Get salary payable account
            $salaryAccount = DB::table('account_event_mapping')
                ->where('account_event_id', 26)
                ->where('entity_name', 'salary_payable')
                ->where('business_id', $accountsBusinessId)
                ->first();
            
            if (!$salaryAccount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Salary payable account not mapped for event 26'
                ], 400);
            }
            $loanAccount = DB::table('account_event_mapping')
                ->where('account_event_id', 26)
                ->where('entity_name', 'loan_deduction')
                ->where('business_id', $accountsBusinessId)
                ->first();
            
            // Get payment account based on mode
            $paymentAccountId = null;
            if ($payment_mode == 'Bank') {
                $paymentAccountId = $bank_account;
            } elseif ($payment_mode == 'Cash') {
                $paymentAccountId = $cash_account;
            }
            
            // Verify payment account exists in correct business
            $paymentAccount = DB::table('account_heads')
                ->where('id_account_heads', $paymentAccountId)
                ->where('business_id', $accountsBusinessId)
                ->where('account_head_status', 'Active')
                ->first();
                
            if (!$paymentAccount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Selected payment account is not valid or inactive'
                ], 400);
            }
            
            $createdVouchers = 0;
            $errors = [];
            
            foreach ($tableData as $index => $row) {
                try {
                    if (empty($row['payingnow']) || floatval($row['payingnow']) <= 0) {
                        continue;
                    }
                    $voucherId = $row['voucher_id'] ?? null;
                    $payableVoucheryear = $row['year'] ?? null;
                    $payableVouchermonth =$row['month'] ?? null;
                   $monthName = '';
                    if ($payableVouchermonth) {
                        $monthNumber = (int) str_replace(',', '', $payableVouchermonth);
                        if ($monthNumber >= 1 && $monthNumber <= 12) {
                            $monthName = \Carbon\Carbon::createFromFormat('m', str_pad($monthNumber, 2, '0', STR_PAD_LEFT))->format('F');
                        }
                    }
                    // if ($voucherId) {
                    //     $voucherBusinessId = DB::table('account_vouchers')->where('id_account_vouchers',$voucherId)->value('business_id');
                    //     return response()->json([
                    //         'success' => false,
                    //         'message' => "voucher business id is   $voucherBusinessId",
                    //         'errors' => $errors
                    //     ], 400);
                    // }
                    // Calculate net amount after loan deduction
                    $loanDeduction = isset($row['loan_deduction']) ? (float)$row['loan_deduction'] : 0;
                    $netPayment = (float)$row['payingnow'] - $loanDeduction;
                    
                    $staffDescription = $row['description'] ?? '';
                    $descriptionContainsLoan = stripos($staffDescription, 'Loan') !== false;
                    
                    // Set voucher description based on condition
                    if ($descriptionContainsLoan) {
                        $voucherDescription = 'Loan Payment to ' . $row['staff_fullname'] . 
                                            ($loanDeduction > 0 ? ' with Loan Deduction' : '');
                    } else {
                       $voucherDescription = "Salary Payment to {$row['staff_fullname']} (ID: {$row['staff_id']}) " . 
                                    "on {$voucher_date} for the Month {$monthName} {$payableVoucheryear}";
                    }
                    $staffPaymentId = DB::table('staff_payments')->insertGetId([
                        'staff_id' => $row['staff_id'],
                        'staff_name' => $row['staff_fullname'],
                        'payment_date' => $voucher_date,
                        'payment_mode' => $payment_mode,
                        'payment_type' => 'Salary',
                        'payment_amount' => $netPayment,
                        'deduction_amount' => $loanDeduction,
                        'payment_remarks' => $voucherDescription,
                        'payment_month' => date('F', strtotime($voucher_date)),
                        'payment_year' => date('Y', strtotime($voucher_date)),
                        'bank_account_id' => $paymentAccountId,
                        'month_number' => date('m', strtotime($voucher_date)),
                        'created_by' => $userName,
                        'created_on' => now(),
                        'account_voucher_id' => $row['voucher_id']
                    ]);
                    
                    $voucherId = DB::table('account_vouchers')->insertGetId([
                        'business_id' =>  $businessId,
                        'description' => $voucherDescription,
                        'voucher_amount' => $row['payingnow'],
                        'voucher_date' => $voucher_date,
                        'voucher_status' => 'Active',
                        'created_by' => $userName,
                        'business_partner' => 3,
                        'business_partner_id' => $row['staff_id'],
                        'business_partner_name' => $row['staff_fullname'],
                        'voucher_type' => 1,
                        'staffpayment_id' => $row['voucher_id'],
                        'payment_mode' => $payment_mode,
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                    
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherId,
                        'created_date' => $voucher_date,
                        'account_head_id' => $salaryAccount->account_head_id,
                        'debit' => $row['payingnow'],
                        'credit' => 0,
                        'detail_remarks' => $voucherDescription, 
                        'cost_center_type' => 'Branches',
                        'cost_center' => 'Main Outlet',
                        'payment_mode' => $payment_mode,
                        'instrument_number' => $instrument_number,
                        'created_by' => $userName,
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                    
                    // Create voucher details 
                    DB::table('account_voucher_detail')->insert([
                        'account_voucher_id' => $voucherId,
                        'created_date' => $voucher_date,
                        'account_head_id' => $paymentAccountId,
                        'debit' => 0,
                        'credit' => $netPayment,
                        'detail_remarks' => $voucherDescription,
                        'cost_center_type' => 'Branches',
                        'cost_center' => 'Main Outlet',
                        'payment_mode' => $payment_mode,
                        'instrument_number' => $instrument_number,
                        'created_by' => $userName,
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                    
                    $createdVouchers++;
                    
                } catch (\Exception $e) {
                    $errors[] = "Row {$index}: " . $e->getMessage();
                    Log::error('Error processing staff payment row ' . $index . ': ' . $e->getMessage());
                }
            }
            
            DB::commit();
            
            if ($createdVouchers > 0) {
                $response = [
                    'success' => true,
                    'message' => "Created {$createdVouchers} payment voucher(s) successfully.",
                    'count' => $createdVouchers
                ];
                
                if (!empty($errors)) {
                    $response['warnings'] = $errors;
                    $response['message'] = "Created {$createdVouchers} voucher(s) with some warnings";
                }
                
                return response()->json($response);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'No payments were processed.',
                    'errors' => $errors
                ], 400);
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating staff payments: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating payments: ' . $e->getMessage()
            ], 500);
        }
    }

    /*Payment slip in salary calculation*/
    public function paymentSlips(Request $request)
    {
        $isHoUser = session('ho') === 'Yes';
        $businessId = session('selected_business_id') ?? session('business_id');
        
        $business = Business::find($businessId);
        
        // Get staff for the dropdown
        $allStaff = Staff::where('business_id', $businessId)
            ->where('staff_active', 'Y')
            ->orderBy('staff_fullname')
            ->get();
        
        // Get businesses for HO users
        $businesses = collect([]);
        if ($isHoUser) {
            if (session('ho_accounts') === 'Yes') {
                $businesses = Business::where('ho_accounts', 'Yes')
                    ->orderBy('business_name', 'ASC')
                    ->get();
            } else {
                $businesses = Business::orderBy('business_name', 'ASC')->get();
            }
        }
        
        // Initialize variables
        $vouchers = [];
        $payments = [];
        $lastMonthYear = [];
        $selectedStaff = null;
        
        // Check if form was submitted
        if ($request->isMethod('post')) {
            $staffId = $request->input('staff_id');
            $month = $request->input('selected_month');
            $year = $request->input('selected_year');
            $monthName = $request->input('month');
            $voucherId = $request->input('voucherid', 0);
            // Get last month/year info
            $lastMonthYear = [
                'Month' => $month,
                'Year' => $year,
                'MonthName' => $monthName
            ];
           
            $salarySlips = $this->getSalarySlips($staffId, $month, $year, $voucherId, $monthName, $businessId);
           
            $paymentData = $this->getStaffPayments($staffId, $month, $year, $voucherId, $businessId);
            
            foreach ($salarySlips as $slip) {
                $vouchers[$slip->id_account_vouchers][] = (array)$slip;
            }
            
            foreach ($paymentData as $payment) {
                $payments[$payment->id_account_vouchers][] = (array)$payment;
            }
            
            // Combine data
            foreach ($vouchers as $key => $val) {
                if (isset($payments[$key])) {
                    $vouchers[$key]['payments'] = $payments[$key];
                }
            }
            // return $vouchers;
            $selectedStaff = $staffId;
        } else {
            $lastMonthYear = [
                'Month' => date('n'),
                'Year' => date('Y'),
                'MonthName' => date('F')
            ];
        }
        
        return view('hrm.payment_slips', compact(
            'business',
            'businesses',
            'allStaff',
            'isHoUser',
            'vouchers',
            'payments',
            'lastMonthYear',
            'selectedStaff'
        ));
    }

    private function getSalarySlips($staffId, $month, $year, $voucherId = 0, $monthName = '', $businessId)
    {
        $query = DB::table('account_vouchers as av')
            ->join('account_voucher_detail as avd', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('account_heads as ta', 'ta.id_account_heads', '=', 'avd.account_head_id')
            ->join('staff as s', 'av.business_partner_id', '=', 's.id_staff')
            ->join('account_event_mapping as aem', function($join) {
                $join->on('aem.account_head_id', '=', 'ta.id_account_heads')
                    ->where('aem.account_event_id', 26);
            })
            ->select(
                's.id_staff',
                's.staff_fullname',
                's.staff_eid',
                's.staff_designation',
                's.staff_department',
                's.staff_eid as staff_number',
                's.staff_date_of_joining',
                's.staff_cnic',
                's.staff_bank',
                's.staff_bank_account',
                's.staff_bank_account_title',
                'av.id_account_vouchers as id_account_vouchers',
                'avd.id_account_voucher_detail',
                DB::raw("DATE_FORMAT(av.voucher_date, '%d-%m-%Y') as payment_date"),
                DB::raw('YEAR(av.voucher_date) as payment_year'),
                'avd.payment_mode',
                'avd.debit as debit_amount',
                'avd.credit as credit_amount',
                'ta.account_head',
                'av.description as payment_remarks',
                'av.business_partner_name as staff_name',
                'aem.entity_name'
            )
            ->where('av.business_partner', 3)
            ->where('av.business_partner_id', $staffId)
            ->where('av.voucher_status', 'Active');
        // Add voucher filter
        if ($voucherId == 0) {
            $query->where('av.description', 'like', '%' . $monthName . ' ' . $year . '%');
        } else {
            $query->where('av.id_account_vouchers', $voucherId)
            ->where('av.description', 'like', '%' . $monthName . ' ' . $year . '%');
        }
        
        $query->orderBy('av.id_account_vouchers')
            ->orderBy('avd.id_account_voucher_detail');
        
        return $query->get();
    }

    private function getStaffPayments($staffId, $month, $year, $voucherId = 0, $businessId)
    {
        $query = DB::table('account_vouchers as av')
            ->join('account_voucher_detail as avd', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
            ->join('account_heads as ta', 'ta.id_account_heads', '=', 'avd.account_head_id')
            ->join('staff as s', 'av.business_partner_id', '=', 's.id_staff')
            ->join('account_event_mapping as aem', function($join) {
                $join->on('aem.account_head_id', '=', 'ta.id_account_heads')
                    ->where('aem.account_event_id', 26)
                    ->where('aem.entity_name', 'salary_payable');
            })
            ->select(
                's.id_staff',
                's.staff_fullname',
                's.staff_eid',
                's.staff_designation',
                's.staff_department',
                's.staff_eid as staff_number',
                's.staff_date_of_joining',
                's.staff_cnic',
                's.staff_bank',
                's.staff_bank_account',
                's.staff_bank_account_title',
                'av.id_account_vouchers',
                'avd.id_account_voucher_detail',
                DB::raw("DATE_FORMAT(av.voucher_date, '%d-%m-%Y') as payment_date"),
                DB::raw('YEAR(av.voucher_date) as payment_year'),
                'avd.payment_mode',
                'avd.debit as debit_amount',
                'avd.credit as credit_amount',
                'ta.account_head',
                'av.description as payment_remarks',
                'av.business_partner_name as staff_name'
            )
            ->where('av.business_partner', 3)
            ->where('av.business_partner_id', $staffId)
            ->where('av.voucher_status', 'Active');
        
        // Add voucher filter
        if ($voucherId == 0) {
            $query->whereMonth('av.voucher_date', $month)
                ->whereYear('av.voucher_date', $year);
        } else {
            $query->where('av.staffpayment_id', $voucherId);
        }
        
        $query->orderBy('av.id_account_vouchers')
            ->orderBy('avd.id_account_voucher_detail');
        
        return $query->get();
    }




    /**Display staff leave report page **/
    public function staffLeaveReport()
    {
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        $isHoUser = session('ho') === 'Yes';
        
        // Get businesses for HO users
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        
        // Get current staff list for non-HO users
        $staffList = collect([]);
        if (!$isHoUser) {
            $staffList = Staff::where('staff_active', 'Y')
                ->where('business_id', $businessId)
                ->orderBy('staff_fullname', 'asc')
                ->get();
        }
        
        return view('hrm.staff_leave_report', compact('business', 'businesses', 'isHoUser', 'staffList'));
    }

  
    /** Get staff leave report data **/
    public function getStaffLeaveReportData(Request $request)
    {
        try {
            $request->validate([
                'date_range' => 'required|array',
                'date_range.start' => 'required|date',
                'date_range.end' => 'required|date|after_or_equal:date_range.start'
            ]);
            
            $startDate = $request->date_range['start'];
            $endDate = $request->date_range['end'];
            $businessId = $request->get('business_id');
            
            // Base query for staff 
            $query = Staff::select(
                    'staff.id_staff',
                    'staff.id_staff as staff_code',
                    'staff.staff_fullname',
                    'staff.leave_policy_id',
                    'business.id_business',
                    'business.business_name',
                    'lp.policy_name',
                    'lp.id_leave_policy'
                )
                ->join('business', 'staff.business_id', '=', 'business.id_business')
                ->leftJoin('leave_policies as lp', 'staff.leave_policy_id', '=', 'lp.id_leave_policy')
                ->where('staff.staff_active', 'Y');
            
            // Filter by business if selected
            if ($businessId) {
                $query->where('staff.business_id', $businessId);
            } else {
                if (session('ho') !== 'Yes') {
                    $query->where('staff.business_id', session('business_id'));
                }
            }
            
            $staffList = $query->orderBy('business.business_name')
                ->orderBy('staff.staff_fullname')
                ->get();
            
            // Process each staff member
            $processedData = [];
            foreach ($staffList as $staff) {
                $staffData = $this->calculateStaffLeaveDetails($staff, $startDate, $endDate);
                $processedData[] = $staffData;
            }
            
            return response()->json([
                'success' => true,
                'data' => $processedData
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching staff leave report: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /** Calculate leave details for a staff member */
    private function calculateStaffLeaveDetails($staff, $startDate, $endDate)
    {
        $staffId = $staff->id_staff;
        $policyId = $staff->leave_policy_id;
        $noPolicy = empty($policyId);
        
        $result = [
            'staff_id' => $staff->staff_code,
            'staff_name' => $staff->staff_fullname,
            'business_name' => $staff->business_name,
            'policy_name' => $staff->policy_name ?? null,
            'no_policy' => $noPolicy,
            'policy_leaves' => '',
            'consumed_leaves' => 0,
            'remaining_leaves' => 0,
            'remaining_leaves_display' => '',
            'leave_details' => 'No leaves taken',
            'exceeded' => false,
            'status_display' => ''
        ];
        
        // If staff has no policy assigned
        if ($noPolicy) {
            $result['policy_leaves'] = '<span class="text-danger fw-bold">Leave policy is not assigned to this staff</span>';
            $result['remaining_leaves_display'] = '<span class="text-muted">N/A</span>';
            $result['status_display'] = '<span class="badge bg-secondary">No Policy</span>';
            return $result;
        }
        
        // Get all leave types from policy
        $policyLeaveTypes = [];
        $totalPolicyLeaves = 0;
        
        $policyDefinitions = DB::table('leave_policies_definition as lpd')
            ->join('leave_type as lt', 'lpd.leave_type_id', '=', 'lt.id_leave_type')
            ->where('lpd.leave_policy_id', $policyId)
            ->where('lt.status', 'Active')
            ->select(
                'lt.id_leave_type',
                'lt.leave_type',
                'lpd.allowed_count'
            )
            ->get();
        
        foreach ($policyDefinitions as $definition) {
            $policyLeaveTypes[$definition->id_leave_type] = [
                'leave_type' => $definition->leave_type,
                'allowed_count' => $definition->allowed_count,
                'consumed' => 0,
                'remaining' => $definition->allowed_count
            ];
            $totalPolicyLeaves += $definition->allowed_count;
        }
        
        // Calculate consumed leaves in date range
        $consumedLeaves = DB::table('staff_leaves')
            ->where('staff_id', $staffId)
            ->where('staff_leave_status', 'Active')
            ->whereBetween('staff_leave_date', [$startDate, $endDate])
            ->get();
        
        $totalConsumed = 0;
        $leaveDetails = [];
        
        foreach ($consumedLeaves as $leave) {
            $leaveTypeId = $leave->leave_type_id;
            $totalConsumed++;
            
            if (isset($policyLeaveTypes[$leaveTypeId])) {
                $policyLeaveTypes[$leaveTypeId]['consumed']++;
                $policyLeaveTypes[$leaveTypeId]['remaining'] = 
                    max(0, $policyLeaveTypes[$leaveTypeId]['allowed_count'] - $policyLeaveTypes[$leaveTypeId]['consumed']);
            }
            
            // Group by leave type for details
            if (!isset($leaveDetails[$leaveTypeId])) {
                $leaveTypeName = $policyLeaveTypes[$leaveTypeId]['leave_type'] ?? $leave->leave_type ?? 'Unknown';
                $leaveDetails[$leaveTypeId] = [
                    'type' => $leaveTypeName,
                    'dates' => [],
                    'count' => 0
                ];
            }
            
            $leaveDetails[$leaveTypeId]['dates'][] = date('d-m-Y', strtotime($leave->staff_leave_date));
            $leaveDetails[$leaveTypeId]['count']++;
        }
        
        // Calculate totals and remaining
        $totalRemaining = max(0, $totalPolicyLeaves - $totalConsumed);
        $exceeded = $totalConsumed > $totalPolicyLeaves;
        
        // Prepare policy leave types string
        $policyLeavesHtml = '<div class="policy-leaves">';
        foreach ($policyLeaveTypes as $type) {
            $statusClass = $type['consumed'] > $type['allowed_count'] ? 'text-danger' : 'text-success';
            $policyLeavesHtml .= '<div class="mb-1">';
            $policyLeavesHtml .= '<strong>' . $type['leave_type'] . ':</strong> ';
            $policyLeavesHtml .= '<span class="' . $statusClass . ' fw-bold">';
            $policyLeavesHtml .= $type['consumed'] . ' / ' . $type['allowed_count'];
            $policyLeavesHtml .= '</span>';
            $policyLeavesHtml .= '</div>';
        }
        $policyLeavesHtml .= '</div>';
        
        // Prepare leave details string
        $leaveDetailsHtml = '<div class="leave-details">';
        if (count($leaveDetails) > 0) {
            foreach ($leaveDetails as $detail) {
                $leaveDetailsHtml .= '<div class="mb-1">';
                $leaveDetailsHtml .= '<strong>' . $detail['type'] . ':</strong> ' . $detail['count'] . ' day(s)';
                if (!empty($detail['dates'])) {
                    $leaveDetailsHtml .= '<div class="small text-muted">' . implode(', ', $detail['dates']) . '</div>';
                }
                $leaveDetailsHtml .= '</div>';
            }
        } else {
            $leaveDetailsHtml .= '<span class="text-muted">No leaves taken</span>';
        }
        $leaveDetailsHtml .= '</div>';
        
        // Prepare remaining leaves display
        $remainingDisplay = '';
        if ($exceeded) {
            $remainingDisplay = '<span class="text-danger fw-bold">' . $totalRemaining . '</span>';
        } else if ($totalRemaining <= 3) {
            $remainingDisplay = '<span class="text-warning fw-bold">' . $totalRemaining . '</span>';
        } else {
            $remainingDisplay = '<span class="text-success fw-bold">' . $totalRemaining . '</span>';
        }
        
        // Prepare status display
        $statusDisplay = '';
        if ($exceeded) {
            $exceededBy = $totalConsumed - $totalPolicyLeaves;
            $statusDisplay = '<span class="badge bg-danger">Exceeded by ' . $exceededBy . '</span>';
        } else {
            $statusDisplay = '<span class="badge bg-success">Within Limit</span>';
        }
        
        return [
            'staff_id' => $staff->staff_code,
            'staff_name' => $staff->staff_fullname,
            'business_name' => $staff->business_name,
            'policy_name' => $staff->policy_name,
            'no_policy' => $noPolicy,
            'policy_leaves' => $policyLeavesHtml,
            'consumed_leaves' => $totalConsumed,
            'remaining_leaves' => $totalRemaining,
            'remaining_leaves_display' => $remainingDisplay,
            'leave_details' => $leaveDetailsHtml,
            'exceeded' => $exceeded,
            'status_display' => $statusDisplay
        ];
    }




    /**Display staff deductions report page*/
    public function staffDeductionsReport(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            
            // Get business ID
            if ($isHoUser && $request->filled('business_id')) {
                $businessId = $request->business_id;
            } else {
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }
            
            $business = Business::find($businessId);
            
            // Get businesses for HO users
            $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
            
            // Get staff list
            $staffList = collect([]);
            if ($isHoUser) {
                if ($request->filled('business_id')) {
                    $selectedBusinessId = $request->business_id;
                } elseif (session('selected_business_id')) {
                    $selectedBusinessId = session('selected_business_id');
                } else {
                    $selectedBusinessId = $businessId;
                }
                
                $staffList = Staff::where('business_id', $selectedBusinessId)
                    ->where('staff_active', 'Y')
                    ->orderBy('staff_fullname')
                    ->get();
            } else {
                $staffList = Staff::where('business_id', $businessId)
                    ->where('staff_active', 'Y')
                    ->orderBy('staff_fullname')
                    ->get();
            }
            
            // Get filter values with default as current month
            $startDate = $request->input('start_date', date('Y-m-01'));
            $endDate = $request->input('end_date', date('Y-m-t'));
            $staffId = $request->input('staff', 0);
            
            return view('hrm.staff_deductions_report', compact(
                'business',
                'staffList',
                'businesses',
                'startDate',
                'endDate',
                'staffId',
                'isHoUser'
            ));
            
        } catch (\Exception $e) {
            Log::error('Error in staffDeductionsReport: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error loading staff deductions report.');
        }
    }

    /**Get staff deductions data for AJAX request*/
    public function getStaffDeductionsData(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            
            // Get business ID from request or session
            if ($isHoUser && $request->filled('business_id') && $request->business_id !== "") {
                $businessId = $request->business_id;
            } else {
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }
            
            $request->validate([
                'start_date' => 'required|date',
                'end_date' => 'required|date|after_or_equal:start_date',
                'staff_id' => 'nullable|integer'
            ]);
            
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
            $staffId = $request->input('staff_id', 0);
            
            // Get current business
            $currentBusiness = Business::find($businessId);
            
            // Determine which business to get events from based on ho_accounts
            if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
                // If business has HO accounts, get events from HO business
                $hoBusiness = Business::where('ho', 'Yes')->first();
                $eventsBusinessId = $hoBusiness ? $hoBusiness->id_business : $businessId;
            } else {
                // Otherwise, use current business for events
                $eventsBusinessId = $businessId;
            }
            
            // Get deduction entities from account event mapping - Event ID 26
            $deductionAccounts = DB::table('account_event_mapping')
                ->where('account_event_id', 26)
                ->where('business_id', $eventsBusinessId)
                ->where('transaction_type', 'credit')
                ->where(function($query) {
                    $query->where('entity_name', 'like', '%deduction%')
                        ->orWhere('entity_name', 'like', '%loan%')
                        ->orWhere('entity_name', 'tax_amount')
                        ->orWhere('entity_name', 'other_deduction')
                        ->orWhere('entity_name', 'absent_deduction')
                        ->orWhere('entity_name', 'late_deduction')
                        ->orWhere('entity_name', 'loan_deduction')
                        ->orWhere('entity_name', 'provident_fund_payable');
                })
                ->select('account_head_id', 'entity_name')
                ->get()
                ->keyBy('account_head_id');
            
            // Create reverse mapping for easy lookup
            $deductionMapping = [];
            foreach ($deductionAccounts as $account) {
                $deductionMapping[$account->account_head_id] = $account->entity_name;
            }
            
            $query = Staff::select(
                    'staff.id_staff',
                    'staff.staff_fullname',
                    'staff.staff_salary',
                    'staff.salary_type',
                    'business.id_business',
                    'business.business_name',
                    'staff.deduction_policy'
                )
                ->join('business', 'staff.business_id', '=', 'business.id_business')
                ->where('staff.staff_active', 'Y');
            
            if ($businessId) {
                $query->where('staff.business_id', $businessId);
            } else {
                if (!$isHoUser) {
                    $query->where('staff.business_id', session('business_id'));
                }
            }
            
            if ($staffId > 0) {
                $query->where('staff.id_staff', $staffId);
            }
            
            $staffList = $query->orderBy('business.business_name')
                ->orderBy('staff.staff_fullname')
                ->get();
            
            $processedData = [];
            $totals = [
                'total_staff' => 0,
                'total_loan_deduction' => 0,
                'total_late_deduction' => 0,
                'total_absent_deduction' => 0,
                'total_tax_deduction' => 0,
                'total_other_deductions' => 0,
                'total_provident_fund' => 0,
                'grand_total' => 0
            ];
            
            foreach ($staffList as $staff) {
                $vouchers = DB::table('account_vouchers as av')
                    ->where('av.business_partner', 3)
                    ->where('av.business_partner_id', $staff->id_staff)
                    ->where('av.voucher_status', 'Active')
                    ->where('av.voucher_type', 3)
                    ->whereDate('av.voucher_date', '>=', $startDate)
                    ->whereDate('av.voucher_date', '<=', $endDate)
                    ->select('av.id_account_vouchers', 'av.voucher_date', 'av.description')
                    ->get();
                
                $monthlyBreakdown = [];
                $staffDeductions = [
                    'loan_deduction' => 0,
                    'late_deduction' => 0,
                    'absent_deduction' => 0,
                    'tax_amount' => 0,
                    'other_deduction' => 0,
                    'provident_fund_payable' => 0,
                    'total' => 0
                ];
                
                foreach ($vouchers as $voucher) {
                    $voucherDetails = DB::table('account_voucher_detail as avd')
                        ->where('avd.account_voucher_id', $voucher->id_account_vouchers)
                        ->where('avd.credit', '>', 0)
                        ->select('avd.account_head_id', 'avd.credit', 'avd.detail_remarks')
                        ->get();
                    
                    $monthYear = date('F Y', strtotime($voucher->voucher_date));
                    
                    if (!isset($monthlyBreakdown[$monthYear])) {
                        $monthlyBreakdown[$monthYear] = [
                            'loan_deduction' => 0,
                            'late_deduction' => 0,
                            'absent_deduction' => 0,
                            'tax_amount' => 0,
                            'other_deduction' => 0,
                            'provident_fund_payable' => 0,
                            'total' => 0
                        ];
                    }
                    
                    foreach ($voucherDetails as $detail) {
                        // Check if this account_head_id is in our deduction mapping
                        if (isset($deductionMapping[$detail->account_head_id])) {
                            $deductionType = $deductionMapping[$detail->account_head_id];
                            
                            $monthlyBreakdown[$monthYear][$deductionType] += $detail->credit;
                            $monthlyBreakdown[$monthYear]['total'] += $detail->credit;
                            
                            $staffDeductions[$deductionType] += $detail->credit;
                            $staffDeductions['total'] += $detail->credit;
                            
                            // Update totals
                            switch ($deductionType) {
                                case 'loan_deduction':
                                    $totals['total_loan_deduction'] += $detail->credit;
                                    break;
                                case 'late_deduction':
                                    $totals['total_late_deduction'] += $detail->credit;
                                    break;
                                case 'absent_deduction':
                                    $totals['total_absent_deduction'] += $detail->credit;
                                    break;
                                case 'tax_amount':
                                    $totals['total_tax_deduction'] += $detail->credit;
                                    break;
                                case 'other_deduction':
                                    $totals['total_other_deductions'] += $detail->credit;
                                    break;
                                case 'provident_fund_payable':
                                    $totals['total_provident_fund'] += $detail->credit;
                                    break;
                            }
                        }
                    }
                }
                
                // Prepare data - keep monthly breakdown raw for view processing
                $processedData[] = [
                    'staff_id' => $staff->id_staff,
                    'staff_name' => $staff->staff_fullname,
                    'business_name' => $staff->business_name,
                    'staff_salary' => $staff->staff_salary,
                    'salary_type' => $staff->salary_type,
                    'loan_deduction' => $staffDeductions['loan_deduction'],
                    'late_deduction' => $staffDeductions['late_deduction'],
                    'absent_deduction' => $staffDeductions['absent_deduction'],
                    'tax_deduction' => $staffDeductions['tax_amount'],
                    'other_deductions' => $staffDeductions['other_deduction'],
                    'provident_fund' => $staffDeductions['provident_fund_payable'],
                    'total_deductions' => $staffDeductions['total'],
                    'monthly_details' => $monthlyBreakdown,
                    'date_range' => [
                        'start' => $startDate,
                        'end' => $endDate
                    ]
                ];
            }
            
            $totals['total_staff'] = count($processedData);
            $totals['grand_total'] = $totals['total_loan_deduction'] + 
                                    $totals['total_late_deduction'] + 
                                    $totals['total_absent_deduction'] + 
                                    $totals['total_tax_deduction'] + 
                                    $totals['total_other_deductions'] + 
                                    $totals['total_provident_fund'];
            
            return response()->json([
                'success' => true,
                'data' => $processedData,
                'totals' => $totals,
                'date_range' => [
                    'start' => $startDate,
                    'end' => $endDate
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error fetching staff deductions report: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching data: ' . $e->getMessage()
            ], 500);
        }
    }



    /*** Display loan report page */
    public function loanReport()
    {
        $businessId = session('business_id');
        $business = Business::where('id_business', $businessId)->first();
        $isHoUser = session('ho') === 'Yes';
        $businesses = $isHoUser ? Business::where('ho_accounts', 'Yes')->orderBy('business_name', 'ASC')->get() : collect([]);
        
       
        $startDate = date('Y-m-01');
        $endDate = date('Y-12-t');
        
        // Get staff list based on user type
        $staffList = collect([]);
        if (!$isHoUser) {
            $staffList = Staff::where('staff_active', 'Y')
                ->where('business_id', $businessId)
                ->orderBy('staff_fullname', 'asc')
                ->get();
        } else {
            if (session('selected_business_id')) {
                $staffList = Staff::where('staff_active', 'Y')
                    ->where('business_id', session('selected_business_id'))
                    ->orderBy('staff_fullname', 'asc')
                    ->get();
            }
        }
        
        return view('hrm.loan_report', compact('business', 'staffList', 'businesses', 'startDate', 'endDate'));
    }

    /** Get loan report data for DataTable */
    public function getLoanReportData(Request $request)
    {
        try {
            $isHoUser = session('ho') === 'Yes';
            
            // Get date range from request or use defaults
            $startDate = $request->input('start_date', date('Y-m-01'));
            $endDate = $request->input('end_date', date('Y-m-t'));
            
            if ($isHoUser && $request->filled('business_id')) {
                $businessId = $request->business_id;
            } else {
                $businessId = session('business_id');
            }
            
            if ($isHoUser && session('selected_business_id')) {
                $businessId = session('selected_business_id');
            }

            // Main query to get staff with loan information
            $query = Staff::select(
                    'staff.id_staff',
                    'staff.staff_fullname',
                    'staff.id_staff as staff_code',
                    'business.business_name',
                    'staff.business_id',
                    
                    // Total loans approved for this staff
                    DB::raw('COALESCE(SUM(CASE WHEN la.application_status IN ("Approved", "Disbursed") THEN la.approved_amount ELSE 0 END), 0) as total_loan_given'),
                    
                    // Count of approved loans
                    DB::raw('COUNT(CASE WHEN la.application_status IN ("Approved", "Disbursed") THEN 1 END) as approved_loans_count'),
                    
                    // Count of pending loan requests
                    DB::raw('COUNT(CASE WHEN la.application_status = "Pending" THEN 1 END) as pending_loans_count'),
                    
                    // Get loan returned amount from account vouchers
                    DB::raw('COALESCE((
                        SELECT SUM(avd.credit) 
                        FROM account_vouchers av
                        JOIN account_voucher_detail avd ON av.id_account_vouchers = avd.account_voucher_id
                        JOIN account_event_mapping aem ON avd.account_head_id = aem.account_head_id
                        WHERE av.business_partner = 3
                        AND av.business_partner_id = staff.id_staff
                        AND av.voucher_status = "Active"
                        AND av.voucher_date BETWEEN "' . $startDate . '" AND "' . $endDate . '"
                        AND aem.account_event_id = 25
                        AND aem.entity_name = "loan_amount"
                        AND avd.credit > 0
                        AND av.business_id = staff.business_id
                    ), 0) as loan_returned'),
                    
                    // Get remaining loan amount (loan amount - returned)
                    DB::raw('COALESCE((
                        SELECT (SUM(avd.debit) - SUM(avd.credit))
                        FROM account_vouchers av
                        JOIN account_voucher_detail avd ON av.id_account_vouchers = avd.account_voucher_id
                        JOIN account_event_mapping aem ON avd.account_head_id = aem.account_head_id
                        WHERE av.business_partner = 3
                        AND av.business_partner_id = staff.id_staff
                        AND av.voucher_status = "Active"
                        AND aem.account_event_id = 25
                        AND aem.entity_name = "loan_amount"
                        AND av.business_id = staff.business_id
                    ), 0) as loan_remaining'),
                    
                    // Get last loan date
                    DB::raw('MAX(CASE WHEN la.application_status IN ("Approved", "Disbursed") THEN la.approved_on END) as last_loan_date')
                )
                ->leftJoin('loan_applications as la', function($join) use ($startDate, $endDate) {
                    $join->on('staff.id_staff', '=', 'la.staff_id')
                        ->whereBetween('la.created_on', [$startDate, $endDate]);
                })
                ->leftJoin('business', 'staff.business_id', '=', 'business.id_business')
                ->where('staff.staff_active', 'Y');

            // Apply business filter
            if ($isHoUser && $request->filled('business_id') && $request->business_id !== "") {
                $query->where('staff.business_id', $request->business_id);
            } else if (!$isHoUser) {
                $query->where('staff.business_id', $businessId);
            } else {
                $query->where('business.ho_accounts', 'Yes');
            }

            // Apply staff filter if provided
            if ($request->filled('staff_id') && $request->staff_id !== "") {
                $query->where('staff.id_staff', $request->staff_id);
            }

            // Apply status filter - MODIFIED TO SHOW ONLY THOSE WITH LOANS BY DEFAULT
            if ($request->filled('loan_status') && $request->loan_status !== "All") {
                if ($request->loan_status === "HasLoans") {
                    $query->having('total_loan_given', '>', 0);
                } elseif ($request->loan_status === "NoLoans") {
                    $query->having('total_loan_given', '=', 0)
                        ->having('pending_loans_count', '=', 0);
                } elseif ($request->loan_status === "PendingLoans") {
                    $query->having('pending_loans_count', '>', 0);
                }
            } else {
                // Default filter: Show only staff with loans or pending loans
                $query->having(function($q) {
                    $q->having('total_loan_given', '>', 0)
                    ->orHaving('pending_loans_count', '>', 0);
                });
            }

            // Group by staff
            $query->groupBy('staff.id_staff', 'staff.staff_fullname', 'staff.id_staff', 'business.business_name', 'staff.business_id');

            // Get total count before filtering for pagination
            $recordsTotal = Staff::where('staff_active', 'Y')
                ->where('staff.business_id', $businessId)
                ->count();

            // Apply ordering
            $orderColumn = $request->input('order.0.column', 3); // Default sort by total_loan_given
            $orderDirection = $request->input('order.0.dir', 'desc');
            
            $columns = [
                'staff_code',
                'staff_fullname',
                'business_name',
                'total_loan_given',
                'approved_loans_count',
                'pending_loans_count',
                'loan_returned',
                'loan_remaining',
                'last_loan_date',
                'total_monthly_installment'
            ];
            
            if (isset($columns[$orderColumn])) {
                $query->orderBy($columns[$orderColumn], $orderDirection);
            } else {
                $query->orderBy('total_loan_given', 'desc');
            }

            // Apply search if provided
            if ($request->filled('search.value')) {
                $search = $request->input('search.value');
                $query->where(function($q) use ($search) {
                    $q->where('staff.staff_fullname', 'like', "%{$search}%")
                    ->orWhere('staff.id_staff', 'like', "%{$search}%")
                    ->orWhere('business.business_name', 'like', "%{$search}%");
                });
            }

            // Get filtered count
            $filteredQuery = clone $query;
            $recordsFiltered = $filteredQuery->get()->count();

            // Apply pagination
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            
            if ($length != -1) {
                $query->offset($start)->limit($length);
            }

            $data = $query->get();

            // Calculate summary totals
            $summary = [
                'total_staff' => $data->count(),
                'total_loan_given' => number_format($data->sum('total_loan_given'), 2),
                'total_loan_returned' => number_format($data->sum('loan_returned'), 2),
                'total_loan_remaining' => number_format($data->sum('loan_remaining'), 2),
                'total_approved_loans' => $data->sum('approved_loans_count'),
                'total_pending_loans' => $data->sum('pending_loans_count'),
                'start_date' => $startDate,
                'end_date' => $endDate
            ];

            $processedData = [];
            foreach ($data as $staff) {
                // Get active loans for this staff
                $activeLoans = DB::table('loan_applications')
                    ->where('staff_id', $staff->id_staff)
                    ->whereIn('application_status', ['Approved', 'Disbursed'])
                    ->where('num_of_months', '>', 0)
                    ->get(['id_loan_applications', 'approved_amount', 'installment', 'num_of_months', 'application_status', 'approved_on']);
                
                $loanDetails = [];
                $totalMonthlyInstallment = 0;
                
                foreach ($activeLoans as $loan) {
                    // Calculate months paid based on loan deductions
                    $monthsPaid = DB::table('account_vouchers as av')
                        ->join('account_voucher_detail as avd', 'av.id_account_vouchers', '=', 'avd.account_voucher_id')
                        ->join('account_event_mapping as aem', 'avd.account_head_id', '=', 'aem.account_head_id')
                        ->where('av.business_partner', 3)
                        ->where('av.business_partner_id', $staff->id_staff)
                        ->where('av.voucher_status', 'Active')
                        ->where('aem.account_event_id', 25)
                        ->where('aem.entity_name', 'loan_amount')
                        ->where('avd.credit', '>', 0)
                        ->where('av.business_id', $staff->business_id)
                        ->where('av.loan_application_id', $loan->id_loan_applications)
                        ->count();
                    
                    $loanDetails[] = [
                        'loan_id' => $loan->id_loan_applications,
                        'amount' => $loan->approved_amount,
                        'installment' => $loan->installment,
                        'months' => $loan->num_of_months,
                        'remaining_months' => max(0, $loan->num_of_months - $monthsPaid),
                        'status' => $loan->application_status,
                        'approved_date' => $loan->approved_on
                    ];
                    $totalMonthlyInstallment += $loan->installment;
                }
                
                $hasLoans = $staff->total_loan_given > 0;
                $hasPending = $staff->pending_loans_count > 0;
                
                $processedData[] = [
                    'id_staff' => $staff->id_staff,
                    'staff_code' => $staff->staff_code ?? $staff->id_staff,
                    'staff_fullname' => $staff->staff_fullname,
                    'business_name' => $staff->business_name,
                    'total_loan_given' => $staff->total_loan_given > 0 ? number_format($staff->total_loan_given, 2) : '0.00',
                    'approved_loans_count' => $staff->approved_loans_count,
                    'pending_loans_count' => $staff->pending_loans_count,
                    'loan_returned' => number_format($staff->loan_returned, 2),
                    'loan_remaining' => number_format($staff->loan_remaining, 2),
                    'last_loan_date' => $staff->last_loan_date ? date('d-m-Y', strtotime($staff->last_loan_date)) : 'N/A',
                    'total_monthly_installment' => number_format($totalMonthlyInstallment, 2),
                    'loan_details' => $loanDetails,
                    'has_loans' => $hasLoans,
                    'has_pending' => $hasPending,
                    'raw_total_loan' => $staff->total_loan_given,
                    'raw_loan_returned' => $staff->loan_returned,
                    'raw_loan_remaining' => $staff->loan_remaining
                ];
            }

            return response()->json([
                'draw' => (int) $request->input('draw', 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $processedData,
                'summary' => $summary
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching loan report: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'draw' => (int) $request->input('draw', 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error fetching loan report data. Please try again.'
            ], 500);
        }
    }


    // Add this method to your HrmController
   public function resetAppRegistration(Request $request)
    {
        try {
            $staffId = $request->input('staffid');
            if (!$staffId) {
                return response()->json(['success' => false, 'message' => 'Staff ID is required']);
            }
            
            $businessId = session('business_id');
            
            if (!$businessId) {
                $businessId = session('businessid');
            }
            
            if (!$businessId) {
                return response()->json(['success' => false, 'message' => 'Business ID not found in session']);
            }
            
            $staff = Staff::where('id_staff', $staffId)
                ->where('business_id', $businessId)
                ->first();
                
            if (!$staff) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Staff not found for business ID: ' . $businessId
                ]);
            }
            
            $staff->fcm_id = null;
            $staff->save();
            
            return response()->json([
                'success' => true, 
                'message' => 'App registration reset successfully',
                'fcm_id' => null
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Error resetting app registration: ' . $e->getMessage());
            return response()->json([
                'success' => false, 
                'message' => 'Error resetting app registration: ' . $e->getMessage()
            ]);
        }
    }

}
