<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\LoginLog;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;

class LoginLogController extends Controller
{
    /**
     * Display the login logs page.
     */
    public function index()
    {
        return view('settings.login-logs.list');
    }

    /**
     * Search users for Select2 dropdown.
     */
    public function userSearch(Request $request)
    {
        $search = $request->input('term', '');
        
        $query = \App\Models\User::select('user_name')
            ->distinct()
            ->orderBy('user_name', 'asc')
            ->where('user_status', 'Active');

        // Filter out Super Users for non-Super Users
        $currentUser = auth()->user();
        if (!$currentUser->hasRole('Super User')) {
            // Exclude users with Super User role
            $query->whereNotExists(function($subquery) {
                $subquery->select(DB::raw(1))
                    ->from('spatie_model_has_roles')
                    ->join('spatie_roles', 'spatie_model_has_roles.role_id', '=', 'spatie_roles.id')
                    ->whereColumn('spatie_model_has_roles.id_users', 'users.id_users')
                    ->where('spatie_model_has_roles.model_type', 'App\\Models\\User')
                    ->where('spatie_roles.name', 'Super User');
            });
        }
        // Super Users: Show ALL users (no filtering needed)

        if ($search) {
            $query->where('user_name', 'like', '%' . $search . '%');
        }
        
        $users = $query->limit(100)
            ->get()
            ->map(function($user) {
                return [
                    'user_name' => $user->user_name
                ];
            });

        return response()->json($users);
    }

    /**
     * Get login logs data for DataTable (AJAX).
     */
    public function getLoginLogsData(Request $request)
    {
        // Clean up expired sessions - mark sessions as logged out if:
        // 1. Session doesn't exist in sessions table (browser closed, session expired)
        // 2. Session is older than session lifetime
        $sessionLifetime = (int) config('session.lifetime', 120); // default 120 minutes
        $expiredCutoff = Carbon::now()->subMinutes($sessionLifetime);
        
        // Get all active sessions that don't exist in sessions table
        $expiredSessions = LoginLog::whereNull('logout_time')
            ->whereNotExists(function($query) {
                $query->select(DB::raw(1))
                      ->from('sessions')
                      ->whereColumn('sessions.id', 'login_logs.session_id');
            })
            ->get();
        
        // Mark each expired session with logout_time = login_time + session lifetime
        foreach ($expiredSessions as $log) {
            $estimatedLogoutTime = Carbon::parse($log->login_time)->addMinutes($sessionLifetime);
            // Don't set logout_time to future, use current time if estimated is in future
            $logoutTime = $estimatedLogoutTime->isFuture() ? Carbon::now() : $estimatedLogoutTime;
            $log->update(['logout_time' => $logoutTime]);
        }
        
        // Also mark very old sessions (older than session lifetime) as expired
        LoginLog::whereNull('logout_time')
            ->where('login_time', '<', $expiredCutoff)
            ->update(['logout_time' => $expiredCutoff]);

        $query = LoginLog::select(
            'login_logs.id_login_log',
            'login_logs.user_id',
            'login_logs.user_name',
            'login_logs.business_id',
            'login_logs.login_time',
            'login_logs.logout_time',
            'login_logs.session_id',
            'business.business_name',
            DB::raw('CASE WHEN sessions.id IS NOT NULL THEN 1 ELSE 0 END as session_exists')
        )
        ->leftJoin('sessions', 'login_logs.session_id', '=', 'sessions.id')
        ->leftJoin('business', 'login_logs.business_id', '=', 'business.id_business')
        ->orderBy('login_logs.login_time', 'desc');

        // Filter out Super User login logs for non-Super Users
        $currentUser = auth()->user();
        if (!$currentUser->hasRole('Super User')) {
            // Exclude login logs where the user has the Super User role
            $query->whereNotExists(function($subquery) {
                $subquery->select(DB::raw(1))
                    ->from('spatie_model_has_roles')
                    ->join('spatie_roles', 'spatie_model_has_roles.role_id', '=', 'spatie_roles.id')
                    ->whereColumn('spatie_model_has_roles.id_users', 'login_logs.user_id')
                    ->where('spatie_model_has_roles.model_type', 'App\\Models\\User')
                    ->where('spatie_roles.name', 'Super User');
            });
        }
        // Super Users: Show ALL login logs (no filtering needed)

        // Apply filters
        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('login_logs.login_time', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('login_logs.login_time', '<=', $request->end_date);
        }

        if ($request->has('user_name') && $request->user_name) {
            $query->where('login_logs.user_name', 'like', '%' . $request->user_name . '%');
        }

        // DataTables server-side processing
        return DataTables::of($query)
            ->addColumn('formatted_login_time', function ($log) {
                if ($log->login_time) {
                    // Convert to Pakistan time (GMT+5)
                    $time = new \DateTime($log->login_time, new \DateTimeZone('UTC'));
                    $time->setTimezone(new \DateTimeZone('Asia/Karachi'));
                    return $time->format('Y-m-d H:i:s');
                }
                return '-';
            })
            ->addColumn('formatted_logout_time', function ($log) {
                if ($log->logout_time) {
                    // Convert to Pakistan time (GMT+5)
                    $time = new \DateTime($log->logout_time, new \DateTimeZone('UTC'));
                    $time->setTimezone(new \DateTimeZone('Asia/Karachi'));
                    return $time->format('Y-m-d H:i:s');
                }
                return null;
            })
            ->addColumn('status', function ($log) {
                // Session is active only if logout_time is null AND session still exists in sessions table
                if ($log->logout_time) {
                    return '<span class="badge bg-success">Logged Out</span>';
                } elseif ($log->session_exists) {
                    return '<span class="badge bg-warning">Active</span>';
                } else {
                    // Session expired or was closed without proper logout
                    return '<span class="badge bg-secondary">Expired</span>';
                }
            })
            ->rawColumns(['status'])
            ->make(true);
    }
}
