<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Illuminate\Support\Facades\File;

class MyAccountController extends Controller
{
    /**
     * Display the My Account page.
     */
    public function index()
    {
        $user = Auth::user();
        
        return view('my-account.profile_settings', [
            'user' => $user,
        ]);
    }

    /**
     * Update user account information.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'user_email' => 'nullable|email|max:255',
            'user_mobile' => 'nullable|string|max:255',
            'user_address' => 'nullable|string|max:255',
            'new_password' => 'nullable|string|min:6',
            'confirm_password' => 'nullable|string|min:6',
            'user_image' => 'nullable|string',
        ]);

        // Validate password match if password is being changed
        if ($request->filled('new_password')) {
            if ($request->new_password !== $request->confirm_password) {
                return back()->withErrors(['confirm_password' => 'Password and confirmation do not match.'])->withInput();
            }
        }

        try {
            DB::beginTransaction();

            // Update email if provided
            if ($request->has('user_email')) {
                $user->user_email = $request->user_email;
            }

            // Update mobile if provided
            if ($request->has('user_mobile')) {
                $user->user_mobile = $request->user_mobile;
            }

            // Update address if provided
            if ($request->has('user_address')) {
                $user->user_address = $request->user_address;
            }

            // Update password if provided
            if ($request->filled('new_password')) {
                $user->user_password = md5($request->new_password);
            }

            // Update image if provided
            if ($request->has('user_image') && $request->user_image) {
                $user->user_image = $request->user_image;
                // Update session
                session(['user_image' => $request->user_image]);
            }

            $user->save();

            DB::commit();

            return redirect()->route('my-account')->with('success', 'Account updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Failed to update account: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Upload user image.
     */
    public function uploadImage(Request $request)
    {
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path('images/users'), $filename);

            return response()->json([
                'success' => true,
                'filename' => $filename
            ]);
        }

        return response()->json(['success' => false], 400);
    }

    /**
     * Get existing user images.
     */
    public function existingImages()
    {
        $path = public_path('images/users');
        $urls = [];

        if (File::exists($path)) {
            foreach (File::files($path) as $file) {
                $urls[] = asset('images/users/' . $file->getFilename());
            }
        }

        return response()->json($urls);
    }
}
