<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class OnlineShopController extends Controller
{
    public function orderCreated(Request $request)
    {
        $shopDomain = null;
        $orderNumber = null;
        $customerName = null;
        $data = null;
        
        try {
            // Step 1: Get request data
            $payload = $request->getContent();
            $shopDomain = $request->header('X-Shopify-Shop-Domain');
            
            // Validate required header
            if (!$shopDomain) {
                throw new \Exception("Missing X-Shopify-Shop-Domain header");
            }
            
            // Step 2: Validate JSON payload
            $data = json_decode($payload, true);
            if (!is_array($data)) {
                throw new \Exception("Invalid JSON payload");
            }
            
            // Step 3: Check all conditions and get integration
            $integration = $this->checkConditionsAndGetIntegration($shopDomain, $data);
            
            // Step 4: Extract order details
            $orderId = $data['id'] ?? null;
            $orderNumber = $data['order_number'] ?? $data['name'] ?? null;
            $customerName = $this->extractCustomerName($data);
            $customerEmail = $data['email'] ?? $data['contact_email'] ?? null;
            $customerPhone = $this->extractCustomerPhone($data);
            
            // Step 5: Save to main orders table
            $saveResult = $this->saveOrderToDatabase(
                $integration->business_id, 
                $shopDomain, 
                $orderId, 
                $orderNumber, 
                $customerName, 
                $customerEmail, 
                $customerPhone, 
                $data
            );
            
            if (!$saveResult['success']) {
                throw new \Exception("Failed to save order to database: " . $saveResult['message']);
            }
            
            // Step 6: Always insert into details table and check for payload changes
            $this->saveToOrderDetailsTable(
                $saveResult['log_id'],
                $integration->business_id,
                $orderId,
                $orderNumber,
                $data,
                'order_created'
            );
            
            return response()->json([
                'message' => 'Order processed successfully',
                'order_number' => $orderNumber,
                'business_id' => $integration->business_id,
                'database_saved' => true,
                'details_inserted' => true,
                'sale_order_created' => false,
            ], 200);
            
        } catch (\Exception $e) {
            // Step 7: Log error with daily file rotation
            $this->logErrorToFile(
                get_class($e),
                $e->getMessage(),
                $shopDomain ?? 'unknown',
                $orderNumber ?? 'unknown',
                $customerName ?? 'unknown',
                $e->getFile(),
                $e->getLine()
            );
            
            return response()->json([
                'error' => 'Order processing failed',
                'message' => $e->getMessage(),
            ], 500);
        }
    }
      /**
     * Check if main order table should be updated
     */
    private function shouldUpdateMainOrder($existingOrder, $newData, $isCancelled)
    {
        // Decode existing order payload
        $existingPayload = json_decode($existingOrder->order_payload, true);
        if (!$existingPayload) {
            return true; // Can't compare, so update
        }
        
        // Check if payload has changed (compare normalized JSON)
        $newJson = json_encode($newData);
        $existingJson = json_encode($existingPayload);
        
        if (json_encode(json_decode($newJson, true)) !== json_encode(json_decode($existingJson, true))) {
            return true; // Payload changed
        }
        
        // Check if order status has changed
        $existingStatus = $existingOrder->status;
        $newStatus = $isCancelled['cancelled'] ? 'order_is_cancelled' : 'sale_order_not_created';
        
        if ($existingStatus !== $newStatus) {
            return true;
        }
        
        return false; 
    }
    
    /**
     * Handle order updates with fallback to create
     * Updates existing orders or creates new ones if they don't exist
     */
     public function orderUpdated(Request $request)
    {
        $shopDomain = null;
        $orderNumber = null;
        $customerName = null;
        $data = null;
        
        try {
            // Step 1: Get request data
            $payload = $request->getContent();
            $shopDomain = $request->header('X-Shopify-Shop-Domain');
            
            // Validate required header
            if (!$shopDomain) {
                throw new \Exception("Missing X-Shopify-Shop-Domain header");
            }
            
            // Step 2: Validate JSON payload
            $data = json_decode($payload, true);
            if (!is_array($data)) {
                throw new \Exception("Invalid JSON payload");
            }
            
            // Step 3: Get integration and extract order details
            $orderId = $data['id'] ?? null;
            $orderNumber = $data['order_number'] ?? $data['name'] ?? null;
            
            // Validate required fields
            if (!$orderId || !$orderNumber) {
                throw new \Exception("Missing order ID or order number");
            }
            
            // Find active integration
            $integration = DB::table('business_integrations')
                ->where('domain', $shopDomain)
                ->where('status', 'Active')
                ->first();
            
            if (!$integration) {
                throw new \Exception("No active integration found for domain: {$shopDomain}");
            }
            
            // Step 4: Check if this is a cancellation update
            $isCancelled = $this->checkIfOrderCancelled($data);
            
            // Step 5: Attempt to locate existing order
            $existingOrder = DB::table('online_order_logs')
                ->where('business_id', $integration->business_id)
                ->where('order_number', $orderNumber)
                ->first();
            
            if ($existingOrder) {
                // Order exists: Update it
                $customerName = $this->extractCustomerName($data);
                $customerEmail = $data['email'] ?? $data['contact_email'] ?? null;
                $customerPhone = $this->extractCustomerPhone($data);
                
                // Determine payload nature based on status
                $payloadNature = 'order_updated';
                if ($isCancelled['cancelled']) {
                    $payloadNature = 'order_cancelled';
                }
                
                // Step 6: Check if we should update main table
                // Only update if payload has changed or order status changed
                $shouldUpdateMainTable = $this->shouldUpdateMainOrder($existingOrder, $data, $isCancelled);
                
                if ($shouldUpdateMainTable) {
                    // Prepare update data for main table
                    $updateData = [
                        'order_payload' => json_encode($data),
                        'customer_name' => $customerName,
                        'customer_email' => $customerEmail,
                        'customer_phone' => $customerPhone,
                        'updated_at' => now(),
                    ];
                    
                    // If order is cancelled, update status and store cancellation reason
                    if ($isCancelled['cancelled']) {
                        $cancellationDetails = $this->extractCancellationDetails($data);
                        
                        $updateData['status'] = 'order_is_cancelled';
                        $updateData['status_reason'] = $cancellationDetails['reason'] ?? 'Order cancelled';
                    }
                    // If order was previously cancelled but is now active again
                    elseif ($existingOrder->status === 'order_is_cancelled' && !$isCancelled['cancelled']) {
                        $updateData['status'] = 'sale_order_not_created';
                        $updateData['status_reason'] = 'Order reactivated';
                    }
                    
                    // Update main order table
                    DB::table('online_order_logs')
                        ->where('id_online_order_logs', $existingOrder->id_online_order_logs)
                        ->update($updateData);
                }
                
                // Step 7: Insert into details table (method will check for payload changes)
                $detailResult = $this->saveToOrderDetailsTable(
                    $existingOrder->id_online_order_logs,
                    $integration->business_id,
                    $orderId,
                    $orderNumber,
                    $data,
                    $payloadNature
                );
                
                return response()->json([
                    'message' => 'Order processed successfully',
                    'order_number' => $orderNumber,
                    'business_id' => $integration->business_id,
                    'payload_nature' => $payloadNature,
                    'main_table_updated' => $shouldUpdateMainTable,
                    'detail_record_inserted' => $detailResult['inserted'] ?? false,
                    'payload_changed' => $detailResult['payload_changed'] ?? false,
                    'cancelled' => $isCancelled['cancelled'],
                    'cancellation_reason' => $isCancelled['cancelled'] ? ($isCancelled['reason'] ?? null) : null,
                ], 200);
            } else {
                // Order does not exist: Fallback to create flow
                return $this->orderCreated($request);
            }
            
        } catch (\Exception $e) {
            // Log error
            $this->logErrorToFile(
                get_class($e),
                $e->getMessage(),
                $shopDomain ?? 'unknown',
                $orderNumber ?? 'unknown',
                $customerName ?? 'unknown',
                $e->getFile(),
                $e->getLine()
            );
            
            return response()->json([
                'error' => 'Order update failed',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Check if order is cancelled in the payload
     */
     private function checkIfOrderCancelled($data)
    {
        // Shopify cancellation indicators
        $cancelledAt = $data['cancelled_at'] ?? null;
        $financialStatus = $data['financial_status'] ?? null;
        $cancelReason = $data['cancel_reason'] ?? null;
        
        $isCancelled = false;
        $reason = null;
        
        // Check if order is cancelled
        if (!empty($cancelledAt)) {
            $isCancelled = true;
            $reason = $cancelReason ?? 'Cancelled via Shopify';
        }
        // Check financial status for refunded/voided orders
        elseif (in_array($financialStatus, ['refunded', 'voided'])) {
            $isCancelled = true;
            $reason = "Financial status: {$financialStatus}";
        }
        
        return [
            'cancelled' => $isCancelled,
            'cancelled_at' => $cancelledAt,
            'reason' => $reason,
            'financial_status' => $financialStatus
        ];
    }
    
    /**
     * Extract cancellation details from payload
     */
      private function extractCancellationDetails($data)
    {
        $cancelledAt = $data['cancelled_at'] ?? null;
        $cancelReason = $data['cancel_reason'] ?? null;
        $financialStatus = $data['financial_status'] ?? null;
        
        // Convert Shopify timestamp to MySQL datetime format
        $cancellationDateTime = null;
        if ($cancelledAt) {
            try {
                $cancellationDateTime = Carbon::parse($cancelledAt)->toDateTimeString();
            } catch (\Exception $e) {
                $cancellationDateTime = now()->toDateTimeString();
            }
        }
        
        // Build detailed cancellation reason
        $reasonParts = [];
        
        if ($cancelReason) {
            $reasonParts[] = "Reason: {$cancelReason}";
        }
        
        if ($financialStatus) {
            $reasonParts[] = "Financial Status: {$financialStatus}";
        }
        
        // Check for refund details
        if (isset($data['refunds']) && is_array($data['refunds']) && count($data['refunds']) > 0) {
            $refund = $data['refunds'][0];
            if (isset($refund['note'])) {
                $reasonParts[] = "Refund Note: {$refund['note']}";
            }
        }
        
        $reason = !empty($reasonParts) ? implode(' | ', $reasonParts) : 'Cancelled by customer';
        
        return [
            'cancelled_at' => $cancellationDateTime,
            'reason' => $reason,
            'original_cancelled_at' => $cancelledAt,
            'cancel_reason' => $cancelReason,
            'financial_status' => $financialStatus
        ];
    }
    
    /**
     * Save to order details table and check for payload changes
     */
    private function saveToOrderDetailsTable($logId, $businessId, $orderId, $orderNumber, $data, $payloadNature)
    {
        try {
            $payloadJson = json_encode($data);
            
            // Get the previous detail record for this order
            $previousDetail = DB::table('online_order_logs_detail')
                ->where('order_id', $orderId)
                ->where('business_id', $businessId)
                ->orderBy('id_order_detail', 'desc')
                ->first();
            
            $currentTime = now();
            
            // Check if payload has changed from previous record
            $payloadChanged = false;
            
            if ($previousDetail && $previousDetail->order_payload) {
                // Normalize JSON for comparison (decode and re-encode)
                $currentPayload = json_encode(json_decode($payloadJson, true));
                $previousPayload = json_encode(json_decode($previousDetail->order_payload, true));
                
                // Check if payload has actually changed
                if ($currentPayload !== $previousPayload) {
                    $payloadChanged = true;
                }
                
                // Also check if payload_nature has changed (e.g., from order_updated to order_cancelled)
                if ($previousDetail->payload_nature !== $payloadNature) {
                    $payloadChanged = true;
                }
            } else {
                // No previous record exists, so this is a new payload
                $payloadChanged = true;
            }
            
            // Only insert new record if:
            // 1. This is a new order (order_created) OR
            // 2. Payload has changed OR
            // 3. Payload nature has changed
            if ($payloadNature === 'order_created' || $payloadChanged) {
                // Prepare insert data
                $insertData = [
                    'business_id' => $businessId,
                    'order_id' => $orderId,
                    'order_number' => $orderNumber,
                    'order_payload' => $payloadJson,
                    'payload_nature' => $payloadNature,
                    'created_at' => $currentTime,
                ];
                
                // Only set updated_at if payload has changed (not for order_created)
                if ($payloadNature !== 'order_created' && $payloadChanged) {
                    $insertData['updated_at'] = $currentTime;
                }
                
                // Insert new record
                DB::table('online_order_logs_detail')->insert($insertData);
                
                return [
                    'success' => true,
                    'inserted' => true,
                    'has_updated_at' => isset($insertData['updated_at']),
                    'payload_changed' => $payloadChanged,
                    'message' => 'Record saved to details table successfully'
                ];
            } else {
                // Payload hasn't changed and it's not a new order
                // Do NOT insert new record
                return [
                    'success' => true,
                    'inserted' => false,
                    'has_updated_at' => false,
                    'payload_changed' => false,
                    'message' => 'Payload unchanged, no new record inserted'
                ];
            }
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'inserted' => false,
                'has_updated_at' => false,
                'payload_changed' => false,
                'message' => 'Failed to process details table: ' . $e->getMessage()
            ];
        }
    }
    /**
     * Function 1: Check all conditions and get integration
     */
    private function checkConditionsAndGetIntegration($shopDomain, $data)
    {
        // Extract order details
        $orderId = $data['id'] ?? null;
        $orderNumber = $data['order_number'] ?? $data['name'] ?? null;
        $customerName = $this->extractCustomerName($data);
        
        // Validate required fields
        if (!$orderId || !$orderNumber) {
            throw new \Exception("Missing order ID or order number");
        }
        
        // Find active integration
        $integration = DB::table('business_integrations')
            ->where('domain', $shopDomain)
            ->where('status', 'Active')
            ->first();
        
        if (!$integration) {
            throw new \Exception("No active integration found for domain: {$shopDomain}");
        }
        
        // Check for duplicate order in main table
        $isDuplicate = DB::table('online_order_logs')
            ->where('business_id', $integration->business_id)
            ->where('order_number', $orderNumber)
            ->exists();
        
        if ($isDuplicate) {
            throw new \Exception("DUPLICATE_ORDER: Order #{$orderNumber} already exists for business ID: {$integration->business_id}");
        }
        
        return $integration;
    }
    
    /**
     * Function 2: Save order to database with try-catch
     */
    private function saveOrderToDatabase($businessId, $shopDomain, $orderId, $orderNumber, $customerName, $customerEmail, $customerPhone, $data)
    {
        try {
            DB::beginTransaction();
            
            // Check if this is a cancelled order during creation
            $isCancelled = $this->checkIfOrderCancelled($data);
            
            $insertData = [
                'business_id' => $businessId,
                'platform' => 'shopify',
                'shop_domain' => $shopDomain,
                'order_id' => $orderId,
                'order_number' => $orderNumber,
                'customer_name' => $customerName,
                'customer_email' => $customerEmail,
                'customer_phone' => $customerPhone,
                'order_payload' => json_encode($data),
                'status' => 'sale_order_not_created',
                'created_at' => now(),
                'updated_at' => now(),
            ];
            
            // If order is cancelled during creation, set cancellation status and reason
            if ($isCancelled['cancelled']) {
                $cancellationDetails = $this->extractCancellationDetails($data);
                
                $insertData['status'] = 'order_is_cancelled';
                $insertData['status_reason'] = $cancellationDetails['reason'] ?? 'Order cancelled via Shopify';
                
                // Store cancellation timestamp if available
                if (isset($cancellationDetails['cancelled_at'])) {
                    $insertData['cancelled_at'] = $cancellationDetails['cancelled_at'];
                }
            }
            
            $logId = DB::table('online_order_logs')->insertGetId($insertData, 'id_online_order_logs');
            
            DB::commit();
            
            return [
                'success' => true,
                'message' => 'Order saved to database successfully',
                'log_id' => $logId,
                'status' => $insertData['status'],
                'is_cancelled' => $isCancelled['cancelled']
            ];
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return [
                'success' => false,
                'message' => $e->getMessage(),
                'error' => 'Database save failed'
            ];
        }
    }
    
    /**
     * Function 3: Log error to text file with daily rotation
     */
    private function logErrorToFile(
        $errorType, 
        $errorMessage, 
        $shopDomain, 
        $orderNumber, 
        $customerName,
        $errorFile = null,
        $errorLine = null
    ) {
        try {
            $timestamp = Carbon::now()->toDateTimeString();
            $day = Carbon::now()->format('l');
            
            $logLine = "[" . $timestamp . "] ";
            $logLine .= "Day: " . $day . " | ";
            $logLine .= "Shop Domain: " . $shopDomain . " | ";
            $logLine .= "Order Number: " . $orderNumber . " | ";
            $logLine .= "Customer Name: " . $customerName . " | ";
            $logLine .= "Error Type: " . $errorType . " | ";
            $logLine .= "Error Message: " . $errorMessage;
            
            if ($errorFile && $errorLine) {
                $logLine .= " | File: " . basename($errorFile) . ":" . $errorLine;
            }
            
            $logLine .= PHP_EOL . str_repeat("-", 100) . PHP_EOL;
            
            // Generate daily filename
            $dailyFilename = 'shopify_webhook_errors_' . Carbon::now()->format('Y-m-d') . '.txt';
            
            // Cleanup old log files (older than 7 days)
            $this->cleanupOldLogFiles();
            
            // Append to today's log file
            Storage::append($dailyFilename, $logLine);
            
        } catch (\Exception $e) {
            \Log::error('Failed to save error to file: ' . $e->getMessage());
        }
    }
    
    /**
     * Cleanup old log files (older than 7 days)
     */
    private function cleanupOldLogFiles()
    {
        try {
            $daysToKeep = 7;
            $cutoffDate = Carbon::now()->subDays($daysToKeep);
            
            // Get all files in storage
            $allFiles = Storage::files();
            
            foreach ($allFiles as $file) {
                // Check if file matches our error log pattern
                if (preg_match('/shopify_webhook_errors_(\d{4}-\d{2}-\d{2})\.txt$/', $file, $matches)) {
                    $fileDate = Carbon::createFromFormat('Y-m-d', $matches[1]);
                    
                    // If file is older than cutoff date, delete it
                    if ($fileDate->lessThan($cutoffDate)) {
                        Storage::delete($file);
                        \Log::info("Deleted old error log file: {$file}");
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to cleanup old log files: ' . $e->getMessage());
        }
    }
    
    /**
     * Extract customer name from payload
     */
    private function extractCustomerName($data)
    {
        if (isset($data['billing_address'])) {
            $firstName = $data['billing_address']['first_name'] ?? '';
            $lastName = $data['billing_address']['last_name'] ?? '';
            $name = trim("{$firstName} {$lastName}");
            if (!empty($name)) return $name;
        }
        
        if (isset($data['shipping_address'])) {
            $firstName = $data['shipping_address']['first_name'] ?? '';
            $lastName = $data['shipping_address']['last_name'] ?? '';
            $name = trim("{$firstName} {$lastName}");
            if (!empty($name)) return $name;
        }
        
        if (isset($data['customer'])) {
            $firstName = $data['customer']['first_name'] ?? '';
            $lastName = $data['customer']['last_name'] ?? '';
            $name = trim("{$firstName} {$lastName}");
            if (!empty($name)) return $name;
        }
        
        return 'Unknown Customer';
    }
    
    /**
     * Extract customer phone with priority
     */
    private function extractCustomerPhone($data)
    {
        // Priority 1: Main order phone
        if (isset($data['phone']) && !empty(trim($data['phone']))) {
            return trim($data['phone']);
        }
        
        // Priority 2: Customer object phone
        if (isset($data['customer']['phone']) && !empty(trim($data['customer']['phone']))) {
            return trim($data['customer']['phone']);
        }
        
        // Priority 3: Customer default address phone
        if (isset($data['customer']['default_address']['phone']) && !empty(trim($data['customer']['default_address']['phone']))) {
            return trim($data['customer']['default_address']['phone']);
        }
        
        // Priority 4: Billing address phone
        if (isset($data['billing_address']['phone']) && !empty(trim($data['billing_address']['phone']))) {
            return trim($data['billing_address']['phone']);
        }
        
        // Priority 5: Shipping address phone
        if (isset($data['shipping_address']['phone']) && !empty(trim($data['shipping_address']['phone']))) {
            return trim($data['shipping_address']['phone']);
        }
        
        return null;
    }
}