<?php

namespace App\Http\Controllers;

use App\Models\AccountEventMapping;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Brands;
use App\Models\Customers;
use App\Models\Products;
use App\Models\KitConfig;
use App\Models\CustomerOrders;
use App\Models\OrderProducts;
use App\Models\Stores;
use App\Models\ProductBatch;
use App\Models\BusinessType;
use App\Models\UnitType;
use App\Models\AdjustmentNotes;
use App\Models\TransferNotes;
use App\Models\Gtn;
use App\Models\onlineOrderLog;  

use App\Services\productBatchService;

use Illuminate\Contracts\Cache\Store;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Yajra\DataTables\Facades\DataTables;


class ProductsController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function search_products_instock(Request $request)
    {
        
        $searchTerm = $request->input('q');

        if (!$searchTerm) {
            return response()->json([]);
        }

        // Extract and sanitize request variables
        $store_id = $request->filled('store_id') ? $request->store_id : '';

        if(empty($store_id)){
            $store_id = Stores::where('business_id', session('business_id'))->value('id_business_stores');
        }

        $business_id = $request->filled('business_id') ? $request->business_id : '';
        $dispatch = $request->filled('dispatch') ? $request->dispatch : '';
        $showprofessional = $request->filled('showprofessional') ? $request->showprofessional : '';
        $phrase = $request->q ? $request->q : 'nothing';

        if(!$phrase) {
            return response()->json(['error' => 'Search term is required'], 400);
        }

        // Optional session-based logic
        $common_products = session('common_products', 'No');
        $business_id = session('business_id', null);

        // Build the query
        $query = DB::table('business_products')
            ->selectRaw("
                id_business_products, business_products.category, business_products.product, business_brands.business_brand_name, measure_unit, qty_per_unit, product_batch.id_batch as id,
                IFNULL(business_products.category, '') as mcategory,
                product_batch.expiry_date,
                business_products.price,
                IFNULL(adj.addition, 0) as total,
                IFNULL(f.qty_purchased, 0) as purchased,
                IFNULL(g.transfer_in, 0) as transferin,
                IFNULL(g.transfer_out, 0) as transferout,
                IFNULL(a.sold, 0) + IFNULL(h.franchise_sale, 0) as sold,
                IFNULL(b.used, 0) as used,
                IFNULL(c.returned, 0) as returned,

                ROUND((IFNULL(adj.addition,0) + IFNULL(f.qty_purchased,0) + IFNULL(g.transfer_in,0))
                - (IFNULL(g.transfer_out,0) + IFNULL(a.sold,0) + IFNULL(b.used,0) + IFNULL(c.returned,0) + IFNULL(h.franchise_sale,0)),3) as instock,

                business_store,
                batch_number as batch,
                product_batch.id_batch as batch_id,
                IFNULL(unit_type,'') as unit_type,
                IFNULL(measure_unit,'') as measure_unit,
                IFNULL(qty_per_unit,'') as qty_per_unit,

                CONCAT(
                    id_business_products, ', ',
                    ((IFNULL(adj.addition,0) + IFNULL(f.qty_purchased,0) + IFNULL(g.transfer_in,0))
                    - (IFNULL(g.transfer_out,0) + IFNULL(a.sold,0) + IFNULL(b.used,0) + IFNULL(c.returned,0))),
                    ', ', product, ', ', IFNULL(category,''), ', ', IFNULL(batch_number,''), ', ',
                    IFNULL(id_batch,0), ', ', IFNULL(unit_type,''), ', ', IFNULL(measure_unit,''), ', ',
                    IFNULL(qty_per_unit,'')
                ) as id
            ")
            ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
            ->join('product_batch', 'product_batch.product_id', '=', 'business_products.id_business_products')
            ->join('business_stores', 'business_stores.id_business_stores', '=', 'product_batch.store_id')
            ->join('business', 'business.id_business', '=', 'business_products.business_id')
            ->leftJoin(DB::raw("(
                SELECT batch_id, product_id, SUM(IFNULL(adjustment_qty,0)) as addition
                FROM adjustment_notes
                GROUP BY batch_id, product_id
            ) as adj"), function($join) {
                $join->on('adj.product_id', '=', 'business_products.id_business_products')
                    ->on('adj.batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT batch_id, product_batch.product_id, SUM(invoice_qty) as sold
                FROM invoice_products
                JOIN invoice ON invoice.id_invoice = invoice_products.invoice_id
                JOIN product_batch ON product_batch.id_batch = invoice_products.batch_id
                WHERE invoice_status = 'valid'
                AND reference_invoice_number = ''
                AND IFNULL(batch_id,0) != 0
                GROUP BY batch_id, product_id
            ) as a"), function($join) {
                $join->on('a.product_id', '=', 'business_products.id_business_products')
                    ->on('a.batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT product_id, batch, batch_id, SUM(IFNULL(dispatch_qty,0)) as used
                FROM dispatch_notes
                WHERE status = 'Active'
                AND IFNULL(batch_id,0) != 0
                GROUP BY product_id, batch, batch_id
            ) as b"), function($join) {
                $join->on('b.product_id', '=', 'business_products.id_business_products')
                    ->on('b.batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT product_id, batch_id, SUM(IFNULL(return_qty,0)) as returned
                FROM return_notes
                WHERE return_status = 'Active'
                AND IFNULL(batch_id,0) != 0
                GROUP BY product_id, batch_id
            ) as c"), function($join) {
                $join->on('c.product_id', '=', 'business_products.id_business_products')
                    ->on('c.batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT grn_product_id, grn_batch_id, SUM(IFNULL(grn_qty_received,0)) as qty_purchased
                FROM grn_details
                JOIN goods_received_note ON goods_received_note.grn_id = grn_details.grn_id
                WHERE grn_batch_id IS NOT NULL
                GROUP BY grn_product_id, grn_batch_id
            ) as f"), function($join) {
                $join->on('f.grn_product_id', '=', 'business_products.id_business_products')
                    ->on('f.grn_batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT product_id, batch_id,
                    SUM(IFNULL(tranfer_out_qty,0)) as transfer_out,
                    SUM(IFNULL(tranfer_in_qty,0)) as transfer_in
                FROM transfer_notes
                WHERE batch_id IS NOT NULL
                AND transfer_notes.status='Active'
                GROUP BY product_id, batch_id
            ) as g"), function($join) {
                $join->on('g.product_id', '=', 'business_products.id_business_products')
                    ->on('g.batch_id', '=', 'product_batch.id_batch');
            })
            ->leftJoin(DB::raw("(
                SELECT product_id, batch_id, SUM(IFNULL(qty,0)) as franchise_sale
                FROM franchise_order_products
                JOIN franchise_orders ON franchise_orders.id_franchise_orders = franchise_order_id
                WHERE batch_id IS NOT NULL
                AND (franchise_orders.order_status='Paid' OR franchise_orders.order_status='Invoiced')
                GROUP BY product_id, batch_id
            ) as h"), function($join) {
                $join->on('h.product_id', '=', 'business_products.id_business_products')
                    ->on('h.batch_id', '=', 'product_batch.id_batch');
            })
            ->where('business_product_active', 'Yes');

        if (!empty($store_id)) {
            $query->where('business_stores.id_business_stores', $store_id);
        }

       // if ($dispatch === "") {
            $query->whereRaw("
                (IFNULL(adj.addition,0) + IFNULL(f.qty_purchased,0) + IFNULL(g.transfer_in,0))
                - (IFNULL(g.transfer_out,0) + IFNULL(a.sold,0) + IFNULL(b.used,0) + IFNULL(c.returned,0) + IFNULL(h.franchise_sale,0)) > 0
            ");
       // }
        if(empty($showprofessional) || $showprofessional === '') {
            //check business config
            $business = Business::find($business_id);
            if ($business && $business->professional === 'y') {
                $query->where('professional', 'y');
            } else if ($business && $business->professional === 'n') {
                $query->where('professional', 'n');
            }
        } else if ($showprofessional === 'Yes') {
            $query->where('professional', 'y');

        } else if ($showprofessional === 'No') {
            $query->where('professional', 'n');
        }

        if (Schema::hasColumn('product_batch', 'batch_hidden')) {
            $query->where('product_batch.batch_hidden', 'No');
        }

        $query->where('business_products.track_inventory', 'Yes');

        // if (session('common_products') === 'No') {
        //     $query->where('business_products.business_id', session('business_id'));
        // } else {
        //     if (empty($business_id)) {
        //         $query->where('business.common_products', "Yes");
        //     } else {
        //         $query->where('business_products.business_id', $business_id);
        //     }
        // }

        if (!empty($phrase)) {
            $phrase = strtoupper($phrase);
            $query->whereRaw("
                (
                    UPPER(CONCAT(business_brand_name, ' ', product, ' ', IFNULL(business_products.category, ''))) LIKE ?
                    OR UPPER(CONCAT(product, ' ', business_brand_name, ' ', IFNULL(business_products.category, ''))) LIKE ?
                    OR UPPER(CONCAT(product, ' ', IFNULL(business_products.category, ''), ' ', business_brand_name)) LIKE ?
                    OR UPPER(CONCAT(IFNULL(business_products.category, ''), ' ', product, ' ', business_brand_name)) LIKE ?
                    OR UPPER(CONCAT(business_brand_name, ' ', IFNULL(business_products.category, ''), ' ', product)) LIKE ?
                    OR IFNULL(business_products.barcode_products, '') LIKE ?
                    OR IFNULL(business_products.sku, '') LIKE ?
                    OR UPPER(id_business_products) LIKE ?
                )
            ", array_fill(0, 8, "%$phrase%"));
        }

        $query->orderBy('business_products.brand_id')
            ->orderBy('business_products.product')
            ->orderBy('business_products.category')
            ->orderBy('expiry_date');

        $results = $query->get();
        $products = $results->map(function($item) {
            return [
                'id' => $item->id_business_products,
                'product' => $item->product,
                'category' => $item->category,
                'business_brand_name' => $item->business_brand_name,
                'expiry_date' => $item->expiry_date,
                'batch_no' => $item->batch,
                'batch_id' => $item->batch_id,
                'store' => $item->business_store,
                'instock' => $item->instock,
                'price' => $item->price,
                'measure_unit' => $item->measure_unit,
                'qty_per_unit' => $item->qty_per_unit,
            ];
        });

        return response()->json($products);
    }

    public function search_products(Request $request)
    {
      // return response()->json([$request->input('q')]);
        $searchTerm = $request->input('q');
        if (!$searchTerm) {
            return response()->json([]);
        }

        $business = Business::find(session('business_id'));
        try{
            $products = Products::select('id_business_products as id', 'business_products.category as mcategory', 
                'business_products.*', 'business_brands.business_brand_name')
                ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
                ->join('business', 'business.id_business', '=', 'business_products.business_id');            
            
            $products= $products->where('business_product_active', 'Yes');
            
            if (!$business->common_products || $business->common_products === 'No') {
                $products = $products->where('business_products.business_id', session('business_id'));
            } else {
                $products = $products->where('business.common_products', 'Yes');
            }

            if ($request->has('brand_id') && !empty($request->input('brand_id')) && $request->input('brand_id') != 0) {
                $products = $products->where('business_products.brand_id', $request->input('brand_id'));
            }

            $products = $products->where(function($query) use ($searchTerm) {
                    $likeTerm = '%' . str_replace(' ', '%', $searchTerm) . '%';
                    $query->where('business_products.product', 'LIKE', $likeTerm)
                        ->orWhere('business_brands.business_brand_name', 'LIKE', $likeTerm)
                        ->orWhere('business_products.sku', 'LIKE', $likeTerm)
                        ->orWhere('business_products.barcode_products', 'LIKE', $likeTerm);
                    return $query;
                });
            $products = $products->orderBy('business_brands.business_brand_name')
                                ->orderBy('business_products.product')
                                ->get();
            
            return response()->json($products);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while searching for products.'.$e->getMessage()], 500);
        }
    }

    public function product_by_brand(Request $request, $id_brand)
    {
        if(session('common_products')=="Yes"){
            if($request->has('business_id')){
                $business_id = $request->input('business_id');
            } else {
                $businesses = Business::select('id_business')->where('common_products', 'Yes')->get();
                $business_ids = $businesses->pluck('id_business')->toArray();
                $business_id = '';
            }
        } else {
            $business_id = session('business_id');
        }

         $products = Products::select('id_business_products as id', 'business_products.category as mcategory', 
            'business_products.*', 'business_brands.business_brand_name')
            ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
            ->where('business_products.brand_id', $id_brand);
            if($business_id == ''){
                $products = $products->whereIn('business_products.business_id', $business_ids);
            }else{
                $products = $products->where('business_products.business_id', $business_id);
            }
            $products = $products->where('business_product_active', 'Yes')
            ->where('business_brands.business_brand_active', 'Yes')            
            ->orderBy('business_products.product')
            ->get();

        return response()->json($products);
    }

    public function create_product_order(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|integer|exists:customers,id_customers',
            'products' => 'required|array|min:1',
            'products.*.product_id' => 'required|integer|exists:business_products,id_business_products',
            'products.*.batch_id' => 'nullable|integer|exists:product_batch,id_batch',
            'products.*.quantity' => 'required|integer|min:1',
            'products.*.price' => 'required|numeric|min:0',
        ]);

        $business_id = session('business_id');
        $user_name = session('user_name');

        DB::beginTransaction();

        try {
            // Create a new order
            $order = new CustomerOrders();
            $order->business_id = $business_id;
            $order->customer_id = $request->customer_id;
            $order->created_by = $user_name;           
            $order->order_type = $request->input('order_type', 'Take Away');
            $order->customer_order_date = now();
            $order->order_status = 'open';
            $order->order_extra = $request->input('order_extra', '');
            $order->save();
            //get the new order id
            $order_id = $order->id_customer_order;

            // Add products to the order
            foreach ($request->products as $item) {

                //fetch product details
                $product = Products::find($item['product_id']);

                $orderItem = new OrderProducts();
                $orderItem->business_id = $business_id;
                $orderItem->customer_order_id = $order_id;
                $orderItem->business_brand_id = $product['brand_id'];
                $orderItem->product_id = $item['product_id'];
                $orderItem->product_name = $item['product_name'];
                $orderItem->qty = $item['quantity'];
                $orderItem->product_unit_price = $item['price'];
                $orderItem->product_discount_amount = $item['discount_amount'] ?? 0;
                $orderItem->product_sale_tax_percentage = (float)$product->product_sales_tax ?? 0;
                $orderItem->product_sale_tax_amount = ((float)$product->product_sales_tax/100) * ((float)$item['price'] * (float)$item['quantity']) ?? 0;
                $orderItem->staff_id = $item['staff_id'] ?? null;
                $orderItem->product_final_price = (float)$item['price'] * (float)$item['quantity'] ?? 0;
                $orderItem->staff_name = $item['staff_name'] ?? '';
                $orderItem->category = $product['category'] ?? '';
                $orderItem->stockfrom = $item['stockfrom'] ?? '';
                $orderItem->batch = $item['batch'] ?? '';
                $orderItem->batch_id = $item['batch_id'] ?? null;
                $order->created_by = session('user_name');
               
                $orderItem->save();
            }

            DB::commit();

            return response()->json([
                'message_type' => 'success',
                'message' => 'Product order created successfully.',
                'message_btn' => 'success',
                'order_id' => $order_id,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message_type' => 'error',
                'message' => 'Failed to create product order: ' . $e->getMessage(),
            ], 500);
        }
    }   

    public function product_orders()
    {
        $business_id = session('business_id');
        $business = Business::find($business_id);
        $common_products = $business ? $business->common_products : 'No';

        return view('products.product_orders', compact('common_products'));
    }

    public function productordersdata(Request $request)
    {
        $business_id = session('business_id');
        $start_date  = $request->input('start_date', null);

        $columns = [
            0 => 'customer_orders.id_customer_order',
            1 => 'customers.customer_name',
            2 => 'customer_orders.order_status',
            3 => 'customer_orders.customer_order_date',
            4 => 'items',
            5 => 'prices',
            6 => 'total_amount',
            7 => 'customer_orders.created_by',
            8 => 'customer_orders.order_type',
        ];

        $limit  = $request->input('length');
        $start  = $request->input('start');
        $order  = $columns[$request->input('order.0.column')];
        $dir    = $request->input('order.0.dir');
        $search = $request->input('search.value');

        // ----------------------------
        // Base Query (shared by both)
        // ----------------------------
        $base_query = CustomerOrders::select(
            'customer_orders.id_customer_order',
            'customers.customer_name',
            'customer_orders.order_status',
            'customer_orders.created_by',
            'customer_orders.order_type',
            DB::raw('DATE_FORMAT(customer_orders.customer_order_date, "%Y-%m-%d") AS customer_order_date'),
            DB::raw('GROUP_CONCAT(CONCAT(order_products.product_name, " x ", order_products.qty) SEPARATOR "<br>") AS items'),
            DB::raw('GROUP_CONCAT(order_products.product_unit_price SEPARATOR "<br>") AS prices'),
            DB::raw('SUM(order_products.product_final_price) AS total_amount')
        )
        ->join('customers', 'customers.id_customers', '=', 'customer_orders.customer_id')
        ->join('order_products', 'order_products.customer_order_id', '=', 'customer_orders.id_customer_order')
        ->where('customer_orders.business_id', $business_id);

        // Filter by start_date
        if (!empty($start_date)) {
            $base_query->where('customer_orders.customer_order_date', '>', $start_date);
        }

        // Search filter
        if (!empty($search)) {
            $base_query->where(function ($query) use ($search) {
                $query->where('customer_orders.id_customer_order', 'LIKE', "%{$search}%")
                    ->orWhere('customer_orders.order_status', 'LIKE', "%{$search}%")
                    ->orWhere('customer_orders.created_by', 'LIKE', "%{$search}%")
                    ->orWhere('customer_orders.order_type', 'LIKE', "%{$search}%")
                    ->orWhere('customers.customer_name', 'LIKE', "%{$search}%");
            });
        }

        // ----------------------------
        // Clone for total (before limit/offset)
        // ----------------------------
        $count_query = clone $base_query;
        $totalRecords = $count_query->groupBy('customer_orders.id_customer_order')->get()->count();

        // ----------------------------
        // Main paginated query
        // ----------------------------
        $orders = $base_query
            ->groupBy('customer_orders.id_customer_order')
            ->orderBy($order, $dir)
            ->offset($start)
            ->limit($limit)
            ->get();

        // ----------------------------
        // Prepare DataTable output
        // ----------------------------
        $data = [];
        foreach ($orders as $order) {
            $nestedData['id_customer_order']   = $order->id_customer_order;
            $nestedData['customer_name']       = $order->customer_name ?? 'N/A';
            $nestedData['order_status']        = $order->order_status;
            $nestedData['customer_order_date'] = $order->customer_order_date;
            $nestedData['items']               = $order->items ?? '-';
            $nestedData['prices']              = $order->prices ?? '-';
            $nestedData['total_amount']        = $order->total_amount ?? 0;
            $nestedData['created_by']          = $order->created_by;
            $nestedData['order_type']          = $order->order_type;
            $data[] = $nestedData;
        }

        // ----------------------------
        // Final JSON response
        // ----------------------------
        return response()->json([
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => intval($totalRecords),
            "recordsFiltered" => intval($totalRecords),
            "data"            => $data,
        ]);
    }

    public function product_orders_online(Request $request)
    {
        $business_id = session('business_id');
        $business = Business::find($business_id);
        
        $status = $request->input('status', 'sale_order_not_created');

        $onlineOrderLogs = onlineOrderLog::where([
            'business_id' => $business_id,
            'status' => $status,
        ])->get();

        $business_products = DB::select("SELECT id_business_products, product, sku FROM `business_products` WHERE business_product_active = 'Yes' AND business_id = ? ;", [$business_id]);
        
        return view('products.product_orders_online', compact('onlineOrderLogs', 'business_products'));
    }

    public function productOrdersOnlineStore(Request $request)
    {
        $business_id = session('business_id');
        $user_name = session('user_name');

        $online_order_log_id = $request->input('online_order_log_id');

        $is_completed = DB::table('online_order_logs')->where([
            'id_online_order_logs' => $online_order_log_id,
            'status' => 'sale_order_created'
        ])->exists();

        if($is_completed){
            return response()->json([
                'success' => false,
                'message' => "Already Process"
            ]);
        }

        $online_order_customer_name = $request->input('online_order_customer_name');
        $online_order_customer_email = $request->input('online_order_customer_email');
        $online_order_customer_phone = $request->input('online_order_customer_phone');
        $online_order_delivery_charges = $request->input('online_order_delivery_charges', '0.00');

        $system_products_gainst_shopify = $request->input('system_product_gainst_shopify', []);
        $order_qtys = $request->input('order_qty', []);
        $item_prices = $request->input('item_price', []);
        $item_discounts = $request->input('item_discount', []);
        $item_tax_rates = $request->input('item_tax_rate', []);
        $item_taxes = $request->input('item_tax', []);

        
        foreach ($system_products_gainst_shopify as $index => $productId) {
            if (!$productId) {
                return response()->json([
                    'success' => false,
                    'message' => "System Product is required for row " . ($index + 1)
                ]);
            }

            $qty = $order_qtys[$index] ?? 0;
            if (!$qty || $qty <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Quantity must be greater than 0 for row " . ($index + 1)
                ]);
            }
        }


        DB::beginTransaction();
        
        try {
            $customer = $this->getOrGetCreateCustomer($online_order_customer_phone, $online_order_customer_email, $online_order_customer_name);
            
            // Extract shipping and billing addresses from order_payload
            $shipping_address = null;
            $billing_address = null;
            
            $onlineOrderLog = onlineOrderLog::find($online_order_log_id);
            if ($onlineOrderLog && $onlineOrderLog->order_payload) {
                $orderPayload = json_decode($onlineOrderLog->order_payload, true);
                
                // Extract shipping address
                if (isset($orderPayload['shipping_address']) && !empty($orderPayload['shipping_address'])) {
                    $shippingAddr = $orderPayload['shipping_address'];
                    $addressParts = array_filter([
                        $shippingAddr['address1'] ?? '',
                        $shippingAddr['address2'] ?? '',
                        $shippingAddr['city'] ?? '',
                        $shippingAddr['province'] ?? '',
                        $shippingAddr['zip'] ?? '',
                        $shippingAddr['country'] ?? ''
                    ]);
                    $shipping_address = !empty($addressParts) ? implode(', ', $addressParts) : null;
                }
                
                // Extract billing address
                if (isset($orderPayload['billing_address']) && !empty($orderPayload['billing_address'])) {
                    $billingAddr = $orderPayload['billing_address'];
                    $addressParts = array_filter([
                        $billingAddr['address1'] ?? '',
                        $billingAddr['address2'] ?? '',
                        $billingAddr['city'] ?? '',
                        $billingAddr['province'] ?? '',
                        $billingAddr['zip'] ?? '',
                        $billingAddr['country'] ?? ''
                    ]);
                    $billing_address = !empty($addressParts) ? implode(', ', $addressParts) : null;
                }
            }
            
            // Create a new order
            $order = new CustomerOrders();
            $order->business_id = $business_id;
            $order->customer_id = $customer->id_customers;
            $order->online_order_log_id = $online_order_log_id;
            $order->created_by = $user_name;           
            $order->order_type = 'Take Away';
            $order->order_status = 'open';
            $order->order_extra = '';
            $order->delivery_charges = $online_order_delivery_charges;
            $order->customer_order_date = now();
            $order->shipping_address = $shipping_address;
            $order->billing_address = $billing_address;
            $order->save();
            //get the new order id
            $order_id = $order->id_customer_order;

            for ($i=0; $i < count($system_products_gainst_shopify); $i++) { 

                //fetch product details
                $product = Products::find($system_products_gainst_shopify[$i]);

                $orderItem = new OrderProducts();
                $orderItem->business_id = $business_id;
                $orderItem->customer_order_id = $order_id;
                $orderItem->business_brand_id = $product->brand_id;
                $orderItem->product_id = $product->id_business_products;
                $orderItem->product_name = $product->product;
                $orderItem->qty = $order_qtys[$i];
                $orderItem->product_unit_price = $item_prices[$i];
                $orderItem->product_discount_amount = $item_discounts[$i] ?? 0;
                $orderItem->product_sale_tax_percentage = (float) $item_tax_rates[$i] ?? 0;
                $orderItem->product_sale_tax_amount = $item_taxes[$i] ?? 0;
                $orderItem->staff_id = null;
                $orderItem->product_final_price = ((float) $item_prices[$i] * (float) $order_qtys[$i] ?? 0) - ($item_discounts[$i] ?? 0);
                $orderItem->staff_name = '';
                $orderItem->category = '';
                $orderItem->stockfrom = '';
                $orderItem->batch = '';
                $orderItem->batch_id = '0';
                $order->created_by = $user_name;
               
                $orderItem->save();
            }

            DB::table('online_order_logs')->where('id_online_order_logs', $online_order_log_id)
            ->update([
                'status' => 'sale_order_created'
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product order created successfully.',
                'order_id' => $order_id,
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create product order: ' . $e->getMessage(),
            ], 500);
        }
    }

    ///** Brands */
    public function brands(Request $request)
    { 
        $business_id = session('business_id');
        if($request->has('business_id')){
            $business_id = $request->business_id;
        }

        if(session('common_products')=="Yes"){
            $brands = Brands::select('business_brands.*')
            ->join('business', 'business.id_business', '=', 'business_brands.business_id')
            ->where('business.common_products', 'Yes')
            ->get();
            
        } else {
            
            $brands = Brands::select('*')->where('business_id', $business_id)->get();
            
        }

        if(session('ho')=="Yes"){
            $branches = Business::where('common_products', 'Yes')->get();
        } else {
            $branches = Business::where('id_business', $business_id)->get();
        }

        $business= Business::find($business_id);

        return view('products.brands', compact('brands', 'business', 'business_id', 'branches'));
    }

    public function save_brand(Request $request){
        try{
            DB::beginTransaction();
            if($request->id_business_brands != '' && $request->id_business_brands != null && $request->id_business_brands > 0){
                //update
                $brand = Brands::find($request->id_business_brands);
                $brand->business_brand_name = $request->business_brand_name;
                $brand->business_brand_short = $request->business_brand_short;
                $brand->business_brand_website = $request->business_brand_website;
                $brand->business_brand_active = $request->business_brand_active;
                $brand->save();

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success',
                    'message' => 'Brand updated successfully.',
                ]);

            } else {
                //create new
                $brand = new Brands();
                $brand->business_id = session('business_id');
                $brand->business_brand_name = $request->business_brand_name;
                $brand->business_brand_short = $request->business_brand_short;
                $brand->business_brand_website = $request->business_brand_website;
                $brand->business_brand_active = $request->business_brand_active;
                $brand->save();

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success',
                    'message' => 'Brand created successfully.',
                ]);

            }

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
                'message' => 'Failed to save brand: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function brand_products($id_brand, Request $request)
    {

        $brand_name = "";
        if($id_brand > 0){
            $brand_name = Brands::where('id_business_brands', $id_brand)->value('business_brand_name');
        } 

        $business_id = session('business_id');

        if(!empty($request->business_id) && $request->business_id != "" && $request->business_id != null){
            $business_id = $request->business_id;
        } 

        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = Business::where('common_products', 'Yes')->get();            
        } else {
            $branches = Business::select('*')->where('id_business', $business_id)->get();
        }
        //return $branches;

        //get all business brands
        if(session('common_products')=="Yes"){
            $brands = Brands::select('business_brands.*')
            ->join('business', 'business.id_business', '=', 'business_brands.business_id')
            ->where('business.common_products', 'Yes')
            ->orderBy('business_brands.business_brand_name')
            ->get();
        } else {
            $brands = Brands::select('*')->where('business_id', $business_id)->orderBy('business_brand_name')->get();
        }
        //get all unit types
        $unit_types = UnitType::all();

        //get measure units
        $measure_units = DB::table('measurement_units')->get();
        
        $store_id = 0; //All stores
        if($request->has('store_id')) {
            $store_id = $request->store_id;
        }

        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $stores = Stores::select('business_stores.*', 'business.business_name')
            ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->where('business.common_products', 'Yes')
            ->orderBy('business.business_name')
            ->get();
        } else {
            $stores = Stores::select('business_stores.*', 'business.business_name')
            ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->where('business.id_business', $business_id)
            ->get();
        }
        

        return view('products.brand_products', compact('id_brand', 'brand_name', 'branches', 'business_id', 'stores', 'store_id', 'brands', 'unit_types', 'measure_units'));
    }

    public function brand_products_data(Request $request, $id_brand)
    {
        if($id_brand == 0){
            $id_brand = '';
        }
        //return $this->get_brand_products_data($request, $id_brand);
        $query = DB::table('business_products')
            ->select([  
                DB::RAW('GROUP_CONCAT(product_image.image_name SEPARATOR "|") as images'),
                'business_products.id_business_products',
                'business_products.product',
                'business_products.category',
                'business_brands.business_brand_name',    
                'business_products.professional',            
                'business_products.price as price',
                'business_products.commission',
                'business_products.sku',
                'business_products.barcode_products as barcode',
                'business_products.updated_at',
                'business_products.purchase_price',
                'business_products.product_threshold as threshold',
                'business_products.business_product_active',
                DB::raw("IFNULL(business_products.category, '') as mcategory"),
                DB::raw("IFNULL(adj.addition,0) as total"),
                DB::raw("IFNULL(f.qty_purchased,0) as purchased"),
                DB::raw("IFNULL(g.transfer_in,0) as transferin"),
                DB::raw("IFNULL(g.transfer_out,0) as transferout"),
                DB::raw("IFNULL(a.sold,0)+IFNULL(h.franchise_sale,0) as sold"),
                DB::raw("IFNULL(b.used,0) as used"),
                DB::raw("IFNULL(c.returned,0) as returned"),
                DB::raw("ROUND(SUM((IFNULL(adj.addition,0)+IFNULL(f.qty_purchased,0)+IFNULL(g.transfer_in,0)) - 
                        (IFNULL(g.transfer_out,0)+IFNULL(a.sold,0)+IFNULL(b.used,0)+IFNULL(c.returned,0)+IFNULL(h.franchise_sale,0))),2) as instock"),               
                DB::raw("IFNULL(unit_type,'') as unit_type"),
                DB::raw("IFNULL(measure_unit,'') as measure_unit"),
                DB::raw("IFNULL(qty_per_unit,'') as qty_per_unit")
            ])
            ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
            ->leftJoin('product_image', 'product_image.product_id', '=', 'business_products.id_business_products')
            
            ->leftJoin('product_batch', 'product_batch.product_id', '=', 'business_products.id_business_products')
            ->leftJoin('business_stores', 'business_stores.id_business_stores', '=', 'product_batch.store_id');

        // Subquery: adjustment_notes
        $adj = DB::table('adjustment_notes')
            ->select('batch_id', 'product_id', DB::raw('SUM(IFNULL(adjustment_qty,0)) as addition'))
            ->groupBy('batch_id', 'product_id');

        $query->leftJoinSub($adj, 'adj', function ($join) {
            $join->on('adj.product_id', '=', 'business_products.id_business_products')
                ->on('adj.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: invoice_products
        $a = DB::table('invoice_products')
            ->join('invoice', 'invoice.id_invoice', '=', 'invoice_products.invoice_id')
            ->join('product_batch', 'product_batch.id_batch', '=', 'invoice_products.batch_id')
            ->select('batch_id', 'product_batch.product_id', DB::raw('SUM(invoice_qty) as sold'))
            ->where('invoice_status', 'valid')
            ->where('reference_invoice_number', '')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('batch_id', 'product_id');

        $query->leftJoinSub($a, 'a', function ($join) {
            $join->on('a.product_id', '=', 'business_products.id_business_products')
                ->on('a.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: dispatch_notes
        $b = DB::table('dispatch_notes')
            ->select('product_id', 'batch', 'batch_id', DB::raw('SUM(IFNULL(dispatch_qty,0)) as used'))
            ->where('status', 'Active')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('product_id', 'batch', 'batch_id');

        $query->leftJoinSub($b, 'b', function ($join) {
            $join->on('b.product_id', '=', 'business_products.id_business_products')
                ->on('b.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: return_notes
        $c = DB::table('return_notes')
            ->select('product_id', 'batch_id', DB::raw('SUM(IFNULL(return_qty,0)) as returned'))
            ->where('return_status', 'Active')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($c, 'c', function ($join) {
            $join->on('c.product_id', '=', 'business_products.id_business_products')
                ->on('c.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: grn_details
        $f = DB::table('grn_details')
            ->join('goods_received_note', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
            ->select('grn_product_id', 'grn_batch_id', DB::raw('SUM(IFNULL(grn_qty_received,0)) as qty_purchased'))
            ->whereNotNull('grn_batch_id')
            ->groupBy('grn_product_id', 'grn_batch_id');

        $query->leftJoinSub($f, 'f', function ($join) {
            $join->on('f.grn_product_id', '=', 'business_products.id_business_products')
                ->on('f.grn_batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: transfer_notes
        $g = DB::table('transfer_notes')
            ->select('product_id', 'batch_id',
                DB::raw('SUM(IFNULL(tranfer_out_qty,0)) as transfer_out'),
                DB::raw('SUM(IFNULL(tranfer_in_qty,0)) as transfer_in')
            )
            ->whereNotNull('batch_id')
            ->where('transfer_notes.status', 'Active')
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($g, 'g', function ($join) {
            $join->on('g.product_id', '=', 'business_products.id_business_products')
                ->on('g.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: franchise_order_products
        $h = DB::table('franchise_order_products')
            ->join('franchise_orders', 'franchise_orders.id_franchise_orders', '=', 'franchise_order_id')
            ->select('product_id', 'batch_id', DB::raw('SUM(IFNULL(qty,0)) as franchise_sale'))
            ->whereNotNull('batch_id')
            ->where(function ($w) {
                $w->where('franchise_orders.order_status', 'Paid')
                ->orWhere('franchise_orders.order_status', 'Invoiced');
            })
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($h, 'h', function ($join) {
            $join->on('h.product_id', '=', 'business_products.id_business_products')
                ->on('h.batch_id', '=', 'product_batch.id_batch');
        });
        if($id_brand != ""){
            $query->where('business_products.brand_id', $id_brand);
        }
        //$query->where('business_product_active', 'Yes'); showing all products including inactive ones

        if ($request->available == "Yes") {
             $query->whereRaw("(IFNULL(adj.addition,0)+IFNULL(f.qty_purchased,0)+IFNULL(g.transfer_in,0))-
                             (IFNULL(g.transfer_out,0)+IFNULL(a.sold,0)+IFNULL(b.used,0)+IFNULL(c.returned,0)+IFNULL(h.franchise_sale,0)) > 0");
        }

        if(session('common_products') === 'No') {
            $query->where('business_products.business_id', session('business_id'));
        } else {
            if($request->business_id != "" && $request->business_id != null){
                $query->where('business_products.business_id', $request->business_id);
            } else {
                //get business ids with common products
                $business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
                $query->whereIn('business_products.business_id', $business_ids);
            }
        }

        // Optional store and branch filters        
        // if ($request->has('store_id') && $request->store_id != "" && $request->store_id != null && $request->store_id != 0) {
        //     $query->where('business_stores.id_business_stores', $request->store_id);
        //     if(session('common_products') === 'No') {
        //         $query->where('business_products.business_id', session('business_id'));
        //     }
        // } else {
        //     if($request->business_id != "" && $request->business_id != null){
        //         $query->where('business_stores.business_id', $request->business_id);
        //     } else if(session('common_products') === 'No') {
        //         $query->where('business_products.business_id', session('business_id'));
        //     } else {
        //         //get business ids with common products
        //         $business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
        //         $query->whereIn('business_products.business_id', $business_ids);
        //     }
        // }

        // Filter by filter_type
        if ($request->filter_type == 'Retail') {
            $query->where('professional', 'n');
        } elseif ($request->filter_type == 'Professional') {
            $query->where('professional', 'y');
        }

        // if (Schema::hasColumn('product_batch', 'batch_hidden')) {
        //     $query->where('product_batch.batch_hidden', 'No');
        // }

        $query->groupBy('business_products.id_business_products');

     
        return DataTables::of($query)
            ->editColumn('product_type', function ($item) {
                return $item->professional == "y" ? 'Professional' : 'Retail';
            })
            ->editColumn('purchase_price', function ($item) {
                return $item->purchase_price ?? '0.00';
            })
            ->editColumn('threshold', function ($item) {
                return $item->threshold ?? '';
            })
            ->addColumn('actions', function ($item) {
                return ''; // placeholder for buttons
            })
            ->filterColumn('product', function ($query, $keyword) {
                $query->where('business_products.product', 'like', "%{$keyword}%")
                    ->orWhere('business_products.category', 'like', "%{$keyword}%")
                    ->orWhere('business_products.sku', 'like', "%{$keyword}%")
                    ->orWhere('business_products.barcode_products', 'like', "%{$keyword}%");
            })
            ->filterColumn('business_brand_name', function ($query, $keyword) {
                $query->where('business_brands.business_brand_name', 'like', "%{$keyword}%");
            })
            ->filterColumn('id_business_products', function ($query, $keyword) {
                $query->where('business_products.id_business_products', 'like', "%{$keyword}%");
            })
            ->filterColumn('category', function ($query, $keyword) {
                $query->where('business_products.category', 'like', "%{$keyword}%");
            })
            ->filterColumn('sku', function ($query, $keyword) {
                $query->where('business_products.sku', 'like', "%{$keyword}%");
            })
            ->filterColumn('barcode', function ($query, $keyword) {
                $query->where('business_products.barcode_products', 'like', "%{$keyword}%");
            })
            ->filterColumn('price', function ($query, $keyword) {
                $query->where('business_products.price', 'like', "%{$keyword}%");
            })
            ->filterColumn('instock', function ($query, $keyword) {
                $query->havingRaw("ROUND(SUM((IFNULL(adj.addition,0)+IFNULL(f.qty_purchased,0)+IFNULL(g.transfer_in,0)) - 
                        (IFNULL(g.transfer_out,0)+IFNULL(a.sold,0)+IFNULL(b.used,0)+IFNULL(c.returned,0)+IFNULL(h.franchise_sale,0))),2) like ?", ["%{$keyword}%"]);
            })
            ->filterColumn('measure_unit', function ($query, $keyword) {
                $query->where('measure_unit', 'like', "%{$keyword}%");
            })
            ->filterColumn('qty_per_unit', function ($query, $keyword) {
                $query->where('qty_per_unit', 'like', "%{$keyword}%");
            })
            ->filterColumn('unit_type', function ($query, $keyword) {
                $query->where('unit_type', 'like', "%{$keyword}%");
            })
            ->rawColumns(['actions', 'images'])
            ->order(function ($query) use ($request) {
                if ($request->has('order')) {
                    $columns = [
                        'images',
                        'id_business_products',
                        'business_brand_name',                        
                        'product',
                        'category',
                        'professional',
                        'price',
                        'purchase_price',
                        'instock',
                        'threshold',
                        'unit_type',                        
                        'qty_per_unit',
                        'measure_unit',
                        'business_product_active',
                        'commission',
                        'sku',
                        'barcode',
                        'updated_at'                        
                        
                    ];
                    if ($request->has('order') && isset($request->order[0])) {
                        $orderColIndex = $request->order[0]['column'] ?? 0;
                        $orderDir = $request->order[0]['dir'] ?? 'asc';
                        $orderCol = $columns[$orderColIndex] ?? 'business_products.id_business_products';
                        $query->orderBy($orderCol, $orderDir);
                    } else {
                        $query->orderBy('business_products.id_business_products', 'ASC');
                    }
                }
            })
            ->make(true);
    }

    public function store_stock_status(Request $request)
    {
        $store_id = $request->input('store_id');
        $business_id = $request->input('business_id');

        $store = Stores::where('id_business_stores', $store_id)
            ->where('business_id', $business_id)
            ->first();

        if (!$store) {
            return response()->json(['error' => 'Store not found'], 404);
        }

        return response()->json(['stock_status' => $store->stock_status]);
    }

    public function store_stock_data(Request $request){
        
        $business_id = $request->input('business_id');

        //get business_types
        $business_types = BusinessType::all();
        $business_type_id = $request->input('business_type_id', '');

        //get branches
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = Business::where('common_products', 'Yes')->get();            
        } else {
            $branches = Business::where('id_business', session('business_id'))->get();
        }

        //get all stores of the business
        if(session('ho')=="Yes"){
            if($business_type_id != "" || $business_type_id != null){
                $business_ids = Business::where('business_type_id', $business_type_id)->pluck('id_business')->toArray();
                $stores = Stores::whereIn('business_id', $business_ids)->get();
            
            } else{
                if($business_id != "" || $business_id != null){
                    $stores = Stores::where('business_id', $business_id)->get();
                
                } else if(session('common_products')=="Yes"){
                    $stores = Stores::join('business', 'business.id_business', '=', 'business_stores.business_id')
                        ->where('business.common_products', 'Yes')
                        ->get();
                
                } else {
                    $stores = Stores::where('business_id', session('business_id'))->get();
                    
                }
            }
        } else {
            $stores = Stores::where('business_id', session('business_id'))->get();
        }

        //return $this->get_brand_products_data($request, $id_brand);
        $query = DB::table('business_products')
            ->select([  
                'business_products.id_business_products as ID',
                'business_products.product as Product',
                'business_products.category as Category',
                'business_brands.business_brand_name as Brand',    
                DB::RAW('CASE business_products.professional = "y" WHEN 1 THEN "Professional" ELSE "Retail" END as product_type'),
                DB::RAW('ROUND(business_products.price, 2) as Retail'),
                DB::RAW('ROUND(business_products.purchase_price, 2) as Purchase'),
                'business_products.sku as SKU',
                'business_products.barcode_products as Barcode',
                
            ]);
        //select store totals
        foreach ($stores as $store) {
            $query->addSelect(DB::raw("
                ROUND(
                    SUM(
                        CASE WHEN business_stores.id_business_stores = {$store->id_business_stores}
                            THEN (IFNULL(adj.addition,0)+IFNULL(f.qty_purchased,0)+IFNULL(g.transfer_in,0))
                                - (IFNULL(g.transfer_out,0)+IFNULL(a.sold,0)+IFNULL(b.used,0)
                                    +IFNULL(c.returned,0)+IFNULL(h.franchise_sale,0))
                            ELSE 0
                        END
                    ), 2
                ) as `{$store->business_store}`
            "));
        }

        $query->Join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
            ->leftJoin('product_batch', 'product_batch.product_id', '=', 'business_products.id_business_products')
            ->leftJoin('business_stores', 'business_stores.id_business_stores', '=', 'product_batch.store_id');

        

        // Subquery: adjustment_notes
        $adj = DB::table('adjustment_notes')
            ->select('batch_id', 'product_id', DB::raw('SUM(IFNULL(adjustment_qty,0)) as addition'))
            ->groupBy('batch_id', 'product_id');

        $query->leftJoinSub($adj, 'adj', function ($join) {
            $join->on('adj.product_id', '=', 'business_products.id_business_products')
                ->on('adj.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: invoice_products
        $a = DB::table('invoice_products')
            ->join('invoice', 'invoice.id_invoice', '=', 'invoice_products.invoice_id')
            ->join('product_batch', 'product_batch.id_batch', '=', 'invoice_products.batch_id')
            ->select('batch_id', 'product_batch.product_id', DB::raw('SUM(invoice_qty) as sold'))
            ->where('invoice_status', 'valid')
            ->where('reference_invoice_number', '')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('batch_id', 'product_id');

        $query->leftJoinSub($a, 'a', function ($join) {
            $join->on('a.product_id', '=', 'business_products.id_business_products')
                ->on('a.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: dispatch_notes
        $b = DB::table('dispatch_notes')
            ->select('product_id', 'batch', 'batch_id', DB::raw('SUM(IFNULL(dispatch_qty,0)) as used'))
            ->where('status', 'Active')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('product_id', 'batch', 'batch_id');

        $query->leftJoinSub($b, 'b', function ($join) {
            $join->on('b.product_id', '=', 'business_products.id_business_products')
                ->on('b.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: return_notes
        $c = DB::table('return_notes')
            ->select('product_id', 'batch_id', DB::raw('SUM(IFNULL(return_qty,0)) as returned'))
            ->where('return_status', 'Active')
            ->whereRaw('IFNULL(batch_id,0) != 0')
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($c, 'c', function ($join) {
            $join->on('c.product_id', '=', 'business_products.id_business_products')
                ->on('c.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: grn_details
        $f = DB::table('grn_details')
            ->join('goods_received_note', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
            ->select('grn_product_id', 'grn_batch_id', DB::raw('SUM(IFNULL(grn_qty_received,0)) as qty_purchased'))
            ->whereNotNull('grn_batch_id')
            ->groupBy('grn_product_id', 'grn_batch_id');

        $query->leftJoinSub($f, 'f', function ($join) {
            $join->on('f.grn_product_id', '=', 'business_products.id_business_products')
                ->on('f.grn_batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: transfer_notes
        $g = DB::table('transfer_notes')
            ->select('product_id', 'batch_id',
                DB::raw('SUM(IFNULL(tranfer_out_qty,0)) as transfer_out'),
                DB::raw('SUM(IFNULL(tranfer_in_qty,0)) as transfer_in')
            )
            ->whereNotNull('batch_id')
            ->where('transfer_notes.status', 'Active')
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($g, 'g', function ($join) {
            $join->on('g.product_id', '=', 'business_products.id_business_products')
                ->on('g.batch_id', '=', 'product_batch.id_batch');
        });

        // Subquery: franchise_order_products
        $h = DB::table('franchise_order_products')
            ->join('franchise_orders', 'franchise_orders.id_franchise_orders', '=', 'franchise_order_id')
            ->select('product_id', 'batch_id', DB::raw('SUM(IFNULL(qty,0)) as franchise_sale'))
            ->whereNotNull('batch_id')
            ->where(function ($w) {
                $w->where('franchise_orders.order_status', 'Paid')
                ->orWhere('franchise_orders.order_status', 'Invoiced');
            })
            ->groupBy('product_id', 'batch_id');

        $query->leftJoinSub($h, 'h', function ($join) {
            $join->on('h.product_id', '=', 'business_products.id_business_products')
                ->on('h.batch_id', '=', 'product_batch.id_batch');
        });

        $query->where('business_product_active', 'Yes');

        if ($request->available == "Yes") {
             $query->whereRaw("(IFNULL(adj.addition,0)+IFNULL(f.qty_purchased,0)+IFNULL(g.transfer_in,0))-
                             (IFNULL(g.transfer_out,0)+IFNULL(a.sold,0)+IFNULL(b.used,0)+IFNULL(c.returned,0)+IFNULL(h.franchise_sale,0)) > 0");
        }

        // Optional store filters
        if($request->has('business_type_id') && $request->business_type_id != "" && $request->business_type_id != null){
            $businesses_of_type = Business::where('business_type_id', $request->business_type_id)->pluck('id_business')->toArray();
            $query->whereIn('business_stores.business_id', $businesses_of_type);
            if(session('common_products') === 'No') {
                $query->where('business_products.business_id', session('business_id'));
            }
        } else {
            if ($request->has('store_id') && $request->store_id != "" && $request->store_id != null && $request->store_id != 0) {
                $query->where('business_stores.id_business_stores', $request->store_id);
                if(session('common_products') === 'No') {
                    $query->where('business_products.business_id', session('business_id'));
                }
            } else {
                if($request->business_id != "" && $request->business_id != null){
                    $query->where('business_stores.business_id', $request->business_id);
                    if(session('common_products') === 'No') {
                        $query->where('business_products.business_id', session('business_id'));
                    }
                } else if(session('common_products') === 'No') {
                    $query->where('business_products.business_id', session('business_id'));
                } 
            }
        }

        // Filter by filter_type
        if ($request->filter_type == 'Retail') {
            $query->where('professional', 'n');
        } elseif ($request->filter_type == 'Professional') {
            $query->where('professional', 'y');
        }

        if (Schema::hasColumn('product_batch', 'batch_hidden')) {
            $query->where('product_batch.batch_hidden', 'No');
        }

        $query->groupBy('business_products.id_business_products', 'business_products.product', 'business_products.category', 
            'business_brands.business_brand_name',
            'business_products.professional', 'business_products.price', 'business_products.purchase_price',
            'business_products.sku', 'business_products.barcode_products');
        $data = $query->get();
       // return $data;

        //get brands
        if(session('common_products')=="Yes"){
            $brands = Brands::select('business_brands.id_business_brands', 'business_brands.business_brand_name')
            ->join('business', 'business.id_business', '=', 'business_brands.business_id')
            ->where('business.common_products', 'Yes')
            ->get();
            
        } else {
            $brands = Brands::select('*')->where('business_id', $business_id)->get();
        }

        //get Unit Types
        $unit_types = UnitType::select('*')->get();

        //get measurement units
        $measure_units = DB::table('measurement_units')->select('*')->get();

        return view('products.store_stock_data', compact('data', 'stores', 'branches', 'business_id', 'business_types', 'business_type_id', 'brands', 'unit_types', 'measure_units'));

    }

    public function product_details($product_id)
    {

        $product = Products::select('business_products.*', DB::RAW('GROUP_CONCAT(IFNULL(product_image.image_name, "") SEPARATOR ",") as image_name'))
        ->leftJoin('product_image', 'product_image.product_id', '=', 'business_products.id_business_products')
        ->where('id_business_products', $product_id)
            ->first();

        if (!$product) {
            return response()->json(['error' => 'Product not found']);
        }

        return response()->json($product);
    }

    public function save_product(Request $request){
        $id_business_products = $request->input('id_business_products');
        try{
            Db::beginTransaction();
            $product = Products::find($id_business_products);
            
            if (!$product) {
                //Insert new product
                $new_product = new Products();
                $new_product->business_id = session('business_id');
                $new_product->product = $request->input('productName');
                $new_product->category = $request->input('category', '');
                $new_product->brand_id = $request->input('brand_id', null);
                $new_product->professional = $request->input('professional', 'n');
                $new_product->price = $request->input('price', 0);
                $new_product->product_sales_tax = $request->input('product_sales_tax', 0);
                $new_product->commission_type = $request->input('commission_type', 'Fixed');
                $new_product->commission = $request->input('commission', 0);
                $new_product->sku = $request->input('sku', '');
                $new_product->barcode_products = $request->input('barcode_products', '');
                $new_product->purchase_price = $request->input('purchase_price', 0);
                $new_product->unit_type = $request->input('unit_type', '');
                $new_product->measure_unit = $request->input('measure_unit', '');
                $new_product->qty_per_unit = $request->input('qty_per_unit', 0);
                $new_product->business_product_active = $request->input('business_product_active', 'Yes');
                $new_product->track_inventory = $request->input('track_inventory', 'Yes');
                $new_product->product_type = $request->input('product_type', 'Physical');
                $new_product->product_threshold = $request->input('product_threshold', 0);
                
                
                $new_product->save();
                DB::commit();
                return response()->json([
                    'message' => 'Product created successfully',
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success'
                ]);

            } else {
                //Update existing product 
                $product->product = $request->input('productName', $product->product);
                $product->category = $request->input('category', $product->category);
                $product->brand_id = $request->input('brand_id', $product->brand_id);
                $product->professional = $request->input('professional', $product->professional);
                $product->price = $request->input('price', $product->price);
                $product->product_sales_tax = $request->input('product_sales_tax', $product->product_sales_tax);
                $product->commission_type = $request->input('commission_type', $product->commission_type);
                $product->commission = $request->input('commission', $product->commission);
                $product->sku = $request->input('sku', $product->sku);
                $product->barcode_products = $request->input('barcode_products', $product->barcode_products);
                $product->purchase_price = $request->input('purchase_price', $product->purchase_price);
                $product->unit_type = $request->input('unit_type', $product->unit_type);
                $product->measure_unit = $request->input('measure_unit', $product->measure_unit);
                $product->qty_per_unit = $request->input('qty_per_unit', $product->qty_per_unit);
                $product->business_product_active = $request->input('business_product_active', $product->business_product_active);

                $product->track_inventory = $request->input('track_inventory', $product->track_inventory);
                $product->product_type = $request->input('product_type', $product->product_type);
                $product->product_threshold = $request->input('product_threshold', $product->product_threshold);
                
                $product->save();

                DB::commit();
                return response()->json([
                    'message' => 'Product updated successfully',
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success'
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Failed to save product: ' . $e->getMessage()]);   
        }
    }

    public function product_batches($id_business_products, Request $request)
    {
        $group = false;
        
        $store_id = $request->input('store_id', 0);

        $product = Products::where('id_business_products', $id_business_products)
            ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
            ->first();

        if (!$product) {
            return response()->json(['error' => 'Product not found']);
        }

         //check if batch_hidden column exists
         $batch_hidden = false;
        if (Schema::hasColumn('product_batch', 'batch_hidden')) {
            $batch_hidden = true;
        }

        $query = DB::table('business_products')
            ->select(
                'id_batch',
                'batch_number',
                'expiry_date',
                'batch_qty',
                'batch_amount',
                DB::raw('date_format(batch_date, "%Y-%m-%d") as batch_date'),
                'created_at',
                'business_store',
                'store_id',
                DB::raw($group == "true" ? "
                    SUM(IFNULL(e.addition, 0)) AS manualQty,
                    SUM(IFNULL(e.qty_purchased, 0)) AS purchasedQty,
                    SUM(IFNULL(e.transfer_in, 0)) AS transfer_in,
                    SUM(IFNULL(e.transfer_out, 0)) AS transfer_out,
                    SUM(IFNULL(e.sold,0) + IFNULL(e.franchise_sale,0)) AS sold,
                    SUM(IFNULL(e.franchise_sale,0)) AS franchise_sale,
                    SUM(IFNULL(e.used,0)) AS used,
                    SUM(IFNULL(e.returned,0)) AS returned,
                    SUM(
                        (IFNULL(e.addition, 0) + IFNULL(e.qty_purchased, 0) + IFNULL(e.transfer_in, 0))
                        - (IFNULL(e.transfer_out, 0) + IFNULL(e.sold, 0) + IFNULL(e.used, 0)
                        + IFNULL(e.returned, 0) + IFNULL(e.franchise_sale,0))
                    ) AS total_stock,
                    DATE_FORMAT(expiry_date, '%d-%m-%Y') AS batch_expiry,
                    ROUND(DATEDIFF(expiry_date, NOW())/30) AS expiry,
                    DATE_FORMAT(batch_date, '%d-%m-%Y') AS bdate,
                    batch_amount
                " : "
                    IFNULL(e.addition, 0) AS manualQty,
                    IFNULL(e.qty_purchased, 0) AS purchasedQty,
                    IFNULL(e.transfer_in, 0) AS transfer_in,
                    IFNULL(e.transfer_out, 0) AS transfer_out,
                    (IFNULL(e.sold,0) + IFNULL(e.franchise_sale,0)) AS sold,
                    IFNULL(e.franchise_sale,0) AS franchise_sale,
                    IFNULL(e.used,0) AS used,
                    IFNULL(e.returned,0) AS returned,
                    ROUND(
                        (IFNULL(e.addition, 0) + IFNULL(e.qty_purchased, 0) + IFNULL(e.transfer_in, 0))
                        - (IFNULL(e.transfer_out, 0) + IFNULL(e.sold, 0) + IFNULL(e.used, 0)
                        + IFNULL(e.returned, 0) + IFNULL(e.franchise_sale,0))
                    ,2) AS total_stock,
                    DATE_FORMAT(expiry_date, '%d-%m-%Y') AS batch_expiry,
                    ROUND(DATEDIFF(expiry_date, NOW())/30) AS expiry,
                    DATE_FORMAT(batch_date, '%d-%m-%Y') AS bdate,
                    batch_amount
                ")
            )
            ->leftJoin(DB::raw($batch_hidden ? "(
                SELECT 
                    id_batch, batch_number, batch_date, expiry_date, batch_qty, qty_purchased,
                    product_batch.product_id, business_store, transfer_in, transfer_out,
                    sold, used, returned, batch_amount, addition, franchise_sale, store_id, business_stores.business_id, batch_hidden
                FROM product_batch
                LEFT JOIN business_stores ON business_stores.id_business_stores = product_batch.store_id
                LEFT JOIN (
                    SELECT batch_id, SUM(adjustment_qty) AS addition FROM adjustment_notes GROUP BY batch_id
                ) AS adj ON adj.batch_id = product_batch.id_batch                
                LEFT JOIN (
                    SELECT batch_id, product_id, SUM(invoice_qty) AS sold
                    FROM invoice_products
                    JOIN invoice ON invoice.id_invoice = invoice_products.invoice_id
                        AND invoice_status = 'valid'
                        AND IFNULL(reference_invoice_number,'') = ''
                        AND batch_id IS NOT NULL
                    GROUP BY batch_id, product_id
                ) AS a ON a.product_id = product_batch.product_id AND a.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch, batch_id, SUM(IFNULL(dispatch_qty,0)) AS used
                    FROM dispatch_notes
                    WHERE status = 'Active' AND batch_id IS NOT NULL
                    GROUP BY product_id, batch, batch_id
                ) AS b ON b.product_id = product_batch.product_id AND b.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id, SUM(IFNULL(return_qty,0)) AS returned
                    FROM return_notes
                    WHERE return_status = 'Active' AND batch_id IS NOT NULL
                    GROUP BY product_id, batch_id
                ) AS c ON c.product_id = product_batch.product_id AND c.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT grn_product_id, grn_batch_id, SUM(IFNULL(grn_qty_received,0)) AS qty_purchased
                    FROM grn_details
                    WHERE grn_batch_id IS NOT NULL
                    GROUP BY grn_product_id, grn_batch_id
                ) AS d ON d.grn_product_id = product_batch.product_id AND d.grn_batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id,
                        SUM(IFNULL(tranfer_out_qty,0)) AS transfer_out,
                        SUM(IFNULL(tranfer_in_qty,0)) AS transfer_in
                    FROM transfer_notes
                    WHERE batch_id IS NOT NULL AND transfer_notes.status = 'Active'
                    GROUP BY product_id, batch_id
                ) AS f ON f.product_id = product_batch.product_id AND f.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id, SUM(IFNULL(qty,0)) AS franchise_sale
                    FROM franchise_orders
                    JOIN franchise_order_products
                        ON franchise_orders.id_franchise_orders = franchise_order_products.franchise_order_id
                    WHERE batch_id IS NOT NULL
                    AND (franchise_orders.order_status='Paid' OR franchise_orders.order_status='Invoiced')
                    GROUP BY product_id, batch_id
                ) AS h ON h.batch_id = product_batch.id_batch AND h.product_id = product_batch.product_id
                ) AS e 
                " : " 
                (
                SELECT 
                    id_batch, batch_number, batch_date, expiry_date, batch_qty, qty_purchased,
                    product_batch.product_id, business_store, transfer_in, transfer_out,
                    sold, used, returned, batch_amount, addition, franchise_sale, store_id, business_stores.business_id
                FROM product_batch
                LEFT JOIN business_stores ON business_stores.id_business_stores = product_batch.store_id
                LEFT JOIN (
                    SELECT batch_id, SUM(adjustment_qty) AS addition FROM adjustment_notes GROUP BY batch_id
                ) AS adj ON adj.batch_id = product_batch.id_batch                
                LEFT JOIN (
                    SELECT batch_id, product_id, SUM(invoice_qty) AS sold
                    FROM invoice_products
                    JOIN invoice ON invoice.id_invoice = invoice_products.invoice_id
                        AND invoice_status = 'valid'
                        AND IFNULL(reference_invoice_number,'') = ''
                        AND batch_id IS NOT NULL
                    GROUP BY batch_id, product_id
                ) AS a ON a.product_id = product_batch.product_id AND a.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch, batch_id, SUM(IFNULL(dispatch_qty,0)) AS used
                    FROM dispatch_notes
                    WHERE status = 'Active' AND batch_id IS NOT NULL
                    GROUP BY product_id, batch, batch_id
                ) AS b ON b.product_id = product_batch.product_id AND b.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id, SUM(IFNULL(return_qty,0)) AS returned
                    FROM return_notes
                    WHERE return_status = 'Active' AND batch_id IS NOT NULL
                    GROUP BY product_id, batch_id
                ) AS c ON c.product_id = product_batch.product_id AND c.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT grn_product_id, grn_batch_id, SUM(IFNULL(grn_qty_received,0)) AS qty_purchased
                    FROM grn_details
                    WHERE grn_batch_id IS NOT NULL
                    GROUP BY grn_product_id, grn_batch_id
                ) AS d ON d.grn_product_id = product_batch.product_id AND d.grn_batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id,
                        SUM(IFNULL(tranfer_out_qty,0)) AS transfer_out,
                        SUM(IFNULL(tranfer_in_qty,0)) AS transfer_in
                    FROM transfer_notes
                    WHERE batch_id IS NOT NULL AND transfer_notes.status = 'Active'
                    GROUP BY product_id, batch_id
                ) AS f ON f.product_id = product_batch.product_id AND f.batch_id = product_batch.id_batch
                LEFT JOIN (
                    SELECT product_id, batch_id, SUM(IFNULL(qty,0)) AS franchise_sale
                    FROM franchise_orders
                    JOIN franchise_order_products
                        ON franchise_orders.id_franchise_orders = franchise_order_products.franchise_order_id
                    WHERE batch_id IS NOT NULL
                    AND (franchise_orders.order_status='Paid' OR franchise_orders.order_status='Invoiced')
                    GROUP BY product_id, batch_id
                ) AS h ON h.batch_id = product_batch.id_batch AND h.product_id = product_batch.product_id
            ) AS e"), 'e.product_id', '=', 'business_products.id_business_products' 
           
            )
            ->where('id_business_products', $id_business_products);

        // session-based filters
        if (session('common_products') === 'No') {
            $query->where('e.business_id', session('business_id'));
        } elseif (!empty($business_id)) {
            $query->where('e.business_id', $business_id);
        } else {
            if(session('ho') === 'Yes'){
                //all business with common products
                $common_business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
                $query->whereIn('e.business_id', $common_business_ids);
            } else {
                $query->where('e.business_id', session('business_id'));
            }
        }

        if (!empty($store_id)) {
            $query->where('e.store_id', $store_id);
        }

        // group by condition
        if ($group == true) {
            $query->groupBy('business_products.id_business_products');
        }
       //batch hidden true
        if ($batch_hidden) {
            $query->where('e.batch_hidden', 'No');
        }
        $query->orderBy('expiry');

        $batches = $query->get();

        if(session('common_products') === 'No') {
            $stores = Stores::where('business_id', session('business_id'))->get();
        } elseif (!empty($business_id)) {
            $stores = Stores::where('business_id', $business_id)->get();
        } else {
            if(session('ho') === 'Yes'){
                //all business with common products
                $common_business_ids = Business::where('common_products', 'Yes')->pluck('id_business')->toArray();
                $stores = Stores::whereIn('business_id', $common_business_ids)->get();
            } else {
                $stores = Stores::where('business_id', session('business_id'))->get();
            }
        }
        
        //if product type is compiled kit get kit components
        if($product->product_type == "Compiled Kit"){
            $kit_components = DB::table('kit_config')
                ->select('kit_config.*', 'business_products.product as component_name', 'business_products.measure_unit as measure_unit')
                ->join('business_products', 'business_products.id_business_products', '=', 'kit_config.business_product_id')
                ->where('kit_config.kit_id', $id_business_products)
                ->whereNull('kit_config.deleted_at')
                ->get();
            $product->kit_components = $kit_components;
        } else {
            $product->kit_components = [];
        }

        return view('products.product_batches', compact('batches', 'id_business_products', 'product', 'stores', 'store_id'));
    }

    public function product_batch_details($id_batch)
    {
        $product_id = Products::select()
        ->join('product_batch', 'product_batch.product_id', '=', 'business_products.id_business_products')
        ->where('id_batch', $id_batch)
        ->first()
        ->product_id ?? null;
        
        if (!$product_id) {
            return response()->json(['error' => 'Product not found']);
        }

        return redirect()->route('products.batches', [
            'id_business_products' => $product_id
        ]);
    }

    public function save_product_batch(Request $request){
        
        try{
            Db::beginTransaction();
            $business_id = session('business_id');
            $id_batch = $request->input('id_batch');
            $batch = ProductBatch::where('id_batch', $id_batch)->first();

   
            // Decode the JSON string into a PHP array
            $adjustments = json_decode($request->input('adjustments'), true);
            if (!$batch) {
                //Insert new batch
                $new_batch = new ProductBatch();
                $new_batch->product_id = $request->input('product_id');
                $new_batch->batch_number = $request->input('batch_number', '');
                $new_batch->expiry_date = $request->input('expiry_date', null);
                //$new_batch->batch_qty = $request->input('batch_qty', 0);
                $new_batch->batch_amount = $request->input('batch_amount', 0);
                $new_batch->store_id = $request->input('store_id', null);
                $new_batch->batch_hidden = $request->input('batch_hidden', 'No');
                $batch_amount =  $request->input('batch_amount', 0);
                
                $new_batch->save();

                //Add adjustment Notes if any
                if($adjustments != null){
                    foreach($adjustments as $adjustment){
                        $new_adjustment = new AdjustmentNotes();
                        $new_adjustment->batch_id = $new_batch->id_batch;
                        $new_adjustment->product_id = $new_batch->product_id;                        
                        $new_adjustment->adjustment_qty = $adjustment['adjustment_qty'];
                        $new_adjustment->adjustment_date = now()->format('Y-m-d');
                        $new_adjustment->adjustment_remarks = $adjustment['adjustment_reason'];
                        $new_adjustment->created_by = session('user_name');
                        $new_adjustment->save();  

                        $adjustment_qty =   $adjustment['adjustment_qty'] ?? 0;
                        $inventory_amount_for_voucher = $batch_amount*$adjustment_qty;
                        $adjustment_note_result =  $this->create_adjustment_note_voucher($business_id,$inventory_amount_for_voucher,$new_adjustment->id_adjustment_notes ,$new_batch->product_id , $batch_amount, $adjustment_qty ,$new_batch->id_batch);
                        $dresponse = json_decode($adjustment_note_result->getContent(), true);

                        if(!$adjustment_note_result || (isset($dresponse['success']) && $dresponse['success'] == false)){
                            DB::rollBack();
                            return response()->json([
                                'message' => $dresponse['message'].'Failed to adjustment voucher for product ID '.$new_batch->product_id,
                                'success' => false,
                                'message_type' => 'error',
                                'message_btn' => 'btn btn-danger',
                            ]);
                        }  

                    }
                } 

                DB::commit();
                return response()->json([
                    'message' => 'Product batch created successfully',
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success'
                ]);

            } else {
                //Update existing batch
                $batch->batch_number = $request->input('batch_number', $batch->batch_number);
                $batch->expiry_date = $request->input('expiry_date', $batch->expiry_date);
                //$batch->batch_qty = $request->input('batch_qty', $batch->batch_qty);
                $batch->batch_amount = $request->input('batch_amount', $batch->batch_amount);
                $batch->store_id = $request->input('store_id', $batch->store_id);
                $batch->batch_hidden = $request->input('batch_hidden', $batch->batch_hidden) ?? 'No';
                $batch_amount =  $request->input('batch_amount', 0);

                $batch->save();
                   
                //Add adjustment Notes if any
                if($adjustments != null){
                    foreach($adjustments as $adjustment){                        
                        $new_adjustment = new AdjustmentNotes();
                        $new_adjustment->batch_id = $batch->id_batch;
                        $new_adjustment->product_id = $batch->product_id;
                        $new_adjustment->adjustment_date = $adjustment['adjustment_date'];
                        $new_adjustment->adjustment_qty = $adjustment['adjustment_qty'];
                        $new_adjustment->adjustment_remarks = $adjustment['adjustment_reason'];
                        $new_adjustment->created_by = session('user_name');
                        $new_adjustment->save(); 


                        $adjustment_qty =   $adjustment['adjustment_qty'] ?? 0;
                        $inventory_amount_for_voucher = $batch_amount*$adjustment_qty;
                        $adjustment_note_result =  $this->create_adjustment_note_voucher($business_id,$inventory_amount_for_voucher,$new_adjustment->id_adjustment_notes ,$batch->product_id, $batch_amount, $adjustment_qty ,$batch->id_batch);
                        $dresponse = json_decode($adjustment_note_result->getContent(), true);

                        if(!$adjustment_note_result || (isset($dresponse['success']) && $dresponse['success'] == false)){
                            DB::rollBack();
                            return response()->json([
                                'message' => $dresponse['message'].'Failed to adjustment voucher for product ID '.$batch->product_id,
                                'success' => false,
                                'message_type' => 'error',
                                'message_btn' => 'btn btn-danger',
                            ]);
                        }  


                    }
                }

                DB::commit();
                return response()->json([
                    'message' => 'Product batch updated successfully',
                    'success' => true,
                    'message_type' => 'success',
                    'message_btn' => 'btn btn-success'
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to save product batch: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',

            ]);   
        }
    }
    
    public function transfer_product_stock(Request $request){
        $product_id = $request->input('product_id');
        $from_store_id = $request->input('from_store_id');
        $to_store_id = $request->input('to_store_id');
        $batch_id = $request->input('batch_id');
        $transfer_qty = $request->input('transfer_qty');
        $business_id = $request->input('business_id');

        //get the batch details from the source store
        $transfer_from_batch = ProductBatch::select(
            'product_batch.*',
            'business_stores.business_store as store_name',
            'business.business_name as business_name',
            'business_stores.business_id as business_id',
            'business_products.product as unit_type'
        )
        ->where('id_batch', $batch_id)
        ->join('business_stores', 'business_stores.id_business_stores', '=', 'product_batch.store_id')
        ->join('business', 'business.id_business', '=', 'business_stores.business_id')
        ->join('business_products', 'business_products.id_business_products', '=', 'product_batch.product_id')
        ->where('product_batch.product_id', $product_id)
        ->first();

        if (!$transfer_from_batch) {            
            //Call the function that will create transfer notes from multiple batches until the requested qty is fulfilled
            $transfer_result = $this->transfer_from_multiple_batches($product_id, $from_store_id, $to_store_id, $transfer_qty);
            return $transfer_result;
            
        } else {
            $batch = $transfer_from_batch->batch_number;
            $expiry_date = $transfer_from_batch->expiry_date;
            $business_id = $transfer_from_batch->business_id;
            $batch_amount = $transfer_from_batch->batch_amount;
            $store_name = $transfer_from_batch->store_name;
            $business_name = $transfer_from_batch->business_name;
            //Call the function that will transfer stock from the specified batch
            $transfer_result = $this->transfer_product_stock_from_batch($product_id, $from_store_id, $to_store_id, $batch_id, $transfer_qty, $batch, $expiry_date, $batch_amount, $store_name);
            return $transfer_result;
        }
    }

    public function transfer_product_stock_from_batch($product_id, $from_store_id, $to_store_id, $batch_id, $transfer_qty, $batch, $expiry_date, $batch_amount, $store_name, $comment = ''){
       try {
            DB::beginTransaction();
            
            //get business id for out store
            $from_business = Stores::select('id_business', 'business_name')
            ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->where('id_business_stores', $from_store_id)->first();            
            $business_id = $from_business->id_business;
            $business_name = $from_business->business_name;

            //get business id for in store
            $to_business = Stores::select('id_business', 'business_name')
            ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->where('id_business_stores', $to_store_id)->first();
            $new_business_id = $to_business->id_business;
            $new_business_name = $to_business->business_name;

            //Create a new GTN entry for transfer out from the source store
            $gtn = new Gtn();
            $gtn->gtn_number = 'GTN'.str_pad(Gtn::max('id_gtn') + 1, 6, '0', STR_PAD_LEFT);
            $gtn->comments = $comment != '' ? $comment : 'Stock transfer from store '.$store_name.' branch : '.$business_name.' to store '.$to_business->business_store.' branch : '.$new_business_name;
            $gtn->gtn_date = now();
            $gtn->business_id = $business_id;
            $gtn->to_store = $to_store_id;
            $gtn->created_by = session('user_name');
            $gtn->save();
            //Get the generated GTN ID
            $gtn_id = $gtn->id_gtn;

            //Create a new batch in the transferred to store to receive the transferred stock
            //get max batch_number for the product in the destination store
            $max_batch_number = ProductBatch::where('product_id', $product_id)
                ->where('store_id', $to_store_id)
                ->max('batch_number');
            $new_batch = new ProductBatch();
            $new_batch->product_id = $product_id;
            $new_batch->batch_number = $max_batch_number + 1;
            $new_batch->expiry_date = $expiry_date;
            $new_batch->batch_qty = 0;
            $new_batch->store_id = $to_store_id;
            $new_batch->batch_amount = $batch_amount;
            $new_batch->save();
            //Get the new batch ID
            $new_batch_id = $new_batch->id_batch;

            //get the new batch details
            $new_batch_details = ProductBatch::select(
                'product_batch.*',
                'business_stores.business_store as store_name',
                'business.business_name as business_name',
                'business_stores.business_id as business_id',
                'business_products.product as unit_type'
            )
            ->where('id_batch', $new_batch_id)
            ->join('business_stores', 'business_stores.id_business_stores', '=', 'product_batch.store_id')
            ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->join('business_products', 'business_products.id_business_products', '=', 'product_batch.product_id')
            ->where('product_batch.product_id', $product_id)
            ->first();

            if( !$new_batch_details) {
                DB::rollBack();
                return response()->json([
                    'message' => 'Failed to create new batch in destination store',
                    'success' => false,
                    'message_type' => 'error',
                    'message_btn' => 'btn btn-danger',
                ]);
            }else{
                $new_batch_id= $new_batch_details->id_batch;
                $new_batch_number= $new_batch_details->batch_number;
                $new_expiry_date = $new_batch_details->expiry_date;
                $new_business_id = $new_batch_details->business_id;
                $new_batch_amount = $new_batch_details->batch_amount;
                $new_store_name = $new_batch_details->store_name;
                $new_business_name = $new_batch_details->business_name;

            }

            // Create a new transfer note for receiving stock in the destination store
            $transfer_note = new TransferNotes();
            $transfer_note->gtn_id = $gtn_id;
            $transfer_note->business_id = $new_business_id;
            $transfer_note->product_id = $product_id;
            $transfer_note->tranfer_out_qty = 0;
            $transfer_note->tranfer_in_qty = $transfer_qty;
            $transfer_note->unit_type = Products::where('id_business_products', $product_id)->value('unit_type');
            $transfer_note->batch_id = $new_batch_id;
            $transfer_note->batch = $new_batch_number;       
            $transfer_note->status = 'Active';
            $transfer_note->created_by = session('user_name');
            $transfer_note->transfer_date = now();
            $transfer_note->save();

            // Create a new transfer note for transferring stock out from the source store
            $transfer_note_out = new TransferNotes();
            $transfer_note_out->gtn_id = $gtn_id;
            $transfer_note_out->business_id = $business_id;
            $transfer_note_out->product_id = $product_id;
            $transfer_note_out->tranfer_out_qty = $transfer_qty;
            $transfer_note_out->tranfer_in_qty = 0;
            $transfer_note_out->unit_type = Products::where('id_business_products', $product_id)->value('unit_type');
            $transfer_note_out->batch_id = $batch_id;
            $transfer_note_out->batch = $batch;
            $transfer_note_out->status = 'Active';
            $transfer_note_out->created_by = session('user_name');
            $transfer_note_out->transfer_date = now();
            $transfer_note_out->save();


            //Create transfered account voucher using account_event 34 at the sending branch
            $account_voucher = new AccountVouchers();
            $account_voucher->voucher_date = now();            
            $account_voucher->voucher_type = 3; //Journal Voucher
            $account_voucher->voucher_status = 'Active';
            $account_voucher->created_by = session('user_name');
            $account_voucher->created_on = date('Y-m-d H:i:s');
            $account_voucher->gtn_id = $gtn_id;     
            $account_voucher->id_account_head = 24; //1 = Accounts Receivable
            $account_voucher->voucher_date = now();
            $account_voucher->description = 'Stock Transfer Voucher for GTNID '.$gtn_id.' on '.now().', to store  '. $new_store_name  .' branch : '.$new_business_name;
            $account_voucher->business_id = $business_id;
            $account_voucher->voucher_amount = $batch_amount * $transfer_qty;
            $account_voucher->cost_center = 2;
            $account_voucher->cost_center_name = 'Back Office';
            $account_voucher->business_partner = '5';
            $account_voucher->business_partner_id = $new_business_id;
            $account_voucher->business_partner_name = $new_business_name;       
            $account_voucher->payment_mode = 'Cash';
            $account_voucher->auto_voucher = 'Yes';
            $account_voucher->save();
            $new_voucher_id = $account_voucher->id_account_vouchers;
            
            //Get the event mappings for Invoice Creation id_events = 34
            if(session('ho_accounts')=='Yes'){
                $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $common_business_id = $business_id;
            }
            $event_mappings = AccountEventMapping::where('account_event_id', 34)->where('business_id', $common_business_id)->get();
            $debit_amounts = [];
            $credit_amounts = [];
            $debit_accounts = array();
            $credit_accounts = array();
            foreach ($event_mappings as $mapping) {
                if($mapping['transaction_type']=='debit'){
                    array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                } else {
                    array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                }

                if($mapping['transaction_type']=='debit'){
                    //make debit entry data
                    if($mapping['entity_name']=='receivable'){array_push($debit_amounts, [ 'entity_name' => 'receivable', 'amount' => $batch_amount * $transfer_qty ]);}                
                } else if($mapping['transaction_type']=='credit'){
                    //make credit entry data
                    if($mapping['entity_name']=='inventory_out'){array_push($credit_amounts, [ 'entity_name' => 'inventory_out', 'amount' => $batch_amount * $transfer_qty ]);}                                        
                }
            }
            //get the sum of debit amounts and credit amounts
            $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
            $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;
            if($debit_sum == null || $credit_sum == null || $debit_sum != $credit_sum){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "btn btn-danger"
                ]);
            } else if($debit_sum == 0 || $credit_sum == 0){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit or Credit amounts are zero. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "btn btn-danger"
                ]);
            } else {
                //Insert voucher Details
                foreach($debit_accounts as $debit){
                    $debit_parts = explode("|", $debit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->detail_remarks = 'Stock Transfer to store  '. $new_store_name  .' branch : '.$new_business_name;
                    $new_voucher_details->account_head_id = $debit_parts[0];
                    $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                    $new_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $debit_amounts));
                    $new_voucher_details->credit = 0;    
                    if($new_voucher_details->debit != 0){                                            
                        $new_voucher_details->save();
                    }
                }
                foreach($credit_accounts as $credit){
                    $credit_parts = explode("|", $credit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->detail_remarks = 'Stock Transfer to store  '. $new_store_name  .' branch : '.$new_business_name;
                    $new_voucher_details->account_head_id = $credit_parts[0];
                    $new_voucher_details->debit = 0;
                    $entity = $credit_parts[2]; // this is 'paid_cash' etc.
                    $new_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $credit_amounts));
                    if($new_voucher_details->credit != 0){           
                        $new_voucher_details->save();
                    }
                }
            }

            //Create transfered account voucher using account_event 34 at the receiving branch
            $account_receiving_voucher = new AccountVouchers();
            $account_receiving_voucher->voucher_date = now();            
            $account_receiving_voucher->voucher_type = 3; //Journal Voucher
            $account_receiving_voucher->voucher_status = 'Active';
            $account_receiving_voucher->created_by = session('user_name');
            $account_receiving_voucher->created_on = date('Y-m-d H:i:s');
            $account_receiving_voucher->gtn_id = $gtn_id;     
            $account_receiving_voucher->id_account_head = 24; //1 = Accounts Receivable
            $account_receiving_voucher->voucher_date = now();
            $account_receiving_voucher->description = 'Stock Transfer Voucher for GTNID '.$gtn_id.' on '.now().', from store  '. $store_name  .' branch : '.$business_name;
            $account_receiving_voucher->business_id = $new_business_id;
            $account_receiving_voucher->voucher_amount = $batch_amount * $transfer_qty;
            $account_receiving_voucher->cost_center = 2;
            $account_receiving_voucher->cost_center_name = 'Back Office';
            $account_receiving_voucher->business_partner = '5';
            $account_receiving_voucher->business_partner_id = $business_id;
            $account_receiving_voucher->business_partner_name = $business_name;       
            $account_receiving_voucher->payment_mode = 'Cash';
            $account_receiving_voucher->auto_voucher = 'Yes';
            $account_receiving_voucher->save();
            $receiving_voucher_id = $account_receiving_voucher->id_account_vouchers;
            
            //Get the event mappings for Invoice Creation id_events = 34
            if(session('ho_accounts')=='Yes'){
                $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $common_business_id = $new_business_id;
            }
            $event_mappings = AccountEventMapping::where('account_event_id', 34)->where('business_id', $common_business_id)->get();
            $debit_amounts = [];
            $credit_amounts = [];
            $debit_accounts = array();
            $credit_accounts = array();
            foreach ($event_mappings as $mapping) {
                if($mapping['transaction_type']=='debit'){
                    array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                } else {
                    array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                }

                if($mapping['transaction_type']=='debit'){
                    //make debit entry data
                    if($mapping['entity_name']=='inventory_in'){array_push($debit_amounts, [ 'entity_name' => 'inventory_in', 'amount' => $batch_amount * $transfer_qty ]);}                
                } else if($mapping['transaction_type']=='credit'){
                    //make credit entry data
                    if($mapping['entity_name']=='payable'){array_push($credit_amounts, [ 'entity_name' => 'payable', 'amount' => $batch_amount * $transfer_qty ]);}                                        
                }
            }
            //get the sum of debit amounts and credit amounts
            $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
            $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;
            if($debit_sum == null || $credit_sum == null || $debit_sum != $credit_sum){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "btn btn-danger"
                ]);
            } else if($debit_sum == 0 || $credit_sum == 0){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit or Credit amounts are zero. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "btn btn-danger"
                ]);
            } else {
                //Insert voucher Details
                foreach($debit_accounts as $debit){
                    $debit_parts = explode("|", $debit);
                    $new_receiving_voucher_details = new AccountVoucherDetail();
                    $new_receiving_voucher_details->account_voucher_id = $receiving_voucher_id;
                    $new_receiving_voucher_details->detail_remarks = 'Stock Transfer from store  '. $store_name  .' branch : '.$business_name;
                    $new_receiving_voucher_details->account_head_id = $debit_parts[0];
                    $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                    $new_receiving_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $debit_amounts));
                    $new_receiving_voucher_details->credit = 0;    
                    if($new_receiving_voucher_details->debit != 0){                                            
                        $new_receiving_voucher_details->save();
                    }
                }
                foreach($credit_accounts as $credit){
                    $credit_parts = explode("|", $credit);
                    $new_receiving_voucher_details = new AccountVoucherDetail();
                    $new_receiving_voucher_details->account_voucher_id = $receiving_voucher_id;
                    $new_receiving_voucher_details->detail_remarks = 'Stock Transfer from store  '. $store_name  .' branch : '.$business_name;
                    $new_receiving_voucher_details->account_head_id = $credit_parts[0];
                    $new_receiving_voucher_details->debit = 0;
                    $entity = $credit_parts[2]; // this is 'paid_cash' etc.
                    $new_receiving_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $credit_amounts));
                    if($new_receiving_voucher_details->credit != 0){           
                        $new_receiving_voucher_details->save();
                    }
                }
            }

            DB::commit();

            return response()->json([
                'message' => 'Product stock transferred successfully',
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to transfer product stock: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

    public function transfer_from_multiple_batches($product_id, $from_store_id, $to_store_id, $transfer_qty){
        //Use ProductBatchService to get batches with sufficient stock in the source store
        $productBatchService = new ProductBatchService();
        $sufficient_batches = $productBatchService->findSufficientBatches($product_id, 0, $from_store_id, $batchId = null);

        if(!$sufficient_batches){
            $transfer_result = response()->json([
                'message' => 'No sufficient stock available in source store to transfer',
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        } else {
            $batches_array = json_decode($sufficient_batches, true);
            $remaining_qty = $transfer_qty;
            foreach($batches_array as $batch){
                if($remaining_qty <= 0){
                    break; //Exit the loop if transfer qty is fulfilled
                }
                $batch_id = $batch['batch_id'];
                $batch_no = $batch['batch_no'];
                $available_qty = $batch['instock'];
                $store_name = $batch['store'];
                if($available_qty >= $remaining_qty){
                    //Transfer the remaining qty from this batch
                    $transfer_result = $this->transfer_product_stock_from_batch($product_id, $from_store_id, $to_store_id, $batch_id, $remaining_qty, $batch_no, $batch['expiry_date'], $batch['batch_amount'], $store_name);                    
                    $remaining_qty = 0;
                } else {
                    //Transfer the available qty from this batch
                    $transfer_result = $this->transfer_product_stock_from_batch($product_id, $from_store_id, $to_store_id, $batch_id, $available_qty, $batch_no, $batch['expiry_date'], $batch['batch_amount'], $store_name);
                    $remaining_qty -= $available_qty;
                }
            }
            if($remaining_qty > 0){
                //Not enough stock to fulfill the transfer request
                $transfer_result = response()->json([
                    'message' => 'Insufficient stock in source store to completely fulfill the transfer request',
                    'success' => false,
                    'message_type' => 'error',
                    'message_btn' => 'btn btn-danger',
                ]);
            }
        }
        return $transfer_result;
    }

    public function multiple_product_batch_transfer_notes(Request $request)
    {
        
        try{
            $requestData = $request->transfers;

            foreach($requestData as $transfer) {

                //get the prouct details and its batch details to validate stock availability
                $productBatch = DB::table('product_batch')
                    ->select('product_batch.id_batch', 'product_batch.batch_amount', 'product_batch.batch_number', 'product_batch.expiry_date',
                    'business_products.id_business_products', 'business_products.product',
                    'business_products.unit_type', 'business_brands.business_brand_name',
                    'business_products.measure_unit', 'business_products.qty_per_unit'
                    )
                    ->join('business_products', 'business_products.id_business_products', '=', 'product_batch.product_id')
                    ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')
                    ->where('id_batch', $transfer['batch_id'])                    
                    ->first();

                $product_id = $transfer['product_id'];
                $from_store_id = $transfer['from_store'];
                $to_store_id = $transfer['to_store'];
                $batch_id = $transfer['batch_id'];
                $remaining_qty = $transfer['transfer_qty'];
                $batch_no = $productBatch->batch_number;
                $store_name = Stores::where('id_business_stores', $from_store_id)->value('business_store');
                $expiry_date = $productBatch->expiry_date;
                $batch_amount = $productBatch->batch_amount;

                $transfer_result = $this->transfer_product_stock_from_batch($product_id, $from_store_id, $to_store_id, $batch_id, $remaining_qty, $batch_no, $expiry_date, $batch_amount, $store_name);

                if (!$transfer_result->getData()->success) {                    
                    return $transfer_result; // Return the error response if transfer failed
                }
                
            }
            
            return response()->json([
                'success' => true, 
                'message_type' => 'success',
                'message_btn' => 'btn btn-success',
                'message' => 'Batch transfer notes for Multiple Products created successfully.']);

        } catch (\Exception $e) {
            
            return response()->json([
                'success' => false, 
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
                'message' => 'Error creating batch transfer notes: ' . $e->getMessage()]);
        }
    }

    public function fetch_store_stock(Request $request){ //pass a product and a store to see available qty
        $product_id = $request->input('product_id');
        $store_id = $request->input('store_id');
        //Get all batches with sufficient available qty in the store using productBatchService.php
        $productBatchService = new ProductBatchService();
        $batches = $productBatchService->findSufficientBatches($product_id, 0, $store_id);
        //Sum up the available qty using string $batches
        $batches_array = json_decode($batches, true);        
        $store_available_qty = array_sum(array_column($batches_array, 'instock'));
         

        return response()->json([
            'message_type' => 'success',
            'message_btn' => 'btn btn-success',
            'message' => 'Store stock fetched successfully',
            'batches' => $batches,
            'data' => $batches_array,
            'available_quantity' => $store_available_qty
        ]);
    }

    public function compiled_kits(Request $request){        
       

        $brands = Brands::select('*');
        
        if(session('common_products')=="Yes"){
            $brands = $brands->whereIn('business_id', Business::where('common_products', 'Yes')->pluck('id_business'));
        } else {
            $brands = $brands->where('business_id', session('business_id'));
        }
        $brands = $brands->get();
        
        $unit_types = DB::table('unit_type')->get();

        $measure_units = DB::table('measurement_units')->get();

        return view('products.compiled_kits.compiled_kits', compact('brands', 'unit_types', 'measure_units'));

    }

    public function compiledkitsdata(Request $request){        

        // $kits = Products::where('product_type', 'Compiled Kit')            
        //     ->select(
        //         'business_products.id_business_products'                
        //     )
        //     ->get();

        $compiled_kits = KitConfig::select(
                'kit_config.*',
                'business_products.product as kit_name',
                'business_brands.business_brand_name as kit_brand_name',
                'component_products.product as component_product_name',                
                'component_products.measure_unit as component_measure_unit',
                'kit_config.kit_measure_qty as component_measure_qty',                
                'kit_config.kit_qty as component_quantity',
                'kit_config.kit_id as compiled_kit_id',
                'kit_config.kit_product_cost as component_product_cost',
                'kit_config.deleted_at as kit_config_deleted_at'
            )
            ->join('business_products', function($join) {
                $join->on('business_products.id_business_products', '=', 'kit_config.kit_id')
                     ->where('business_products.product_type', '=', 'Compiled Kit');
            })
            ->join('business_brands', 'business_brands.id_business_brands', '=', 'business_products.brand_id')                        
            ->join('business_products as component_products', 'component_products.id_business_products', '=', 'kit_config.business_product_id')
            ;
            
            return DataTables::of($compiled_kits)
            ->order(function ($query) {
                if (request()->has('order')) {
                    $order = request('order')[0];
                    $columns = request('columns');
                    $orderColumnIndex = $order['column'];
                    $orderColumnName = $columns[$orderColumnIndex]['data'];
                    $orderDirection = $order['dir'];
                    $query->orderBy($orderColumnName, $orderDirection);
                } else {
                    $query->orderBy('kit_config.kit_id', 'asc');
                }
            })
            ->filterColumn('kit_brand_name', function ($query, $keyword) {
                $query->where('business_brands.business_brand_name', 'like', "%{$keyword}%");
            })
            ->filterColumn('kit_name', function ($query, $keyword) {
                $query->where('business_products.product', 'like', "%{$keyword}%");
            })
            ->filterColumn('component_product_name', function ($query, $keyword) {
                $query->where('component_products.product', 'like', "%{$keyword}%");
            })
            ->filterColumn('component_measure_unit', function ($query, $keyword) {
                $query->where('component_measure_unit', 'like', "%{$keyword}%");
            })
            ->filterColumn('id_kit_config', function ($query, $keyword) {
                $query->where('id_kit_config', 'like', "%{$keyword}%");
            })
            ->filterColumn('kit_id', function ($query, $keyword) {
                $query->where('kit_id', 'like', "%{$keyword}%");
            })
            ->make(true);

    }

    public function compiled_kit_components($id_compiled_kits) {

        $components = KitConfig::select(
            'kit_config.*',
            'business_products.product as component_product_name',                
            'business_products.measure_unit as component_measure_unit',
            'kit_config.kit_measure_qty as component_measure_qty',                
            'kit_config.kit_qty as component_quantity',
            'kit_config.kit_product_cost as component_product_cost'
        )
        ->where('kit_id', $id_compiled_kits)
        ->whereNull('kit_config.deleted_at')
        ->join('business_products', 'business_products.id_business_products', '=', 'kit_config.business_product_id')
        ->get();

        return response()->json([
            'components' => $components
        ]);
        
    }

    function add_component_to_kit(Request $request) {
        try{
            DB::beginTransaction();

            $kit_id = $request->input('kit_id');
            $business_product_id = $request->input('business_product_id');
            $kit_qty = $request->input('kit_qty');
            $kit_measure_qty = $request->input('kit_measure_qty');
            $kit_product_cost = $request->input('kit_product_cost');

            $new_component = new KitConfig();
            
            $new_component->kit_id = $kit_id;
            $new_component->business_product_id = $business_product_id;
            $new_component->kit_qty = $kit_qty;
            $new_component->kit_measure_qty = $kit_measure_qty;
            $new_component->kit_product_cost = $kit_product_cost;
            $new_component->kit_created_by = session('user_name');
            $new_component->created_at = now();
            $new_component->save();

            DB::commit();
            return response()->json([
                'message' => 'Component added to compiled kit successfully',
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to add component to compiled kit: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

    public function remove_component_from_kit(Request $request) {
        try{
            DB::beginTransaction();

            $id_kit_config = $request->input('id_kit_config');

            if($id_kit_config <= 0 || $id_kit_config == null){
                DB::rollBack();
                return response()->json([
                    'message' => 'Invalid component ID',
                    'success' => false,
                    'message_type' => 'error',
                    'message_btn' => 'btn btn-danger',
                ]);
            }

            // KitConfig::where('id_kit_config', $id_kit_config)->delete();
            KitConfig::where('id_kit_config', $id_kit_config)
            ->update([
                'deleted_at' => now(),
                'deleted_by'=>session('user_name')
            ]);

            DB::commit();
            return response()->json([
                'message' => 'Component removed from compiled kit successfully',
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to remove component from compiled kit: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

    public function save_compiled_kit(Request $request) {
        // /return response()->json($request->all());
        try{
            DB::beginTransaction();

            $id_business_products = $request->input('id_business_products');
            $kit_name = $request->input('product');
            $kit_brand_id = $request->input('brand_id');
            $kit_unit_type = $request->input('unit_type');
            $kit_measure_unit = $request->input('measure_unit');
            $kit_commission_type = $request->input('commission_type');
            $kit_commission_value = $request->input('commission');
            $kit_track_inventory = $request->input('track_inventory');
            $kit_tax_percentage =   $request->input('tax_percentage');
            $price =   $request->input('price');

            if($id_business_products == null || $id_business_products <= 0){
                //Create the compiled kit product
                $new_kit_product = new Products();
                $new_kit_product->product = $kit_name;
                $new_kit_product->brand_id = $kit_brand_id;
                $new_kit_product->product_type = 'Compiled Kit';
                $new_kit_product->unit_type = $kit_unit_type;
                $new_kit_product->measure_unit = $kit_measure_unit;
                $new_kit_product->commission_type = $kit_commission_type;
                $new_kit_product->commission = $kit_commission_value;
                $new_kit_product->track_inventory = $kit_track_inventory;
                $new_kit_product->product_sales_tax = $kit_tax_percentage;
                $new_kit_product->price = $price;
                $new_kit_product->created_by = session('user_name');
                $new_kit_product->created_at = now();
                $new_kit_product->business_id = session('business_id');
                $new_kit_product->save();

                //add compiled kit id to the components
                $compiled_kit_id = $new_kit_product->id_business_products;
                $components = json_decode($request->input('components'), true);

                if(!is_array($components) || count($components) == 0){
                    DB::rollBack();
                    return response()->json([
                        'message' => 'No components provided for the compiled kit',
                        'success' => false,
                        'message_type' => 'error',
                        'message_btn' => 'btn btn-danger',
                    ]);
                } 

                foreach($components as $component){
                    $kit_component = new KitConfig();
                    $kit_component->kit_id = $compiled_kit_id;
                    $kit_component->kit_qty = $component['component_quantity'];
                    $kit_component->kit_measure_qty = $component['component_measure_qty'];
                    $kit_component->kit_product_cost = $component['component_product_cost'];
                    $kit_component->business_product_id = $component['component_product_id'];
                    $kit_component->kit_created_by = session('user_name');
                    $kit_component->created_at = now();
                    $kit_component->save();
                }


            } else {
                //Update the compiled kit product
                $existing_kit_product = Products::where('id_business_products', $id_business_products)->first();

                $existing_kit_product->product = $kit_name;
                $existing_kit_product->brand_id = $kit_brand_id;
                $existing_kit_product->unit_type = $kit_unit_type;
                $existing_kit_product->measure_unit = $kit_measure_unit;
                $existing_kit_product->commission_type = $kit_commission_type;
                $existing_kit_product->commission = $kit_commission_value;
                $existing_kit_product->track_inventory = $kit_track_inventory;
                $existing_kit_product->product_sales_tax = $kit_tax_percentage;
                $existing_kit_product->price = $price;
                $existing_kit_product->updated_by = session('user_name');
                $existing_kit_product->updated_at = now();
                $existing_kit_product->save();
            }
            DB::commit();
            return response()->json([
                'message' => 'Compiled kit saved successfully',
                'success' => true,
                'message_type' => 'success',
                'message_btn' => 'btn btn-success'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to save compiled kit: ' . $e->getMessage(),
                'success' => false,
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ]);
        }
    }

    function create_adjustment_note_voucher($business_id,$voucher_amount,$id_adjustment_notes ,$product_id , $batch_unit_amount , $adjustment_qty , $batch_id){
        try{
            DB::beginTransaction();
            if(session('ho_accounts')=='Yes'){
                    $business_id_for_account_head= Business::where('business.ho', 'Yes')->value('id_business');
                } else {
                    $business_id_for_account_head = $business_id;
                }

                $new_voucher = new AccountVouchers();
                $new_voucher->business_id = $business_id;
                $new_voucher->description = 'Cost of Goods Adjusted for goods found in inventory through Adjustment Note ID: ' . $id_adjustment_notes . ' Product ID: ' . $product_id .' with unit price' . $batch_unit_amount. ' Qty: ' . $adjustment_qty;
                $new_voucher->voucher_type = 3; //Journal Voucher
                $new_voucher->voucher_date =  now();
                $new_voucher->created_by = session('user_name');
                $new_voucher->voucher_status = 'Active';
                $new_voucher->voucher_amount = $voucher_amount;
                $new_voucher->cost_center = 1;
                $new_voucher->cost_center_name = 'Front Desk';
                $new_voucher->business_partner=5; //Branch
                $new_voucher->business_partner_id = $business_id;
                $new_voucher->business_partner_name = session('business_name');
                $new_voucher->payment_mode = 'Cash';
                $new_voucher->auto_voucher = 'Yes';            
                $new_voucher->adjustment_id = $id_adjustment_notes;            
                $new_voucher->save();
                $id_voucher = $new_voucher->id_account_vouchers;

                //Get the event mappings for Invoice Creation id_events = 22
                if($adjustment_qty > 0){
                    $event_mappings = AccountEventMapping::where('account_event_id', 22)->where('business_id', $business_id_for_account_head)->get();
                } else {
                    $event_mappings = AccountEventMapping::where('account_event_id', 23)->where('business_id', $business_id_for_account_head)->get();
                }

                $debit_amounts = [];
                $credit_amounts = [];
                $debit_accounts = array();
                $credit_accounts = array();
                foreach ($event_mappings as $mapping) {
                    if($mapping['transaction_type']=='debit'){
                        array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    } else {
                        array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    }

                    if($adjustment_qty > 0){
                        if($mapping['transaction_type']=='debit'){
                            //make debit entry data
                            if($mapping['entity_name']=='inventory'){array_push($debit_amounts, [ 'entity_name' => 'inventory', 'amount' => round((float) $voucher_amount ,2)]);} 

                        } else if($mapping['transaction_type']=='credit'){
                            //make credit entry data
                            if($mapping['entity_name']=='cogd'){array_push($credit_amounts, [ 'entity_name' => 'cogd', 'amount' => round((float) $voucher_amount ,2) ]);}                                        
                        }
                    } else {
                        if($mapping['transaction_type']=='debit'){
                            //make debit entry data
                            if($mapping['entity_name']=='cogd'){array_push($debit_amounts, [ 'entity_name' => 'cogd', 'amount' => round((float) abs($voucher_amount) ,2)]);} 

                        } else if($mapping['transaction_type']=='credit'){
                            //make credit entry data
                            if($mapping['entity_name']=='inventory'){array_push($credit_amounts, [ 'entity_name' => 'inventory', 'amount' => round((float) abs($voucher_amount) ,2) ]);}                                        
                        }
                    }
                }
                //get the sum of debit amounts and credit amounts
                $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
                $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

                if(bccomp($debit_sum, $credit_sum, 2) !== 0){
                    DB::rollBack(); //Rollback the transaction
                    return response()->json([
                        "success" => false,
                        "message" => " debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                        "message_type" => "error",
                        "message_btn" => "btn btn-danger"
                    ]);
                
                } else {
                    
                    //Insert voucher Details
                    foreach($debit_accounts as $debit){
                        $debit_parts = explode("|", $debit);
                        $new_receiving_voucher_details = new AccountVoucherDetail();
                        $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
                        $new_receiving_voucher_details->detail_remarks = 'Adjustment Note ID '. $id_adjustment_notes  .' Batch ID : '. $batch_id;
                        $new_receiving_voucher_details->account_head_id = $debit_parts[0];
                        $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                        $new_receiving_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $debit_amounts));
                        $new_receiving_voucher_details->credit = 0;          
                        //only save if amount is greater than zero
                        if($new_receiving_voucher_details->debit != 0){                        
                            $new_receiving_voucher_details->save();
                        }
                    }
                    foreach($credit_accounts as $credit){
                        $credit_parts = explode("|", $credit);
                        $new_receiving_voucher_details = new AccountVoucherDetail();
                        $new_receiving_voucher_details->account_voucher_id = $new_voucher->id_account_vouchers;
                        $new_receiving_voucher_details->detail_remarks = 'Adjustment Note ID '. $id_adjustment_notes  .' Batch ID : '. $batch_id;
                        $new_receiving_voucher_details->account_head_id = $credit_parts[0];
                        $new_receiving_voucher_details->debit = 0;
                        $entity = $credit_parts[2]; // this is 'paid_cash' etc.
                        $new_receiving_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $credit_amounts));
                        if($new_receiving_voucher_details->credit != 0){           
                            $new_receiving_voucher_details->save();
                        }
                    }
                }
          
              DB::commit();
            return response()->json([
                "success" => true,
                'message' => 'Adjustment note voucher created successfully.',
                'message_type' => 'success',
                'message_button' => 'btn btn-success',
            ]);    
        } catch(\Exception $e){
            DB::rollBack();
            return response()->json([
                'message' => 'Error adding Adjustment note voucher: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
                "success" => false
            ]);
        }        
    }

    public function getOrGetCreateCustomer($phone='', $email='', $name='')
    {
        // if (!empty($email)) {
        //     $customer = Customers::where('customer_email', $email)->first();
        //     if ($customer) {
        //         return $customer;
        //     }
        // }

        if (!empty($phone)) {
            $customer = Customers::where('customer_cell', $phone)->first();
            if ($customer) {
                return $customer;
            }
        }

        $customer = Customers::create([
            'business_id' => session('business_id'),
            'customer_name' => $name ?? ' ',
            'customer_email' => $email ?? ' ',
            'password' => '12345',
            'customer_cell' => $phone ?? ' ',
            'customer_allergies' => ' ',
            'customer_alert' => ' ',
            'customer_type' => ' ',
            'profession' => ' ',
            'loyalty_points' => '0.00',
            'created_on' => NOW(),
            'customer_gender' => ' ',
            'customer_advance' => '0.00',
            'fcm_id' => ' ',
            'pin_code' => ' ',
            'is_verified' => '0',
            'created_by' => session('user_name'),
            'allowed_balance' => '50000.00',
        ]);

        return $customer;
    }

    function get_new_batch_number(Request $request)
    {
        try {
            $store_id   = $request->store_id;
            $product_id = $request->product_id;

            if (!$store_id || !$product_id) {
                return response()->json([
                    'message' => 'Store ID or Product ID missing',
                    'message_type' => 'error',
                    'message_btn' => 'btn btn-danger',
                ], 422);
            }

            $maxBatch = ProductBatch::where('store_id', $store_id)
                ->where('product_id', $product_id)
                ->select(DB::raw('MAX(CAST(batch_number AS UNSIGNED)) as max_batch'))
                ->value('max_batch');

            $new_batch_number = 1;

            if (!empty($maxBatch)) {
                $new_batch_number = $maxBatch + 1;
            }

            // 4 digit format
            $new_batch_number = str_pad($new_batch_number, 4, '0', STR_PAD_LEFT);

            return response()->json([
                'batch_number' => $new_batch_number,
                'message' => 'New batch number generated',
                'message_type' => 'success',
                'message_btn' => 'btn btn-success',
            ]);

        } catch (\Throwable $e) {

        
            return response()->json([
                'message' => 'Something went wrong while generating batch number',
                'message_type' => 'error',
                'message_btn' => 'btn btn-danger',
            ], 500);
        }
    }

}