<?php

namespace App\Http\Controllers;

use App\Models\AccountVouchers;
use App\Models\Business;
use App\Models\BusinessType;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderDetails;
use App\Models\Supplier;
use App\Models\Brands;
use App\Models\GoodsReceivedNote;
use App\Models\GoodsReceivedNoteDetails;
use App\Models\Stores;
use App\Models\ProductBatch;
use App\Models\AccountEventMapping;
use App\Models\AccountVoucherDetail;
use App\Models\CostCenter;
use App\Models\ReturnNotes;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\View\View;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    
    public function purchase_orders(Request $request)
    {
        $business_type_id = $request->input('business_type_id');

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        $business_id = session('business_id');
        if(!empty($request->business_id)){
            $business_id = $request->business_id;
        } 

        if(session('ho')=="Yes"){
            $branches = Business::select('id_business', 'business_name')
            ->where('common_products', 'Yes')
            ->get();

            $business_types = BusinessType::select('id_business_type', 'business_type')
            ->get();

            $suppliers = Supplier::select('business_supplier.*')
            ->join('business', 'business_supplier.business_id', '=', 'business.id_business')
            ->where('common_products', "Yes")->get();

            $brands = Brands::select('id_business_brands', 'business_brand_name')
            ->whereIn('business_id', function ($query) {
                $query->select('id_business')
                    ->from('business')
                    ->where('common_products', 'Yes');
            })->get();

        } else {
            $branches = Business::select('id_business', 'business_name')
            ->where('id_business', $business_id)
            ->get();
            $business_types = [];
            if(session('common_products')=="Yes"){
                $suppliers = Supplier::select('business_supplier.*')
                ->whereIn('business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('common_products', 'Yes');
                })->get();

                $brands = Brands::select('id_business_brands', 'business_brand_name')
                ->whereIn('business_id', function ($query) {
                    $query->select('id_business')
                        ->from('business')
                        ->where('common_products', 'Yes');
                })->get();
            } else {
                $suppliers = Supplier::where('business_id', $business_id)->get();
                $brands = Brands::where('business_id', $business_id)->get();
            }
            
        }

        return view('products.purchase.purchase_orders', compact('branches', 'business_type_id', 'start_date', 'end_date', 'business_id', 'business_types', 'suppliers', 'brands'));

    }

    public function purchase_orders_data(Request $request)
    {

        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        //get accounts payable head from account event mapping
        $accounts_payable_head = DB::table('account_event_mapping')
            ->where('account_event_id', 24)->where('entity_name', 'payable')
            ->value('account_head_id');

        // Define subquery properly with alias
        $subQuery = DB::raw("
            (
                SELECT 
                    account_vouchers.purchase_order_id, 
                    SUM(IFNULL(debit, 0)) AS debit, 
                    SUM(IFNULL(credit, 0)) AS credit
                FROM account_vouchers
                JOIN account_voucher_detail 
                    ON id_account_vouchers = account_voucher_id
                WHERE account_voucher_detail.account_head_id = {$accounts_payable_head}
                AND account_vouchers.voucher_status = 'Active'
                AND COALESCE(account_vouchers.purchase_order_id, '') != ''
                GROUP BY account_vouchers.purchase_order_id
            ) AS av
        ");

        $query = PurchaseOrder::select(
                'purchase_order.idpurchase_order',
                'purchase_order.purchase_order_number',
                'purchase_order.business_id',
                'purchase_order.supplier_id',
                'purchase_order.supplier_name',
                'purchase_order.status',
                'business.business_name',
                'purchase_order.created_by',
                DB::raw('ROUND(SUM(purchase_order_details.product_purchase_price * product_qty) - SUM(purchase_order_details.unit_discount * product_qty), 2) AS total_value'),
                DB::raw('ROUND(SUM(purchase_order_details.product_qty), 2) AS total_qty'),
                DB::raw('ROUND(SUM(purchase_order_details.product_received_qty), 2) AS total_received'),
                DB::raw('ROUND(SUM(purchase_order_details.unit_discount * product_qty), 2) AS total_discount'),
                DB::RAW('date_format(purchase_order.order_date, "%d-%m-%Y") as order_date'),
                DB::raw('ROUND(IFNULL(av.credit, 0) - IFNULL(av.debit, 0), 2) AS amount_due')
            )
            ->join('purchase_order_details', 'purchase_order.idpurchase_order', '=', 'purchase_order_details.purchase_order_id')
            ->join('business', 'business.id_business', '=', 'purchase_order.business_id')
            ->leftJoin($subQuery, 'purchase_order.idpurchase_order', '=', 'av.purchase_order_id')
            ->whereDate('purchase_order.order_date', '>=', $start_date)
            ->whereDate('purchase_order.order_date', '<=', $end_date)
            
            ->groupBy('purchase_order.idpurchase_order');

            // Apply business filter if provided
        if ($request->has('business_id') && !empty($request->input('business_id'))) {
            $business_id = $request->input('business_id');
            $query->where('purchase_order.business_id', $business_id);
        }

        return DataTables::eloquent($query)
            ->filterColumn('business_name', function ($query, $keyword) {
                $query->where('business.business_name', 'like', "%{$keyword}%");
            })
            ->filterColumn('purchase_order_number', function ($query, $keyword) {
                $query->where('purchase_order.purchase_order_number', 'like', "%{$keyword}%");
            })
            ->filterColumn('total_value', function ($query, $keyword) {
                $query->havingRaw('ROUND(SUM(purchase_order_details.product_purchase_price * product_qty), 2) like ?', ["%{$keyword}%"]);
            })
            ->filterColumn('total_received', function ($query, $keyword) {
                $query->havingRaw('ROUND(SUM(purchase_order_details.product_received_qty), 2) like ?', ["%{$keyword}%"]);
            })
            ->make(true);

    }   

    public function open_purchase_details($idpurchase_order)
    {
        if(empty($idpurchase_order)){
            return response()->json([ 
                'message' => 'Purchase Order ID is required',
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }


        $purchase_order = PurchaseOrder::select('*', DB::raw('DATE_FORMAT(order_date, "%Y-%m-%d") AS formatted_order_date'))
        ->leftJoin('purchase_order_details', 'purchase_order.idpurchase_order', '=', 'purchase_order_details.purchase_order_id')
        ->where('idpurchase_order', $idpurchase_order)->get();

        if(empty($purchase_order)){
            return response()->json([ 
                'message' => 'Purchase Order not found',
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }

       //check if active account vouchers exist for this purchase order
       $active_voucher_count = AccountVouchers::where('purchase_order_id', $idpurchase_order)
       ->where('voucher_status', 'Active');

        if($active_voucher_count->count() > 0){
                return response()->json([ 
                    'message' => 'Purchase Order details can no longer be edited. Active account vouchers exist for this Purchase Order.',
                    'message_type' => 'info',
                    'message_btn'=> "btn btn-info",
                    'items' => []
                ]);
        }

        return response()->json([ 
            'message' => 'Success',
            'message_type' => 'success',
            'message_btn'=> "btn btn-success",
            'items' => $purchase_order,
        ]);

    }

    public function save_purchase_order(Request $request)
    {
        try{
            $id_purchase_order = $request->input('id_purchase_order');
            if(empty($id_purchase_order)){
                //Add new purchase order
                $new_purchase_order = new PurchaseOrder();
                // Set attributes from request
                $new_purchase_order->business_id = session('business_id');
                $new_purchase_order->supplier_id = $request->input('supplier_id');
                $new_purchase_order->supplier_name = $request->input('supplier_name');
                $new_purchase_order->total_purchase_pirce = $request->input('total_purchase_price');
                $new_purchase_order->status = 'Pending';
                $new_purchase_order->product_total_qty = $request->input('product_total_qty');
                $new_purchase_order->order_date = $request->input('order_date');
                $new_purchase_order->created_by = session('user_name');
                $new_purchase_order->created_on = now();
                
                $new_purchase_order->save();

                //update the purchase order number
                $new_purchase_order->purchase_order_number = 'PO-' . date('Y-m').'-'. $new_purchase_order->idpurchase_order;
                $new_purchase_order->save();


                // Save purchase order details
                $items = $request->input('items', []);
                foreach ($items as $item) {

                  $business_products =  DB::table('business_products')
                    ->where('id_business_products',$item['product_id'])
                    ->first();

                    $new_purchase_order->details()->create([
                        'purchase_order_id' => $new_purchase_order->idpurchase_order,
                        'brand_id' => $item['brand_id'],
                        'product_id' => $item['product_id'],
                        'brand_name' => $item['brand_name'],
                        'product_name' => $business_products->product,
                        'product_purchase_price' => $item['product_purchase_price'],                    
                        'product_qty' => $item['product_qty'],
                        'unit_discount' => $item['unit_discount'],                                 
                    ]);
                }

            } else {
                //Update existing purchase order
                $existing_purchase_order = PurchaseOrder::where('idpurchase_order', $id_purchase_order)->first();
                if(!$existing_purchase_order){
                    return response()->json([ 
                        'message' => 'Purchase Order not found',
                        'message_type' => 'error',
                        'message_btn'=> "btn btn-danger",
                    ]);
                }
                // Update attributes from request
                $existing_purchase_order->supplier_id = $request->input('supplier_id');
                $existing_purchase_order->supplier_name = $request->input('supplier_name');
                $existing_purchase_order->total_purchase_pirce = $request->input('total_purchase_price');
                $existing_purchase_order->product_total_qty = $request->input('total_qty');
                $existing_purchase_order->order_date = $request->input('order_date');
                $existing_purchase_order->save();
                //if idpurchase_order_details is 0, add new detail, else update existing detail
                $items = $request->input('items', []);
                foreach ($items as $item) {
                    if($item['idpurchase_order_details'] == 0){
                        //add new detail
                        $existing_purchase_order->details()->create([
                            'purchase_order_id' => $existing_purchase_order->idpurchase_order,
                            'brand_id' => $item['brand_id'],
                            'product_id' => $item['product_id'],
                            'brand_name' => $item['brand_name'],
                            'product_name' => $item['product_name'],
                            'product_purchase_price' => $item['product_purchase_price'],
                            'unit_discount' => $item['unit_discount'],                    
                            'product_qty' => $item['product_qty'],                                        
                        ]);
                    } else {
                        //update existing detail
                        $existing_detail = $existing_purchase_order->details()->find($item['idpurchase_order_details']);
                        if($existing_detail){
                            $existing_detail->brand_id = $item['brand_id'];
                            $existing_detail->product_id = $item['product_id'];
                            $existing_detail->brand_name = $item['brand_name'];
                            $existing_detail->product_name = $item['product_name'];
                            $existing_detail->product_purchase_price = $item['product_purchase_price'];                    
                            $existing_detail->product_qty = $item['product_qty'];
                            $existing_detail->unit_discount = $item['unit_discount'];                                        
                            $existing_detail->save();
                        }
                    }
                }
            }
            return response()->json([ 
                'message' => 'Purchase Order saved successfully',
                'message_type' => 'success',
                'message_btn'=> "btn btn-success",
            ]);
        } catch (\Exception $e) {
            return response()->json([ 
                'message' => 'Error saving Purchase Order: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }
            
    }

    public function open_purchase_order($idpurchase_order)
    {
        
        $purchase_order = PurchaseOrder::where('idpurchase_order', $idpurchase_order)
        ->join('business_supplier', 'purchase_order.supplier_id', '=', 'business_supplier.id_business_supplier')
        ->join('business', 'purchase_order.business_id', '=', 'business.id_business')
        ->select('purchase_order.*', DB::raw('date_format(purchase_order.order_date, "%d-%m-%Y") as formatted_order_date'),
        'business_supplier.ho_address', 'business_supplier.contact_person', 'business_supplier.contact_number', 'business_supplier.email',
        'business.business_name', 'business.business_logo'
        )
        ->first();
        if(!$purchase_order){
            return Redirect::back()->with('error', 'Purchase Order not found');
        }

        $sub_query = DB::raw("
            (
                SELECT 
                goods_received_note.purchase_order_id,  grn_details.grn_product_id, 
                round(sum(ifnull(grn_qty_received,0)),2) total_received, 
                round(sum(ifnull(grn_unit_price * grn_qty_received,0)),2) total_cost
                FROM goods_received_note
                JOIN grn_details on goods_received_note.grn_id = grn_details.grn_id
                WHERE goods_received_note.purchase_order_id = {$idpurchase_order}
                GROUP BY goods_received_note.purchase_order_id, grn_product_id
            ) AS av");

        $sub_query2 = DB::raw("
            (
               SELECT 
                return_notes.grn_id,  return_notes.product_id, goods_received_note.purchase_order_id,
                round(sum(ifnull(return_notes.return_qty,0)),2) total_returned
                FROM return_notes                
                JOIN goods_received_note on goods_received_note.grn_id = return_notes.grn_id
                WHERE goods_received_note.purchase_order_id = {$idpurchase_order}
                AND return_notes.return_status = 'Active'
                GROUP BY goods_received_note.grn_id, return_notes.product_id
            ) AS rp");

        $purchase_order_details = PurchaseOrderDetails::where('purchase_order_details.purchase_order_id', $idpurchase_order)
        ->leftJoin($sub_query, function($join) {
            $join->on('purchase_order_details.purchase_order_id', '=', 'av.purchase_order_id')
                 ->on('purchase_order_details.product_id', '=', 'av.grn_product_id');
        })
        ->leftJoin($sub_query2, function($join) {
            $join->on('purchase_order_details.purchase_order_id', '=', 'rp.purchase_order_id')
                 ->on('purchase_order_details.product_id', '=', 'rp.product_id');
        })
        ->groupBy('purchase_order_details.idpurchase_order_details')
        ->get();

        $goods_received_notes = GoodsReceivedNote::select('goods_received_note.*', 
        DB::raw('GROUP_CONCAT(concat(grn_details.grn_product_id, " ", grn_details.grn_product_name, "| Batch:", grn_batch_id) SEPARATOR "<br>") as products_received'), 
        DB::raw('date_format(goods_received_note.grn_created_date, "%d-%m-%Y") as formatted_grn_date'),
        DB::raw('round(sum(ifnull(grn_details.grn_qty_received,0)),2) as total_received'),
        DB::raw('round(sum(ifnull(grn_details.grn_unit_price * grn_details.grn_qty_received,0)),2) as total_cost'),
        DB::raw('round(sum(ifnull(rp.total_returned,0)),2) as total_returned')
        )
        ->join('grn_details', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
        // ->leftJoin($sub_query2, function($join) {
        //     $join->on('goods_received_note.grn_id', '=', 'rp.grn_id' and 'grn_details.grn_product_id', '=', 'rp.product_id');
        // })
        ->leftJoin($sub_query2, function($join) {
            $join->on('goods_received_note.grn_id', '=', 'rp.grn_id')
                 ->on('grn_details.grn_product_id', '=', 'rp.product_id');
        })
        
        ->where('goods_received_note.purchase_order_id', $idpurchase_order)
        ->groupBy('goods_received_note.grn_id')
        ->get();

        //get all account vouchers with grn_id linked to this purchase order
        //get accounts payable from account event mapping 20
        
        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $purchase_order->business_id;
        }

        $accounts_payable_event_id = 20;
        $accounts_payable_head = AccountEventMapping::where('account_event_id', 20)->where('business_id', $common_business_id)->where('entity_name', 'payables')->first()->account_head_id;
        $account_heads_name ='';
        if($accounts_payable_head){
            $account_heads_name = DB::table('account_heads')->where('id_account_heads',$accounts_payable_head)->first()->account_head ?? '';
        }

        $account_vouchers = AccountVoucherDetail::select('account_vouchers.*', 
        DB::raw('date_format(account_vouchers.voucher_date, "%d-%m-%Y") as formatted_voucher_date'),
            DB::raw('round(sum(ifnull(account_voucher_detail.debit,0)),2) as total_debit'),
            DB::raw('round(sum(ifnull(account_voucher_detail.credit,0)),2) as total_credit')
        )
            ->join('account_vouchers', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
            //->where('account_vouchers.grn_id', '!=', null)
            ->where('account_vouchers.purchase_order_id', $idpurchase_order)
            ->where('account_voucher_detail.account_head_id', $accounts_payable_head)
            ->where('account_vouchers.voucher_status', 'Active')
            ->groupBy('account_vouchers.id_account_vouchers', 'account_vouchers.grn_id')
            ->get();

        $business_id = $purchase_order->business_id;

        return view('products.purchase.purchase_order', compact('idpurchase_order', 'purchase_order_details', 'purchase_order', 'goods_received_notes', 'business_id', 'account_vouchers','account_heads_name'));
    }

    public function new_grn($idpurchase_order)
    {
        $purchase_order = PurchaseOrder::select('purchase_order.*', 'business_supplier.*', 'business.business_name', 'business.business_logo')
        ->where('idpurchase_order', $idpurchase_order)
        ->join('business_supplier', 'business_supplier.id_business_supplier', '=', 'purchase_order.supplier_id')
        ->join('business', 'business.id_business', '=', 'purchase_order.business_id')
        ->first();
        if(!$purchase_order){
            return Redirect::back()->with('error', 'Purchase Order not found');
        }


         $sub_query = DB::raw("
            (
                SELECT 
                goods_received_note.purchase_order_id,  grn_details.grn_product_id, 
                round(sum(ifnull(grn_qty_received,0)),2) total_received, 
                round(sum(ifnull(grn_unit_price * grn_qty_received,0)),2) total_cost
                FROM goods_received_note
                JOIN grn_details on goods_received_note.grn_id = grn_details.grn_id
                WHERE goods_received_note.purchase_order_id = {$idpurchase_order}
                GROUP BY goods_received_note.purchase_order_id, grn_product_id
            ) AS av");

        $sub_query2 = DB::raw("
            (
               SELECT 
                return_notes.grn_id,  return_notes.product_id, goods_received_note.purchase_order_id,
                round(sum(ifnull(return_notes.return_qty,0)),2) total_returned
                FROM return_notes                
                JOIN goods_received_note on goods_received_note.grn_id = return_notes.grn_id
                WHERE goods_received_note.purchase_order_id = {$idpurchase_order}
                AND return_notes.return_status = 'Active'
                GROUP BY goods_received_note.grn_id, return_notes.product_id
            ) AS rp");

        $purchase_order_items = PurchaseOrderDetails::select('purchase_order_details.*',
        DB::raw('IFNULL(av.total_received, 0) AS total_received'),
        DB::raw('IFNULL(rp.total_returned, 0) AS total_returned')
        )
        ->where('purchase_order_details.purchase_order_id', $idpurchase_order)
        ->leftJoin($sub_query, function($join) {
            $join->on('purchase_order_details.purchase_order_id', '=', 'av.purchase_order_id')
                 ->on('purchase_order_details.product_id', '=', 'av.grn_product_id');
        })
        ->leftJoin($sub_query2, function($join) {
            $join->on('purchase_order_details.purchase_order_id', '=', 'rp.purchase_order_id')
                 ->on('purchase_order_details.product_id', '=', 'rp.product_id');
        })
        ->get();

        $stores = Stores::where('business_id', $purchase_order->business_id)->get();

        return view('products.purchase.new_grn', compact('idpurchase_order',  'purchase_order', 'purchase_order_items', 'stores'));
    }   

    public function save_grn(Request $request)
    {
        try{
            $id_purchase_order = $request->input('idpurchase_order');
            if(empty($id_purchase_order)){
                return response()->json([ 
                    'message' => 'Purchase Order ID is required',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }

            $purchase_order = PurchaseOrder::where('idpurchase_order', $id_purchase_order)->first();
            if(!$purchase_order){
                return response()->json([ 
                    'message' => 'Purchase Order not found',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }

            //Create new GRN
            $new_grn = new GoodsReceivedNote();
            $new_grn->purchase_order_id = $id_purchase_order;
            $new_grn->business_id = $purchase_order->business_id;
            $new_grn->grn_vendor_name = $purchase_order->supplier_name;
            $new_grn->grn_created_date = $request->input('grn_date');
            $new_grn->grn_comment = $request->input('grn_comment');
            $new_grn->created_by = session('user_name');
            $new_grn->grn_created_date = now();
            $new_grn->save();
            //update the GRN number
            $new_grn->grn_number = 'GRN-' . date('Y-m').'-'. $new_grn->grn_id;
            $new_grn->save();

            $store_id = $request->input('store_id');
            $total_inventory_value = 0;
            //Save GRN details
            $items = json_decode($request->input('items', '[]'), true);
            
            foreach ($items as $item) {

                
                $batches = ProductBatch::where('store_id', $store_id)
                ->where('product_id', $item['product_id'])
                ->select(DB::raw('Max(batch_number) as batch_number'))
                ->orderBy('created_at', 'desc')
                ->first();

                $new_batch_number = '0001';
                if($batches && !empty($batches->batch_number)){
                    $new_batch_number = $batches->batch_number + 1;
                }
                $new_batch_number = str_pad($new_batch_number, 4, '0', STR_PAD_LEFT);
                //create new batch in the store 
                $new_batch = new ProductBatch();                
                $new_batch->store_id = $store_id;                    
                $new_batch->product_id = $item['product_id'];
                $new_batch->batch_number = $new_batch_number;                
                $new_batch->batch_date = now();
                $new_batch->expiry_date = now()->addYears(1);
                $new_batch->batch_qty = $item['receiving_quantity'];
                $new_batch->batch_amount = $item['unit_price'] - $item['unit_discount'];
                $new_batch->batch_hidden = 'No';
                $new_batch->save();

                $new_batch_id = $new_batch->id_batch;
                $new_batch_number = $new_batch->batch_number;

                //create new grn details
                $new_grn_details = new GoodsReceivedNoteDetails();
                $new_grn_details->grn_id = $new_grn->grn_id;
                $new_grn_details->grn_brand_id = $item['brand_id'];
                $new_grn_details->grn_product_id = $item['product_id'];
                $new_grn_details->grn_brand_name = $item['brand_name'];
                $new_grn_details->grn_product_name = $item['product_name'];
                $new_grn_details->grn_sku = $item['product_sku'];
                $new_grn_details->grn_unit_price = $item['unit_price'];
                $new_grn_details->grn_unit_discount = $item['unit_discount'];
                $new_grn_details->grn_qty_received = $item['receiving_quantity'];
                $new_grn_details->grn_batch_id = $new_batch_id;
                $new_grn_details->grn_batch_number = $new_batch_number;                
                $new_grn_details->save();
                $total_inventory_value += ($item['unit_price'] - $item['unit_discount']) * $item['receiving_quantity'];
            }

            //update purchase order status to 'Partially Received' or 'Received'
            $total_ordered = PurchaseOrderDetails::where('purchase_order_id', $id_purchase_order)
            ->sum('product_qty');
            $total_received = GoodsReceivedNoteDetails::whereIn('grn_id', function($query) use ($id_purchase_order) {
                $query->select('grn_id')
                      ->from('goods_received_note')
                      ->where('purchase_order_id', $id_purchase_order);
            })->sum('grn_qty_received');

            if($total_received >= $total_ordered){
                $purchase_order->status = 'Received';
            } else {
                $purchase_order->status = 'Partial Receive';
            }
            $purchase_order->save();

            //Create account voucher for inventory increase and accounts payable
            $business_id = $purchase_order->business_id;
            
            $account_voucher = new AccountVouchers();
            $account_voucher->voucher_date = $new_grn->grn_created_date;            
            $account_voucher->voucher_type = 3; //Receive Voucher
            $account_voucher->voucher_status = 'Active';
            $account_voucher->created_by = session('user_name');
            $account_voucher->created_on = date('Y-m-d H:i:s');
            $account_voucher->purchase_order_id = $id_purchase_order;                    
            $account_voucher->id_account_head = 24; //1 = Accounts Receivable
            $account_voucher->voucher_date = $new_grn->grn_created_date;
            $account_voucher->description = 'Inventory received for Purchase Order ' . $purchase_order->purchase_order_number . ' via GRN ' . $new_grn->grn_number;
            $account_voucher->business_id = $business_id;
            $account_voucher->voucher_amount = $total_inventory_value;
            $account_voucher->cost_center = 2;
            $account_voucher->cost_center_name = 'Back Office';
            $account_voucher->business_partner = '2'; //Supplier
            $account_voucher->business_partner_id = $purchase_order->supplier_id;
            $account_voucher->business_partner_name = $purchase_order->supplier_name;                
            $account_voucher->grn_id = $new_grn->grn_id;            
            $account_voucher->payment_mode = 'Credit';
            $account_voucher->auto_voucher = 'Yes';
            $account_voucher->save();
            $new_voucher_id = $account_voucher->id_account_vouchers;
            
            //get account heads from account event mapping 20
            if(session('ho_accounts')=='Yes'){
                $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $common_business_id = $purchase_order->business_id;
            }
            $event_mappings = AccountEventMapping::where('account_event_id', 20)->where('business_id', $common_business_id)->get();

            $debit_amounts = [];
            $credit_amounts = [];
            $debit_accounts = array();
            $credit_accounts = array();
            foreach($event_mappings as $mapping){
                
                if($mapping['transaction_type']=='debit'){
                    array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                } else {
                    array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                }
                
                if($mapping['transaction_type']=='debit'){
                    if($mapping['entity_name']=='inventory'){array_push($debit_amounts, [ 'entity_name' => 'inventory', 'amount' => $total_inventory_value ]);}
                } else if($mapping['transaction_type']=='credit'){
                    if($mapping['entity_name']=='payables'){array_push($credit_amounts, [ 'entity_name' => 'payables', 'amount' => $total_inventory_value ]);}
                }
            }

            //get the sum of debit amounts and credit amounts
            $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
            $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

            //create account voucher details
            if($debit_sum == null || $credit_sum == null || $debit_sum != $credit_sum){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "danger"
                ]);
            } else if($debit_sum == 0 || $credit_sum == 0){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit or Credit amounts are zero. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "danger"
                ]);
            }else{   
                //Insert voucher Details
                foreach($debit_accounts as $debit){
                    $debit_parts = explode("|", $debit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->account_head_id = $debit_parts[0];
                    $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                    $new_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $debit_amounts));
                    $new_voucher_details->credit = 0;    
                    if($new_voucher_details->debit != 0){                                            
                        $new_voucher_details->save();
                    }
                }
                foreach($credit_accounts as $credit){
                    $credit_parts = explode("|", $credit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->account_head_id = $credit_parts[0];
                    $entity = $credit_parts[2]; // this is 'payables' etc.
                    $new_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $credit_amounts));
                    $new_voucher_details->debit = 0;    
                    if($new_voucher_details->credit != 0){                                            
                        $new_voucher_details->save();
                    }
                }
            }
            return response()->json([ 
                'message' => 'Goods Received Note created successfully',
                'message_type' => 'success',
                'message_btn'=> "btn btn-success",
                'idgrn' => $new_grn->grn_id,
            ]);

        } catch (\Exception $e) {
            return response()->json([ 
                'message' => 'Error creating GRN: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }
    }

    public function print_grn($idgrn)
    {
        $grn = GoodsReceivedNote::where('grn_id', $idgrn)
        ->join('purchase_order', 'goods_received_note.purchase_order_id', '=', 'purchase_order.idpurchase_order')
        ->join('business_supplier', 'purchase_order.supplier_id', '=', 'business_supplier.id_business_supplier')
        ->join('business', 'goods_received_note.business_id', '=', 'business.id_business')
        ->select('goods_received_note.*', 'purchase_order.purchase_order_number', 'business_supplier.contact_person', 
        'business_supplier.contact_number', 'business_supplier.email', 'business.business_name', 'business.business_logo',
        DB::raw('date_format(goods_received_note.grn_created_date, "%d-%m-%Y") as formatted_grn_date')
        )
        ->first();

        $grn_items = GoodsReceivedNoteDetails::where('grn_id', $idgrn)->get();

        return view('products.purchase.print_grn', compact('grn', 'grn_items'));
    }

    public function purchase_order_print($idpurchase_order)
    {
        $purchase_order = PurchaseOrder::where('idpurchase_order', $idpurchase_order)
        ->join('business_supplier', 'purchase_order.supplier_id', '=', 'business_supplier.id_business_supplier')
        ->join('business', 'purchase_order.business_id', '=', 'business.id_business')
        ->select('purchase_order.*', DB::raw('date_format(purchase_order.order_date, "%d-%m-%Y") as formatted_order_date'),
        'business_supplier.ho_address', 'business_supplier.contact_person', 'business_supplier.contact_number', 'business_supplier.email',
        'business.business_name', 'business.business_logo'
        )
        ->first();
        if(!$purchase_order){
            return Redirect::back()->with('error', 'Purchase Order not found');
        }

        $purchase_order_details = PurchaseOrderDetails::where('purchase_order_details.purchase_order_id', $idpurchase_order)
        ->get();

        return view('products.purchase.purchase_order_print', compact('idpurchase_order', 'purchase_order_details', 'purchase_order'));
    }

    public function new_supplier_payment($purchase_order_id)
    {
        $purchase_order = PurchaseOrder::where('idpurchase_order', $purchase_order_id)
        ->join('business_supplier', 'purchase_order.supplier_id', '=', 'business_supplier.id_business_supplier')
        ->join('business', 'purchase_order.business_id', '=', 'business.id_business')
        ->select('purchase_order.*', DB::raw('date_format(purchase_order.order_date, "%d-%m-%Y") as formatted_order_date'),
        'business_supplier.ho_address', 'business_supplier.contact_person', 'business_supplier.contact_number', 'business_supplier.email',
        'business.business_name', 'business.business_logo'
        )
        ->first();
        
        if(!$purchase_order){
            return Redirect::back()->with('error', 'Purchase Order not found');
        }

        //get the remaining amount to be paid for this purchase order using account vouchers
        $common_business_id = $purchase_order->business_id;
        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $purchase_order->business_id;
        }
        
        $remaining_amount = AccountVouchers::select(
            DB::raw('ROUND(SUM(account_voucher_detail.credit) - SUM(account_voucher_detail.debit), 2) as remaining_amount')
        )
        ->where('purchase_order_id', $purchase_order_id)
        ->join('account_voucher_detail', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
        ->where('voucher_status', 'Active')
        ->where('account_voucher_detail.account_head_id', function($query) use ($common_business_id) {
            //get accounts payable head from account event mapping 20
            $sub_query = AccountEventMapping::where('account_event_id', 20)
            ->where('business_id', $common_business_id)
            ->where('entity_name', 'payables')
            ->select('account_head_id')
            ->limit(1);
            $query->fromSub($sub_query, 'a')->select('account_head_id');
        })
        ->groupBy('purchase_order_id')
        ->first();
       
        $cost_centers = CostCenter::all();

        return view('products.purchase.new_supplier_payment', compact('purchase_order_id', 'purchase_order', 'remaining_amount', 'cost_centers'));
    }

    public function save_supplier_payment(Request $request){

        try {
            DB::beginTransaction(); //Start transaction

            $purchase_order = PurchaseOrder::where('idpurchase_order', $request->input('purchase_order_id'))->first();
            if(!$purchase_order){
                return response()->json([ 
                    'message' => 'Purchase Order not found',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }
        
            $purchase_order_id = $request->input('purchase_order_id');
            //data posted supplier_name=Zareen&payment_date=2025-11-16&payment_mode=Cash&account_head_id=2&instrument_number=123&discount=0.00&additional_charge=0.00&sale_tax_percentage=0.00&sale_tax=0.00&withholding_tax_percentage=0.00&withholding_tax=0.00&notes=Payment%20against%20PO-2025-11-53%20to%20Zareen.&paying_now=7000.00&purchase_order_id=53
            $supplier_name = $request->input('supplier_name');
            $payment_date = $request->input('payment_date');
            $payment_mode = $request->input('payment_mode');
            $account_head_id = $request->input('account_head_id');
            $instrument_number = $request->input('instrument_number');
            $description = $request->input('notes');
            $total_amount = $request->input('total_payment_amount');
            $paying_now = $request->input('paying_now');
            $sale_tax = $request->input('sale_tax');;
            $withholding_tax = $request->input('withholding_tax');
            $additional_charge = $request->input('additional_charge');
            $discount = $request->input('discount');
            $business_id = $request->input('business_id');
            $payable_adjusted = $paying_now - $sale_tax - $additional_charge + $discount + $withholding_tax;

            //Create account voucher for supplier payment
            $account_voucher = new AccountVouchers();
            $account_voucher->voucher_date = $payment_date;            
            $account_voucher->voucher_type = 1; //Receive Voucher
            $account_voucher->voucher_status = 'Active';
            $account_voucher->created_by = session('user_name');
            $account_voucher->created_on = date('Y-m-d H:i:s');
            $account_voucher->purchase_order_id = $purchase_order_id;
            $account_voucher->sale_type = 'sale';        
            $account_voucher->id_account_head = 24; //1 = Accounts Receivable
            $account_voucher->description = $description;
            $account_voucher->business_id = $business_id;
            $account_voucher->voucher_amount = $paying_now;
            $account_voucher->cost_center = 1;
            $account_voucher->cost_center_name = 'Back Office';
            $account_voucher->business_partner = '2'; //Supplier
            $account_voucher->business_partner_id = $purchase_order->supplier_id;
            $account_voucher->business_partner_name = $purchase_order->supplier_name;          
            $account_voucher->payment_mode = $payment_mode;
            $account_voucher->auto_voucher = 'Yes';
            $account_voucher->save();
            $new_voucher_id = $account_voucher->id_account_vouchers;

            //Get the event mappings voucher Creation id_events = 24
            if(session('ho_accounts')=='Yes'){
                $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
            } else {
                $common_business_id = $purchase_order->business_id;
            }
            $event_mappings = AccountEventMapping::where('account_event_id', 24)->where('business_id', $common_business_id)->get();
            $debit_amounts = [];
            $credit_amounts = [];
            $debit_accounts = array();
            $credit_accounts = array();
            foreach($event_mappings as $mapping){
                
                if($mapping['transaction_type']=='debit'){
                    array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                } else {
                    array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                }
                
                if($mapping['transaction_type']=='debit'){
                    if($mapping['entity_name']=='payable'){array_push($debit_amounts, [ 'entity_name' => 'payable', 'amount' => $payable_adjusted ]);}
                    if($mapping['entity_name']=='purchase_salestax'){array_push($debit_amounts, [ 'entity_name' => 'purchase_salestax', 'amount' => $sale_tax ]);}
                    if($mapping['entity_name']=='purchase_charges'){array_push($debit_amounts, [ 'entity_name' => 'purchase_charges', 'amount' => $additional_charge ]);}
                    
                } else if($mapping['transaction_type']=='credit'){
                    if($mapping['entity_name']=='purchase_discount'){array_push($credit_amounts, [ 'entity_name' => 'purchase_discount', 'amount' => $discount ]);}
                    if($mapping['entity_name']=='purchase_withholdingtax'){array_push($credit_amounts, [ 'entity_name' => 'purchase_withholdingtax', 'amount' => $withholding_tax ]);}
                    
                }
            }
            
            //get the sum of debit amounts and credit amounts
            $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
            $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;
            //add the paying now amount to credit sum
            $credit_sum += $paying_now;

            if($debit_sum == null || $credit_sum == null || $debit_sum != $credit_sum){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "danger"
                ]);
            } else if($debit_sum == 0 || $credit_sum == 0){
                DB::rollBack(); //Rollback the transaction
                return response()->json([
                    "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit or Credit amounts are zero. Please contact system administrator.",
                    "message_type" => "error",
                    "message_btn" => "danger"
                ]);
            }else{                    
                //Insert voucher Details
                foreach($debit_accounts as $debit){
                    $debit_parts = explode("|", $debit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->account_head_id = $debit_parts[0];
                    $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                    $new_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $debit_amounts));
                    $new_voucher_details->credit = 0;    
                    if($new_voucher_details->debit != 0){                                            
                        $new_voucher_details->save();
                    }
                }
                foreach($credit_accounts as $credit){
                    $credit_parts = explode("|", $credit);
                    $new_voucher_details = new AccountVoucherDetail();
                    $new_voucher_details->account_voucher_id = $new_voucher_id;
                    $new_voucher_details->account_head_id = $credit_parts[0];
                    $new_voucher_details->debit = 0;
                    $entity = $credit_parts[2]; // this is 'paid_cash' etc.
                    $new_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                        return $item['entity_name'] === $entity ? $item['amount'] : 0;
                    }, $credit_amounts));
                    if($new_voucher_details->credit != 0){           
                        $new_voucher_details->save();
                    }
                }

                //additionally credit the cash/bank account from which payment is made
                $new_voucher_details = new AccountVoucherDetail();
                $new_voucher_details->account_voucher_id = $new_voucher_id;
                $new_voucher_details->account_head_id = $account_head_id;
                $new_voucher_details->debit = 0;
                $new_voucher_details->credit = $paying_now;
                if($new_voucher_details->credit != 0){           
                    $new_voucher_details->save();
                }

            }

            //check if purchase order is fully paid
            $total_paid = AccountVouchers::select(
                DB::raw('ROUND(SUM(account_voucher_detail.debit), 2) as total_paid')
            )
            ->where('purchase_order_id', $purchase_order_id)
            ->join('account_voucher_detail', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
            ->where('voucher_status', 'Active')
            ->where('account_voucher_detail.account_head_id', function($query) use ($common_business_id) {
                //get accounts payable head from account event mapping 20
                $sub_query = AccountEventMapping::where('account_event_id', 20)
                ->where('business_id', $common_business_id)
                ->where('entity_name', 'payables')
                ->select('account_head_id')
                ->limit(1);
                $query->fromSub($sub_query, 'a')->select('account_head_id');
            })
            ->first();

            // if total paid >= total amount, update purchase order status to 'Paid'
            if($total_paid && $total_paid->total_paid >= $total_amount){
                $purchase_order->status = 'Paid';
                $purchase_order->save();
            }

            DB::commit(); //Commit the transaction
            return response()->json([ 
                'message' => 'Supplier payment recorded successfully',
                'message_type' => 'success',
                'message_btn'=> "btn btn-success",
                'id_account_voucher' => $new_voucher_id,
            ]);
        
        } catch (\Exception $e) {
            DB::rollBack(); //Rollback the transaction
            return response()->json([
                "message" => "Error in saving supplier payment: " . $e->getMessage(),
                "message_type" => "error",
                "message_btn" => "btn btn-danger"
            ]);
        }
    }

    public function receiving_notes()
    {

        //get all goods received notes between two dates
        $start_date = request()->input('start_date');
        $end_date = request()->input('end_date');

        if($start_date && $end_date){
            $goods_received_notes = GoodsReceivedNote::whereBetween('grn_created_date', [$start_date, $end_date])
            ->join('grn_details', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
            ->select('goods_received_note.*', 'purchase_order.purchase_order_number',
            DB::raw('date_format(goods_received_note.grn_created_date, "%d-%m-%Y") as formatted_grn_date')
            )
            ->get();
        } else {
            $start_date = date('Y-m-01');
            $end_date = date('Y-m-t');
            $goods_received_notes = GoodsReceivedNote::whereBetween('grn_created_date', [$start_date, $end_date])
            ->join('grn_details', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
            ->select('goods_received_note.*', 'purchase_order.purchase_order_number',
            DB::raw('date_format(goods_received_note.grn_created_date, "%d-%m-%Y") as formatted_grn_date')
            )
            ->get();
        }

        return view('reports.purchase.receiving_notes', compact('goods_received_notes'));
    }

    public function cancel_purchase_order($idpurchase_order)
    {
        try {
            $purchase_order = PurchaseOrder::where('idpurchase_order', $idpurchase_order)->first();
            if(!$purchase_order){
                return response()->json([ 
                    'message' => 'Purchase Order not found',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }

            //only pending purchase orders can be cancelled
            if($purchase_order->status != 'Pending'){
                return response()->json([ 
                    'message' => 'Only pending purchase orders can be cancelled',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }

            $purchase_order->status = 'Cancelled';
            $purchase_order->save();

            return response()->json([ 
                'message' => 'Purchase Order cancelled successfully',
                'message_type' => 'success',
                'message_btn'=> "btn btn-success",
            ]);

        } catch (\Exception $e) {
            return response()->json([ 
                'message' => 'Error cancelling Purchase Order: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }
    }

    public function return_note(Request $request, $idgrn)
    {
        $grn = GoodsReceivedNote::select('goods_received_note.*', 'purchase_order.purchase_order_number', 'business.business_name', 'business.business_logo')
        ->join('purchase_order', 'goods_received_note.purchase_order_id', '=', 'purchase_order.idpurchase_order')
        ->join('business', 'purchase_order.business_id', '=', 'business.id_business')
        ->where('grn_id', $idgrn)->first();
        if(!$grn){
            return Redirect::back()->with('error', 'Goods Received Note not found');
        }

        // $grn_items = GoodsReceivedNoteDetails::where('grn_id', $idgrn)->get();
        
        $grn_items = GoodsReceivedNoteDetails::query()
            ->leftJoin(
                DB::raw('
                    (
                        SELECT 
                            product_id,
                            grn_id,
                            IFNULL(SUM(return_qty),0) AS total_returned_qty
                        FROM return_notes
                        WHERE grn_id = ' . (int)$idgrn . '
                        GROUP BY grn_id , product_id 
                    ) AS rn
                '),
                'grn_details.grn_product_id',
                '=',
                'rn.product_id'
            )
            ->where('grn_details.grn_id', $idgrn)
            ->select(
                'grn_details.*',
                DB::raw('IFNULL(rn.total_returned_qty, 0) AS total_returned_qty')
            )
            ->get();

        return view('products.purchase.return_note', compact('idgrn', 'grn', 'grn_items'));
    }

    public function save_return_note(Request $request)
    {
        try {
        DB::beginTransaction(); //Start transaction
            $id_grn = $request->input('idgrn');
            $return_date = $request->input('return_date') ?? now();

            $goods_received_note =  DB::table('goods_received_note')
                ->where('grn_id',$id_grn)->first();

            $grn = GoodsReceivedNote::where('grn_id', $id_grn)
            ->join('purchase_order', 'goods_received_note.purchase_order_id', '=', 'purchase_order.idpurchase_order')
            ->select('goods_received_note.*', 'idpurchase_order', 'purchase_order.supplier_id', 'purchase_order.supplier_name', 'purchase_order.business_id')
            ->first();

            if(!$grn){
                return response()->json([ 
                    'success' =>'false',
                    'message' => 'Goods Received Note not found',
                    'message_type' => 'error',
                    'message_btn'=> "btn btn-danger",
                ]);
            }

            //Save return note
            /* 
            Columns in return_notes table
            'id_return_notes',
            'return_date',
            'return_qty',
            'batch_id',
            'product_id',
            'user_id',
            'business_id',
            'return_status',
            'grn_id',
            */
            $items = json_decode($request->input('items', '[]'), true);
            foreach ($items as $item) {

                if(!$item['return_qty'] > 0){
                    continue;
                }

                $return_note = new ReturnNotes();
                $return_note->grn_id = $id_grn;
                $return_note->business_id = $grn->business_id;
                $return_note->return_date = $return_date;
                $return_note->return_qty = $item['return_qty'];
                $return_note->batch_id = $item['batch_id'];
                $return_note->product_id = $item['product_id'];   
                $return_note->user_id = session('user_id');
                $return_note->created_at = now();
                $return_note->return_status = 'Active';
                $return_note->save();

                //Create account voucher for inventory decrease and accounts payable
                $total_return_value = $item['return_qty'] * $item['unit_price'];
                $business_id = $grn->business_id;

                $account_voucher = new AccountVouchers();
                $account_voucher->voucher_date = $return_date;
                $account_voucher->voucher_type = 3; //Return Voucher
                $account_voucher->voucher_status = 'Active';
                $account_voucher->created_by = session('user_name');
                $account_voucher->created_on = date('Y-m-d H:i:s');
                $account_voucher->purchase_order_id = $grn->idpurchase_order;                    
                $account_voucher->grn_id = $id_grn;                    
                $account_voucher->id_account_head = 24; //1 = Accounts Receivable
                $account_voucher->description = 'Inventory returned for GRN ' . $grn->grn_number . ' of PO '. $grn->idpurchase_order . ' via Return Note ' . $return_note->id_return_notes;
                $account_voucher->business_id = $business_id;
                $account_voucher->voucher_amount = $total_return_value;
                $account_voucher->cost_center = 2;
                $account_voucher->cost_center_name = 'Back Office';
                $account_voucher->business_partner = '2'; //Supplier
                $account_voucher->business_partner_id = $grn->supplier_id;
                $account_voucher->business_partner_name = $grn->supplier_name;
                $account_voucher->payment_mode = 'Credit';
                $account_voucher->auto_voucher = 'Yes';
                $account_voucher->return_note_id = $return_note->id_return_notes;
                $account_voucher->save();
                $new_voucher_id = $account_voucher->id_account_vouchers;
                
                //get account heads from account event mapping 21
                if(session('ho_accounts')=='Yes'){
                    $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
                } else {
                    $common_business_id = $grn->business_id;
                }

                $event_mappings = AccountEventMapping::where('account_event_id', 21)->where('business_id', $common_business_id)->get();

                $debit_amounts = [];
                $credit_amounts = [];

                $debit_accounts = array();
                $credit_accounts = array();

                foreach($event_mappings as $mapping){
                    
                    if($mapping['transaction_type']=='debit'){
                        array_push($debit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    } else {
                        array_push($credit_accounts, $mapping['account_head_id']."|".$mapping['transaction_type']."|".$mapping['entity_name']);
                    }
                    
                    if($mapping['transaction_type']=='debit'){
                        if($mapping['entity_name']=='payable'){array_push($debit_amounts, [ 'entity_name' => 'payable', 'amount' => $total_return_value ]);}
                    } else if($mapping['transaction_type']=='credit'){
                        if($mapping['entity_name']=='inventory'){array_push($credit_amounts, [ 'entity_name' => 'inventory', 'amount' => $total_return_value ]);}
                    }
                }

                //get the sum of debit amounts and credit amounts
                $debit_sum = !empty($debit_amounts) ? array_sum(array_column($debit_amounts, 'amount')) : null;
                $credit_sum = !empty($credit_amounts) ? array_sum(array_column($credit_amounts, 'amount')) : null;

                if($debit_sum == null || $credit_sum == null || $debit_sum != $credit_sum){
                    DB::rollBack(); //Rollback the transaction
                    return response()->json([
                        "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit and Credit amounts do not match. Please contact system administrator.",
                        "message_type" => "error",
                        "message_btn" => "danger"
                    ]);
                } else if($debit_sum == 0 || $credit_sum == 0){
                    DB::rollBack(); //Rollback the transaction
                    return response()->json([
                        "message" => "debit=". $debit_sum . " credit=". $credit_sum . " Error in Voucher Creation. Debit or Credit amounts are zero. Please contact system administrator.",
                        "message_type" => "error",
                        "message_btn" => "danger"
                    ]);
                }else{   
                    //Insert voucher Details
                    foreach($debit_accounts as $debit){
                        $debit_parts = explode("|", $debit);
                        $new_voucher_details = new AccountVoucherDetail();
                        $new_voucher_details->account_voucher_id = $new_voucher_id;
                        $new_voucher_details->account_head_id = $debit_parts[0];
                        $entity = $debit_parts[2]; // this is 'paid_cash' etc.
                        $new_voucher_details->debit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $debit_amounts));
                        $new_voucher_details->credit = 0;    
                        if($new_voucher_details->debit != 0){                                            
                            $new_voucher_details->save();
                        }
                    }
                    foreach($credit_accounts as $credit){
                        $credit_parts = explode("|", $credit);
                        $new_voucher_details = new AccountVoucherDetail();
                        $new_voucher_details->account_voucher_id = $new_voucher_id;
                        $new_voucher_details->account_head_id = $credit_parts[0];
                        $entity = $credit_parts[2]; // this is 'payables' etc.
                        $new_voucher_details->credit = array_sum(array_map(function($item) use ($entity) {
                            return $item['entity_name'] === $entity ? $item['amount'] : 0;
                        }, $credit_amounts));
                        $new_voucher_details->debit = 0;    
                        if($new_voucher_details->credit != 0){                                            
                            $new_voucher_details->save();
                        }
                    }
                }
            }

            $purchase_order_id =  $goods_received_note->purchase_order_id ?? 0 ;
            $po_detail_url = route('purchase.open_purchase_order', [
                'idpurchase_order' => $purchase_order_id
            ]);   
            DB::commit(); //Commit the transaction
            return response()->json([ 
                'success' =>true,
                'message' => 'Return Note created successfully',
                'message_type' => 'success',
                'message_btn'=> "btn btn-success",
                'po_detail_url'=> $po_detail_url,
            ]);

        } catch (\Exception $e) {
            DB::rollBack(); //Rollback the transaction
            return response()->json([ 
                'success' =>false,
                'message' => 'Error creating Return Note: ' . $e->getMessage(),
                'message_type' => 'error',
                'message_btn'=> "btn btn-danger",
            ]);
        }
    }

    public function print_return_note($idgrn)
    {
        $return_note = ReturnNotes::where('return_notes.grn_id', $idgrn)
        ->join('goods_received_note', 'return_notes.grn_id', '=', 'goods_received_note.grn_id')
        ->join('grn_details', function($join) {
            $join->on('return_notes.grn_id', '=', 'grn_details.grn_id')
                 ->on('return_notes.batch_id', '=', 'grn_details.grn_batch_id')
                 ->on('return_notes.product_id', '=', 'grn_details.grn_product_id');
        })
        ->join('purchase_order', 'goods_received_note.purchase_order_id', '=', 'purchase_order.idpurchase_order')
        ->join('business_supplier', 'purchase_order.supplier_id', '=', 'business_supplier.id_business_supplier')
        ->join('business', 'purchase_order.business_id', '=', 'business.id_business')
        ->select('return_notes.*', 
       
        'goods_received_note.grn_number',
        DB::RAW('grn_details.grn_brand_name as brand_name'),
        DB::RAW('grn_details.grn_product_name as product_name'),
        DB::RAW('grn_details.grn_sku as product_sku'),
        DB::RAW('grn_details.grn_unit_price as unit_price'),
        'purchase_order.purchase_order_number', 'business_supplier.contact_person', 
        'business_supplier.contact_number', 'business_supplier.email', 'business.business_name', 'business.business_logo',
        DB::raw('date_format(return_notes.return_date, "%d-%m-%Y") as formatted_return_date')
        )
        ->get();

        if(!$return_note){
            return Redirect::back()->with('error', 'Return Note not found');
        }

        return view('products.purchase.return_note_print', compact('return_note'));
    }

}