<?php

namespace App\Http\Controllers;



use Illuminate\Support\Facades\DB;
use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\View\View;
use App\Models\Reports;
use SebastianBergmann\CodeCoverage\Report\Xml\Report;
use App\Services\ProductsStockService;
use App\Http\Controllers\AccountsController;
use App\Models\Business;

class ReportsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function reports(Request $request)
    {

        $report_category = DB::table('report_category')
        ->orderBy('id_report_category', 'ASC')
        ->get();

        $report_list = DB::table('reports')
        ->select(
        'id_reports', 
        'report_name', 
        'report_description',
        'report_file_path', 
        DB::RAW('reports.report_category_id'),
        'report_category.category_name',
        DB::RAW('CASE WHEN coalesce(users.business_id, 0) = '.session('business_id').' THEN reports.last_viewed_by ELSE NULL END as last_viewed_by'),
        DB::RAW('CASE WHEN coalesce(users.business_id, 0) = '.session('business_id').' THEN reports.updated_at ELSE NULL END as last_viewed_at'),
        'users.user_image as last_viewed_by_image'
        )
        ->join('report_category', 'reports.report_category_id', '=', 'report_category.id_report_category')
        ->leftJoin('users', function($join) {
            $join->on('reports.last_viewed_by', '=', 'users.user_name')->where('users.business_id', session('business_id'));
        })
        ->where('report_active', 'Yes')
        ->where('report_file_path', '>', 0)
        ->orderBy('report_category.category_name', 'ASC')
        ->orderBy('reports.id_reports', 'ASC')
        ->get();

        foreach ($report_list as $report) {         
            $report_permissions = DB::table('report_role_permissions')
            ->where('role_id', session('role_id'))
            ->where('report_category_id', $report->report_category_id) 
            ->get();

            $report->permissions = $report_permissions;
            
        }

        //return $report_list;

        return view('reports.reports', compact('report_list', 'report_category'));
    }

    public function run(Request $request, $id)
    {
        if(!$request->start_date){
            // Set a default start date to the first of the current month
           $start_date = now()->startOfMonth()->toDateString();
           $end_date = now()->endOfMonth()->toDateString();
          // $request->merge(['start_date' => $start_date, 'end_date' => $end_date]);
        }
        //Get permission for this report category and role
        $report_permissions = DB::table('report_role_permissions')
        ->select('permission_type')
        ->join('report_category', 'report_role_permissions.report_category_id', '=', 'report_category.id_report_category')
        ->join('reports', 'report_category.id_report_category', '=', 'reports.report_category_id')
            ->where('role_id', session('role_id'))
            ->where('reports.report_file_path', $id)
            ->first();
        if(!$report_permissions){
            $request->merge(['permission' => 'Any']);
        }else{
            $request->merge(['permission' => $report_permissions->permission_type]);
        }

        switch ($id) {
            case 1:
                return $this->cancelled_appointments_report($request);
                break;
            case 2:
                return $this->appointments($request);
                break;
            case 3:
                return $this->customer_profiles($request);
                break;
            case 4:
                return $this->new_customers($request);
                break;

            case 6:
                return $this->returning_customers($request);
                break;
            case 7:
                return $this->care_of_customers($request);
                break;
            case 8:
                return $this->shared_customers($request);
                break;

             case 9:
                return $this->online_appointments($request);
                break;

            case 10:                
                return $this->invoices($request);
                break;
          
            case 12: 
                return $this->monthly_collection($request);
                break;

            case 13:
                return $this->product_sale_summary($request);
                break;

            case 16:
                return $this->service_sale_summary($request);
                break;
            
            case 17:
                return $this->service_sale_detail($request);
                break;
        
            case 18:
                return $this->gift_vouchers_sold($request);
                break;

            case 20:
                return $this->payment_breakup($request);
                break;

            case 22:
                return $this->receivables_from_open_visits($request);
                break;

            case 23:
                return $this->service_advances($request);
                break;

            case 24:
                return $this->retained_amount($request);
                break;

            case 26:
                return $this->open_recovery_invoices($request);
                break;

            case 31:
                return $this->grn_details($request);
                break;

            case 33:
                return $this->dispatch_note_details($request);
                break;

            case 34:
                return $this->adjustment_notes($request);

            case 35:
                return $this->return_notes($request);
            case 36:
                return $this->store_wise_stock($request);
            case 37:
                return $this->stock_aging_report($request);
            case 38:
                return $this->category_wise_stock_sales($request);
            case 39:
                return $this->product_stock_status($request);
            case 40:
                return $this->franchise_sales_summary($request);
            case 41:
                return $this->customer_sales_summary($request);
            case 42:
                return $this->customer_products_sale_details($request);
            case 43:
                return $this->expenses($request);
            case 44:
                return $this->stock_by_category($request);
            default:
                return redirect()->back()->with('error', 'Report not found.');
        }
    }

    // Report 1: Cancelled Appointments Report
    public function cancelled_appointments_report(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $cancelled_appointments_report = DB::table('customer_visits')
            ->select('id_customer_visits as ID', 
            'customers.customer_name as Customer', 'customers.customer_cell as Cell',
             DB::raw('DATE_FORMAT(MIN(visit_service_start), "%d-%m-%Y %H:%i") as "Visit Start"'),
             DB::RAW('GROUP_CONCAT(CASE WHEN visit_services.visit_service_status != "Active" THEN CONCAT(visit_services.service_name, " (Removed)") ELSE visit_services.service_name END SEPARATOR ", ") as Services'),
             DB::RAW('SUM(CASE WHEN visit_service_status = "Active" THEN visit_services.s_rate END) as "Total Price"'),
             DB::RAW('customer_visits.created_by as "Booked By"'),
             DB::RAW('customer_visits.canceled_by as "Cancelled By"'),
             DB::RAW('DATE_FORMAT(customer_visits.cancelled_on, "%d-%m-%Y %H:%i") as "Cancelled On"'),
             DB::RAW('customer_visits.cancelreason as "Cancel Reason"'),
             DB::RAW('customer_visits.visit_status as "Visit Status"'),
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('customer_visits.reminder_sms as "SMS Sent"'),
            DB::RAW('customer_visits.reminder_call as "Call"')
             )
            ->join('customers', 'customer_visits.customer_id', '=', 'customers.id_customers')
            ->join('visit_services', 'customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
            ->join('business', 'customer_visits.business_id', '=', 'business.id_business')
            ->whereNotIn('customer_visits.visit_status', ['open', 'invoiced'])
            ->whereDate('visit_services.visit_service_start', '>=', $start_date)
            ->whereDate('visit_services.visit_service_start', '<=', $end_date);
        if($request->business_id && $request->business_id != ''){
            $cancelled_appointments_report = $cancelled_appointments_report->where('customer_visits.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $cancelled_appointments_report = $cancelled_appointments_report->whereIn('customer_visits.business_id', $business_ids);
            } else {
                $cancelled_appointments_report = $cancelled_appointments_report->where('customer_visits.business_id', session('business_id'));
            }
        } else {
            $cancelled_appointments_report = $cancelled_appointments_report->where('customer_visits.business_id', session('business_id'));
        }
        $cancelled_appointments_report = $cancelled_appointments_report
            ->groupBy('customer_visits.id_customer_visits', 'customers.customer_name', 'customers.customer_cell',
             'customer_visits.created_by', 'customer_visits.canceled_by', 'customer_visits.cancelled_on',
             'customer_visits.cancelreason', 'customer_visits.visit_status', 'business.business_name')
            ->orderBy('customer_visits.cancelled_on', 'DESC');
        $cancelled_appointments_report = $cancelled_appointments_report->get();

        $data = $cancelled_appointments_report;

        $reportID=$request->id;
        $reportName = 'Cancelled Appointments Report';

        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;

        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
       
        if($request->datareport){
            return response()->json([
                'reportName' => 'Cancelled Appointments Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $request->business_id ?? '',
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id = $request->business_id ?? '';
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'permission_type', 'business_types'));
        }
    }

    // Report 2: Appointments Report
    public function appointments(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }
        $appointments = DB::table('customer_visits')
            ->select(
            DB::RAW('id_customer_visits as "ID"'), 
            DB::RAW('customer_id as "Customer ID"'), 
            DB::RAW('customers.customer_name as "Customer Name"'),
            DB::RAW('customers.customer_cell as "Customer Cell"'), 
            DB::RAW('GROUP_CONCAT(visit_services.service_name SEPARATOR ", ") as "Services"'),
            DB::RAW('MIN(DATE_FORMAT(visit_service_start, "%d-%m-%Y %H:%i:%s")) as "Visit Start"'), 
            DB::RAW('SUM(visit_services.s_rate) as "Total Price"'),
            DB::RAW('visit_status as "Status"'),
            DB::RAW('business.business_name as "Branch"')
            )
            ->join('business', 'customer_visits.business_id', '=', 'business.id_business')
            ->join('customers', 'customer_visits.customer_id', '=', 'customers.id_customers')
            ->join('visit_services', 'customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
            ->whereDate('visit_service_start', '>=', $start_date)
            ->whereDate('visit_service_start', '<=', $end_date)
            ->where('customer_visits.visit_status', 'open');
        if($request->business_id && $request->business_id != ''){
            $appointments = $appointments->where('customer_visits.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $appointments = $appointments->whereIn('customer_visits.business_id', $business_ids);
            } else {
                $appointments = $appointments->where('customer_visits.business_id', session('business_id'));
            }
        } else {
            $appointments = $appointments->where('customer_visits.business_id', session('business_id'));
        }
        $appointments = $appointments->groupBy('customer_visits.id_customer_visits', 'customers.customer_name', 'customers.customer_cell',
            'customer_visits.visit_status');

        $appointments = $appointments->get();

        $data = $appointments;

        $reportID=$request->id;
        $reportName = 'Appointments Report';
        $business_id = $request->business_id ?? '';

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_types = DB::table('business_type')->get();

        $permission_type = $request->permission;
        $business_id = $request->business_id ?? '';
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => 'Appointments Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }   
    }

    // Report 3: Customer Profiles Report
    public function customer_profiles(Request $request)
    {

        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $customer_profiles = DB::table('customers')
        ->select(
            DB::RAW('id_customers as "Customer ID"'), 
            DB::RAW('customers.customer_name as "Name"'), 
            DB::RAW('customers.customer_cell as "Cell"'), 
            DB::RAW('customers.customer_email as "Email"'), 
            DB::RAW('DATE_FORMAT(customers.created_on, "%d-%m-%Y") as "Added On"'), 
            DB::RAW('concat(customers.customer_birthday, " ", customers.customer_birthmonth) as "Birthday"'), 
            DB::RAW('customers.customer_anniversary as "Anniversary"'),
            DB::RAW('customers.customer_allergies as "Allergies"'),             
            DB::RAW('customers.customer_alert as "Alert"'),
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('DATE_FORMAT(MIN(invoice.invoice_date), "%d-%m-%Y") as "First Invoice"'),
            DB::RAW('COUNT(invoice.id_invoice) as "Total Invoices"'),
            DB::RAW('SUM(invoice.gross_wo_tax) as "Total Spent"')
            )
        ->join('business', 'customers.business_id', '=', 'business.id_business');
        //join invoices where invoice status is invoiced
        $customer_profiles->leftJoin('invoice', function($join) use ($request) {
            $join->on('customers.id_customers', '=', 'invoice.customer_id')
                 ->where('invoice.invoice_status', 'valid');
       
                if($request->business_id && $request->business_id != ''){
                    $join->where('invoice.business_id', $request->business_id);
                } else if($request->business_id && $request->business_id == '') { //get all common products businesses
                    if(session('common_products')=="Yes" && session('ho')=="Yes"){
                        $business_ids = DB::table('business')
                        ->where('common_products', 'Yes')
                        ->pluck('id_business')
                        ->toArray();

                        $join->whereIn('invoice.business_id', $business_ids);
                    } else {
                        $join->where('invoice.business_id', session('business_id'));
                    }
                } else {
                    $join->where('invoice.business_id', session('business_id'));
                }
            });
        $customer_profiles = $customer_profiles->groupBy('customers.id_customers', 'customers.customer_name', 'customers.customer_cell',
            'customers.customer_email', 'customers.created_at', 'business.business_name');
        
        $customer_profiles =   $customer_profiles->having('Total Invoices', '>', 0);
        $customer_profiles = $customer_profiles->orderBy('Total Invoices', 'DESC')
        ->limit(100);

        $customer_profiles = $customer_profiles->get();

        $data = $customer_profiles;

        $business_id = $request->business_id ?? '';

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_types = DB::table('business_type')->get();

        $reportID = $request->id;
        $reportName = 'Customer Profiles Report';

        $permission_type = $request->permission;
       
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $customer_profiles,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }
    }

    // Report 4: New Customers Report
    public function new_customers(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }


        $new_customers = DB::table('customers as c')
            ->join('business as b', 'c.business_id', '=', 'b.id_business')
            ->join('customer_visits as cv', 'cv.customer_id', '=', 'c.id_customers')
            
            ->select([
                'c.id_customers as Customer ID',
                'c.customer_name as Name',
                'c.customer_cell as Cell',
                'c.customer_email as Email',
                DB::raw("DATE_FORMAT(c.created_on, '%d-%m-%Y') as `Added On`"),
                'b.business_name as Branch',
                DB::raw("DATE_FORMAT(MIN(cv.customer_visit_date), '%d-%m-%Y') as `First Visit`"),
                DB::RAW('visit_status as Status')
            ])
            ->groupBy(
                'c.id_customers',
                'c.customer_name',
                'c.customer_cell',
                'c.customer_email',
                'c.created_on',
                'b.business_name'
                
            )
            ->havingRaw(
                "DATE(MIN(cv.customer_visit_date)) BETWEEN ? AND ?",
                [$start_date, $end_date]
            );
            
        $new_customers = $new_customers->whereIn('cv.visit_status', ['invoiced', 'open']);

        if($request->business_id && $request->business_id != ''){
            $new_customers = $new_customers->where('cv.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $new_customers = $new_customers->whereIn('cv.business_id', $business_ids);
            } else {
                $new_customers = $new_customers->where('cv.business_id', session('business_id'));
            }
        } else {
            $new_customers = $new_customers->where('cv.business_id', session('business_id'));
        }
        $new_customers = $new_customers->orderByRaw("MIN(cv.customer_visit_date) DESC");
        $new_customers = $new_customers->get();

        $data = $new_customers;

        $reportID = $request->id;
        $reportName = 'New Customers Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    // Report 6: Returning Customers Report
    public function returning_customers(Request $request){
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $returning_customers = DB::table('customers as c')
            ->join('business as b', 'c.business_id', '=', 'b.id_business')
            ->join('invoice as i', 'i.customer_id', '=', 'c.id_customers')
            ->select([
                'c.id_customers as Customer ID',
                'c.customer_name as Name',
                'c.customer_cell as Cell',
                'c.customer_email as Email',
                DB::raw("DATE_FORMAT(c.created_on, '%d-%m-%Y') as `Added On`"),
                'b.business_name as Branch',
                DB::raw("COUNT(i.id_invoice) as `Total Visits`"),
                DB::raw("DATE_FORMAT(MAX(i.invoice_date), '%d-%m-%Y') as `Last Visit`")
            ])
            ->groupBy(
                'c.id_customers',
                'c.customer_name',
                'c.customer_cell',
                'c.customer_email',
                'c.created_on',
                'b.business_name'
            )
            ->havingRaw(
                "DATE(MIN(i.invoice_date)) < ? AND MAX(DATE(i.invoice_date)) BETWEEN ? AND ?",
                [$start_date, $start_date, $end_date]
            );
            
        $returning_customers = $returning_customers->where('i.invoice_status', 'valid');
        if($request->business_id && $request->business_id != ''){
            $returning_customers = $returning_customers->where('i.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $returning_customers = $returning_customers->whereIn('i.business_id', $business_ids);
            } else {
                $returning_customers = $returning_customers->where('i.business_id', session('business_id'));
            }
        } else {
            $returning_customers = $returning_customers->where('i.business_id', session('business_id'));
        }
        $returning_customers = $returning_customers->orderByRaw("MAX(i.invoice_date) DESC");
        $returning_customers = $returning_customers->get();
        $data = $returning_customers;
        $reportID = $request->id;
        $reportName = 'Returning Customers Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }

    }

    // Report 7: Care Of Customers Report
    public function care_of_customers(Request $request){
        // Similar structure to other customer reports
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }
        
        $care_of_customers = DB::table('customers')
            ->select(
                DB::RAW('id_customers as "Customer ID"'), 
                DB::RAW('customers.customer_name as Name'), 
                DB::RAW('customers.customer_cell as Cell'), 
                DB::RAW('customers.customer_email as Email'), 
                DB::RAW('DATE_FORMAT(customers.created_on, "%d-%m-%Y") as "Added On"'), 
                DB::RAW('business.business_name as "Branch"'),
                DB::RAW('customers.customer_careof as "Care Of"')
                
            )
            ->join('business', 'customers.business_id', '=', 'business.id_business')
            ->where(DB::RAW('ifnull(customers.customer_careof, "")'), '!=', '')
            ;
        if($request->business_id && $request->business_id != ''){
            $care_of_customers = $care_of_customers->where('customers.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $care_of_customers = $care_of_customers->whereIn('customers.business_id', $business_ids);
            } else {
                $care_of_customers = $care_of_customers->where('customers.business_id', session('business_id'));
            }
        } else {
            $care_of_customers = $care_of_customers->where('customers.business_id', session('business_id'));
        }            

        
        $care_of_customers = $care_of_customers->get();

        $data = $care_of_customers;
        $reportID = $request->id;
        $reportName = 'Care Of Customers Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    // Report 8: Shared Customers Report
    public function shared_customers(Request $request){
        // Similar structure to other customer reports
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }
        
       $multiBranchCustomers = DB::table('invoice')
            ->select('customer_id')
            ->groupBy('customer_id')
            ->havingRaw('COUNT(DISTINCT business_id) > 1');

        $shared_customers = DB::table('customers as c')
            ->join('invoice as i', 'i.customer_id', '=', 'c.id_customers')
            ->join('business as b', 'i.business_id', '=', 'b.id_business')
            ->whereIn('c.id_customers', $multiBranchCustomers)
            ->select(
                DB::raw('c.id_customers as "Customer ID"'),
                DB::raw('c.customer_name as "Name"'),
                DB::raw('c.customer_cell as "Cell"'),
                DB::raw('c.customer_email as "Email"'),
                DB::raw('b.business_name as "Branch"'),
                DB::raw('COUNT(i.id_invoice) as "Total Invoices"')
            )
            ->groupBy(
                'c.id_customers',
                'c.customer_name',
                'b.business_name'
            );
                    
        $shared_customers = $shared_customers->get();

        $data = $shared_customers;
        $reportID = $request->id;
        $reportName = 'Shared Customers Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    // Report 9: Online Appointments Report
    public function online_appointments(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }

        $online_appointments = DB::table('customer_visits')
        ->select(
            
            DB::RAW('customers.id_customers as "Customer ID"'), 
            DB::RAW('customers.customer_name as "Name"'), 
            DB::RAW('customers.customer_cell as "Cell"'), 
            DB::RAW('CASE WHEN customer_visits.visit_status not in ("open", "invoiced") THEN \'Yes\' ELSE \'No\' END as "Cancelled"'),
            DB::RAW('customer_visits.id_customer_visits as "ID Visit"'), 
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('date_format(customer_visits.created_on, "%d-%m-%Y") as "Booking Date"'),   
            DB::RAW('Min(date_format(visit_services.visit_service_start, "%d-%m-%Y %H:%i")) as "Scheduled Date"'),
            DB::RAW('GROUP_CONCAT(visit_services.service_name SEPARATOR "<br> ") as "Services Booked"'),
            DB::RAW('group_concat(visit_service_staffs.staff_name SEPARATOR "<br> ") as "Staff Assigned"'),
            DB::RAW('customer_visits.created_by as "Booked Via"'),
            DB::RAW('customer_visits.visit_status as "Status"')
            )
        ->join('customers', 'customers.id_customers', '=', 'customer_visits.customer_id')
        ->join('business', 'customer_visits.business_id', '=', 'business.id_business')
        ->join('visit_services', 'customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
        ->join('visit_service_staffs', 'visit_services.id_visit_services', '=', 'visit_service_staffs.visit_service_id')
        ->where('customer_visits.created_by', 'Online');
        
        $online_appointments = $online_appointments->whereDate('customer_visits.created_on', '>=', $start_date)
            ->whereDate('customer_visits.created_on', '<=', $end_date);
        $online_appointments = $online_appointments->groupBy('customer_visits.id_customer_visits', 'customers.id_customers', 
            'customers.customer_name', 'customers.customer_cell', 'customer_visits.created_on', 'business.business_name',
            'customer_visits.visit_status');

        if($request->business_id && $request->business_id != ''){
            $online_appointments = $online_appointments->where('customer_visits.business_id',
            $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $online_appointments = $online_appointments->whereIn('customer_visits.business_id', $business_ids);
            } else {
                $online_appointments = $online_appointments->where('customer_visits.business_id', session('business_id'));
            }
        } else {
            $online_appointments = $online_appointments->where('customer_visits.business_id', session('business_id'));
        }
        $online_appointments = $online_appointments->get();

        $data = $online_appointments;
        $reportID=$request->id;
        $reportName = 'Online Appointments Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => 'Online Appointments Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    // Report 10: Invoices Report
    public function invoices(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date =  $start_date; //same date
        }

        $businessId = $request->business_id ?? session('business_id');

        $business_id_for_account_head = session('business_id');  
        if(session('ho_accounts')=="Yes"){
            $business_id_for_account_head = DB::table('business')->where('ho','Yes')->value('id_business');
        }

        //get account heads
        $ac_head = [];
        
         /* Service Gross Total */
        $ac_head[0] = DB::table('account_event_mapping')
            ->where('account_event_id', 1)
            ->where('entity_name', 'grosstotal_service')
            ->where('business_id', $business_id_for_account_head)
            ->value('account_head_id');

        /* Retail Gross Total */
        $ac_head[1] = DB::table('account_event_mapping')
            ->where('account_event_id', 1)
            ->where('entity_name', 'grosstotal_retail')
            ->where('business_id', $business_id_for_account_head)
            ->value('account_head_id');

        /* Service Discount */
        $ac_head[2] = DB::table('account_event_mapping')
            ->where('account_event_id', 1)
            ->where('entity_name', 'sale_discount_service')
            ->where('business_id', $business_id_for_account_head)
            ->value('account_head_id');

        /* Retail Discount */
        $ac_head[3] = DB::table('account_event_mapping')
            ->where('account_event_id', 1)
            ->where('entity_name', 'sale_discount_retail')
            ->where('business_id', $business_id_for_account_head)
            ->value('account_head_id');

        $isShUser = session('is_sh_user') == 'Yes' ? true : false;



        //code for invoices report
        $serviceQuery = DB::table('invoice')
            ->leftJoin('invoice_details', 'invoice_details.invoice_id', '=', 'invoice.id_invoice')
            ->join('business', 'invoice.business_id', '=', 'business.id_business')
            ->leftJoinSub(
                DB::table('account_voucher_detail')
                    ->join('account_vouchers', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
                    ->join('invoice as i2', 'i2.id_invoice', '=', 'account_vouchers.invoice_id')
                    ->select(
                        'invoice_id',
                        'id_account_vouchers',
                        DB::raw('SUM(credit) - SUM(debit) AS saleincome')
                    )
                    ->whereBetween(DB::raw('DATE(i2.invoice_date)'), [$start_date.' 00:00:00', $end_date.' 23:59:59'])
                    ->whereIn('account_voucher_detail.account_head_id', [$ac_head[0], $ac_head[2]])
                    ->groupBy('id_account_vouchers'),
                'a',
                'a.invoice_id',
                '=',
                'invoice.id_invoice'
            )
            ->whereBetween(DB::raw('DATE(invoice.invoice_date)'), [$start_date.' 00:00:00', $end_date.' 23:59:59'])
            ->where('invoice_type', 'service')
            ->when($isShUser, fn ($q) => $q->where('invoice_seq', '>', 0))
            ->groupBy('id_invoice')
            ->selectRaw("
                id_invoice as Invoice_ID,
                invoice.invoice_day_seq as Invoice_Day_Seq,
                 DATE_FORMAT(invoice_date,'%d-%m-%Y %H:%i:%s') AS invoice_date,
                CASE WHEN invoice.invoice_status = 'valid' THEN 'No' ELSE 'Yes' END AS Cancelled,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')<>'' THEN '<h5><span class=\"badge badge-outline-primary\">Recovery</span></h5>' ELSE '' END AS recovery,                
                reference_invoice_number as ref,
                business.business_name as Branch,
                    
                invoice_number as Invoice_Number,
                invoice_type as Invoice_Type,
                IFNULL(invoice_extra,'') AS invoice_Extra,
                payment_mode,
                ifnull(instrument_number,'') AS instrument_number,
               
                invoice.created_by,
                customer_id,
                customer_name,
                customer_cell,                
               ifnull(saleincome, 0) as Total_Sale_Income,

                CASE WHEN invoice.invoice_type='service' AND IFNULL(invoice.reference_invoice_number,'')=''
                    THEN ROUND(SUM(IFNULL(invoice_details.service_addition,0)),2) ELSE 0 END AS service_addition_total,   

                    IFNULL(discount_reason,'') AS discount_reason,                
                CASE WHEN invoice.invoice_type='service' AND IFNULL(invoice.reference_invoice_number,'')=''
                    THEN SUM(invoice_details.discount) ELSE 0 END AS service_Discount_Total,
                    sub_Total,

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN invoice.discount ELSE 0 END AS invoice_Discount,

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN other_charges ELSE 0 END AS other_Charges,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN tax_total ELSE 0 END AS tax_Total,
                invoice.cc_charge as cc_Charge,
                 IFNULL(cctip,0) AS cc_Tip_Total,
                 CASE WHEN IFNULL(invoice.reference_invoice_number,'') = ''
                    THEN ROUND(IFNULL(invoice.total_payable,0),2)
                    ELSE 0 END AS payable_Total,

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN invoice.advance_amount ELSE 0 END AS advance_Adjusted,
                
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN retained_amount_used ELSE 0 END AS retained_Amount_Used,
              
                invoice.paid_cash + invoice.paid_card + invoice.paid_voucher + invoice.paid_check AS paid_amount,
                invoice.balance as Balance_Total,

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN retained_amount ELSE 0 END AS retained_Amount,
                
                paid_cash as Paid_Cash_Total,
                paid_card as Paid_Card_Total,
                paid_voucher as Paid_Voucher_Total,
                paid_check as Paid_Check_Total,
               
                invoice.remarks,
                imsfiscal_invoice_number AS ims_ref
            ");

        if($request->business_id && $request->business_id != ''){
            $serviceQuery = $serviceQuery->where('invoice.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('ho_accounts', 'Yes')
                ->orwhere('ho','Yes')
                ->pluck('id_business')
                ->toArray();
    
                $serviceQuery = $serviceQuery->whereIn('invoice.business_id', $business_ids);
            } else {
                $serviceQuery = $serviceQuery->where('invoice.business_id', session('business_id'));
            }
        } else {
            $serviceQuery = $serviceQuery->where('invoice.business_id', session('business_id'));
        }

        $serviceInvoices = $serviceQuery->get();
        // return $serviceInvoices;

        $retailQuery = DB::table('invoice')
            ->leftJoin('invoice_products', 'invoice_products.invoice_id', '=', 'invoice.id_invoice')
            ->join('business', 'business.id_business', '=', 'invoice.business_id')
            ->leftJoinSub(
                DB::table('account_voucher_detail')
                    ->join('account_vouchers', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
                    ->join('invoice as i2', 'i2.id_invoice', '=', 'account_vouchers.invoice_id')
                    ->select(
                        'invoice_id',
                        'id_account_vouchers',
                        DB::raw('SUM(credit) - SUM(debit) AS saleincome')
                    )
                    ->whereBetween(DB::raw('DATE(i2.invoice_date)'), [$start_date.' 00:00:00', $end_date.' 23:59:59'])
                    
                    ->whereIn('account_voucher_detail.account_head_id', [$ac_head[1], $ac_head[3]])
                    
                    ->groupBy('id_account_vouchers'),
                'a',
                'a.invoice_id',
                '=',
                'invoice.id_invoice'
            )
            ->whereBetween(DB::raw('DATE(invoice.invoice_date)'), [$start_date.' 00:00:00', $end_date.' 23:59:59'])
           
            ->where('invoice_type', 'sale')
            ->when($isShUser, fn ($q) => $q->where('invoice_seq', '>', 0))
            ->groupBy('id_invoice')
            ->selectRaw("
                id_invoice as Invoice_ID,
                invoice.invoice_day_seq as Invoice_Day_Seq,
                DATE_FORMAT(invoice_date,'%d-%m-%Y %H:%i:%s') AS Invoice_Date,
                CASE WHEN invoice.invoice_status = 'valid' THEN 'No' ELSE 'Yes' END AS Cancelled,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')<>'' THEN '<h5><span class=\"badge badge-outline-primary\">Recovery</span></h5>' ELSE '' END AS recovery,                
                reference_invoice_number as ref,

                business.business_name as Branch,
                
                invoice_number as Invoice_Number,
                invoice_type as Invoice_Type,
                IFNULL(instrument_number,'') AS Instrument_Number,
                payment_mode,                
                IFNULL(invoice_extra,'') AS Invoice_Extra,
                
                invoice.created_By,
                customer_id,
                customer_name,
                customer_cell,
                saleincome as Total_Sale_Income,
                               
                CASE WHEN invoice.invoice_type='sale' AND IFNULL(invoice.reference_invoice_number,'')=''
                    THEN ROUND(SUM(IFNULL(invoice_products.service_addition,0)),2) ELSE 0 END AS service_Addition_Total,   

                IFNULL(discount_reason,'') AS discount_Reason,                
                CASE WHEN invoice.invoice_type='sale' AND IFNULL(invoice.reference_invoice_number,'')=''
                    THEN SUM(invoice_products.discount) ELSE 0 END AS service_Discount_Total,
                sub_total as sub_Total, 

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN invoice.discount ELSE 0 END AS invoice_Discount,                

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN other_charges ELSE 0 END AS other_Charges,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN tax_total ELSE 0 END AS tax_Total,
                invoice.cc_charge as cc_Charge,
                IFNULL(cctip,0) AS cc_Tip_Total,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN ROUND(IFNULL(invoice.total_payable,0),2)
                    ELSE 0 END AS payable_Total,

                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN invoice.advance_amount ELSE 0 END AS advance_Adjusted,
                
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN retained_amount_used ELSE 0 END AS retained_Amount_Used,               
                
                invoice.paid_cash + invoice.paid_card + invoice.paid_voucher + invoice.paid_check AS paid_amount,
                invoice.balance as Balance_Total,
                CASE WHEN IFNULL(invoice.reference_invoice_number,'')=''
                    THEN retained_amount ELSE 0 END AS retained_Amount,
                
                paid_cash as Paid_Cash_Total,
                paid_card as Paid_Card_Total,
                paid_voucher as Paid_Voucher_Total,
                paid_check as Paid_Check_Total,
                
                invoice.remarks,
                imsfiscal_invoice_number AS ims_ref
            ");

        if($isShUser) {
            $retailQuery = $retailQuery->where('invoice_seq', '>', 0);
        }

        if($request->business_id && $request->business_id != ''){
            $retailQuery = $retailQuery->where('invoice.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('ho_accounts', 'Yes')
                ->orwhere('ho','Yes')
                ->pluck('id_business')
                ->toArray();
    
                $retailQuery = $retailQuery->whereIn('invoice.business_id', $business_ids);
            } else {
                $retailQuery = $retailQuery->where('invoice.business_id', session('business_id'));
            }
        } else {
            $retailQuery = $retailQuery->where('invoice.business_id', session('business_id'));
        }

        $retailInvoices = $retailQuery->get();

        $invoices = $serviceInvoices->merge($retailInvoices)->sortByDesc('invoice_date')->values();
        $data = $invoices;

        $branches = [];
        if(session('ho')=="Yes"){
            $branches = DB::table('business')->where('ho_accounts','Yes')->orwhere('ho','Yes')->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_types = DB::table('business_type')->get();
        $reportID = $request->id;
        $reportName = 'Invoices Report';

        $permission_type = $request->permission;
        $business_id = $request->business_id ?? '';
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }
    }

    // Report 13: product Sale Summary Report
    public function product_sale_summary(Request $request)
    {

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }

        //code for product sale summary report
        $product_sale_summary = DB::table('invoice_products')
             ->select(
                DB::RAW('business.id_business'),
                DB::RAW('business.business_name as "Branch"'), 
                DB::RAW('invoice_products.product_id as "Product ID"'), 
                DB::RAW('invoice_products.brand_name as "Brand"'), 
                DB::RAW('invoice_products.product_name as "Product Name"'),                 
                DB::RAW('invoice_products.category as "Category"'), 
                DB::RAW('SUM(invoice_products.invoice_qty) as "Total Quantity Sold"'), 
                DB::RAW('ROUND(SUM((invoice_products.price + invoice_products.service_addition - invoice_products.discount)) * invoice_products.invoice_qty, 2) as "Total Sales Amount"'),
                DB::RAW('ROUND(SUM(product_batch.batch_amount * invoice_products.invoice_qty), 2) as "Total COGS "')
            )

            ->join('invoice', 'invoice.id_invoice', '=', 'invoice_products.invoice_id')           
            ->join('business', 'invoice.business_id', '=', 'business.id_business')
            ->join('business_type', 'business.business_type_id', '=', 'business_type.id_business_type')
            ->join('product_batch', 'invoice_products.batch_id', '=', 'product_batch.id_batch')
           
           
            ->where(DB::RAW('DATE(invoice.invoice_date)'), '>=', $start_date)
            ->where(DB::RAW('DATE(invoice.invoice_date)'), '<=', $end_date)
            ->where('invoice.invoice_status', 'valid')
            ->where('invoice.invoice_type', 'sale')
            ->groupBy('invoice_products.product_id', 'business.id_business');

        //filter by business type
        if($request->business_type_id != null && $request->business_type_id != ''){
            $product_sale_summary = $product_sale_summary->where('business.business_type_id', $request->business_type_id);
        }

        if($request->business_id != null && $request->business_id != ''){
            $product_sale_summary = $product_sale_summary->where('invoice.business_id', $request->business_id);
        } else if($request->business_id == null || $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $product_sale_summary = $product_sale_summary->whereIn('invoice.business_id', $business_ids);
            } else {
                $product_sale_summary = $product_sale_summary->where('invoice.business_id', session('business_id'));
            }
        } else {
            $product_sale_summary = $product_sale_summary->where('invoice.business_id', session('business_id'));
        }

        $product_sale_summary = $product_sale_summary->toRawSql();

        $base_sql = "SELECT `Product ID`, `Brand`, `Product Name`, `Category` ";

        foreach($branches as $branch){
            if($request->business_type_id > 0){
                if($branch->business_type_id == $request->business_type_id){
                    $base_sql .= ", SUM( CASE WHEN `id_business` = '". $branch->id_business . "' THEN `Total Quantity Sold` ELSE 0 END ) AS `".$branch->business_name." Quantity Sold` ";
                    $base_sql .= ", SUM( CASE WHEN `id_business` = '". $branch->id_business . "' THEN `Total Sales Amount` ELSE 0 END ) AS `".$branch->business_name." Sales Amount` ";
                    $base_sql .= ", SUM( CASE WHEN `id_business` = '". $branch->id_business . "' THEN `Total COGS ` ELSE 0 END ) AS `".$branch->business_name." COGS` ";
                }
            } else {
                $base_sql .= ", SUM( CASE WHEN `Branch` = '". $branch->business_name . "' THEN `Total Quantity Sold` ELSE 0 END ) AS `".$branch->business_name." Quantity Sold` ";
                $base_sql .= ", SUM( CASE WHEN `Branch` = '". $branch->business_name . "' THEN `Total Sales Amount` ELSE 0 END ) AS `".$branch->business_name." Sales Amount` ";
                $base_sql .= ", SUM( CASE WHEN `Branch` = '". $branch->business_name . "' THEN `Total COGS ` ELSE 0 END ) AS `".$branch->business_name." COGS` ";
            }
        }
       
        $base_sql .= " 
            FROM ( " . $product_sale_summary . " ) AS subquery
            GROUP BY `Product ID`, `Brand`, `Product Name`, `Category`
            ORDER BY `Total Quantity Sold` DESC";
        // print_r($base_sql); exit;
        $data = DB::select($base_sql);

        $reportID=$request->id;
        $reportName = 'Product Sale Summary Report';
        $business_id = $request->business_id ?? '';
        
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }
    }

    // Report 12: Monthly Collection Report
    public function monthly_collection(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;

        if($request->start_date == null){
            $start_date = now()->startOfMonth()->toDateString(); //yesterday date
        } else {
            //make $start_date the first day of the month
            $start_date = \Carbon\Carbon::parse($start_date)->startOfMonth();
        }

        
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = now()->endOfMonth()->toDateString(); //last day of  the month
        } else {
            //make $end_date the last day of the month
            $end_date = \Carbon\Carbon::parse($end_date)->endOfMonth();
        }

        $business_id = $request->business_id;

        if($business_id == ''){
            //get all common products businesses
            if(session('ho')=="Yes"){
                $business_ids = DB::table('business')
               ->where('ho_accounts', 'Yes')->orwhere('ho', 'Yes')
                ->pluck('id_business')
                ->toArray();
                $business_id = implode(',', $business_ids);
            } else {
                $business_id = session('business_id');
            }
        }
        
        $business_id_for_account_head = session('business_id');
        if (session('ho_accounts') === 'Yes') {
            $business_id_for_account_head = DB::table('business')
                ->where('ho', 'Yes')
                ->value('id_business');
        }


        //code for monthly collection report and return in reports.report_page view
        //get all entities from account_event_mapping for account_event_id = 1
        $entities = DB::table('account_event_mapping')
        ->join('account_heads', 'account_heads.id_account_heads', '=', 'account_event_mapping.account_head_id')
        ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
        ->join('account_control', 'account_sub_control.account_control_id', '=', 'account_control.id_account_control')
        ->select('account_head_id', 'account_head', 'transaction_type')
        ->where('account_event_id', 1)
        ->where('account_event_mapping.business_id', $business_id_for_account_head)
        ->orderBy('account_control.id_account_control',  'ASC')
        ->orderBy('account_sub_control.id_account_sub_control',  'ASC')
        ->orderBy('account_heads.id_account_heads',  'ASC')
        ->get();

        $sql = "SELECT concat('<h5><span class=\"badge badge-outline-primary\">', date_format(account_voucher_date, '%b-%Y'), '</span></h5>') as Period, business_name as Branch ,  ";
        foreach($entities as $entity){
            if($entity->transaction_type == 'debit'){
                
                $sql .= " ROUND(SUM(CASE WHEN dataset.account_head = '".$entity->account_head."' THEN dataset.debit  ELSE 0 END), 2) as '".$entity->account_head." Total', ";
            }
        }   
        $sql .= "  Month(account_voucher_date) as Month, Year(account_voucher_date) as Year
            FROM
                (
                    SELECT business.business_name, account_vouchers.voucher_date as account_voucher_date,                   
                    id_account_vouchers, account_voucher_detail.account_head_id, account_head, 
                    sum(debit) as debit, sum(credit) as credit
                    FROM account_vouchers
                    join business on account_vouchers.business_id = business.id_business
                    join account_voucher_detail on account_vouchers.id_account_vouchers = account_voucher_detail.account_voucher_id
                    join account_heads on account_heads.id_account_heads = account_voucher_detail.account_head_id                               
                    where account_vouchers.voucher_date >= '".$start_date."'
                    and account_vouchers.voucher_date <= '".$end_date."'
                    and account_vouchers.business_id IN (".$business_id.")
                    group by id_account_vouchers, id_account_heads
                   
                ) as dataset
                group by Month(dataset.account_voucher_date), Year(dataset.account_voucher_date)
                order by Year(dataset.account_voucher_date), Month(dataset.account_voucher_date) ASC
                ";
       
        $monthly_collection = DB::select($sql);

        $data = $monthly_collection;
        $reportID=$request->id;
        $reportName = 'Monthly Collection Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('ho')=="Yes"){
            $branches = DB::table('business')->where('ho_accounts', 'Yes')->orwhere('ho', 'Yes')->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => 'Monthly Collection Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
       
    }

    // Report 16: Service Sale Summary Report
    public function service_sale_summary(Request $request)
    {
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }

          $isShUser = session('is_sh_user') == 'Yes' ? true : false;
        
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = now()->endOfMonth()->toDateString(); //last day of  the month
        } else {
            //make $end_date the last day of the month
            $end_date = \Carbon\Carbon::parse($end_date)->endOfMonth();
        }

        //code for service sale summary report
        $service_sale_summary = DB::table('visit_services as vs')
                ->select(
                    DB::RAW('b.id_business'),
                    DB::RAW('vs.service_id as Service_Code'),
                     DB::RAW('b.business_name as Branch'),
                    DB::RAW('vs.s_type as Type'),
                    DB::RAW('vs.s_category as Category'),                    
                    DB::RAW('concat("<h4><span class=\"badge badge-outline-success\">", vs.service_name, "</span></h4>") as Service'),                  
                    DB::RAW('COUNT(vs.id_visit_services) as Total_Quantity_Sold'),
                    DB::RAW('ROUND(SUM(idat.discounted_price + idat.service_addition), 2) as Total_Sales_Amount')
                )
                ->leftJoin('invoice_details as idat', 'vs.id_visit_services', '=', 'idat.visit_service_id')
                ->leftJoin('invoice as i', function($join) {
                    $join->on('idat.invoice_id', '=', 'i.id_invoice')
                         ->where('i.invoice_status', 'valid');
                })
                
                ->join('customer_visits as cv', 'vs.customer_visit_id', '=', 'cv.id_customer_visits')
                ->join('business as b', 'cv.business_id', '=', 'b.id_business')
                ->whereBetween('i.invoice_date', [$start_date, $end_date])
                ->where('cv.visit_status', 'invoiced')
                ->groupBy('vs.service_id', 'b.id_business');

            if($isShUser) {
                $service_sale_summary = $service_sale_summary->where('invoice_seq', '>', 0);
            }

            //filter by business type
            if($request->business_type_id != null && $request->business_type_id != ''){
                $service_sale_summary = $service_sale_summary->where('b.business_type_id', $request->business_type_id);
            }

            if($request->business_id != '' && $request->business_id != null){               
                $service_sale_summary = $service_sale_summary->where('cv.business_id', $request->business_id);
            } else if($request->business_id == '' || $request->business_id == null) { //get ho business id
                if(session('ho')=="Yes"){
                    // get all businesses with common services
                    $business_ids = DB::table('business')
                    ->where('common_services', 'Yes')
                    ->pluck('id_business')
                    ->toArray();

                    $service_sale_summary = $service_sale_summary->whereIn('cv.business_id', $business_ids);
                    
                } else {
                    $service_sale_summary = $service_sale_summary->where('cv.business_id', session('business_id'));
                }
            } else {
                $service_sale_summary = $service_sale_summary->where('cv.business_id', session('business_id'));
            }

            //Apply filters for service type, category and service department
            if($request->service_type && $request->service_type != ''){
                $service_sale_summary = $service_sale_summary->where('vs.id_service_type', $request->service_type);
            }
            if($request->service_category && $request->service_category != ''){
                $service_sale_summary = $service_sale_summary->where('vs.id_service_category', $request->service_category);
            }
            if($request->service_department && $request->service_department != ''){
                
                $service_sale_summary = $service_sale_summary->where('bs.service_department', $request->service_department);
            }


            $service_sale_summary = $service_sale_summary->toRawSql();
        
            $final_sql = "SELECT subquery.Service_Code, subquery.Type, subquery.Category, 
            subquery.Service ";
            
            foreach($branches as $branch){
                if($request->business_type_id > 0){                   
                    if($branch->business_type_id == $request->business_type_id){
                        $final_sql .= ", SUM(CASE WHEN subquery.id_business = '".$branch->id_business."' THEN subquery.Total_Quantity_Sold ELSE 0 END) AS `".$branch->business_name." Qty` ";
                        $final_sql .= ", SUM(CASE WHEN subquery.id_business = '".$branch->id_business."' THEN subquery.Total_Sales_Amount ELSE 0 END) AS `".$branch->business_name." Amount` ";
                    }
                } else {
                    $final_sql .= ", SUM(CASE WHEN subquery.id_business = '".$branch->id_business."' THEN subquery.Total_Quantity_Sold ELSE 0 END) AS `".$branch->business_name." Qty` ";
                    $final_sql .= ", SUM(CASE WHEN subquery.id_business = '".$branch->id_business."' THEN subquery.Total_Sales_Amount ELSE 0 END) AS `".$branch->business_name." Amount` ";
                }
            }

            $final_sql .= " FROM (".$service_sale_summary.") AS subquery 
            GROUP BY Service_Code, Type, Category, Service
            ORDER BY Total_Quantity_Sold DESC";
            
        $data = DB::select($final_sql);

        //$data = $service_sale_summary;
        $reportID=$request->id;
        $reportName = 'Service Sale Summary Report';
        $business_id = $request->business_id ?? '';
        
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        //get service types, package types and service departments for filters
        $service_types1 = DB::table('service_type')
        ->select('id_service_types', 'service_type')
        ->where('service_type_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            //pluck ho business id
            $ho_business_id = DB::table('business')->where('ho', 'Yes')->value('id_business');
            $service_types1 = $service_types1->where('business_id', $ho_business_id);
        } else {
            $service_types1 = $service_types1->where('business_id', session('business_id'));
        }
        $service_types1 = $service_types1->get();

        $package_types = DB::table('package_type')
        ->select(
            DB::raw('id_package_type as id_service_types'),
            DB::raw("CONCAT(service_type, ' (Package)') as service_type")
        )
        ->where('service_type_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $package_types = $package_types->where('business_id', $ho_business_id);
        } else {
            $package_types = $package_types->where('business_id', session('business_id'));
        }

        $package_types = $package_types->get();

        $service_types = $service_types1->merge($package_types)->sortBy('service_type');
    
        $service_categories1 = DB::table('service_category')
        ->select('id_service_category', 'service_category', 'service_type')
        ->join('service_type', 'service_category.service_type_id', '=', 'service_type.id_service_types')
        ->where('service_category_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $service_categories1 = $service_categories1->where('service_category.business_id', $ho_business_id);
        } else {
            $service_categories1 = $service_categories1->where('service_category.business_id', session('business_id'));
        }

        $service_categories1 = $service_categories1->get();
        $package_categories = DB::table('package_category')
        ->select(
            DB::raw('id_package_category as id_service_category'),
            DB::raw("CONCAT(service_category, ' (Package)') as service_category"),
            DB::RAW('package_type.service_type as service_type')
        )
        ->join('package_type', 'package_category.package_type_id', '=', 'package_type.id_package_type')
        ->where('service_category_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $package_categories = $package_categories->where('package_category.business_id', $ho_business_id);
        } else {
            $package_categories = $package_categories->where('package_category.business_id', session('business_id'));
        }
        $package_categories = $package_categories->get();

        $service_categories = $service_categories1->merge($package_categories)->sortBy('service_category');

        $service_departments = DB::table('service_department')
        ->select('id_service_department', 'service_department')
        ->where('department_status', 'Active');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $service_departments = $service_departments->where('business_id', $ho_business_id);
        } else {
            $service_departments = $service_departments->where('business_id', session('business_id'));
        }
        $service_departments = $service_departments->get();

        //////////


        if($request->datareport){
            return response()->json([
                'reportName' => 'Service Sale Summary Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'service_types', 'service_categories', 'service_departments', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
       
    }

    // Report 17: Service Sale Detail Report
    public function service_sale_detail(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $isShUser = session('is_sh_user') == 'Yes' ? true : false;       
        

        $service_sale_detail = DB::table('visit_services as vs')
            ->select([
                'cv.id_customer_visits as Visit_ID',
                'i.id_invoice as Invoice_ID',
                DB::RAW('b.business_name as Branch'),
                DB::raw("DATE_FORMAT(vs.visit_service_start, '%d-%m-%Y %H:%i:%s') as Service_Start"),
                DB::raw("DATE_FORMAT(i.invoice_date, '%d-%m-%Y %H:%i:%s') as Invoiced_On"),
                'i.customer_name',
                'i.customer_cell',
                DB::raw("CASE 
                    WHEN vs.visit_service_status = 'Cancelled' THEN 'Yes' 
                    ELSE 'No' 
                END as Removed"),
                'vs.s_type as Type',
                'vs.s_category as Category',
                'vs.service_id as Service Code',
                DB::RAW('concat("<h4><span class=\"badge badge-outline-success\">", vs.service_name, "</span></h4>") as Service'),
                DB::raw("
                    GROUP_CONCAT(
                        CONCAT(
                            ist.staff_name, ' ',
                            CASE 
                                WHEN ist.additional_staff = 'Yes' THEN '(Help)' 
                                ELSE '' 
                            END
                        )
                        ORDER BY ist.additional_staff DESC
                        SEPARATOR '<br>'
                    ) as staff
                "),
                'vs.s_rate as Price',
                DB::RAW('ROuND(idat.service_addition, 2) as total_addition'),
                DB::RAW('ROUND(idat.discount, 2) as total_discount'),
                'idat.discount_type',
                DB::RAW('ROUND(idat.discounted_price + idat.service_addition, 2) as total_discounted_price'),
                DB::raw("
                    GROUP_CONCAT(
                        ist.commission 
                        ORDER BY ist.additional_staff DESC 
                        SEPARATOR '<br>'
                    ) as commission
                "),
            ])
            ->join('customer_visits as cv', 'cv.id_customer_visits', '=', 'vs.customer_visit_id')
            ->leftJoin('invoice_details as idat', 'vs.id_visit_services', '=', 'idat.visit_service_id')
            ->leftJoin('invoice_staff as ist', 'idat.id_invoice_details', '=', 'ist.invoice_detail_id')
            ->leftJoin('invoice as i', function($join) {
                $join->on('idat.invoice_id', '=', 'i.id_invoice')
                     ->where('i.invoice_status', 'valid');
            })
            ->join('business as b', 'cv.business_id', '=', 'b.id_business')
            ->where('i.invoice_date', '>=', $start_date . ' 00:00:00')
            ->where('i.invoice_date', '<=', $end_date . ' 23:59:59')
            ->where('cv.visit_status', 'invoiced')
            ->groupBy('vs.id_visit_services', 'id_business');

            if($isShUser) {
                $service_sale_detail = $service_sale_detail->where('invoice_seq', '>', 0);
            }
            
            //filter by business type
            if($request->business_type_id != null && $request->business_type_id != ''){
                $service_sale_detail = $service_sale_detail->where('b.business_type_id', $request->business_type_id);
            }

            if($request->business_id != '' && $request->business_id != null){               
                $service_sale_detail = $service_sale_detail->where('cv.business_id', $request->business_id);
            } else if($request->business_id == '' || $request->business_id == null) { 
                if(session('ho')=="Yes"){
                    //get all businesses with common services
                    $business_ids = DB::table('business')->where('common_services', 'Yes')->pluck('id_business')->toArray();                   
                    $service_sale_detail = $service_sale_detail->whereIn('cv.business_id', $business_ids);
                } else {
                    $service_sale_detail = $service_sale_detail->where('cv.business_id', session('business_id'));
                }
            } else {
                $service_sale_detail = $service_sale_detail->where('cv.business_id', session('business_id'));
            }

            //get service types and package types filter
            if($request->service_type && $request->service_type != ''){
                $service_sale_detail = $service_sale_detail->where('vs.id_service_type', $request->service_type);
            }
            if($request->service_category && $request->service_category != ''){
                $service_sale_detail = $service_sale_detail->where('vs.id_service_category', $request->service_category);
            }
            if($request->service_department && $request->service_department != ''){
                
                $service_sale_detail = $service_sale_detail->where('bs.service_department', $request->service_department);
            }


           $service_sale_detail = $service_sale_detail->get();
            //    / return $service_sale_detail;
        $data = $service_sale_detail;
        $reportID=$request->id;
        $reportName = 'Service Sale Detail Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        //get service types, package types and service departments for filters
        $service_types1 = DB::table('service_type')
        ->select('id_service_types', 'service_type')
        ->where('service_type_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            //pluck ho business id
            $ho_business_id = DB::table('business')->where('ho', 'Yes')->value('id_business');
            $service_types1 = $service_types1->where('business_id', $ho_business_id);
        } else {
            $service_types1 = $service_types1->where('business_id', session('business_id'));
        }
        $service_types1 = $service_types1->get();

        $package_types = DB::table('package_type')
        ->select(
            DB::raw('id_package_type as id_service_types'),
            DB::raw("CONCAT(service_type, ' (Package)') as service_type")
        )
        ->where('service_type_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $package_types = $package_types->where('business_id', $ho_business_id);
        } else {
            $package_types = $package_types->where('business_id', session('business_id'));
        }

        $package_types = $package_types->get();

        $service_types = $service_types1->merge($package_types)->sortBy('service_type');
    
        $service_categories1 = DB::table('service_category')
        ->select('id_service_category', 'service_category', 'service_type')
        ->join('service_type', 'service_category.service_type_id', '=', 'service_type.id_service_types')
        ->where('service_category_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $service_categories1 = $service_categories1->where('service_category.business_id', $ho_business_id);
        } else {
            $service_categories1 = $service_categories1->where('service_category.business_id', session('business_id'));
        }

        $service_categories1 = $service_categories1->get();
        $package_categories = DB::table('package_category')
        ->select(
            DB::raw('id_package_category as id_service_category'),
            DB::raw("CONCAT(service_category, ' (Package)') as service_category"),
            DB::RAW('package_type.service_type as service_type')
        )
        ->join('package_type', 'package_category.package_type_id', '=', 'package_type.id_package_type')
        ->where('service_category_active', 'Yes');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $package_categories = $package_categories->where('package_category.business_id', $ho_business_id);
        } else {
            $package_categories = $package_categories->where('package_category.business_id', session('business_id'));
        }
        $package_categories = $package_categories->get();

        $service_categories = $service_categories1->merge($package_categories)->sortBy('service_category');

        $service_departments = DB::table('service_department')
        ->select('id_service_department', 'service_department')
        ->where('department_status', 'Active');
        if(session('common_services')=="Yes" || session('ho')=="Yes"){
            $service_departments = $service_departments->where('business_id', $ho_business_id);
        } else {
            $service_departments = $service_departments->where('business_id', session('business_id'));
        }
        $service_departments = $service_departments->get();

        //////////


        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'service_types', 'service_categories', 'service_departments',     'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }

    }

    // Report 18: Vouchers Sold Report
    public function gift_vouchers_sold(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }
        
        $vouchers_sold = DB::table('order_vouchers')
            ->select([
                'order_vouchers.id_order_vouchers as Voucher_ID',
                DB::raw('business.business_name as branch'),
                DB::raw('voucher_number as "Voucher #"'),
                DB::raw('order_vouchers.type as "Voucher Type"'),
                DB::raw('REPLACE(service_names, "|", "<br>") as service_names'),
                DB::raw('date_format(valid_until, "%d-%m-%Y") as "Validity"'),
                'voucher_value',
                DB::raw('voucher_tax as tax_amount'),
                DB::raw('order_vouchers.amount as voucher_amount'),
                'remaining_amount',
                'paid_cash',
                'paid_card',
                'paid_check',
                'paid_voucher',
                'paid_online'
            ])
            ->join('business', 'order_vouchers.business_id', '=', 'business.id_business')
           
            ->whereBetween('voucher_date', [
                $start_date." 00:00:00",
                $end_date." 23:59:59"
            ]);
            //filter by business type
            if($request->business_type_id != null && $request->business_type_id != ''){
                $vouchers_sold = $vouchers_sold->where('b.business_type_id', $request->business_type_id);
            }

            if($request->business_id != '' && $request->business_id != null){               
                $vouchers_sold = $vouchers_sold->where('order_vouchers.business_id', $request->business_id);
            } else if($request->business_id == '' || $request->business_id == null) { 
                if(session('ho')=="Yes"){
                    //get all businesses with common services
                    $business_ids = DB::table('business')->where('common_services', 'Yes')->pluck('id_business')->toArray();                   
                    $vouchers_sold = $vouchers_sold->whereIn('order_vouchers.business_id', $business_ids);
                } else {
                    $vouchers_sold = $vouchers_sold->where('order_vouchers.business_id', session('business_id'));
                }
            } else {
                $vouchers_sold = $vouchers_sold->where('order_vouchers.business_id', session('business_id'));
            }
        
        $vouchers_sold = $vouchers_sold->get();
                
        $data = $vouchers_sold;
        $reportID=$request->id;
        $reportName = 'Vouchers Sold Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => 'Vouchers Sold Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    // Report 20: Payment Breakup Services Report
    public function payment_breakup(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //today date
        }
    
        $business_id = $request->business_id ?? session('business_id');

        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $business_id;
        }

        $result = DB::table('invoice as i')
            ->select([
                DB::raw('COUNT(DISTINCT i.id_invoice) as Total_invoices'),
                DB::raw('b.business_name as Branch'),
                DB::raw("DATE_FORMAT(i.invoice_date, '%d-%m-%Y') as invoice_date"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN avd.debit ELSE 0 END) AS Cash_In"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN avd.credit ELSE 0 END) as Cash_out"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_cash' THEN avd.debit - avd.credit ELSE 0 END) as Cash_Total"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_card' THEN avd.debit ELSE 0 END) as Card_Total"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_check' THEN avd.debit ELSE 0 END) as Check_Total"),
                DB::raw("SUM(CASE WHEN aem.entity_name = 'paid_online' THEN avd.debit ELSE 0 END) as Online_Total"),
            ])
            ->join('account_vouchers as av', 'i.id_invoice', '=', 'av.invoice_id')
            ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('account_heads as ah', 'ah.id_account_heads', '=', 'avd.account_head_id')
            ->join('account_event_mapping as aem', function ($join) use ($common_business_id) {
                $join->on('aem.account_head_id', '=', 'avd.account_head_id')
                    ->where('aem.business_id', '=', $common_business_id)
                    ->where('aem.account_event_id', '=', 1);
            })
            ->join('business as b', 'b.id_business', '=', 'i.business_id')
            ->where('i.invoice_status', 'valid')
            ->where('av.voucher_status', 'Active')          
            ->whereBetween('av.voucher_date', [
                $start_date. ' 00:00:00',
                $end_date.' 23:59:59'
            ])
            ->groupBy(DB::raw('DATE(i.invoice_date)'));
        //filter by invoice type
        if($request->invoice_type != null && $request->invoice_type != '' && $request->invoice_type != 0){
            $result = $result->where('i.invoice_type', $request->invoice_type);
        }

        //filter by business type
        if($request->business_type_id != null && $request->business_type_id != ''){
            $result = $result->where('b.business_type_id', $request->business_type_id);
        }

        //filter by business id
        if($request->business_id != '' && $request->business_id != null){
            $result = $result->where('i.business_id', $request->business_id);
        } else if($request->business_id == '' || $request->business_id == null) { 
            if(session('ho')=="Yes"){
                //get all businesses with common services
                $business_ids = DB::table('business')->where('common_services', 'Yes')->pluck('id_business')->toArray();                   
                $result = $result->whereIn('i.business_id', $business_ids);
            } else {
                $result = $result->where('i.business_id', session('business_id'));
            }
        } else {
            $result = $result->where('i.business_id', session('business_id'));
        }

        $result = $result->get();

        $data = $result;
        $reportID=$request->id;
        $reportName = 'Payment Breakup Services Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => 'Payment Breakup Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }

    }

    // Report 22: Receivables From Open Visits Report
    public function receivables_from_open_visits(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $results = DB::table('customer_visits as cv')
            ->select(
                'cv.id_customer_visits as Visit_ID',
                'cv.visit_status as Visit_Status',
                DB::raw('b.business_name as Branch'),
                "c.id_customers as Customer_ID",
                'c.customer_name',
                'c.customer_cell',
                'c.customer_email',
                'vs.s_type as Type',
                'vs.s_category as Category',
                'vs.service_name as Service',
                'vs.s_rate as Price',
                DB::raw("DATE_FORMAT(vs.visit_service_start, '%d-%m-%Y %H:%i') as visit_service_start"),
                DB::raw("CASE 
                            WHEN vs.visit_service_status != 'Active' 
                            THEN 'Yes'
                            WHEN cv.visit_status != 'open' AND cv.visit_status != 'invoiced' 
                            THEN 'Yes' 
                            ELSE '' 
                        END as Cancelled"),
                'vss.staff_name',
                'vss.additional_staff',
                'vss.requested',
                DB::raw("CASE 
                            WHEN vss.visit_service_staff_status != 'Active' 
                            THEN 'Yes' 
                            ELSE '' 
                        END as 'Removed'")
            )
            ->join('visit_services as vs', 'vs.customer_visit_id', '=', 'cv.id_customer_visits')
            ->join('visit_service_staffs as vss', 'vss.visit_service_id', '=', 'vs.id_visit_services')
            ->join('customers as c', 'c.id_customers', '=', 'cv.customer_id')
            ->join('business as b', 'cv.business_id', '=', 'b.id_business')
            
            ->whereBetween('vs.visit_service_start', [
                $start_date. ' 00:00:00',
                $end_date.' 23:59:59'
            ]);
            //filter by business type
            if($request->business_type_id != null && $request->business_type_id != ''){
                $results = $results->where('b.business_type_id', $request->business_type_id);
            }
            if($request->business_id != '' && $request->business_id != null){               
                $results = $results->where('cv.business_id', $request->business_id);
            } else if($request->business_id == '' || $request->business_id == null) { 
                if(session('ho')=="Yes"){
                    //get all businesses with common services
                    $business_ids = DB::table('business')->where('common_services', 'Yes')->pluck('id_business')->toArray();                   
                    $results = $results->whereIn('cv.business_id', $business_ids);
                } else {
                    $results = $results->where('cv.business_id', session('business_id'));
                }
            } else {
                $results = $results->where('cv.business_id', session('business_id'));
            }

            // filter by visit  status if $request->visit_status is provided
            if($request->visit_status && $request->visit_status != '' && $request->visit_status != '0'){
                if($request->visit_status == 'open'){
                    $results = $results->where('cv.visit_status', 'open');
                } else if($request->visit_status == 'invoiced'){
                    $results = $results->where('cv.visit_status', 'invoiced');
                } else {
                    $results = $results->whereNotIn('cv.visit_status', ['open', 'invoiced']);
                }                
            } else {
                //default filter to get only open and invoiced visits
                $results = $results->where('cv.visit_status', 'open');
            }

        $results = $results->get();
            
        $data = $results;
        $reportID=$request->id;
        $reportName = 'Receivables From Open Visits Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => 'Receivables From Open Visits Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }


    }


    // Report 23: Services Advances Report
    public function service_advances(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $result = DB::table('visit_advance as va')
            ->select(
                'va.id_visit_advance as Advance_ID',
                'cv.id_customer_visits as Visit_ID',
                DB::raw("DATE_FORMAT(va.advance_date, '%d-%m-%Y') as advance_date"),
                'c.id_customers as Customer_ID',
                'c.customer_name',
                'c.customer_cell',
                'c.customer_email',
                
                DB::raw('GROUP_CONCAT(DISTINCT vs.s_type) as services'),
                DB::raw("va.advance_user as `Added By`"),
                DB::raw("
                    CASE 
                        WHEN va.advance_status != 'Active' THEN 'Yes'
                        WHEN cv.visit_status NOT IN ('open', 'invoiced') THEN 'Yes'
                        ELSE 'No'
                    END as Cancelled
                "),
                'va.advance_amount'
            )
            ->join('customer_visits as cv', 'cv.id_customer_visits', '=', 'va.customer_visit_id')
            ->join('visit_services as vs', 'vs.customer_visit_id', '=', 'cv.id_customer_visits')
            ->join('customers as c', 'c.id_customers', '=', 'cv.customer_id')
            ->join('account_vouchers as av', 'av.visit_advance_id', '=', 'va.id_visit_advance')
            ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->where('va.advance_type', '=', 'service')
            ->whereBetween('va.advance_date', [
                $start_date.' 00:00:00',
                $end_date.' 23:59:59'
            ])
            ->groupBy(
                'va.id_visit_advance',
                'cv.id_customer_visits'
            );

        //filter by business type
        if($request->business_type_id != null && $request->business_type_id != ''){
            $result = $result->where('b.business_type_id', $request->business_type_id);
        }

        //filter by business id
        if($request->business_id != '' && $request->business_id != null){
            $result = $result->where('cv.business_id', $request->business_id);
        } else if($request->business_id == '' || $request->business_id == null) { 
            if(session('ho')=="Yes"){
                //get all businesses with common services
                $business_ids = DB::table('business')->where('common_services', 'Yes')->pluck('id_business')->toArray();                   
                $result = $result->whereIn('cv.business_id', $business_ids);
            } else {
                $result = $result->where('cv.business_id', session('business_id'));
            }
        } else {
            $result = $result->where('cv.business_id', session('business_id'));
        }

        $result = $result->get();
        //print_r($result); exit;

        $data = $result;
        $reportID=$request->id;
        $reportName = 'Advance Against Services Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_services')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_services', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;

        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => 'Services Advances Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }

    }

    //Report 24: Retained Amount on Invoices Report
    public function retained_amount(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }
        $business_id = $request->business_id ?? session('business_id');

        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $business_id;
        }

        //get retained account head using account event mapping  for event = 1
        $retained_account_head = DB::table('account_event_mapping')
            ->where('business_id', $common_business_id)
            ->where('account_event_id', 1)
            ->where('entity_name', 'retained_amount')
            ->value('account_head_id');

        $retained_amount = DB::table('account_vouchers as av')
            ->select(
                'av.id_account_vouchers as Account_Voucher_ID',
                'av.invoice_id as Invoice_ID',                
                DB::raw('b.business_name as Branch'),
                DB::raw("DATE_FORMAT(av.voucher_date, '%d-%m-%Y') as Account_Voucher_Date"),
                'c.id_customers as Customer_ID',
                'c.customer_name',
                'c.customer_cell',
                'av.created_by as By',
                DB::raw('concat(left(av.description,20), "...") as Description'),
                DB::raw('SUM(avd.debit) as Retained_Amount'),
                DB::raw('SUM(avd.credit) as Retained_Amount_Used')
            )            
            ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
            ->join('customers as c', 'av.business_partner_id', '=', 'c.id_customers')
            ->join('business as b', 'av.business_id', '=', 'b.id_business')
            ->where('av.business_partner', '1') //customer
            ->where('avd.account_head_id', $retained_account_head)
            ->where('av.voucher_status', 'Active')
            ->whereDate('av.voucher_date', '>=', $start_date)
            ->whereDate('av.voucher_date', '<=', $end_date)
            ->groupBy('av.invoice_id', 'av.id_account_vouchers', 'c.id_customers');

        if($request->business_id && $request->business_id != ''){
            $retained_amount = $retained_amount->where('i.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
            }
        }

        $retained_amount = $retained_amount->get();

        $data = $retained_amount;
        $reportID=$request->id;
        $reportName = 'Retained Amount on Invoices Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;

        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => 'Retained Amount on Invoices Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }

    }

    // Report 26: Goods Received Notes Report
    public function open_recovery_invoices(Request $request)
    {
         $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }
        $business_id = $request->business_id ?? session('business_id');

        if(session('ho_accounts')=='Yes'){
            $common_business_id = Business::where('business.ho', 'Yes')->value('id_business');
        } else {
            $common_business_id = $business_id;
        }

       
        //get account head from account event mapping for event = 1 and entity_name = 'accounts_receivable' 
         $retained_account_head = DB::table('account_event_mapping')
            ->where('business_id', $common_business_id)
            ->where('account_event_id', 1)
            ->where('entity_name', 'balance')
            ->value('account_head_id');

        $recovery_invoices = DB::table('invoice as i')
        ->select(
            DB::raw('i.id_invoice as Invoice_ID'),
            DB::raw("date_format(i.invoice_date, '%d-%m-%Y') as invoice_Date"),
            DB::raw('i.customer_name'),
            DB::raw('i.customer_cell'),
            DB::raw('i.reference_invoice_number'),
            DB::raw('i.balance as Total_Balance'),
            DB::raw('i.is_recovery'),
            DB::raw('sum(avd.debit) as Total_Receivable'),
            DB::raw('sum(avd.credit) as Total_Recovered')
        )
        ->join('account_vouchers as av', 'i.id_invoice', '=', 'av.invoice_id')
        ->join('account_voucher_detail as avd', 'avd.account_voucher_id', '=', 'av.id_account_vouchers')
        ->join('business as b', 'i.business_id', '=', 'b.id_business')
        ->where('avd.account_head_id', $retained_account_head)
        ->where('i.invoice_status', 'valid')
        ->where('i.is_recovery', 'Yes')
        ->where('av.voucher_status', 'Active');

        $recovery_invoices = $recovery_invoices->whereDate('i.invoice_date', '>=', $start_date)
        ->whereDate('i.invoice_date', '<=', $end_date);
        //filter by business type
        if($request->business_type_id != null && $request->business_type_id != ''){
            $recovery_invoices = $recovery_invoices->where('b.business_type_id', $request->business_type_id);
        }

        if($request->business_id && $request->business_id != ''){
            $recovery_invoices = $recovery_invoices->where('i.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $recovery_invoices = $recovery_invoices->whereIn('i.business_id', $business_ids);
            } else {
                $recovery_invoices = $recovery_invoices->where('i.business_id', session('business_id'));
            }
        } else {
            $recovery_invoices = $recovery_invoices->where('i.business_id', session('business_id'));
        }
        $recovery_invoices = $recovery_invoices->groupBy('i.id_invoice', 'i.invoice_date', 'i.customer_name', 'i.customer_cell', 'i.reference_invoice_number', 'i.balance', 'i.is_recovery');


        $recovery_invoices = $recovery_invoices->get();
        $data = $recovery_invoices;
        $reportID=$request->id;
        $reportName = 'Open Recovery Invoices Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $reportID)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => 'Open Recovery Invoices Report',
                'data' => $data,
                'reportID' => $reportID,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'reportID', 'start_date', 'end_date', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }

    }


    public function grn_details(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;

        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }
        //code for grn details report
        $grn_details = DB::table('goods_received_note')
        ->select(
            DB::RAW('goods_received_note.grn_id as "GRN ID"'), 
            DB::RAW('goods_received_note.grn_number as "GRN Number"'),
            DB::RAW('goods_received_note.purchase_order_id as "PO ID"'),
            DB::RAW('purchase_order.purchase_order_number as "PO Number"'), 
            DB::RAW('date_format(goods_received_note.grn_created_date, "%d-%m-%Y") as "GRN Date"'),
             DB::RAW('business.business_name as "Branch"'),
            DB::RAW('goods_received_note.grn_vendor_name as "Supplier"'),
            DB::RAW('GROUP_CONCAT(grn_details.grn_product_name SEPARATOR "<br>") as "Products"'),
            DB::RAW('GROUP_CONCAT(grn_details.grn_batch_id SEPARATOR ",") as "Batch IDs"'),
            DB::RAW('GROUP_CONCAT(grn_details.grn_qty_received SEPARATOR "<br>") as "Quantities Received"'),
            DB::RAW('GROUP_CONCAT(ROUND(return_notes.return_qty,2) SEPARATOR "<br>") as "Quantities Returned"'),
            DB::RAW('GROUP_CONCAT(ROUND(grn_details.grn_unit_price,2) SEPARATOR "<br>") as "Unit Prices"'),
            DB::RAW('SUM(grn_details.grn_qty_received)-SUM(IFNULL(return_notes.return_qty, 0)) as "Total Quantity Received"'),
            DB::RAW('ROUND(SUM((grn_details.grn_qty_received - IFNULL(return_notes.return_qty, 0)) * grn_details.grn_unit_price),2) as "Total Value"'),
           
            )
        ->join('business', 'goods_received_note.business_id', '=', 'business.id_business')
        ->join('grn_details', 'goods_received_note.grn_id', '=', 'grn_details.grn_id')
        ->leftJoin('purchase_order', 'goods_received_note.purchase_order_id', '=', 'purchase_order.idpurchase_order')
        //join return notes where return note grn_id = goods received note grn_id and return note . product_id = grn_details.grn_product_id
        ->leftJoin('return_notes', function($join) {
            $join->on('return_notes.grn_id', '=', 'goods_received_note.grn_id')
                 ->on('return_notes.product_id', '=', 'grn_details.grn_product_id');
        })
        ->whereDate('goods_received_note.grn_created_date', '>=', $start_date)
        ->whereDate('goods_received_note.grn_created_date', '<=', $end_date);
        if($request->business_id && $request->business_id != ''){
            $grn_details = $grn_details->where('goods_received_note.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $grn_details = $grn_details->whereIn('goods_received_note.business_id', $business_ids);
            } else {
                $grn_details = $grn_details->where('goods_received_note.business_id', session('business_id'));
            }
        } else {
            $grn_details = $grn_details->where('goods_received_note.business_id', session('business_id'));
        }
        $grn_details = $grn_details->groupBy('goods_received_note.grn_id', 'goods_received_note.grn_number',
            'goods_received_note.purchase_order_id', 'purchase_order.purchase_order_number',
            'goods_received_note.grn_created_date', 'business.business_name', 'goods_received_note.grn_vendor_name');

        $grn_details = $grn_details->get();

        $data = $grn_details;

        $business_id = $request->business_id ?? '';

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_types = DB::table('business_type')->get();

        $reportID = $request->id;
        $reportName = 'Goods Received Notes Report';

        $permission_type = $request->permission;
       
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $grn_details,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }
    }

    public function dispatch_note_details(Request $request)
    {

        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }
        
        $dispatch_note_details = DB::table('dispatch_notes')
        ->select(
            DB::RAW('dispatch_notes.id_dispatch_note as "Dispatch ID"'), 
            DB::RAW('date_format(dispatch_notes.dispatch_date, "%d-%m-%Y %H:%i") as "Dispatch Date"'),
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('customers.customer_name as "Customer"'),
            DB::RAW('customers.customer_cell as "Cell"'),
            DB::RAW('business_products.id_business_products as "Product Code"'),
            DB::RAW('business_products.product as "Product"'),
            DB::RAW('business_products.unit_type as "Container Type"'),
            DB::RAW('business_products.qty_per_unit as "Quantity Per Unit"'),
            DB::RAW('business_products.measure_unit as "Measure Unit"'),
            DB::RAW('dispatch_notes.batch_id as "Batch ID"'),
            DB::RAW('dispatch_notes.batch as "Batch No."'),
            DB::RAW('staff.staff_fullname as "Dispatched To"'),
            DB::RAW('ROUND(dispatch_notes.dispatch_qty, 4) as "Total Dispatched"'),
            DB::RAW('ROUND(dispatch_notes.dispatch_measure, 2) as "Total Measure Dispatched"'),
            DB::RAW('dispatch_notes.request_id as "Dispatch Request ID"'),
            DB::RAW('dispatch_notes.visit_id as "Dispatch Visit ID"'),
            DB::RAW('dispatch_notes.created_by as "Dispatched By"'),
            DB::RAW('dispatch_notes.dispatch_comment as "Comment"')
            )
        ->join('business', 'dispatch_notes.business_id', '=', 'business.id_business')
        ->join('business_products', 'dispatch_notes.product_id', '=', 'business_products.id_business_products')
        ->leftjoin('staff', 'dispatch_notes.dispatch_to_staff', '=', 'staff.id_staff')
        ->leftJoin('dispatch_request', function($join) {
            $join->on('dispatch_notes.request_id', '=', 'dispatch_request.id_dispatch_request');
        })
        ->leftJoin('customer_visits', function($join) {
            $join->on('dispatch_request.customer_visit_id', '=', 'customer_visits.id_customer_visits');
        })
        ->leftJoin('customers', function($join) {
            $join->on('customer_visits.customer_id', '=', 'customers.id_customers');
        })
        ->whereDate('dispatch_notes.dispatch_date', '>=', $start_date)
        ->whereDate('dispatch_notes.dispatch_date', '<=', $end_date);
        if($request->business_id && $request->business_id != ''){
            $dispatch_note_details = $dispatch_note_details->where('dispatch_notes.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $dispatch_note_details = $dispatch_note_details->whereIn('dispatch_notes.business_id', $business_ids);
            } else {
                $dispatch_note_details = $dispatch_note_details->where('dispatch_notes.business_id', session('business_id'));
            }
        } else {
            $dispatch_note_details = $dispatch_note_details->where('dispatch_notes.business_id', session('business_id'));
        }
        $dispatch_note_details = $dispatch_note_details->orderBy('dispatch_notes.dispatch_date', 'DESC');
        $data = $dispatch_note_details->get();

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }

        $business_id = $request->business_id ?? '';
        $business_types = DB::table('business_type')->get();

        $reportID = $request->id;
        $reportName = 'Dispatch Note Details Report';
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }
    }

    public function stock_status(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }

        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }
        //code for stock status report
        $stock_status = DB::table('business_products')
        ->select(
            DB::RAW('id_business_products as "Product ID"'), 
            DB::RAW('product as "Product Name"'),
            DB::RAW('id_business_products as "Product Code"'),
            DB::RAW('unit_type as "Container Type"'),
            DB::RAW('qty_per_unit as "Quantity Per Unit"'),
            DB::RAW('measure_unit as "Measure Unit"'),
            
            DB::RAW('ROUND(stock_measure, 2) as "Stock Measure"'),
            DB::RAW('reorder_level as "Reorder Level"'),
            DB::RAW('business.business_name as "Branch"')
            )
        ->join('business', 'business_products.business_id', '=', 'business.id_business')
        ->where('business_products.product_active', 'Yes')
        ->orderBy('business_products.product', 'ASC');

        if($request->business_id && $request->business_id != ''){
            $stock_status = $stock_status->where('business_products.business_id', $request->business_id);
        } else if($request->business_id && $request->business_id == '') { //get all common products businesses
            if(session('common_products')=="Yes" && session('ho')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $stock_status = $stock_status->whereIn('business_products.business_id', $business_ids);
            } else {
                $stock_status = $stock_status->where('business_products.business_id', session('business_id'));
            }
        } else {
            $stock_status = $stock_status->where('business_products.business_id', session('business_id'));
        }

        $data = $stock_status->get();

        $reportID = $request->id;
        $reportName = 'Stock Status Report';

        $business_id = $request->business_id ?? '';

        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }   
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_id' => $business_id,
                'permission_type' => $permission_type,
            ]);
        } else {
            
            return view('reports.report_page', compact('data', 'reportID', 'reportName', 'branches', 'business_id', 'business_types', 'permission_type'));
        }
    }

    public function adjustment_notes(Request $request)
    {
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
             $end_date = $start_date; //same date
        }

        //code for adjustment notes report
        $adjustment_notes = DB::table('adjustment_notes')
        ->select(
            DB::RAW('adjustment_notes.id_adjustment_notes as "Adjustment ID"'), 
            DB::RAW('date_format(adjustment_notes.adjustment_date, "%d-%m-%Y") as "Adjustment Date"'),
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('business_stores.business_store as "Store"'),
            DB::RAW('product_batch.id_batch as "Batch ID"'),
            DB::RAW('product_batch.batch_number as "Batch No."'),            
            DB::RAW('adjustment_notes.adjustment_remarks as "Reason"'),
            DB::RAW('business_products.id_business_products as "Product Code"'),
            DB::RAW('business_products.product as "Product"'),
            DB::RAW('ROUND(adjustment_notes.adjustment_qty, 4) as "Adjusted Quantity"'),
            DB::RAW('adjustment_notes.created_by as "Adjusted By"'),
            DB::RAW('adjustment_notes.adjustment_remarks as "Comment"')
            )
        ->join('product_batch', 'adjustment_notes.batch_id', '=', 'product_batch.id_batch')
        ->join('business_products', 'adjustment_notes.product_id', '=', 'business_products.id_business_products')
        ->join('business_stores', 'product_batch.store_id', '=', 'business_stores.id_business_stores')
        ->join('business', 'business_stores.business_id', '=', 'business.id_business')
        
        ->whereDate('adjustment_notes.adjustment_date', '>=', $start_date)
        ->whereDate('adjustment_notes.adjustment_date', '<=', $end_date);
        
        if($request->business_id && $request->business_id != ''){
            $adjustment_notes = $adjustment_notes->where('business_stores.business_id', $request->business_id);
        } else if(session('ho') == "Yes") { //get all common products businesses
            if(session('common_products')=="Yes"){
                $business_ids = DB::table('business')
                ->where('common_products', 'Yes')
                ->pluck('id_business')
                ->toArray();
    
                $adjustment_notes = $adjustment_notes->whereIn('business_stores.business_id', $business_ids);
            } else {
                $adjustment_notes = $adjustment_notes->where('business_stores.business_id', session('business_id'));
            }

        } else {
            $adjustment_notes = $adjustment_notes->where('business_stores.business_id', session('business_id'));
        }
        $adjustment_notes = $adjustment_notes->orderBy('adjustment_notes.adjustment_date', 'DESC');

        $data = $adjustment_notes->get();

        $reportID = $request->id;
        $reportName = 'Adjustment Notes Report';
        $business_id = $request->business_id ?? '';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
      
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();

        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();

        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'start_date' => $start_date,
                'end_date' => $end_date,
                'branches' => $branches,
                'business_id' => $business_id,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'request', 'branches', 'business_id', 'start_date', 'end_date', 'reportName', 'reportID', 'business_types', 'permission_type'));
        }

    }

    public function return_notes(Request $request)
    {
        //code for return notes report
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        if($request->start_date == null){
            $start_date = now()->subDay()->toDateString(); //yesterday date
        }
        if($request->end_date == null || $request->end_date == '' || $request->end_date == 'undefined'){
            $end_date = $start_date; //same date
        }

        $return_notes = DB::table('return_notes')
        ->select(
            DB::RAW('return_notes.id_return_notes as "Return ID"'), 
            DB::RAW('date_format(return_notes.return_date, "%d-%m-%Y") as "Return Date"'),
            DB::RAW('business.business_name as "Branch"'),
            DB::RAW('return_notes.grn_id as "GRN ID"'),
            DB::RAW('goods_received_note.purchase_order_id as "PO ID"'),
            DB::RAW('goods_received_note.grn_vendor_name as "Supplier"'),
            DB::RAW('return_notes.product_id as "Product ID"'),
            DB::RAW('business_products.product as "Product Name"'),
            DB::RAW('ROUND(return_notes.return_qty, 4) as "Returned Quantity"'),
            DB::RAW('users.user_name as "Returned By"')

            )
        ->join('business', 'return_notes.business_id', '=', 'business.id_business')
        ->join('business_products', 'return_notes.product_id', '=', 'business_products.id_business_products')
        ->join('users', 'return_notes.user_id', '=', 'users.id_users')
        ->join('goods_received_note', 'return_notes.grn_id', '=', 'goods_received_note.grn_id')
        ->whereDate('return_notes.return_date', '>=', $start_date)
        ->whereDate('return_notes.return_date', '<=', $end_date)
        ->orderBy('return_notes.return_date', 'DESC');
        $data = $return_notes->get();
        $reportID = $request->id;
        $reportName = 'Return Notes Report';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type'));
        }
    }

    public function store_wise_stock(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;

        $ProductsStockService = new ProductsStockService();
        $finalProducts = [];
        if($request->datareport){
                $stores = DB::table('business_stores')
                    ->join('business','business.id_business','business_stores.business_id')
                    ->select('id_business_stores', 'business_id', 'business_store');
                if ($business_id > 0) {
                    $stores->where('business_id', $business_id);
                }
                if($business_id ==0){
                 $stores->where('common_products', 'Yes');
                }
                $stores = $stores->get();
                if ($stores->isEmpty()) {
                        return "No stores found'";
                }
                foreach ($stores as $store) {
                    $productsStock = $ProductsStockService->gettingProductsStock(
                        $store->id_business_stores,
                        $store->business_id
                    );
                    foreach ($productsStock as $stock) {
                        $pid = $stock['product_id'];
                        $business_store = $store->business_store;
                        // create base row once
                        if (!isset($finalProducts[$pid])) {
                            $finalProducts[$pid] = [
                                'Branch' => $stock['business_name'],
                                'Brand' => $stock['business_brand_name'],
                                'product id' => $pid,
                                'Product' => $stock['product'],
                                'Pro' => $stock['is_professional'],
                                'Category' => $stock['category'],
                                'unit_type' => $stock['unit_type'],
                                'qty_per_unit' => $stock['qty_per_unit'],
                                'measure_unit' => $stock['measure_unit'],
                            ];

                            foreach ($stores as $s) {
                                $finalProducts[$pid]["{$s->business_store} stock"] = 0;
                                $finalProducts[$pid]["{$s->business_store} value"] = 0;
                            }

                            // initialize totals at the end
                            $finalProducts[$pid]['Total Stock'] = 0;
                            $finalProducts[$pid]['Total Value'] = 0;
                        }

                        // fill current store data
                        $finalProducts[$pid]["{$business_store} stock"] = (float)$stock['instock'];
                        $finalProducts[$pid]["{$business_store} value"] = (float)$stock['total_value'];

                        // add to totals
                        $finalProducts[$pid]['Total Stock'] += (float)$stock['instock'];
                        $finalProducts[$pid]['Total Value'] += (float)$stock['total_value'];
                    }
                }

                $data = array_values($finalProducts);
        }else{
            $data =[];
        }        

        $reportID = $request->id;
        $reportName = 'Store Wise Stock';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
             $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public function stock_aging_report(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;

        $lastUpdatedAt = DB::table('mis_store_product_stock')
        ->selectRaw("DATE_FORMAT(MAX(updated_at), '%d-%b-%Y, %h:%i %p') as last_updated_at")
        ->value('last_updated_at') ?? '';
        if($request->datareport){

        $agingReport = DB::table('mis_store_product_stock')
            ->join('business_products','business_products.id_business_products','mis_store_product_stock.product_id')    
            ->select(
                'business_name as branch',
                'store as store',
                'business_brand_name as brand',
                'product_id',
                'mis_store_product_stock.product as product_name',
                DB::raw("
                        CASE 
                            WHEN business_products.professional = 'y' 
                                THEN 'Professional'
                            ELSE 'Retail'
                        END AS Pro
                "),
                'mis_store_product_stock.category',
                DB::raw("SUM(CASE WHEN DATEDIFF(CURDATE(), DATE(batch_date)) <= 30 THEN instock ELSE 0 END) as 'stock 0-30' "),
                DB::raw("SUM(CASE WHEN DATEDIFF(CURDATE(), DATE(batch_date)) BETWEEN 31 AND 60 THEN instock ELSE 0 END) as 'stock 31-60'"),
                DB::raw("SUM(CASE WHEN DATEDIFF(CURDATE(), DATE(batch_date)) BETWEEN 61 AND 90 THEN instock ELSE 0 END) as 'stock 61-90'"),
                DB::raw("SUM(CASE WHEN DATEDIFF(CURDATE(), DATE(batch_date)) > 90 THEN instock ELSE 0 END) as 'stock above 90' "),
                DB::raw("ROUND(SUM(instock), 2) as 'Total Stock'"),
                DB::raw("ROUND(SUM(instock * average_batch_unit_price), 2) as 'Total Value'")

            )
            ->where('instock', '>', 0);

        if ($business_id > 0) {
            $agingReport->where('business_id', $business_id);
        }

        $data = $agingReport
            ->groupBy(
                'business_name',
                'store_id',
                'product_id'
            )
            ->orderBy('store_id')
            ->orderBy('product_id')
            ->get();

        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Stock Aging Report (Last Updated at '.$lastUpdatedAt.')';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }


    public function category_wise_stock_sales(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;
        $ProductsStockService = new ProductsStockService();

        $lastUpdatedAt = DB::table('mis_store_product_stock')
        ->selectRaw("DATE_FORMAT(MAX(updated_at), '%d-%b-%Y, %h:%i %p') as last_updated_at")
        ->value('last_updated_at') ?? '';

        if($request->datareport){
        $stores = DB::table('business_stores')
            ->join('business','business.id_business','business_stores.business_id')    
            ->select('id_business_stores', 'business_id');
        if ($business_id > 0) {
            $stores->where('business_id', $business_id);
        }
        if($business_id ==0){
         $stores->where('common_products', 'Yes');
        }
        $stores = $stores->get();
        $finalData = collect();
        foreach ($stores as $store) {

            $productsData = $ProductsStockService
                ->getInvoiceSaleAndFranchiseSaleQty(
                    $store->id_business_stores,
                    $store->business_id,
                    $start_date,
                    $end_date
                );

            $productsData = $productsData->map(function ($item) {

                $item['category'] = empty(trim($item['category']))
                    ? 'Uncategorized'
                    : trim($item['category']);

                $item['total_sale_qty'] =
                    ($item['sold_qty'] ?? 0) + ($item['franchise_sale_qty'] ?? 0);

                return $item;
            });

            $finalData = $finalData->merge($productsData);
        }



        $groupedData = $finalData->groupBy('category')->map(function ($items, $category) {

            $soldQty      = $items->sum('sold_qty');
            $franchiseQty = $items->sum('franchise_sale_qty');
            $totalQty     = $items->sum('total_sale_qty');

           return [
                'category'                  => $category,
                'stock_sold_to_customer'    => $soldQty,
                'stock_sold_to_franchise'   => $franchiseQty,
                'total_stock_sold'          => $totalQty,
            ];

        })


        ->filter(function ($item) {
            return $item['total_stock_sold'] > 0;
        });

        $grandTotalSale = $groupedData->sum('total_stock_sold');

        $data = $groupedData->map(function ($item) use ($grandTotalSale) {

            $item['sale_percentage'] = $grandTotalSale > 0
                ? round(($item['total_stock_sold'] / $grandTotalSale) * 100, 2)
                : 0;

            return $item;
        })->values();
        
        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Category-wise Stock Sales (Last Updated at '.$lastUpdatedAt.')';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }
    
    public function product_stock_status(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;
        $stroe_id =$request->stroe_id ?? 0;
        $ProductsStockService = new ProductsStockService();

        $lastUpdatedAt = DB::table('mis_store_product_stock')
        ->selectRaw("DATE_FORMAT(MAX(updated_at), '%d-%b-%Y, %h:%i %p') as last_updated_at")
        ->value('last_updated_at') ?? '';

        if($request->datareport){
        $stores = DB::table('business_stores')
            ->join('business','business.id_business','business_stores.business_id')    
            ->select('id_business_stores', 'business_id');

        if ($business_id > 0) {
            $stores->where('business_id', $business_id);
        }
        
        if($business_id ==0){
          $stores->where('common_products', 'Yes');
        }

        $stores = $stores->get();
        $finalData = collect();
        $data = $ProductsStockService->gettingProductsStockWithBF($stroe_id, $business_id,$start_date, $end_date);
        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Product Stock Status';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public function franchise_sales_summary(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;

        if($request->datareport){

        $account_receivable =  AccountsController::get_event_account('37', 'total_receivable');
        $account_receivable_id = $account_receivable->account_head_id;

        $account_discount =  AccountsController::get_event_account('37', 'invoice_discount');
        $account_discount_id = $account_discount->account_head_id;

        $account_total_invoice_value =  AccountsController::get_event_account('37', 'total_invoice_value');
        $account_total_invoice_value_id = $account_total_invoice_value->account_head_id;

        $account_other_charges =  AccountsController::get_event_account('37', 'other_charges');
        $account_other_charges_id = $account_other_charges->account_head_id;

        $account_sales_tax =  AccountsController::get_event_account('37', 'sales_tax');
        $account_sales_tax_id = $account_sales_tax->account_head_id;
        // Ye QUERY HAI PERFECT WORK KAR RAHI HAI LAKIN COMMENT ES WAJH SY KI HAI 
        //AGR ZERO VALUE PER KOI ODER AATA HAI TOU US KA VOUCHER NAHI BNATA TOU PHIR QTY B NAHI AATI  
        // $sql=" SELECT
        //         fo.franchise_id,
        //         f.franchise_name,

        //         IFNULL(fop.total_qty, 0) AS total_sold_qty,
        //         IFNULL(ret.return_qty, 0) AS total_return_qty,
        //         (IFNULL(fop.total_qty, 0) - IFNULL(ret.return_qty, 0)) AS actual_total_sold_qty,

        //         ROUND(
        //             SUM(CASE 
        //                 WHEN avd.account_head_id = $account_total_invoice_value_id 
        //                 THEN avd.credit - avd.debit ELSE 0 END), 2
        //         ) AS total_value,

        //         ROUND(
        //             SUM(CASE 
        //                 WHEN avd.account_head_id = $account_discount_id 
        //                 THEN avd.debit - avd.credit ELSE 0 END), 2
        //         ) AS total_discount,

        //         ROUND(
        //             SUM(CASE 
        //                 WHEN avd.account_head_id = $account_other_charges_id 
        //                 THEN avd.credit - avd.debit ELSE 0 END), 2
        //         ) AS total_other_charges,

        //         ROUND(
        //             SUM(CASE 
        //                 WHEN avd.account_head_id = $account_sales_tax_id 
        //                 THEN avd.credit - avd.debit ELSE 0 END), 2
        //         ) AS total_sales_tax,

        //         ROUND(
        //             (
        //                 SUM(CASE WHEN avd.account_head_id = $account_total_invoice_value_id 
        //                     THEN avd.credit - avd.debit ELSE 0 END)
        //                 - SUM(CASE WHEN avd.account_head_id = $account_discount_id 
        //                     THEN avd.debit - avd.credit ELSE 0 END)
        //                 + SUM(CASE WHEN avd.account_head_id = $account_other_charges_id 
        //                     THEN avd.credit - avd.debit ELSE 0 END)
        //                 + SUM(CASE WHEN avd.account_head_id = $account_sales_tax_id 
        //                     THEN avd.credit - avd.debit ELSE 0 END)
        //             ), 2
        //         ) AS net_amount

        //     FROM account_vouchers av
        //     INNER JOIN account_voucher_detail avd 
        //         ON avd.account_voucher_id = av.id_account_vouchers
        //     INNER JOIN franchise_orders fo 
        //         ON fo.id_franchise_orders = av.franchise_order_id
        //     LEFT JOIN franchises f 
        //         ON f.id_franchises = fo.franchise_id
        //     LEFT JOIN (
        //         SELECT 
        //             franchise_id,
        //             SUM(qty) AS total_qty,
        //             SUM(qty * unit_price) AS total_value
        //         FROM franchise_order_products
        //         JOIN franchise_orders 
        //             ON franchise_orders.id_franchise_orders = franchise_order_products.franchise_order_id
        //         GROUP BY franchise_orders.franchise_id
        //     ) fop ON fop.franchise_id = fo.franchise_id
        //     LEFT JOIN (
        //         SELECT 
        //             fo2.franchise_id,
        //             SUM(IFNULL(adjustment_qty, 0)) AS return_qty
        //         FROM adjustment_notes an
        //         JOIN franchise_return_notes frn 
        //             ON frn.id_franchise_return_notes = an.franchise_return_note_id
        //         JOIN franchise_orders fo2 
        //             ON fo2.id_franchise_orders = frn.franchise_order_id
        //         WHERE LOWER(frn.status) != 'cancelled'
        //         GROUP BY fo2.franchise_id
        //     ) ret ON ret.franchise_id = fo.franchise_id

        //     WHERE 
        //         LOWER(av.voucher_status) = 'active'
        //         AND DATE(av.voucher_date) BETWEEN '$start_date' AND '$end_date'
        //         AND fo.business_id = $business_id
        //         AND LOWER(fo.order_status) != 'cancelled'
        //     GROUP BY fo.franchise_id;";  
        $voucher_business_condition ='';   
        $franchise_order_business_condition ='';   
        if($business_id > 0){
            $voucher_business_condition = " AND av.business_id = $business_id ";
            $franchise_order_business_condition = " AND fo.business_id = $business_id ";

        }

      $sql = "
            SELECT
                business.business_name AS branch,    
                fo.franchise_id,
                f.franchise_name AS franchise_name,
                f.franchise_owner AS franchise_owner,
                SUM(IFNULL(franchise_order_products.qty, 0)) AS total_qty_sold,
                IFNULL(return_data.return_qty, 0) AS total_return_qty,
                (SUM(IFNULL(franchise_order_products.qty, 0)) - IFNULL(return_data.return_qty, 0)) AS actual_total_sold_qty,
                #SUM(IFNULL(franchise_order_products.qty * franchise_order_products.unit_price, 0)) AS total_qty_value,
                ROUND(IFNULL(av_tot.franchise_sale_value, 0), 2) AS total_value,
                ROUND(IFNULL(av_tot.total_discount, 0), 2) AS total_discount,
                ROUND(IFNULL(av_tot.franchise_sale_other_charges, 0), 2) AS total_other_charges,
                ROUND(IFNULL(av_tot.sales_tax, 0), 2) AS total_sales_tax,
                ROUND(
                    (
                        IFNULL(av_tot.franchise_sale_value, 0)
                        - IFNULL(av_tot.total_discount, 0)
                        + IFNULL(av_tot.franchise_sale_other_charges, 0)
                        + IFNULL(av_tot.sales_tax, 0)
                    ), 2
                ) AS net_amount
            FROM
                franchise_orders fo
            LEFT JOIN business ON business.id_business = fo.business_id    
            JOIN 
                franchise_order_products 
                ON franchise_order_products.franchise_order_id = fo.id_franchise_orders
            JOIN
                franchises f 
                ON f.id_franchises = fo.franchise_id
            LEFT JOIN (
                SELECT
                    franchise_orders.franchise_id,
                    COALESCE(SUM(CASE WHEN avd.account_head_id = $account_discount_id THEN avd.debit - avd.credit END), 0) AS total_discount,
                    COALESCE(SUM(CASE WHEN avd.account_head_id = $account_total_invoice_value_id THEN avd.credit - avd.debit END), 0) AS franchise_sale_value,
                    COALESCE(SUM(CASE WHEN avd.account_head_id = $account_other_charges_id THEN avd.credit - avd.debit END), 0) AS franchise_sale_other_charges,
                    COALESCE(SUM(CASE WHEN avd.account_head_id = $account_sales_tax_id THEN avd.credit - avd.debit END), 0) AS sales_tax
                FROM
                    account_voucher_detail avd
                JOIN
                    account_vouchers av 
                    ON av.id_account_vouchers = avd.account_voucher_id
                JOIN
                    franchise_orders 
                    ON franchise_orders.id_franchise_orders = av.franchise_order_id
                WHERE
                    LOWER(av.voucher_status) = 'active'
                    AND DATE(av.voucher_date) BETWEEN '$start_date' AND '$end_date'
                    $voucher_business_condition
                GROUP BY
                    franchise_orders.franchise_id
            ) AS av_tot 
                ON av_tot.franchise_id = fo.franchise_id
            LEFT JOIN (
                SELECT
                    franchise_orders.franchise_id,
                    SUM(IFNULL(adjustment_qty, 0)) AS return_qty
                FROM
                    adjustment_notes
                JOIN
                    franchise_return_notes 
                    ON franchise_return_notes.id_franchise_return_notes = adjustment_notes.franchise_return_note_id
                JOIN
                    franchise_orders 
                    ON franchise_orders.id_franchise_orders = franchise_return_notes.franchise_order_id
                WHERE
                    franchise_return_note_id > 0 
                    AND LOWER(franchise_return_notes.status) != 'cancelled'
                    AND DATE(franchise_return_notes.return_date) BETWEEN '$start_date' AND '$end_date' 
                GROUP BY
                    franchise_orders.franchise_id
            ) AS return_data 
                ON return_data.franchise_id = fo.franchise_id
            WHERE
             DATE(fo.invoice_date) BETWEEN '$start_date' AND '$end_date'
             AND LOWER(fo.order_status) != 'cancelled'
            $franchise_order_business_condition
            GROUP BY
               fo.business_id, fo.franchise_id
            ";
            $data = DB::select($sql);

        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Franchise Sales Summary';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public function customer_sales_summary(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;

        if($request->datareport){
            $sql = "SELECT  
                        business.business_name AS branch,    
                        invoice.customer_id,
                        invoice.customer_name AS customer_name,
                        IFNULL(invoice.customer_cell,'') as customer_cell, 
                        SUM(COALESCE(invoice_detail_data.invoice_qty, 0)) AS total_qty_sold,
                        SUM(COALESCE(invoice_detail_data.total_price, 0)) AS total_price,
                        SUM(COALESCE(invoice_detail_data.service_addition, 0)) AS total_additional_charges,
                        SUM(COALESCE(invoice_detail_data.total_price_after_other_charges, 0)) AS total_price_after_additional_charges,
                        SUM(COALESCE(invoice_detail_data.total_discount_amount, 0)) AS total_discount,
                        SUM(COALESCE(invoice_detail_data.discounted_price, 0)) AS total_discounted_amount,
                        SUM(COALESCE(invoice_detail_data.total_tax_amount, 0)) AS total_sales_tax,
                        SUM(COALESCE(invoice_detail_data.discounted_price, 0) + COALESCE(invoice_detail_data.total_tax_amount, 0)) AS net_amount

                    FROM invoice
                    LEFT JOIN business ON business.id_business = invoice.business_id
                    LEFT JOIN (
                        SELECT 
                            invoice_products.invoice_id,
                            SUM(invoice_products.invoice_qty) AS invoice_qty,
                            SUM(invoice_products.invoice_qty * invoice_products.price) AS total_price,
                            SUM(invoice_products.service_addition * invoice_products.invoice_qty) AS service_addition,
                            SUM(
                                invoice_products.invoice_qty * 
                                (invoice_products.price + invoice_products.service_addition)
                            ) AS total_price_after_other_charges,
                            SUM(invoice_products.invoice_qty * invoice_products.discount) AS total_discount_amount,
                            SUM(
                                invoice_products.invoice_qty *
                                ((invoice_products.price + invoice_products.service_addition) - invoice_products.discount)
                            ) AS discounted_price,
                            SUM(invoice_products.taxes) AS total_tax_amount
                        FROM invoice_products
                        GROUP BY invoice_products.invoice_id
                    ) AS invoice_detail_data 
                        ON invoice_detail_data.invoice_id = invoice.id_invoice

                    WHERE
                        DATE(invoice.invoice_date) BETWEEN ? AND ?
                        AND invoice.invoice_type = 'sale'
                        AND LOWER(invoice.invoice_status) = 'valid' ";
                 $params = [$start_date, $end_date];
                if ($business_id > 0) {
                    $sql .= " AND invoice.business_id = ?";
                    $params[] = $business_id;
                }
                $sql .= " GROUP BY invoice.business_id, invoice.customer_id";
                $data = DB::select($sql, $params); 

        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Customer Sales Summary';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public function customer_products_sale_details(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? now()->toDateString();
        $business_id =$request->business_id ?? 0;

        if($request->datareport){
            $sql = "SELECT
                    b.business_name AS branch,
                    i.id_invoice as invoice_id,
                    DATE_FORMAT(i.invoice_date, '%d-%m-%Y') AS Date,
                    IFNULL(i.customer_name,'') as customer,
                    IFNULL(i.customer_cell,'') as customer_cell,
                    bb.business_brand_name AS brand,
                    p.id_business_products AS product_id,
                    p.product AS product,
                    p.category,
                    bs.business_store as store,
                    pb.id_batch as batch_id,
                    #pb.batch_number as batch,
                    IFNULL(SUM(ip.invoice_qty), 0) AS total_invoice_qty,
                    IFNULL(SUM(ip.invoice_qty * ip.price), 0) AS total_price,
                    IFNULL(SUM(ip.service_addition * ip.invoice_qty), 0) AS total_addition_charges,
                    IFNULL(SUM(ip.invoice_qty * (ip.price + ip.service_addition)), 0) AS total_price_after_addition_charges,
                    IFNULL(SUM(ip.invoice_qty * ip.discount), 0) AS total_discount,
                    IFNULL(SUM(ip.invoice_qty * ((ip.price + ip.service_addition) - ip.discount)), 0) AS total_discounted_amount,
                    IFNULL(SUM(ip.taxes), 0) AS total_tax,
                    IFNULL(SUM((ip.invoice_qty * ((ip.price + ip.service_addition) - ip.discount)) + (ip.taxes)), 0) AS net_amount,
                    IFNULL(SUM(pb.batch_amount * ip.invoice_qty),0) AS total_cogs
                FROM invoice i
                JOIN invoice_products ip ON ip.invoice_id = i.id_invoice
                JOIN business_products p ON p.id_business_products = ip.product_id
                JOIN business_brands bb ON bb.id_business_brands = p.brand_id
                JOIN product_batch pb ON pb.id_batch = ip.batch_id
                JOIN business_stores bs ON bs.id_business_stores = pb.store_id
                JOIN business b ON b.id_business = i.business_id
                WHERE DATE(i.invoice_date) BETWEEN ? AND ?
                AND i.invoice_type = 'sale'
                AND LOWER(i.invoice_status) = 'valid' ";
                 $params = [$start_date, $end_date];
                if ($business_id > 0) {
                    $sql .= " AND i.business_id = ?";
                    $params[] = $business_id;
                }
                $sql .= " GROUP BY
                            b.id_business,
                            i.id_invoice,
                            ip.product_id,
                            ip.batch_id
                            ORDER BY
                                b.id_business,
                                i.id_invoice,
                                ip.product_id,
                                ip.batch_id ";
                $data = DB::select($sql, $params);
        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Products Sale Details';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public function expenses(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ?? $start_date;
        $business_id =$request->business_id ?? 0;

        if(session('ho_accounts')=='Yes'){
            $business_id_for_account_head = DB::table('business')->where('ho','Yes')->value('id_business');
        } else {
            $business_id_for_account_head = session('business_id');
        }

        if($request->datareport){
            $sql = "
                    SELECT 
                        business.business_name AS branch,
                        av.id_account_vouchers AS voucher_id,
                        DATE_FORMAT(av.voucher_date, '%d-%m-%Y') AS voucher_date,
                        DATE_FORMAT(av.created_on, '%d-%m-%Y') AS created_at,
                        IFNULL(av.business_partner_name,'None') AS paid_to,
                        GROUP_CONCAT(ah.account_head SEPARATOR ' ➜ ') AS account_heads,
                        av.description,
                        SUM(avd.debit) AS payment_amount,
                        av.created_by
                    FROM account_vouchers av
                    inner JOIN business ON business.id_business = av.business_id
                    INNER JOIN account_voucher_detail avd 
                        ON av.id_account_vouchers = avd.account_voucher_id
                    INNER JOIN account_heads ah 
                        ON ah.id_account_heads = avd.account_head_id
                    INNER JOIN account_sub_control asctr 
                        ON asctr.id_account_sub_control = ah.account_sub_control_id
                    INNER JOIN account_control ac 
                        ON ac.id_account_control = asctr.account_control_id
                    INNER JOIN account_main am 
                        ON am.id_account_main = ac.account_main_id
                    WHERE av.voucher_date BETWEEN ? AND ?
                    AND IFNULL(av.purchase_order_id, 0) = 0
                    AND av.voucher_type = 1
                    AND av.voucher_status = 'Active'
                    #AND ah.account_head = 'Office Expense'
                    #AND am.account_main = 'Expenses'
                ";
                $params = [
                    $start_date,
                    $end_date
                ];

                if ($business_id != 0) {
                    $sql .= " AND av.business_id = ? ";
                    $params[] = $business_id;
                }


                $sql .= " AND ah.business_id = ? ";
                $params[] = $business_id_for_account_head;

                $sql .= "
                    GROUP BY business.id_business, av.id_account_vouchers
                    HAVING SUM(CASE WHEN am.account_main = 'Expenses' THEN 1 ELSE 0 END) > 0
                    ORDER BY business.id_business, av.created_on DESC
                ";
                $data = DB::select($sql, $params);
        }else{
            $data =[];
        }   

        $reportID = $request->id;
        $reportName = 'Expenses';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')
            ->where('common_products', 'Yes')
            ->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    } 

    public function stock_by_category(Request $request)
    {
        $start_date = $request->start_date ?? now()->subDay()->toDateString();
        $end_date = $request->end_date ??  $start_date; //same date
        $business_id =$request->business_id ?? 0;
        $ProductsStockService = new ProductsStockService();

        $lastUpdatedAt = DB::table('mis_store_product_stock')
        ->selectRaw("DATE_FORMAT(MAX(updated_at), '%d-%b-%Y, %h:%i %p') as last_updated_at")
        ->value('last_updated_at') ?? '';
        $where = '';
        if($business_id > 0){
            $where = " where business_id = $business_id ";
        }

        if($request->datareport){
    
        $sql =" SELECT
                    business_name AS branch,
                    store_id,
                    store,
                    category,
                    IFNULL(product_count,0) AS product_count,
                    IFNULL(addition_qty,0) AS manual_qty,
                    IFNULL(purchased_qty,0) AS purchased_qty,
                    IFNULL(transfer_in_qty,0) AS transfer_in_qty,
                    IFNULL(transfer_out_qty,0) AS transfer_out_qty,
                    IFNULL((sold_qty+franchise_sale_qty),0) AS sold_qty,
                    IFNULL(used_qty,0) AS used_qty,
                    IFNULL(returned_qty,0) AS returned_qty,
                    IFNULL(instock,0) AS instock,
                    IFNULL(total_stock_value,0) AS total_stock_value,
                    ROUND(
                        (instock / NULLIF(SUM(instock) OVER (), 0)) * 100,
                        2
                    ) AS stock_percentage,
                    ROUND(
                        (total_stock_value / NULLIF(SUM(total_stock_value) OVER (), 0)) * 100,
                        2
                    ) AS stock_value_percentage

                FROM (
                    SELECT
                        store_id,
                        store,
                        business_name,
                        CASE
                            WHEN category IS NULL OR TRIM(category) = ''
                            THEN 'Uncategorized'
                            ELSE category
                        END AS category,

                        COUNT(DISTINCT product_id) AS product_count,

                        SUM(addition_qty) AS addition_qty,
                        SUM(purchased_qty) AS purchased_qty,
                        SUM(transfer_in_qty) AS transfer_in_qty,
                        SUM(transfer_out_qty) AS transfer_out_qty,
                        SUM(sold_qty) AS sold_qty,
                        SUM(used_qty) AS used_qty,
                        SUM(returned_qty) AS returned_qty,
                        SUM(franchise_sale_qty) AS franchise_sale_qty,

                        SUM(instock) AS instock,
                        SUM(total_value) AS total_stock_value

                    FROM mis_store_product_stock

                    $where 

                    GROUP BY
                        store_id,
                        CASE
                            WHEN category IS NULL OR TRIM(category) = ''
                            THEN 'Uncategorized'
                            ELSE category
                        END
                ) t
                #where instock > 0
                ORDER BY store_id,  total_stock_value DESC;";
        $data = DB::select($sql);
        
        }else{
            $data =[];
        }        
        $reportID = $request->id;
        $reportName = 'Stock by Category (Last Updated at '.$lastUpdatedAt.')';
        $branches = [];
        if(session('common_products')=="Yes" && session('ho')=="Yes"){
            $branches = DB::table('business')->where('common_products', 'Yes')->get();
        } else {
         $branches = DB::table('business')->where('id_business', session('business_id'))->get();
        }
        $business_types = DB::table('business_type')->get();
        $permission_type = $request->permission;
        //update report last viewed
        $report = Reports::where('report_file_path', $request->id)->first();
        $report->last_viewed_at = now();
        $report->last_viewed_by = session('user_name');
        $report->save();
        if($request->datareport){
            return response()->json([
                'reportName' => $reportName,
                'data' => $data,
                'reportID' => $request->id,
                'branches' => $branches,
                'business_types' => $business_types,
                'permission_type' => $permission_type,
            ]);
        } else {
            $business_id=session('business_id');
            return view('reports.report_page', compact('data', 'start_date', 'end_date', 'reportID', 'reportName', 'branches', 'business_types', 'permission_type','business_id'));
        }
    }

    public  function get_branch_store(Request $request) {
        $businessId = $request->branch_id ?? 0;
        $query = DB::table('business_stores')
            ->select('id_business_stores', 'business_store'); 
        if ($businessId > 0) {
            $query->where('business_id', $businessId);
        }
        $stores = $query->get();
        if ($stores->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No stores found.'
            ]);
        }
        return response()->json([
            'success' => true,
            'data'    => $stores
        ]);
    }

   

}