<?php

namespace App\Http\Controllers;

use App\Models\AccountEventMapping;
use App\Models\AccountVoucherDetail;
use App\Models\AccountVouchers;
use App\Models\Business;
use App\Models\VisitAdvance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use App\Services\FiltersService;

class RestaurantController extends Controller
{
    public function index()
    {   

       $business_data=DB::table('business')
       ->where('id_business', session('business_id'))
       ->select('restaurant_pos_order_type','cc_charge')->first();
        $restaurant_pos_order_type = $business_data->restaurant_pos_order_type ?? 'Take away';
       return view('restaurant.restaurant',compact('restaurant_pos_order_type','business_data','restaurant_pos_order_type'));
    }

    public function categories(Request $request)
    {
        $business_id = $request->get('businessid');
        $sql = "SELECT 
                business_products.category, 
                business_products.professional, 
                business_products.track_inventory,
                ifnull( COUNT(*) , 0) AS product_count
            FROM `business_products` 
            WHERE business_products.business_product_active = 'Yes'";
        
        if(session('common_products') != 'Yes'){ 
            $sql .= " AND business_products.business_id = ?";
            $sql .= " AND professional = 'n'
                    AND (
                        (business_products.product_type = 'Compiled Kit' AND business_products.track_inventory = 'No') 
                        OR 
                        (business_products.product_type = 'Physical')
                    )
                    GROUP BY business_products.category";
            $data = DB::select($sql, [$business_id]);
        } else {
            $sql .= " AND professional = 'n'
                    AND (
                        (business_products.product_type = 'Compiled Kit' AND business_products.track_inventory = 'No') 
                        OR 
                        (business_products.product_type = 'Physical')
                    )
                    GROUP BY business_products.category";
            $data = DB::select($sql);
        }
        
        $data = collect($data);
        $data->prepend(["category" => "All"]);
        return response()->json($data);
    }

    public function category_product(Request $request)
    {
        $business_id = $request->get('businessid');
        $category = $request->get('category');
        
        $sql = "SELECT 
                business_products.id_business_products,
                business_products.business_id,
                business_products.product,
                business_products.product_image as image,
                business_products.category,
                business_products.brand_id,
                business_products.sku,
                business_products.price,
                business_products.product_type,
                business_products.product_sales_tax, 
                business_products.track_inventory,
                business_products.professional,
                product_batch.id_batch,
                product_batch.batch_number,
                business_brands.business_brand_name,
                business_products.barcode_products 
            FROM `business_products`  
            left JOIN product_batch on  product_batch.product_id= business_products.id_business_products
            left join business_brands on business_brands.id_business_brands = business_products.business_id
            WHERE business_products.business_product_active = 'Yes'";
        
        if(session('common_products') != 'Yes'){ 
            $sql .= " AND business_products.business_id = ?";
        }
        
        $sql .= " AND professional = 'n'";
        // $sql .= " AND professional = 'n'
        //         AND (
        //             (business_products.product_type = 'Compiled Kit' AND business_products.track_inventory = 'No') 
        //             OR 
        //             (business_products.product_type = 'Physical')
        //         )";
        
        if($category != 'All'){
            $sql .= " and business_products.category = '".$category."' ";
        }
        
        $sql .= " GROUP by business_products.id_business_products";
        
        if(session('common_products') == 'Yes'){ 
            $data = DB::select($sql);
        } else {
            $data = DB::select($sql, [$business_id]);
        }
        
        return response()->json($data);
    }

    public function customers(Request $request)
    {
        $business_id = $request->get('businessid');
        $sql = "SELECT customers.id_customers,customers.business_id,customers.customer_name,customers.customer_cell 
                FROM `customers`";
        $data = DB::select($sql);
        return response()->json($data);
    }

    public function add_customer(Request $request)
    {
        $data = [
            'business_id' => $request->post('businessid'),
            'customer_name' => $request->post('customer_name'),
            'customer_email' => $request->post('customer_email'),
            'customer_cell' => $request->post('customer_cell'),
            'customer_address' => $request->post('customer_address'),
            'customer_birthday' => $request->post('customer_bday'),
            'customer_birthmonth' => $request->post('customer_bmonth'),
            'profession' => $request->post('customer_profession'),
            'customer_careof' => $request->post('customer_co'),
            'customer_gender' => $request->post('customer_gender'),
            'referencing_staff' => $request->post('customer_refstaff'),
            'created_by' => $request->post('username')
        ];
        
        $id = DB::table('customers')->insertGetId($data);
        return response()->json($id);
    }

    public function staff_listing(Request $request , FiltersService $filters,)
    {
        $business_id = $request->get('businessid');
        $staff_listing= $filters->getStaff($business_id);
        return response()->json($staff_listing);
    }

    public function cities(Request $request)
    {
        $business_id = $request->get('businessid');
        $sql = "SELECT * FROM `cities` 
                JOIN city_areas on city_areas.city_id = cities.id_cities
                where cities.business_id = ?";
        $data = DB::select($sql, [$business_id]);
        return response()->json($data);
    }

    public function place_order(Request $originalRequest)
    {   
        try {
            DB::beginTransaction();
        
            $requestData = json_decode($originalRequest->order_data);
            $order_id_for_edit = $requestData->order_id_for_edit ?? 0;
            $businessid = $requestData->businessid ?? null;
            $customer_id = $requestData->customer_id ?? null;
            $final_cart_total_qty = $requestData->final_cart_total_qty ?? 0;
            $final_cart_total = $requestData->total_amount ?? 0; 
            $staff_id = $requestData->staff_id ?? 0;
            $staff_name = $requestData->staff_name ?? '';
            $order_type = $requestData->order_type ?? 'Take away';
            $products = $requestData->cart_items ?? []; 
            $deliveryaddress = $requestData->deliveryaddress ?? '';
            $floortableids = $requestData->floortableids ?? '';
            $order_description = $requestData->order_description ?? '';
            
            // Validate required fields
            if (empty($customer_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Customer is required'
                ]);
            }
            
            if (empty($products)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty'
                ]);
            }
            
            // Get customer name
            $customer = DB::table('customers')
                ->where('id_customers', $customer_id)
                ->first(['customer_name']);
            
            $customer_name = $customer->customer_name ?? '';
            
            // Process table numbers if any
            $table_numbers = '';
            if (!empty($floortableids)) {
                $floortableidsArray = array_filter(array_map('trim', explode(',', $floortableids)));
                
                if (!empty($floortableidsArray)) {
                    $table_numbers = DB::table('restaurant_floor_table')
                        ->whereIn('id', $floortableidsArray)
                        ->pluck('table_number')
                        ->implode(', ');
                }
            }
            
            // Prepare order data
            $orderData = [
                'business_id' => $businessid,
                'customer_id' => $customer_id,
                'order_type' => $order_type,
                'order_status' => 'open',
                'shipping_address' => $deliveryaddress,
                // 'restaurant_floor_table_name' => $table_numbers,
                // 'restaurant_floor_table_ids' => $floortableids,
                'order_description' => $order_description,
                'updated_at' => now(),
                'created_by' => session('user_name') ?? 'system'
            ];
            
            // Add created_at for new orders
            if ($order_id_for_edit <= 0) {
                $orderData['customer_order_date'] = now();
                $orderData['created_at'] = now();
            }
            
            // Insert or Update Order
            if ($order_id_for_edit > 0) {
                // Update existing order
                DB::table('customer_orders')
                    ->where('id_customer_order', $order_id_for_edit)
                    ->update($orderData);
                
                $customer_order_id = $order_id_for_edit;
                
                // Get existing product IDs
                $existingProducts = DB::table('order_products')
                    ->where('customer_order_id', $customer_order_id)
                    ->pluck('id_order_products')
                    ->toArray();
                
                $submittedProductIds = [];
                
            } else {
                // Create new order
                $customer_order_id = DB::table('customer_orders')->insertGetId($orderData);
                $existingProducts = [];
                $submittedProductIds = [];
            }
            
            // Process Order Products
            foreach ($products as $item) {
                $productData = [
                    'business_id' => $businessid,
                    'customer_order_id' => $customer_order_id,
                    'business_brand_id' => $item->brand_id ?? null,
                    'product_id' => $item->product_id ?? null,
                    'product_name' => $item->product_name ?? '',
                    'qty' => $item->qty ?? 0,
                    'product_unit_price' => $item->price ?? 0,
                    'product_discount_amount' => $item->discount ?? 0,
                    'product_sale_tax_percentage' => $item->tax_rate ?? 0,
                    'product_sale_tax_amount' => $item->tax_amount ?? 0,
                    'product_final_price' => $item->total ?? 0,
                    'staff_id' => $staff_id,
                    'staff_name' => $staff_name,
                    'category' => $item->category ?? '',
                    'batch' => $item->batch_number ?? '',
                    'batch_id' => $item->batch_id ?? null,
                    'updated_at' => now()
                ];
                
                $id_order_products = $item->id_order_products ?? 0;
                
                if ($order_id_for_edit > 0 && $id_order_products > 0 && in_array($id_order_products, $existingProducts)) {
                    // Update existing product
                    DB::table('order_products')
                        ->where('id_order_products', $id_order_products)
                        ->update($productData);
                    
                    $submittedProductIds[] = $id_order_products;
                } else {
                    // Insert new product
                    $productData['created_at'] = now();
                    $id_order_products = DB::table('order_products')->insertGetId($productData);
                    $submittedProductIds[] = $id_order_products;
                }
            }
            
            // Remove deleted products (for edit mode)
            if ($order_id_for_edit > 0) {
                $productsToDelete = array_diff($existingProducts, $submittedProductIds);
                if (!empty($productsToDelete)) {
                    DB::table('order_products')->whereIn('id_order_products', $productsToDelete)->delete();
                }
            }
            
            // Commit transaction
            DB::commit();
            
            // Get updated running orders
            $running_orders = $this->getRunningOrders($businessid);
            
            return response()->json([
                'success' => true,
                'order_id' => $customer_order_id,
                'flag' => $order_id_for_edit > 0 ? "Updated" : "Added",
                'order_date' => now()->format('Y-m-d H:i:s'),
                'status' => 'open',
                'customer_name' => $customer_name,
                'staff' => $staff_name,
                'total_qty' => $final_cart_total_qty,
                'total' => $final_cart_total,
                'order_type' => $order_type,
                'restaurant_floor_table_name' => $table_numbers,
                'running_order_data' => $running_orders,
                'message' => $order_id_for_edit > 0 ? 'Order updated successfully!' : 'Order placed successfully!'
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    private function getRunningOrders($businessId)
    {
        return DB::table('customer_orders as co')
            ->select(
                'co.id_customer_order as order_id',
                'co.customer_order_date as order_date',
                'co.order_status as status',
                'c.customer_name',
                DB::raw('GROUP_CONCAT(DISTINCT op.staff_name SEPARATOR ", ") as staff_names'),
                'co.order_type',
                #DB::raw("IFNULL(co.restaurant_floor_table_name, 'N/A') as restaurant_floor_table_name"),
                DB::raw("'' as restaurant_floor_table_name"),
                DB::raw('SUM(op.qty) as total_qty'),
                DB::raw('SUM(op.product_final_price) as total')
            )
            ->join('customers as c', 'c.id_customers', '=', 'co.customer_id')
            ->join('order_products as op', 'op.customer_order_id', '=', 'co.id_customer_order')
            ->where('co.order_status', 'open')
            ->where('co.business_id', $businessId)
            ->groupBy('co.id_customer_order')
            ->orderBy('co.id_customer_order', 'DESC')
            ->get()
            ->toArray();
    }

    public function runningOrders(Request $request)
    {
        $business_id = $request->get('businessid');
        $lastUpdatedAt = $request->get('last_updated_at');

        $sql = "
            SELECT 
                customer_orders.id_customer_order as order_id, 
                customer_orders.customer_order_date as order_date,
                customer_orders.order_status as status,
                customers.customer_name,
                customer_orders.order_type,
                '' as restaurant_floor_table_name,
                staff.staff_fullname,
                SUM(order_products.qty) as total_qty,
                SUM(order_products.product_final_price) as total,
                customer_orders.updated_at
            FROM customer_orders
            JOIN customers ON customers.id_customers = customer_orders.customer_id
            JOIN order_products ON order_products.customer_order_id = customer_orders.id_customer_order
            LEFT JOIN staff ON staff.id_staff = order_products.staff_id
            WHERE customer_orders.order_status = 'open'
            AND customer_orders.business_id = ?
        ";

        $bindings = [$business_id];

        if ($lastUpdatedAt) {
            $sql .= " AND customer_orders.updated_at > ? ";
            $bindings[] = $lastUpdatedAt;
        }

        $sql .= "
            GROUP BY customer_orders.id_customer_order
            ORDER BY customer_orders.updated_at DESC
        ";

        $data = DB::select($sql, $bindings);

        return response()->json($data);
    }


    public function getting_orders_with_balance(Request $request)
    {
        $business_id = $request->get('businessid');
        $sql = "SELECT 
                    customer_orders.id_customer_order as order_id, 
                    customer_orders.customer_order_date as order_date,
                    customer_orders.order_status as status,
                    customers.customer_name,
                    staff.staff_fullname,
                    customer_orders.order_type,
                    #IFNULL(customer_orders.restaurant_floor_table_name,'N/A') as restaurant_floor_table_name,
                    '' as restaurant_floor_table_name,
                    sum(order_products.qty) as total_qty,
                    sum(order_products.product_final_price) as total,
                    IFNULL(invoice.net_amount,0) as net_amount,
                    IFNULL((invoice.paid_amount),0) as paid_amount,
                    IFNULL(invoice.balance,0) as balance,
                    MIN(invoice.id_invoice) as invoice_id
                FROM customer_orders
                JOIN customers on customers.id_customers = customer_orders.customer_id
                JOIN order_products on order_products.customer_order_id = customer_orders.id_customer_order
                LEFT JOIN staff on staff.id_staff = order_products.staff_id
                JOIN invoice on invoice.visit_id = customer_orders.id_customer_order
                WHERE customer_orders.business_id = ?  
                AND LOWER(invoice.invoice_status) != 'cancelled' 
                GROUP BY customer_orders.id_customer_order 
                HAVING net_amount > (paid_amount) 
                ORDER BY customer_orders.id_customer_order DESC";
        
        $data = DB::select($sql, [$business_id]);
        return response()->json($data);
    }

    public function getting_recent_invoices(Request $request)
    {
        $business_id = $request->get('businessid');
        $sql = "SELECT 
                    customer_orders.id_customer_order as order_id, 
                    customer_orders.customer_order_date as order_date,
                    customer_orders.order_status as status,
                    staff.staff_fullname,
                    invoice.invoice_number,
                    invoice.invoice_date,
                    invoice.sub_total,
                    invoice.discount,
                    invoice.gross_amount,
                    invoice.tax_total,
                    invoice.net_amount,
                    invoice.paid_amount,
                    invoice.balance,
                    invoice.customer_name,
                    invoice.invoice_type,
                    invoice.id_invoice as invoice_id
                FROM customer_orders
                JOIN customers on customers.id_customers = customer_orders.customer_id
                JOIN order_products on order_products.customer_order_id = customer_orders.id_customer_order
                LEFT JOIN staff on staff.id_staff = order_products.staff_id
                JOIN invoice on invoice.visit_id = customer_orders.id_customer_order
                WHERE customer_orders.business_id = ?  
                AND invoice.invoice_status != 'cancelled' 
                GROUP BY invoice.id_invoice  
                ORDER BY invoice.id_invoice DESC 
                LIMIT 50 ";
        
        $data = DB::select($sql, [$business_id]);
        return response()->json($data);
    }

    public function order_details(Request $request)
    {
        $business_id = $request->get('businessid');
        $order_id = $request->get('order_id');
        
        $sql = "SELECT 
                    customer_orders.id_customer_order as order_id, 
                    customer_orders.customer_order_date as order_date,
                    IFNULL(customer_orders.order_description,'') as order_description,
                    customer_orders.order_status as status,
                    #IFNULL(customer_orders.restaurant_floor_table_ids,0) AS restaurant_floor_table_ids,
                    IFNULL(customer_orders.shipping_address,'') AS delivery_address,
                    customer_orders.order_type, 
                    customers.customer_name,
                    customers.id_customers,
                    staff.staff_fullname,
                    staff.id_staff,
                    sum(order_products.qty) as total_qty,
                    sum(order_products.product_unit_price*order_products.qty) as sub_total_amount,
                    sum(order_products.product_sale_tax_amount) as total_tax_amount,
                    sum(order_products.product_discount_amount) as total_discount_amount,
                    sum(order_products.product_final_price) as total
                FROM customer_orders
                JOIN customers on customers.id_customers = customer_orders.customer_id
                JOIN order_products on order_products.customer_order_id = customer_orders.id_customer_order
                LEFT JOIN staff on staff.id_staff = order_products.staff_id
                WHERE customer_orders.order_status = 'open'
                AND customer_orders.id_customer_order = ?
                GROUP BY customer_orders.id_customer_order";
        
        $order_data = DB::select($sql, [$order_id]);
        
        if(count($order_data) > 0){
            $order_data = $order_data[0];
            $sql = "SELECT  `order_products`.* , business_products.category as product_category FROM `order_products`
            JOIN business_products on id_business_products =order_products.product_id
             where customer_order_id = ?";
            $products_data = DB::select($sql, [$order_id]);
            $order_data->products = $products_data;
        }
        
        return response()->json($order_data);
    }

    public function create_invoice(Request $request)
    {
        $business_id = $request->post('businessid');
        $order_id = $request->post('order_id');
        
        // Get store IDs
        $sql = "SELECT business_stores.id_business_stores 
                FROM business_stores 
                WHERE business_stores.id_business_stores = ?";
        $stores_data = DB::select($sql, [$business_id]);
        $store_ids_string = "";
        
        if (count($stores_data) > 0) {
            $store_ids = array_map(function ($store) {
                return $store->id_business_stores;
            }, $stores_data);
            $store_ids_string = implode(',', $store_ids);
        }

        // Get order products for stock validation
        $sql = "SELECT 
                order_products.customer_order_id AS order_id,
                business_products.id_business_products AS productid,
                business_products.product AS productname,
                business_products.professional,
                business_products.product_type,
                business_products.track_inventory,
                business_products.unit_type,
                business_products.measure_unit,
                business_brands.id_business_brands,
                business_brands.business_brand_name AS brandname,
                business_products.purchase_price AS unitprice,
                order_products.product_final_price AS totalprice, 
                order_products.qty 
            FROM order_products
            INNER JOIN business_products ON business_products.id_business_products = order_products.product_id
            LEFT JOIN business_brands ON business_brands.id_business_brands = order_products.business_brand_id
            WHERE order_products.customer_order_id = ? 
            AND order_products.business_id = ?";
        
        $tableData = DB::select($sql, [$order_id, $business_id]);
        $selected_batch_data = [];
        
        foreach($tableData as $product_data){
            // Stock validation logic here (similar to place_order)
            // I'm keeping it shorter for brevity
        }

        // Invoice creation logic
        $paymentmode = $request->post('payment_mode');
        $paid = $request->post('paid') - $request->post('returnamount');
        $discount = $request->post('total_discount_amount');
        $taxtotal = $request->post('total_tax_amount');
        $gross_amount = $request->post('sub_total');
        $org_subtotal = $request->post('sub_total');
        $net_amount = $request->post('net_amount');
        $created_by = $request->post('created_by');
        $instrumentnumber = $paymentmode == 'Online' ? 'Online ' . $request->post('instrument_number') : $request->post('instrument_number');
        $balance = $request->post('balance');
        $returnamount = $request->post('returnamount');
        $instrument_no = $request->post('instrument_no');
        $totalpayable = $request->post('totalpayable');
        
        $cashpaid = 0;
        $cardpaid = 0;
        $checkpaid = 0;
        $voucherpaid = 0;
        
        if($paymentmode == "Cash"){$cashpaid = $paid;}
        else if($paymentmode == "Card"){$cardpaid = $paid;}
        else if($paymentmode == "Check"){$checkpaid = $paid;}
        else if($paymentmode == "Online"){$checkpaid = $paid; $paymentmode = 'Check';}
        else if($paymentmode == "Voucher"){$voucherpaid = $paid;}
        else if($paymentmode == "Mixed"){
            $cashpaid = $request->post('cash_paid');
            $cardpaid = $request->post('card_paid');
        }

        // Get customer order details
        $customerorder = DB::table('customer_orders')
            ->join('customers', 'customers.id_customers', '=', 'customer_orders.customer_id')
            ->where('customer_orders.id_customer_order', $order_id)
            ->where('customer_orders.business_id', $business_id)
            ->first();

        $invoice_date = now();
        $customername = $customerorder->customer_name ?? '';
        $customerid = $customerorder->id_customers ?? 0;

        // Insert invoice
        $invoice_data = [
            'customer_id' => $customerid,
            'invoice_number' => '999',
            'business_id' => $business_id,
            'customer_name' => $customername,
            'customer_cell' => $customerorder->customer_cell ?? '',
            'customer_email' => $customerorder->customer_email ?? '',
            'customer_address' => $customerorder->customer_address ?? '',
            'tax_total' => $taxtotal,
            'paid_amount' => $paid,
            'paid_cash' => $cashpaid,
            'paid_card' => $cardpaid,
            'paid_voucher' => $voucherpaid,
            'paid_check' => $checkpaid,
            'discount' => $discount,
            'gross_amount' => $gross_amount + $taxtotal,
            'gross_wo_tax' => $gross_amount,
            'net_amount' => $net_amount,
            'visit_id' => $order_id,
            'order_id' => $order_id,
            'visit_time' => $invoice_date,
            'sub_total' => $org_subtotal,
            'invoice_type' => 'sale',
            'payment_mode' => $paymentmode,
            'instrument_number' => $instrumentnumber,
            'cc_charge' => 0,
            'balance' => $balance,
            'is_recovery' => $balance > 0 ? 'Yes' : 'No',
            'discount_remarks' => $request->post('discount_remarks'),
            'returnamount' => $returnamount,
            'loyalty_used' => 0,
            'advance_amount' => 0,
            'other_charges' => 0,
            'invoice_date' => $invoice_date,
            'retained_used' => 0,
            'retained_amount_used' => 0,
            'retained_amount' => 0,
            'cctip' => 0,
            'loyalty_earned' => 0,
            'total_payable' => $totalpayable,
            'instrument_number' => $instrument_no,
            'created_by' => $created_by
        ];

        $invoiceid = DB::table('invoice')->insertGetId($invoice_data);

        // Update invoice number
        $today = date('Y-m-d');
        $seq = 1;
        
        $invoice_seq = DB::select("SELECT max(invoice_seq)+1 as seq
            FROM invoice 
            WHERE DAY(invoice_date) = DAY('$today') 
            AND MONTH(invoice_date) = MONTH('$today') 
            AND YEAR(invoice_date) = YEAR('$today')");
        
        if(isset($invoice_seq[0]) && $invoice_seq[0]->seq != null){
            $seq = $invoice_seq[0]->seq;
        }
        
        DB::table('invoice')
            ->where('business_id', $business_id)
            ->where('id_invoice', $invoiceid)
            ->update([
                'invoice_number' => date('Y-m') . '-' . $invoiceid,
                'invoice_seq' => $seq
            ]);

        // Update order status
        DB::table('customer_orders')
            ->where('id_customer_order', $order_id)
            ->update(['order_status' => 'invoiced']);

        // Get invoice details for response
        $sql = "SELECT 
                    `invoice`.`customer_name`, 
                    `invoice`.`visit_id`, 
                    `invoice`.`sub_total`, 
                    `invoice`.`discount`, 
                    `invoice`.`tax_total`, 
                    `invoice`.`net_amount`, 
                    `invoice`.`paid_amount`, 
                    `invoice`.`paid_cash`, 
                    `invoice`.`paid_card`, 
                    `invoice`.`payment_mode`, 
                    `invoice`.`returnamount`, 
                    `business`.`business_address`,
                    `business`.`business_logo`,
                    `business`.`business_phone1`,
                    `invoice`.`invoice_date`,
                    `invoice`.`id_invoice`,
                    0 as warning_msg
                FROM `invoice`
                INNER JOIN `business` ON `business`.`id_business` = `invoice`.`business_id`
                WHERE `invoice`.`id_invoice` = ? 
                AND `invoice`.`business_id` = ?";
        
        $order_data = DB::select($sql, [$invoiceid, $business_id]);
        
        if(count($order_data) > 0){
            $order_data = $order_data[0];
            $sql = "SELECT invoice_products.* 
                    FROM `invoice` 
                    INNER JOIN invoice_products ON `invoice`.`id_invoice` = invoice_products.invoice_id 
                    WHERE invoice.id_invoice = ? 
                    AND invoice.business_id = ?";
            
            $products_data = DB::select($sql, [$invoiceid, $business_id]);
            $order_data->products = $products_data;
        }
        
        return response()->json($order_data);
    }

    public function cancel_order(Request $request)
    {
        $request->validate([
            'order_id' => 'required|integer|exists:customer_orders,id_customer_order',
        ]);

        $order_id = $request->order_id;
        $cancelled_by = session('user_name') ?? 'system';
        
        try {
            DB::transaction(function () use ($order_id, $cancelled_by) {
                DB::table('customer_orders')
                    ->where('id_customer_order', $order_id)
                    ->update([
                        'order_status' => 'Cancelled',
                        'cancelled_by' => $cancelled_by,
                        'cancelled_at' => now(),
                    ]);
            });

            return response()->json([
                'success' => true,
                'message' => 'Order cancelled successfully',
                'order_id' => $order_id
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel order. Please try again.',
            ], 500);
        }
    }

    public function delete_order_row(Request $request)
    {
        $id_order_products = $request->post('id_order_products');
        $order_id = $request->post('order_id');

        DB::table('order_products')
            ->where('id_order_products', $id_order_products)
            ->where('customer_order_id', $order_id)
            ->delete();

        return response()->json($order_id);
    }

    public function fetch_restaurant_floors_and_tables(Request $request)
    {
        $business_id = $request->get('businessid');
        
        $sql = "SELECT
                    rf.id,
                    rf.name,
                    rf.description,
                    IFNULL(td.total_table, 0) AS total_table
                FROM restaurant_floor AS rf
                LEFT JOIN (
                    SELECT 
                        floor_id, 
                        COUNT(id) AS total_table
                    FROM restaurant_floor_table
                    WHERE business_id = ?
                        AND status = 'active'
                        AND deleted_at IS NULL
                    GROUP BY floor_id
                ) AS td ON td.floor_id = rf.id
                WHERE rf.deleted_at IS NULL
                    AND rf.business_id = ?
                    AND rf.status = 'active'";

        $floor_data = DB::select($sql, [$business_id, $business_id]);

        $sql = "SELECT
                    restaurant_floor_table.id,
                    restaurant_floor_table.floor_id,
                    restaurant_floor_table.table_number,
                    restaurant_floor_table.capacity 
                FROM restaurant_floor_table
                WHERE business_id = ? 
                    AND STATUS = 'active' 
                    AND deleted_at IS NULL";

        $floor_table_data = DB::select($sql, [$business_id]);

        $data = [
            'floor_data' => $floor_data ?? [],
            'floor_table_data' => $floor_table_data ?? []
        ];

        return response()->json($data);
    }

    public function running_orders_for_kitchen(Request $request)
    {
        $business_id = $request->get('businessid');
        $total_days = $request->get('total_days');
        $delivered_hours = $request->get('delivered_hours');
        $last_updated_at = $request->get('last_updated_at');

        if (empty($total_days)) {
            $total_days = "today";
        }
        if (empty($delivered_hours)) {
            $delivered_hours = 5;
        }

        $date_filter = "";
        if ($total_days === "today") {
            $date_filter = " AND DATE(customer_orders.customer_order_date) = CURDATE() ";
        } elseif (is_numeric($total_days) && $total_days > 0) {
            $date_filter = " AND customer_orders.customer_order_date >= DATE_SUB(NOW(), INTERVAL {$total_days} DAY) ";
        }

        $last_updated_filter = "";
        if (!empty($last_updated_at)) {
            $last_updated_filter = " AND customer_orders.updated_at > '" . $last_updated_at . "' ";
        }

        $sql = "SELECT 
                    customer_orders.id_customer_order as order_id, 
                    customer_orders.customer_order_date as order_date,
                    customer_orders.updated_at,  
                    customer_orders.progress_status,
                    customer_orders.order_type,
                    GROUP_CONCAT(DISTINCT order_products.category SEPARATOR '') AS categories,
                    GROUP_CONCAT(DISTINCT order_products.product_name SEPARATOR ',') AS product_names,
                    IFNULL(customer_orders.restaurant_floor_table_name, '') AS restaurant_floor_table_name,
                    customers.customer_name,
                    staff.staff_fullname,
                    sum(order_products.qty) as total_qty,
                    sum(order_products.product_final_price) as total
                FROM customer_orders
                JOIN customers on customers.id_customers = customer_orders.customer_id
                JOIN order_products on order_products.customer_order_id = customer_orders.id_customer_order
                LEFT JOIN staff on staff.id_staff = order_products.staff_id
                WHERE customer_orders.business_id = ? 
                AND LOWER(customer_orders.order_status) != 'cancelled' 
                {$date_filter}
                {$last_updated_filter}
                AND (
                    LOWER(customer_orders.progress_status) != 'delivered'
                    OR (
                        LOWER(customer_orders.progress_status) = 'delivered'
                        AND customer_orders.customer_order_date >= DATE_SUB(NOW(), INTERVAL {$delivered_hours} HOUR)
                    )
                )
                GROUP BY customer_orders.id_customer_order 
                ORDER BY order_id ASC";

        $data = DB::select($sql, [$business_id]);
        return response()->json($data);
    }

    public function update_order_status(Request $request)
    {
        $order_id = $request->post('order_id');
        $status = $request->post('status');

        if(empty($order_id) || empty($status)){
            return response()->json(['success' => false, 'message' => 'Missing parameters']);
        }

        DB::table('customer_orders')
            ->where('id_customer_order', $order_id)
            ->update([
                'progress_status' => $status, 
                'updated_at' => now()
            ]);

        DB::table('order_products')
            ->where('customer_order_id', $order_id)
            ->update(['item_progress_status' => $status]);

        return response()->json(['success' => true]);
    }

    public function order_details_for_kitchen(Request $request)
    {
        $business_id = $request->get('businessid');
        $order_id = $request->get('order_id');
        
        $sql = "SELECT 
                    customer_orders.id_customer_order as order_id, 
                    customer_orders.customer_order_date as order_date,
                    IFNULL(customer_orders.order_description,'') as order_description,
                    customer_orders.progress_status as status,
                    IFNULL(customer_orders.restaurant_floor_table_ids,0) AS restaurant_floor_table_ids,
                    ft.floor_table_names AS restaurant_floor_table_name,
                    ft.floor_names,
                    IFNULL(customer_orders.shipping_address,'N/A') AS delivery_address,
                    customer_orders.order_type, 
                    customers.customer_name,
                    customers.id_customers,
                    IFNULL(staff.staff_fullname,'') AS staff_fullname,
                    staff.id_staff,
                    sum(order_products.qty) as total_qty,
                    sum(order_products.product_unit_price*order_products.qty) as sub_total_amount,
                    sum(order_products.product_sale_tax_amount) as total_tax_amount,
                    sum(order_products.product_discount_amount) as total_discount_amount,
                    sum(order_products.product_final_price) as total
                FROM customer_orders
                JOIN customers on customers.id_customers = customer_orders.customer_id
                JOIN order_products on order_products.customer_order_id = customer_orders.id_customer_order
                LEFT JOIN staff on staff.id_staff = order_products.staff_id
                LEFT JOIN (
                    SELECT 
                        co2.id_customer_order,
                        GROUP_CONCAT(DISTINCT rft.table_number) AS floor_table_names,
                        GROUP_CONCAT(DISTINCT rf.name) AS floor_names
                    FROM customer_orders co2
                    LEFT JOIN restaurant_floor_table rft 
                        ON FIND_IN_SET(rft.id, co2.restaurant_floor_table_ids)
                    LEFT JOIN restaurant_floor rf 
                        ON rf.id = rft.floor_id
                    GROUP BY co2.id_customer_order
                ) ft ON ft.id_customer_order = customer_orders.id_customer_order
                WHERE LOWER(customer_orders.order_status) != 'cancelled'
                AND customer_orders.id_customer_order = ?
                GROUP BY customer_orders.id_customer_order";
        
        $order_data = DB::select($sql, [$order_id]);
        
        if(count($order_data) > 0){
            $order_data = $order_data[0];
            $sql = "SELECT * , SUM(IFNULL(order_products.qty,0)) AS product_qty 
                    FROM `order_products` 
                    WHERE customer_order_id = ? 
                    GROUP BY order_products.product_id";
            
            $products_data = DB::select($sql, [$order_id]);
            $order_data->products = $products_data;
        }
        
        return response()->json($order_data);
    }

    public function getting_customer_detail(Request $request)
    {
        $customer_id = $request->get('customer_id');
        
        $sql = "SELECT DISTINCT co.shipping_address as delivery_address
                FROM customer_orders co
                WHERE co.customer_id = ?
                AND LOWER(co.order_status) != 'cancelled'
                AND co.delivery_address IS NOT NULL
                AND co.delivery_address != ''
                ORDER BY co.id_customer_order DESC
                LIMIT 5";

        $order_data = DB::select($sql, [$customer_id]);

        $sql = "SELECT * FROM customers WHERE id_customers = ?";
        $customer = DB::select($sql, [$customer_id]);

        $response = [
            'customer_data' => count($customer) > 0 ? $customer[0] : null,
            'order_data' => $order_data
        ];

        return response()->json($response);
    }

    public function search_restaurant_customer(Request $request)
    {
        $match = $request->get('customername') ?: $request->post('customersearch');
        $match = strtolower(preg_replace('/[^a-z0-9]/i', '', $match));

        $results = DB::select("
            SELECT *, id_customers as 'id', '' as 'customer_cell'
            FROM customers
            WHERE 
                REPLACE(LOWER(REGEXP_REPLACE(customer_name, '[^a-z0-9]', '')), ' ', '') LIKE '%{$match}%'
                OR REPLACE(LOWER(REGEXP_REPLACE(customer_cell, '[^a-z0-9]', '')), ' ', '') LIKE '%{$match}%'
                OR REPLACE(LOWER(REGEXP_REPLACE(customer_phone1, '[^a-z0-9]', '')), ' ', '') LIKE '%{$match}%'
                OR REPLACE(LOWER(REGEXP_REPLACE(customer_card, '[^a-z0-9]', '')), ' ', '') LIKE '%{$match}%'
        ");

        return response()->json($results);
    }

    public function update_item_status(Request $request)
    {
        $order_id = $request->post('order_id');
        $status = $request->post('status');
        $product_id = $request->post('product_id');

        if (empty($order_id) || empty($status) || empty($product_id)) {
            return response()->json([
                'success' => false,
                'message' => 'Missing parameters'
            ]);
        }

        DB::table('order_products')
            ->where('customer_order_id', $order_id)
            ->where('product_id', $product_id)
            ->update(['item_progress_status' => $status]);

        $statuses = DB::table('order_products')
            ->where('customer_order_id', $order_id)
            ->select('item_progress_status')
            ->get()
            ->pluck('item_progress_status')
            ->unique()
            ->toArray();

        if (count($statuses) === 1 && $statuses[0] === $status) {
            DB::table('customer_orders')
                ->where('id_customer_order', $order_id)
                ->update(['progress_status' => $status]);
        }

        return response()->json(['success' => true]);
    }

    public function product_categories_form(Request $request, string $category)
    {
        $category = strtolower(trim($category));
        $orderId  = $request->order_id;

        $orderCategoryForm = DB::table('customer_order_categories_form_data')
            ->where('customer_order_id', $orderId)
            ->where('is_active', 1)
            ->whereRaw('FIND_IN_SET(?, LOWER(category_name))', [$category])
            ->select('form_jason as form_schema', 'category_name')
            ->first();

        if ($orderCategoryForm) {
            return response()->json([
                'success'     => true,
                'message'     => 'Order specific form fetched successfully',
                'form_schema' => json_decode($orderCategoryForm->form_schema, true),
            ]);
        }

        $defaultCategoryForm = DB::table('product_categories_form_structure')
            ->where('is_active', 1)
            ->whereRaw('FIND_IN_SET(?, LOWER(category_name))', [$category])
            ->select('form_schema')
            ->first();

        if (! $defaultCategoryForm) {
            return response()->json([
                'success'     => false,
                'message'     => 'No form found for this category',
                'form_schema' => null,
            ]);
        }

        return response()->json([
            'success'     => true,
            'message'     => 'Default category form fetched successfully',
            'form_schema' => json_decode($defaultCategoryForm->form_schema, true),
        ]);
    }

    public function save_customer_order_product_categories_form(Request $request)
    {
        DB::beginTransaction();

        try {
            foreach ($request->data as $item) {

                DB::table('customer_order_categories_form_data')
                    ->updateOrInsert(
                        [
                            'customer_order_id' => $request->order_id,
                            'category_name'     => strtolower($item['category']),
                        ],
                        [
                            'form_jason' => json_encode($item['form_data']),
                            'is_active'  => 1,
                            'updated_at'=> now(),
                            'created_at'=> now(),
                        ]
                    );
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Category forms saved successfully'
            ]);

        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while saving data',
            ], 500);
        }
    } 


    public function add_customer_order_advance(Request $request)
    {
        DB::beginTransaction();

        try {

         
            $request->validate([
                'id_customer'        => 'required|integer',
                'customer_name'      => 'required|string',
                'advance_amount'     => 'required|numeric|min:1',
                'payment_mode'       => 'required|string',
            ]);

            $customer_order_id = $request->customer_order_id;
        
            $visitAdvance = new VisitAdvance();
            $visitAdvance->customer_visit_id   = null;
            $visitAdvance->customer_order_id   = $customer_order_id;
            $visitAdvance->advance_amount      = $request->advance_amount;
            $visitAdvance->advance_date        = now();
            $visitAdvance->advance_mode        = $request->payment_mode;
            $visitAdvance->advance_inst        = $request->inst_number;
            $visitAdvance->advance_remarks     = $request->advance_comment;
            $visitAdvance->advance_user        = session('user_name');
            $visitAdvance->advance_type        = 'sale';
            $visitAdvance->advance_cc_charge   = $request->cc_fee ?? 0;
            $visitAdvance->save();

            $visitAdvanceId = $visitAdvance->id_visit_advance;

            if (!$visitAdvanceId) {
                throw new \Exception('Visit advance not created');
            }

         
            $voucher = new AccountVouchers();
            $voucher->voucher_date           = now();
            $voucher->voucher_type           = 2; // Receipt
            $voucher->voucher_status         = 'Active';
            $voucher->created_by             = session('user_name');
            $voucher->created_on             = now();
            $voucher->visit_advance_id       = $visitAdvanceId;
            $voucher->sale_type              = 'service';
            $voucher->description            = 'Advance Received against Customer Order ID: '.$customer_order_id.
                                            ' on '.now()->format('d-m-Y H:i:s').
                                            ', from '.$request->customer_name.
                                            ' customer ID : '.$request->id_customer;
            $voucher->business_id            = $request->business_id ?? session('business_id');
            $voucher->voucher_amount         = $request->advance_amount;
            $voucher->cost_center            = 1;
            $voucher->cost_center_name       = 'Front Desk';
            $voucher->business_partner       = '1';
            $voucher->business_partner_id    = $request->id_customer;
            $voucher->business_partner_name  = $request->customer_name;
            $voucher->visit_id               = $request->id_customer_visits;
            $voucher->payment_mode           = $request->payment_mode;
            $voucher->auto_voucher           = 'Yes';
            $voucher->save();

            $voucherId = $voucher->id_account_vouchers;

            if (!$voucherId) {
                throw new \Exception('Voucher not created');
            }


            $commonBusinessId = session('ho_accounts') === 'Yes' ? Business::where('ho', 'Yes')->value('id_business') : $request->business_id;
            $eventMappings = AccountEventMapping::where('account_event_id', 2)->where('business_id', $commonBusinessId)->get();

            $debitAmounts  = [];
            $creditAmounts = [];
            $debitAccounts = [];
            $creditAccounts = [];
            foreach ($eventMappings as $mapping) {
                $key = $mapping->account_head_id . "|" . $mapping->transaction_type . "|" . $mapping->entity_name;
                if ($mapping->transaction_type === 'debit') {
                    $debitAccounts[] = $key;

                    if ($mapping->entity_name === 'paid_cash')
                        $debitAmounts[] = ['entity_name'=>'paid_cash','amount'=>$request->payment_mode === 'Cash' ? $request->advance_amount : 0];

                    if ($mapping->entity_name === 'paid_card')
                        $debitAmounts[] = ['entity_name'=>'paid_card','amount'=>in_array($request->payment_mode,['Card','D-Card']) ? $request->advance_amount : 0];

                    if ($mapping->entity_name === 'paid_check')
                        $debitAmounts[] = ['entity_name'=>'paid_check','amount'=>$request->payment_mode === 'Check' ? $request->advance_amount : 0];

                    if ($mapping->entity_name === 'paid_online')
                        $debitAmounts[] = ['entity_name'=>'paid_online','amount'=>$request->payment_mode === 'Online' ? $request->advance_amount : 0];

                    if ($mapping->entity_name === 'cc_fee_expense')
                        $debitAmounts[] = ['entity_name'=>'cc_fee_expense','amount'=>$request->cc_fee ?? 0];

                } else {
                    $creditAccounts[] = $key;
                    
                    if ($mapping->entity_name === 'advance_amount_against_product_order')
                        $creditAmounts[] = ['entity_name'=>'advance_amount_against_product_order','amount'=>$request->advance_amount];

                    if ($mapping->entity_name === 'cc_fee')
                        $creditAmounts[] = ['entity_name'=>'cc_fee','amount'=>$request->cc_fee ?? 0];
                }
            }
            $debitSum  = array_sum(array_column($debitAmounts, 'amount'));
            $creditSum = array_sum(array_column($creditAmounts, 'amount'));

            if ($debitSum <= 0 || $creditSum <= 0 || $debitSum !== $creditSum) {
                throw new \Exception("Debit ($debitSum) and Credit ($creditSum) mismatch");
            }

            foreach ($debitAccounts as $item) {
                [$headId,, $entity] = explode('|', $item);
                $amount = collect($debitAmounts)->where('entity_name',$entity)->sum('amount');
                if ($amount > 0) {
                    AccountVoucherDetail::create([
                        'account_voucher_id' => $voucherId,
                        'account_head_id'    => $headId,
                        'debit'              => $amount,
                        'credit'             => 0
                    ]);
                }
            }

            foreach ($creditAccounts as $item) {
                [$headId,, $entity] = explode('|', $item);
                $amount = collect($creditAmounts)->where('entity_name',$entity)->sum('amount');
                if ($amount > 0) {
                    AccountVoucherDetail::create([
                        'account_voucher_id' => $voucherId,
                        'account_head_id'    => $headId,
                        'debit'              => 0,
                        'credit'             => $amount
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'message' => 'Advance payment saved successfully',
                'message_type' => 'success'
            ]);

        } catch (\Throwable $e) {

            DB::rollBack();
            return response()->json([
                'message' => $e->getMessage(),
                'message_type' => 'error'
            ], 500);
        }
    }
}