<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Customers;
use App\Models\Business;
use App\Models\SMSLog;
use App\Models\ScheduledTask;
use App\Services\SMSService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SMSController extends Controller
{
    public function customSms()
    {
        $business = Business::select('business_name', 'id_business')
            ->where('id_business', '=', session('business_id'))
            ->first();

        return view('sms.custom_sms')->with(compact('business'));
    }

    public function customersData(Request $request)
    {
        $columns = [
            'id_customers',
            'customer_name',
            'customer_gender',
            'care_of',
            'customer_cell'
        ];

        $search = $request->input('search.value');
        $orderColumnIndex = $request->input('order.0.column', 0);
        $orderColumn = $columns[$orderColumnIndex] ?? 'id_customers';
        $orderDir = $request->input('order.0.dir', 'desc');
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        // Get filter parameters
        $filterGender = $request->input('filter_gender');
        $filterMobileLength = $request->input('filter_mobile_length');
        $filterDisplayFormat = $request->input('filter_display_format', 'as_saved');

        // Base query with gender formatting
        $query = Customers::select(
            'id_customers',
            'customer_name',
            'customer_gender',
            'customer_careof',
            'customer_cell',
            'country_code'
        );

        // Apply gender filter
        if ($filterGender) {
            if ($filterGender === 'Male') {
                $query->where('customer_gender', 'M');
            } elseif ($filterGender === 'Female') {
                $query->where('customer_gender', 'F');
            } elseif ($filterGender === 'Other') {
                // Other: NULL, empty, or any value not equal to M or F
                $query->where(function ($q) {
                    $q->whereNull('customer_gender')
                      ->orWhere('customer_gender', '')
                      ->orWhereNotIn('customer_gender', ['M', 'F']);
                });
            }
        }

        // Apply mobile number length filter
        if ($filterMobileLength) {
            $query->whereRaw('LENGTH(customer_cell) = ?', [$filterMobileLength]);
        }

        // Get total count before search/filtering
        $recordsTotal = Customers::count();

        // Apply search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('customer_name', 'LIKE', "%{$search}%")
                  ->orWhere('id_customers', 'LIKE', "%{$search}%");
            });
        }

        // Get filtered count
        $recordsFiltered = $query->count();

        // Get data with ordering and pagination
        $data = $query->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get()
            ->map(function ($customer) use ($filterDisplayFormat) {
                // Format gender
                $gender = 'N/A';
                if ($customer->customer_gender === 'M') {
                    $gender = 'Male';
                } elseif ($customer->customer_gender === 'F') {
                    $gender = 'Female';
                } elseif (!empty($customer->customer_gender)) {
                    $gender = $customer->customer_gender;
                }

                // Format mobile number based on display format
                $mobileNumber = $customer->customer_cell ?? '';
                $formattedMobile = $this->formatMobileNumber($mobileNumber, $customer->country_code, $filterDisplayFormat);

                return [
                    'id_customers' => $customer->id_customers,
                    'customer_name' => $customer->customer_name,
                    'customer_gender' => $gender,
                    'care_of' => $customer->customer_careof ?? 'N/A',
                    'customer_cell' => $formattedMobile
                ];
            });

        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data
        ]);
    }

    /**
     * Format mobile number based on display format option
     * 
     * @param string $mobileNumber
     * @param string|null $countryCode
     * @param string $format
     * @return string
     */
    private function formatMobileNumber($mobileNumber, $countryCode, $format)
    {
        if (empty($mobileNumber)) {
            return '';
        }

        switch ($format) {
            case 'prefix_0':
                // Strip any existing prefix (+, country code) and prefix with 0
                $cleaned = $mobileNumber;
                
                // Remove + if present
                $cleaned = ltrim($cleaned, '+');
                
                // Remove country code (92) if number starts with it and is 12 digits
                if (strlen($cleaned) == 12 && str_starts_with($cleaned, '92')) {
                    $cleaned = substr($cleaned, 2);
                }
                
                // Remove leading 0 if already present
                $cleaned = ltrim($cleaned, '0');
                
                return '0' . $cleaned;

            case 'prefix_country':
                // Use country_code, default to 92 if NULL
                $code = $countryCode ?? '92';
                $cleaned = $mobileNumber;
                
                // Remove + if present
                $cleaned = ltrim($cleaned, '+');
                
                // Remove country code if number starts with it (handle both 92 and other codes)
                if (strlen($cleaned) >= 11 && str_starts_with($cleaned, $code)) {
                    $cleaned = substr($cleaned, strlen($code));
                } elseif (strlen($cleaned) == 12 && str_starts_with($cleaned, '92')) {
                    // Handle legacy 92 prefix
                    $cleaned = substr($cleaned, 2);
                }
                
                // Remove leading 0 if present
                $cleaned = ltrim($cleaned, '0');
                
                return '+' . $code . $cleaned;

            case 'as_saved':
            default:
                // Show as saved in database
                return $mobileNumber;
        }
    }

    public function customerLookup(Request $request)
    {
        $term = $request->input('term');

        if (strlen($term) < 4) {
            return response()->json([]);
        }

        // Normalize the search term - handle both 92xxx and 0xxx formats
        $normalizedTerm = $term;
        
        // Build query to search with normalization
        $customers = Customers::where(function ($query) use ($term, $normalizedTerm) {
                // Direct match
                $query->where('customer_cell', 'LIKE', '%' . $term . '%');
                
                // If searching with leading 0, also search for 92 format
                if (str_starts_with($term, '0')) {
                    $without0 = substr($term, 1);
                    $query->orWhere('customer_cell', 'LIKE', '%92' . $without0 . '%');
                }
                
                // If searching without prefix, search both formats
                if (!str_starts_with($term, '0') && !str_starts_with($term, '92')) {
                    $query->orWhere('customer_cell', 'LIKE', '%0' . $term . '%')
                          ->orWhere('customer_cell', 'LIKE', '%92' . $term . '%');
                }
            })
            ->select('id_customers', 'customer_name', 'customer_cell')
            ->limit(10)
            ->get()
            ->map(function ($customer) {
                // Normalize cell for autofill - extract 10 digit portion
                $cell = $customer->customer_cell;
                $normalized = $cell;
                
                // Remove 92 prefix
                if (str_starts_with($cell, '92') && strlen($cell) == 12) {
                    $normalized = substr($cell, 2);
                }
                // Remove leading 0
                elseif (str_starts_with($cell, '0') && strlen($cell) == 11) {
                    $normalized = substr($cell, 1);
                }

                return [
                    'id_customers' => $customer->id_customers,
                    'customer_name' => $customer->customer_name,
                    'customer_cell' => $customer->customer_cell,
                    'cell_normalized' => $normalized
                ];
            });

        return response()->json($customers);
    }

    public function send(Request $request)
    {
        $request->validate([
            'cell_number' => 'required|string|size:12',
            'message' => 'required|string'
        ]);

        $cellNumber = $request->input('cell_number');
        $message = $request->input('message');
        $businessId = session('business_id');

        // Convert 92xxx to 0xxx format (11 digits) as expected by SMSService
        $phoneForService = '0' . substr($cellNumber, 2);

        try {
            $smsService = new SMSService();
            $result = $smsService->sendSMS(
                $message,
                $phoneForService,
                'CustomSMS',
                $businessId,
                null,
                false,
                false
            );

            return response()->json([
                'success' => true,
                'message' => 'SMS sent successfully',
                'raw_response' => $result
            ]);
        } catch (\Exception $e) {
            Log::error('Custom SMS Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function send_invoice_sms($id_invoice)
    {
        $business_id = session('business_id');
        $business = Business::find($business_id);

        if($business && $business->sms_enabled){

            //Fetch invoice details
            $invoice = DB::table('invoices')
                ->leftJoin('customers', 'invoices.customer_id', '=', 'customers.id_customers')
                ->select('invoices.*', 'customers.name as customer_name', 'customers.phone as customer_phone')
                ->where('invoices.id_invoices', $id_invoice)
                ->where('invoices.business_id', $business_id)
                ->first();

            if($invoice && $invoice->customer_phone){

                $message = "Dear " . $invoice->customer_name . ", your invoice #" . $invoice->id_invoices . " amounting to " . number_format($invoice->total_amount, 2) . " has been generated. Thank you for your business!";

                //Send SMS using a hypothetical SMS service
                try {
                    // Assuming there's an SMS service class to handle sending
                    (new SMSService())->sendSMS($message, $invoice->customer_phone, 'InvoiceSMS', $business_id, $id_invoice);
                } catch (\Exception $e) {
                    // Log the error but do not interrupt the main flow
                    Log::error("Failed to send SMS for invoice {$id_invoice}: " . $e->getMessage());
                }

            }

        }

    }

    public function smsLog()
    {
        return view('sms.sms_log');
    }

    public function smsLogData(Request $request)
    {
        $columns = [
            'id_sms_log',
            'msgdata',
            'receiver',
            'senton',
            'using'
        ];

        $orderColumnIndex = $request->input('order.0.column', 0);
        $orderColumn = $columns[$orderColumnIndex] ?? 'id_sms_log';
        $orderDir = $request->input('order.0.dir', 'desc');
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        $query = SMSLog::select(
            'id_sms_log',
            'msgdata',
            'receiver',
            DB::raw('DATE_FORMAT(senton, "%Y-%m-%d %H:%i") as senton'),
            'using'
        )->where('business_id', session('business_id'));

        $recordsTotal = SMSLog::where('business_id', session('business_id'))->count();

        // Apply filters
        if ($request->filled('filter_id')) {
            $query->where('id_sms_log', 'LIKE', '%' . $request->input('filter_id') . '%');
        }

        if ($request->filled('filter_message')) {
            $searchTerm = $request->input('filter_message');
            $query->where('msgdata', 'LIKE', '%' . $searchTerm . '%');
        }

        if ($request->filled('filter_receiver')) {
            $query->where('receiver', 'LIKE', '%' . $request->input('filter_receiver') . '%');
        }

        if ($request->filled('filter_date')) {
            $query->whereDate('senton', $request->input('filter_date'));
        }

        if ($request->filled('filter_using')) {
            $query->where('using', $request->input('filter_using'));
        }

        $recordsFiltered = $query->count();

        $data = $query->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get();

        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data
        ]);
    }

    public function smsLogUsingOptions()
    {
        $options = SMSLog::where('business_id', session('business_id'))
            ->distinct()
            ->pluck('using')
            ->filter()
            ->values();

        return response()->json($options);
    }

    public function resend(Request $request)
    {
        $request->validate([
            'id' => 'required|integer'
        ]);

        $smsLog = SMSLog::where('id_sms_log', $request->input('id'))
            ->where('business_id', session('business_id'))
            ->first();

        if (!$smsLog) {
            return response()->json([
                'success' => false,
                'message' => 'SMS log not found'
            ], 404);
        }

        try {
            $smsService = new SMSService();
            $result = $smsService->sendSMS(
                $smsLog->msgdata,
                $smsLog->receiver,
                $smsLog->using ?? '',
                session('business_id'),
                null,
                false,
                false
            );

            return response()->json([
                'success' => true,
                'message' => 'SMS resent successfully',
                'raw_response' => $result
            ]);
        } catch (\Exception $e) {
            Log::error('Resend SMS Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function scheduledTasks()
    {
        return view('sms.scheduled_tasks');
    }

    public function scheduledTasksList()
    {
        $tasks = DB::table('scheduled_tasks')
            ->select('id_scheduled_tasks', 'task_name')
            ->where('status', 'enabled')
            ->whereNotNull('task_name')
            ->distinct()
            ->get();

        return response()->json($tasks);
    }

    public function scheduledTaskDetails(Request $request)
    {
        $request->validate([
            'id' => 'required|integer'
        ]);

        $task = DB::table('scheduled_tasks')
            ->select('msg', 'day_difference')
            ->where('id_scheduled_tasks', $request->input('id'))
            ->first();

        if (!$task) {
            return response()->json(['error' => 'Task not found'], 404);
        }

        return response()->json($task);
    }

    public function scheduledTaskQuery(Request $request)
    {
        $request->validate([
            'id' => 'required|integer'
        ]);

        $task = DB::table('scheduled_tasks')
            ->select('s_query')
            ->where('id_scheduled_tasks', $request->input('id'))
            ->first();

        if (!$task || empty($task->s_query)) {
            return response()->json(['error' => 'Query not found'], 404);
        }

        try {
            // Execute the stored query with a limit
            $query = $task->s_query;
            
            // Add LIMIT if not already present
            if (stripos($query, 'LIMIT') === false) {
                $query = rtrim($query, ';') . ' LIMIT 2000';
            }

            $results = DB::select($query);

            if (empty($results)) {
                return response()->json([
                    'columns' => [],
                    'data' => []
                ]);
            }

            // Get column names from first result
            $columns = array_keys((array) $results[0]);

            // Convert objects to arrays
            $data = array_map(function($row) {
                return (array) $row;
            }, $results);

            return response()->json([
                'columns' => $columns,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            Log::error('Scheduled Task Query Error: ' . $e->getMessage());
            return response()->json([
                'error' => 'Query execution failed: ' . $e->getMessage()
            ], 500);
        }
    }
}
