<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\CashRegister;
use App\Models\Staff;
use App\Models\CustomerVisits;
use App\Models\VisitServices;
use App\Models\AccountHeads;
use App\Models\AccountVouchers;
use App\Models\AccountVoucherDetail;
use App\Models\AccountEventMapping;
use App\Models\BlockStaffTime;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Bus;

class SchedulerController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;


    public function scheduler(Request $request)
    {
        //Get Business Info
        $business = Business::select('*', DB::raw('CONCAT(business_opening_time, ":00") as business_opening_time'), DB::raw('CONCAT(business_closing_time, ":00") as business_closing_time'))
        ->where('id_business', session('business_id'))
        ->first();
        
        //Get Business Taxes
        $business_taxes = DB::table('business_taxes')
            ->select(
                'business.cc_charge',
                DB::raw('
                    SUM(CASE WHEN tax_payment_mode = "Cash"   AND tax_invoice_type = "service" THEN tax_percentage ELSE 0 END) AS tax_percentage_cash_service,
                    SUM(CASE WHEN tax_payment_mode = "Card"   AND tax_invoice_type = "service" THEN tax_percentage ELSE 0 END) AS tax_percentage_card_service,
                    SUM(CASE WHEN tax_payment_mode = "Online" AND tax_invoice_type = "service" THEN tax_percentage ELSE 0 END) AS tax_percentage_online_service,
                    SUM(CASE WHEN tax_payment_mode = "Check"  AND tax_invoice_type = "service" THEN tax_percentage ELSE 0 END) AS tax_percentage_check_service,
                    SUM(CASE WHEN tax_payment_mode = "Voucher"AND tax_invoice_type = "service" THEN tax_percentage ELSE 0 END) AS tax_percentage_voucher_service,

                    SUM(CASE WHEN tax_payment_mode = "Cash"   AND tax_invoice_type = "sale" THEN tax_percentage ELSE 0 END) AS tax_percentage_cash_sale,
                    SUM(CASE WHEN tax_payment_mode = "Card"   AND tax_invoice_type = "sale" THEN tax_percentage ELSE 0 END) AS tax_percentage_card_sale,
                    SUM(CASE WHEN tax_payment_mode = "Online" AND tax_invoice_type = "sale" THEN tax_percentage ELSE 0 END) AS tax_percentage_online_sale,
                    SUM(CASE WHEN tax_payment_mode = "Check"  AND tax_invoice_type = "sale" THEN tax_percentage ELSE 0 END) AS tax_percentage_check_sale,
                    SUM(CASE WHEN tax_payment_mode = "Voucher"AND tax_invoice_type = "sale" THEN tax_percentage ELSE 0 END) AS tax_percentage_voucher_sale
                ')
            )
            ->join('business', 'business_taxes.business_id', '=', 'business.id_business');

        // if (session('ho_accounts') == "Yes") {
        //     $business_taxes = $business_taxes->where('business.ho', 'Yes');
        // } else {
            $business_taxes = $business_taxes->where('business_id', session('business_id')); //provinces have different tax rates
        // }

        $business_taxes = $business_taxes
            ->where('tax_active', 'Y')
            ->groupBy('business.id_business')
            ->first();
        //if no taxes found, set all tax percentages to 0
        if(!$business_taxes) { $business_taxes = (object) [
            'cc_charge' => 0,
            'tax_percentage_cash_service' => 0,
            'tax_percentage_card_service' => 0,
            'tax_percentage_online_service' => 0,
            'tax_percentage_check_service' => 0,
            'tax_percentage_voucher_service' => 0,
            'tax_percentage_cash_sale' => 0,
            'tax_percentage_card_sale' => 0,
            'tax_percentage_online_sale' => 0,
            'tax_percentage_check_sale' => 0,
            'tax_percentage_voucher_sale' => 0,
        ]; }

        //Get Staff List
        $staff_list = Staff::select('id_staff', 'staff_fullname', 'staff_firstname', 'staff_lastname',
        'staff_designation', 'staff_image', 'floor', 
        DB::RAW('date_format(day_start_time, "%H:%i") as day_start_time'),
         DB::RAW('date_format(day_end_time, "%H:%i") as day_end_time'), 
         DB::RAW('CASE WHEN business_id = ' . session('business_id') . ' THEN "No" ELSE "Yes" END as staff_shared'),
         'week_day_off', 'staff_active', 'staff_order', 'staff_scheduler'
         )
        ->where('staff_active', 'Y')
        ->where('staff_scheduler', 'On')
        ->where(function($q) {
            $q->where('business_id', session('business_id'))
            ->orWhere('staff_shared', 'Yes');
        });

        if($business->common_services == "Yes"){
            $staff_list = $staff_list->orWhere('staff_shared', 'Yes');
        }
        
         $staff_list = $staff_list->orderby('staff_order', 'ASC')
        ->get();

        $resourceList  = $staff_list->map(function($staff) {
            return [
                'id'          => $staff->id_staff,
                'image'       =>  $staff->staff_image == null || $staff->staff_image == '' ?  asset('images/staff/nu.png') : asset('images/staff/'. $staff->staff_image),
                'title'       => $staff->staff_fullname,
                'designation' => $staff->staff_designation,
                'floor'      => $staff->floor,
                'staff_shared'=> $staff->staff_shared ?? '',
                'days_off'    => $staff->week_day_off ?? '',
                'day_start_time' => $staff->day_start_time ?? '',
                'day_end_time'   => $staff->day_end_time ?? '',
                'staff_order'   => $staff->staff_order,
            ];
        });

        //return $resourceList;

        //Get Staff Blocking Reasons
        $staff_block_reasons = DB::table('block_events')
            ->select('id_block_events','block_event_name', 'block_event_duration');
            //if common services is Yes, get blocking reasons for HO else only for the business
        if (session('common_services') == "Yes") {
            $staff_block_reasons = $staff_block_reasons->where(function($q) {
               //query business table to get HO business id
                $ho_business = DB::table('business')
                    ->where('ho', 'Yes')
                    ->first();
                $q->where('business_id', $ho_business->id_business ?? session('business_id'));
            });
        } else {
            $staff_block_reasons = $staff_block_reasons->where('business_id', session('business_id'));
        }
        $staff_block_reasons = $staff_block_reasons->get();
        

        return view('scheduler.scheduler', compact('business', 'resourceList', 'staff_list', 'business_taxes', 'staff_block_reasons')); //Scheduler for Reception
       
    }

    public function eventList(Request $request){

        //return($request->lastupdatedate);

        $business = Business::select('common_services', DB::raw('CONCAT(business_opening_time, ":00") as business_opening_time'), DB::raw('CONCAT(business_closing_time, ":00") as business_closing_time'))
        ->where('id_business', session('business_id'))
        ->first();

        $date_start = date('Y-m-d H:i:s', strtotime($business->business_opening_time));
        $date_end = date('Y-m-d H:i:s', strtotime($business->business_closing_time));

        if($request->start != null || $request->start != ''){
            $date_start = date('Y-m-d H:i:s', strtotime($request->start));
            $date_end = date('Y-m-d H:i:s', strtotime($request->end));
        }
        
        $lastupdate = $request->lastupdatedate ?? 0;

        if($request->lastupdatedate != null && $request->lastupdatedate != '' && $request->lastupdatedate != 0){
            $lastupdate = $request->lastupdatedate;
        } 
        
        $events = CustomerVisits::select(
                'visit_service_id as id',
                'id_customer_visits as visitid',                
                'customer_visit_date as visit_date',
                DB::raw('"eventClass" as classNames'), // Event Class
                'visit_color as backgroundColor', // FullCalendar color
                'visit_color_type',
                'visit_services.visit_service_start as start',
                'visit_services.visit_service_end as end',
                DB::RAW('CONCAT(visit_services.service_id, "-" , visit_services.service_name) as title'), // FullCalendar title
                DB::RAW('CASE WHEN customer_visits.visit_status != "open" && customer_visits.visit_status != "invoiced" THEN "canceled"  ELSE customer_visits.visit_status END as visit_status'),
                DB::RAW('CASE WHEN customer_visits.business_id = ' . session('business_id') . ' THEN "No" ELSE "Yes" END as staff_shared'),
                'customers.id_customers',
                'customers.customer_name',
                'customers.customer_cell',
                'customers.customer_type',
                'visit_service_staffs.staff_id as resourceId',
                'staff.staff_fullname as staff_name',
                'visit_service_staffs.requested',
                'visit_service_staffs.block_other',
                'customer_visits.reminder_sms',
                'customer_visits.reminder_email',
                'customer_visits.reminder_call',
                'customer_visits.reminder_stricttime',
                'visit_services.promo',
                'customer_visits.created_by',
                DB::raw('IFNULL(advances.total_advance, 0) as total_advance'),                
                DB::raw('CASE WHEN customer_visits.visit_status in ("open", "invoiced") AND customer_visits.business_id = '.session('business_id').' THEN visit_services.s_rate ELSE 0 END as service_rate'),
                'id_visit_services as visitserviceid',
                'id_visit_service_staffs as visitstaffid',
                'customer_visits.inservice',
                'visit_services.update_date as lastupdatedate'
            )
            ->leftJoin('customers', 'customer_visits.customer_id', '=', 'customers.id_customers')
            ->leftJoin('visit_services', function ($join) {
            $join->on('customer_visits.id_customer_visits', '=', 'visit_services.customer_visit_id')
                ->where('visit_services.visit_service_status', 'Active');
            })
            ->leftJoin('visit_service_staffs', function ($join) {
                $join->on('visit_services.id_visit_services', '=', 'visit_service_staffs.visit_service_id')
                    ->where('visit_service_staffs.visit_service_staff_status', 'Active');
            })
            ->leftJoin('staff', 'visit_service_staffs.staff_id', '=', 'staff.id_staff')
            ->leftJoin(DB::raw('(SELECT customer_visit_id, SUM(advance_amount) as total_advance 
                            FROM visit_advance 
                            where visit_advance.advance_status = "Active"
                            GROUP BY customer_visit_id) advances'),
                'advances.customer_visit_id', '=', 'customer_visits.id_customer_visits')
            
            ->whereDate('visit_service_start', '>=', $date_start)
            ->whereDate('visit_service_end', '<', $date_end);

        if($lastupdate != null && $lastupdate != '' && $lastupdate != 0){
            $events = $events->where('visit_services.update_date', '>', $lastupdate);
        }

        if($business->common_services == "Yes"){
            $events = $events->where(function($q) {
                $q->where('customer_visits.business_id', session('business_id'))
                ->orWhere('staff.staff_shared', 'Yes');
            });
        } else {
            $events = $events->where('customer_visits.business_id', session('business_id'));
        }
        $events = $events->orderBy('visit_services.visit_service_start', 'ASC');
        //debug the query
        //dd($events->toSql(), $events->getBindings());exit();

        $events = $events->get();
       
        $events = $events->map(function ($event) {
            // Keep at root
            $event->visitid          = $event->visitid;
            $event->visit_status     = $event->visit_status;
            $event->customer_name    = $event->customer_name;
            $event->customer_cell    = $event->customer_cell;
            $event->staff_name       = $event->staff_name;
            $event->visit_color_type = $event->visit_color_type;
            $event->textColor        = $event->visit_color_type == 'Light' ? '#000000' : '#FFFFFF';
            $event->id_customers     = $event->id_customers;
            $event->visit_date       = $event->visit_date;
            $event->requested        = $event->requested;
            $event->block_other      = $event->block_other;
            $event->reminder_sms     = $event->reminder_sms;
            $event->reminder_email   = $event->reminder_email;
            $event->reminder_call    = $event->reminder_call;
            $event->reminder_stricttime = $event->reminder_stricttime;
            $event->promo            = $event->promo;
            $event->total_advance    = $event->total_advance;
            $event->created_by       = $event->created_by;
            $event->service_rate     = $event->service_rate;
            $event->visitserviceid   = $event->visitserviceid;
            $event->visitstaffid     = $event->visitstaffid;
            $event->customer_type    = $event->customer_type;
            $event->inservice       = $event->inservice;
            $event->staff_shared    = $event->staff_shared ?? '';
            // Also pack into extendedProps for FullCalendar
            $event->extendedProps = [
                'visitid'          => $event->visitid,
                'visit_date'       => $event->visit_date,
                'visit_status'     => $event->visit_status,
                'customer_name'    => $event->customer_name,
                'customer_cell'    => $event->customer_cell,
                'staff_name'       => $event->staff_name,
                'visit_color_type' => $event->visit_color_type,
                'textColor'        => $event->visit_color_type == 'Light' ? '#000000' : '#FFFFFF',
                'id_customers'     => $event->id_customers,
                'requested'        => $event->requested,
                'block_other'      => $event->block_other,
                'reminder_sms'     => $event->reminder_sms,
                'reminder_email'   => $event->reminder_email,
                'reminder_call'    => $event->reminder_call,
                'reminder_stricttime' => $event->reminder_stricttime,
                'promo'            => $event->promo,
                'total_advance'    => $event->total_advance,
                'created_by'       => $event->created_by,
                'service_rate'     => $event->service_rate,
                'visitserviceid'   => $event->visitserviceid,
                'visitstaffid'     => $event->visitstaffid,
                'customer_type'    => $event->customer_type,
                'inservice'       => $event->inservice,
                'staff_shared'    => $event->staff_shared ?? '',
            ];

            return $event;
        }); 
        
        //get blocked staff events
        $block_events = BlockStaffTime::select(
                'block_time_event_id as id',
                DB::raw('"Yes" as is_blocking_event'),
                'block_event_id',
                'id_staff_time_blocked as visitid',
                'staff_id as resourceId',
                'staff_name',
                DB::raw('"blockedClass" as classNames'), // Event Class
                'start_time as start',
                'end_time as end',
                DB::RAW('block_events.block_event_name as title'), // FullCalendar title
                DB::RAW('"Blocked" as visit_status'),
                DB::RAW('CASE WHEN block_staff_time.business_id = ' . session('business_id') . ' THEN "No" ELSE "Yes" END as staff_shared'),
                'remarks'
            )
            ->join('block_events', 'block_staff_time.block_event_id', '=', 'block_events.id_block_events')
            ->where('block_staff_time.business_id', session('business_id'))
            ->whereDate('start_time', '>=', $date_start)
            ->whereDate('end_time', '<', $date_end);
            if($lastupdate != null && $lastupdate != '' && $lastupdate != 0){
                $block_events = $block_events->where('block_staff_time.updated_at', '>', $lastupdate);
            }
            $block_events = $block_events->get();
            

        $lastupdatedate = VisitServices::select(DB::RAW('max(update_date) as lastupdatedate'))
            ->whereDate('visit_services.visit_service_start', '>=', $date_start)
            ->whereDate('visit_services.visit_service_end', '<', $date_end);

        if($lastupdate != null && $lastupdate != '' && $lastupdate != 0){
            $lastupdatedate = $lastupdatedate->where('update_date', '>', $lastupdate);
        }            
        $lastupdatedate = $lastupdatedate->first();

        // return "date_start: ". $date_start . " date_end: ". $date_end . " lastupdatedate: ". $lastupdatedate->lastupdatedate.' lastupdate: '.$lastupdate;
       
        $service_rate = $events->sum('service_rate');
        $service_count = count($events);
        return response()->json([
                'mdata' => $events,
                'bdata' => $block_events,
                'res_type' => 'success',
                'msg' => 'Events fetched successfully.',                
                'service_rate' => $service_rate,
                'service_count' => $service_count,
                'lastupdatedate' => $lastupdatedate->lastupdatedate ?? $lastupdate,
        ]);
    }

    public function getCashRegister(Request $request){

        try {
            $cash_register_date = $request->cash_register_date;
            if($cash_register_date == null || $cash_register_date == '' || $cash_register_date == '0'){
                $cash_register_date = date('Y-m-d');
            }

            $business = Business::select('business_name')
                ->where('id_business', session('business_id'))
                ->first();

            // $cash_register = CashRegister::select('*')
            //     ->where('cash_register_date', $cash_register_date)
            //     ->first();

            // $account_heads = AccountHeads::select('account_heads.id_account_heads', 'account_heads.account_head',
            //  'account_events.account_event', 'account_event_mapping.entity_name')
            // ->where('account_heads.business_id', session('business_id'))
            //     ->join('account_event_mapping', 'account_heads.id_account_heads', '=', 'account_event_mapping.account_head_id')
            //     ->join('account_events', 'account_events.id_account_events', '=', 'account_event_mapping.account_event_id')
            //     ->where('id_account_events', 1) // Sales Event                
            //     ->get();

            // Get Service Income using the account heads fetched above
            if(session('ho_accounts') == "Yes") {
                $common_business_id = DB::table('business')
                    ->where('ho', 'Yes')
                    ->first()->id_business;
            } else {
                $common_business_id = session('business_id');
            }

            $income_data = AccountVouchers::select(                
                DB::RAW('ROUND(SUM(COALESCE(account_voucher_detail.debit, 0)), 2) as debit'),
                DB::RAW('ROUND(SUM(COALESCE(account_voucher_detail.credit, 0)), 2) as credit'),
                'account_heads.account_head',
                'account_heads.id_account_heads',
                'account_heads.account_head_number',
                'entity_name as entity'
            )
            ->join('account_voucher_detail', 'account_vouchers.id_account_vouchers', '=', 'account_voucher_detail.account_voucher_id')
            ->join('account_heads', 'account_voucher_detail.account_head_id', '=', 'account_heads.id_account_heads')
            ->join('account_event_mapping as account_events', function($join) {
                $join->on('account_heads.id_account_heads', '=', 'account_events.account_head_id')
                    ->where('account_events.account_event_id', 1); // Sales Event
            })
            ->where('account_events.account_event_id', 1) // Sales Event
            ->where('account_events.business_id', $common_business_id)
            ->where('account_vouchers.business_id', session('business_id'))
            ->where('account_vouchers.voucher_status', 'Active')
            //->whereIn('account_vouchers.business_partner', [1,2]) // Customer Vouchers only
            ->whereDate('account_vouchers.voucher_date', $cash_register_date)
            ->groupBy('account_heads.id_account_heads', 'account_heads.account_head', 
            'account_heads.account_head_number', 'account_events.id_account_event_mapping')
            ->get();
                
            // Get Cash Register Data
            $cash_register_data = CashRegister::select('*')
                ->where('cash_register_date', $cash_register_date)
                ->first();

            return response()->json([
                'res_type' => 'success',
                    'msg' => 'Cash Register fetched successfully.',
                    
                    'invoice_data' => $income_data,
                    'cash_register_date' => $cash_register_date,
                    'business_name' => $business->business_name,
                    'cash_register_data' => $cash_register_data,
                ]);
            
               
            
        } catch (\Exception $e) {
            return response()->json([
                'res_type' => 'error',
                'msg' => 'Error fetching Cash Register: ' . $e->getMessage(),
                'cash_register' => null,
            ]);
        }
    }

    public function save_cash_register(Request $request){
        try {
            $cash_register_date = $request->cash_register_date;
            if($cash_register_date == null || $cash_register_date == ''){
               return response()->json([
                    'res_type' => 'error',
                    'msg' => 'Cash Register Date is required.',
                ]);
            }
            // Check if cash register for the date already exists
            $cash_register = CashRegister::where('cash_register_date', $cash_register_date)
                ->first();

            if ($cash_register) {
                //check if the cash register is closed
                if($cash_register->register_status == 'closed'){
                    return response()->json([
                        'res_type' => 'error',
                        'btn_type' => 'btn-danger',
                        'msg' => 'Cash Register for this date is already closed and cannot be updated.',
                    ]);
                }

                //check if the cash register is open and is not from a previous date
                $current_date = date('Y-m-d');
                if($cash_register->register_status == 'open' && $cash_register_date < $current_date && session('user_role') != 'Super Admin' && session('user_role') != 'Super User'){
                    return response()->json([
                        'res_type' => 'error',
                        'btn_type' => 'btn-danger',
                        'msg' => 'Cash Register for previous dates cannot be updated even if it is still open. Please ask your admin to update and close the Cash Register and Cash Till.',
                    ]);
                }

                // Update existing record
                
                $cash_register->business_id = session('business_id');
                $cash_register->x5000 = $request->x5000 ?? '0';
                $cash_register->x1000 = $request->x1000 ?? '0';
                $cash_register->x500 = $request->x500 ?? '0';
                $cash_register->x100 = $request->x100 ?? '0';
                $cash_register->x75 = $request->x75 ?? '0';
                $cash_register->x50 = $request->x50 ?? '0';
                $cash_register->x20 = $request->x20 ?? '0';
                $cash_register->x10 = $request->x10 ?? '0';
                $cash_register->x5 = $request->x5 ?? '0';
                $cash_register->x1 = $request->x1 ?? '0';
                $cash_register->difference = $request->difference ?? '0';
                $cash_register->daily_expense = $request->daily_expense ?? '0';
                $cash_register->remarks = $request->remarks ?? '';
                $cash_register->cash_register_date = $request->cash_register_date;
                $cash_register->cash_addition = $request->cash_addition ?? '0';
                $cash_register->register_status = $request->register_status;
                $cash_register->till_amounts = $request->till_amounts ?? '0.00';

                $cash_register->save();

                $message = 'Cash Register updated successfully.';
            } else {
                // Create new record
                CashRegister::create([

                    'till_amounts'=> $request->till_amounts ?? '0.00',
                    'business_id'=> session('business_id'),
                    'x5000'=> $request->x5000 ?? '0',
                    'x1000'=> $request->x1000 ?? '0',
                    'x500'=> $request->x500 ?? '0',
                    'x100'=> $request->x100 ?? '0',
                    'x75'=> $request->x75 ?? '0',
                    'x50'=> $request->x50 ?? '0',
                    'x20'=> $request->x20 ?? '0',
                    'x10'=> $request->x10 ?? '0',
                    'x5'=> $request->x5 ?? '0',
                    'x1'=> $request->x1 ?? '0',
                    'difference'=> $request->difference ?? '0',
                    'daily_expense'=> $request->daily_expense ?? '0',
                    'remarks'=> $request->remarks ?? '',
                    'cash_register_date'=> $request->cash_register_date ?? '',
                    'cash_addition'=> $request->cash_addition ?? '0',
                    'register_status'=> $request->register_status ?? '',

                ]);

                $message = 'Cash Register created successfully.';
            }

            return response()->json([
                'res_type' => 'success',
                'btn_type' => 'btn-success',
                'msg' => $message,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'res_type' => 'error',
                'msg' => 'Error saving Cash Register: ' . $e->getMessage(),
            ]);
        }
    }   
  
    public function get_till_closing(Request $request){

        $today = date('Y-m-d');

        if($request->date != null && $request->date != ''){
            $today = date('Y-m-d', strtotime($request->date));
        }

        //Get the till account heads for the business
        if(session('ho_accounts') == "Yes") {
            $common_business_id = DB::table('business')
                ->where('ho', 'Yes')
                ->first()->id_business;
        } else {
            $common_business_id = session('business_id');
        }

        $till_account_heads = AccountEventMapping::select(
            
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "paid_cash" THEN debit ELSE 0 END) as paid_cash'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "paid_card" THEN debit ELSE 0 END) as paid_card'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "paid_online" THEN debit ELSE 0 END) as paid_online'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "paid_check" THEN debit ELSE 0 END) as paid_check'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "paid_voucher" THEN debit ELSE 0 END) as paid_voucher'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "grosstotal_service" THEN credit ELSE 0 END) as grosstotal_service'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "grosstotal_retail" THEN credit ELSE 0 END) as grosstotal_retail'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "grosstotal_training" THEN credit ELSE 0 END) as grosstotal_training'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "sale_discount_service" THEN debit ELSE 0 END) as sale_discount_service'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "sale_discount_retail" THEN debit ELSE 0 END) as sale_discount_retail'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "retained_amount" THEN credit ELSE 0 END) as retained_amount'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "retained_amount_used" THEN debit ELSE 0 END) as retained_amount_used'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "cc_tip" THEN credit ELSE 0 END) as cc_tip'),
            DB::raw('SUM(CASE WHEN account_event_mapping.entity_name = "cc_charge" THEN credit ELSE 0 END) as cc_charge')
        )
            ->join('account_heads', 'account_event_mapping.account_head_id', '=', 'account_heads.id_account_heads')
            ->join('account_voucher_detail', 'account_heads.id_account_heads', '=', 'account_voucher_detail.account_head_id')
            ->join('account_vouchers', 'account_voucher_detail.account_voucher_id', '=', 'account_vouchers.id_account_vouchers')
            ->whereDate('account_vouchers.voucher_date', $today)
            ->where('account_event_mapping.account_event_id', 1) // Till Closing Event
            ->where('account_heads.business_id', $common_business_id ?? session('business_id'))
            ->where('account_vouchers.business_id', session('business_id'))
            ->where('account_vouchers.voucher_status', 'Active')
            ->groupBy(DB::raw('date(account_vouchers.voucher_date)'))
            ->first();
        
        //Get Cash Register Data for today
        $cash_register = CashRegister::select('*',
        DB::RAW('(x5000 * 5000) + (x1000 * 1000) + (x500 * 500) + (x100 * 100) + (x75 * 75) + (x50 * 50) + (x20 * 20) + (x10 * 10) + (x5 * 5) + (x1 * 1) as denomination_total')
        )
            ->where('business_id', session('business_id'))
            ->where('cash_register_date', $today)
            ->first();

        //Get Yesterday's Cash Register to get the previous till amount
        $previous_cash_register = CashRegister::select('till_amounts')
            ->where('business_id', session('business_id'))
            ->where('cash_register_date', date('Y-m-d', strtotime($today . ' -1 day')))
            ->first();

        //Get cash account heads
        $cash_account_heads = AccountHeads::select('id_account_heads', 'account_head')
            ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
            ->where('account_heads.business_id', $common_business_id ?? session('business_id'))
            ->where('account_sub_control.account_sub_control', 'like', '%Cash%')
            ->where('account_heads.account_head', 'not like', '%Till%')
            ->get();

        //Get Bank account heads
        $bank_account_heads = AccountHeads::select('id_account_heads', 'account_head')
            ->join('account_sub_control', 'account_heads.account_sub_control_id', '=', 'account_sub_control.id_account_sub_control')
            ->where('account_heads.business_id', $common_business_id ?? session('business_id'))
            ->where('account_sub_control.account_sub_control', 'like', '%Bank%')
            ->get();

        return response()->json([
            'till_account_heads' => $till_account_heads,
            'cash_register' => $cash_register,
            'previous_till_amount' => $previous_cash_register ? $previous_cash_register->till_amounts : 0,
            'till_date' => $today,
            'cash_account_heads' => $cash_account_heads,
            'bank_account_heads' => $bank_account_heads,
            'message' => 'Till account heads fetched successfully.',
            'success' => true,
            'message_btn' => 'btn btn-success',
        ]);
    }

    public function create_till_closing_voucher(Request $request){
       //return $request->all();
        try {
            DB::beginTransaction();
            $till_date = $request->till_date;
            if($till_date == null || $till_date == ''){
               return response()->json([
                    'res_type' => 'error',
                    'msg' => 'Till Date is required.',
                ]);
            }

            $business = Business::select('id_business', 'business_name')
                ->where('id_business', session('business_id'))
                ->first();


            // Create Account Voucher
            $account_voucher = new AccountVouchers();
            $account_voucher->voucher_date = $till_date;            
            $account_voucher->voucher_type = 3; //Receive Voucher
            $account_voucher->voucher_status = 'Active';
            $account_voucher->created_by = session('user_name');
            $account_voucher->created_on = date('Y-m-d H:i:s');            
            $account_voucher->sale_type = '';        
            $account_voucher->id_account_head = 24; //1 = Accounts Receivable
            $account_voucher->voucher_date = $till_date;
            $account_voucher->description = 'Cash Till Transfer Voucher on Day Closing for '.$till_date.' on '.date('d-m-Y H:i:s').', Branch '.$business->business_name;
            $account_voucher->business_id = $business->id_business;
            $account_voucher->voucher_amount = $request->total_transfer ?? 0;
            $account_voucher->cost_center = 1;
            $account_voucher->cost_center_name = 'Front Desk';
            $account_voucher->business_partner = '5';
            $account_voucher->business_partner_id = $business->id_business;
            $account_voucher->business_partner_name = $business->business_name;          
            $account_voucher->payment_mode = 'Cash';
            $account_voucher->auto_voucher = 'Yes';
            $account_voucher->save();
            $new_voucher_id = $account_voucher->id_account_vouchers;

            // Create Account Voucher Detail for Debit and Credit entries
            //Get Till Account Head
            if(session('ho_accounts')=="Yes"){
                $common_business_id = DB::table('business')
                    ->where('ho', 'Yes')
                    ->first()->id_business;
            }else{
                $common_business_id = session('business_id');
            }

            $till_account_head = AccountEventMapping::select('account_head_id', 'entity_name')
                ->where('business_id', $common_business_id ?? session('business_id'))                
                ->where('entity_name', '=', 'paid_cash') // Till Closing Event - Paid Cash
                ->first();

            // Debit Entry - Cash/Bank Account
            foreach($request->transfers as $account){
                $debit_entry = new AccountVoucherDetail();
                $debit_entry->account_voucher_id = $new_voucher_id;
                $debit_entry->account_head_id = $account['to'];
                $debit_entry->credit = 0;
                $debit_entry->debit = $account['amount'] ?? 0;
                $debit_entry->save();
            }

            // Credit Entry - Till Account
            $credit_entry = new AccountVoucherDetail();
            $credit_entry->account_voucher_id = $new_voucher_id;
            $credit_entry->account_head_id = $till_account_head->account_head_id; // Till Account
            $credit_entry->credit = $request->total_transfer ?? 0;
            $credit_entry->debit = 0;
            $credit_entry->save();

            //update the cash register as closed for the till date
            $cash_register = CashRegister::where('business_id', session('business_id'))
                ->where('cash_register_date', $till_date)
                ->first();

            if($cash_register){
                $cash_register->register_status = 'closed';
                $cash_register->save();
            }

            DB::commit();
            return response()->json([
                'message_type' => 'success',
                'btn_type' => 'btn btn-success',
                'message' => 'Till Closing Voucher creation completed. voucher ID: '.$new_voucher_id,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message_type' => 'error',
                'btn_type' => 'btn btn-danger',
                'message' => 'Error creating Till Closing Voucher: ' . $e->getMessage(),
            ]);
        }
    }

    public function reopen_cash_register(Request $request){
        try {
            $cash_register_id = $request->id_cash_register;
            if($cash_register_id == null || $cash_register_id == ''){
               return response()->json([
                    'res_type' => 'error',
                    'msg' => 'Cash Register ID is required.',
                ]);
            }

            // Find the cash register by ID
            $cash_register = CashRegister::where('id_cash_register', $cash_register_id)
                ->first();

            if (!$cash_register) {
                return response()->json([
                    'res_type' => 'error',
                    'msg' => 'Cash Register not found.',
                ]);
            }

            // Update the register status to 'open'
            $cash_register->register_status = 'open';
            $cash_register->save();

            return response()->json([
                'res_type' => 'success',
                'btn_type' => 'btn-success',
                'msg' => 'Cash Register reopened successfully.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'res_type' => 'error',
                'msg' => 'Error reopening Cash Register: ' . $e->getMessage(),
            ]);
        }
    }

    public function cash_register_print(Request $request){
        $register_date = $request->id_cash_register;        
        
        return view('scheduler.cash_register_print', compact('register_date'));
    }

    public function save_staff_blocking(Request $request){
        try {
            $staff_id = $request->staff_id;
            $staff_name = $request->staff_name;
            $block_date = $request->block_date;
            $block_start_time = $request->block_start_time;
            $block_end_time = $request->block_end_time;
            $block_reason_id = $request->block_reason_id;

            if($staff_id == null || $staff_id == '' || $block_date == null || $block_date == '' || $block_start_time == null || $block_start_time == '' || $block_end_time == null || $block_end_time == '' || $block_reason_id == null || $block_reason_id == ''){
               return response()->json([
                    'message_type' => 'error',
                    'message' => 'All fields are required to block staff time.',
                ]);
            }

            // Create new Staff Blocking Entry and get the id
            $new_block = new BlockStaffTime();
            $new_block->business_id = session('business_id');
            $new_block->block_event_id = $block_reason_id;
            $new_block->block_time_event_id = 0; // No specific event
            $new_block->staff_id = $staff_id;
            $new_block->staff_name = $staff_name;
            $new_block->start_time = $block_date . 'T' . $block_start_time.':00';
            $new_block->end_time = $block_date . 'T' . $block_end_time.':00';
            $new_block->remarks = 'Blocked by ' . session('user_name') . ' on ' . date('d-m-Y H:i:s');
            $new_block->save();
            $new_block_id = $new_block->id_staff_time_blocked;
            //update block_time_event_id with the created id
            
            $new_block->block_time_event_id = 'b'. $new_block_id;
            $new_block->save();

            return response()->json([
                'message' => 'Staff blocking saved successfully.',
                'message_type' => 'success',
                'message_button' => 'btn btn-success',
                'msg' => 'Staff blocking saved successfully.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
                'msg' => 'Error saving staff blocking: ' . $e->getMessage(),
            ]);
        }
    }

    function get_staff_blocking(Request $request){
        $date_start = $request->block_date . ' 00:00:00';
        $date_end = $request->block_date . ' 23:59:59';
        $staff_id = $request->staff_id;

        //get blocked staff events
        $block_events = BlockStaffTime::select(
            'block_time_event_id as id',
            DB::raw('"Yes" as is_blocking_event'),
            'block_event_id',
            'id_staff_time_blocked',
            'staff_id as resourceId',
            'staff_name',
            DB::raw('"blockedClass" as classNames'), // Event Class
            'start_time as start',
            'end_time as end',
            DB::RAW('date_format(start_time, "%H:%i") as block_start_time'),
            DB::RAW('date_format(end_time, "%H:%i") as block_end_time'),
            DB::RAW('block_events.block_event_name as title'), // FullCalendar title
            DB::RAW('"Blocked" as visit_status'),
            DB::RAW('CASE WHEN block_staff_time.business_id = ' . session('business_id') . ' THEN "No" ELSE "Yes" END as staff_shared'),
            'remarks'
        )
        ->join('block_events', 'block_staff_time.block_event_id', '=', 'block_events.id_block_events')
        ->where('block_staff_time.business_id', session('business_id'))
        ->whereDate('start_time', '>=', $date_start)
        ->whereDate('end_time', '<', $date_end);

        if($staff_id != null && $staff_id != '' && $staff_id != 0) {
            $block_events = $block_events->where('staff_id', $staff_id);
        }        
        
        $block_events = $block_events->get();

        return response()->json([
            'blocking_events' => $block_events,
            'message_type' => 'success',
            'message' => 'Blocked staff events fetched successfully.',                
            'message_button' => 'btn btn-success',
        ]);
    }

    public function remove_staff_blocking(Request $request){
        try {
            $id_staff_time_blocked = $request->id_staff_time_blocked;

            if($id_staff_time_blocked == null || $id_staff_time_blocked == ''){
               return response()->json([
                    'message_type' => 'error',
                    'message' => 'Block Time Event ID is required to remove staff blocking.',
                ]);
            }

            // Delete the Staff Blocking Entry
            BlockStaffTime::where('id_staff_time_blocked', $id_staff_time_blocked)
                ->where('business_id', session('business_id'))
                ->delete();

            return response()->json([
                'message' => 'Staff blocking removed successfully.',
                'message_type' => 'success',
                'message_button' => 'btn btn-success',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message_type' => 'error',
                'message_button' => 'btn btn-danger',
                'message' => 'Error removing staff blocking: ' . $e->getMessage(),
            ]);
        }
    }

}


