<?php

namespace App\Http\Controllers;


use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Bus;

use App\Models\Services;
use App\Models\ServiceType;
use App\Models\ServiceCategory;
use App\Models\PackageCategory;
use App\Models\PackageType;
use App\Models\Business;
use App\Models\Department;
use App\Models\ServicesProducts;


use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;
            

class ServicesController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

    public function price_list(Request $request)
    {
        $business_id = session('business_id');

         // if session common services is Yes, 
         if(session("common_services")=="Yes"){
            $mQuery = DB::table('business')
                ->where('ho', 'Yes')->first();
            $business_id = $mQuery->id_business ?? session('business_id');
        } else {
            $business_id = session('business_id');
        }

        //get tax rates
        $tax_rate = DB::table('business_taxes')
        ->select(DB::RAW('sum(tax_percentage) as tax_rate'))
        ->where('business_id', $business_id)
        ->where('tax_active', 'Y')
        ->where('tax_payment_mode', 'Cash')
        ->where('tax_invoice_type', 'service')
        ->groupBy('business_id')
        ->first();

        $tax_rate_value = $tax_rate->tax_rate ?? 0;

        // get services
        $service_types = ServiceType::where('service_type.business_id', $business_id)->where('service_type_active', 'Yes')->get();
        $service_categories = ServiceCategory::where('service_category.business_id', $business_id)->where('service_category_active', 'Yes')
        ->join('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
        ->where('service_type.service_type_active', 'Yes')
        ->get();
        $services = Services::where('business_services.business_id', $business_id)->where('service_active', 'Yes')
        ->leftJoin('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
        ->leftJoin('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
        ->where('service_category.service_category_active', 'Yes')
        ->where('service_type.service_type_active', 'Yes')
        ->get();

        if(session("ho")=="Yes"){
            $businesses = Business::where('common_services', 'No')->get();
        } else {
            $businesses = Business::where('id_business', $business_id)->get();
        }

        return view('services.price_list', compact('businesses', 'service_types', 'service_categories', 'services', 'business_id', 'tax_rate_value'));
    }

    public function get_services_list(Request $request)
    {
        // if session common_services is Yes, get business_type 
        $business_type=0;
        if(session("common_services")=="Yes"){
            $mQuery = DB::table('business')
                ->where('ho', 'Yes');
            $firstrow = $mQuery->first();
            $business_type = $firstrow->business_type_id ?? 0;
        } else {
            $business_type = 0;
        }

        // First query: services linked via service_category → service_type
        $q1 = DB::table('business_services as bs')
            ->select(
                'bs.*',
                'sc.id_service_category as category_id',
                'sc.service_type_id',
                DB::raw('"servicetype" as service_flag'),
                //'bs.service_rate as service_price',
                DB::raw('COALESCE(services_business_mapping.price, bs.service_rate) as service_price'),
                'st.service_type as service_type_name',
                'sc.service_category as service_category_name'
            )
            ->join('business as b', 'b.id_business', '=', 'bs.business_id')
            ->leftJoin('service_category as sc', 'sc.id_service_category', '=', 'bs.service_category_id')
            ->leftJoin('service_type as st', 'st.id_service_types', '=', 'sc.service_type_id')
            ->leftJoin("services_business_mapping", function($join) {
                $join->on("services_business_mapping.service_id", "=", "bs.id_business_services")
                ->Where('services_business_mapping.business_id', session('business_id'));
            })     
            ->where('bs.service_active', 'Yes')
            ->where('sc.service_category_active', 'Yes')
            ->where('st.service_type_active', 'Yes');
           
            if(session("common_services")=="No"  || empty(session("common_services"))  || null == session("common_services")){
                $q1->where('bs.business_id', '=', session('business_id'));
            }else {
                $q1->where('b.ho', '=', "Yes");
            }

            $q1->where(function($query) {
                $query->whereNull('services_business_mapping.is_visible') // if no mapping exists, include it
                    ->orWhere('services_business_mapping.is_visible', 'Yes'); // include only visible
            });


        // Second query: services linked via package_services → package_category → package_type
        $q2 = DB::table('business_services as bs')
            ->select(
                'bs.*',
                'pc.id_package_category as category_id',
                'pc.package_type_id as service_type_id',
                DB::raw('"packagetype" as service_flag'),
                //'ps.service_rate as service_price',
                DB::raw('COALESCE(package_services_business_mapping.price, ps.service_rate) as service_price'),
                'pt.service_type as service_type_name',
                'pc.service_category as service_category_name'
            )
            ->join('business as b', 'b.id_business', '=', 'bs.business_id')
            ->leftJoin('package_services as ps', 'ps.service_id', '=', 'bs.id_business_services')
            ->leftJoin("package_services_business_mapping", function($join) {
                $join->on("package_services_business_mapping.package_services_id", "=", "ps.id_package_services")
                ->Where('package_services_business_mapping.business_id', session('business_id'));
            })     
            ->leftJoin('package_category as pc', 'pc.id_package_category', '=', 'ps.package_category_id')
            ->leftJoin('package_type as pt', 'pt.id_package_type', '=', 'pc.package_type_id')
            ->where('bs.service_active', 'Yes')
            ->where(DB::RAW('ifnull(ps.option_for, 0)'), '=', 0);
            if(session("common_services")=="No"  || empty(session("common_services"))  || null == session("common_services")){
                $q2->where('bs.business_id', '=', session('business_id'));
            }else {
                $q2->where('b.ho', '=', "Yes");
            }
            $q2->where(function($query) {
                $query->whereNull('package_services_business_mapping.is_visible') // if no mapping exists, include it
                    ->orWhere('package_services_business_mapping.is_visible', 'Yes'); // include only visible
            });
        // Merge them with UNION ALL
        $query = $q1->unionAll($q2);
        $services_list = DB::table(DB::raw("({$query->toSql()}) as combined"))
            ->mergeBindings($query) // you need to get the bindings from the union query            
            ->orderBy('combined.order_id')
            ->get();

            //print_r($services_list); exit();

        ///Service Types
        $q1_1 = DB::table("service_type")->select("service_type.id_service_types", "service_type.service_type", "service_type.business_id", "service_type.order_id", DB::raw('"servicetype" as serviceflag'),
        DB::RAW('COALESCE(service_type.service_type_image, "nu.png") as service_type_image')
        )
        ->join("business", "business.id_business", "=", "service_type.business_id")
        ->leftJoin("service_type_business_mapping", function($join) {
            $join->on("service_type_business_mapping.service_type_id", "=", "service_type.id_service_types")
            ->Where('service_type_business_mapping.business_id', session('business_id'));
        });       
        
        if(session("common_services")=="No" || empty(session("common_services"))  || null == session("common_services")){
            $q1_1->where('service_type.business_id', '=', session('business_id'));
        }else {
            $q1_1->where('business.ho', '=', "Yes");
        }
        $q1_1->where('service_type.service_type_active', 'Yes');
        if($business_type > 0){ 
           // $q1_1->where('service_type.business_type_id', '=', $business_type); // Restrict by business type
        }
        
        $q1_1->where(function($query) {
            $query->whereNull('service_type_business_mapping.is_visible') // if no mapping exists, include it
                ->orWhere('service_type_business_mapping.is_visible', 'Yes'); // include only visible
        });


        $q1_2 = DB::table("package_type")->select("package_type.id_package_type as id_service_types", "package_type.service_type", "package_type.business_id",  "package_type.order_id", DB::raw('"packagetype" as serviceflag'),
        DB::RAW('COALESCE(package_type.service_type_image, "nu.png") as service_type_image')
        )
        ->join("business", "business.id_business", "=", "package_type.business_id")
        ->leftJoin("package_type_business_mapping", function($join) {
            $join->on("package_type_business_mapping.package_type_id", "=", "package_type.id_package_type")
            ->Where('package_type_business_mapping.business_id', session('business_id'));
        });
        
        if(session("common_services")=="No"  || empty(session("common_services"))  || null == session("common_services")){
            $q1_2->where('package_type.business_id', '=', session('business_id'));
        }else {
            $q1_2->where('business.ho', '=', "Yes");
        }
        $q1_2->where('package_type.service_type_active', 'Yes');
        if($business_type > 0){ 
           // $q1_2->where('package_type.business_type_id', '=', $business_type); // Restrict by business type
        }
        $q1_2->where('package_type.expiry_date', '>=' , Carbon::now()->format('Y-m-d'));
        $q1_2->where(function($query) {
            $query->whereNull('package_type_business_mapping.is_visible')
                ->orWhere('package_type_business_mapping.is_visible', 'Yes');
        });

        $query1 = $q1_1->union($q1_2);
        $service_types = DB::table(DB::raw("({$query1->toSql()}) as combined"))
        ->mergeBindings($query1)
        ->orderBy("serviceflag", "DESC")->orderBy("order_id")
        ->get();


        ///Service Categories
        $q2_1 = DB::table("service_category")->select("service_category.id_service_category", "service_type.service_type", "service_category.service_category", "service_category.service_type_id", "service_category.business_id",  DB::RAW('COALESCE(service_category.service_category_image, "nu.png") as service_category_image'), "service_category.order_id", DB::raw('"servicetype" as serviceflag'))
        ->join("business", "business.id_business", "=", "service_category.business_id")
        ->join("service_type", "service_type.id_service_types", "=", "service_category.service_type_id");
        if(session("common_services")=="No"  || empty(session("common_services"))  || null == session("common_services")){
            $q2_1->where('service_category.business_id', '=', session('business_id'));
        }else {
            $q2_1->where('business.ho', '=', "Yes");
            if($business_type > 0){
                //$q2_1->where('service_type.business_type_id', '=', $business_type); // Restrict by business type
            }
        }
        $q2_1->where('service_category.service_category_active', 'Yes');        
        $q2_1->where('service_type.service_type_active', 'Yes');

        $q2_2 = DB::table("package_category")->select('package_category.id_package_category as id_service_category', 'package_type.service_type', 'package_category.service_category as service_category', 'package_category.package_type_id as service_type_id', 'package_category.business_id', DB::RAW('COALESCE(package_category.service_category_image, "nu.png") as service_category_image'), 'package_category.order_id', DB::raw('"packagetype" as serviceflag'))
        ->join("business", "business.id_business", "=", "package_category.business_id")
        ->join("package_type", "package_type.id_package_type", "=", "package_category.package_type_id");
        if(session("common_services")=="No"  || empty(session("common_services"))  || null == session("common_services")){
            $q2_2->where('package_category.business_id', '=', session('business_id')); 
        }else {
            $q2_2->where('business.ho', '=', "Yes");
            if($business_type > 0){
                //$q2_2->where('package_type.business_type_id', '=', $business_type); // Restrict by business type
            }
        }
        $q2_2->where('package_category.service_category_active', 'Yes');
        $q2_2->where('package_type.service_type_active', 'Yes');        
        $query2 = $q2_1->union($q2_2);
        $service_categories = DB::table(DB::raw("({$query2->toSql()}) as combined"))
        ->mergeBindings($query2)
        ->orderBy('order_id')
        ->get();
        

        return response()->json([           
            "service_types" => $service_types, 
            "services_list" => $services_list,
            "service_categories" => $service_categories
        ]);

        // return response()->json([
        //     "debug" => [
        //         "session" => [
        //             "common_services" => session('common_services'),
        //             "business_id" => session('business_id'),
        //             "business_type" => session('business_type')
        //         ],
        //         "counts" => [
        //             "types" => $service_types->count(),
        //             "categories" => $service_categories->count(),
        //             "services" => $services_list->count()
        //         ]
        //     ],
        //     "service_categories" => $service_categories
        // ]);


    }

    public function services(Request $request)
    {
        $business=Business::select("ho", "business_name", "id_business", "common_services", "common_products")
        ->where('id_business', session('business_id'))->first();

       if ($business->common_services == 'Yes' && $business->ho == 'No') {
            // Redirect unauthorized users to a dedicated page
           return response()->view('error.403', ['message' => 'Your services are managed by the Head Office.'], 403);
        }

        
        $service_type = ServiceType::select("*", DB::RAW('COALESCE(service_type_image, "nu.png") as service_type_image'));
        $service_type = $service_type->leftJoin('business_type', 'business_type.id_business_type', '=', 'service_type.business_type_id');
        
        if($business->common_services=="Yes"){
            $service_type = $service_type->join("business", "business.id_business", "=", "service_type.business_id")->where('business.ho', '=', "Yes");
        } else {
            $service_type = $service_type->where('business_id', session('business_id'));
        }
        
        $service_type = $service_type->orderby('order_id', 'ASC');
        $service_type = $service_type->get();

        $departments = Department::where('business_id', session('business_id'))->where('department_status', 'Active')->get();

        $business_types = DB::table('business_type')
        ->select('business_type.id_business_type', 'business_type.business_type');
         $business_types = $business_types->join("business", "business.business_type_id", "=", "business_type.id_business_type");
        if(session('common_services')=="Yes"){
            $business_types = $business_types->where('business.common_services', '=', "Yes");
        } else {
            $business_types = $business_types->where('business.id_business', '=', session('business_id'));
        }
        $business_types = $business_types->get();

        $business_with_common_services =  DB::table('business')->select('id_business','business_name')->where('common_services','Yes')->get();

        return view('services.services', compact('business', 'service_type', 'departments', 'business_types','business_with_common_services'));
    }

    public function get_categories_by_service_type(Request $request)
    {
        $service_type_id = $request->input('id_service_type');
        if(empty($service_type_id)){
            return response()->json([
                "message" => "Error",
                "service_category" => []
            ]);
        }

        $service_category = ServiceCategory::select("*", DB::RAW('COALESCE(service_category_image, "nu.png") as service_category_image'));
       
        if(!empty($service_type_id)){
            $service_category = $service_category->where('service_type_id', $service_type_id);
        }
        $service_category = $service_category->orderby('order_id', 'ASC');
        $service_category = $service_category->get();

        return response()->json([
            "message" => "Success",
            "service_category" => $service_category
        ]);
    }

    public function get_services_by_category(Request $request)
    {
        $category_id = $request->input('id_service_category');
        if(empty($category_id)){
            return response()->json([
                "message" => "Error",
                "services" => []
            ]);
        }

        $services = Services::select("business_services.*", DB::RAW('COALESCE(service_category.service_category_image, "nu.png") as service_image'))
        ->join('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
        ->where('service_category_id', $category_id)
        ->orderby('order_id', 'ASC')
        ->get();

        return response()->json([
            "message" => "Success",
            "services" => $services
        ]);
    }

    public function get_existing_type_images(Request $request)
    {
        $path = public_path('images/services/types');
        $urls = [];

        if (File::exists($path)) {
            foreach (File::files($path) as $file) {
                $urls[] = asset('images/services/types/' . $file->getFilename());
            }
        }

        return response()->json($urls);
    }

    public function type_uploadimage(Request $request)
    {
        $request->validate([
            'file' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // max 2MB
        ]);

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('images/services/types'), $filename);

            return response()->json(['filename' => $filename]);
        }

        return response()->json(['error' => 'No file uploaded'], 400);
    }

    public function update_service_type_image(Request $request)
    {

        if(empty($request->id_service_type)){
            return response()->json(['error' => 'Service type ID is required'], 400);
        }

        $typeimage = "nu.png";
        if(!$request->input('service_type_image')){
            $typeimage = $request->input('existing_type_image');
        } else {
            $typeimage = $request->input('service_type_image');
        }

        $service_type = ServiceType::find($request->id_service_type);
        if ($service_type) {
            $service_type->service_type_image = $typeimage;
            $service_type->save();

            return response()->json(['success' => true, 'message' => 'Service type image updated successfully']);
        }

        return response()->json(['error' => true, 'message' => 'Service type not found'], 404);
    }

    public function get_existing_category_images(){
        $path = public_path('images/services/categories');
        $urls = [];

        if (File::exists($path)) {
            foreach (File::files($path) as $file) {
                $urls[] = asset('images/services/categories/' . $file->getFilename());
            }
        }

        return response()->json($urls);
    }

    public function category_uploadimage(Request $request)
    {
        $request->validate([
            'file' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // max 2MB
        ]);

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('images/services/categories'), $filename);

            return response()->json(['filename' => $filename]);
        }

        return response()->json(['error' => 'No file uploaded'], 400);
    }

    public function update_service_category_image(Request $request)
    {

        if(empty($request->id_service_category)){
            return response()->json(['error' => 'Service category ID is required'], 400);
        }

        $categoryimage = "nu.png";
        if(!$request->input('service_category_image')){
            $categoryimage = $request->input('existing_category_image');
        } else {
            $categoryimage = $request->input('service_category_image');
        }

        $service_category = ServiceCategory::find($request->id_service_category);
        if ($service_category) {
            $service_category->service_category_image = $categoryimage;
            $service_category->save();

            $id_service_type = $service_category->service_type_id;

            return response()->json(['success' => true, 'message' => 'Service category image updated successfully', 'id_service_type' => $id_service_type]);
        }

        return response()->json(['error' => true, 'message' => 'Service category not found'], 404);
    }

    public function get_service_type($id_service_type)
    {
        if(empty($id_service_type)){
            return response()->json([
                "message" => "Error",
                "service_type" => null
            ]);
        }

        $service_type = ServiceType::select("*", DB::RAW('COALESCE(service_type_image, "nu.png") as service_type_image'))
        ->leftJoin('business_type', 'business_type.id_business_type', '=', 'service_type.business_type_id')
        ->where('id_service_types', $id_service_type)
        ->first();

        $service_type_business_mapping = DB::table('service_type_business_mapping')
        ->where('service_type_id',$id_service_type)
        ->get() ?? [];

        return response()->json([
            "message" => "Success",
            "service_type" => $service_type,
            "service_type_business_mapping" => $service_type_business_mapping,
        ]);
    }

    public function save_service_type(Request $request)
    {
        try {
            // Validation
            $request->validate([
                'service_type' => 'required|string|max:255',
                'service_type_image' => 'nullable|string|max:255',
                'business_type_id' => 'nullable|integer',
            ]);

            // Prepare identifier
            $id = $request->input('id_service_type');

            // Insert or update using updateOrCreate
            $service_type = ServiceType::updateOrCreate(
                ['id_service_types' => $id], // Make sure this matches your actual primary key column name
                [
                    'service_type' => $request->input('service_type'),
                    'order_id' => $request->input('order_id') ?? 0,
                    'business_id' => session('business_id'),
                    'service_type_active' => $request->input('service_type_active', 'Yes'),
                    'service_type_image' => $request->input('service_type_image', 'nu.png'),
                    'created_by' => session('user_name'),
                    // 'business_type_id' => $request->input('business_type_id', null),
                    'business_type_id' =>  null,
                ]
            );

           $serviceTypeId = $service_type->id_service_types;
           if(isset($request->service_type_visibility_unchecked)){
                foreach ($request->service_type_visibility_unchecked as $businessId) {
                    $exists = DB::table('service_type_business_mapping')
                        ->where('business_id', $businessId)
                        ->where('service_type_id', $serviceTypeId)
                        ->exists();

                    if (!$exists) {
                        DB::table('service_type_business_mapping')->insert([
                            'business_id'     => $businessId,
                            'service_type_id' => $serviceTypeId,
                            'is_visible'      => 'No',
                            'created_by'      => session('user_name'),
                            'created_at'      => now(),
                        ]);
                    }
                }
            }

            if(isset($request->service_type_visibility_checked)){
                foreach ($request->service_type_visibility_checked as $businessId) {
                    DB::table('service_type_business_mapping')
                        ->where('business_id', $businessId)
                        ->where('service_type_id', $serviceTypeId)
                        ->update([
                            'is_visible' => 'Yes',
                            'updated_at' => now(),
                        ]);
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Service type saved successfully',
                'id_service_type' => $service_type->id_service_type,
                'service_type' => $service_type
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    public function get_service_category($id_service_category)
    {
        if(empty($id_service_category)){
            return response()->json([
                "message" => "Error",
                "service_category" => null
            ]);
        }

        $service_category = ServiceCategory::select("*", DB::RAW('COALESCE(service_category_image, "nu.png") as service_category_image'))
        ->where('id_service_category', $id_service_category)
        ->first();

        return response()->json([
            "message" => "Success",
            "service_category" => $service_category
        ]);
    }

    public function save_service_category(Request $request)
    {
        // return response()->json($request->all());

        try {
           
            // Validation
            $request->validate([
                'service_category' => 'required|string|max:255',
                'service_type_id' => 'required|integer',
                'order_id' => 'required|integer',
                'service_category_image' => 'nullable|string|max:255',
            ]);

            // Prepare identifier
            $id = $request->input('id_service_category');

            // Insert or update using updateOrCreate
            $service_category = ServiceCategory::updateOrCreate(
                ['id_service_category' => $id], // Make sure this matches your actual primary key column name
                [
                    'service_category' => $request->input('service_category'),
                    'service_type_id' => $request->input('service_type_id'),
                    'order_id' => $request->input('order_id'),
                    'business_id' => session('business_id'),
                    'service_category_active' => $request->input('service_category_active', 'Yes'),
                    'service_category_image' => $request->input('service_category_image', 'nu.png'),
                    'created_by' => session('user_name'),
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Service category saved successfully',
                'id_service_category' => $service_category->id_service_category,
                'service_category' => $service_category
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    public function get_service($id_service)
    {
        if(empty($id_service)){
            return response()->json([
                "message" => "Error",
                "service" => null
            ]);
        }

        $service = Services::select("business_services.*", DB::RAW('COALESCE(service_category.service_category_image, "nu.png") as service_image'))
        ->leftJoin('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
        ->where('business_services.id_business_services', $id_service)
        ->first();

        $services_business_mapping = DB::table('services_business_mapping')->where('service_id',$id_service)->get();

        return response()->json([
            "message" => "Success",
            "service" => $service,
            "services_business_mapping"=> $services_business_mapping
        ]);  
    }

    public function save_service(Request $request)
    {
        // return response()->json($request->all());

        try {
           
            // Validation
            $request->validate([
                'service_name' => 'required|string|max:255',
                'service_category_id' => 'nullable|integer',
                'order_id' => 'required|integer',
                'service_rate' => 'required|numeric',
                'service_active' => 'required|string|max:10',
                'commission_type' => 'nullable|string|max:20',
                'commission_perc' => 'nullable|numeric',
                'commission_fixed' => 'nullable|numeric',
                'helper_perc' => 'nullable|numeric',
                'helper_fixed' => 'nullable|numeric',
                'service_color' => 'nullable|string|max:20',
                'service_duration' => 'nullable|string|max:20',
                'max_per_day' => 'nullable|integer',
                'service_mode' => 'nullable|string|max:20',
                'service_department' => 'nullable|string|max:50',
            ]);

            // Prepare identifier
            $id = $request->input('id_business_services');

            // Insert or update using updateOrCreate
            $service = Services::updateOrCreate(
                ['id_business_services' => $id], // Make sure this matches your actual primary key column name
                [
                    'service_name' => $request->input('service_name'),
                    'service_desc' => $request->input('service_desc'),
                    'service_category_id' => $request->input('service_category_id'),
                    'order_id' => $request->input('order_id'),
                    'business_id' => session('business_id'),
                    'service_color' => $request->input('service_color', ''),
                    'service_rate' => $request->input('service_rate'),
                    'service_active' => $request->input('service_active', 'Yes'),
                    'commission_type' => $request->input('commission_type', ''),
                    'commission_perc' => $request->input('commission_perc', 0),
                    'commission_fixed' => $request->input('commission_fixed', 0),
                    'helper_perc' => $request->input('helper_perc', 0),
                    'helper_fixed' => $request->input('helper_fixed', 0),
                    'service_color' => $request->input('service_color', ''),
                    'service_duration' => $request->input('service_duration', '00:30:00'),
                    'max_per_day' => $request->input('max_per_day', 0),
                    'service_mode' => $request->input('service_mode', ''),
                    'service_department' => $request->input('service_department', ''),
                    'created_by' => session('user_name'),
                ]
            );

            $serviceId  =  $service->id_business_services;

            // Decode JSON arrays 
            $checked = json_decode($request->services_visibility_checked, true) ?? [];
            $unchecked = json_decode($request->services_visibility_unchecked, true) ?? [];

            foreach ($unchecked as $row) {
                $businessId = $row['business_id'];
                $price = isset($row['price']) ? $row['price'] : null;

                $exists = DB::table('services_business_mapping')
                    ->where('business_id', $businessId)
                    ->where('service_id', $serviceId)
                    ->exists();

                if ($exists) {
                    DB::table('services_business_mapping')
                        ->where('business_id', $businessId)
                        ->where('service_id', $serviceId)
                        ->update([
                            'is_visible' => 'No',
                            'price'      => $price !='' ? $price : null,
                            'updated_at' => now(),
                        ]);
                } else {
                    DB::table('services_business_mapping')->insert([
                        'business_id'         => $businessId,
                        'service_id' => $serviceId,
                        'is_visible'          => 'No',
                        'price'               => $price !='' ? $price : null,
                        'created_by'          => session('user_name'),
                        'created_at'          => now(),
                    ]);
                }
            }


            foreach ($checked as $row) {
                $businessId = $row['business_id'];
                $price = isset($row['price']) ? $row['price'] : null;

                $exists = DB::table('services_business_mapping')
                    ->where('business_id', $businessId)
                    ->where('service_id', $serviceId)
                    ->exists();

                if ($exists) {
                    DB::table('services_business_mapping')
                        ->where('business_id', $businessId)
                        ->where('service_id', $serviceId)
                        ->update([
                            'is_visible' => 'Yes',
                            'price'      => $price !='' ? $price : null,
                            'updated_at' => now(),
                        ]);
                } else {
                    DB::table('services_business_mapping')->insert([
                        'business_id'         => $businessId,
                        'service_id' => $serviceId,
                        'is_visible'          => 'Yes',
                        'price'               => $price !='' ? $price : null,
                        'created_by'          => session('user_name'),
                        'created_at'          => now(),
                    ]);
                }
            }

            return response()->json([
                'success' => true,
                "message" => "Update Service Successful",
                "service" => $service
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error " . $e->getMessage(),
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function get_service_recipes($id_service)
    {
        if(empty($id_service)){
            return response()->json([
                "message" => "Error",
                "recipes" => []
            ]);
        }
        try{
            $recipes = Services::select('services_products.usage_qty', 'services_products.id_services_products', 'services_products.business_product_id', 'business_services.service_name', 'bp.product', 'bp.sku', 'bp.unit_type', 'bp.measure_unit', 'bp.qty_per_unit')
            ->leftJoin('services_products', 'business_services.id_business_services', '=', 'services_products.business_service_id')
            ->leftJoin('business_products as bp', 'bp.id_business_products', '=', 'services_products.business_product_id')
            ->where('business_services.id_business_services', $id_service)
            ->where('services_products.status', 'Y')
            ->get();

            return response()->json([
                "message" => "Success",
                "recipes" => $recipes
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "message" => "error",
                "error" => $e->getMessage()
            ], 500);
        }
    }

    public function deactivate_service_product($id_services_products)
    {
        $service_product = ServicesProducts::find($id_services_products);
        if ($service_product) {
            $service_product->update([
                'status' => 'N'
            ]);

            return response()->json(['success' => true, 'message' => 'Service product deactivated successfully']);
        }

        return response()->json(['error' => true, 'message' => 'Service product not found'], 404);
    }
    
    public function save_service_recipe(Request $request)
    {
        //return response()->json($request->all());

        try {
         
            // Prepare identifier
            $service_id = $request->input('service_id');

            if(empty($service_id)){
                return response()->json([
                    'error' => true,
                    'message' => 'Service ID is required.'
                ], 400);
            }
            
            // Decode the JSON string into an array
            $rows = json_decode($request->input('recipes'), true);

            if(empty($rows)){
                return response()->json([
                    'error' => true,
                    'message' => 'No recipe items provided.'
                ], 400);
            }

            foreach($rows as $row){
                $business_product_id = $row['business_product_id'];
                //update or insert each recipe item based on business_product_id and business_service_id
                $existing = ServicesProducts::where('business_service_id', $service_id)
                    ->where('business_product_id', $business_product_id)
                    ->first();
                if($existing){
                    //update
                    $existing->update([
                        'usage_qty' => $row['usage_qty'],
                        'status' => 'Y',
                        'created_by' => session('user_name'),
                    ]);
                }else{
                    //insert
                    ServicesProducts::create([
                        'business_service_id' => $service_id,
                        'business_product_id' => $row['business_product_id'],
                        'usage_qty' => $row['usage_qty'],
                        'status' => 'Y',
                        'created_by' => session('user_name'),
                    ]);
                }
            }
            $service_products = ServicesProducts::where('business_service_id', $service_id)->get();
            return response()->json([
                'success' => true,
                'message' => 'Service recipe product saved successfully',
                'service_products' => $service_products
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update_type_order(Request $request)
    {
      
        $order = $request->input('order');

        foreach ($order as $index => $item) {
            ServiceType::where('id_service_types', $item['id_service_types'])->update(['order_id' => $index + 1]);
        }

        return response()->json(['success' => true, 'message' => 'Service type order updated successfully']);
    }

    public function update_category_order(Request $request)
    {
        $return_type_id = $request->input('return_type_id');
      
        $order = $request->input('order');

        foreach ($order as $index => $item) {            
            ServiceCategory::where('id_service_category', $item['id_service_category'])->update(['order_id' => $index + 1]);
        }

        return response()->json(['success' => true, 'message' => 'Service category order updated successfully',  'return_type_id' => $return_type_id ]);
    }

    public function update_service_order(Request $request)
    {
       $return_category_id = $request->input('return_category_id');

        $order = $request->input('order');

        foreach ($order as $index => $item) {
            Services::where('id_business_services', $item['id_business_services'])->update(['order_id' => $index + 1]);
        }

        return response()->json(['success' => true, 'message' => 'Service order updated successfully', 'return_category_id' => $return_category_id]);
    }

    public function open_all_services_list(Request $request)
    {
        if(session('common_services')=="Yes"){ 
            // Business ID exists in session
            if(session('ho')=="No"){
                // If HO, set business_id to HO business
                $business_ho = Business::select("id_business")
                ->where('ho', 'Yes')
                ->first();
            }else{
                $business_id = session('business_id');
            }
        } else {
            $business_id = session('business_id');
        }

        $services = Services::select("*", 
        DB::RAW('COALESCE(service_category.service_category_image, "nu.png") as service_image'),
        DB::RAW('COALESCE(service_type.service_type_image, "nu.png") as service_type_name')
        
        )
        ->leftJoin('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
        ->leftJoin('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
        ->join('business', 'business.id_business', '=', 'business_services.business_id')
        ->leftJoin('business_type', 'business_type.id_business_type', '=', 'service_type.business_type_id')
        ->where('business_services.business_id', session('business_id'))
        ->orderBy('service_type.id_service_types', 'ASC')
        ->orderBy('service_category.id_service_category', 'ASC')
        ->orderBy('business_services.order_id', 'ASC')
        ->get();

        return view('services.all_service_list', compact('services'));
    }

    public function search_service(Request $request)
    {

        $business_id = session('business_id');

        if(session('common_services')=="Yes"){ 
            // Business ID exists in session
            if(session('ho')=="No"){
                // If HO, set business_id to HO business
                $business_ho = Business::select("id_business")
                ->where('ho', 'Yes')
                ->first();
                $business_id = $business_ho->id_business;
            }else{
                $business_id = session('business_id');
            }
        } else {
            $business_id = session('business_id');
        }

        $search_term = $request->input('search_term');
        $services = Services::select("id_business_services as id", "id_business_services", DB::RAW(" IFNULL(business_type.business_type,'') as business_type "), 
        "service_category.service_category", "service_type.service_type",
        "business_services.service_name", 
        #DB::RAW("ROUND(business_services.service_rate) as service_rate"), 
        DB::raw('COALESCE(services_business_mapping.price, business_services.service_rate) as service_rate'),
        "business_services.service_duration",
        DB::RAW('COALESCE(service_category.service_category_image, "nu.png") as service_image')
        )
        ->Join('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
        ->Join('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
        ->LeftJoin('business_type', 'business_type.id_business_type', '=', 'service_type.business_type_id')
        ->leftJoin("services_business_mapping", function($join) {
                $join->on("services_business_mapping.service_id", "=", "business_services.id_business_services")
                ->Where('services_business_mapping.business_id', session('business_id'));
        })     
        ->where('business_services.business_id', $business_id)
        ->where('business_services.service_active', 'Yes')
        ->where('service_type.service_type_active', 'Yes')
        ->where('service_category.service_category_active', 'Yes')
        ->where(function($query) use ($search_term) {
            $query->where('business_services.service_name', 'LIKE', '%' . $search_term . '%');
        })
        ->where(function($query) {
                $query->whereNull('services_business_mapping.is_visible') // if no mapping exists, include it
                    ->orWhere('services_business_mapping.is_visible', 'Yes'); // include only visible
        })
        ->get();

        return $services;
    }

    public function service_department_search(Request $request){

           $search = request()->query('term');

       if(!$search || strlen($search) < 3) {
           return response()->json(['results' => []]);
       }

        $business_id = request()->query('business_id', session('business_id'));
        if($business_id == null || $business_id == '' || $business_id == 0) {
           if(session('ho')=="Yes"){
               $business_id = 0; // for HO, show all staff across businesses
           } else {
               $business_id = session('business_id');
           }            
        } else {
           $business_id = session('business_id');
        } 

        $service_departments = DB::table('service_department')
            ->select(
                'service_department.id_service_department as id',
                'service_department.service_department as text'
            )
            ->join('business', 'service_department.business_id', '=', 'business.id_business')
            ->where(function($query) use ($search) {
                $query->where('service_department', 'LIKE', "%$search%");
            })
            ->when($business_id != 0, function($query) use ($business_id) {
                $query->where('service_department.business_id', $business_id);
            }, function($query) {
                // for HO, show staff only from businesses with common_products = Yes
                $query->where('business.common_services', 'Yes');
            })
            ->get();
       return response()->json($service_departments);
    }
}