<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\PermissionCategory;
use App\Models\Permission;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\Models\Role;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RoleManagementController extends Controller
{
    /**
     * Temporarily switch cache to array driver to avoid database cache table issues
     */
    protected function useArrayCache(): void
    {
        // Temporarily use array cache which doesn't require a database table
        config(['permission.cache.store' => 'array']);
        config(['cache.default' => 'array']);
    }

    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:spatie_permissions,name'
        ]);

        Role::create(['name' => $request->name]);

        return response()->json([
            'success' => true,
            'message' => 'Role created successfully'
        ]);
    }
    /**
     * Display the role management page.
     */
    public function index()
    {
        $permissions = Permission::all();
        $totalPermissions = Permission::count();
        $roles = Role::withCount(['permissions', 'users'])
        ->where('is_template', 0)
        ->orderBy('name')
        ->get();
        
        // Filter out Super User role if current user is not a Super User
        $currentUser = Auth::user();
        if (!$currentUser->hasRole('Super User')) {
            $roles = $roles->filter(function($role) {
                return $role->name !== 'Super User';
            });
        }
        
        $roles = $roles->map(function ($role) use ($totalPermissions) {
            $role->total_permissions = $totalPermissions;
            $role->total_unassigned_permissions = $totalPermissions - $role->permissions_count;
            return $role;
        });
        
        return view('settings.role-management.roles', compact('roles', 'permissions'));
    }

    /**
     * Get live counts for a single role (for AJAX card updates).
     */
    public function getRoleCounts($roleId)
    {
        $role = Role::withCount(['permissions', 'users'])->findOrFail($roleId);
        $totalPermissions = Permission::count();
        return response()->json([
            'success' => true,
            'role_id' => $role->id,
            'permissions_count' => $role->permissions_count,
            'totalPermissions' => $totalPermissions,
            'users_count' => $role->users_count,
        ]);
    }

    /**
     * Get all permissions grouped by category for a role
     */
    public function getRolePermissions($roleId)
    {
        try {

            $role = Role::with('permissions')->findOrFail($roleId);
    
            $assignedPermissionIds = $role->permissions->pluck('id')->toArray();
            
            $all_permission = Permission::with('childs')
            ->where(function($q){
                $q->whereNull('parent_id');
                $q->orWhere('parent_id', '0');
            })
            ->get();
    
            return response()->json([
                'success' => true,
                'assignedPermissionIds' => $assignedPermissionIds,
                'all_permission' => $all_permission
            ]);
            
        } catch (Exception $e) {
            return response()->json([
                'success' => true,
                'message' => $e->getMessage()
            ]);
        }

    }

    /**
     * Get all users for a role
     */
    public function getRoleUsers($roleId)
    {
        $role = Role::findOrFail($roleId);
        $allUsers = User::orderBy('user_name')->get(['id_users', 'user_name', 'user_fullname', 'user_email', 'user_hidden']);
        $assignedUsers = $role->users;
        $assignedUserIds = $assignedUsers->pluck('id_users')->toArray();
        
        // Filter out users with user_hidden = "Yes" if current user is not a Super User
        $currentUser = Auth::user();
        if (!$currentUser->hasRole('Super User')) {
            $allUsers = $allUsers->filter(function($user) {
                return $user->user_hidden !== 'Yes';
            });
            $assignedUsers = $assignedUsers->filter(function($user) {
                return $user->user_hidden !== 'Yes';
            });
            $assignedUserIds = $assignedUsers->pluck('id_users')->toArray();
        }
        
        return response()->json([
            'allUsers' => $allUsers->map(function($user) use ($assignedUserIds) {
                return [
                    'id' => $user->id_users,
                    'name' => $user->user_fullname ?: $user->user_name,
                    'username' => $user->user_name,
                    'email' => $user->user_email,
                    'assigned' => in_array($user->id_users, $assignedUserIds),
                ];
            }),
            'assignedUsers' => $assignedUsers->map(function($user) {
                return [
                    'id' => $user->id_users,
                    'name' => $user->user_fullname ?: $user->user_name,
                    'username' => $user->user_name,
                    'email' => $user->user_email,
                ];
            }),
        ]);
    }

    /**
    * Assign permission to role (drag from left to right)
    */
    public function assignPermission(Request $request, $roleId)
    {
        $request->validate([
            'permission_id' => 'required|exists:spatie_permissions,id',
        ]);
        
        try {
            $role = Role::findOrFail($roleId);
            $permission = Permission::findOrFail($request->permission_id);
            
            $role->givePermissionTo($permission); 
            
            $childPermissions = Permission::where('parent_id', $permission->id)->get();
            if($childPermissions){
                foreach($childPermissions as $childPermission){
                    $role->givePermissionTo($childPermission);

                    $grandChildPermissions = Permission::where('parent_id', $childPermission->id)->get();
                    if($grandChildPermissions){
                        foreach($grandChildPermissions as $grandChildPermission){
                            $role->givePermissionTo($grandChildPermission);

                            $subGrandChildPermissions = Permission::where('parent_id', $grandChildPermission->id)->get();
                            if($subGrandChildPermissions){
                                foreach ($subGrandChildPermissions as $subGrandChildPermission) {
                                    $role->givePermissionTo($subGrandChildPermission);
                                }
                            }
                        }
                    }

                }
            }

            $this->forgotSpatieCache();

            return response()->json([
                'success' => true,
                'message' => 'Permission assigned successfully.',
            ]);
            
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign permission: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
    * Remove permission from role (drag from right to left)
    */
    public function removePermission(Request $request, $roleId)
    {
        $request->validate([
            'permission_id' => 'required|exists:spatie_permissions,id',
        ]);
        
        try {
            $role = Role::findOrFail($roleId);
            $permission = Permission::findOrFail($request->permission_id);
            
            $role->revokePermissionTo($permission);

            $childPermissions = Permission::where('parent_id', $permission->id)->get();
            if($childPermissions){
                foreach($childPermissions as $childPermission){
                    $role->revokePermissionTo($childPermission);

                    $grandChildPermissions = Permission::where('parent_id', $childPermission->id)->get();
                    if($grandChildPermissions){
                        foreach($grandChildPermissions as $grandChildPermission){
                            $role->revokePermissionTo($grandChildPermission);

                            $subGrandChildPermissions = Permission::where('parent_id', $grandChildPermission->id)->get();
                            if($subGrandChildPermissions){
                                foreach ($subGrandChildPermissions as $subGrandChildPermission) {
                                    $role->revokePermissionTo($subGrandChildPermission);
                                }
                            }
                        }
                    }

                }
            }

            $this->forgotSpatieCache();
            return response()->json([
                'success' => true,
                'message' => 'Permission removed successfully.',
            ]);
        } catch (Exception $e) {
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove permission: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function assignAllPermissions(Role $role) 
    {
        try {
            
            $permissions = Permission::all();
            foreach ($permissions as $permission) {
                if (!$role->permissions->contains($permission->id)) {
                    $role->givePermissionTo($permission);
                }
            }
            $this->forgotSpatieCache();
            return response()->json([
                'success' => true,
                'message' => 'All Permission assigned.',
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign all permissions: ' . $e->getMessage(),
            ]);
        }

    }

    public function removeAllPermissions(Role $role)
    {
        try {
        
            $role->permissions()->detach();
            $this->forgotSpatieCache();
            return response()->json([
                'success' => true,
                'message' => 'All Permission removed.',
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove all permissions: ' . $e->getMessage(),
            ]);
        }


    }

    public function resetDefaultPermissions(Role $role)
    {
        try {
        
            $template_role = Role::with('permissions')->templateRole($role->name)->first();
            
            if (!$template_role || $template_role->permissions->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Template role has no permissions, cannot reset.'
                ]);
            }

            $role->syncPermissions($template_role->permissions ?? []);

            $this->forgotSpatieCache();
            return response()->json([
                'success' => true,
                'message' => 'Default permissions applied successfully.',
            ]);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to apply default permissions at this time. ' . $e->getMessage(),
            ]);
        }


    }


    /**
     * Assign user to role (drag from left to right)
     */
    public function assignUser(Request $request, $roleId)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id_users',
        ]);

        // Use array cache to avoid database cache table issues
        $this->useArrayCache();
        
        try {
            $role = Role::findOrFail($roleId);
            $user = User::findOrFail($request->user_id);
            
            
            if ($user->hasRole($role)) {
                return response()->json([
                    'success' => true,
                    'message' => 'User already has this role.',
                ]);
            }
            
            $exists = DB::table('spatie_model_has_roles')
                ->where('role_id', $role->id)
                ->where('id_users', $user->id_users)
                ->where('model_type', get_class($user))
                ->exists();
            
            if (!$exists) {

                DB::table('spatie_model_has_roles')->insert([
                    'role_id' => $role->id,
                    'id_users' => $user->id_users,
                    'model_type' => get_class($user),
                ]);
            }
            
            try {
                $user->assignRole($role);
            } catch (Exception $e) {
                Log::debug('Spatie assignRole had issue (non-critical): ' . $e->getMessage());
            }
            
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            
            $dbRecord = DB::table('spatie_model_has_roles')
                ->where('role_id', $role->id)
                ->where('id_users', $user->id_users)
                ->where('model_type', get_class($user))
                ->first();
            
            if (!$dbRecord) {
                throw new Exception('Role assignment failed - record not found in database.');
            }
            
            return response()->json([
                'success' => true,
                'message' => 'User assigned to role successfully.',
            ]);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign user: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove user from role (drag from right to left)
     */
    public function removeUser(Request $request, $roleId)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id_users',
        ]);

        // Use array cache to avoid database cache table issues
        $this->useArrayCache();
        
        try {
            $role = Role::findOrFail($roleId);
            $user = User::findOrFail($request->user_id);
            
            // Remove the role - use direct database delete to ensure correct column name
            $deleted = DB::table('spatie_model_has_roles')
                ->where('role_id', $role->id)
                ->where('id_users', $user->id_users)
                ->where('model_type', get_class($user))
                ->delete();
            
            // Also call Spatie's method to keep cache in sync (but don't rely on it)
            try {
                $user->removeRole($role);
            } catch (\Exception $e) {
                // Ignore - we've already deleted directly
                \Log::debug('Spatie removeRole had issue (non-critical): ' . $e->getMessage());
            }
            
            // Clear permission cache
            app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();
            
            // Verify the removal was successful by checking database directly
            $dbRecord = DB::table('spatie_model_has_roles')
                ->where('role_id', $role->id)
                ->where('id_users', $user->id_users)
                ->where('model_type', get_class($user))
                ->first();
            
            if ($dbRecord) {
                \Log::error('Role removal verification failed', [
                    'user_id' => $user->id_users,
                    'role_id' => $role->id,
                    'role_name' => $role->name,
                    'db_record' => (array)$dbRecord,
                ]);
                throw new \Exception('Role removal failed - record still exists in database.');
            }
            
            return response()->json([
                'success' => true,
                'message' => 'User removed from role successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error removing user from role: ' . $e->getMessage());
            \Log::error('User ID: ' . $request->user_id);
            \Log::error('Role ID: ' . $roleId);
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove user: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function forgotSpatieCache()
    {
        $cache_key = config('permission.cache.key');

        DB::table('cache')
        ->where('key', $cache_key)
        ->delete();

        app()->make(\Spatie\Permission\PermissionRegistrar::class)->forgetCachedPermissions();
        app()->make(\Spatie\Permission\PermissionRegistrar::class)->initializeCache();
    }

}
