<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;

class RolePermissionController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {

    }

    /**
     * Display the role permissions management page.
     */
    public function index(Request $request)
    {
        $roles = Role::with('permissions')->orderBy('name')->get();
        $permissions = Permission::orderBy('name')->get();
        
        // Group permissions by module/prefix if they follow a pattern
        $groupedPermissions = $this->groupPermissions($permissions);
        
        return view('settings.role-permissions.permissions', compact('roles', 'permissions', 'groupedPermissions'));
    }

    /**
     * Store a new permission.
     */
    public function storePermission(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:spatie_permissions,name',
            'guard_name' => 'nullable|string|max:255',
        ]);

        Permission::create([
            'name' => $request->name,
            'parent_id' => $request->parent_id ?? null,
            'guard_name' => $request->guard_name ?? 'web',
        ]);

        return response()->json(['success' => true, 'message' => 'Permission created successfully.']);
    }

    /**
     * Update a permission.
     */
    public function updatePermission(Request $request, $id)
    {
        $permission = Permission::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255|unique:spatie_permissions,name,' . $id,
            'guard_name' => 'nullable|string|max:255',
        ]);

        $permission->update([
            'name' => $request->name,
            'guard_name' => $request->guard_name ?? 'web',
        ]);

        return redirect()->route('settings.role-permissions.index')
            ->with('success', 'Permission updated successfully.');
    }

    /**
     * Delete a permission.
     */
    public function destroyPermission($id)
    {
        $permission = Permission::findOrFail($id);
        $permission->delete();

        return redirect()->route('settings.role-permissions.index')
            ->with('success', 'Permission deleted successfully.');
    }

    /**
     * Assign permissions to a role.
     */
    public function assignPermissions(Request $request, $roleId)
    {
        $role = Role::findOrFail($roleId);

        $request->validate([
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:spatie_permissions,id',
        ]);

        $permissionIds = $request->permissions ?? [];
        $permissions = Permission::whereIn('id', $permissionIds)->get();

        $role->syncPermissions($permissions);

        return redirect()->route('settings.role-permissions.index')
            ->with('success', 'Permissions assigned to role successfully.');
    }

    /**
     * Get role with permissions for editing.
     */
    public function getRole($id)
    {
        $role = Role::with('permissions')->findOrFail($id);
        return response()->json($role);
    }

    /**
     * Get permission for editing.
     */
    public function getPermission($id)
    {
        $permission = Permission::findOrFail($id);
        return response()->json($permission);
    }

    /**
     * Group permissions by their prefix (e.g., 'users.create' -> 'users' group)
     */
    private function groupPermissions($permissions)
    {
        $grouped = [];
        
        foreach ($permissions as $permission) {
            $parts = explode('.', $permission->name);
            $group = count($parts) > 1 ? $parts[0] : 'Other';
            
            if (!isset($grouped[$group])) {
                $grouped[$group] = [];
            }
            
            $grouped[$group][] = $permission;
        }
        
        ksort($grouped);
        return $grouped;
    }

    
    public function updateSort(Request $request)
    {
        $request->validate([
            'parent_id' => 'nullable|integer',
            'order'     => 'required|array',
        ]);

        foreach ($request->order as $index => $permissionId) {
            Permission::where('id', $permissionId)->update([
                // 'parent_id' => $request->parent_id,
                // 'sort_order' => $index + 1,
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Permission order updated'
        ]);
    }
}
