<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Business;
use App\Models\Role;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display the user management page.
     */
    public function index(Request $request)
    {
        $query = User::with(['business', 'roles'])
            ->orderBy('user_name', 'ASC');
        
        // Filter by status
        if ($request->status && $request->status != 'all') {
            $query->where('user_status', $request->status);
        }
        
        // Filter by username search
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('user_name', 'like', '%' . $request->search . '%')
                  ->orWhere('user_fullname', 'like', '%' . $request->search . '%')
                  ->orWhere('user_email', 'like', '%' . $request->search . '%');
            });
        }
        
        $users = $query->get();
        
        // Filter users based on current user's role
        $currentUser = auth()->user();
        if (!$currentUser->hasRole('Super User')) {
            // Non-Super Users: Hide users with Super User role AND users with user_hidden = "Yes"
            $users = $users->filter(function($user) {
                // Hide users with Super User role
                if ($user->hasRole('Super User')) {
                    return false;
                }
                // Hide users with user_hidden = "Yes"
                if ($user->user_hidden === 'Yes') {
                    return false;
                }
                return true;
            });
        }
        // Super Users: Show ALL users (no filtering needed)
        
        $selectedStatus = $request->status ?? 'all';
        $searchTerm = $request->search ?? '';
        
        // Get all businesses for the form (roles no longer needed)
        $businesses = Business::select('id_business', 'business_name')->get();
        
        return view('settings.users.list', compact('users', 'selectedStatus', 'searchTerm', 'businesses'));
    }

    /**
     * Get user data for editing.
     */
    public function show($id)
    {
        $user = User::with(['business', 'roles'])->findOrFail($id);
        
        // Check if current user can view this user
        $currentUser = auth()->user();
        
        // If user is hidden and current user is not Super User, deny access
        if ($user->user_hidden === 'Yes' && !$currentUser->hasRole('Super User')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to view this user.'], 403);
        }
        
        // If user has Super User role and current user is not Super User, deny access
        if ($user->hasRole('Super User') && !$currentUser->hasRole('Super User')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to view this user.'], 403);
        }
        
        // Get Spatie roles for the user
        $spatieRoles = $user->getRoleNames()->map(function($roleName) {
            return ['name' => $roleName];
        });
        
        $userData = $user->toArray();
        $userData['spatie_roles'] = $spatieRoles->toArray();
        
        return response()->json(['success' => true, 'data' => $userData]);
    }

    /**
     * Store a new user.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_name' => 'required|string|unique:users,user_name',
            'user_password' => 'required|string|min:6',
            'user_firstname' => 'nullable|string',
            'user_lastname' => 'nullable|string',
            'user_fullname' => 'nullable|string',
            'user_email' => 'nullable|email',
            'user_mobile' => 'nullable|string',
            'user_phone' => 'nullable|string',
            'user_address' => 'nullable|string',
            'business_id' => 'nullable|integer|exists:business,id_business',
            'user_status' => 'required|string|in:Active,Inactive',
            'user_hidden' => 'nullable|string|in:Yes,No',
        ]);
        
        // Check permission to set user_hidden
        $currentUser = auth()->user();
        if ($request->has('user_hidden') && !$currentUser->can('hide_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to hide/unhide users.'], 403);
        }

        DB::beginTransaction();
        try {
            $userData = [
                'business_id' => $request->business_id,
                'user_name' => $request->user_name,
                'user_password' => md5($request->user_password), // Using MD5 as per existing codebase
                'user_firstname' => $request->user_firstname,
                'user_lastname' => $request->user_lastname,
                'user_fullname' => $request->user_fullname,
                'user_email' => $request->user_email,
                'user_mobile' => $request->user_mobile,
                'user_phone' => $request->user_phone,
                'user_address' => $request->user_address,
                'user_image' => $request->user_image ?? 'avatar-15.jpg',
                'user_status' => $request->user_status,
            ];
            
            // Only set user_hidden if user has permission
            if ($currentUser->can('hide_users')) {
                $userData['user_hidden'] = $request->user_hidden ?? 'No';
            } else {
                $userData['user_hidden'] = 'No'; // Default to 'No' if no permission
            }
            
            $user = User::create($userData);

            // No role assignment - roles must be assigned via Role Management page

            DB::commit();
            return response()->json(['success' => true, 'message' => 'User created successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error creating user: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Update an existing user.
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        
        // Check if current user can modify this user
        $currentUser = auth()->user();
        
        // If user is hidden and current user is not Super User, deny access
        if ($user->user_hidden === 'Yes' && !$currentUser->hasRole('Super User')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
        }
        
        // If user has Super User role and current user is not Super User, deny access
        if ($user->hasRole('Super User') && !$currentUser->hasRole('Super User')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
        }
        
        $request->validate([
            'user_name' => [
                'required',
                'string',
                Rule::unique('users', 'user_name')->ignore($id, 'id_users')
            ],
            'user_password' => 'nullable|string|min:6',
            'user_firstname' => 'nullable|string',
            'user_lastname' => 'nullable|string',
            'user_fullname' => 'nullable|string',
            'user_email' => 'nullable|email',
            'user_mobile' => 'nullable|string',
            'user_phone' => 'nullable|string',
            'user_address' => 'nullable|string',
            'business_id' => 'nullable|integer|exists:business,id_business',
            'user_status' => 'required|string|in:Active,Inactive',
            'user_hidden' => 'nullable|string|in:Yes,No',
        ]);
        
        // Check permission to set user_hidden
        if ($request->has('user_hidden') && !$currentUser->can('hide_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to hide/unhide users.'], 403);
        }

        DB::beginTransaction();
        try {
            $updateData = [
                'business_id' => $request->business_id,
                'user_name' => $request->user_name,
                'user_firstname' => $request->user_firstname,
                'user_lastname' => $request->user_lastname,
                'user_fullname' => $request->user_fullname,
                'user_email' => $request->user_email,
                'user_mobile' => $request->user_mobile,
                'user_phone' => $request->user_phone,
                'user_address' => $request->user_address,
                'user_status' => $request->user_status,
            ];
            
            // Only update user_hidden if user has permission
            if ($currentUser->can('hide_users')) {
                $updateData['user_hidden'] = $request->user_hidden ?? $user->user_hidden ?? 'No';
            }

            // Only update password if provided
            if ($request->user_password) {
                $updateData['user_password'] = md5($request->user_password);
            }

            // Only update image if provided
            if ($request->user_image) {
                $updateData['user_image'] = $request->user_image;
            }

            $user->update($updateData);

            // No role assignment - roles must be assigned via Role Management page

            DB::commit();
            return response()->json(['success' => true, 'message' => 'User updated successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error updating user: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Delete a user.
     */
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);
            
            // Check if current user can modify this user
            $currentUser = auth()->user();
            
            // If user is hidden and current user is not Super User, deny access
            if ($user->user_hidden === 'Yes' && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to delete this user.'], 403);
            }
            
            // If user has Super User role and current user is not Super User, deny access
            if ($user->hasRole('Super User') && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to delete this user.'], 403);
            }
            
            $user->delete();
            return response()->json(['success' => true, 'message' => 'User deleted successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error deleting user: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Search users for dropdown (similar to login logs).
     */
    public function userSearch(Request $request)
    {
        $search = $request->input('term', '');
        
        $query = User::select('user_name', 'user_fullname', 'id_users')
            ->distinct()
            ->orderBy('user_name', 'asc');
            
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('user_name', 'like', '%' . $search . '%')
                  ->orWhere('user_fullname', 'like', '%' . $search . '%');
            });
        }
        
        $users = $query->limit(100)
            ->get()
            ->map(function($user) {
                return [
                    'user_name' => $user->user_name,
                    'user_fullname' => $user->user_fullname,
                    'id_users' => $user->id_users
                ];
            });

        return response()->json($users);
    }

    /**
     * Update user status.
     */
    public function updateStatus(Request $request)
    {
        $request->validate([
            'id_users' => 'required|integer',
            'user_status' => 'required|string|in:Active,Inactive',
        ]);

        try {
            $user = User::findOrFail($request->id_users);
            
            // Check if current user can modify this user
            $currentUser = auth()->user();
            
            // If user is hidden and current user is not Super User, deny access
            if ($user->user_hidden === 'Yes' && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
            }
            
            // If user has Super User role and current user is not Super User, deny access
            if ($user->hasRole('Super User') && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
            }
            
            $user->update(['user_status' => $request->user_status]);
            return response()->json(['success' => true, 'message' => 'User status updated successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error updating user status: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Upload user image.
     */
    public function uploadImage(Request $request)
    {
        $request->validate([
            'file' => 'required|image|mimes:jpeg,jpg,png,gif|max:2048', // 2MB max
        ]);

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
            
            // Ensure directory exists
            $uploadPath = public_path('images/users');
            if (!File::exists($uploadPath)) {
                File::makeDirectory($uploadPath, 0755, true);
            }
            
            $file->move($uploadPath, $filename);

            return response()->json([
                'success' => true,
                'filename' => $filename
            ]);
        }

        return response()->json(['success' => false, 'message' => 'No file uploaded.'], 400);
    }

    /**
     * Update user image.
     */
    public function updateImage(Request $request)
    {
        $request->validate([
            'id_users' => 'required|integer',
            'existing_image' => 'nullable|string',
            'user_image' => 'nullable|string',
        ]);

        $image_user = "";
        if ($request->existing_image != null && $request->existing_image != '') {
            $image_user = $request->existing_image;
        } else if ($request->user_image != null && $request->user_image != '') {
            $image_user = $request->user_image;
        } else {
            return response()->json(['success' => false, 'message' => 'No image selected.'], 400);
        }

        // Validate that the image file exists
        $imagePath = public_path('images/users/' . $image_user);
        if (!File::exists($imagePath)) {
            return response()->json(['success' => false, 'message' => 'Selected image file does not exist.'], 400);
        }

        try {
            $user = User::findOrFail($request->id_users);
            
            // Check if current user can modify this user
            $currentUser = auth()->user();
            
            // If user is hidden and current user is not Super User, deny access
            if ($user->user_hidden === 'Yes' && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
            }
            
            // If user has Super User role and current user is not Super User, deny access
            if ($user->hasRole('Super User') && !$currentUser->hasRole('Super User')) {
                return response()->json(['success' => false, 'message' => 'You do not have permission to modify this user.'], 403);
            }
            
            $user->update(['user_image' => $image_user]);
            return response()->json(['success' => true, 'message' => 'User image updated successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error updating user image: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get existing user images.
     */
    public function existingImages()
    {
        $path = public_path('images/users');
        $urls = [];

        if (File::exists($path)) {
            foreach (File::files($path) as $file) {
                $urls[] = asset('images/users/' . $file->getFilename());
            }
        }

        return response()->json($urls);
    }
}
