<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Models\Business;
use App\Models\Staff;
use App\Models\Allowance;
use App\Models\StaffAllowance;
use App\Models\LeavePolicy;
use App\Models\DeductionPolicy;
use App\Models\Floor;
use App\Models\FaceRecognitionDevices;
use App\Models\FaceRecognitionDeviceStaff;


// use DB;
use Illuminate\Container\Attributes\DB as AttributesDB;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;

class StaffController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;

  public function staff_list(Request $request)
    {
        $currentBusiness = Business::find(session('business_id'));
        
        if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
            $hoBusiness = Business::where('ho', 'Yes')->first();
            $policiesBusinessId = $hoBusiness ? $hoBusiness->id_business : session('business_id');
        } else {
            $policiesBusinessId = session('business_id');
        }
        
        $staffMembers = Staff::select('*')
            ->where('staff.business_id', session('business_id'))
            ->orderBy('staff.staff_order', 'ASC');
        
        if($request->status && $request->status != 'all') {
            $staffMembers = $staffMembers->where('staff.staff_active', request('status'));
        }

        $staffMembers = $staffMembers->get();
        $selectedStatus = $request->status ?? 'all';

        // Policies from determined business (HO or current)
        $deductionPolicies = DeductionPolicy::where('business_id', $policiesBusinessId)
            ->orWhere('business_id', null)
            ->get();

        // Business-specific data always from current business
        $business_floors = Floor::where('business_id', session('business_id'))->get();

        $faceRecognitionDevices = FaceRecognitionDevices::where('business_id', session('business_id'))->get();
        
        // Allowances from determined business (HO or current)
        $allAllowances = Allowance::where('business_id', $policiesBusinessId)
            ->orderBy('allowance_name', 'ASC')
            ->get();

        // Leave policies from determined business (HO or current)
        $leavePolicies = LeavePolicy::where('business_id', $policiesBusinessId)
            ->orWhereNull('business_id')
            ->with('definitions')
            ->get();

        return view('staff.staff_list', compact(
            'staffMembers', 
            'leavePolicies', 
            'selectedStatus', 
            'deductionPolicies', 
            'business_floors', 
            'faceRecognitionDevices',
            'allAllowances'
        ));
    }


   public function get_staff($id)
    {
        $currentBusiness = Business::find(session('business_id'));
        if ($currentBusiness && $currentBusiness->ho_accounts === 'Yes') {
            $hoBusiness = Business::where('ho', 'Yes')->first();
            $allowancesBusinessId = $hoBusiness ? $hoBusiness->id_business : session('business_id');
        } else {
            $allowancesBusinessId = session('business_id');
        }
        
        $staffMember = Staff::with(['staffAllowances' => function($query) {
                $query->where('staff_allowance_status', 'Active')
                    ->with('allowance');
            },
            'leavePolicy.definitions'
            ])->findOrFail($id);
        $allAllowances = Allowance::where('business_id', $allowancesBusinessId)
            ->orderBy('allowance_name', 'ASC')
            ->get();
        
        $formattedAllowances = [];
        foreach ($staffMember->staffAllowances as $staffAllowance) {
            $formattedAllowances[] = [
                'allowance_id' => $staffAllowance->allowance_id,
                'allowance_name' => $staffAllowance->allowance->allowance_name ?? '',
                'allowance_amount' => $staffAllowance->allowance_amount
            ];
        }
        
        return response()->json([
            'success' => true, 
            'data' => $staffMember,
            'staff_allowances' => $formattedAllowances,
            'all_allowances' => $allAllowances,
            'allowances_business_id' => $allowancesBusinessId 
        ]);
    }
   
   public function device_assignment ($id_staff)
   {
       $id_staff = intval($id_staff);
       if($id_staff) {
           $devicestaffassignment = FaceRecognitionDeviceStaff::where('staff_id', $id_staff)->first();
           return response()->json(['success' => true, 'data' => $devicestaffassignment]);
       } else {
           return response()->json(['success' => false, 'message' => 'Staff ID is required to access device assignment.']);
       }
       
   }

   public function save_device_assignment(Request $request)
   {
        $id_face_recognition_device_staff = $request->id_face_recognition_device_staff;
        $id_staff = $request->id_staff;
        $device_id = $request->device_id;
        $device_staff_id = $request->device_staff_id;


        if(!$id_staff) {
            return response()->json(['success' => false, 'message' => 'Staff ID is required.'], 400);
        }

       try {
            if($id_face_recognition_device_staff) {
                $id_face_recognition_device_staff = intval($id_face_recognition_device_staff);
                $assignment = FaceRecognitionDeviceStaff::findOrFail($id_face_recognition_device_staff);
                $assignment->update([
                    'device_id' => $device_id,
                    'device_staff_id' => $device_staff_id,
                    'staff_id' => $id_staff,
                    's_name' => $request->s_name ? $request->s_name : null
                ]);
                return response()->json(['success' => true, 'message' => 'Device assignment updated successfully.']);
            } else {
                FaceRecognitionDeviceStaff::create([
                    'device_id' => $device_id,
                    'staff_id' => $id_staff,
                    'device_staff_id' => $device_staff_id,
                    'created_on' => now(),
                    'created_by' => session('user_name'),
                    's_name' => $request->s_name ? $request->s_name : null
                ]);
                return response()->json(['success' => true, 'message' => 'Device assignment created successfully.']);
            }
           
       } catch (\Exception $e) {
           return response()->json(['error' => true, 'message' => 'Error saving device assignment: '.$e->getMessage()], 500);
       }
   }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $validatedData = $request->validate([
                'staff_fullname' => 'required|string|max:255',
                'staff_firstname' => 'required|string|max:255',
                'staff_lastname' => 'required|string|max:255',
                'staff_cell' => 'required|string|max:255',
                'staff_phone1' => 'nullable|string|max:255',
                'staff_phone2' => 'nullable|string|max:255',
                'staff_address' => 'nullable|string|max:255',
                'staff_eid' => 'nullable|string|max:255',
                'staff_image' => 'nullable|string|max:255',
                'staff_email' => 'nullable|string|email|max:255',
                'staff_active' => 'required|string|max:1',
                'staff_order' => 'required|integer',
                'staff_salary' => 'required|numeric',
                'staff_comment' => 'nullable|string|max:255',
                'staff_shared' => 'nullable|string|max:255',
                'fcm_id' => 'nullable|string|max:255',
                'staff_work_min' => 'nullable|integer',
                'week_day_off' => 'nullable|string|max:255',
                'commission_mode' => 'nullable|string|max:255',
                'staff_commission_perc' => 'nullable|numeric',
                'commission_target' => 'nullable|numeric',
                'commission_fixed_amount' => 'nullable|numeric',
                'deduction_policy' => 'nullable|integer',
                'provident_fund_rate' => 'nullable|decimal:2',
                'salary_monthly_tax' => 'nullable|numeric',
                'staff_designation' => 'nullable|string|max:255',
                'staff_department' => 'nullable|string|max:255',
                'staff_cnic' => 'nullable|string|max:255',
                'staff_date_of_joining' => 'nullable|date',
                'staff_bank_account' => 'nullable|string|max:255',
                'staff_bank_account_title' => 'nullable|string|max:255',
                'staff_bank' => 'nullable|string|max:255',
                'staff_annual_leaves' => 'nullable|integer',
                'day_start_time' => 'nullable|string|max:255',
                'day_end_time' => 'nullable|string|max:255',
                'floor_id' => 'nullable|integer',
                'floor' => 'nullable|string|max:255',
                'salary_type' => 'nullable|string|max:255',
                'next_of_kin_name' => 'nullable|string|max:255',
                'next_of_kin_number' => 'nullable|string|max:255',
                'date_of_birth' => 'nullable|date',
                'contract_validity' => 'nullable|date',
                'contract_number' => 'nullable|string|max:255',
                'staff_fingerprint_template_base64' => 'nullable|string',
                'staff_fingerprint' => 'nullable|string',
                'auto_time_out' => 'nullable|boolean',
                'leave_policy_id' => 'nullable|integer',
                'staff_shared' => 'nullable|string|max:255',
                'overtime_allowed' => 'nullable|string|in:Yes,No',
                'overtime_hourly_rate' => 'nullable|numeric|min:0',
            ]);
            $validatedData['business_id'] = session('business_id');
            $validatedData['created_by'] = session('user_name');
            $validatedData['created_at'] = now();

            $staff = Staff::create($validatedData);
            if ($request->has('allowances')) {
                $allowances = json_decode($request->allowances, true);
                
                if (is_array($allowances) && count($allowances) > 0) {
                    foreach ($allowances as $allowanceData) {
                        if (isset($allowanceData['allowance_id']) && 
                            isset($allowanceData['allowance_amount']) && 
                            $allowanceData['allowance_amount'] > 0) {
                            
                            StaffAllowance::create([
                                'staff_id' => $staff->id_staff,
                                'allowance_id' => $allowanceData['allowance_id'],
                                'allowance_amount' => $allowanceData['allowance_amount'],
                                'staff_allowance_status' => 'Active',
                                'created_on' => now(),
                                'created_by' => session('user_name')
                            ]);
                        }
                    }
                }
            }

            DB::commit();
            
            return response()->json([
                'success' => true, 
                'message' => 'Staff member created successfully.',
                'data' => $staff
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => true, 
                'message' => 'Error creating staff member: ' . $e->getMessage()
            ], 500);
        }
    }


   

   public function update_scheduler_status(Request $request)
   {
       $id = $request->id_staff;
       $status = $request->staff_scheduler;

       if(!$id) {
           return response()->json(['success' => false, 'message' => 'Staff ID is required.'], 400);
       }

       try {
           $id = intval($id);
           $staffMember = Staff::findOrFail($id);
           $staffMember->update(['staff_scheduler' => $status]);
           return response()->json(['success' => true, 'message' => 'Scheduler status updated successfully.']);
       } catch (\Exception $e) {
           return response()->json(['error' => true, 'message' => 'Invalid Staff ID.'], 400);
       }
   }
  
    public function update(Request $request)
    {
        $id = $request->id_staff;
        
        if(!$id) {
            return response()->json(['success' => false, 'message' => 'Staff ID is required.'], 400);
        }
        
        try {
            DB::beginTransaction();
            
            $id = intval($id);
            $validatedData = $request->validate([
                'staff_fullname' => 'required|string|max:255',
                'staff_firstname' => 'required|string|max:255',
                'staff_lastname' => 'required|string|max:255',
                'staff_cell' => 'required|string|max:255',
                'staff_phone1' => 'nullable|string|max:255',
                'staff_phone2' => 'nullable|string|max:255',
                'staff_address' => 'nullable|string|max:255',
                'staff_eid' => 'nullable|string|max:255',
                'staff_image' => 'nullable|string|max:255',
                'staff_email' => 'nullable|string|email|max:255',
                'staff_active' => 'required|string|max:1',
                'staff_order' => 'required|integer',
                'staff_salary' => 'required|numeric',
                'staff_comment' => 'nullable|string|max:255',
                'staff_shared' => 'nullable|string|max:255',
                'fcm_id' => 'nullable|string|max:255',
                'staff_work_min' => 'nullable|integer',
                'week_day_off' => 'nullable|string|max:255',
                'commission_mode' => 'nullable|string|max:255',
                'staff_commission_perc' => 'nullable|numeric',
                'commission_target' => 'nullable|numeric',
                'commission_fixed_amount' => 'nullable|numeric',
                'deduction_policy' => 'nullable|integer',
                'provident_fund_rate' => 'nullable|decimal:2',
                'salary_monthly_tax' => 'nullable|numeric',
                'staff_designation' => 'nullable|string|max:255',
                'staff_department' => 'nullable|string|max:255',
                'staff_cnic' => 'nullable|string|max:255',
                'staff_date_of_joining' => 'nullable|date',
                'staff_bank_account' => 'nullable|string|max:255',
                'staff_bank_account_title' => 'nullable|string|max:255',
                'staff_bank' => 'nullable|string|max:255',
                'staff_annual_leaves' => 'nullable|integer',
                'day_start_time' => 'nullable|string|max:255',
                'day_end_time' => 'nullable|string|max:255',
                'floor_id' => 'nullable|integer',
                'floor' => 'nullable|string|max:255',
                'salary_type' => 'nullable|string|max:255',
                'next_of_kin_name' => 'nullable|string|max:255',
                'next_of_kin_number' => 'nullable|string|max:255',
                'date_of_birth' => 'nullable|date',
                'contract_validity' => 'nullable|date',
                'contract_number' => 'nullable|string|max:255',
                'staff_fingerprint_template_base64' => 'nullable|string',
                'staff_fingerprint' => 'nullable|string',
                'auto_time_out' => 'nullable|boolean',
                'leave_policy_id' => 'nullable|integer',
                'staff_shared' => 'nullable|string|max:255',
                'overtime_allowed' => 'nullable|string|in:Yes,No',
                'overtime_hourly_rate' => 'nullable|numeric|min:0',
            ]);

            $staffMember = Staff::findOrFail($id);
            $staffMember->update($validatedData);

            if ($request->has('allowances')) {
                $allowances = json_decode($request->allowances, true);
                
                StaffAllowance::where('staff_id', $id)
                    ->update([
                        'staff_allowance_status' => 'Cancelled',
                        'cancelled_by' => session('user_name'),
                        'cancelled_on' => now()
                    ]);
                if (is_array($allowances)) {
                    foreach ($allowances as $allowanceData) {
                        if (isset($allowanceData['allowance_id']) && 
                            isset($allowanceData['allowance_amount']) && 
                            $allowanceData['allowance_amount'] > 0) {
                            $existingAllowance = StaffAllowance::where('staff_id', $id)
                                ->where('allowance_id', $allowanceData['allowance_id'])
                                ->first();
                            
                            if ($existingAllowance) {
                                $existingAllowance->update([
                                    'allowance_amount' => $allowanceData['allowance_amount'],
                                    'staff_allowance_status' => 'Active',
                                    'cancelled_by' => null,
                                    'cancelled_on' => null
                                ]);
                            } else {
                                StaffAllowance::create([
                                    'staff_id' => $id,
                                    'allowance_id' => $allowanceData['allowance_id'],
                                    'allowance_amount' => $allowanceData['allowance_amount'],
                                    'staff_allowance_status' => 'Active',
                                    'created_on' => now(),
                                    'created_by' => session('user_name')
                                ]);
                            }
                        }
                    }
                }
            }else{
                 StaffAllowance::where('staff_id', $id)
                    ->update([
                        'staff_allowance_status' => 'Cancelled',
                        'cancelled_by' => session('user_name'),
                        'cancelled_on' => now()
                    ]);
            }

            DB::commit();
            
            return response()->json([
                'success' => true, 
                'message' => 'Staff member updated successfully.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => true, 
                'message' => 'Error updating staff: ' . $e->getMessage()
            ], 400);
        }
    }

   public function uploadimage(Request $request)
   {
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time().'_'.$file->getClientOriginalName();
            $file->move(public_path('images/staff'), $filename);
            return response()->json([
                'success' => true,
                'filename' => $filename
            ]);
        }

        return response()->json(['success' => false], 400);
   }

   public function update_image(Request $request)
   {
        
        $image_staff = "";
        if($request->existing_image != null || $request->existing_image != '') {
            $image_staff = $request->existing_image;
        } else {
            $image_staff = $request->staff_image;
        }
       $staffMember = Staff::findOrFail($request->id_staff);
       $staffMember->update(['staff_image' => $image_staff]);
         return response()->json(['success' => true, 'message' => 'Staff image updated successfully.']);    

   }

   public function destroy($id)
   {
       $staffMember = Staff::findOrFail($id);
       $staffMember->delete();
       return redirect()->route('staff.staff_list')->with('success', 'Staff member deleted successfully.');
   }

   public function upload(Request $request)
   {
       $request->validate([
           'file' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
       ]);

       $imageName = time().'_'.$request->file->getClientOriginalName();
       $request->file->move(public_path('images/staff'), $imageName);

       return response()->json(['success' => true, 'image' => $imageName]);
   }

   public function remove_image(Request $request)
   {
       $request->validate([
           'image' => 'required|string',
       ]);

       $imagePath = public_path('images/staff/' . $request->image);
       if (file_exists($imagePath)) {
           unlink($imagePath);
           return response()->json(['success' => true]);
       } else {
           return response()->json(['success' => false, 'message' => 'Image not found.'], 404);
       }
   }

   public function existing_images()
   {
        
       $path = public_path('images/staff');
        $urls = [];

        if (File::exists($path)) {
            foreach (File::files($path) as $file) {
                $urls[] = asset('images/staff/' . $file->getFilename());
            }
        }

        return response()->json($urls);
    

   }

   public function update_order(Request $request)
   {
       $order = $request->input('order');
       foreach ($order as $index => $id) {
           Staff::where('id_staff', $id)->update(['staff_order' => $index + 1]);
       }
       return response()->json(['success' => true, 'message' => 'Order updated successfully.']);
   }

   public function staff_search(Request $request){
       $search = request()->query('term');

       if(!$search || strlen($search) < 3) {
           return response()->json(['results' => []]);
       }

       $business_id = request()->query('business_id', session('business_id'));
       
        if($business_id == null || $business_id == '' || $business_id == 0) {
           if(session('ho')=="Yes"){
               $business_id = 0;
           } else {
               $business_id = session('business_id');
           }            
        } else {
           $business_id = session('business_id');
        }

       $staffMembers = Staff::select('staff.id_staff as id',
       'staff_fullname as text',
       'staff_cell',
       'staff_email',
       'staff.business_id',
       'staff_image',
       DB::RAW('CASE WHEN staff.staff_active = "y" THEN "Active" ELSE "Inactive" END as staff_type'),
       'business.business_name')
       ->join('business', 'staff.business_id', '=', 'business.id_business')
       ->where(function($query) use ($search) {
            $query->where('staff_fullname', 'LIKE', "%$search%")
                    ->orWhere('staff_cell', 'LIKE', "%$search%")
                    ->orWhere('staff_email', 'LIKE', "%$search%");
        });
         if($business_id != 0) {
            $staffMembers = $staffMembers->where('business_id', $business_id);
         } else {
            $staffMembers = $staffMembers->where(function($query) {
                $query->where('business.common_products', '=', "Yes");
            });
         }
         $staffMembers = $staffMembers->get();

       return response()->json($staffMembers);
   }
   
   



    // Fingerprint methods
    public function getStaffFingerprint($id)
    {
        try {
            $staffMember = Staff::select('id_staff', 'staff_fullname', 'staff_fingerprint', 'staff_fingerprint_template_base64')
                ->findOrFail($id);
            
            return response()->json([
                'success' => true, 
                'data' => $staffMember
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false, 
                'message' => 'Error fetching fingerprint: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateFingerprint(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'id_staff' => 'required|integer|exists:staff,id_staff',
                'staff_fingerprint' => 'nullable|string',
                'staff_fingerprint_template_base64' => 'nullable|string'
            ]);
            
            $staffMember = Staff::findOrFail($validatedData['id_staff']);
            $staffMember->update([
                'staff_fingerprint' => $validatedData['staff_fingerprint'],
                'staff_fingerprint_template_base64' => $validatedData['staff_fingerprint_template_base64']
            ]);
            
            return response()->json([
                'success' => true, 
                'message' => 'Staff fingerprint updated successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false, 
                'message' => 'Error updating fingerprint: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getAllFingerprints()
    {
        try {
            $fingerprints = Staff::select('id_staff', 'staff_firstname', 'staff_cell', 'fcm_id', 'staff_fullname', 'staff_fingerprint_template_base64')
                ->where('business_id', session('business_id'))
                ->whereNotNull('staff_fingerprint_template_base64')
                ->where('staff_fingerprint_template_base64', '!=', '')
                ->get();
            
            return response()->json([
                'success' => true, 
                'data' => $fingerprints
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false, 
                'message' => 'Error fetching fingerprints: ' . $e->getMessage()
            ], 500);
        }
    }



}


