<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\BusinessTaxes;
use App\Models\Business;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class TaxSettingsController extends Controller
{
    private function isHoAccountsUser()
    {
        return session('ho_accounts') === 'Yes';
    }

    private function getHeadOfficeBusinessId()
    {
        $hoBusiness = Business::where('ho', 'Yes')->first();
        return $hoBusiness ? $hoBusiness->id_business : session('business_id');
    }

    private function getAllBusinesses()
    {
        return Business::orderBy('business_name', 'ASC')
            ->when(session('ho_accounts') === 'Yes', function($query) {
                return $query->where('ho_accounts', 'Yes');
            })
            ->get();
    }

    private function getBusinessIdForQuery(Request $request = null)
    {
        // If ho_accounts is 'Yes', use head office business_id
        if ($this->isHoAccountsUser()) {
            return $this->getHeadOfficeBusinessId();
        }
        
        // For non-HO users, use their own business ID from session
        return session('business_id');
    }

    /**
     * Display the tax settings page
     */
    public function index(Request $request)
    {
        // If ho_accounts is 'Yes', use head office business_id; otherwise use session business_id
        if ($this->isHoAccountsUser()) {
            $businessId = $this->getHeadOfficeBusinessId();
        } else {
            $businessId = session('business_id');
        }
        
        $business = Business::where('id_business', $businessId)->first();
        $businesses = $this->isHoAccountsUser() ? $this->getAllBusinesses() : collect([]);
        
        return view('tax_settings.tax_configuration', compact('business', 'businesses', 'businessId'));
    }

    /**
     * Get tax settings data for DataTables
     */
    public function getData(Request $request)
    {
        try {
            $businessId = $this->getBusinessIdForQuery($request);
            $isHoAccountsUser = $this->isHoAccountsUser();

            $query = BusinessTaxes::join('business', 'business_taxes.business_id', '=', 'business.id_business')
                ->select([
                    'business_taxes.*',
                    'business.business_name'
                ]);

            // Filter by business_id
            // If ho_accounts is 'Yes', use head office business_id; otherwise use session business_id
            $query->where('business_taxes.business_id', $businessId);

            // Apply filters
            if ($request->filled('id_business_taxes')) {
                $query->where('business_taxes.id_business_taxes', $request->id_business_taxes);
            }
            
            if ($request->filled('tax_name')) {
                $query->where('business_taxes.tax_name', 'like', '%' . $request->tax_name . '%');
            }
            
            if ($request->filled('business_name')) {
                $query->where('business.business_name', 'like', '%' . $request->business_name . '%');
            }
            
            if ($request->filled('tax_active')) {
                $query->where('business_taxes.tax_active', $request->tax_active);
            }
            
            if ($request->filled('tax_invoice_type')) {
                $query->where('business_taxes.tax_invoice_type', $request->tax_invoice_type);
            }
            
            if ($request->filled('tax_payment_mode')) {
                $query->where('business_taxes.tax_payment_mode', $request->tax_payment_mode);
            }

            return DataTables::of($query)
                ->addIndexColumn()
                ->addColumn('status', function($row) {
                    $badge = $row->tax_active === 'Y' 
                        ? '<span class="badge badge-outline-success">Active</span>'
                        : '<span class="badge badge-outline-danger">Inactive</span>';
                    return $badge;
                })
                ->addColumn('apply_on', function($row) {
                    return ucfirst($row->tax_invoice_type);
                })
                ->addColumn('payment_mode', function($row) {
                    return ucfirst($row->tax_payment_mode);
                })
                ->addColumn('percentage', function($row) {
                    return number_format($row->tax_percentage, 2) . '%';
                })
                ->addColumn('action', function($row) {
                    $editBtn = '';
                    $deleteBtn = '';
                    
                    if(auth()->user()->can('edit_tax_settings')) {
                        $editBtn = '<button class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-info open-edit-btn" 
                            onclick="openEditModal('.$row->id_business_taxes.')"
                            data-bs-toggle="tooltip" title="Edit Tax Setting">
                            <iconify-icon icon="line-md:edit" class="fs-20"></iconify-icon></button>';
                    }
                    
                    if(auth()->user()->can('delete_tax_settings')) {
                        $deleteBtn = '<button class="btn flex-shrink-0 rounded-circle btn-icon btn-ghost-danger" 
                            onclick="deleteTax('.$row->id_business_taxes.')"
                            data-bs-toggle="tooltip" title="Delete Tax Setting">
                            <iconify-icon icon="solar:trash-bin-trash-bold-duotone" class="fs-20"></iconify-icon></button>';
                    }
                    
                    return '<div class="d-flex gap-1">' . $editBtn . $deleteBtn . '</div>';
                })
                ->rawColumns(['status', 'action'])
                ->make(true);
                
        } catch (\Exception $e) {
            \Log::error('Error getting tax settings data: ' . $e->getMessage());
            return response()->json([
                'error' => 'Error loading tax settings data'
            ], 500);
        }
    }

    /**
     * Get single tax setting for editing
     */
    public function getTaxSetting($id)
    {
        try {
            $tax = BusinessTaxes::findOrFail($id);
            return response()->json([
                'success' => true,
                'tax' => $tax
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Tax setting not found'
            ], 404);
        }
    }

    /**
     * Save or update tax setting
     */
    public function saveOrUpdate(Request $request)
    {
        try {
            $request->validate([
                'id_business_taxes' => 'nullable|integer',
                'tax_name' => 'required|string|max:255',
                'tax_percentage' => 'required|numeric|min:0|max:100',
                'tax_active' => 'required|in:Y,N',
                'tax_invoice_type' => 'required|in:service,retail',
                'tax_payment_mode' => 'required|in:All,cash,card,check,online,voucher,mixed',
                'business_id' => 'required|integer|exists:business,id_business',
            ]);

            $id = $request->input('id_business_taxes');
            $businessId = $request->input('business_id');

            // If ho_accounts is 'Yes', use head office business_id; otherwise use session business_id
            if ($this->isHoAccountsUser()) {
                $businessId = $this->getHeadOfficeBusinessId();
            } else {
                $businessId = session('business_id');
            }

            $data = [
                'business_id' => $businessId,
                'tax_name' => $request->input('tax_name'),
                'tax_percentage' => $request->input('tax_percentage'),
                'tax_active' => $request->input('tax_active'),
                'tax_invoice_type' => $request->input('tax_invoice_type'),
                'tax_payment_mode' => $request->input('tax_payment_mode'),
            ];

            if (empty($id)) {
                // Create new tax setting
                $tax = BusinessTaxes::create($data);
                $message = 'Tax setting created successfully';
            } else {
                // Update existing tax setting
                $tax = BusinessTaxes::findOrFail($id);
                $tax->update($data);
                $message = 'Tax setting updated successfully';
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'tax' => $tax
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Error saving tax setting: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving tax setting: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete tax setting
     */
    public function delete($id)
    {
        try {
            $tax = BusinessTaxes::findOrFail($id);
            $tax->delete();

            return response()->json([
                'success' => true,
                'message' => 'Tax setting deleted successfully'
            ]);
        } catch (\Exception $e) {
            \Log::error('Error deleting tax setting: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error deleting tax setting'
            ], 500);
        }
    }
}
