<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Ticket;
use App\Models\Business;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class TicketController extends Controller
{
    public function index()
    {
        $business = Business::select('business_name', 'id_business')
            ->where('id_business', '=', session('business_id'))
            ->first();

        return view('tickets.support_tickets')->with(compact('business'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:50',
            'description' => 'required|string|max:1000',
            'severity' => 'required|in:Critical,High,Medium,Low',
        ]);

        try {
            $businessId = session('business_id');
            $userId = session('user_id');
            
            // Client identifier: business_id will be used to match ref_sys_business_id in external system
            // The external system's ref_sys_businesses table will have the mapping
            $clientIdentifier = $businessId;

            // Create ticket
            $ticket = Ticket::create([
                'business_id' => $businessId,
                'user_id' => $userId,
                'client_identifier' => $clientIdentifier,
                'title' => $request->title,
                'description' => $request->description,
                'severity' => $request->severity,
                'status' => 'Open',
            ]);

            // Forward to external ticketing system (stub for now)
            $this->forwardToExternalSystem($ticket);

            return response()->json([
                'success' => true,
                'message' => 'Ticket submitted successfully',
                'ticket_id' => $ticket->id_tickets,
            ]);

        } catch (\Exception $e) {
            Log::error('Ticket submission failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to submit ticket. Please try again.',
            ], 500);
        }
    }

    public function ticketsData(Request $request)
    {
        $columns = [
            'id_tickets',
            'title',
            'severity',
            'status',
            'created_at'
        ];

        $search = $request->input('search.value');
        $orderColumnIndex = $request->input('order.0.column', 0);
        $orderColumn = $columns[$orderColumnIndex] ?? 'id_tickets';
        $orderDir = $request->input('order.0.dir', 'desc');
        $start = $request->input('start', 0);
        $length = $request->input('length', 10);

        $query = Ticket::select(
            'id_tickets',
            'title',
            'severity',
            'status',
            'created_at'
        )
        ->where('business_id', session('business_id'))
        ->where('user_id', session('user_id'))
        ->where('created_at', '>=', Carbon::now()->subDays(30));

        $recordsTotal = $query->count();

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('id_tickets', 'LIKE', "%{$search}%")
                  ->orWhere('severity', 'LIKE', "%{$search}%")
                  ->orWhere('status', 'LIKE', "%{$search}%");
            });
        }

        $recordsFiltered = $query->count();

        $data = $query->orderBy($orderColumn, $orderDir)
            ->skip($start)
            ->take($length)
            ->get()
            ->map(function ($ticket) {
                return [
                    'id_tickets' => $ticket->id_tickets,
                    'title' => $ticket->title,
                    'severity' => $ticket->severity,
                    'status' => $ticket->status,
                    'created_at' => $ticket->created_at ? $ticket->created_at->format('Y-m-d H:i:s') : '',
                ];
            });

        return response()->json([
            'draw' => intval($request->input('draw')),
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    /**
     * Forward ticket to external ticketing system
     * 
     * The external ticketing system maintains a ref_sys_businesses table with:
     * - id_ref_sys_businesses: primary key
     * - ref_sys_business_name: business name
     * - ref_sys_url: base URL of this ERP system (e.g., "https://skedwise.xyz/coya")
     * - ref_sys_business_id: maps to our business_id (used for lookup)
     * - ref_sys_logo: business logo URL
     * 
     * Flow:
     * 1. ERP sends ticket data with ref_sys_business_id (our business_id)
     * 2. External system looks up business in ref_sys_businesses using ref_sys_business_id
     * 3. External system can use ref_sys_url to call back to this ERP if needed
     * 4. External system stores ticket and returns external ticket ID
     */
    private function forwardToExternalSystem(Ticket $ticket)
    {
        try {
            // Get external ticketing system API endpoint from config
            // This should be set in .env as TICKETING_SYSTEM_API_URL
            $externalApiUrl = config('services.ticketing.api_url');
            
            // Prepare payload for external ticketing system
            $payload = [
                'ref_sys_business_id' => $ticket->business_id, // Maps to ref_sys_business_id in external system's ref_sys_businesses table
                'user_id' => $ticket->user_id,
                'title' => $ticket->title,
                'description' => $ticket->description,
                'severity' => $ticket->severity,
                'timestamp' => $ticket->created_at->toIso8601String(),
            ];

            // TODO: Uncomment and configure when external ticketing system is ready
            // 1. Add TICKETING_SYSTEM_API_URL to .env file
            // 2. Add 'ticketing' => ['api_url' => env('TICKETING_SYSTEM_API_URL')] to config/services.php
            // 3. Uncomment the code below
            
            // if ($externalApiUrl) {
            //     $response = Http::timeout(10)
            //         ->post($externalApiUrl . '/tickets', $payload);
            //     
            //     if ($response->successful()) {
            //         Log::info('Ticket forwarded successfully', [
            //             'ticket_id' => $ticket->id_tickets,
            //             'external_ticket_id' => $response->json('ticket_id'),
            //         ]);
            //         
            //         // Optionally store external ticket ID in local ticket record
            //         // $ticket->update(['external_ticket_id' => $response->json('ticket_id')]);
            //     } else {
            //         Log::warning('Failed to forward ticket', [
            //             'ticket_id' => $ticket->id_tickets,
            //             'status' => $response->status(),
            //             'response' => $response->body(),
            //         ]);
            //     }
            // } else {
            //     Log::warning('External ticketing system API URL not configured');
            // }

            // For now, log the attempt with the payload structure
            Log::info('Ticket forwarded to external system (stub)', [
                'ticket_id' => $ticket->id_tickets,
                'external_api_url' => $externalApiUrl ?? 'Not configured',
                'payload' => $payload,
            ]);

        } catch (\Exception $e) {
            // Log error but don't fail ticket creation
            Log::error('Failed to forward ticket to external system: ' . $e->getMessage(), [
                'ticket_id' => $ticket->id_tickets,
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }
}
