<?php

namespace App\Http\Controllers;

use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;

use App\Models\Stores;
use App\Models\TransferNotes;
use App\Models\Gtn;
use App\Models\ProductBatch;
use App\Models\Business;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\Facades\DataTables;


class TransferNotesController extends BaseController
{
    use AuthorizesRequests, ValidatesRequests;


    public function TransferNotes(Request $request)
    {
        $businessId = session('business.id_business');
       

        return view('products.transfer_notes.transfer_notes');
    }

    public function getTransferNotesData(Request $request)
    {
        $startDate = $request->input('start_date', '2025-11-12');
        $endDate   = $request->input('end_date', '2025-11-12');
        $additionalColumns = $request->input('additional_columns', []);

        if (!$startDate || !$endDate) {
            $startDate = date('Y-m-d');
            $endDate = date('Y-m-d');
        }

        // Ensure additional_columns is an array
        if (!is_array($additionalColumns)) {
            $additionalColumns = [];
        }

        // Check permissions and which columns to show
        $user = Auth::user();
        $showUnitPrice = $user && $user->can('view_tn_unit_price') && in_array('unit_price', $additionalColumns);
        $showAmount = $user && $user->can('view_tn_amount') && in_array('amount', $additionalColumns);

        // Build base select fields
        $selectFields = [
            'g.id_gtn',
            'tn.id_transfer_notes',
            'tn.requisition_id',
            'tn.transfer_date',
            'b.business_brand_name as brand_name',
            'tn.product_id',
            'bp.product',
            'tn.batch_id',
            'tn.batch',
            'bs.business_store',
            'pb.store_id',
            'tn.tranfer_in_qty',
            'tn.tranfer_out_qty',
            'tn.created_by',
            'tn.created_at'
        ];

        // Add conditional fields based on permissions and selection
        // Include unit_price if either unit_price is selected OR amount is selected (needed for amount calculation)
        if ($showUnitPrice || $showAmount) {
            $selectFields[] = DB::raw('ROUND(IFNULL(pb.batch_amount, 0), 2) as unit_price');
        }
        if ($showAmount) {
            $selectFields[] = DB::raw('ROUND(IFNULL(pb.batch_amount, 0) * IFNULL(tn.tranfer_out_qty, 0), 2) as amount');
        }

        $query = DB::table('transfer_notes as tn')
            ->join('gtn as g', 'g.id_gtn', '=', 'tn.gtn_id')
            ->join('product_batch as pb', 'pb.id_batch', '=', 'tn.batch_id')
            ->join('business_stores as bs', 'pb.store_id', '=', 'bs.id_business_stores')
            ->join('business_products as bp', 'bp.id_business_products', '=', 'tn.product_id')
            ->join('business_brands as b', 'b.id_business_brands', '=', 'bp.brand_id')
            ->join('business as bus', 'bs.business_id', '=', 'bus.id_business')
            ->select($selectFields)
            ->whereBetween(DB::raw('DATE(tn.transfer_date)'), [$startDate, $endDate]);

        // Apply business filtering based on session and request
        $isHoUser = session('ho') === 'Yes';
        $requestBusinessId = $request->input('business_id');
        
        if ($isHoUser) {
            // Condition commented because HO should be able to see all, regardless of business_id
            // if ($requestBusinessId) {
            //     if (is_array($requestBusinessId)) {
            //         $query->whereIn('bs.business_id', $requestBusinessId);
            //     } else {
            //         $query->where('bs.business_id', $requestBusinessId);
            //     }
            // }
        } else {
            // Non-HO users: filter by session business_id unless overridden by request
            if ($requestBusinessId) {
                // Only allow if they have permission to view that business
                if (is_array($requestBusinessId)) {
                    $query->whereIn('bs.business_id', $requestBusinessId);
                } else {
                    $query->where('bs.business_id', $requestBusinessId);
                }
            } else {
                $query->where('bs.business_id', session('business_id'));
            }
        }

        $query->orderBy('g.id_gtn')
            ->orderByDesc('tn.id_transfer_notes');

        $datatable = DataTables::of($query)
            ->addIndexColumn()
            ->editColumn('created_at', function ($row) {
                return date('d-m-Y H:i', strtotime($row->created_at));
            })
            ->filterColumn('id_gtn', function ($query, $keyword) {
                $query->whereRaw("g.id_gtn LIKE ?", ["%{$keyword}%"]);
            })
            ->filterColumn('id_transfer_notes', function ($query, $keyword) {
                $query->whereRaw("tn.id_transfer_notes LIKE ?", ["%{$keyword}%"]);
            })
            ->filterColumn('product', function ($query, $keyword) {
                $query->whereRaw("bp.product LIKE ?", ["%{$keyword}%"]);
            })
            ->filterColumn('brand_name', function ($query, $keyword) {
                $query->whereRaw("b.business_brand_name LIKE ?", ["%{$keyword}%"]);
            })
            ->filterColumn('batch', function ($query, $keyword) {
                $query->whereRaw("tn.batch LIKE ?", ["%{$keyword}%"]);
            })           
            ->filterColumn('transfer_date', function ($query, $keyword) {
                $query->whereRaw("tn.transfer_date LIKE ?", ["%{$keyword}%"]);
            })            
            ->filterColumn('to_store', function ($query, $keyword) {
                $query->whereRaw("bs.business_store LIKE ?", ["%{$keyword}%"]);
            })
            ->filterColumn('from_store', function ($query, $keyword) {
                $query->whereRaw("bs.business_store LIKE ?", ["%{$keyword}%"]);
            });

        // Format unit_price and amount columns if they exist in the response
        // Note: unit_price may be included even if not selected (needed for amount calculation)
        if ($showUnitPrice || $showAmount) {
            $datatable->editColumn('unit_price', function ($row) {
                return isset($row->unit_price) ? number_format((float)$row->unit_price, 2, '.', '') : '0.00';
            });
        }

        if ($showAmount) {
            $datatable->editColumn('amount', function ($row) {
                return isset($row->amount) ? number_format((float)$row->amount, 2, '.', '') : '0.00';
            });
        }

        return $datatable->make(true);
    }

    public function batch_transfers(Request $request)
    {
        $businessId = session('business.id_business');

        $stores = Stores::select('id_business_stores', 'business_store')
        ->join('business', 'business.id_business', '=', 'business_stores.business_id')
            ->where('business.common_products', "Yes")
            ->orderBy('business_store', 'asc')
            ->get();
       

        return view('products.transfer_notes.batch_transfers', compact('stores'));
    }

    

}