<?php

namespace App\Http\Controllers\staffApp;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\ResponseFormatter;
use App\Models\Staff;
use App\Models\Business;

class StaffAppController extends Controller
{
    protected $formatter;

    public function __construct(ResponseFormatter $formatter)
    {
        $this->formatter = $formatter;
    }

    // ==================== Authentication Methods ====================

    /**
     * Login endpoint - matches legacy login_get() behavior
     * 
     * GET /api/login?cellNumber=...&username=...&fcm_id=...
     */
    public function login(Request $request)
    {
        $cellNumber = $request->get('cellNumber');
        $username = $request->get('username');
        $fcm_id = $request->get('fcm_id');

        if ($cellNumber == null) {
            return response()->json("Missing Paramaters");
        }

        $params = [
            'cellNumber' => $cellNumber,
            'username' => $username,
            'fcm_id' => $fcm_id,
        ];

        $result = $this->performLogin($params);

        if ($result) {
            return response()->json($result);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Check mobile number endpoint - matches legacy checkMobileNumber_get() behavior
     * 
     * GET /api/check_mobile_number?cellNumber=...
     */
    public function checkMobileNumber(Request $request)
    {
        $cellNumber = $request->get('cellNumber');

        if (isset($cellNumber)) {
            $result = $this->checkMobile($cellNumber);
        } else {
            $result = null;
        }

        if ($result) {
            return response()->json($result);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Update FCM Token endpoint - matches legacy updateFCMToken_get() behavior
     * 
     * GET /api/update_fcm_token?idStaff=...&fcmid=...
     */
    public function updateFCMToken(Request $request)
    {
        $idStaff = $request->get('idStaff');
        $fcmid = $request->get('fcmid');

        if (isset($idStaff)) {
            $updated = DB::table('staff')
                ->where('id_staff', $idStaff)
                ->update(['fcm_id' => $fcmid]);

            if ($updated) {
                $res = $this->formatter->formatResponse("true", "Registration Successfull");
                return response()->json($res);
            } else {
                $res = $this->formatter->formatResponse("false", "Not Registrated");
                return response()->json($res);
            }
        }

        return response()->json("Invalid Access");
    }

    // ==================== Attendance Methods ====================

    /**
     * Mark attendance endpoint - matches legacy attendance_get() behavior
     * 
     * GET /api/mark_attendance?cellNumber=...&fcm_id=...&key=...&date=...&time=...&businessId=...&address=...&longitude=...&latitude=...
     */
    public function markAttendance(Request $request)
    {
        $params = [
            'cellNumber' => $request->get('cellNumber'),
            'fcm_id' => $request->get('fcm_id'),
            'key' => $request->get('key'),
            'date' => $request->get('date'),
            'time' => $request->get('time'),
            'businessId' => $request->get('businessId'),
            'address' => $request->get('address'),
            'longitude' => $request->get('longitude'),
            'latitude' => $request->get('latitude'),
        ];

        $resp = null;
        if (isset($params['cellNumber']) && isset($params['fcm_id']) && isset($params['key']) && 
            isset($params['date']) && isset($params['time']) && isset($params['businessId']) && 
            isset($params['address']) && isset($params['longitude']) && isset($params['latitude'])) {
            
            $resp = $this->markStaffAttendance(
                $params['cellNumber'],
                $params['fcm_id'],
                $params['key'],
                $params['date'],
                $params['time'],
                $params['businessId'],
                $params['address'],
                $params['longitude'],
                $params['latitude']
            );

            if ($resp) {
                return response()->json($resp);
            } else {
                return response()->json("Invalid Access");
            }
        }

        return response()->json("Invalid Access");
    }

    /**
     * Staff attendance report endpoint - matches legacy staff_attendance_report_get() behavior
     * 
     * GET /api/staff_attendance_report?staff_id=...&month=...&year=...
     */
    public function staffAttendanceReport(Request $request)
    {
        $params = [
            'staff_id' => $request->get('staff_id'),
            'month' => $request->get('month'),
            'year' => $request->get('year'),
        ];

        $resp = null;
        if (isset($params['staff_id']) && isset($params['month']) && isset($params['year'])) {
            $resp = $this->getStaffMonthAttendanceDetail(
                $params['staff_id'],
                $params['year'],
                $params['month']
            );
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Attendance request endpoint - matches legacy attendance_request_get() behavior
     * 
     * GET /api/attendance_request?cellNumber=...&fcm_id=...&key=...&date=...&time=...&businessId=...&address=...&longitude=...&latitude=...
     */
    public function attendanceRequest(Request $request)
    {
        $params = [
            'cellNumber' => $request->get('cellNumber'),
            'fcm_id' => $request->get('fcm_id'),
            'key' => $request->get('key'),
            'date' => $request->get('date'),
            'time' => $request->get('time'),
            'businessId' => $request->get('businessId'),
            'address' => $request->get('address'),
            'longitude' => $request->get('longitude'),
            'latitude' => $request->get('latitude'),
        ];

        $resp = null;
        if (isset($params['cellNumber']) && isset($params['fcm_id']) && isset($params['key']) && 
            isset($params['date']) && isset($params['time']) && isset($params['businessId']) && 
            isset($params['address']) && isset($params['longitude']) && isset($params['latitude'])) {
            
            $resp = $this->staffAttendanceRequest(
                $params['cellNumber'],
                $params['fcm_id'],
                $params['key'],
                $params['date'],
                $params['time'],
                $params['businessId'],
                $params['address'],
                $params['longitude'],
                $params['latitude']
            );
        } else {
            $resp = "Missing Paramaters";
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Attendance request report endpoint - matches legacy attendance_request_report_get() behavior
     * 
     * GET /api/attendance_request_report?staff_id=...&month=...&year=...&businessid=...
     */
    public function attendanceRequestReport(Request $request)
    {
        $params = [
            'staff_id' => $request->get('staff_id'),
            'month' => $request->get('month'),
            'year' => $request->get('year'),
            'business_id' => $request->get('businessid'),
        ];

        $resp = null;
        if (isset($params['staff_id']) && isset($params['month']) && isset($params['year'])) {
            $resp = $this->getAttendanceRequestReport(
                $params['staff_id'],
                $params['year'],
                $params['month'],
                $params['business_id']
            );
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    // ==================== Staff Service Methods ====================

    /**
     * Get visits endpoint - matches legacy getVisits_get() behavior
     * 
     * GET /api/get_visits?staffId=...&startDate=...&endDate=...&businessId=...
     */
    public function getVisits(Request $request)
    {
        $params = [
            'staffId' => $request->get('staffId'),
            'startDate' => $request->get('startDate'),
            'endDate' => $request->get('endDate'),
            'businessId' => $request->get('businessId'),
        ];

        $resp = null;
        if (isset($params['staffId']) && isset($params['startDate']) && isset($params['endDate']) && isset($params['businessId'])) {
            $resp = $this->getVisitsData(
                $params['staffId'],
                $params['startDate'],
                $params['endDate'],
                $params['businessId']
            );

            if ($resp) {
                return response()->json($resp);
            } else {
                return response()->json("Invalid Access");
            }
        }

        return response()->json("Invalid Access");
    }

    /**
     * Staff service report endpoint - matches legacy staffservicereport_get() behavior
     * 
     * GET /api/staff_service_report?staffId=...&startDate=...&endDate=...&businessId=...&typeid=...&categoryid=...
     */
    public function staffServiceReport(Request $request)
    {
        $params = [
            'staffId' => $request->get('staffId'),
            'startDate' => $request->get('startDate'),
            'endDate' => $request->get('endDate'),
            'businessId' => $request->get('businessId'),
            'typeid' => $request->get('typeid'),
            'categoryid' => $request->get('categoryid'),
        ];

        $resp = null;
        if (isset($params['staffId']) && isset($params['startDate']) && isset($params['endDate']) && isset($params['businessId'])) {
            $resp = $this->getCommissionDetails(
                $params['staffId'],
                $params['startDate'],
                $params['endDate'],
                $params['typeid'] ?? 'all',
                $params['categoryid'] ?? 'all',
                $params['businessId']
            );

            if ($resp) {
                return response()->json($resp);
            } else {
                return response()->json("Invalid Access");
            }
        }

        return response()->json("Invalid Access");
    }

    /**
     * Staff retail report endpoint - matches legacy staffretailreport_get() behavior
     * 
     * GET /api/staff_retail_report?staffId=...&startDate=...&endDate=...&businessId=...
     */
    public function staffRetailReport(Request $request)
    {
        $params = [
            'staffId' => $request->get('staffId'),
            'startDate' => $request->get('startDate'),
            'endDate' => $request->get('endDate'),
            'businessId' => $request->get('businessId'),
        ];

        $resp = null;
        if (isset($params['staffId']) && isset($params['startDate']) && isset($params['endDate']) && isset($params['businessId'])) {
            $resp = $this->getRetailCommissionDetails(
                $params['staffId'],
                $params['startDate'],
                $params['endDate'],
                $params['businessId']
            );

            if ($resp) {
                return response()->json($resp);
            } else {
                return response()->json("Invalid Access");
            }
        }

        return response()->json("Invalid Access");
    }

    /**
     * Appointments endpoint - matches legacy appointments_get() behavior
     * 
     * GET /api/appointments?customer_cell=...
     * Returns plain text response (not JSON) - preserves exact behavior
     */
    public function appointments(Request $request)
    {
        $customer_cell = $request->get('customer_cell');

        $data = DB::table('customer_visits')
            ->join('customers', 'customers.id_customers', '=', 'customer_visits.customer_id')
            ->join('visit_services', 'visit_services.customer_visit_id', '=', 'customer_visits.id_customer_visits')
            ->join('business', 'business.id_business', '=', 'customer_visits.business_id')
            ->where('customers.customer_cell', $customer_cell)
            ->where('customer_visits.visit_status', 'open')
            ->whereRaw("date(visit_services.visit_service_start) >= ?", [date('Y-m-d')])
            ->select(
                'customers.customer_name',
                'customers.customer_cell',
                'customers.id_customers',
                'customers.customer_gender',
                'customers.customer_email',
                'customer_visits.id_customer_visits',
                'customer_visits.business_id',
                'visit_services.id_visit_services',
                DB::raw("date_format(visit_service_start, '%a, %b %Y') as visit_date"),
                DB::raw("date_format(visit_service_start, '%H:%i') as start_time"),
                DB::raw("date_format(visit_service_end, '%H:%i') as end_time"),
                'business.id_business',
                'business.business_name',
                'business.business_address'
            )
            ->get();

        $response = '';
        foreach ($data as $d) {
            $response .= "\n📝 Appointment No.: " . $d->id_customer_visits . "\n";
            $response .= "🦱 Name: " . $d->customer_name . "\n";
            $response .= "📆 Date: " . $d->visit_date . "\n";
            $response .= "⏰ Slot: " . $d->start_time . " to " . $d->end_time . "\n";
        }

        return response($response)->header('Content-Type', 'text/plain');
    }

    /**
     * Service types endpoint - matches legacy service_types_get() behavior
     * 
     * GET /api/service_types?businessid=...
     * Returns plain array (not wrapped in ResponseFormatter)
     */
    public function serviceTypes(Request $request)
    {
        $business_id = $request->get('businessid');

        $servicesBase = asset('assets/images/servicetype/');

        $services = DB::table('service_type')
            ->where('business_id', $business_id)
            ->where('service_type_active', 'Yes')
            ->select(
                'id_service_types as id',
                'business_id',
                'service_type as name',
                DB::raw("CONCAT('" . $servicesBase . "', IFNULL(service_type_image, 'nu.jpg')) as service_type_image"),
                'service_type_active',
                'order_id',
                DB::raw("'servicetype' as flag")
            )
            ->get()
            ->toArray();

        // Convert to array and sort by order_id
        $servicesArray = array_map(function ($item) {
            return (array) $item;
        }, $services);

        $key_values = array_column($servicesArray, 'order_id');
        array_multisort($key_values, SORT_ASC, $servicesArray);

        return response()->json($servicesArray);
    }

    /**
     * Service categories endpoint - matches legacy service_categories_get() behavior
     * 
     * GET /api/service_categories?businessid=...&typeid=...&flag=...
     * Returns plain array (not wrapped in ResponseFormatter)
     */
    public function serviceCategories(Request $request)
    {
        $business_id = $request->get('businessid');
        $type_id = $request->get('typeid');
        $flag = $request->get('flag');

        if ($flag == '') {
            return response()->json("Invalid Flag passed");
        }

        $categoryBase = asset('assets/images/category/');

        if ($flag === "servicetype") {
            $data = DB::table('service_category')
                ->join('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
                ->join('business', 'business.id_business', '=', 'service_category.business_id')
                ->where('service_category.service_category_active', 'Yes')
                ->when($type_id !== null, function ($query) use ($type_id) {
                    return $query->where('service_category.service_type_id', $type_id);
                })
                ->select(
                    'service_category.id_service_category',
                    'service_category.id_service_category as id',
                    'service_category.service_type_id',
                    'service_type.service_type',
                    'service_category.order_id',
                    'service_category.service_category',
                    'service_category.service_category as name',
                    'service_category.business_id',
                    'business.business_name',
                    'service_category.service_category_active',
                    DB::raw("CONCAT('" . $categoryBase . "', IFNULL(service_category_image, 'nu.png')) as service_category_image"),
                    DB::raw("'servicetype' as flag")
                )
                ->get()
                ->toArray();
        } else {
            $data = DB::table('package_category')
                ->join('package_type', 'package_type.id_package_type', '=', 'package_category.package_type_id')
                ->join('business', 'business.id_business', '=', 'package_category.business_id')
                ->where('package_category.service_category_active', 'Yes')
                ->when($type_id !== null, function ($query) use ($type_id) {
                    return $query->where('package_category.package_type_id', $type_id);
                })
                ->select(
                    'package_category.id_package_category',
                    'package_category.id_package_category as id',
                    'package_category.package_type_id',
                    'package_type.service_type',
                    'package_category.order_id',
                    'package_category.service_category',
                    'package_category.service_category as name',
                    'package_category.business_id',
                    'business.business_name',
                    'package_category.service_category_active',
                    DB::raw("CONCAT('" . $categoryBase . "', IFNULL(service_category_image, 'nu.png')) as service_category_image"),
                    DB::raw("'packagetype' as flag")
                )
                ->get()
                ->toArray();
        }

        $dataArray = array_map(function ($item) {
            return (array) $item;
        }, $data);

        return response()->json($dataArray);
    }

    /**
     * Services endpoint - matches legacy services_get() behavior
     * 
     * GET /api/services?businessid=...&categoryid=...&flag=...
     * Returns plain array (not wrapped in ResponseFormatter)
     */
    public function services(Request $request)
    {
        $business_id = $request->get('businessid');
        $category_id = $request->get('categoryid');
        $flag = $request->get('flag');

        if ($flag === "packagetype") {
            $data = DB::table('business_services')
                ->join('package_services', 'business_services.id_business_services', '=', 'package_services.service_id')
                ->join('package_category', 'package_category.id_package_category', '=', 'package_services.package_category_id')
                ->join('package_type', 'package_type.id_package_type', '=', 'package_category.package_type_id')
                ->where('package_category_id', $category_id)
                ->where('service_active', 'Yes')
                ->select(
                    'business_services.*',
                    'business_services.id_business_services as id',
                    DB::raw("CONCAT(business_services.service_name, ' | ', service_rate, ' | ', service_duration) as name"),
                    'package_services.service_rate as price',
                    'package_type.service_type',
                    'package_type.id_package_type as id_service_type',
                    'package_category.service_category',
                    'package_services.package_category_id as service_category_id',
                    DB::raw("'packagetype' as flag"),
                    'business_services.service_duration',
                    DB::raw("'20' as discount"),
                    DB::raw("FORMAT(package_services.service_rate, 2) as service_rate")
                )
                ->orderBy('business_services.order_id', 'ASC')
                ->get();
        } else {
            $data = DB::table('business_services')
                ->join('service_category', 'service_category.id_service_category', '=', 'business_services.service_category_id')
                ->join('service_type', 'service_type.id_service_types', '=', 'service_category.service_type_id')
                ->where('service_category_id', $category_id)
                ->where('service_active', 'Yes')
                ->select(
                    'business_services.*',
                    'business_services.id_business_services as id',
                    DB::raw("CONCAT(business_services.service_name, ' | ', service_rate, ' | ', service_duration) as name"),
                    'business_services.service_rate as price',
                    'service_type.service_type',
                    'service_type.id_service_types as id_service_type',
                    'service_category.service_category',
                    DB::raw("'servicetype' as flag"),
                    'business_services.service_duration',
                    DB::raw("'20' as discount"),
                    DB::raw("FORMAT(business_services.service_rate, 2) as service_rate")
                )
                ->orderBy('business_services.order_id', 'ASC')
                ->get();
        }

        return response()->json($data);
    }

    // ==================== HRM Methods ====================

    /**
     * Get leave types endpoint - matches legacy getleavetypes_get() behavior
     * 
     * GET /api/get_leave_types?businessId=...
     */
    public function getLeaveTypes(Request $request)
    {
        $businessId = $request->get('businessId');

        $resp = null;
        if (isset($businessId)) {
            $resp = $this->getLeaveTypesData($businessId);
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Get remaining leaves endpoint - matches legacy get_remaining_leaves_get() behavior
     * 
     * GET /api/get_remaining_leaves?staff_id=...
     */
    public function getRemainingLeaves(Request $request)
    {
        $staff_id = $request->get('staff_id');

        $resp = null;
        if (isset($staff_id)) {
            $resp = $this->getRemainingLeavesData($staff_id);
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Leave application endpoint - matches legacy leave_application_get() behavior
     * 
     * GET /api/leave_application?staff_id=...&fcm_id=...&required_from=...&required_to=...&business_id=...&leave_type=...&comment=...
     */
    public function leaveApplication(Request $request)
    {
        $params = [
            'staff_id' => $request->get('staff_id'),
            'fcm_id' => $request->get('fcm_id'),
            'required_from' => $request->get('required_from'),
            'required_to' => $request->get('required_to'),
            'business_id' => $request->get('business_id'),
            'leave_type' => $request->get('leave_type'),
            'comment' => $request->get('comment'),
        ];

        $resp = null;
        if (isset($params['staff_id']) && isset($params['fcm_id']) && isset($params['required_from']) && 
            isset($params['required_to']) && isset($params['business_id']) && isset($params['leave_type']) && 
            isset($params['comment'])) {
            $resp = $this->createLeaveApplication(
                $params['staff_id'],
                $params['fcm_id'],
                $params['required_from'],
                $params['required_to'],
                $params['business_id'],
                $params['leave_type'],
                $params['comment']
            );
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Loan application endpoint - matches legacy loan_application_get() behavior
     * 
     * GET /api/loan_application?staff_id=...&fcm_id=...&amount=...&num_of_months=...&comment=...
     */
    public function loanApplication(Request $request)
    {
        $params = [
            'staff_id' => $request->get('staff_id'),
            'fcm_id' => $request->get('fcm_id'),
            'amount' => $request->get('amount'),
            'num_of_months' => $request->get('num_of_months'),
            'comment' => $request->get('comment'),
        ];

        $resp = null;
        if (isset($params['staff_id']) && isset($params['fcm_id']) && isset($params['amount']) && 
            isset($params['num_of_months']) && isset($params['comment'])) {
            $resp = $this->createLoanApplication(
                $params['staff_id'],
                $params['fcm_id'],
                $params['amount'],
                $params['num_of_months'],
                $params['comment']
            );
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Leave application report endpoint - matches legacy leave_application_report_get() behavior
     * 
     * GET /api/leave_application_report?staff_id=...
     */
    public function leaveApplicationReport(Request $request)
    {
        $staff_id = $request->get('staff_id');

        $resp = null;
        if (isset($staff_id)) {
            $resp = $this->getLeaveApplicationReport($staff_id);
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    /**
     * Loan application report endpoint - matches legacy loan_application_report_get() behavior
     * 
     * GET /api/loan_application_report?staff_id=...
     */
    public function loanApplicationReport(Request $request)
    {
        $staff_id = $request->get('staff_id');

        $resp = null;
        if (isset($staff_id)) {
            $resp = $this->getLoanApplicationReport($staff_id);
        }

        if ($resp) {
            return response()->json($resp);
        } else {
            return response()->json("Invalid Access");
        }
    }

    // ==================== Private Helper Methods ====================

    /**
     * Perform login logic - replicates legacy login() method
     */
    private function performLogin($params)
    {
        $fcm_id = $params['fcm_id'];
        $cellNumber = $params['cellNumber'];
        $username = $params['username'];

        // First attempt: Find staff by fcm_id and cellNumber
        $staffImageBase = asset('assets/images/staff/');
        $businessLogoBase = asset('assets/images/business/');
        
        $staff = DB::table('staff')
            ->join('business', 'business.id_business', '=', 'staff.business_id')
            ->where('staff.staff_active', 'Y')
            ->where('staff.fcm_id', $fcm_id)
            ->where('staff.staff_cell', $cellNumber)
            ->select(
                'staff.id_staff',
                'staff.staff_fullname',
                'staff.staff_firstname',
                'staff.staff_lastname',
                'staff.staff_cell',
                'staff.staff_email',
                'staff.staff_eid',
                DB::raw("CONCAT('" . $staffImageBase . "', staff.staff_image) as staff_image"),
                'staff.staff_comment',
                'staff.staff_shared',
                DB::raw('staff.business_id as id_business'),
                'business.business_name',
                DB::raw("CONVERT(LEFT(business.business_opening_time, 2), DOUBLE) as business_opening_time"),
                DB::raw("CONVERT(LEFT(business.business_closing_time, 2), DOUBLE) as business_closing_time"),
                DB::raw("CONVERT(LEFT(IFNULL(staff.day_start_time, '11'), 2), DOUBLE) as staff_opening_time"),
                DB::raw("CONVERT(LEFT(IFNULL(staff.day_end_time, '20'), 2), DOUBLE) as staff_closing_time"),
                DB::raw("CONCAT('" . $businessLogoBase . "', business.business_logo) as logo"),
                'staff.fcm_id',
                DB::raw("'correct' as status")
            )
            ->first();

        if ($staff) {
            $row = (array) $staff;
            return $this->formatter->formatResponse("true", "Login Succesfull", "staff", [$row]);
        }

        // Second attempt: Find staff by username and cellNumber (fallback)
        $staffCheck = DB::table('staff')
            ->where('staff_active', 'Y')
            ->where('staff_firstname', $username)
            ->where('staff_cell', $cellNumber)
            ->select('staff.id_staff', DB::raw("IFNULL(staff.fcm_id, '') as fcm_id"))
            ->first();

        if ($staffCheck) {
            // Check if staff has no FCM ID or empty FCM ID
            if ($staffCheck->fcm_id == null || $staffCheck->fcm_id == '') {
                // Check if FCM ID is already in use by another staff
                $fcmUsed = DB::table('staff')
                    ->where('fcm_id', $fcm_id)
                    ->where('staff_active', 'Y')
                    ->first();

                if (!$fcmUsed) {
                    // Update staff with FCM ID
                    DB::table('staff')
                        ->where('id_staff', $staffCheck->id_staff)
                        ->update(['fcm_id' => $fcm_id]);

                    // Get full staff data after update
                    $staffImageBase = asset('assets/images/staff/');
                    $businessLogoBase = asset('assets/images/business/');
                    
                    $staff = DB::table('staff')
                        ->join('business', 'business.id_business', '=', 'staff.business_id')
                        ->where('staff.staff_active', 'Y')
                        ->where('staff.staff_firstname', $username)
                        ->where('staff.staff_cell', $cellNumber)
                        ->select(
                            'staff.id_staff',
                            'staff.staff_fullname',
                            'staff.staff_firstname',
                            'staff.staff_lastname',
                            'staff.staff_cell',
                            'staff.staff_email',
                            'staff.staff_eid',
                            DB::raw("CONCAT('" . $staffImageBase . "', staff.staff_image) as staff_image"),
                            'staff.staff_comment',
                            'staff.staff_shared',
                            DB::raw('staff.business_id as id_business'),
                            'business.business_name',
                            DB::raw("IFNULL(staff.fcm_id, '') as fcm_id"),
                            DB::raw("CONVERT(LEFT(business.business_opening_time, 2), DOUBLE) as business_opening_time"),
                            DB::raw("CONVERT(LEFT(business.business_closing_time, 2), DOUBLE) as business_closing_time"),
                            DB::raw("CONVERT(LEFT(IFNULL(staff.day_start_time, '11'), 2), DOUBLE) as staff_opening_time"),
                            DB::raw("CONVERT(LEFT(IFNULL(staff.day_end_time, '20'), 2), DOUBLE) as staff_closing_time"),
                            DB::raw("CONCAT('" . $businessLogoBase . "', business.business_logo) as logo")
                        )
                        ->first();

                    if ($staff) {
                        $row = (array) $staff;
                        return $this->formatter->formatResponse("true", "Login Succesfull", "staff", [$row]);
                    }
                } else {
                    return $this->formatter->formatResponse("false", "fcm ID already in Use");
                }
            } else {
                return $this->formatter->formatResponse("false", "Please Use Registered Cell Number");
            }
        } else {
            return $this->formatter->formatResponse("false", "Invalid User Password");
        }

        return null;
    }

    /**
     * Check mobile number logic - replicates legacy checkMobileNumber() method
     */
    private function checkMobile($cellNumber)
    {
        $staff = DB::table('staff')
            ->where('staff_cell', $cellNumber)
            ->get();

        if ($staff->count() > 0) {
            $row = $staff->map(function ($item) {
                return (array) $item;
            })->toArray();
            return $this->formatter->formatResponse("true", "Number Registerd", "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "No User Found with this Cell Number");
        }
    }

    /**
     * Mark staff attendance - replicates legacy mark_staff_attendance() method
     */
    private function markStaffAttendance($cellNumber, $fcm_id, $key, $date, $time, $businessId, $address, $longitude, $latitude)
    {
        // Legacy code overwrites date/time with current date/time
        $date = date('Y-m-d');
        $time = date('H:i:s');

        $staff = DB::table('staff')
            ->where('staff_cell', $cellNumber)
            ->whereRaw("IFNULL(fcm_id, '') = ?", [$fcm_id])
            ->where('business_id', $businessId)
            ->first();

        if (!$staff) {
            return $this->formatter->formatResponse("false", "Please Use Registered Cell Number Or your finger print is not registered");
        }

        $staff_id = $staff->id_staff;

        // Check existing attendance for today
        $exists = DB::table('staff_attendance')
            ->where('staff_id', $staff_id)
            ->whereRaw("DATE(time_in) = ?", [$date])
            ->selectRaw("*, IFNULL(time_out, '') as t_out")
            ->first();

        $timeParts = explode(":", $time);
        $hour = (int) $timeParts[0];

        // Case 1: Attendance exists and time is before 14:00 (2 PM)
        if ($exists && $hour < 14) {
            return $this->formatter->formatResponse("false", "Attendance already marked for today");
        }

        // Case 2: Attendance exists, time >= 14:00, and no time_out yet
        if ($exists && $hour >= 14 && ($exists->t_out == '' || $exists->t_out == null)) {
            return $this->handleTimeOut($exists, $date, $time, $businessId, $key, $staff);
        }

        // Case 3: No attendance record exists
        if (!$exists) {
            return $this->handleTimeIn($staff_id, $date, $time, $businessId, $address, $longitude, $latitude, $key, $staff);
        }

        // Case 4: Attendance exists and time_out needs to be updated
        return $this->handleTimeOutUpdate($exists, $date, $time, $businessId, $key, $staff);
    }

    /**
     * Handle time in (first attendance of the day)
     */
    private function handleTimeIn($staff_id, $date, $time, $businessId, $address, $longitude, $latitude, $key, $staff)
    {
        $keyCode = DB::table('key_code')
            ->where('business_id', $businessId)
            ->select('current_key')
            ->first();

        if (!$keyCode) {
            return $this->formatter->formatResponse("true", "Key Code not available at the server", "staff", [(array) $staff]);
        }

        if ($keyCode->current_key != $key) {
            return $this->formatter->formatResponse("false", "Your Phone or finger print is not registered or QR code is not valid ", "staff", [(array) $staff]);
        }

        // Insert attendance record
        DB::table('staff_attendance')->insert([
            'business_id' => $businessId,
            'staff_id' => $staff_id,
            'time_in' => $date . ' ' . $time,
            'address' => $address,
            'longitude' => $longitude,
            'latitude' => $latitude,
        ]);

        // Generate new key
        $newKey = $this->getRandomString(8);
        DB::table('key_code')
            ->where('business_id', $businessId)
            ->update(['current_key' => $newKey]);

        return $this->formatter->formatResponse("true", "ATTENDANCE MARKED SUCCESSFULLY AT " . $date . ' ' . $time, "staff", [(array) $staff]);
    }

    /**
     * Handle time out (when attendance exists and time >= 14:00)
     */
    private function handleTimeOut($exists, $date, $time, $businessId, $key, $staff)
    {
        $keyCode = DB::table('key_code')
            ->where('business_id', $businessId)
            ->select('current_key')
            ->first();

        if (!$keyCode) {
            return $this->formatter->formatResponse("true", "Key Code not available at the server", "staff", [(array) $staff]);
        }

        if ($keyCode->current_key != $key) {
            return $this->formatter->formatResponse("false", "Your Phone or finger print is not registered or QR code is not valid ", "staff", [(array) $staff]);
        }

        // Update time_out
        DB::table('staff_attendance')
            ->where('id_staff_attendance', $exists->id_staff_attendance)
            ->update(['time_out' => $date . ' ' . $time]);

        // Generate new key
        $newKey = $this->getRandomString(8);
        DB::table('key_code')
            ->where('business_id', $businessId)
            ->update(['current_key' => $newKey]);

        return $this->formatter->formatResponse("true", "Your TIME OUT has been saved", "staff", [(array) $staff]);
    }

    /**
     * Handle time out update (when attendance exists and updating time_out)
     */
    private function handleTimeOutUpdate($exists, $date, $time, $businessId, $key, $staff)
    {
        $keyCode = DB::table('key_code')
            ->where('business_id', $businessId)
            ->select('current_key')
            ->first();

        if (!$keyCode) {
            return $this->formatter->formatResponse("true", "Key Code not available at the server", "staff", [(array) $staff]);
        }

        if ($keyCode->current_key != $key) {
            return $this->formatter->formatResponse("true", "Expired or Invalid Key Code. Wait for new code before retrying.", "staff", [(array) $staff]);
        }

        // Update time_out
        DB::table('staff_attendance')
            ->where('id_staff_attendance', $exists->id_staff_attendance)
            ->update(['time_out' => $date . ' ' . $time]);

        // Generate new key
        $newKey = $this->getRandomString(8);
        DB::table('key_code')
            ->where('business_id', $businessId)
            ->update(['current_key' => $newKey]);

        return $this->formatter->formatResponse("true", "Updated TIME OUT TO " . $date . ' ' . $time, "staff", [(array) $staff]);
    }

    /**
     * Staff attendance request - replicates legacy staff_attendance_request() method
     */
    private function staffAttendanceRequest($cellNumber, $fcm_id, $key, $date, $time, $businessId, $address, $longitude, $latitude)
    {
        $staff = DB::table('staff')
            ->where('staff_cell', $cellNumber)
            ->where('fcm_id', $fcm_id)
            ->where('business_id', $businessId)
            ->first();

        if (!$staff) {
            return $this->formatter->formatResponse("false", "Please Use Registered Cell Number");
        }

        $staff_id = $staff->id_staff;

        // Check existing attendance for today
        $exists = DB::table('staff_attendance')
            ->where('staff_id', $staff_id)
            ->whereRaw("DATE(time_in) = ?", [$date])
            ->selectRaw("*, IFNULL(time_out, '') as t_out")
            ->first();

        $timeParts = explode(":", $time);
        $hour = (int) $timeParts[0];

        // Case 1: Attendance exists and time is before 14:00
        if ($exists && $hour < 14) {
            return $this->formatter->formatResponse("false", "Attendance already marked for today");
        }

        // Case 2: Attendance exists and time > 14:00 (time out request)
        if ($exists && $hour > 14) {
            // Check if time out request already exists
            $timeOutRequest = DB::table('attendance_request')
                ->whereRaw("DATE(request_time_out) = ?", [$date])
                ->where('staff_id', $staff_id)
                ->whereRaw("IFNULL(request_time_out, '') != ''")
                ->first();

            if (!$timeOutRequest) {
                DB::table('attendance_request')->insert([
                    'business_id' => $businessId,
                    'staff_id' => $staff_id,
                    'request_date' => $date,
                    'request_time_out' => $date . ' ' . $time,
                    'address' => $address,
                    'longitude' => $longitude,
                    'latitude' => $latitude,
                ]);
                return $this->formatter->formatResponse("true", "Attendance Time Out Requested Succesfully", "staff", [(array) $staff]);
            } else {
                return $this->formatter->formatResponse("true", "Your attendance time out request are already received", "staff", [(array) $staff]);
            }
        }

        // Case 3: No attendance exists (time in request)
        // Check if time in request already exists
        $timeInRequest = DB::table('attendance_request')
            ->whereRaw("DATE(request_time_in) = ?", [$date])
            ->where('staff_id', $staff_id)
            ->whereRaw("IFNULL(request_time_in, '') != ''")
            ->first();

        if (!$timeInRequest) {
            DB::table('attendance_request')->insert([
                'business_id' => $businessId,
                'staff_id' => $staff_id,
                'request_date' => $date,
                'request_time_in' => $date . ' ' . $time,
                'address' => $address,
                'longitude' => $longitude,
                'latitude' => $latitude,
            ]);
            return $this->formatter->formatResponse("true", "Attendance Time In Requested Succesfully", "staff", [(array) $staff]);
        } else {
            // Time in request exists, check if we can add time out request
            $timeIn = $timeInRequest->request_time_in;
            $id_attendance_request = $timeInRequest->id_attendance_request;

            // Check if time out request exists
            $timeOutRequest = DB::table('attendance_request')
                ->whereRaw("DATE(request_time_out) = ?", [$date])
                ->where('staff_id', $staff_id)
                ->whereRaw("IFNULL(request_time_out, '') != ''")
                ->first();

            if (!$timeOutRequest && $time != $timeIn) {
                DB::table('attendance_request')
                    ->where('id_attendance_request', $id_attendance_request)
                    ->update([
                        'request_time_out' => $date . ' ' . $time,
                        'address' => $address,
                        'longitude' => $longitude,
                        'latitude' => $latitude,
                    ]);
                return $this->formatter->formatResponse("true", "Attendance Time Out Requested Succesfully", "staff", [(array) $staff]);
            } else {
                return $this->formatter->formatResponse("true", "Doubled Clicked or Your attendance time in and time out requests are already received", "staff", [(array) $staff]);
            }
        }
    }

    /**
     * Get staff month attendance detail - replicates legacy staff_month_attendance_detail() method
     */
    private function getStaffMonthAttendanceDetail($staffid, $year, $month)
    {
        // Get staff details with deduction policy
        // Note: provident_fund_rate, week_day_off, salary_monthly_tax are from staff table, not deduction_policy
        $staff = DB::table('staff')
            ->join('deduction_policy', 'deduction_policy.id_deduction_policy', '=', 'staff.deduction_policy')
            ->join('business', 'business.id_business', '=', 'staff.business_id')
            ->where('staff.id_staff', $staffid)
            ->select(
                'staff.id_staff',
                'staff.staff_fullname',
                'staff.staff_image',
                'staff.staff_salary',
                'staff.provident_fund_rate',
                'staff.week_day_off',
                'deduction_policy.deduction_days_absent',
                'deduction_policy.deduction_days_late',
                'deduction_policy.total_late_allowed',
                'deduction_policy.total_absents_allowed',
                'staff.salary_monthly_tax',
                'staff.day_start_time',
                'business.timein_margin'
            )
            ->first();

        if (!$staff) {
            return $this->formatter->formatResponse("false", "No Records Found for this Staff");
        }

        $week_day_off = $staff->week_day_off ?? '';
        $day_start_time = $staff->day_start_time ?? '11';
        $margin = $staff->timein_margin ?? 20;

        // Get business_id for holiday filtering
        $businessId = DB::table('staff')
            ->where('id_staff', $staffid)
            ->value('business_id');

        // Check if business has common_products
        $commonProducts = DB::table('business')
            ->where('id_business', $businessId)
            ->value('common_products');

        // Build the complex SQL query
        $sql = "SELECT 
                mDays.Date,
                mDays.Day,
                '" . addslashes($week_day_off) . "' as week_day_off,
                " . $staff->id_staff . " as id_staff, 
                '" . addslashes($staff->staff_fullname) . "' AS staff_fullname,
                min(attendance.time_in) time_in,
                max(attendance.time_out) time_out,
                staff_leaves.staff_leave_date,
                staff_leaves.leave_reason,
                holidays.holiday_date,
                holidays.holiday_reason,
                CONCAT(ROUND(TIMESTAMPDIFF(MINUTE, min(attendance.time_in), max(attendance.time_out)) / 60,2),' Hours') AS 'Work_Hours',
                CASE WHEN
                        IFNULL(attendance.time_in, '') != '' AND attendance.time_in != CONCAT(DATE(attendance.time_in), ' 00:00:00')
                    THEN 1
                    ELSE 0
                END AS 'Present',
                CASE
                    WHEN INSTR('" . addslashes($week_day_off) . "',  mDays.Day) > 0 AND IFNULL(attendance.time_in, '') = '' THEN 0
                    WHEN '" . addslashes($week_day_off) . "' Not Like concat('%',mDays.Day,'%') AND IFNULL(attendance.time_in, '') = '' 
                        AND IFNULL(staff_leave_date, '') = '' 
                        AND IFNULL(holidays.holiday_date,'') = ''  THEN 1 
                    WHEN IFNULL(holidays.holiday_date,'') != '' THEN 0    
                    WHEN IFNULL(staff_leave_date, '') != '' THEN 0
                    ELSE 0
                END AS 'Absent',
                CASE
                    WHEN
                        TIMESTAMPDIFF(MINUTE,
                            CONCAT(mDays.Date, ' ', '" . addslashes($day_start_time) . "'),
                            min(attendance.time_in)) > " . $margin . "
                    THEN
                        1
                    WHEN min(attendance.time_in) = CONCAT(DATE(min(attendance.time_in)), ' 00:00:00') THEN 0
                    ELSE 0
                END AS 'Late',
                CASE
                    WHEN IFNULL(staff_leave_date, '') = '' THEN 0
                    WHEN '" . addslashes($week_day_off) . "' NOT LIKE concat('%',mDays.Day,'%') AND IFNULL(holidays.holiday_date,'') = '' AND IFNULL(staff_leave_date, '') != '' AND IFNULL(attendance.time_in, '') = '' THEN 1        
                    ELSE 0
                END AS 'Leaves',
                CASE 
                    WHEN IFNULL(holidays.holiday_date,'') != '' THEN 1
                    WHEN INSTR('" . addslashes($week_day_off) . "', mDays.Day) > 0 THEN 1
                    ELSE 0 
                END AS 'Holiday',
                attendance.address,   
                business.business_opening_time,
                business.business_closing_time
            FROM (SELECT 
                    FROM_UNIXTIME(UNIX_TIMESTAMP(CONCAT('" . $year . "-" . $month . "-', n)), '%Y-%m-%d') AS Date,
                    FROM_UNIXTIME(UNIX_TIMESTAMP(CONCAT('" . $year . "-" . $month . "-', n)), '%a') AS Day
                            FROM
                    (SELECT 
                                    (((b4.0 << 1 | b3.0) << 1 | b2.0) << 1 | b1.0) << 1 | b0.0 AS n
                            FROM
                                    (SELECT 0 UNION ALL SELECT 1) AS b0, (SELECT 0 UNION ALL SELECT 1) AS b1, (SELECT 0 UNION ALL SELECT 1) AS b2, (SELECT 0 UNION ALL SELECT 1) AS b3, (SELECT 0 UNION ALL SELECT 1) AS b4) t
                            WHERE
                                    n > 0 AND n <= DAY(LAST_DAY('" . $year . "-" . $month . "-01'))) AS mDays
            LEFT JOIN
                (SELECT staff_id, time_in, time_out, address
                FROM staff_attendance
                WHERE
                    MONTH(time_in) = " . $month . "
                    AND YEAR(time_in) = '" . $year . "'
                    AND staff_id = " . $staff->id_staff . ") AS attendance ON DATE(time_in) = mDays.Date
            LEFT JOIN
                (SELECT id_staff,
                        staff_fullname,
                        staff_image,
                        week_day_off,
                        business_id
                FROM staff WHERE id_staff = " . $staff->id_staff . ") AS staff ON staff.id_staff = attendance.staff_id
            LEFT JOIN 
                (SELECT staff_id,
                        staff_leave_date,
                        leave_reason
                 FROM staff_leaves 
                 WHERE staff_id = " . $staff->id_staff . "
                 AND staff_leave_status = 'Active' 
                 AND MONTH(staff_leave_date) =  " . $month . "
                 AND YEAR(staff_leave_date) =  '" . $year . "') As staff_leaves 
                 ON date(staff_leaves.staff_leave_date) = mDays.Date
            LEFT JOIN (
                SELECT holiday_date,
                holiday_day,
                holiday_reason
                FROM holidays WHERE";

        // Add business_id filter if common_products is 'No'
        if ($commonProducts == 'No') {
            $sql .= " business_id= " . $businessId . " AND ";
        }

        $sql .= " MONTH(holiday_date) = " . $month . "
                AND YEAR(holiday_date)='" . $year . "'       
                ) As holidays ON holiday_date = mDays.Date
            LEFT JOIN
                (SELECT id_business, business_opening_time, business_closing_time
                FROM business) AS business ON staff.business_id = business.id_business
                Group by mDays.Date
            ORDER BY mDays.Date, time_in;";

        $results = DB::select($sql);

        if (count($results) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $results);
            return $this->formatter->formatResponse("true", "Attendance", "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "No Records Found for this Staff");
        }
    }

    /**
     * Get attendance request report - replicates legacy attendance_request_report() method
     */
    private function getAttendanceRequestReport($staff_id, $year, $month, $business_id)
    {
        $sql = "SELECT staff_id, staff_fullname, DATE(request_date) request_date, 
                IFNULL(DATE_FORMAT(request_time_in, '%d-%m-%Y %H:%i'),'-') as request_time_in, 
                IFNULL(DATE_FORMAT(request_time_out, '%d-%m-%Y %H:%i'),'-') as request_time_out, 
                request_status, approved_by
                FROM attendance_request
                JOIN staff ON staff.id_staff = attendance_request.staff_id
                WHERE staff_id = " . (int) $staff_id . " 
                AND MONTH(request_date) = " . (int) $month . "  
                AND YEAR(request_date) = '" . addslashes($year) . "'
                AND attendance_request.business_id = " . (int) $business_id;

        $results = DB::select($sql);

        if (count($results) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $results);
            return $this->formatter->formatResponse("true", "Attendance Requests", "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "No Records Found for this Staff");
        }
    }

    /**
     * Get visits data - replicates legacy getVisits() method
     */
    private function getVisitsData($staffId, $startDate, $endDate, $businessId)
    {
        $sql = "SELECT id_customer_visits, customer_name, customer_cell, service_name, staff_id, staff_name, 
            date_format(visit_services.visit_service_start,'%Y-%m-%d %H:%i:%s') as service_start,
            date_format(visit_services.visit_service_end,'%Y-%m-%d %H:%i:%s') as service_end,
            customer_visits.business_id, business.business_name, customer_visits.visit_status,
            Case when customer_visits.business_id = ? then
                'same branch'
            Else
                'other branch'
            End as branch,
            visit_color, visit_color_type, business.business_opening_time, business.business_closing_time
            FROM customer_visits
            join customers on customers.id_customers = customer_visits.customer_id
            join visit_services on visit_services.customer_visit_id = customer_visits.id_customer_visits
            join visit_service_staffs on visit_service_staffs.customer_visit_id = customer_visits.id_customer_visits
            and visit_service_staffs.visit_service_id = visit_services.id_visit_services
            join business on customer_visits.business_id = business.id_business
            where visit_status in ('open', 'invoiced')  
            and date(visit_services.visit_service_start) >= ? 
            and date(visit_services.visit_service_end) <= ?  
            and visit_service_staffs.staff_id = ?";

        $results = DB::select($sql, [
            $businessId,
            $startDate . ' 00:00:00',
            $endDate . ' 23:59:59',
            $staffId
        ]);

        if (count($results) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $results);
            return $this->formatter->formatResponse("true", count($results), "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "0", []);
        }
    }

    /**
     * Get commission details - replicates legacy commission_details() method
     * Note: business_id=0 case handled by using provided businessId parameter (stateless API)
     */
    private function getCommissionDetails($staff_id, $startdate, $enddate, $typeid, $categoryid, $business_id)
    {
        $query = DB::table('invoice_staff')
            ->join('invoice', 'invoice.id_invoice', '=', 'invoice_staff.invoice_id')
            ->join('staff', 'staff.id_staff', '=', 'invoice_staff.staff_id')
            ->join('invoice_details', 'invoice_staff.invoice_detail_id', '=', 'invoice_details.id_invoice_details')
            ->join('business_services as bs', 'invoice_details.service_id', '=', 'bs.id_business_services')
            ->where('invoice.invoice_status', 'valid')
            ->where('invoice.invoice_type', 'service')
            ->where('invoice.business_id', $business_id)
            ->whereRaw('invoice_date >= ?', [$startdate . ' 00:00'])
            ->whereRaw('invoice_date <= ?', [$enddate . ' 23:59'])
            ->where('invoice_staff.staff_id', $staff_id)
            ->when($typeid !== 'all', function ($query) use ($typeid) {
                return $query->where('invoice_details.service_type', $typeid);
            })
            ->when($categoryid !== 'all', function ($query) use ($categoryid) {
                return $query->where('invoice_details.service_category', $categoryid);
            })
            ->select(
                'invoice.reference_invoice_number',
                'staff.staff_fullname',
                'invoice.customer_name',
                'invoice_staff.staff_id',
                'invoice_staff.service_type',
                'invoice_staff.service_category',
                'invoice_staff.service_name',
                'invoice_staff.additional_staff',
                'bs.commission_perc as staff_commission',
                'bs.helper_perc as helper_commission',
                'invoice_staff.paid',
                DB::raw("date_format(invoice.invoice_date,'%Y-%m-%d %H:%i') as staff_service_date"),
                'invoice_staff.invoice_id',
                'invoice_details.invoice_discount',
                'invoice_details.discounted_price',
                'invoice_details.paid as final_price',
                DB::raw("date_format(invoice.invoice_date, '%Y-%m-%d %H:%i') as customer_visit_date")
            )
            ->get();

        if (count($query) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $query->toArray());
            return $this->formatter->formatResponse("true", count($query), "performance", $row);
        } else {
            return $this->formatter->formatResponse("false", "0", []);
        }
    }

    /**
     * Get retail commission details - replicates legacy retail_commission_details() method
     * Note: business_id=0 case handled by using provided businessId parameter (stateless API)
     */
    private function getRetailCommissionDetails($staff_id, $startdate, $enddate, $business_id)
    {
        $query = DB::table('invoice_products')
            ->join('business_products as bp', 'invoice_products.product_id', '=', 'bp.id_business_products')
            ->join('business_brands as bb', 'bb.id_business_brands', '=', 'bp.brand_id')
            ->join('invoice', 'invoice.id_invoice', '=', 'invoice_products.invoice_id')
            ->join('staff', 'staff.id_staff', '=', 'invoice_products.staff_id')
            ->where('invoice.invoice_status', 'valid')
            ->where('invoice.business_id', $business_id)
            ->whereRaw('invoice_date >= ?', [$startdate . ' 00:00'])
            ->whereRaw('invoice_date <= ?', [$enddate . ' 23:59'])
            ->where('invoice_products.staff_id', $staff_id)
            ->select(
                'invoice.reference_invoice_number',
                'invoice.customer_name',
                'staff.staff_fullname',
                'invoice_products.staff_id',
                'bb.business_brand_name',
                'bp.product',
                'bp.qty_per_unit',
                'bp.measure_unit',
                'invoice_products.invoice_qty',
                'bp.commission as commission',
                'invoice_products.paid',
                DB::raw("date_format(invoice.invoice_date, '%Y-%m-%d %H:%i') as invoice_date"),
                'invoice_products.invoice_id',
                'invoice_products.discounted_price',
                DB::raw("round((invoice_products.discounted_price*bp.commission)/100,2) as staff_commission")
            )
            ->get();

        if (count($query) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $query->toArray());
            return $this->formatter->formatResponse("true", count($query), "performance", $row);
        } else {
            return $this->formatter->formatResponse("false", "0", []);
        }
    }

    /**
     * Get leave types - replicates legacy get_leave_types() method
     */
    private function getLeaveTypesData($businessid)
    {
        $query = DB::table('leave_type')
            ->select('id_leave_type', 'leave_type')
            ->get();

        if (count($query) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $query->toArray());
            return $this->formatter->formatResponse("true", count($query), "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "0", []);
        }
    }

    /**
     * Get remaining leaves - replicates legacy get_remaining_leaves() method
     */
    private function getRemainingLeavesData($staffid)
    {
        $sql = "SELECT id_staff, staff_fullname, staff_annual_leaves, count(id_staff_leaves) as leaves_taken,
            staff_annual_leaves - count(id_staff_leaves) as remainin_leaves 
            FROM staff
            LEFT JOIN staff_leaves ON staff.id_staff = staff_leaves.staff_id  
            AND YEAR(staff_leave_date) = YEAR(NOW()) 
            AND staff_leaves.staff_leave_status = 'Active'
            WHERE staff.id_staff = ?";

        $results = DB::select($sql, [$staffid]);

        if (count($results) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $results);
            return $this->formatter->formatResponse("true", count($results), "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "0", []);
        }
    }

    /**
     * Create leave application - replicates legacy leave_application() method
     */
    private function createLeaveApplication($staff_id, $fcm_id, $required_from, $required_to, $business_id, $leave_type, $comment)
    {
        $staff = DB::table('staff')
            ->where('id_staff', $staff_id)
            ->whereRaw("IFNULL(fcm_id, '') = ?", [$fcm_id])
            ->where('business_id', $business_id)
            ->select('id_staff', 'staff_fullname')
            ->first();

        if (!$staff) {
            return $this->formatter->formatResponse("false", "This Staff ID is not allowed to use this cellphone", "0");
        }

        // Insert into leave application
        DB::table('leave_applications')->insert([
            'staff_id' => $staff->id_staff,
            'staff_name' => $staff->staff_fullname,
            'required_from' => $required_from,
            'required_to' => $required_to,
            'leave_type' => $leave_type,
            'comments' => $leave_type . " - " . $comment,
            'created_by' => $staff->staff_fullname,
            'business_id' => $business_id,
        ]);

        return $this->formatter->formatResponse("true", "Leave Application Forwarded Succesfully", "staff");
    }

    /**
     * Create loan application - replicates legacy loan_application() method
     */
    private function createLoanApplication($staff_id, $fcm_id, $amount, $num_of_months, $comment)
    {
        $staff = DB::table('staff')
            ->where('id_staff', $staff_id)
            ->whereRaw("IFNULL(fcm_id, '') = ?", [$fcm_id])
            ->select('id_staff', 'staff_fullname', 'business_id')
            ->first();

        if (!$staff) {
            return $this->formatter->formatResponse("false", "This Staff ID is not allowed to use this cellphone", "0");
        }

        $installment = round($amount / $num_of_months, 2);

        // Insert into loan application
        DB::table('loan_applications')->insert([
            'staff_id' => $staff->id_staff,
            'staff_name' => $staff->staff_fullname,
            'required_loan_amount' => $amount,
            'num_of_months' => $num_of_months,
            'installment' => $installment,
            'comments' => $comment,
            'created_by' => $staff->staff_fullname,
            'business_id' => $staff->business_id,
        ]);

        return $this->formatter->formatResponse("true", "Application For Loan of " . $amount . " Forwarded Succesfully", "staff");
    }

    /**
     * Get leave application report - replicates legacy leave_application_report() method
     */
    private function getLeaveApplicationReport($staffid)
    {
        $query = DB::table('leave_applications')
            ->where('staff_id', $staffid)
            ->select(
                'id_leave_applications as id',
                'leave_type',
                'required_from as from',
                'required_to as to',
                'application_status',
                'approved_on'
            )
            ->orderBy('created_on', 'desc')
            ->get();

        if (count($query) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $query->toArray());
            return $this->formatter->formatResponse("true", "Leave Applications", "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "No Records Found for this Staff");
        }
    }

    /**
     * Get loan application report - replicates legacy loan_application_report() method
     */
    private function getLoanApplicationReport($staffid)
    {
        $query = DB::table('loan_applications')
            ->where('staff_id', $staffid)
            ->select(
                'id_loan_applications as id',
                'required_loan_amount as amount',
                'num_of_months',
                'installment',
                'application_status',
                'approved_on'
            )
            ->orderBy('created_on', 'desc')
            ->get();

        if (count($query) > 0) {
            $row = array_map(function ($item) {
                return (array) $item;
            }, $query->toArray());
            return $this->formatter->formatResponse("true", "Loan Applications", "staff", $row);
        } else {
            return $this->formatter->formatResponse("false", "No Records Found for this Staff");
        }
    }

    /**
     * Generate random string - replicates legacy getRandomString() method
     */
    private function getRandomString($length = 8)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $string = '';

        for ($i = 0; $i < $length; $i++) {
            $string .= $characters[mt_rand(0, strlen($characters) - 1)];
        }

        return $string;
    }
}
