<?php

namespace App\Listeners;

use App\Models\User;
use App\Models\UserRole;
use App\Services\RoleSyncService;
use Illuminate\Support\Facades\Log;

class SyncRoleChanges
{
    protected $syncService;

    /**
     * Create the event listener.
     */
    public function __construct(RoleSyncService $syncService)
    {
        $this->syncService = $syncService;
    }

    /**
     * Handle user created event - sync roles if user has roles
     */
    public function handleUserCreated(User $user): void
    {
        if ($user->relationLoaded('roles') || $user->roles()->exists()) {
            $this->syncService->syncUserRolesToSpatie($user);
        }
    }

    /**
     * Handle user updated event - sync roles if roles relationship changed
     */
    public function handleUserUpdated(User $user): void
    {
        // Only sync if roles are loaded or if we detect role changes
        // This is a lightweight check - full sync happens on login
        if ($user->wasChanged() && $user->relationLoaded('roles')) {
            $this->syncService->syncUserRolesToSpatie($user);
        }
    }

    /**
     * Handle UserRole pivot created event
     */
    public function handleUserRoleCreated(UserRole $userRole): void
    {
        $user = User::find($userRole->user_id);
        if ($user) {
            $this->syncService->syncUserRolesToSpatie($user);
        }
    }

    /**
     * Handle UserRole pivot deleted event
     */
    public function handleUserRoleDeleted(UserRole $userRole): void
    {
        $user = User::find($userRole->user_id);
        if ($user) {
            $this->syncService->syncUserRolesToSpatie($user);
        }
    }
}
