<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LeaveApplication extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'leave_applications';
    protected $primaryKey = 'id_leave_applications';

    protected $fillable = [
        'staff_id',
        'business_id',
        'staff_name',
        'required_from',
        'required_to',
        'leave_type',
        'leave_type_id',
        'application_status',
        'approved_by',
        'approved_on',
        'comments',
        'created_by',
        'created_on',
    ];

    protected $casts = [
        'required_from' => 'date',
        'required_to' => 'date',
        'approved_on' => 'datetime',
        'created_on' => 'datetime',
    ];

    // Relationships
    public function staff()
    {
        return $this->belongsTo(Staff::class, 'staff_id', 'id_staff');
    }

    public function business()
    {
        return $this->belongsTo(Business::class, 'business_id', 'id_business');
    }

    public function leaveType()
    {
        return $this->belongsTo(LeaveType::class, 'leave_type_id', 'id_leave_type');
    }

    public function staffLeaves()
    {
        return $this->hasMany(StaffLeave::class, 'staff_id', 'staff_id')->where('leave_type', $this->leave_type);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('application_status', 'Pending');
    }

    public function scopeApproved($query)
    {
        return $query->where('application_status', 'Approved');
    }

    public function scopeRejected($query)
    {
        return $query->where('application_status', 'Rejected');
    }

    public function scopeByBusiness($query, $businessId)
    {
        return $query->where('business_id', $businessId);
    }

    public function scopeByStaff($query, $staffId)
    {
        return $query->where('staff_id', $staffId);
    }

    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->where(function($q) use ($startDate, $endDate) {
            $q->whereBetween('required_from', [$startDate, $endDate])
              ->orWhereBetween('required_to', [$startDate, $endDate])
              ->orWhere(function($query) use ($startDate, $endDate) {
                  $query->where('required_from', '<=', $startDate)
                        ->where('required_to', '>=', $endDate);
              });
        });
    }
}