<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class LoanAdvSalary extends Model
{
    use HasFactory;

    protected $table = 'loan_applications';
    protected $primaryKey = 'id_loan_applications';
    public $timestamps = false;
    protected $fillable = [
        'staff_id',
        'staff_name',
        'required_loan_amount',
        'approved_amount',
        'application_status',
        'created_by',
        'approved_by',
        'approved_on',
        'rejected_by',
        'rejected_on',
        'comments',
        'num_of_months',
        'installment',
        'voucher_id',
        'business_id'
    ];

    protected $dates = [
        'created_on',
        'approved_on',
        'rejected_on'
    ];

    // Relationships
    public function staff()
    {
        return $this->belongsTo(Staff::class, 'staff_id', 'id_staff');
    }

    public function business()
    {
        return $this->belongsTo(Business::class, 'business_id', 'id_business');
    }

    public function voucher()
    {
        return $this->belongsTo(AccountVoucher::class, 'voucher_id', 'id_account_voucher');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('application_status', 'Pending');
    }

    public function scopeApproved($query)
    {
        return $query->where('application_status', 'Approved');
    }

    public function scopeRejected($query)
    {
        return $query->where('application_status', 'Rejected');
    }

    public function scopeDisbursed($query)
    {
        return $query->where('application_status', 'Disbursed');
    }

    public function scopeForBusiness($query, $businessId)
    {
        return $query->where('business_id', $businessId);
    }

    // Calculate monthly installment
    public function getMonthlyInstallmentAttribute()
    {
        if ($this->num_of_months > 0) {
            $amount = $this->approved_amount ?? $this->required_loan_amount;
            return round($amount / $this->num_of_months, 2);
        }
        return 0;
    }

    // Check if approved
    public function isApproved()
    {
        return $this->application_status == 'Approved' || $this->application_status == 'Disbursed';
    }

    // Format date for display
    public function getFormattedCreatedOnAttribute()
    {
        return $this->created_on ? date('d-m-Y', strtotime($this->created_on)) : '';
    }

    public function getFormattedApprovedOnAttribute()
    {
        return $this->approved_on ? date('d-m-Y H:i:s', strtotime($this->approved_on)) : '';
    }
}