<?php

namespace App\Services;

use App\Models\Menu;
use Exception;
use Illuminate\Support\Facades\Route;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\ValidationException;

class MenuService
{

    public function __construct()
    {
        
    }

    public function getMenus($arg=[])
    {
        $is_active = $arg['is_active'] ?? null;

        $menus =  Menu::query();
        
        if($is_active !== null){
                $menus->where('is_active', $is_active ? '1' : '0');
                $menus->with($is_active ? 'childsActive' : 'childsInactive');
        }else{
            $menus->with('childs');
        }

        $menus->where(function($q){
            $q->whereNull('parent_id');
            $q->orWhere('parent_id', '0');
        });

        return $menus->orderBy('sort_order', 'ASC')->get(); 
    }

    public function createMenu(Request $request, $parent_id = '')
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'permission' => 'nullable|string',
                'route' => 'nullable|string',
                'link_target' => 'nullable|in:_self,_blank',
                'route_params' => 'nullable|string',
                'icon' => 'nullable|string',
                'child_icon' => 'nullable|string',
                'child_icon_position' => 'nullable|in:start,end',
            ]);

            $validator->after(function ($validator) use ($request) {
                // Route existence check
                if ($request->route && !Route::has($request->route)) {
                    $validator->errors()->add(
                        'route',
                        'Given route does not exist'
                    );
                }

                $rawParams = trim($request->route_params ?? '');
                if (!empty($rawParams)) {
                    $parsedParams = $this->parseRouteParams($rawParams);

                    if ($parsedParams === false) {
                        $validator->errors()->add(
                            'route_params',
                            'Invalid format for route parameters. Use "key:value,key2:value2" or "param1,param2".'
                        );
                    }
                }
            });

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $routeParams = $this->parseRouteParams($request->route_params ?? '');

            $menu = Menu::create([
                'parent_id' => $parent_id,
                'name' => $request->name,
                'permission' => $request->permission,
                'route' => $request->route,
                'link_target' => $request->link_target,
                'route_params' => !empty($routeParams) ? json_encode($routeParams) : null,
                'icon' => $request->icon,
                'child_icon' => $request->child_icon,
                'child_icon_position' => $request->child_icon_position,
                'before_separator' => $request->boolean('before_separator') ? '1' : '0',
                'after_separator' => $request->boolean('after_separator') ? '1' : '0',
                'sort_order' => $this->getMenuSortOrder($parent_id),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Menu created successfully',
                'menu' => $menu,
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->errors(),
            ], 422);

        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function updateMenu(Request $request, int $id)
    {
        try {

            $menu = Menu::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'permission' => 'nullable|string',
                'route' => 'nullable|string',
                'link_target' => 'nullable|in:_self,_blank',
                'route_params' => 'nullable|string',
                'icon' => 'nullable|string',
                'child_icon' => 'nullable|string',
                'child_icon_position' => 'nullable|in:start,end',
            ]);

            $validator->after(function ($validator) use ($request) {
                if ($request->route && !Route::has($request->route)) {
                    $validator->errors()->add(
                        'route',
                        'Given route does not exist'
                    );
                }

                $rawParams = trim($request->route_params ?? '');
                if (!empty($rawParams)) {
                    $parsedParams = $this->parseRouteParams($rawParams);

                    // If parsing fails, add error
                    if ($parsedParams === false) {
                        $validator->errors()->add(
                            'route_params',
                            'Invalid format for route parameters. Use "key:value,key2:value2" or "param1,param2".'
                        );
                    }
                }
            });

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
            $routeParams = $this->parseRouteParams($request->route_params ?? '');
            $menu->update([
                'name' => $request->name,
                'permission' => $request->permission,
                'route' => $request->route,
                'link_target' => $request->link_target,
                'route_params' => !empty($routeParams) ? json_encode($routeParams) : null,
                'icon' => $request->icon,
                'child_icon' => $request->child_icon,
                'child_icon_position' => $request->child_icon_position,
                'before_separator' => $request->boolean('before_separator') ? '1' : '0',
                'after_separator' => $request->boolean('after_separator') ? '1' : '0',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Menu updated successfully',
            ]);

        } catch (ValidationException $e) {

            return response()->json([
                'success' => false,
                'message' => $e->errors(),
            ], 422);

        } catch (Exception $e) {

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    function updateMenuStatus(Request $request, int $id)
    {
        try {

            $menu = Menu::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'is_active' => 'required|in:0,1',
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $menu->update([
                'is_active' => $request->is_active,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Menu status updated successfully',
            ]);

        } catch (ValidationException $e) {

            return response()->json([
                'success' => false,
                'message' => $e->errors(),
            ], 422);

        } catch (Exception $e) {

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

        
    /**
     * Parse route params from user input
     *
     * Accepts:
     *   - Simple array: param1,param2
     *   - Key-value: key1:value1,key2:value2
     *
     * Returns:
     *   - Array if valid
     *   - false if invalid format
     */
    private function parseRouteParams(string $raw)
    {
        $raw = str_replace(' ', '', $raw); // remove spaces
        if (empty($raw)) return [];

        $json = json_decode($raw, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $json;
        }

        $params = [];
        $parts = explode(',', $raw);

        foreach ($parts as $part) {
            if (empty($part)) continue;

            if (str_contains($part, ':')) {
                // Key-value pair
                [$key, $value] = explode(':', $part, 2);
                if ($key === '' || $value === '') return false; // invalid
                $params[$key] = $value;
            } else {
                // Simple array
                $params[] = $part;
            }
        }

        return $params;
    }

    public function getMenuSortOrder($parent_id = '')
    {
        if ($parent_id) {
            $maxSort = Menu::where('parent_id', $parent_id)->max('sort_order');
        } else {
            $maxSort = Menu::whereNull('parent_id')->max('sort_order');
        }
        return $maxSort ? $maxSort + 1 : 1;
    }

    /**
     * Extract searchable menu items from the menu structure
     * Only includes items that are visible to the current user (permission-aware)
     * Only includes items with actual routes
     * 
     * @param \Illuminate\Database\Eloquent\Collection $menus
     * @return array Array of searchable items with name, url, and group
     */
    public function getSearchableMenuItems($menus)
    {
        $items = [];
        
        foreach ($menus as $menu) {
            // Check if user has permission for this menu
            // If permission is empty, allow it (no permission required)
            if (!empty($menu->permission)) {
                try {
                    if (!Gate::allows($menu->permission)) {
                        continue;
                    }
                } catch (\Exception $e) {
                    // If permission check fails (e.g., permission doesn't exist), skip this menu
                    continue;
                }
            }
            
            // If menu has a route, add it to searchable items
            if (!empty($menu->route)) {
                $params = $menu->route_params ? json_decode($menu->route_params, true) : [];
                try {
                    $url = route($menu->route, $params);
                    
                    $items[] = [
                        'name' => $menu->name,
                        'url' => $url,
                        'group' => '', // Top-level menus have no group
                    ];
                } catch (\Exception $e) {
                    // If route doesn't exist, skip this menu item
                    continue;
                }
            }
            
            // Recursively process children
            if ($menu->childsActive && $menu->childsActive->isNotEmpty()) {
                $childItems = $this->getSearchableMenuItemsRecursive($menu->childsActive, $menu->name);
                $items = array_merge($items, $childItems);
            }
        }
        
        return $items;
    }
    
    /**
     * Recursively extract searchable items from child menus
     * 
     * @param \Illuminate\Database\Eloquent\Collection $childMenus
     * @param string $parentGroup The parent menu name for grouping
     * @return array
     */
    private function getSearchableMenuItemsRecursive($childMenus, $parentGroup = '')
    {
        $items = [];
        
        foreach ($childMenus as $child) {
            // Check if user has permission for this child menu
            // If permission is empty, allow it (no permission required)
            if (!empty($child->permission)) {
                try {
                    if (!Gate::allows($child->permission)) {
                        continue;
                    }
                } catch (\Exception $e) {
                    // If permission check fails, skip this menu
                    continue;
                }
            }
            
            // If child has a route, add it to searchable items
            if (!empty($child->route)) {
                $params = $child->route_params ? json_decode($child->route_params, true) : [];
                try {
                    $url = route($child->route, $params);
                    
                    // Use parent group for this item
                    $items[] = [
                        'name' => $child->name,
                        'url' => $url,
                        'group' => $parentGroup,
                    ];
                } catch (\Exception $e) {
                    // If route doesn't exist, skip this menu item
                    continue;
                }
            }
            
            // Recursively process sub-children
            if ($child->childsActive && $child->childsActive->isNotEmpty()) {
                // For sub-children, build group as "Parent > Child"
                $subParentGroup = $parentGroup;
                if (!empty($subParentGroup)) {
                    $subParentGroup .= ' > ' . $child->name;
                } else {
                    $subParentGroup = $child->name;
                }
                
                $subChildItems = $this->getSearchableMenuItemsRecursive($child->childsActive, $subParentGroup);
                $items = array_merge($items, $subChildItems);
            }
        }
        
        return $items;
    }

}
