<?php

namespace App\Services;

/**
 * ResponseFormatter - Replicates the exact printResponse() format from legacy CodeIgniter API
 * 
 * This service ensures byte-for-byte compatible JSON responses with the legacy API.
 * The format wraps all values in arrays with type information: [value, "string"]
 */
class ResponseFormatter
{
    /**
     * Format a response matching the legacy printResponse() method
     * 
     * @param string $status Status value (e.g., "true", "false")
     * @param string $message Message string
     * @param string|null $key Optional key for data (e.g., "staff", "performance")
     * @param mixed $data Optional data to include
     * @return array Formatted response array
     */
    public function formatResponse($status, $message, $key = null, $data = null)
    {
        if ($key && $data) {
            if (is_array($data)) {
                $arr = [];
                foreach ($data as $k => $r) {
                    $arr[] = $this->getFormat($r);
                }
                $r = [
                    [
                        "status" => [$status, "string"],
                        "message" => [$message, "string"],
                        $key => [$arr, 'array']
                    ],
                    'struct'
                ];
            } else {
                $r = [
                    [
                        "status" => [$status, "string"],
                        "message" => [$message, "string"],
                        $key => $this->getFormat($data)
                    ],
                    'struct'
                ];
            }
        } else {
            $r = [
                [
                    "status" => [$status, "string"],
                    "message" => [$message, "string"]
                ],
                'struct'
            ];
        }
        return $r;
    }

    /**
     * Format data values matching the legacy getFormat() method
     * 
     * Wraps each non-null, non-array value in [value, "string"] format
     * 
     * @param array $data Data array to format
     * @return array Formatted data array
     */
    public function getFormat($data)
    {
        $arr = [];
        foreach ($data as $key => $val) {
            if (!is_array($val)) {
                if (strcmp(gettype($val), "NULL") != 0) {
                    $arr[$key] = [$val, "string"];
                }
            }
        }
        return [$arr, 'struct'];
    }

    /**
     * Return a simple error string (for endpoints that return plain strings)
     * 
     * Some legacy endpoints return plain strings like "Invalid Access"
     * This method ensures proper JSON encoding of these strings
     * 
     * @param string $message Error message
     * @return string JSON-encoded string
     */
    public function formatSimpleError($message)
    {
        // Return as plain string - JSON encode will handle it
        return $message;
    }
}
