<?php

namespace App\Services;

use App\Models\User;
use App\Models\Role as LegacyRole;
use Spatie\Permission\Models\Role as SpatieRole;
use Illuminate\Support\Facades\DB;

class RoleSyncService
{
    /**
     * Temporarily switch cache to array driver to avoid database cache table issues
     */
    protected function useArrayCache(): void
    {
        // Temporarily use array cache which doesn't require a database table
        config(['permission.cache.store' => 'array']);
        config(['cache.default' => 'array']);
    }

    /**
     * Sync a legacy role to Spatie's roles table
     * 
     * @param LegacyRole $legacyRole
     * @return SpatieRole
     */
    public function syncRoleToSpatie(LegacyRole $legacyRole): SpatieRole
    {
        // Use array cache to avoid database cache table issues
        $this->useArrayCache();
        
        try {
            // Find or create the role in Spatie's table
            $spatieRole = SpatieRole::firstOrCreate(
                [
                    'name' => $legacyRole->name,
                    'guard_name' => 'web',
                ],
                [
                    'name' => $legacyRole->name,
                    'guard_name' => 'web',
                ]
            );

            return $spatieRole;
        } catch (\Exception $e) {
            // If still fails, log and re-throw
            \Log::warning('Failed to sync role to Spatie: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Sync a user's roles from legacy system to Spatie
     * 
     * @param User $user
     * @return void
     */
    public function syncUserRolesToSpatie(User $user): void
    {
        // Use array cache to avoid database cache table issues
        $this->useArrayCache();
        
        // Get all legacy roles for this user
        $legacyRoles = $user->roles;

        // Collect Spatie role names
        $spatieRoleNames = [];

        foreach ($legacyRoles as $legacyRole) {
            if ($legacyRole->name) {
                // Sync the role to Spatie first
                $spatieRole = $this->syncRoleToSpatie($legacyRole);
                $spatieRoleNames[] = $spatieRole->name;
            }
        }

        // Sync roles to user (this will replace existing Spatie roles)
        try {
            if (!empty($spatieRoleNames)) {
                $user->syncRoles($spatieRoleNames);
            } else {
                // If user has no legacy roles, remove all Spatie roles
                $user->syncRoles([]);
            }
        } catch (\Exception $e) {
            \Log::warning('Failed to sync user roles to Spatie: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Sync all roles from legacy system to Spatie
     * 
     * @return int Number of roles synced
     */
    public function syncAllRoles(): int
    {
        $legacyRoles = LegacyRole::all();
        $syncedCount = 0;

        foreach ($legacyRoles as $legacyRole) {
            if ($legacyRole->name) {
                $this->syncRoleToSpatie($legacyRole);
                $syncedCount++;
            }
        }

        return $syncedCount;
    }

    /**
     * Sync all users' roles from legacy system to Spatie
     * 
     * @return int Number of users synced
     */
    public function syncAllUsers(): int
    {
        $users = User::with('roles')->get();
        $syncedCount = 0;

        foreach ($users as $user) {
            $this->syncUserRolesToSpatie($user);
            $syncedCount++;
        }

        return $syncedCount;
    }
}
