<?php

namespace App\Services;
use App\Models\SMSCred;
use App\Models\SMSLog;
use App\Models\Invoice;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class SMSService
{
    

    public function __construct()
    {
        
    }


    /**
     * Fetch credentials for generic SMS (not OTP or Receipts)
     */
    private function getSmsCred($business_id = 1)
    {
        return SmsCred::where('status', 'valid')
            ->where('business_id', $business_id)
            ->where('channel', 'not like', '%OTP%')
            ->where('channel', 'not like', '%Receipt%')
            ->get()
            ->toArray();
    }

    /**
     * Fetch credentials for OTP SMS
     */
    private function getOtpCred($business_id = 1)
    {
        return SmsCred::where('status', 'valid')
            ->where('business_id', $business_id)
            ->where('channel', 'like', '%OTP%')
            ->get()
            ->toArray();
    }

    /**
     * Fetch credentials for Receipt SMS
     */
    private function getReceiptsCred($business_id = 1)
    {
        return SmsCred::where('status', 'valid')
            ->where('business_id', $business_id)
            ->where('channel', 'like', '%Receipt%')
            ->get()
            ->toArray();
    }

    /**
     * Send SMS
     *
     * @param string $msg
     * @param string $phone
     * @param string $using
     * @param int $business_id
     * @param int|null $invoice_id
     * @param bool $debug
     * @param bool $echo
     * @return mixed
     * @throws Exception
     */
    public function sendSMS($msg, $phone, $using, $business_id, $invoice_id = null, $debug = false, $echo = false)
    {
        if ($msg === '' || $phone === '' ) {
            throw new Exception('Invalid phone or message!');
        }

        // Credentials
        if ($using === 'LoyaltyOTP') {
            $sms_cred = $this->getOTPCred($business_id);
        } elseif (in_array($using, ['ReceiptsSMS', 'ReceiptsSMSretail'])) {
            $sms_cred = $this->getReceiptsCred($business_id);
        } else {
            $sms_cred = $this->getSmsCred($business_id);
        }

        if (!$sms_cred) {
            Log::error('No SMS credentials found.');
        }



        // --- Handle Receipts SMS ---
        if ($using === "ReceiptsSMS") {
            $data = $this->receiptsjson_service($invoice_id, $sms_cred, $msg, $business_id);
            $urltouse = $sms_cred[0]['domain'] . "insert-receipt";
           // Log::debug("Sending receipt SMS for url: $urltouse with data: $data");

            return $this->postJsonCurl($urltouse, $data, $sms_cred[0]['password'], $invoice_id, $debug);
        }

        if ($using === "ReceiptsSMSretail") {
            $data = $this->receiptsjson_sale($invoice_id, $sms_cred, $msg, $business_id);
            $urltouse = $sms_cred[0]['domain'] . "insert-receipt";

            return $this->postJsonCurl($urltouse, $data, $sms_cred[0]['password'], $invoice_id, $debug);
        }

        // --- Handle domain-based providers ---
        switch ($sms_cred[0]['domain']) {
            case 'https://secure.h3techs.com/sms/api/send':
                return $this->sendViaH3($sms_cred, $msg, $phone, $using, $debug, $echo, $business_id);

            case 'https://www.tilismtechservices.com/messaging/':
                return $this->sendViaTilism($sms_cred, $msg, $phone, $using, $debug, $echo, $business_id);

            case 'https://bsms.ufone.com/bsms_v8_api/sendapi-0.3.jsp?id=03348970448':
                return $this->sendViaUfone($sms_cred, $msg, $phone, $using, $debug, $echo, $business_id);

            case 'https://bsms.its.com.pk/':
                return $this->sendViaITS($sms_cred, $msg, $phone, $using, $debug, $echo, $business_id);

            default:
                throw new Exception('SMS Service is not setup!');
        }
    }

    /* ---------------- Helper methods ---------------- */

    private function postJsonCurl($url, $data, $secret, $invoice_id, $debug = false)
    {
       // Log::debug("Posting JSON to $url with secret $secret and invoice ID $invoice_id");
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'secret-key: ' . $secret
            ],
        ]);

        $response = curl_exec($curl);
       // Log::debug("cURL Response: " . $response);
        if (curl_errno($curl)) {
            Log::error('cURL Error: ' . curl_error($curl));
        }

        curl_close($curl);

        if ($debug) {
            Log::debug("Request: $url");
            Log::debug("Response: " . $response);
        }

        return $this->updateslipcode($invoice_id, $response);
    }

    private function sendViaH3($cred, $msg, $phone, $using, $debug, $echo, $business_id)
    {
        $email = $cred[0]['username'];
        $key   = $cred[0]['password'];
        $mask  = urlencode($cred[0]['mask']);
        $to    = urlencode($phone);
        $message = str_replace('%5Cn','%0A',urlencode($msg));

        $data = "email={$email}&key={$key}&mask={$mask}&to={$to}&message={$message}";

        $ch = curl_init('https://secure.h3techs.com/sms/api/send');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($ch);
        curl_close($ch);

        if ($debug) Log::debug($result);

        return $this->update_smslog_simple($result, $phone, $using, $msg, $echo, $business_id);
    }

    private function sendViaTilism($cred, $msg, $phone, $using, $debug, $echo, $business_id)
    {
        $url = 'https://www.tilismtechservices.com/messaging/url/msgurl.aspx?'
             . 'username=' . urlencode($cred[0]['username'])
             . '&password=' . urlencode($cred[0]['password'])
             . '&mask=' . urlencode($cred[0]['mask'])
             . '&mobileno=' . urlencode($phone)
             . '&msgid=123'
             . '&message=' . str_replace('%5Cn','%0A',urlencode($msg))
             . '&channel=' . urlencode($cred[0]['channel']);

        $response = file_get_contents($url);

        if ($debug) Log::debug($response);

        return $this->update_smslog_simple($response, $phone, $using, $msg, $echo, $business_id);
    }

    private function sendViaUfone($cred, $msg, $phone, $using, $debug, $echo, $business_id)
    {
        $url = "https://sms.skedwise.xyz/sendsms.php?"
             . 'message=' . str_replace('%5Cn','%0A',urlencode($msg))
             . '&shortcode=' . urlencode($cred[0]['mask'])
             . '&lang=English'
             . '&mobilenum=' . $phone
             . '&password=' . urlencode($cred[0]['password']);

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        $response = curl_exec($ch);
        curl_close($ch);

        if ($debug) Log::debug($response);

        return $this->update_smslog_simple($response, $phone, $using, $msg, $echo, $business_id);
    }

    private function sendViaITS($cred, $msg, $phone, $using, $debug, $echo, $business_id)
    {
        $url = "https://bsms.its.com.pk/api.php?key=" . urlencode($cred[0]['password'])
             . '&receiver=' . urlencode($phone)
             . '&sender=' . urlencode($cred[0]['mask'])
             . '&msgdata=' . str_replace('%5Cn','%0A',urlencode($msg))
             . '&channel=' . urlencode($cred[0]['channel']);

        $response = file_get_contents($url);

        if ($debug) Log::debug($response);

        return $this->update_smslog_simple($response, $phone, $using, $msg, $echo, $business_id);
    }

    /* ---------------- Placeholder for original methods ---------------- */
    private function receiptsjson_service($invoice_id, $sms_cred, $loyaltypoints, $business_id) { 
        //Log::debug("Generating receipt JSON for invoice ID: $invoice_id");
            // Step 1 — Tax Info
            $business = DB::table('business')
                ->select(
                    'tax_mode',
                    DB::raw("CASE WHEN srb_fiscal_service = 'Yes' THEN 'SRB' ELSE 'PRA' END as tax_org")
                )
                ->where('id_business', $business_id)
                ->first();

            $taxmode = $business->tax_mode ?? null;
            $taxorg  = $business->tax_org ?? null;

            // Step 2 — Business branch info from SMS credentials
            $businessbranch = explode("|", $sms_cred[0]['mask']);

            // Step 3 — Fetch invoice with joins
            $invoice_data = DB::table('invoice')
                ->select(
                    'invoice.*',
                    'invoice_details.*',
                    'invoice_staff.staff_id',
                    'invoice_staff.staff_name',
                    DB::raw("invoice.payment_mode as mode"),
                    'business.id_business',
                    'business.business_name',
                    'business.business_address',
                    DB::raw("SUBSTRING(business_address, position('BR>' IN business_address)+3, length(business_address)) as business_taxcode"),
                    'business.business_phone1',
                    'customers.customer_email',
                    DB::raw("CASE WHEN instrument_number like 'Online%' THEN 'Online' ELSE payment_mode END as payment_mode"),
                    DB::raw("invoice.discount as invoice_discount"),
                    DB::raw("invoice_details.invoice_discount as invoice_discount_item"),
                    DB::raw("round(100*taxes/ invoice_details.discounted_price ,2) as beforerate")
                )
                ->join('customers', 'customers.id_customers', '=', 'invoice.customer_id')
                ->join('business', 'invoice.business_id', '=', 'business.id_business')
                ->join('invoice_details', 'invoice_details.invoice_id', '=', 'invoice.id_invoice')
                ->join('invoice_staff', 'invoice_staff.invoice_detail_id', '=', 'invoice_details.id_invoice_details')
                ->where('id_invoice', $invoice_id)
                ->groupBy('id_invoice_details')
                ->get();

            if ($invoice_data->isEmpty()) {
                return json_encode([]);
            }

            $invoice_data = $invoice_data->toArray();

            // Step 4 — Build invoice object
            $invoice = new \stdClass();
            $first   = $invoice_data[0];

            $invoice->inv_id          = $first->id_invoice;
            $invoice->invoice_number  = $first->invoice_number;
            $invoice->date            = $first->invoice_date;
            $invoice->userId          = $sms_cred[0]['username'];
            $invoice->business_id     = $businessbranch[0];
            $invoice->branch_id       = $businessbranch[1];
            $invoice->taxation_code   = $first->business_taxcode;
            $invoice->ims_code        = $first->imsfiscal_invoice_number;

            if (!empty($first->imsfiscal_invoice_number)) {
                if ($taxorg === "SRB") {
                    $invoice->tax_org      = "SRB";
                    $invoice->tax_org_link = "https://pos.srb.gos.pk/InvoiceVerification/InvoiceStatus.jsp?invoiceVerification=" . $first->imsfiscal_invoice_number;
                } else {
                    $invoice->tax_org      = "PRA";
                    $invoice->tax_org_link = "https://e.pra.punjab.gov.pk/IMSFiscalReport/SearchPOSInvoice_Report.aspx?PRAInvNo=" . $first->imsfiscal_invoice_number;
                }
            } else {
                $invoice->tax_org      = "";
                $invoice->tax_org_link = "";
            }

            $invoice->customer_name   = $first->customer_name;
            $invoice->customer_cell   = $first->customer_cell;
            $invoice->customer_cnic   = '';
            $invoice->customer_email  = $first->customer_email;

            $invoice->invoice_status   = "Active";
            $invoice->payment_mode     = $first->mode;
            $invoice->instrument_number = $first->instrument_number;
            $invoice->cashier          = $first->created_by;

            $invoice->subtotal        = $first->sub_total;
            $invoice->other_charges   = $first->other_charges;

            $invoice->points_earned   = $first->loyalty_earned;
            $invoice->points_used     = $first->loyalty_used;
            $invoice->total_points    = $loyaltypoints + $first->loyalty_earned; // ⚠️ here `$msg` is filling the same role as `$loyaltypoints`

            $invoice->discount = 0;

            if ($taxmode === 'after') {
                $invoice->total_before_tax = $first->gross_wo_tax;
            } elseif ($taxmode === 'before') {
                $invoice->total_before_tax = $first->sub_total;
            }

            $invoice->total_tax      = $first->tax_total;
            $invoice->total_after_tax = $first->total_payable;
            $invoice->paid_amount    = $first->paid_amount;
            $invoice->remaining_amount = $first->balance;

            // Step 5 — Line Items
            $invoice->detail = [];
            foreach ($invoice_data as $d) {
                $detail = new \stdClass();

                $detail->inv_detail_id = $d->id_invoice_details;
                $detail->product_id    = $d->service_id;
                $detail->staff_id      = $d->staff_id;
                $detail->staff_name    = $d->staff_name;
                $detail->deal_name     = $d->service_flag === 'packagetype' ? $d->service_type : '';
                $detail->type          = $d->service_type;
                $detail->category      = $d->service_category;
                $detail->product_name  = $d->service_name;
                $detail->barcode       = $d->service_id;

                $detail->unit_price    = $d->price + $d->service_addition;
                $detail->qty           = 1.0;
                $detail->discount_percentage = $d->final_price > 0
                    ? ceil((($d->discount + $d->invoice_discount_item) * 100) / ($d->price + $d->service_addition))
                    : 0;
                $detail->discount_amount = round($d->invoice_discount_item + $d->discount, 2);

                $detail->tax_percentage = 0;
                $detail->tax_amount     = 0;

                $detail->total_price = round($d->final_price, 2);

                $invoice->detail[] = $detail;
            }

            return json_encode($invoice);

     }

    private function receiptsjson_sale($invoice_id, $sms_cred, $loyaltypoints, $business_id) { 

        // Step 1 — Tax Info
        $business = DB::table('business')
            ->select(
                'tax_mode',
                DB::raw("CASE WHEN srb_fiscal_sale = 'Yes' THEN 'SRB' ELSE 'PRA' END as tax_org")
            )
            ->where('id_business', $business_id)
            ->first();

        $taxmode = $business->tax_mode ?? null;
        $taxorg  = $business->tax_org ?? null;

        // Step 2 — Business branch info from SMS credentials
        $businessbranch = explode("|", $sms_cred[0]['mask']);

        // Step 3 — Fetch invoice with relations
        $invoice = \App\Models\Invoice::with([
                'business:id_business,business_name,business_address,business_phone1',
                'customer:id_customers,customer_email,customer_name,customer_cell',
                'products.businessProduct:id_business_products,barcode_products,brand_name,product_name'
            ])
            ->where('id_invoice', $invoice_id)
            ->first();

        if (!$invoice) {
            return json_encode([]);
        }

        // Step 4 — Build invoice object
        $data = new \stdClass();
        $data->inv_id        = $invoice->id_invoice;
        $data->invoice_number = $invoice->invoice_number;
        $data->date          = $invoice->invoice_date;
        $data->userId        = $sms_cred[0]['username'];
        $data->business_id   = $businessbranch[0];
        $data->branch_id     = $businessbranch[1];
        $data->taxation_code = substr($invoice->business->business_address, strpos($invoice->business->business_address, 'BR>') + 3);
        $data->ims_code      = $invoice->imsfiscal_invoice_number;

        if (!empty($invoice->imsfiscal_invoice_number)) {
            if ($taxorg === "SRB") {
                $data->tax_org      = "SRB";
                $data->tax_org_link = "https://pos.srb.gos.pk/InvoiceVerification/InvoiceStatus.jsp?invoiceVerification=" . $invoice->imsfiscal_invoice_number;
            } else {
                $data->tax_org      = "PRA";
                $data->tax_org_link = "https://e.pra.punjab.gov.pk/IMSFiscalReport/SearchPOSInvoice_Report.aspx?PRAInvNo=" . $invoice->imsfiscal_invoice_number;
            }
        } else {
            $data->tax_org      = "";
            $data->tax_org_link = "";
        }

        $data->customer_name   = $invoice->customer->customer_name ?? '';
        $data->customer_cell   = $invoice->customer->customer_cell ?? '';
        $data->customer_cnic   = '';
        $data->customer_email  = $invoice->customer->customer_email ?? '';

        $data->invoice_status   = "Active";
        $data->payment_mode     = str_starts_with($invoice->instrument_number, 'Online') ? 'Online' : $invoice->payment_mode;
        $data->instrument_number = $invoice->instrument_number;
        $data->cashier          = $invoice->created_by;

        $data->subtotal         = $invoice->sub_total;
        $data->other_charges    = $invoice->other_charges;
        $data->points_earned    = $invoice->loyalty_earned;
        $data->points_used      = $invoice->loyalty_used;
        $data->total_points     = $loyaltypoints; // ⚠️ here I assume `$msg` means "loyalty points" from your original function

        if ($taxmode === 'after') {
            $data->total_before_tax = $invoice->gross_wo_tax;
        } elseif ($taxmode === 'before') {
            $data->total_before_tax = $invoice->sub_total;
        }

        $data->total_tax      = $invoice->tax_total;
        $data->total_after_tax = $invoice->total_payable;

        $data->paid_amount    = $invoice->paid_amount;
        $data->remaining_amount = $invoice->balance;

        // Step 5 — Line Items
        $data->detail = [];
        foreach ($invoice->products as $prod) {
            $d = new \stdClass();
            $d->inv_detail_id     = $prod->id_invoice_products;
            $d->product_id        = $prod->product_id;
            $d->staff_id          = $prod->staff_id;
            $d->staff_name        = $prod->staff_name;
            $d->deal_name         = '';

            $d->type              = $prod->businessProduct->brand_name ?? '';
            $d->category          = $prod->businessProduct->brand_name ?? '';
            $d->product_name      = $prod->businessProduct->product_name ?? '';
            $d->barcode           = $prod->businessProduct->barcode_products ?? '';

            $d->unit_price        = $prod->price + $prod->service_addition;
            $d->qty               = 1.0;
            $d->discount_percentage = ceil(($prod->discount * 100) / ($prod->price + $prod->service_addition));
            $d->discount_amount   = round($prod->invoice_discount + $prod->discount, 2);

            $d->tax_percentage    = 0;
            $d->tax_amount        = 0;
            $d->total_price       = round($prod->final_price, 2);

            $data->detail[] = $d;
        }

        return json_encode($data);

    
    }

    private function updateslipcode($invoice_id, $receipt_response) { 
        
        $response = json_decode($receipt_response, true);
        $receipt_url = '';

        if ($invoice_id !== null) {
            $invoice = Invoice::find($invoice_id);

            if (!$invoice) {
                Log::error("Invoice ID $invoice_id not found for updating slip code.");
                return 0; // invoice not found
            }

            foreach ($response as $key => $value) {
                if ($key === 'receipt_code') {
                    $invoice->receipt_code = $value;
                } elseif ($key === 'reciept_url') { // ⚠️ check spelling
                    $invoice->receipt_url = $value;
                    $receipt_url = $value;
                }
            }

            if ($receipt_url !== '') {
                $invoice->save();   // update row
                return $receipt_url;
            } else {
                return $receipt_response; // no URL in response, return original
            }
        }

        return 0;

     }
     
    private function update_smslog_simple($response, $receiver, $using = "SMSReminder", $msg = "", $echo = false, $businessid = 1)
    {
        $today = now()->format('Y-m-d H:i'); // correct datetime format (24h clock)

        $log = new SMSLog();
        $log->errorno     = $response;
        $log->status      = $response;
        $log->msgdata     = $msg;
        $log->using       = $using;
        $log->business_id = $businessid;
        $log->senton      = $today;
        $log->receiver    = $receiver;
        $log->save();

        if ($echo) {
            echo "Message ready for delivery<br/>";
            echo "error = {$response}<br/>";
            echo "{$msg}<br/>";
        }

        return $log->id; // returns inserted row's primary key
    }
}
