<?php

namespace App\Traits;

use Illuminate\Support\Facades\Auth;

trait HasPermissionHelpers
{
    /**
     * Check if the authenticated user has a role (checks both legacy session and Spatie)
     * 
     * @param string|array $roleName Role name(s) to check
     * @return bool
     */
    protected function checkRole($roleName): bool
    {
        $user = Auth::user();
        
        if (!$user) {
            return false;
        }

        // Check legacy session first (for backward compatibility)
        $sessionRole = session('user_role');
        if (is_array($roleName)) {
            if ($sessionRole && in_array($sessionRole, $roleName)) {
                return true;
            }
        } else {
            if ($sessionRole === $roleName) {
                return true;
            }
        }

        // Check Spatie roles
        if (is_array($roleName)) {
            return $user->hasAnyRole($roleName);
        } else {
            return $user->hasRole($roleName);
        }
    }

    /**
     * Check if the authenticated user has any of the given roles
     * 
     * @param array $roles Array of role names
     * @return bool
     */
    protected function hasAnyRole(array $roles): bool
    {
        return $this->checkRole($roles);
    }

    /**
     * Check if the authenticated user has a specific permission (Spatie only)
     * 
     * @param string|array $permission Permission name(s) to check
     * @return bool
     */
    protected function hasPermission($permission): bool
    {
        $user = Auth::user();
        
        if (!$user) {
            return false;
        }

        if (is_array($permission)) {
            return $user->hasAnyPermission($permission);
        } else {
            return $user->hasPermissionTo($permission);
        }
    }

    /**
     * Check if the authenticated user has any of the given permissions
     * 
     * @param array $permissions Array of permission names
     * @return bool
     */
    protected function hasAnyPermission(array $permissions): bool
    {
        return $this->hasPermission($permissions);
    }

    /**
     * Get the current user's role name (from legacy system or Spatie)
     * 
     * @return string|null
     */
    protected function getUserRole(): ?string
    {
        $user = Auth::user();
        
        if (!$user) {
            return null;
        }

        // Try legacy session first
        $sessionRole = session('user_role');
        if ($sessionRole) {
            return $sessionRole;
        }

        // Fall back to Spatie
        $spatieRoles = $user->getRoleNames();
        return $spatieRoles->first();
    }
}
