<?php

use App\Models\Menu;

if (!function_exists('getCurrentMenuBreadcrumb')) {
    /**
     * Get the breadcrumb path for the currently active menu item.
     * 
     * This function determines which menu item is active based on the current route,
     * then walks up the parent hierarchy to build the breadcrumb path.
     * 
     * @return array An ordered array of menu names from root → highlighted item
     *               Example: ['Accounts', 'Reports', 'Accounts Report']
     */
    function getCurrentMenuBreadcrumb(): array
    {
        $request = request();
        $routeName = $request->route()?->getName();
        $segments = $request->segments();
        $currentUrl = $request->url();

        // First, try to find menu by exact route name match
        // Load parent recursively to avoid N+1 queries
        $activeMenu = Menu::where('route', $routeName)
            ->with(['parent' => function($query) {
                $query->with('parent');
            }])
            ->first();

        // If not found, try to match by route_params containing segments
        if (!$activeMenu && !empty($segments)) {
            // Get all menus with route_params and check if segments match
            $menus = Menu::whereNotNull('route_params')
                ->whereNotNull('route')
                ->with('parent')
                ->get();

            foreach ($menus as $menu) {
                $params = json_decode($menu->route_params, true);
                if (is_array($params)) {
                    // Check if segments match the route_params
                    // route_params can be an array like ["accounts", "vouchers"] or ["scheduler"]
                    if ($params === $segments || 
                        (count($params) <= count($segments) && 
                         array_slice($segments, 0, count($params)) === $params)) {
                        // Also verify the generated URL matches
                        try {
                            $menuUrl = route($menu->route, $params);
                            if (str_starts_with($currentUrl, $menuUrl)) {
                                $activeMenu = $menu;
                                break;
                            }
                        } catch (\Exception $e) {
                            // If route generation fails, still use this menu if params match
                            if ($params === $segments) {
                                $activeMenu = $menu;
                                break;
                            }
                        }
                    }
                }
            }
        }

        // If still not found, try to match by URL comparison (like JavaScript does)
        if (!$activeMenu) {
            $menus = \App\Models\Menu::whereNotNull('route')
                ->with('parent')
                ->get();

            foreach ($menus as $menu) {
                try {
                    $params = $menu->route_params ? json_decode($menu->route_params, true) : [];
                    $menuUrl = route($menu->route, $params);
                    // Remove query string and fragment for comparison
                    $menuUrlClean = strtok($menuUrl, '?#');
                    $currentUrlClean = strtok($currentUrl, '?#');
                    
                    if ($menuUrlClean === $currentUrlClean) {
                        $activeMenu = $menu;
                        break;
                    }
                } catch (\Exception $e) {
                    // Skip menus with invalid routes
                    continue;
                }
            }
        }

        // If no menu found, return empty array
        if (!$activeMenu) {
            return [];
        }

        // Build breadcrumb by walking up the parent hierarchy
        $breadcrumb = [];
        $currentMenu = $activeMenu;

        // Walk up the parent chain
        while ($currentMenu) {
            array_unshift($breadcrumb, $currentMenu->name);
            
            // Check if this is a root menu (no parent)
            $parentId = $currentMenu->parent_id;
            if ($parentId === null || $parentId === '0' || $parentId === 0) {
                break;
            }
            
            // Load parent if not already loaded
            if (!$currentMenu->relationLoaded('parent')) {
                $currentMenu->load('parent');
            }
            
            $currentMenu = $currentMenu->parent;
            
            // Safety check - if parent didn't load, stop
            if (!$currentMenu) {
                break;
            }
        }

        return $breadcrumb;
    }
}
