/**
 * Post-build script to fix font paths in compiled CSS
 * 
 * Vite generates absolute font URLs (/assets/... or /build/assets/...) which
 * don't work when the app is deployed in a subdirectory.
 * 
 * Since CSS and fonts are in the same directory, we convert to relative paths
 * (e.g., ./font.woff2) which work in both root and subdirectory deployments.
 */

import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

function fixFontPaths() {
    try {
        const assetsDir = path.join(__dirname, 'public', 'build', 'assets');
        
        if (!fs.existsSync(assetsDir)) {
            console.log('Build directory not found. Run "yarn build" first.');
            return;
        }

        // Find all CSS files
        const files = fs.readdirSync(assetsDir);
        const cssFiles = files.filter(file => file.endsWith('.css'));
        
        if (cssFiles.length === 0) {
            console.log('No CSS files found in public/build/assets/');
            return;
        }

        let totalReplacements = 0;

        cssFiles.forEach(file => {
            const filePath = path.join(assetsDir, file);
            let content = fs.readFileSync(filePath, 'utf8');
            const originalContent = content;
            
            // Replace absolute paths with relative paths
            // Since CSS and fonts are in the same directory, use just the filename
            // Match patterns like:
            // - url(/assets/font.woff2)
            // - url("/build/assets/font.woff2")
            // - url('/assets/font.woff2?v=1.0')
            // - url(/build/assets/font.woff2?v=3.4.0)
            const regex = /url\((['"]?)(\/build\/assets\/|\/assets\/)([^'")]+?)(\?[^'")]*)?(['"]?)\)/g;
            const matches = [...content.matchAll(regex)];
            
            if (matches.length > 0) {
                content = content.replace(regex, (match, quote1, pathPrefix, fontPath, queryParams, quote2) => {
                    // Extract just the filename (handle query params)
                    const fontFile = fontPath.split('/').pop();
                    // Use relative path (same directory) - preserve query params if any
                    const query = queryParams || '';
                    // Use the same quote style, or default to the first quote
                    const finalQuote = quote2 || quote1 || '';
                    return `url(${finalQuote}./${fontFile}${query}${finalQuote})`;
                });
                
                fs.writeFileSync(filePath, content, 'utf8');
                totalReplacements += matches.length;
                console.log(`✓ Fixed ${matches.length} font path(s) in ${file} (using relative paths)`);
            }
        });

        if (totalReplacements > 0) {
            console.log(`\n✅ Fixed ${totalReplacements} total font path(s) in ${cssFiles.length} CSS file(s)`);
        } else {
            console.log('No font paths found to fix (they may already be correct)');
        }
    } catch (error) {
        console.error('Error fixing font paths:', error);
        process.exit(1);
    }
}

fixFontPaths();
