@extends('layouts.horizontal', ['title' => 'Profit and Loss', 'topbarTitle' => 'Profit and Loss']) 
@section('css')
<style>
    .control-account-td { padding-left: 10px !important; }
    .sub-control-account-td {padding-left: 30px !important; }
    .transaction-account-td { padding-left: 50px !important; }
    .sub-control-account-td-total {padding-left: 30px !important; }
    
    .main-total-row { font-weight: bold;  font-size: 1.1em; }
    .main-account { font-weight: bold;  font-size: 1.1em; }
    .gross-profit-row {  font-weight:bold; }
    .net-profit-row {  font-weight:bold; font-size: 17px;}
    .account-head-account{font-size: 13px; font-weight: normal; }
    .sub-control-account{font-size: 14px; font-weight: normal;font-weight:500;}
    .control-account{font-size: 14px;font-weight:500;}
    .period-column { min-width: 120px; text-align: center; }
    .dataTables_empty { display: none !important; }


</style>
@endsection
@section('content')
<div class="profit-loss-container">
    <!-- start page title -->
    <div class="row mb-2">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Profit and Loss</h2>
                 <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                            <li class="breadcrumb-item"><a href="javascript: void(0);">Accounts</a></li>
                            <li class="breadcrumb-item active">Profit and Loss</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
    
    <!-- FILTERS -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <h4 class="card-title mb-4">Filters :</h4>

                    <div class="row g-3">
                        <div class="col-sm-2">
                           
                            <label class="form-label">Business</label>
                            <select class="form-select" id="business">
                                @foreach($businesses as $business)
                                    <option value="{{ $business->id_business }}"
                                        {{ $business->id_business == session('business_id') ? 'selected' : '' }}>
                                        {{ $business->business_name }}
                                    </option>
                                @endforeach
                                 @if (session('ho') =='Yes')
                                 <option value="">All Branches</option>
                                @endif
                            </select>
                        </div>

                        <div class="col-sm-2">
                            <label class="form-label">From Date</label>
                            <input type="date" class="form-control" id="from_date" value="{{ date('Y-m-01') }}">
                        </div>
                        <div class="col-sm-2">
                            <label class="form-label">To Date</label>
                            <input type="date" class="form-control" id="to_date" value="{{ date('Y-m-t') }}">
                        </div>

                        <div class="col-sm-2">
                            <label class="form-label">Period Type</label>
                            <select class="form-select" id="period_type">
                                <option value="" selected>Total Only</option>
                                <option value="day">Day Wise</option>
                                <option value="month" >Month Wise</option>
                                <option value="year">Year Wise</option>
                            </select>
                        </div>

                        <div class="col-sm-4 align-self-end">
                            <button type="button" class="btn btn-outline-success " onclick="loadProfitLoss()" id="profitTable"> Refresh</button>
                            <button type="button" class="btn btn-ghost-primary rounded-circle btn-icon" onclick="printPL()">
                                <iconify-icon icon="ri:printer-fill" class="fs-20"></iconify-icon>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- TABLE -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h4 class="card-title mb-0">Profit and Loss Statement</h4>
                        <div>
                            <button class="btn btn-outline-primary btn-sm" onclick="showLevel('main')">LEVEL-1</button>
                            <button class="btn btn-outline-info btn-sm" onclick="showLevel('control')">LEVEL-2</button>
                            <button class="btn btn-outline-success btn-sm" onclick="showLevel('sub')">LEVEL-3</button>
                            <button class="btn btn-outline-dark btn-sm" onclick="showLevel('head')">LEVEL-4</button>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-centered table-bordered table-nowrap mb-0 table-sm fs-12" id="profitLossTable">
                            <thead class="table-light">
                                <tr class="bg-light">
                                    <th style="min-width: 400px;">Account</th>
                                    <!-- Dynamic headers will be inserted here -->
                                </tr>
                            </thead>
                            <tbody id="profit-loss-body">
                                <!-- Data will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

</div>
@endsection

@section('scripts')
<script>
// Global variable to store current data
let currentProfitLossData = null;

document.addEventListener("DOMContentLoaded", function() { 
    loadProfitLoss();  
});

function formatNumber(num) {
    if (num === null || num === undefined) return '0.00';
    num = parseFloat(num);
    if (isNaN(num)) return '0.00';
    
    if (num < 0) {
        return '(' + Math.abs(num).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",") + ')';
    }
    return Math.abs(num).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

function formatPeriod(period, periodType) {
    if (!period) return '';
    
    switch(periodType) {
        case 'year':
            return period;
        case 'month':
            const [year, month] = period.split('-');
            const monthNames = ["Jan", "Feb", "Mar", "Apr", "May", "Jun",
                                "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
            return `${monthNames[parseInt(month)-1]} ${year}`;
        case 'day':
            return new Date(period).toLocaleDateString('en-US', { 
                day: '2-digit', 
                month: 'short', 
                year: 'numeric' 
            });
        default:
            return period;
    }
}

function loadProfitLoss() { 
    // Destroy existing DataTable if exists
    if ($.fn.DataTable.isDataTable('#profitLossTable')) {
        $('#profitLossTable').DataTable().destroy();
    }
    
    const start_date = document.getElementById("from_date").value;
    const end_date   = document.getElementById("to_date").value;
    const business   = document.getElementById("business").value;
    const period_type = document.getElementById("period_type").value;
    
    $("#profitTable").html(`<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span> Loading...`);
    
    fetch("{{ route('accounts.profit_and_loss') }}" +
          `?start_date=${start_date}&end_date=${end_date}&business_id=${business}&period_type=${period_type}`)
    .then(res => res.json())
    .then(data => {
        $("#profitTable").html(`Refresh`);
        currentProfitLossData = data;
        renderProfitLossTable(data);
    })
    .catch(error => {
        console.error('Error:', error);
        $("#profitTable").html(`Refresh`);
        alert('Error loading profit and loss data');
    });
} 

function renderProfitLossTable(data) {
    const tbody = document.getElementById("profit-loss-body");
    tbody.innerHTML = "";
    
    const period_type = data.period_type;
    const structuredData = data.structured_data || {};
    const all_periods = data.all_periods || [];

    // Update table header
    updateTableHeader(all_periods, period_type);

    // Update business name
    const reportNameEl = document.querySelector('.card-title');
    if(reportNameEl && data.business_name) {
        reportNameEl.textContent = `Profit and Loss Statement - ${data.business_name}`;
    }

    let sectionOrder = ["Revenue", "Cost of Goods", "Expenses"];  

    let totalRevenue = structuredData["Revenue"]?.total || 0;
    let totalCOGS    = structuredData["Cost of Goods"]?.total || 0;
    let totalExpense = structuredData["Expenses"]?.total || 0;

    let grossProfit  = totalRevenue - totalCOGS;
    let netProfit    = grossProfit - totalExpense;

    // Build table rows
    let tableHTML = '';

    sectionOrder.forEach(mainType => {
        if (!structuredData[mainType]) return;
        const main = structuredData[mainType];
        
        // Main Account Row
        tableHTML += createMainAccountRow(mainType, main, all_periods, period_type);
        
        // Controls
        if (main.controls) {
            Object.values(main.controls).forEach(control => {
                if (control.name !== "Cost of Goods") {
                    tableHTML += createControlAccountRow(control.name, all_periods, period_type);
                }
                
                // Sub-controls
                if (control.sub_controls) {
                    Object.values(control.sub_controls).forEach(sub => {
                        tableHTML += createSubControlAccountRow(sub.name, all_periods, period_type);
                        
                        // Account Heads
                        if (sub.heads) {
                            Object.values(sub.heads).forEach(head => {
                                tableHTML += createAccountHeadRow(head, all_periods, period_type);
                            });
                        }
                        
                        // Sub-control Total
                        tableHTML += createSubControlTotalRow(sub, all_periods, period_type);
                    });
                }
                
                // Control Total
                if (control.name !== "Cost of Goods") {
                    tableHTML += createControlTotalRow(control, all_periods, period_type);
                }
            });
        }

        // Main Account Total
        tableHTML += createMainTotalRow(mainType, main, all_periods, period_type);

        // Gross Profit after Cost of Goods
        if(mainType === "Cost of Goods") {
            tableHTML += createGrossProfitRow(grossProfit, all_periods, period_type);
        }
    });
    
    // NET PROFIT
    tableHTML += createNetProfitRow(netProfit, all_periods, period_type);

    tbody.innerHTML = tableHTML;
    
    // Initialize DataTable
    initializeDataTable();
    
    // Show level 4 by default
    showLevel('head');
}

function updateTableHeader(all_periods, period_type) {
    const headerRow = document.querySelector('#profitLossTable thead tr');
    let headerHTML = '<th style="min-width: 400px;">Account</th>';
    
    if (period_type && all_periods.length > 0) {
        all_periods.forEach(period => {
            headerHTML += `<th class="text-center period-column">${formatPeriod(period, period_type)}</th>`;
        });
        headerHTML += '<th class="text-center period-column">Total</th>';
    } else {
        headerHTML += '<th class="text-center">Total</th>';
    }
    
    headerRow.innerHTML = headerHTML;
}

function createMainAccountRow(mainType, main, all_periods, period_type) {
    let html = `<tr class="level-1 level-one main-account">
        <td class="control-account-td "><strong>${mainType.toUpperCase()}</strong></td>`;
    
    // Add empty cells for period columns
    const columnCount = getColumnCount(all_periods, period_type);
    for(let i = 0; i < columnCount; i++) {
        html += `<td class="text-center"></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createControlAccountRow(controlName, all_periods, period_type) {
    let html = `<tr class="level-2 level-two control-account">
        <td class="sub-control-account-td ">${controlName}</td>`;
    
    const columnCount = getColumnCount(all_periods, period_type);
    for(let i = 0; i < columnCount; i++) {
        html += `<td class="text-center"></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createSubControlAccountRow(subControlName, all_periods, period_type) {
    let html = `<tr class="level-2 level-two sub-control-account">
        <td class="sub-control-account-td ">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;${subControlName}</td>`;
    
    const columnCount = getColumnCount(all_periods, period_type);
    for(let i = 0; i < columnCount; i++) {
        html += `<td class="text-center"></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createAccountHeadRow(head, all_periods, period_type) {
    let html = `<tr class="level-3 level-three account-head-account">
        <td class="transaction-account-td">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;— ${head.account_head}</td>`;
    
    if (period_type && all_periods.length > 0) {
        all_periods.forEach(period => {
            const amount = head.period_totals?.[period] || head.net_amount || 0;
            html += `<td class="text-start">${formatNumber(amount)}</td>`;
        });
        // Total column
        const total = head.total || head.net_amount || 0;
        html += `<td class="text-start">${formatNumber(total)}</td>`;
    } else {
        const amount = head.net_amount || 0;
        html += `<td class="text-start">${formatNumber(amount)}</td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createSubControlTotalRow(sub, all_periods, period_type) {
    let html = `<tr class="level-2 level-two total-row sub-control-account sub-control-account-total-row">
        <td class="sub-control-account-td "><strong>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Total ${sub.name}</strong></td>`;
    
    if (period_type && all_periods.length > 0) {
        all_periods.forEach(period => {
            let periodTotal = 0;
            if (sub.heads) {
                Object.values(sub.heads).forEach(head => {
                    periodTotal += head.period_totals?.[period] || head.net_amount || 0;
                });
            }
            html += `<td class="text-center "><strong>${formatNumber(periodTotal)}</strong></td>`;
        });
        html += `<td class="text-center "><strong>${formatNumber(sub.total)}</strong></td>`;
    } else {
        html += `<td class="text-center "><strong>${formatNumber(sub.total)}</strong></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createControlTotalRow(control, all_periods, period_type) {
    let html = `<tr class="level-2 level-two total-row control-account control-account-total-row">
        <td class="sub-control-account-td "><strong>Total ${control.name}</strong></td>`;
    
    if (period_type && all_periods.length > 0) {
        all_periods.forEach(period => {
            let periodTotal = 0;
            if (control.sub_controls) {
                Object.values(control.sub_controls).forEach(sub => {
                    if (sub.heads) {
                        Object.values(sub.heads).forEach(head => {
                            periodTotal += head.period_totals?.[period] || head.net_amount || 0;
                        });
                    }
                });
            }
            html += `<td class="text-center ">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<strong>${formatNumber(periodTotal)}</strong></td>`;
        });
        html += `<td class="text-center ">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<strong>${formatNumber(control.total)}</strong></td>`;
    } else {
        html += `<td class="text-center ">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<strong>${formatNumber(control.total)}</strong></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createMainTotalRow(mainType, main, all_periods, period_type) {
    let html = `<tr class="level-1 level-one main-total-row">
        <td class="control-account-td "><strong >TOTAL ${mainType.toUpperCase()}</strong></td>`;
    
    if (period_type && all_periods.length > 0) {
        all_periods.forEach(period => {
            let periodTotal = 0;
            if (main.controls) {
                Object.values(main.controls).forEach(control => {
                    if (control.sub_controls) {
                        Object.values(control.sub_controls).forEach(sub => {
                            if (sub.heads) {
                                Object.values(sub.heads).forEach(head => {
                                    periodTotal += head.period_totals?.[period] || head.net_amount || 0;
                                });
                            }
                        });
                    }
                });
            }
            html += `<td class="text-end "><strong>${formatNumber(periodTotal)}</strong></td>`;
        });
        html += `<td class="text-end "><strong>${formatNumber(main.total)}</strong></td>`;
    } else {
        html += `<td class="text-end "><strong>${formatNumber(main.total)}</strong></td>`;
    }
    
    html += `</tr>`;
    return html;
}

function createGrossProfitRow(grossProfit, all_periods, period_type) {
    let html = `<tr class="gross-profit-row">
        <td><strong>GROSS PROFIT</strong></td>`;
    
    const columnCount = getColumnCount(all_periods, period_type);
    for(let i = 0; i < columnCount - 1; i++) {
        html += `<td class="text-end"></td>`;
    }
    html += `<td class="text-end"><strong>${formatNumber(grossProfit)}</strong></td>`;
    
    html += `</tr>`;
    return html;
}

function createNetProfitRow(netProfit, all_periods, period_type) {
    let html = `<tr class="net-profit-row">
        <td><strong>NET PROFIT</strong></td>`;
    
    const columnCount = getColumnCount(all_periods, period_type);
    for(let i = 0; i < columnCount - 1; i++) {
        html += `<td class="text-end"></td>`;
    }
    html += `<td class="text-end"><strong>${formatNumber(netProfit)}</strong></td>`;
    
    html += `</tr>`;
    return html;
}

function getColumnCount(all_periods, period_type) {
    if (period_type && all_periods.length > 0) {
        return all_periods.length + 1; // periods + total column
    }
    return 1; // only total column
}

function initializeDataTable() {
    // Destroy existing DataTable if exists
    if ($.fn.DataTable.isDataTable('#profitLossTable')) {
        $('#profitLossTable').DataTable().destroy();
    }


    table = $('#profitLossTable').DataTable({
            destroy: true,
            paging: false,
            searching: false,
            info: false,
            ordering: false,
             "layout": {
                topStart: "buttons",
                topEnd: "search",
                bottomStart: {                 // page length selector
                    pageLength: {
                        menu: [[10, 25, 50, -1], [10, 25, 50, "Show all"]]
                    },
                    info: {}
                },                
              
            },
            "buttons": [
             {
                extend: 'csv',
                className: 'btn btn-sm btn-light',
                text: 'CSV',
                exportOptions: {
                    format: {
                        body: function (data, row, column, node) {
                            // Remove all HTML tags and fix special characters
                            return data
                                .replace(/<[^>]*>/g, '') // Remove HTML tags
                                .replace(/&nbsp;/g, ' ')  // Replace &nbsp; with space
                                .replace(/â€”|—/g, '')   // Replace em dash with regular dash
                                .replace(/\s+/g, ' ')     // Replace multiple spaces with single space
                                .trim();                  // Remove leading/trailing spaces
                        }
                    }
                }
            },
              {
                extend: 'print',
                className: 'btn btn-sm btn-light',
                text: 'Print',
               
            },
           
           
        ]
        });
    
}

function showLevel(level) {
    // Hide all rows first
    
    // Hide all rows first
    document.querySelectorAll(
        ".main-account,  .control-account, .sub-control-account, .account-head-account, .main-total-row , .control-account-total-row , .sub-control-account-total-row , .account-head-account-total-row"
    ).forEach(row => row.style.display = "none");

    if(level === "main") {
        document.querySelectorAll(".main-total-row").forEach(r => r.style.display = "");
    }

    if(level === "control") {
        document.querySelectorAll(".main-account,.main-total-row, .control-account-total-row ").forEach(r => r.style.display = "");
    }

    if(level === "sub") {
        document.querySelectorAll(".main-account,.main-total-row, .control-account , .control-account-total-row, .sub-control-account-total-row ")
            .forEach(r => r.style.display = "");
    }

    if(level === "head") {
        document.querySelectorAll(
            ".main-account,.main-total-row, .control-account , .control-account-total-row, .sub-control-account , .sub-control-account-total-row, .account-head-account"
        ).forEach(r => r.style.display = "");
    }
}

function printPL() {
      const sheet = document.getElementById("profitLossTable");
    if (!sheet) return;

    const businessName = document.getElementById("report-business-name")?.textContent || "";
    const fromDate = document.getElementById("from_date")?.value || "";
    const toDate = document.getElementById("to_date")?.value || "";
    const periodType = document.getElementById("period_type")?.value || "Total Only";

    // Save original page content
    const originalContents = document.body.innerHTML;

    // Create temporary print content
    const tempDiv = document.createElement("div");
    tempDiv.innerHTML = `
        <div style="text-align:center; margin-bottom:20px;">
            <h2 style="margin:0;">${businessName}</h2>
            <h3 style="margin:0;">Profit and Loss Statement</h3>
            <p>Date Range: <b>${fromDate}</b> to <b>${toDate}</b> | Period: <b>${periodType.toUpperCase()}</b></p>
        </div>
        ${sheet.outerHTML}
    `;

    document.body.innerHTML = tempDiv.innerHTML;


    window.print();

   
    document.body.innerHTML = originalContents;
    location.reload(); // Restore JS events
}
</script>
@endsection