@extends('layouts.horizontal', ['title' => 'Branch Targets', 'topbarTitle' => 'Branch Targets'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    /* Prevent body padding adjustment on modal open/close */
    body.modal-open {
        padding-right: 0 !important;
    }
    /* Rowspan styling */
    .group-header {
        vertical-align: middle;
        font-weight: 500;
    }
    .month-row td {
        vertical-align: middle;
    }
    .month-row:not(:first-child) .group-header {
        border-top: none;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Branch Targets</h2>
                <div class="page-title-right">
                    @can('create_branch_targets')
                    <button type="button" class="btn btn-sm btn-outline-success" id="addTargetBtn">
                        <iconify-icon icon="mdi:plus" class="me-1"></iconify-icon> Add Target
                    </button>
                    @endcan
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->

    <!-- Success/Error Messages -->
    <div id="alert-container"></div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="branch_targets_table" class="table table-bordered dt-responsive fs-13 w-100">
                            <thead>
                                <tr class="table-light">
                                    <th class="text-start">Business</th>
                                    <th class="text-start">Target Type</th>
                                    <th class="text-start">Year</th>
                                    <th class="text-start">Month</th>
                                    <th class="text-start">Target Amount</th>
                                    <th class="text-start">Actions</th>
                                </tr>
                                <!-- Filter Row -->
                                <tr class="filter filter-row">
                                    <th>
                                        <input type="text" class="form-control form-filter input-sm" name="business_name" placeholder="Search Business">
                                    </th>
                                    <th>
                                        <select class="form-control form-filter input-sm" name="target_type">
                                            <option value="">All</option>
                                            <option value="Service">Service</option>
                                            <option value="Retail">Retail</option>
                                        </select>
                                    </th>
                                    <th>
                                        <input type="text" class="form-control form-filter input-sm" name="year" placeholder="Search Year">
                                    </th>
                                    <th>
                                        <select class="form-control form-filter input-sm" name="month">
                                            <option value="">All</option>
                                            <option value="1">January</option>
                                            <option value="2">February</option>
                                            <option value="3">March</option>
                                            <option value="4">April</option>
                                            <option value="5">May</option>
                                            <option value="6">June</option>
                                            <option value="7">July</option>
                                            <option value="8">August</option>
                                            <option value="9">September</option>
                                            <option value="10">October</option>
                                            <option value="11">November</option>
                                            <option value="12">December</option>
                                        </select>
                                    </th>
                                    <th></th>
                                    <th>
                                        <div class="filter-buttons">
                                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit">
                                                <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                            </button>
                                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel">
                                                <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                            </button>
                                        </div>
                                    </th>
                                </tr>
                            </thead>
                            <tbody></tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Target Modal -->
<div class="modal fade" id="targetModal" tabindex="-1" aria-labelledby="targetModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="targetModalLabel">Add Branch Targets</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="targetForm">
                    @csrf
                    
                    <!-- Business, Target Type, and Year in the same row -->
                    <div class="row mb-3">
                        <div class="col-md-4 mb-3 mb-md-0">
                            <label for="modal_business_id" class="form-label">Business <span class="text-danger">*</span></label>
                            <select class="form-control" id="modal_business_id" name="business_id" required>
                                <option value="">Select Business</option>
                                @foreach($businesses as $biz)
                                    <option value="{{ $biz->id_business }}"
                                        {{ session('business_id') == $biz->id_business ? 'selected' : '' }}>
                                        {{ $biz->business_name }}
                                    </option>
                                @endforeach
                            </select>
                            <div class="invalid-feedback" id="business_id_error"></div>
                        </div>
                        
                        <div class="col-md-4 mb-3 mb-md-0">
                            <label for="target_type" class="form-label">Target Type <span class="text-danger">*</span></label>
                            <select class="form-control" id="target_type" name="target_type" required>
                                <option value="Service">Service</option>
                                <option value="Retail">Retail</option>
                            </select>
                            <div class="invalid-feedback" id="target_type_error"></div>
                        </div>
                        
                        <div class="col-md-4 mb-3 mb-md-0">
                            <label for="year" class="form-label">Year <span class="text-danger">*</span></label>
                            <select class="form-control" id="year" name="year" required>
                                <option value="">Select Year</option>
                                @php
                                    $currentYear = (int)date('Y');
                                    $startYear = $currentYear - 2;
                                    $endYear = $currentYear + 10;
                                @endphp
                                @for($y = $endYear; $y >= $startYear; $y--)
                                    <option value="{{ $y }}">{{ $y }}</option>
                                @endfor
                            </select>
                            <div class="invalid-feedback" id="year_error"></div>
                        </div>
                    </div>

                    <!-- Months section - shown after year is selected -->
                    <div id="monthsSection" style="display: none;">
                        <hr>
                        <h6 class="mb-3">Monthly Targets</h6>
                        <div class="row">
                            @php
                                $months = [
                                    1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
                                    5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
                                    9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
                                ];
                            @endphp
                            @foreach($months as $monthNum => $monthName)
                                <div class="col-md-4 mb-3">
                                    <label for="month_{{ $monthNum }}" class="form-label">{{ $monthName }}</label>
                                    <input type="number" 
                                           class="form-control month-amount" 
                                           id="month_{{ $monthNum }}" 
                                           name="months[{{ $monthNum }}]" 
                                           step="0.01" 
                                           min="0" 
                                           placeholder="0.00">
                                    <div class="invalid-feedback" id="month_{{ $monthNum }}_error"></div>
                                </div>
                            @endforeach
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveTargetBtn">Save Targets</button>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeTargetTable();
        setupEventHandlers();
    });
    
    let targetTable = null;
    let targetModal = null;
    let currentGroupKey = null;
    let yearPicker = null;
    
    function initializeTargetTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        targetTable = $('#branch_targets_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            autoWidth: false,
            fixedHeader: {
                header: true,
                footer: false
            },
            layout: {
                topStart: 'buttons',
                topEnd: 'search',
                bottomStart: {
                    pageLength: {
                        menu: [10, 20, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            responsive: false,
            ajax: {
                url: "{{ route('branch-targets.data') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.business_name = $('input[name="business_name"]').val();
                    d.target_type = $('select[name="target_type"]').val();
                    d.year = $('input[name="year"]').val();
                    d.month = $('select[name="month"]').val();
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load data. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            columns: [
                { 
                    data: 'business_name', 
                    name: 'business_name', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        if (row.is_first_row) {
                            return '<span class="group-header">' + data + '</span>';
                        }
                        return '';
                    }
                },
                { 
                    data: 'target_type_display', 
                    name: 'target_type', 
                    orderable: true, 
                    searchable: false,
                    render: function(data, type, row) {
                        if (row.is_first_row) {
                            return '<span class="group-header">' + data + '</span>';
                        }
                        return '';
                    }
                },
                { 
                    data: 'year', 
                    name: 'year', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        if (row.is_first_row) {
                            return '<span class="group-header">' + data + '</span>';
                        }
                        return '';
                    }
                },
                { data: 'month_display', name: 'month', orderable: true, searchable: false },
                { 
                    data: 'amount_display', 
                    name: 'amount', 
                    orderable: true, 
                    searchable: false,
                    render: function(data, type, row) {
                        return data || '-';
                    }
                },
                { 
                    data: 'action', 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        if (row.is_first_row) {
                            return data;
                        }
                        return '';
                    }
                }
            ],
            order: [[2, 'desc'], [0, 'asc'], [1, 'asc']],
            drawCallback: function(settings) {
                // Apply rowspan to grouped columns
                applyRowspan();
            }
        });
        
        // Initialize modal
        targetModal = new bootstrap.Modal(document.getElementById('targetModal'));
        
        // Fix horizontal shrink issue when modal closes
        const targetModalElement = document.getElementById('targetModal');
        if (targetModalElement) {
            targetModalElement.addEventListener('hidden.bs.modal', function() {
                if (targetTable) {
                    targetTable.columns.adjust().draw(false);
                }
                document.body.style.paddingRight = '';
                const saveBtn = document.getElementById('saveTargetBtn');
                if (saveBtn) {
                    saveBtn.disabled = false;
                }
            });
        }
    }

    function applyRowspan() {
        const table = document.getElementById('branch_targets_table');
        const tbody = table.querySelector('tbody');
        const rows = tbody.querySelectorAll('tr');
        
        let currentGroup = null;
        let groupStartRow = null;
        let groupRowCount = 0;
        
        rows.forEach((row, index) => {
            const cells = row.querySelectorAll('td');
            if (cells.length < 6) return;
            
            const businessName = cells[0].textContent.trim();
            const targetType = cells[1].textContent.trim();
            const year = cells[2].textContent.trim();
            const groupKey = businessName + '|' + targetType + '|' + year;
            
            if (groupKey !== currentGroup) {
                // Close previous group
                if (currentGroup !== null && groupStartRow !== null && groupRowCount > 0) {
                    const startCells = groupStartRow.querySelectorAll('td');
                    if (startCells[0]) startCells[0].setAttribute('rowspan', groupRowCount);
                    if (startCells[1]) startCells[1].setAttribute('rowspan', groupRowCount);
                    if (startCells[2]) startCells[2].setAttribute('rowspan', groupRowCount);
                    if (startCells[5]) startCells[5].setAttribute('rowspan', groupRowCount);
                }
                
                // Start new group
                currentGroup = groupKey;
                groupStartRow = row;
                groupRowCount = 1;
            } else {
                // Remove cells from subsequent rows
                cells[0].remove();
                cells[1].remove();
                cells[2].remove();
                cells[5].remove();
                groupRowCount++;
            }
        });
        
        // Close last group
        if (currentGroup !== null && groupStartRow !== null && groupRowCount > 0) {
            const startCells = groupStartRow.querySelectorAll('td');
            if (startCells[0]) startCells[0].setAttribute('rowspan', groupRowCount);
            if (startCells[1]) startCells[1].setAttribute('rowspan', groupRowCount);
            if (startCells[2]) startCells[2].setAttribute('rowspan', groupRowCount);
            if (startCells[5]) startCells[5].setAttribute('rowspan', groupRowCount);
        }
    }

    function setupEventHandlers() {
        // Add Target Button
        document.getElementById('addTargetBtn')?.addEventListener('click', function() {
            openTargetModal();
        });
        
        // Save Target Button
        document.getElementById('saveTargetBtn')?.addEventListener('click', function() {
            saveTarget();
        });
        
        // Year select - show months section when year is selected
        document.getElementById('year')?.addEventListener('change', function() {
            const year = this.value;
            const monthsSection = document.getElementById('monthsSection');
            if (year && year.length === 4) {
                monthsSection.style.display = 'block';
            } else {
                monthsSection.style.display = 'none';
            }
        });
        
        // Filter Submit
        document.querySelector('.filter-submit')?.addEventListener('click', function(e) {
            e.preventDefault();
            refreshData();
        });
        
        // Filter Cancel
        document.querySelector('.filter-cancel')?.addEventListener('click', function(e) {
            e.preventDefault();
            document.querySelectorAll('.form-filter').forEach(input => {
                input.value = '';
            });
            refreshData();
        });
        
        // Enter key in filters
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }

    function refreshData() {
        if (targetTable) {
            targetTable.ajax.reload(null, false);
        }
    }

    function openTargetModal(businessId = null, targetType = null, year = null) {
        resetModalForm();
        clearValidationErrors();
        
        const saveBtn = document.getElementById('saveTargetBtn');
        if (saveBtn) {
            saveBtn.disabled = false;
        }
        
        const modalTitle = document.getElementById('targetModalLabel');
        if (modalTitle) {
            modalTitle.textContent = (businessId && targetType && year) ? 'Edit Branch Targets' : 'Add Branch Targets';
        }
        
        if (businessId && targetType && year) {
            loadTargetsForYear(businessId, targetType, year);
        } else {
            targetModal.show();
        }
    }

    function resetModalForm() {
        const form = document.getElementById('targetForm');
        if (form) {
            form.reset();
        }
        
        // Reset year select
        document.getElementById('year').value = '';
        document.getElementById('target_type').value = 'Service';
        document.getElementById('monthsSection').style.display = 'none';
        
        // Clear all month inputs
        document.querySelectorAll('.month-amount').forEach(input => {
            input.value = '';
        });
    }

    function loadTargetsForYear(businessId, targetType, year) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('branch-targets.get-year-targets') }}", {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
                'Accept': 'application/json',
            },
            body: JSON.stringify({
                business_id: businessId,
                target_type: targetType,
                year: year
            })
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success && response.targets) {
                document.getElementById('modal_business_id').value = response.business_id;
                document.getElementById('target_type').value = response.target_type;
                document.getElementById('year').value = response.year;
                
                // Populate month inputs
                Object.keys(response.targets).forEach(month => {
                    const input = document.getElementById('month_' + month);
                    if (input && response.targets[month].amount !== null) {
                        input.value = response.targets[month].amount;
                    }
                });
                
                // Show months section
                document.getElementById('monthsSection').style.display = 'block';
                
                clearValidationErrors();
                targetModal.show();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load target data.',
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading target data. Please try again.',
                customClass: {
                    confirmButton: 'btn btn-danger'
                }
            });
        });
    }

    function saveTarget() {
        const saveBtn = document.getElementById('saveTargetBtn');
        
        if (saveBtn) {
            saveBtn.disabled = true;
        }
        
        const form = document.getElementById('targetForm');
        if (!form.checkValidity()) {
            form.classList.add('was-validated');
            if (saveBtn) {
                saveBtn.disabled = false;
            }
            return;
        }
        
        const formData = new FormData(form);
        
        // Convert FormData to object for JSON
        const data = {
            business_id: formData.get('business_id'),
            target_type: formData.get('target_type'),
            year: formData.get('year'),
            months: {}
        };
        
        // Collect month amounts
        for (let i = 1; i <= 12; i++) {
            const monthInput = document.getElementById('month_' + i);
            if (monthInput && monthInput.value) {
                data.months[i] = parseFloat(monthInput.value);
            } else {
                data.months[i] = null;
            }
        }
        
        Swal.fire({
            title: 'Saving...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('branch-targets.save') }}", {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
                'Accept': 'application/json',
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                Swal.fire({
                    icon: 'success',
                    title: 'Success',
                    text: response.message || 'Branch targets saved successfully.',
                    timer: 2000,
                    showConfirmButton: false
                }).then(() => {
                    targetModal.hide();
                    refreshData();
                });
            } else {
                let errorMsg = response.message || 'Error saving branch targets.';
                if (response.errors) {
                    displayValidationErrors(response.errors);
                    errorMsg = 'Please fix the validation errors.';
                }
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: errorMsg,
                });
                if (saveBtn) {
                    saveBtn.disabled = false;
                }
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving branch targets. Please try again.',
            });
            if (saveBtn) {
                saveBtn.disabled = false;
            }
        });
    }

    function openEditModal(businessId, targetType, year) {
        openTargetModal(businessId, targetType, year);
    }

    function deleteYearTargets(businessId, targetType, year) {
        Swal.fire({
            title: 'Are you sure?',
            text: "This will delete all targets for this year. You won't be able to revert this!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Yes, delete it!',
            customClass: {
                confirmButton: 'btn btn-danger me-2',
                cancelButton: 'btn btn-warning'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Deleting...',
                    allowOutsideClick: false,
                    didOpen: () => Swal.showLoading()
                });
                
                fetch("{{ route('branch-targets.delete-year') }}", {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                    },
                    body: JSON.stringify({
                        business_id: businessId,
                        target_type: targetType,
                        year: year
                    })
                })
                .then(response => response.json())
                .then(response => {
                    Swal.close();
                    
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Deleted!',
                            text: response.message || 'Branch targets deleted successfully.',
                            timer: 2000,
                            showConfirmButton: false
                        }).then(() => {
                            refreshData();
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: response.message || 'Error deleting branch targets.',
                        });
                    }
                })
                .catch(error => {
                    Swal.close();
                    console.error('Error:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: 'Error deleting branch targets. Please try again.',
                    });
                });
            }
        });
    }

    function clearValidationErrors() {
        document.querySelectorAll('.invalid-feedback').forEach(el => {
            el.textContent = '';
        });
        document.getElementById('targetForm')?.classList.remove('was-validated');
    }

    function displayValidationErrors(errors) {
        clearValidationErrors();
        Object.keys(errors).forEach(field => {
            const errorElement = document.getElementById(field + '_error');
            if (errorElement) {
                errorElement.textContent = errors[field][0];
            }
        });
    }

    function showAlert(type, message) {
        const alertContainer = document.getElementById('alert-container');
        const alertClass = type === 'error' ? 'alert-danger' : 'alert-success';
        alertContainer.innerHTML = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;
        setTimeout(() => {
            alertContainer.innerHTML = '';
        }, 5000);
    }
</script>
@endsection
