@extends('layouts.horizontal', ['title' => 'Color Records', 'topbarTitle' => 'Color Records'])

@section('css')
@vite([
    'node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
    'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
    'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
    'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
    'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
    'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
    'node_modules/flatpickr/dist/flatpickr.min.css',
    'node_modules/select2/dist/css/select2.min.css'
])
@endsection

@section('content')

<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Color Records</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="ColorRecordsTable" class="table table-striped table-bordered w-100 fs-13">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Customer</th>
                                    <th>Color Type</th>
                                    <th>Color #</th>
                                    <th>Water Content</th>
                                    <th>Date</th>
                                    <th>Time</th>
                                    <th>Charge</th>
                                    <th>Remarks</th>
                                    <th>Recommendation</th>
                                    <th class="text-end">Edit</th>
                                </tr>
                                <tr class="filters">
                                    <td><input type="text" class="form-control form-control-sm" id="filter_id" placeholder="ID"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_customer" placeholder="Customer"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_color_type" placeholder="Color Type"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_color_no" placeholder="Color No"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_water_content" placeholder="Water Content"></td>
                                    <td><input type="text" class="form-control form-control-sm flatpickr" id="filter_date" placeholder="Date"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_time" placeholder="Time"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_charge" placeholder="Charge"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_remarks" placeholder="Remarks"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_recommendation" placeholder="Recommendation"></td>
                                    <td></td>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Add/Edit Record Modal --}}
<div class="modal fade" id="recordModal" tabindex="-1" aria-labelledby="recordModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="recordModalLabel"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="recordModalForm" onsubmit="return false;">
                @csrf
                <div class="modal-body">
                    <input type="hidden" id="id" name="id">
                    <div class="mb-3" id="id_display_container" style="display: none;">
                        <label for="id_display" class="form-label">ID</label>
                        <input type="text" class="form-control" id="id_display" readonly>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="customer_name" class="form-label">Customer</label>
                            <select class="form-select" id="customer_name" name="customer_name" required style="width: 100%;">
                            </select>
                            <input type="hidden" id="customer_id" name="customer_id">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="color_type" class="form-label">Color Type</label>
                            <input type="text" class="form-control" id="color_type" name="color_type">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="color_no" class="form-label">Color No</label>
                            <input type="text" class="form-control" id="color_no" name="color_no">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="water_content" class="form-label">Water Content</label>
                            <input type="text" class="form-control" id="water_content" name="water_content">
                        </div>
                       
                        <div class="col-md-6 mb-3">
                            <label for="date" class="form-label">Date</label>
                            <input type="text" class="form-control flatpickr" id="date" name="date">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="time_minutes" class="form-label">Time Required (minutes)</label>
                            <input type="number" class="form-control" id="time_minutes" name="time_minutes" min="0" step="1" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="charge" class="form-label">Charge</label>
                            <input type="number" class="form-control" id="charge" name="charge" min="0" step="0.01">
                        </div>
                        <div class="col-12 mb-3">
                            <label for="remarks" class="form-label">Remarks</label>
                            <textarea class="form-control" id="remarks" name="remarks" rows="3"></textarea>
                        </div>
                        <div class="col-12 mb-3">
                            <label for="recommendation" class="form-label">Recommendation</label>
                            <textarea class="form-control" id="recommendation" name="recommendation" rows="3"></textarea>
                            </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" onclick="save_record()" class="btn btn-primary" id="saveRecordBtn">Save changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@section('scripts')

<script>
    let recordsTable;
    let filterTimeout = null;

    $(document).ready(function() {
        initRecordsTable();
        initFilters();
        initDatePicker();
        initCustomerSelect();
    });

    function initRecordsTable() {
        if ($.fn.DataTable.isDataTable('#ColorRecordsTable')) {
            $('#ColorRecordsTable').DataTable().clear().destroy();
        }

        recordsTable = $('#ColorRecordsTable').DataTable({
            processing: true,
            serverSide: false,
            ajax: {
                url: '{{ route("color_records.records_data") }}',
                type: 'GET',
                data: function(d) {
                    d.filter_id = $('#filter_id').val();
                    d.filter_customer = $('#filter_customer').val();
                    d.filter_color_type = $('#filter_color_type').val();
                    d.filter_color_no = $('#filter_color_no').val();
                    d.filter_water_content = $('#filter_water_content').val();
                    d.filter_date = $('#filter_date').val();
                    d.filter_time = $('#filter_time').val();
                    d.filter_charge = $('#filter_charge').val();
                    d.filter_remarks = $('#filter_remarks').val();
                    d.filter_recommendation = $('#filter_recommendation').val();
                }
            },
            columns: [
                { data: 'id', name: 'id' },
                { data: 'customer', name: 'customer' },
                { data: 'color_type', name: 'color_type' },
                { data: 'color_number', name: 'color_no' },
                { data: 'water_content', name: 'water_content' },
                { data: 'date', name: 'date' },
                { 
                    data: 'formatted_time', 
                    name: 'time',
                    render: function(data) {
                        return data || '0:00:00';
                    }
                },
                { data: 'charge', name: 'charge' },
                { data: 'remarks', name: 'remarks' },
                { data: 'recommendation', name: 'recommendation' },
                { 
                    data: null,
                    orderable: false,
                    className: 'text-end',
                    render: function(data, type, row) {
                        return '<button class="btn btn-sm btn-ghost-info rounded-circle btn-icon" onclick="openEditRecordModal(' + row.id + ')"><iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon></button>';
                    }
                }
            ],
            pageLength: 10,
            order: [[0, 'desc']],
            orderCellsTop: true,
            searching: false,
            layout: {
                topStart: 'buttons',
                topEnd: function() {
                    return '<button class="btn btn-sm btn-outline-success" onclick="openAddRecordModal()"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Record</button>';
                },
                bottomStart: {
                    pageLength: { menu: [10, 20, 50, 100] },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ]
        });
    }

    function initFilters() {
        const filterInputs = ['filter_id', 'filter_customer', 'filter_color_type', 'filter_color_no', 'filter_water_content', 'filter_date', 'filter_time', 'filter_charge', 'filter_remarks', 'filter_recommendation'];
        
        filterInputs.forEach(function(id) {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('input', function() {
                    clearTimeout(filterTimeout);
                    filterTimeout = setTimeout(function() {
                        recordsTable.ajax.reload();
                    }, 300);
                });
            }
        });
    }

    function initDatePicker() {
        flatpickr('#filter_date', {
            dateFormat: 'Y-m-d',
            allowInput: true,
            onChange: function() {
                recordsTable.ajax.reload();
            }
        });

        flatpickr('#date', {
            dateFormat: 'Y-m-d',
            allowInput: true
        });
    }

    function initCustomerSelect() {
        $('#customer_name').select2({
            dropdownParent: $('#recordModal'),
            placeholder: 'Select Customer',
            minimumInputLength: 3,
            ajax: {
                url: '{{ route("customer_search") }}',
                dataType: 'json',
                delay: 250,
                data: function (params) {
                    return { term: params.term };
                },
                transport: function (params, success, failure) {
                    let $select = $('#customer_name');
                    $.ajax(params).then(function (data) {
                        success(data);

                        setTimeout(() => {
                            let select2 = $select.data('select2');
                            let $first = select2.$results.find('.select2-results__option:first');
                            if ($first.length) {
                                $first.addClass('select2-results__option--highlighted');
                            }
                        }, 500);
                    }, failure);
                },
                processResults: function (data) {
                    return {
                        results: data.map(function(item){
                            return {
                                id: item.id,
                                text: item.customer_name || (item.customer_name + ' (' + item.customer_cell + ')'),
                                customer_name: item.customer_name,
                                customer_cell: item.customer_cell,
                                customer_type: item.customer_type
                            };
                        })
                    };
                },
                cache: true
            },
            templateResult: formatCustomer,
            templateSelection: formatCustomerSel
        }).on('select2:select', function (e) {
            const data = e.params.data;
            $('#customer_id').val(data.id);
        });
    }

    function formatCustomer(customer) {
        if (!customer || !customer.id) {
            return customer && customer.text ? customer.text : '';
        }

        var type = (customer.customer_type || '').toString().toLowerCase();
        var iconClass = 'ti-user fs-4 text-secondary';
        var nameClass = '';

        if (type === 'orange' || type === '1') {
            iconClass = 'ti-star fs-4 text-warning';
            nameClass = 'text-warning';
        } else if (type === 'green' || type === '2') {
            iconClass = 'ti-flag fs-4 text-success';
            nameClass = 'text-success';
        } else if (type === 'red' || type === '3') {
            iconClass = 'ti-flag-question fs-4 text-danger';
        }

        var nameEsc = $('<div/>').text(customer.customer_name || customer.text || '').html();
        var cellEsc = $('<div/>').text(customer.customer_cell || '').html();

        var $node = $(`
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <div class="fw-bold ${nameClass}">${nameEsc}</div>
                    <div class="small">${cellEsc}</div>
                </div>
                <i class="ti ${iconClass} ms-2" aria-hidden="true"></i>
            </div>
        `);

        return $node[0];
    }

    function formatCustomerSel(customer) {
        if (!customer || !customer.id) {
            return customer && customer.text ? customer.text : '';
        }
        var name = customer.customer_name || customer.text;
        var cell = customer.customer_cell ? ' (' + customer.customer_cell + ')' : '';
        return $('<span>').text(name + cell);
    }

    function openAddRecordModal() {
        $('#id').val('');
        $('#id_display').val('');
        $('#id_display_container').hide();
        $('#customer_id').val('');
        
        if ($('#customer_name').hasClass('select2-hidden-accessible')) {
            $('#customer_name').select2('destroy');
        }
        $('#customer_name').empty();
        $('#customer_name').prop('disabled', false);
        initCustomerSelect();
        
        $('#Color').val('');
        $('#exfoliant').val('');
        $('#mask').val('');
        $('#cleanser').val('');
        $('#date').val('');
        $('#time_minutes').val('');
        $('#charge').val('');
        $('#remarks').val('');

        $('#recordModalLabel').text('Add Record');

        let el = document.getElementById('recordModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function openEditRecordModal(id) {
        $.ajax({
            url: '{{ route("color_records.records_data") }}',
            type: 'GET',
            success: function(response) {
                const record = response.data.find(r => r.id == id);
                if (record) {
                    $('#id').val(record.id);
                    $('#id_display').val(record.id);
                    $('#id_display_container').show();
                    $('#customer_id').val(record.customer_id || '');
                    
                    // Disable customer field in edit mode
                    if ($('#customer_name').hasClass('select2-hidden-accessible')) {
                        $('#customer_name').select2('destroy');
                    }
                    $('#customer_name').empty().append(new Option(record.customer || '', record.customer_id, true, true));
                    $('#customer_name').prop('disabled', true);
                    $('#customer_name').select2({
                        dropdownParent: $('#recordModal'),
                        minimumResultsForSearch: -1
                    });
                    
                    $('#Color').val(record.Color || '');
                    $('#exfoliant').val(record.exfoliant || '');
                    $('#mask').val(record.mask || '');
                    $('#cleanser').val(record.cleanser || '');
                    $('#date').val(record.date || '');
                    
                    // Convert stored minutes to minutes for input
                    const timeMinutes = record.time ? parseInt(record.time) : 0;
                    $('#time_minutes').val(timeMinutes);
                    
                    $('#charge').val(record.charge || '');
                    $('#remarks').val(record.remarks || '');

                    $('#recordModalLabel').text('Edit Record');

                    let el = document.getElementById('recordModal');
                    let modal = bootstrap.Modal.getInstance(el);
                    if (!modal) modal = new bootstrap.Modal(el);
                    modal.show();
                }
            }
        });
    }

    function save_record() {
        if (!$('#customer_name').val()) {
            toastr.warning('Customer is required.');
            return;
        }
        if (!$('#time_minutes').val() || parseInt($('#time_minutes').val()) < 0) {
            toastr.warning('Time (minutes) is required and must be 0 or greater.');
            return;
        }

        $('#saveRecordBtn').prop('disabled', true);

        // Get customer name from Select2
        let customerName = '';
        const customerData = $('#customer_name').select2('data');
        if (customerData && customerData.length > 0 && customerData[0].customer_name) {
            customerName = customerData[0].customer_name;
        } else {
            // Fallback: try to get from the select option text
            const selectedOption = $('#customer_name option:selected');
            if (selectedOption.length) {
                customerName = selectedOption.text().split(' (')[0];
            }
        }

        const formData = $('#recordModalForm').serializeArray().reduce(function(obj, item) {
            obj[item.name] = item.value;
            return obj;
        }, {});
        
        formData.customer = customerName;

        $.ajax({
            url: '{{ route("color_records.save_or_update_record") }}',
            method: 'POST',
            data: formData,
            success: function(res) {
                $('#saveRecordBtn').prop('disabled', false);
                if (res.success) {
                    toastr.success(res.message);
                    // Close modal
                    $('#recordModal').modal('hide');
                    // Reload table data after a short delay to ensure modal closes
                    setTimeout(function() {
                        recordsTable.ajax.reload();
                    }, 300);
                } else {
                    toastr.error(res.message || 'Failed to save record');
                }
            },
            error: function(xhr) {
                $('#saveRecordBtn').prop('disabled', false);
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    toastr.error(xhr.responseJSON.message);
                } else {
                    toastr.error('Failed to save record: ' + xhr.responseText);
                }
            }
        });
    }
</script>
@endsection
