@extends('layouts.horizontal', ['title' => 'Eyelash Records', 'topbarTitle' => 'Eyelash Records'])

@section('css')
@vite([
    'node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
    'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
    'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
    'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
    'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
    'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
    'node_modules/flatpickr/dist/flatpickr.min.css',
    'node_modules/select2/dist/css/select2.min.css'
])
<style>
    .config-container {
        height: 500px;
        overflow-y: auto;
        border: 1px solid #dee2e6;
        border-radius: 0.375rem;
        padding: 15px;
    }
    .config-item {
        padding: 8px 12px;
        margin: 5px 0;
        border: 1px solid #e9ecef;
        border-radius: 4px;
        cursor: pointer;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .config-item:hover {
        background-color: #f8f9fa;
    }
    .config-item.editing {
        border-color: #0d6efd;
    }
</style>
@endsection

@section('content')

<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Eyelash Records</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header border-bottom border-dashed">
                    <ul class="nav nav-tabs card-header-tabs nav-bordered" role="tablist">
                        <li class="nav-item">
                            <a href="#eyelash-records-tab" data-bs-toggle="tab" aria-expanded="true" class="nav-link active">
                                Eyelash Records
                            </a>
                        </li>
                        <li class="nav-item">
                            <a href="#eyelash-config-tab" data-bs-toggle="tab" aria-expanded="false" class="nav-link">
                                Eyelash Configuration
                            </a>
                        </li>
                    </ul>
                </div>
                <div class="card-body">
                    <div class="tab-content">
                        <!-- Eyelash Records Tab -->
                        <div class="tab-pane fade show active" id="eyelash-records-tab" role="tabpanel">
                            <div class="table-responsive">
                                <table id="eyelashRecordsTable" class="table table-striped table-bordered w-100 fs-13">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Customer</th>
                                            <th>Type</th>
                                            <th>Thickness</th>
                                            <th>Length</th>
                                            <th>Curl</th>
                                            <th>Full Set / Refill</th>
                                            <th>Date</th>
                                            <th>Price</th>
                                            <th>Remarks</th>
                                            <th class="text-end">Action</th>
                                        </tr>
                                        <tr class="filters">
                                            <td><input type="text" class="form-control form-control-sm" id="filter_id" placeholder="ID"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_customer" placeholder="Customer"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_type" placeholder="Type"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_thickness" placeholder="Thickness"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_length" placeholder="Length"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_curl" placeholder="Curl"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_full_set_refill" placeholder="Full Set/Refill"></td>
                                            <td><input type="text" class="form-control form-control-sm flatpickr" id="filter_date" placeholder="Date"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_price" placeholder="Price"></td>
                                            <td><input type="text" class="form-control form-control-sm" id="filter_remarks" placeholder="Remarks"></td>
                                            <td></td>
                                        </tr>
                                    </thead>
                                    <tbody>
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <!-- Eyelash Configuration Tab -->
                        <div class="tab-pane fade" id="eyelash-config-tab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <h5 class="mb-0">Type Of Lashes</h5>
                                        <button class="btn btn-sm btn-outline-success" onclick="addConfigItem('type')">
                                            <iconify-icon icon="mdi:plus" class="fs-18"></iconify-icon>
                                        </button>
                                    </div>
                                    <div class="config-container" id="type-container"></div>
                                </div>
                                <div class="col-md-3">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <h5 class="mb-0">Thickness</h5>
                                        <button class="btn btn-sm btn-outline-success" onclick="addConfigItem('thickness')">
                                            <iconify-icon icon="mdi:plus" class="fs-18"></iconify-icon>
                                        </button>
                                    </div>
                                    <div class="config-container" id="thickness-container"></div>
                                </div>
                                <div class="col-md-3">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <h5 class="mb-0">Length</h5>
                                        <button class="btn btn-sm btn-outline-success" onclick="addConfigItem('length')">
                                            <iconify-icon icon="mdi:plus" class="fs-18"></iconify-icon>
                                        </button>
                                    </div>
                                    <div class="config-container" id="length-container"></div>
                                </div>
                                <div class="col-md-3">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <h5 class="mb-0">Curl</h5>
                                        <button class="btn btn-sm btn-outline-success" onclick="addConfigItem('curl')">
                                            <iconify-icon icon="mdi:plus" class="fs-18"></iconify-icon>
                                        </button>
                                    </div>
                                    <div class="config-container" id="curl-container"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Add/Edit Record Modal --}}
<div class="modal fade" id="recordModal" tabindex="-1" aria-labelledby="recordModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="recordModalLabel"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="recordModalForm" onsubmit="return false;">
                @csrf
                <div class="modal-body">
                    <input type="hidden" id="id_eyelashes" name="id_eyelashes">

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="customer_name" class="form-label">Customer</label>
                            <select class="form-select" id="customer_name" name="customer_name" required style="width: 100%;">
                            </select>
                            <input type="hidden" id="customer_id" name="customer_id">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="eyelash_type" class="form-label">Type</label>
                            <select class="form-select" id="eyelash_type" name="eyelash_type" required>
                                <option value="">Select Type</option>
                                @foreach($eyelashTypes as $type)
                                    <option value="{{ $type->eyelash_type }}">{{ $type->eyelash_type }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="thickness" class="form-label">Thickness</label>
                            <select class="form-select" id="thickness" name="thickness">
                                <option value="">Select Thickness</option>
                                @foreach($thicknesses as $thickness)
                                    <option value="{{ $thickness->thickness }}">{{ $thickness->thickness }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="length" class="form-label">Length</label>
                            <select class="form-select" id="length" name="length">
                                <option value="">Select Length</option>
                                @foreach($lengths as $length)
                                    <option value="{{ $length->length }}">{{ $length->length }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="curl" class="form-label">Curl</label>
                            <select class="form-select" id="curl" name="curl">
                                <option value="">Select Curl</option>
                                @foreach($curls as $curl)
                                    <option value="{{ $curl->curl }}">{{ $curl->curl }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="full_set_refill" class="form-label">Full Set / Refill</label>
                            <select class="form-select" id="full_set_refill" name="full_set_refill">
                                <option value="">Select</option>
                                <option value="Full Set">Full Set</option>
                                <option value="Refill">Refill</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="date" class="form-label">Date</label>
                            <input type="text" class="form-control flatpickr" id="date" name="date">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="price" class="form-label">Price</label>
                            <input type="text" class="form-control" id="price" name="price">
                        </div>
                        <div class="col-12 mb-3">
                            <label for="remarks" class="form-label">Remarks</label>
                            <textarea class="form-control" id="remarks" name="remarks" rows="3"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" onclick="save_record()" class="btn btn-primary" id="saveRecordBtn">Save changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script>
    let recordsTable;
    let filterTimeout = null;
    let editingItem = null;

    $(document).ready(function() {
        // Handle tab changes
        $('a[data-bs-toggle="tab"]').on('shown.bs.tab', function (e) {
            const targetTab = $(e.target).attr('href');
            if (targetTab === '#eyelash-config-tab') {
                $('#add-record-btn').hide();
                loadConfigurationData();
            } else {
                $('#add-record-btn').show();
            }
        });

        initRecordsTable();
        initFilters();
        initDatePicker();
        initCustomerSelect();
        loadConfigurationData();
    });

    function initRecordsTable() {
        if ($.fn.DataTable.isDataTable('#eyelashRecordsTable')) {
            $('#eyelashRecordsTable').DataTable().clear().destroy();
        }

        recordsTable = $('#eyelashRecordsTable').DataTable({
            processing: true,
            serverSide: false,
            ajax: {
                url: '{{ route("eyelash_records.records_data") }}',
                type: 'GET',
                data: function(d) {
                    d.filter_id = $('#filter_id').val();
                    d.filter_customer = $('#filter_customer').val();
                    d.filter_type = $('#filter_type').val();
                    d.filter_thickness = $('#filter_thickness').val();
                    d.filter_length = $('#filter_length').val();
                    d.filter_curl = $('#filter_curl').val();
                    d.filter_full_set_refill = $('#filter_full_set_refill').val();
                    d.filter_date = $('#filter_date').val();
                    d.filter_price = $('#filter_price').val();
                    d.filter_remarks = $('#filter_remarks').val();
                }
            },
            columns: [
                { data: 'id_eyelashes', name: 'id_eyelashes' },
                { data: 'customer_name', name: 'customer_name' },
                { data: 'eyelash_type', name: 'eyelash_type' },
                { data: 'thickness', name: 'thickness' },
                { data: 'length', name: 'length' },
                { data: 'curl', name: 'curl' },
                { data: 'full_set_refill', name: 'full_set_refill' },
                { data: 'date', name: 'date' },
                { data: 'price', name: 'price' },
                { data: 'remarks', name: 'remarks' },
                { 
                    data: null,
                    orderable: false,
                    className: 'text-end',
                    render: function(data, type, row) {
                        return '<button class="btn btn-sm btn-ghost-info rounded-circle btn-icon" onclick="openEditRecordModal(' + row.id_eyelashes + ')"><iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon></button>';
                    }
                }
            ],
            pageLength: 10,
            order: [[0, 'desc']],
            orderCellsTop: true,
            searching: false,
            layout: {
                topStart: 'buttons',
                topEnd: function() {
                    return '<button class="btn btn-sm btn-outline-success" id="add-record-btn" onclick="openAddRecordModal()"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Record</button>';
                },
                bottomStart: {
                    pageLength: { menu: [10, 20, 50, 100] },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ]
        });
    }

    function initFilters() {
        const filterInputs = ['filter_id', 'filter_customer', 'filter_type', 'filter_thickness', 'filter_length', 'filter_curl', 'filter_full_set_refill', 'filter_price', 'filter_remarks'];
        
        filterInputs.forEach(function(id) {
            const element = document.getElementById(id);
            if (element) {
                element.addEventListener('input', function() {
                    clearTimeout(filterTimeout);
                    filterTimeout = setTimeout(function() {
                        recordsTable.ajax.reload();
                    }, 300);
                });
            }
        });
    }

    function initDatePicker() {
        flatpickr('#filter_date', {
            dateFormat: 'Y-m-d',
            allowInput: true,
            onChange: function() {
                recordsTable.ajax.reload();
            }
        });

        flatpickr('#date', {
            dateFormat: 'Y-m-d',
            allowInput: true
        });
    }

    function initCustomerSelect() {
        $('#customer_name').select2({
            dropdownParent: $('#recordModal'),
            placeholder: 'Select Customer',
            minimumInputLength: 3,
            ajax: {
                url: '{{ route("customer_search") }}',
                dataType: 'json',
                delay: 250,
                data: function (params) {
                    return { term: params.term };
                },
                transport: function (params, success, failure) {
                    let $select = $('#customer_name');
                    $.ajax(params).then(function (data) {
                        success(data);

                        // Executed AFTER results loaded
                        setTimeout(() => {
                            let select2 = $select.data('select2');
                            let $first = select2.$results.find('.select2-results__option:first');
                            if ($first.length) {
                                $first.addClass('select2-results__option--highlighted');
                            }
                        }, 500);
                    }, failure);
                },
                processResults: function (data) {
                    return {
                        results: data.map(function(item){
                            return {
                                id: item.id,
                                text: item.customer_name || (item.customer_name + ' (' + item.customer_cell + ')'),
                                customer_name: item.customer_name,
                                customer_cell: item.customer_cell,
                                customer_type: item.customer_type
                            };
                        })
                    };
                },
                cache: true
            },
            templateResult: formatCustomer,
            templateSelection: formatCustomerSel
        }).on('select2:select', function (e) {
            const data = e.params.data;
            $('#customer_id').val(data.id);
        });
    }

    function formatCustomer(customer) {
        if (!customer || !customer.id) {
            return customer && customer.text ? customer.text : '';
        }

        var type = (customer.customer_type || '').toString().toLowerCase();
        var iconClass = 'ti-user fs-4 text-secondary';
        var nameClass = '';

        if (type === 'orange' || type === '1') {
            iconClass = 'ti-star fs-4 text-warning';
            nameClass = 'text-warning';
        } else if (type === 'green' || type === '2') {
            iconClass = 'ti-flag fs-4 text-success';
            nameClass = 'text-success';
        } else if (type === 'red' || type === '3') {
            iconClass = 'ti-flag-question fs-4 text-danger';
        }

        var nameEsc = $('<div/>').text(customer.customer_name || customer.text || '').html();
        var cellEsc = $('<div/>').text(customer.customer_cell || '').html();

        var $node = $(`
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <div class="fw-bold ${nameClass}">${nameEsc}</div>
                    <div class="small">${cellEsc}</div>
                </div>
                <i class="ti ${iconClass} ms-2" aria-hidden="true"></i>
            </div>
        `);

        return $node[0];
    }

    function formatCustomerSel(customer) {
        if (!customer || !customer.id) {
            return customer && customer.text ? customer.text : '';
        }
        var name = customer.customer_name || customer.text;
        var cell = customer.customer_cell ? ' (' + customer.customer_cell + ')' : '';
        return $('<span>').text(name + cell);
    }

    function openAddRecordModal() {
        $('#id_eyelashes').val('');
        $('#customer_id').val('');
        
        // Reinitialize Select2 for add mode
        if ($('#customer_name').hasClass('select2-hidden-accessible')) {
            $('#customer_name').select2('destroy');
        }
        $('#customer_name').empty();
        $('#customer_name').prop('disabled', false);
        initCustomerSelect();
        
        $('#eyelash_type').val('');
        $('#thickness').val('');
        $('#length').val('');
        $('#curl').val('');
        $('#full_set_refill').val('');
        $('#date').val('');
        $('#price').val('');
        $('#remarks').val('');

        $('#recordModalLabel').text('Add Record');

        let el = document.getElementById('recordModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function openEditRecordModal(id) {
        $.ajax({
            url: '{{ route("eyelash_records.records_data") }}',
            type: 'GET',
            success: function(response) {
                const record = response.data.find(r => r.id_eyelashes == id);
                if (record) {
                    $('#id_eyelashes').val(record.id_eyelashes);
                    $('#customer_id').val(record.customer_id || '');
                    
                    // Disable customer field in edit mode and show as read-only
                    if ($('#customer_name').hasClass('select2-hidden-accessible')) {
                        $('#customer_name').select2('destroy');
                    }
                    $('#customer_name').empty().append(new Option(record.customer_name + ' (' + (record.customer_cell || '') + ')', record.customer_id, true, true));
                    $('#customer_name').prop('disabled', true);
                    $('#customer_name').select2({
                        dropdownParent: $('#recordModal'),
                        minimumResultsForSearch: -1
                    });
                    
                    $('#eyelash_type').val(record.eyelash_type);
                    $('#thickness').val(record.thickness);
                    $('#length').val(record.length);
                    $('#curl').val(record.curl);
                    $('#full_set_refill').val(record.full_set_refill);
                    $('#date').val(record.date);
                    $('#price').val(record.price);
                    $('#remarks').val(record.remarks);

                    $('#recordModalLabel').text('Edit Record');

                    let el = document.getElementById('recordModal');
                    let modal = bootstrap.Modal.getInstance(el);
                    if (!modal) modal = new bootstrap.Modal(el);
                    modal.show();
                }
            }
        });
    }

    function save_record() {
        if (!$('#customer_name').val()) {
            toastr.warning('Customer is required.');
            return;
        }
        if (!$('#eyelash_type').val()) {
            toastr.warning('Type is required.');
            return;
        }

        $('#saveRecordBtn').prop('disabled', true);

        $.ajax({
            url: '{{ route("eyelash_records.save_or_update_record") }}',
            method: 'POST',
            data: $('#recordModalForm').serialize(),
            success: function(res) {
                $('#saveRecordBtn').prop('disabled', false);
                if (res.success) {
                    toastr.success(res.message);
                    // Close modal
                    $('#recordModal').modal('hide');
                    // Reload table data after a short delay to ensure modal closes
                    setTimeout(function() {
                        recordsTable.ajax.reload();
                    }, 300);
                } else {
                    toastr.error(res.message || 'Failed to save record');
                }
            },
            error: function(xhr) {
                $('#saveRecordBtn').prop('disabled', false);
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    toastr.error(xhr.responseJSON.message);
                } else {
                    toastr.error('Failed to save record: ' + xhr.responseText);
                }
            }
        });
    }

    function loadConfigurationData() {
        $.ajax({
            url: '{{ route("eyelash_records.get_configuration_data") }}',
            type: 'GET',
            success: function(response) {
                renderConfigItems('type', response.types, 'id_eyelash_types', 'eyelash_type');
                renderConfigItems('thickness', response.thicknesses, 'id_eyelash_thickness', 'thickness');
                renderConfigItems('length', response.lengths, 'id_eyelash_length', 'length');
                renderConfigItems('curl', response.curls, 'id_eyelash_curl', 'curl');
            }
        });
    }

    function renderConfigItems(type, items, idField, valueField) {
        const container = $('#' + type + '-container');
        container.empty();
        
        items.forEach(function(item) {
            const div = $('<div class="config-item" data-id="' + item[idField] + '" data-type="' + type + '"></div>');
            div.append('<span class="config-value">' + (item[valueField] || '') + '</span>');
            div.append('<iconify-icon icon="iconamoon:trash-thin" class="fs-18 text-danger delete-item" style="cursor: pointer;"></iconify-icon>');
            div.on('dblclick', function() {
                startEditing($(this), type, idField, valueField);
            });
            div.find('.delete-item').on('click', function(e) {
                e.stopPropagation();
                deleteConfigItem(item[idField], type);
            });
            container.append(div);
        });
    }

    function startEditing(element, type, idField, valueField) {
        if (editingItem) {
            return;
        }
        
        editingItem = element;
        const currentValue = element.find('.config-value').text();
        const id = element.data('id');
        
        element.addClass('editing');
        element.find('.config-value').html('<input type="text" class="form-control form-control-sm" value="' + currentValue + '" style="display: inline-block; width: auto;">');
        const input = element.find('input');
        input.focus().select();
        
        input.on('blur', function() {
            finishEditing(element, type, id, idField, valueField);
        });
        
        input.on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                finishEditing(element, type, id, idField, valueField);
            }
        });
    }

    function finishEditing(element, type, id, idField, valueField) {
        if (!editingItem) return;
        
        const newValue = element.find('input').val().trim();
        if (newValue === '') {
            toastr.warning('Value cannot be empty');
            element.removeClass('editing');
            element.find('.config-value').text(element.data('original-value') || '');
            editingItem = null;
            return;
        }

        const routeMap = {
            'type': '{{ route("eyelash_records.save_type") }}',
            'thickness': '{{ route("eyelash_records.save_thickness") }}',
            'length': '{{ route("eyelash_records.save_length") }}',
            'curl': '{{ route("eyelash_records.save_curl") }}'
        };

        const data = {};
        data[idField] = id;
        data[valueField] = newValue;
        data._token = '{{ csrf_token() }}';

        $.ajax({
            url: routeMap[type],
            method: 'POST',
            data: data,
            success: function(res) {
                if (res.success) {
                    element.removeClass('editing');
                    element.find('.config-value').text(newValue);
                    loadConfigurationData();
                    toastr.success(res.message);
                } else {
                    toastr.error(res.message);
                }
                editingItem = null;
            },
            error: function(xhr) {
                editingItem = null;
                element.removeClass('editing');
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    toastr.error(xhr.responseJSON.message);
                } else {
                    toastr.error('Failed to save');
                }
            }
        });
    }

    function addConfigItem(type) {
        const typeLabels = {
            'type': 'Type Of Lashes',
            'thickness': 'Thickness',
            'length': 'Length',
            'curl': 'Curl'
        };

        Swal.fire({
            title: 'Add ' + typeLabels[type],
            input: 'text',
            inputLabel: 'Enter ' + typeLabels[type] + ' value',
            inputPlaceholder: 'Enter value...',
            showCancelButton: true,
            confirmButtonText: 'Add',
            cancelButtonText: 'Cancel',
            inputValidator: (value) => {
                if (!value || value.trim() === '') {
                    return 'Value cannot be empty';
                }
            },
            customClass: {
                confirmButton: 'btn btn-success me-2',
                cancelButton: 'btn btn-secondary'
            }
        }).then((result) => {
            if (result.isConfirmed && result.value && result.value.trim() !== '') {
                const value = result.value.trim();

                const routeMap = {
                    'type': '{{ route("eyelash_records.save_type") }}',
                    'thickness': '{{ route("eyelash_records.save_thickness") }}',
                    'length': '{{ route("eyelash_records.save_length") }}',
                    'curl': '{{ route("eyelash_records.save_curl") }}'
                };

                const fieldMap = {
                    'type': 'eyelash_type',
                    'thickness': 'thickness',
                    'length': 'length',
                    'curl': 'curl'
                };

                const data = {};
                data[fieldMap[type]] = value;
                data._token = '{{ csrf_token() }}';

                $.ajax({
                    url: routeMap[type],
                    method: 'POST',
                    data: data,
                    success: function(res) {
                        if (res.success) {
                            loadConfigurationData();
                            toastr.success(res.message);
                        } else {
                            toastr.error(res.message);
                        }
                    },
                    error: function(xhr) {
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            toastr.error(xhr.responseJSON.message);
                        } else {
                            toastr.error('Failed to save');
                        }
                    }
                });
            }
        });
    }

    function deleteConfigItem(id, type) {
        Swal.fire({
            title: 'Delete?',
            text: 'Are you sure you want to delete this item?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, delete',
            cancelButtonText: 'Cancel'
        }).then((result) => {
            if (result.isConfirmed) {
                const routeMap = {
                    'type': '{{ route("eyelash_records.delete_type", ":id") }}',
                    'thickness': '{{ route("eyelash_records.delete_thickness", ":id") }}',
                    'length': '{{ route("eyelash_records.delete_length", ":id") }}',
                    'curl': '{{ route("eyelash_records.delete_curl", ":id") }}'
                };

                const url = routeMap[type].replace(':id', id);

                $.ajax({
                    url: url,
                    method: 'DELETE',
                    data: {
                        _token: '{{ csrf_token() }}'
                    },
                    success: function(res) {
                        if (res.success) {
                            loadConfigurationData();
                            toastr.success(res.message);
                        } else {
                            toastr.error(res.message);
                        }
                    },
                    error: function(xhr) {
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            toastr.error(xhr.responseJSON.message);
                        } else {
                            toastr.error('Failed to delete');
                        }
                    }
                });
            }
        });
    }
</script>
@endsection
