@extends('layouts.horizontal', ['title' => 'Discount Configuration', 'topbarTitle' => 'Discount Configuration'])

@section('css')
@vite([
    'node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
    'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
    'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
    'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
    'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
    'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css'
])
@endsection

@section('content')

<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Discount Configuration</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header border-bottom border-dashed">
                    <ul class="nav nav-tabs card-header-tabs nav-bordered" role="tablist">
                        <li class="nav-item">
                            <a href="#discount-reasons-tab" data-bs-toggle="tab" aria-expanded="true" class="nav-link active">
                                Discount Reasons
                            </a>
                        </li>
                        @can('view_discount_users')
                        <li class="nav-item">
                            <a href="#discount-users-tab" data-bs-toggle="tab" aria-expanded="false" class="nav-link">
                                Discount Users
                            </a>
                        </li>
                        @endcan
                    </ul>
                </div>
                <div class="card-body">
                    <div class="tab-content">
                        <!-- Discount Reasons Tab -->
                        <div class="tab-pane fade show active" id="discount-reasons-tab" role="tabpanel">
                            <table id="discount_reasons_table" class="table table-bordered dt-responsive fs-13 w-100">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Reason</th>
                                        <th>Type</th>
                                        <th>Value</th>
                                        <th>Active</th>
                                        <th class="noprint">Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($discountReasons as $reason)
                                        @php
                                            $type = 'None';
                                            $value = 0;
                                            if ($reason->discount_perc == 0 && $reason->fixed_amount > 0) {
                                                $type = 'Fixed Amount';
                                                $value = $reason->fixed_amount;
                                            } elseif ($reason->discount_perc > 0) {
                                                $type = 'Percentage';
                                                $value = $reason->discount_perc;
                                            }
                                        @endphp
                                        <tr>
                                            <td>{{ $reason->id_discount_reasons }}</td>
                                            <td>{{ $reason->discount_reason }}</td>
                                            <td>{{ $type }}</td>
                                            <td>{{ $value }}</td>
                                            <td>{{ $reason->active ?? 'No' }}</td>
                                            <td>
                                                @can('edit_discount_reasons')
                                                <button
                                                    onclick="openEditReasonModal({{ $reason->id_discount_reasons }})"
                                                    class="btn btn-sm btn-ghost-info me-2 rounded-circle btn-icon edit-reason-btn"
                                                    data-id="{{ $reason->id_discount_reasons }}"
                                                    data-discount_reason="{{ $reason->discount_reason }}"
                                                    data-type="{{ $type }}"
                                                    data-value="{{ $value }}"
                                                    data-active="{{ $reason->active ?? 'No' }}"
                                                >
                                                    <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                                                </button>
                                                @endcan
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>

                        <!-- Discount Users Tab -->
                        @can('view_discount_users')
                        <div class="tab-pane fade" id="discount-users-tab" role="tabpanel">
                            <table id="discount_users_table" class="table table-bordered dt-responsive fs-13 w-100">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Username</th>
                                        <th>E-mail</th>
                                        <th class="noprint">Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($discountUsers as $user)
                                        <tr>
                                            <td>{{ $user->id }}</td>
                                            <td>{{ $user->name }}</td>
                                            <td>{{ $user->username }}</td>
                                            <td>{{ $user->email }}</td>
                                            <td>
                                                @can('edit_discount_users')
                                                <button
                                                    onclick="openEditUserModal({{ $user->id }})"
                                                    class="btn btn-sm btn-ghost-info me-2 rounded-circle btn-icon edit-user-btn"
                                                    data-id="{{ $user->id }}"
                                                    data-name="{{ $user->name }}"
                                                    data-username="{{ $user->username }}"
                                                    data-email="{{ $user->email }}"
                                                >
                                                    <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                                                </button>
                                                @endcan
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        @endcan
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Add/Edit Discount Reason Modal --}}
<div class="modal fade" id="reasonModal" tabindex="-1" aria-labelledby="reasonModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-sm">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="reasonModalLabel"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="reasonModalForm">
                @csrf
                <div class="modal-body">
                    <input type="hidden" id="id_discount_reasons" name="id_discount_reasons">

                    <div class="mb-3" id="id_display_container" style="display: none;">
                        <label for="id_display" class="form-label">ID</label>
                        <input type="text" class="form-control" id="id_display" readonly>
                    </div>

                    <div class="mb-3">
                        <label for="discount_reason" class="form-label">Reason</label>
                        <input type="text" class="form-control" id="discount_reason" name="discount_reason" required>
                    </div>

                    <div class="mb-3">
                        <label for="type" class="form-label">Type</label>
                        <select class="form-select" id="type" name="type" required>
                            <option value="Percentage">Percentage</option>
                            <option value="Fixed Amount">Fixed Amount</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="value" class="form-label">Value</label>
                        <input type="number" step="0.01" min="0" class="form-control" id="value" name="value" required>
                    </div>

                    <div class="mb-3">
                        <label for="active" class="form-label">Active</label>
                        <select class="form-select" id="active" name="active" required>
                            <option value="Yes">Yes</option>
                            <option value="No">No</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" onclick="save_reason()" class="btn btn-primary" id="saveReasonBtn">Save changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

{{-- Add/Edit Discount User Modal --}}
<div class="modal fade" id="userModal" tabindex="-1" aria-labelledby="userModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-sm">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="userModalLabel"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="userModalForm">
                @csrf
                <div class="modal-body">
                    <input type="hidden" id="user_id" name="id">

                    <div class="mb-3">
                        <label for="user_name" class="form-label">Name</label>
                        <input type="text" class="form-control" id="user_name" name="name" required>
                    </div>

                    <div class="mb-3">
                        <label for="user_username" class="form-label">Username</label>
                        <input type="text" class="form-control" id="user_username" name="username" required>
                    </div>

                    <div class="mb-3">
                        <label for="user_email" class="form-label">E-mail</label>
                        <input type="email" class="form-control" id="user_email" name="email" required>
                    </div>

                    <div class="mb-3" id="password_section">
                        <label class="form-label" id="password_label">Password</label>
                        <div id="old_password_container" style="display: none;">
                            <input type="password" class="form-control mb-2" id="old_password" name="old_password" placeholder="Old Password">
                        </div>
                        <input type="password" class="form-control mb-2" id="new_password" name="new_password" placeholder="New Password">
                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" placeholder="Confirm New Password">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" onclick="save_user()" class="btn btn-primary" id="saveUserBtn">Save changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script>
    let discountReasonsTable;
    let discountUsersTable;

    $(document).ready(function() {
        initDiscountReasonsTable();
        initDiscountUsersTable();
    });

    function initDiscountReasonsTable() {
        if ($.fn.DataTable.isDataTable('#discount_reasons_table')) {
            $('#discount_reasons_table').DataTable().clear().destroy();
        }

        discountReasonsTable = $('#discount_reasons_table').DataTable({
            responsive: true,
            fixedHeader: true,
            lengthChange: true,
            pageLength: 10,
            searching: true,
            info: true,
            paging: true,
            buttons: [
                { extend: "copy",  className: "btn btn-sm btn-soft-primary" },
                { extend: "csv",   className: "btn btn-sm btn-soft-primary" },
                { extend: "print", className: "btn btn-sm btn-soft-primary" },
                { extend: "pdf",   className: "btn btn-sm btn-soft-primary" }
            ],
            layout: {
                topStart: "buttons",
                topEnd: function() {
                    // Return a flex container with search input and button
                    return '<div class="d-flex align-items-center gap-2" style="justify-content: flex-end; width: 100%;"><label class="dataTables_filter" style="margin: 0;"><input type="search" class="form-control form-control-sm" placeholder="Search..." aria-controls="discount_reasons_table" id="discount_reasons_table_search"></label>' +
                           @can('add_discount_reasons')
                           '<button class="btn btn-sm btn-outline-success" id="addReasonBtn" onclick="openAddReasonModal()"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Discount Reason</button>' +
                           @endcan
                           '</div>';
                },
                bottomStart: {
                    pageLength: { menu: [[10, 25, 50, 100, 1000], [10, 25, 50, 100, 1000]] },
                    info: {}
                },
                bottomEnd: "paging"
            },
            initComplete: function(settings) {
                // Manually wire up the search input to DataTables
                var searchInput = $('#discount_reasons_table_search');
                if (searchInput.length) {
                    searchInput.on('keyup', function() {
                        discountReasonsTable.search(this.value).draw();
                    });
                }
            }
        });
    }

    function initDiscountUsersTable() {
        if ($.fn.DataTable.isDataTable('#discount_users_table')) {
            $('#discount_users_table').DataTable().clear().destroy();
        }

        discountUsersTable = $('#discount_users_table').DataTable({
            responsive: true,
            fixedHeader: true,
            lengthChange: true,
            pageLength: 10,
            searching: true,
            info: true,
            paging: true,
            buttons: [
                { extend: "copy",  className: "btn btn-sm btn-soft-primary" },
                { extend: "csv",   className: "btn btn-sm btn-soft-primary" },
                { extend: "print", className: "btn btn-sm btn-soft-primary" },
                { extend: "pdf",   className: "btn btn-sm btn-soft-primary" }
            ],
            layout: {
                topStart: "buttons",
                topEnd: function() {
                    // Return a flex container with search input and button
                    return '<div class="d-flex align-items-center gap-2" style="justify-content: flex-end; width: 100%;"><label class="dataTables_filter" style="margin: 0;"><input type="search" class="form-control form-control-sm" placeholder="Search..." aria-controls="discount_users_table" id="discount_users_table_search"></label>' +
                           @can('add_discount_users')
                           '<button class="btn btn-sm btn-outline-success" id="addUserBtn" onclick="openAddUserModal()"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Discount User</button>' +
                           @endcan
                           '</div>';
                },
                bottomStart: {
                    pageLength: { menu: [[10, 25, 50, 100, 1000], [10, 25, 50, 100, 1000]] },
                    info: {}
                },
                bottomEnd: "paging"
            },
            initComplete: function(settings) {
                // Manually wire up the search input to DataTables
                var searchInput = $('#discount_users_table_search');
                if (searchInput.length) {
                    searchInput.on('keyup', function() {
                        discountUsersTable.search(this.value).draw();
                    });
                }
            }
        });
    }

    function reloadDiscountReasonsTable() {
        $.ajax({
            url: window.location.href,
            type: 'GET',
            success: function(response) {
                const $newContent = $(response);
                const $newTableBody = $newContent.find('#discount-reasons-tab #discount_reasons_table tbody');
                const $newTable = $newContent.find('#discount-reasons-tab #discount_reasons_table');
                
                if ($newTableBody.length && $newTable.length) {
                    $('#discount-reasons-tab #discount_reasons_table tbody').html($newTableBody.html());
                    initDiscountReasonsTable();
                } else {
                    window.location.reload();
                }
            },
            error: function() {
                window.location.reload();
            }
        });
    }

    function reloadDiscountUsersTable() {
        $.ajax({
            url: window.location.href,
            type: 'GET',
            success: function(response) {
                const $newContent = $(response);
                const $newTableBody = $newContent.find('#discount-users-tab #discount_users_table tbody');
                const $newTable = $newContent.find('#discount-users-tab #discount_users_table');
                
                if ($newTableBody.length && $newTable.length) {
                    $('#discount-users-tab #discount_users_table tbody').html($newTableBody.html());
                    initDiscountUsersTable();
                } else {
                    window.location.reload();
                }
            },
            error: function() {
                window.location.reload();
            }
        });
    }

    function openAddReasonModal() {
        $('#id_discount_reasons').val('');
        $('#id_display').val('');
        $('#id_display_container').hide();
        $('#discount_reason').val('');
        $('#type').val('Percentage');
        $('#value').val('');
        $('#active').val('Yes');

        $('#reasonModalLabel').text('Add Discount Reason');

        let el = document.getElementById('reasonModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function openEditReasonModal(id) {
        let button = $('.edit-reason-btn[data-id="' + id + '"]');
        $('#id_discount_reasons').val(button.data('id'));
        $('#id_display').val(button.data('id'));
        $('#id_display_container').show();
        $('#discount_reason').val(button.data('discount_reason'));
        $('#type').val(button.data('type') === 'None' ? 'Percentage' : button.data('type'));
        $('#value').val(button.data('value'));
        $('#active').val(button.data('active'));

        $('#reasonModalLabel').text('Edit Discount Reason');

        let el = document.getElementById('reasonModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function save_reason() {
        if ($('#discount_reason').val().trim() === '') {
            toastr.warning('Reason is required.');
            return;
        }
        if ($('#value').val().trim() === '' || parseFloat($('#value').val()) < 0) {
            toastr.warning('Value must be a positive number.');
            return;
        }

        $('#saveReasonBtn').prop('disabled', true);

        $.ajax({
            url: '{{ route("discount_configuration.save_or_update_reason") }}',
            method: 'POST',
            data: $('#reasonModalForm').serialize(),
            success: function(res) {
                $('#saveReasonBtn').prop('disabled', false);
                if (res.success) {
                    toastr.success(res.message);
                    $('#reasonModal').modal('hide');
                    reloadDiscountReasonsTable();
                } else {
                    toastr.error(res.message || 'Failed to save discount reason');
                }
            },
            error: function(xhr) {
                $('#saveReasonBtn').prop('disabled', false);
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    toastr.error(xhr.responseJSON.message);
                } else {
                    toastr.error('Failed to save discount reason: ' + xhr.responseText);
                }
            }
        });
    }

    function openAddUserModal() {
        $('#user_id').val('');
        $('#user_name').val('');
        $('#user_username').val('');
        $('#user_email').val('');
        $('#old_password').val('');
        $('#new_password').val('');
        $('#confirm_password').val('');
        
        // Show/hide password fields for add mode
        $('#old_password_container').hide();
        $('#new_password').prop('required', true);
        $('#confirm_password').prop('required', true);
        $('#password_label').text('Password');
        
        $('#userModalLabel').text('Add Discount User');

        let el = document.getElementById('userModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function openEditUserModal(id) {
        let button = $('.edit-user-btn[data-id="' + id + '"]');
        $('#user_id').val(button.data('id'));
        $('#user_name').val(button.data('name'));
        $('#user_username').val(button.data('username'));
        $('#user_email').val(button.data('email'));
        $('#old_password').val('');
        $('#new_password').val('');
        $('#confirm_password').val('');
        
        // Show/hide password fields for edit mode
        $('#old_password_container').show();
        $('#new_password').prop('required', false);
        $('#confirm_password').prop('required', false);
        $('#password_label').text('Change Password (leave blank to keep current)');
        
        $('#userModalLabel').text('Edit Discount User');

        let el = document.getElementById('userModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function save_user() {
        if ($('#user_name').val().trim() === '') {
            toastr.warning('Name is required.');
            return;
        }
        if ($('#user_username').val().trim() === '') {
            toastr.warning('Username is required.');
            return;
        }
        if ($('#user_email').val().trim() === '') {
            toastr.warning('E-mail is required.');
            return;
        }

        const isAddMode = $('#user_id').val() === '';
        
        // For add mode, password is required
        if (isAddMode) {
            if ($('#new_password').val().trim() === '') {
                toastr.warning('Password is required.');
                return;
            }
            if ($('#new_password').val() !== $('#confirm_password').val()) {
                toastr.warning('Password and confirmation do not match.');
                return;
            }
        } else {
            // For edit mode, validate password change if new password is provided
            if ($('#new_password').val().trim() !== '') {
                if ($('#old_password').val().trim() === '') {
                    toastr.warning('Old password is required to change password.');
                    return;
                }
                if ($('#new_password').val() !== $('#confirm_password').val()) {
                    toastr.warning('New password and confirmation do not match.');
                    return;
                }
            }
        }

        $('#saveUserBtn').prop('disabled', true);

        $.ajax({
            url: '{{ route("discount_configuration.save_or_update_user") }}',
            method: 'POST',
            data: $('#userModalForm').serialize(),
            success: function(res) {
                $('#saveUserBtn').prop('disabled', false);
                if (res.success) {
                    toastr.success(res.message);
                    $('#userModal').modal('hide');
                    reloadDiscountUsersTable();
                } else {
                    toastr.error(res.message || 'Failed to save discount user');
                }
            },
            error: function(xhr) {
                $('#saveUserBtn').prop('disabled', false);
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    toastr.error(xhr.responseJSON.message);
                } else {
                    toastr.error('Failed to save discount user: ' + xhr.responseText);
                }
            }
        });
    }
</script>
@endsection
